/*
 *  @(#)Monitor.java
 *
 * Copyright (C) 2002-2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Part of the GroboUtils package at:
 *  http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */
package net.sourceforge.groboutils.autodoc.v1.testserver;


import java.util.Hashtable;


/**
 * Monitors the state of multiple tests through the creation, retrieval, and
 * sending to a server of the TestData for a specific test.  Instances
 * will be in charge of pooling together events from a {@link TestCorrelate}.
 * <P>
 * This class itself is not a singleton, but each framework implementation
 * should have only one Monitor.  Hence, this is called a "pseudo-singleton".
 * <P>
 * A Monitor may have multiple tests being tested at the same time.  However,
 * each test must be uniquely identifiable through a {@link TestInfo}; if two
 * tests execute at the same time with the same <tt>TestInfo</tt>, the monitor
 * will identify them as the same test.  Uniqueness is guaranteed through the
 * <tt>TestInfo</tt> instances.
 *
 * @author     Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @since      March 17, 2002
 * @version    $Date: 2003/02/10 22:52:13 $
 */
public interface Monitor
{
    /**
     * Adds a new <tt>TestData</tt> instance related to the given
     * <tt>TestInfo</tt>.  If there already is a <tt>TestData</tt> for the
     * given <tt>TestInfo</tt>, then an exception is thrown.
     *
     * @param info the unique test identifier to create a new <tt>TestData</tt>
     *      instance for.
     * @exception IllegalStateException if <tt>info</tt> is already
     *      been added without having been sent.
     * @exception IllegalArgumentException if <tt>info</tt> is <tt>null</tt>.
     */
    public void addTestData( TestInfo info );
    
    
    /**
     * Retrieves the data associated with the given <tt>TestInfo</tt>, as was
     * created through {@link #addTestData( TestInfo )}.  If the <tt>info</tt>
     * was never passed to the add method, then an exception is thrown.
     *
     * @param info the unique test identifier.
     * @return the data for the given <tt>info</tt>.
     * @exception IllegalStateException if <tt>info</tt> has not been added,
     *      or has been removed through the send call.
     * @exception IllegalArgumentException if <tt>info</tt> is <tt>null</tt>.
     */
    public TestData getTestData( TestInfo info );
    
    
    /**
     * Sends the <tt>TestData</tt> associated with <tt>info</tt> to the
     * inner server, and removes the data from the inner cache.  If the
     * <tt>info</tt> has not been added, then an exception is thrown.
     *
     * @param info the unique test identifier
     * @exception IllegalStateException if <tt>info</tt> has not been added,
     *      or has been removed through the send call.
     * @exception IllegalArgumentException if <tt>info</tt> is <tt>null</tt>.
     */
    public void sendTestData( TestInfo info );
}

