/*
 * PropertyDatabase.java
 *
 * Copyright (C) 2001,,2003 2002 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */

 
package net.sourceforge.groboutils.util.io.v1;
 
import java.io.IOException;
import java.io.FileNotFoundException;
import java.io.File;
import java.io.PrintWriter;
import java.io.FileInputStream;
import java.io.FileOutputStream;

import java.util.ResourceBundle;
import java.util.Locale;
import java.util.Properties;
import java.util.Enumeration;
import java.util.Hashtable;



/**
 * A database of property files. Internally, it uses a
 * ResourceBundle, so that locale specific properties can be used.
 * The format is for the application defined property files to be defined
 * as "read-only", and for a user defined property file to be the
 * readable/writeable properties (i.e. modifications or additions or
 * removals of the default properties). User settings override the
 * read-only settings. The user property file is not localized, and is
 * stored at <tt>$home/.<i>app-name</i>/user.properties</tt> (or the filename
 * may be specified).
 * <P>
 * By default, the properties are loaded from the Resource streams,
 * although this can be changed.
 * <P>
 * Before using this class, you must initialize the user property file
 * by either {@link #setApplicationName( String )} or
 * {@link #setUserPropertyFile( String )}.
 * <P>
 * The stored data is only of type String, and multiple identical key
 * entries are not possible - only the first one is allowed.
 *
 * @author   Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @since    January 7, 2001
 * @version  $Date: 2003/02/10 22:52:45 $
 */
public class PropertyDatabase
{
    //---------------------------------------------------------------------
    // Public Static Fields
     
     
    //---------------------------------------------------------------------
    // Protected Static Fields
     
     
    //---------------------------------------------------------------------
    // Private Static Fields
    
    private static final String USER_FILE_NAME = "user.properties";
    private static final String USER_REMOVED = "<<REMOVED>>";
     
     
    //---------------------------------------------------------------------
    // Public Fields
     
     
    //---------------------------------------------------------------------
    // Protected Fields
     
     
    //---------------------------------------------------------------------
    // Private Fields
    
    private Hashtable resourceValues = new Hashtable();
    private Properties userValues = null;
    private boolean doAutosave = false;
    private File userProps = null;
    private Locale locale = Locale.getDefault();
    private PrintWriter trace = null;
    private File appDirectory = null;
     
     
    //---------------------------------------------------------------------
    // Constructors
     
     
    /**
     * Default Constructor
     */
    public PropertyDatabase()
    {
        // do nothing
    }
     
    /**
     * Specify the Locale to load the properties from.
     */
    public PropertyDatabase( Locale l )
    {
        this.locale = l;
    }
     
     
     
    //---------------------------------------------------------------------
    // Public Methods
    
    
    /**
     * Set the application name and thus the corresponding directory that will
     * store the user properties.  The user properties are located
     * at <tt>$home/.<i>app-name</i>/user.properties</tt>. If the user property
     * file is already set, then an IllegalStateException is thrown.
     *
     * @param name name of the application.
     */
    public void setApplicationName( String name )
            throws IOException
    {
        if (this.userProps != null)
        {
            throw new IllegalStateException(
                "property file already in use" );
        }
        if (name == null)
        {
            throw new IllegalArgumentException(
                "no null args" );
        }
        
        // find the home directory.
        File home = new File( System.getProperty("user.home") );
        if (home == null || !home.isDirectory() || !home.exists())
        {
            throw new FileNotFoundException(
                "user.home property not valid" );
        }
        
        // for hidden unix home path
        name = "." + name;
        this.appDirectory = new File( home, name );
        if (!this.appDirectory.exists())
        {
            // need to create the appDir directory
            this.appDirectory.mkdir();
        }
        else if (!this.appDirectory.isDirectory())
        {
            // error - app is not a directory
            throw new FileNotFoundException(
                "application directory "+this.appDirectory.toString()+
                " is not a directory" );
        }
        
        // get the user file
        this.userProps = new File( this.appDirectory, USER_FILE_NAME );
        // if the file doesn't exist, create it
        this.userProps.createNewFile();
        
        loadUserProperties();
    }
    
    
    /**
     * Returns the directory where the user properties are stored for the
     * current application, or <tt>null</tt> if there is no application set.
     */
    public File getApplicationDirectory()
    {
        return this.appDirectory;
    }
    
    
    /**
     * Set the user property file name exactly. If the user property file
     * is already set, then an IllegalStateException is thrown.
     */
    public void setUserPropertyFile( String name )
            throws IOException
    {
        if (this.userProps != null)
        {
            throw new IllegalStateException(
                "property file already in use" );
        }
        if (name == null)
        {
            throw new IllegalArgumentException(
                "no null args" );
        }
        File f = new File( name );
        if (f.isDirectory())
        {
            throw new IOException("user property file "+f+" is a directory");
        }
        // if the file doesn't exist, create it
        f.createNewFile();
        this.userProps = f;
        
        loadUserProperties();
    }
    
    
    /**
     * Saves the current user properties.
     */
    public void saveUserProperties()
            throws IOException
    {
        if (this.userValues == null || this.userProps == null)
        {
            throw new IllegalStateException("database not initialized");
        }
        synchronized( this )
        {
            FileOutputStream fos = null;
            try
            {
                fos = new FileOutputStream( this.userProps );
                this.userValues.store( fos, "User values" );
            }
            finally
            {
                if (fos != null) fos.close();
                fos = null;
            }
        }
    }
    
    
    /**
     * Adds a resource bundle of the given name to the database, from the
     * specified locale. Note that
     * if a user property already exists with a given key, then the
     * user property overrides the resource property.
     */
    public void addResourceBundle( String resourceName )
    {
        ResourceBundle rb = ResourceBundle.getBundle( resourceName,
            this.locale );
        if (rb == null)
        {
            return;
        }
        
        Hashtable rv = this.resourceValues;
        Enumeration enum = rb.getKeys();
        String key, val;
        synchronized( rv )
        {
            while (enum.hasMoreElements())
            {
                key = (String)enum.nextElement();
                if (rv.contains( key ) && this.trace != null)
                {
                    this.trace.println("Resource "+resourceName+
                        " contains a duplicate key '"+key+"'.");
                }
                val = rb.getString( key );
                if (val == null)
                {
                    if (this.trace != null)
                    {
                        this.trace.println("Resource "+resourceName+
                            " contains a null key '"+key+"'." );
                    }
                    if (rv.contains( key ))
                    {
                        rv.remove( key );
                    }
                }
                else
                {
                    rv.put( key, val );
                }
            }
        }
    }
    
    
    /**
     * Retrieves the auto-save setting.
     */
    public boolean isAutosaveOn()
    {
        return this.doAutosave;
    }
    
    
    /**
     * Sets the current autosave setting.
     */
    public void setAutosaveOn( boolean yes )
    {
        this.doAutosave = yes;
    }
    
    
    /**
     * Removes a value from the properties. If the property is defined
     * by a resource, then the user list must specify that it is
     * removed.
     *
     * @param key the key to remove
     * @return the value the key was assigned to, or <tt>null</tt> if nothing
     *      was removed.
     */
    public String removeValue( String key )
    {
        String val = null;
        val = this.userValues.getProperty( key );
        if (val != null)
        {
            if (val.equals( USER_REMOVED ))
            {
                return null;
            }
            if (this.resourceValues.contains( key ))
            {
                // remove from the user list
                this.userValues.setProperty( key, USER_REMOVED );
            }
            else
            {
                this.userValues.remove( key );
            }
            autoSave();
            return val;
        }
        val = (String)this.resourceValues.get( key );
        if (val != null)
        {
            // we know the user values doesn't have this key
            this.userValues.setProperty( key, USER_REMOVED );
            autoSave();
            return val;
        }
        
        // no one had this key.
        return null;
    }
    
    
    /**
     * Sets the given value to the user properties.
     *
     * @param key the key to assign the value to
     * @param value the value to be assigned to the key
     */
    public void setValue( String key, String value )
    {
        if (value == null)
        {
            removeValue( key );
            return;
        }
        this.userValues.setProperty( key, value );
        autoSave();
    }
    
    
    /**
     * Retrieves the value associated with the given key.
     *
     * @param key the key to pull the value out of
     */
    public String getValue( String key )
    {
        String val = this.userValues.getProperty( key );
        if (val != null)
        {
            if (val.equals( USER_REMOVED ))
            {
                return null;
            }
            return val;
        }
        return (String)this.resourceValues.get( key );
    }
    
    
    /**
     * Resets the user property to the resource bundle's default value.
     *
     * @param key the key to be reset.
     * @return the default value for the key.
     */
    public String setValueToDefault( String key )
    {
        boolean needSave = this.userValues.contains( key );
        this.userValues.remove( key );
        if (needSave) autoSave();
        return (String)this.resourceValues.get( key );
    }
    
    
    //---------------------------------------------------
    // Convenience functions
    
    /**
     * Convenience function to convert a property to an int value.
     *
     * @return the given key converted to an integer, or Integer.MIN_VALUE
     *      if there was a parse error.
     */
    public int getIntValue( String key )
    {
        try
        {
            return Integer.parseInt( getValue( key ) );
        }
        catch (NumberFormatException nfe)
        {
            return Integer.MIN_VALUE;
        }
        catch (NullPointerException nfe)
        {
            return Integer.MIN_VALUE;
        }
    }
    
    
    /**
     * Convenience function to convert an int value to a String property.
     */
    public void setIntValue( String key, int value )
    {
        setValue( key, Integer.toString( value ) );
    }
    
    
    /**
     * Convenience function to convert a property to a boolean value.
     *
     * @return the given key converted to a boolean, or false
     *      if there was a parse error.
     */
    public boolean getBooleanValue( String key )
    {
        try
        {
            return Boolean.getBoolean( getValue( key ) );
        }
        catch (NullPointerException nfe)
        {
            return false;
        }
    }
    
    
    /**
     * Convenience function to convert a boolean value to a String property.
     */
    public void setBooleanValue( String key, boolean value )
    {
        setValue( key,
            (value ? Boolean.TRUE.toString() : Boolean.FALSE.toString() ) );
    }
    
    
    /**
     * Convenience function to convert a property to a byte value.
     *
     * @return the given key converted to a byte, or Byte.MIN_VALUE
     *      if there was a parse error.
     */
    public byte getByteValue( String key )
    {
        try
        {
            return Byte.parseByte( getValue( key ) );
        }
        catch (NumberFormatException nfe)
        {
            return Byte.MIN_VALUE;
        }
        catch (NullPointerException nfe)
        {
            return Byte.MIN_VALUE;
        }
    }
    
    
    /**
     * Convenience function to convert a byte value to a String property.
     */
    public void setByteValue( String key, byte value )
    {
        setValue( key, Byte.toString( value ) );
    }
    
    
    /**
     * Convenience function to convert a property to a char value.
     *
     * @return the given key converted to a char, or Character.MIN_VALUE
     *      if there was a parse error.
     */
    public char getCharValue( String key )
    {
        String val = getValue( key );
        if (val == null || val.length() <= 0) return Character.MIN_VALUE;
        return val.charAt(0);
    }
    
    
    /**
     * Convenience function to convert a char value to a String property.
     */
    public void setCharValue( String key, char value )
    {
        setValue( key, ""+value );
    }
    
    
    /**
     * Convenience function to convert a property to a double value.
     *
     * @return the given key converted to a double, or Double.MIN_VALUE
     *      if there was a parse error.
     */
    public double getDoubleValue( String key )
    {
        try
        {
            return Double.parseDouble( getValue( key ) );
        }
        catch (NumberFormatException nfe)
        {
            return Double.MIN_VALUE;
        }
        catch (NullPointerException nfe)
        {
            return Double.MIN_VALUE;
        }
    }
    
    
    /**
     * Convenience function to convert a byte value to a String property.
     */
    public void setDoubleValue( String key, double value )
    {
        setValue( key, Double.toString( value ) );
    }
    
    
    /**
     * Convenience function to convert a property to a float value.
     *
     * @return the given key converted to a float, or Float.MIN_VALUE
     *      if there was a parse error.
     */
    public float getFloatValue( String key )
    {
        try
        {
            return Float.parseFloat( getValue( key ) );
        }
        catch (NumberFormatException nfe)
        {
            return Float.MIN_VALUE;
        }
        catch (NullPointerException nfe)
        {
            return Float.MIN_VALUE;
        }
    }
    
    
    /**
     * Convenience function to convert a float value to a String property.
     */
    public void setFloatValue( String key, float value )
    {
        setValue( key, Float.toString( value ) );
    }
    
    
    /**
     * Convenience function to convert a property to a long value.
     *
     * @return the given key converted to a long, or Long.MIN_VALUE
     *      if there was a parse error.
     */
    public long getLongValue( String key )
    {
        try
        {
            return Long.parseLong( getValue( key ) );
        }
        catch (NumberFormatException nfe)
        {
            return Long.MIN_VALUE;
        }
        catch (NullPointerException nfe)
        {
            return Long.MIN_VALUE;
        }
    }
    
    
    /**
     * Convenience function to convert a long value to a String property.
     */
    public void setLongValue( String key, long value )
    {
        setValue( key, Long.toString( value ) );
    }
    
    
    /**
     * Convenience function to convert a property to a short value.
     *
     * @return the given key converted to a short, or Short.MIN_VALUE
     *      if there was a parse error.
     */
    public short getShortValue( String key )
    {
        try
        {
            return Short.parseShort( getValue( key ) );
        }
        catch (NumberFormatException nfe)
        {
            return Short.MIN_VALUE;
        }
        catch (NullPointerException nfe)
        {
            return Short.MIN_VALUE;
        }
    }
    
    
    /**
     * Convenience function to convert a short value to a String property.
     */
    public void setShortValue( String key, short value )
    {
        setValue( key, Short.toString( value ) );
    }
    
    

    
    //--------------------------------------------------
    // Debug aid function
    
    /**
     * Sets the trace stream. If you set this to non-null, then warnings,
     * such as ResourceBundles containing duplicate keys, will be reported
     * to the stream. Errors will still be thrown as exceptions. Autosave
     * will send any exceptions to this trace.
     */
    public void setTrace( PrintWriter tracer )
    {
        this.trace = tracer;
    }
    
    //---------------------------------------------------------------------
    // Protected Methods
    
    
    /**
     * 
     */
    protected void loadUserProperties()
            throws IOException
    {
        Properties prop = new Properties();
        FileInputStream fis = new FileInputStream( this.userProps );
        prop.load( fis );
        fis.close();
        
        // if no exception was thrown...
        synchronized( this )
        {
            this.userValues = prop;
        }
    }
    
    
    /**
     * 
     */
    protected void autoSave()
    {
        if (isAutosaveOn())
        {
            try
            {
                // attempt to save
                saveUserProperties();
            }
            catch (IOException ioe)
            {
                if (this.trace != null)
                {
                    ioe.printStackTrace( this.trace );
                }
            }
        }
    }
    
     
    //---------------------------------------------------------------------
    // Private Methods
     
     
}
 

