/* GNOME Transcript
 * Copyright (C) 1999 the Free Software Foundation
 * Authors : Matias Mutchinick
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */



#include <glib.h>
#include "gtrans_ifase_result.h"



/************************************************************************
 * Memory managment                                                     *
 ************************************************************************/



/**
 * gtrans_ifase_result_alloc
 *
 * Allocate memory and set some default falues for the GTransIFaseResult.
 * Returns : A pointer to a GTransIFaseResult.
 */
GTransIFaseResult *
gtrans_ifase_result_alloc()
{
	GTransIFaseResult *res;
	
	res = (GTransIFaseResult *)g_malloc(sizeof(GTransIFaseResult));
	
	res->rows     = NULL;
	res->fields   = NULL;
	res->status   = GTRANS_IFASE_RESULT_OK;
	res->type     = GTRANS_IFASE_RESULT_COMMAND;
	res->n_rows   = 0;
	res->n_cols   = 0;
	res->errormsg = NULL;
	
	return res;
}




/*
 * gtrans_ifase_result_free
 * @res : The GTransIFaseResult to free.
 *
 * Free all memory used by @res, this functions check
 * that wathever is being freed at the moment is not NULL.
 * The things that are freed are:
 * * @res->fields
 * * @res->rows
 * * @res->errormsg
 */
void
gtrans_ifase_result_free(GTransIFaseResult *res)
{
	GList   *l;
	gint     i;
	gchar  **tmp;
	
	g_return_if_fail(res != NULL);
	
	/* Free the field names */
	if (res->fields){
		for (i = 0 ; i < res->n_cols ; i++)
			if(res->fields[i])
				g_free(res->fields[i]);
		
		g_free(res->fields);
	}
	
	/* Free the tupples */
	for ( l = res->rows ; l != NULL ; l = l->next ){
		
		tmp = (gchar **) l->data;
		if (tmp){
			for (i = 0 ; i > res->n_cols ; i++)
				if(tmp[i])
					g_free(tmp[i]);
			
			g_free(tmp);
		}
	}
	g_list_free(res->rows);

	
	/* Free the error message */
	if (res->errormsg)
		g_free(res->errormsg);
	
	/* Free the result */
	g_free(res);
}





/************************************************************************
 * Functions for retriving from a GTransIFaseResult                     *
 ************************************************************************/
	

/**
 * gtrans_ifase_result_get_item
 * @res  : A GTransIFaseResult of a query exec.
 * @row  : number of a row.
 * @col  : number of a column.
 *
 * Get the result item placed at (@row,@col).
 * Returns : A string.
 */
gchar *
gtrans_ifase_result_get_item(GTransIFaseResult  *res,
			     gint                row,
			     gint                col)
{
	gchar **tmp;
	
	tmp = (gchar **) g_list_nth_data(res->rows,row);
	
	return tmp[col];
}




/**
 * gtrans_ifase_result_get_row
 * @res  : A GTransIFaseResult of a query exec.
 * @row  : number of a row.
 *
 * Get the @row from the result.
 * Returns : An arry of strings.
 */
gchar **
gtrans_ifase_result_get_row(GTransIFaseResult  *res,
			    gint                row)
{
	return (gchar **) g_list_nth_data(res->rows,row);
}




/**
 * gtrans_ifase_result_fetch_row
 * @res  : A GTransIFaseResult of a query exec.
 *
 * Fetch the next row from the result.
 * Returns : An arry of strings.
 */
gchar **
gtrans_ifase_result_fetch_row(GTransIFaseResult  *res)
{
	gchar **data = NULL;
	
	g_return_val_if_fail(res != NULL, NULL);
	
	if (!res->rows)
		return NULL;
	
	data = (gchar **) (res->rows)->data;
	if (res->rows->next)
		res->rows = res->rows->next;
	
	return data;
}



/**
 * gtrans_ifase_result_rewind
 * @res  : A GTransIFaseResult of a query exec.
 *
 * Go to the begining of the result rows
 * in case we must after fetching rows.
 */
void
gtrans_ifase_result_rewind(GTransIFaseResult *res)
{
	res->rows = g_list_first(res->rows);
}




/************************************************************************
 * Functions for modifing a       GTransIFaseResult                     *
 ************************************************************************/
	


/**
 * gtrans_ifase_result_replace_row
 * @res  : The GTransIFaseResult
 * @row  : The index of a row
 * @data : An arry of string
 *
 * Replace de contents of row @row with @data.
 */
void
gtrans_ifase_result_replace_row(GTransIFaseResult *res,
				gint               row,
				gchar            **data)
{
	GList              *tmp;
	gchar             **old_data;
	gint                cols , i;
	
	cols = res->n_cols;
	
	tmp = g_list_nth(res->rows,row);
	old_data = (gchar **) tmp->data;  
	tmp->data = data;
	
	for (i = 0 ; i < cols ; i++)
		if (old_data[i] != NULL)
			g_free(old_data[i]);
	
	g_free (old_data);
}




/**
 * gtrans_ifase_result_append_row
 * @res  : The GTransIFaseResult
 * @data : An arry of string
 *
 * Append a new row with @data to @res. 
 */
void
gtrans_ifase_result_append_row(GTransIFaseResult *res,
			       gchar            **data)
{
	res->rows = g_list_insert(res->rows,data,res->n_rows);
	res->n_rows++;
}



/**
 * gtrans_ifase_result_remove_row
 * @res  : The GTransIFaseResult
 * @row  : The index of a row
 *
 * Remove the row @row from the result.
 */
void
gtrans_ifase_result_remove_row(GTransIFaseResult *res,
			       gint               row)
{
	GList  *tmp;
	gchar **data;
	
	tmp = g_list_nth(res->rows,row);
	data = (gchar**)tmp->data;
	res->rows = g_list_remove_link(res->rows,tmp);
	res->n_rows--;
	g_free(data);
}




/************************************************************************
 * Functions for status and error messages                              *
 ************************************************************************/


/**
 * gtrans_ifase_result_error_msg
 * @res  : The result.
 *
 * Return a pointer to the result error message.
 */
gchar *
gtrans_ifase_result_error_msg(GTransIFaseResult *res)
{
	return res->errormsg;
}




/**
 * gtrans_ifase_result_status
 * @res  : The result.
 *
 * Return the status of the result.
 */
gint
gtrans_ifase_result_status(GTransIFaseResult *res)
{
	return res->status;
}



/**
 * gtrans_ifase_result_type
 * @res  : The result.
 *
 * Return the type of the result.
 */
gint
gtrans_ifase_result_type(GTransIFaseResult *res)
{
	return res->type;
}





