/**
 * GUI Commands
 * Copyright 2004 Andrew Pietsch
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * $Id: DefaultToolbarFactory.java,v 1.8 2005/06/10 08:45:45 pietschy Exp $
 */

package org.pietschy.command;

import javax.swing.*;
import java.awt.*;

/**
 * Provides the default implementation of {@link ToolbarFactory}.  In its default configuration
 * this factory creates toolbars for which {@link JToolBar#isFloatable} is <tt>false</tt> and
 * buttons for which {@link JButton#isRolloverEnabled} is <tt>true</tt>.  All buttons are created
 * with a margin of 1 pixel.
 * @see #setFloatable
 * @see #setRolloverEnabled
 * @see #setInsets
 */
public class
DefaultToolbarFactory
implements ToolbarFactory
{
   private Insets insets;
   private boolean floatable;
   private boolean rolloverEnabled;
   private boolean focusable = false;

   /**
    * Creates a new factory that creates non-floatable toolbars and rollover enabled buttons.
    */
   public DefaultToolbarFactory()
   {
      this(null, false, true);
   }

   /**
    * Creates a new factory.
    * @param insets the insets to use for all buttons.
    * @param floatable <tt>true</tt> if the toolbars created are to be
    * floatable, <tt>false</tt> otherwise.
    * @param rolloverEnabled <tt>true</tt> if buttons created are to be rollover enabled,
    * <tt>false</tt> otherwise.
    */
   public DefaultToolbarFactory(Insets insets, boolean floatable, boolean rolloverEnabled)
   {
      this.insets = insets;
      this.floatable = floatable;
      this.rolloverEnabled = rolloverEnabled;
   }

   /**
    * Creates a new {@link JButton}.
    * @return a new {@link JButton}.
    */
   public JButton createButton()
   {
      JButton button = new JButton();
      configureButton(button);
      return button;
   }


   /**
    * Creates a new {@link JCheckBox}.
    * @return a new {@link JCheckBox}.
    */
   public JCheckBox createCheckBox()
   {
      JCheckBox button = new JCheckBox();
      configureButton(button);
      return button;
   }

   /**
    * Creates a new {@link JRadioButton}.
    * @return a new {@link JRadioButton}.
    */
   public JRadioButton createRadioButton()
   {
      JRadioButton button = new JRadioButton();
      configureButton(button);
      return button;
   }

   /**
    * Creates a new {@link JToggleButton}.
    * @return a new {@link JToggleButton}.
    */
   public AbstractButton createToggleButton()
   {
      JToggleButton button = new JToggleButton();
      configureButton(button);
      return button;
   }

   /**
    * Creates a new {@link JToolBar}.
    * @return a new {@link JToolBar}.
    */
   public JToolBar createToolbar()
   {
      JToolBar toolBar = new JToolBar();
      toolBar.setRollover(rolloverEnabled);
      toolBar.setFloatable(floatable);
      toolBar.setFocusable(focusable);
      return toolBar;
   }


   private void configureButton(AbstractButton button)
   {
      button.setMargin(insets);
      button.setFocusPainted(false);
      button.setFocusable(focusable);
   }

   /**
    * Gets the {@link Insets} this factory is applying to the buttons it creates.
    * @return the {@link Insets} this factory is applying to the buttons it creates.
    */
   public Insets getInsets()
   {
      return insets;
   }

   /**
    * Sets the {@link Insets} this factory will apply to the buttons it creates.
    * @param insets the {@link Insets} to use.
    */
   public void setInsets(Insets insets)
   {
      this.insets = insets;
   }

   /**
    * Checks if this factory is creating floatable toolbars.
    * @return <tt>true</tt> if this factory is setting {@link JToolBar#setFloatable floatable }
    * to <tt>true</tt>, <tt>false</tt> otherwise.
    */
   public boolean isFloatable()
   {
      return floatable;
   }

   /**
    * Configures if this factory is creating floatable toolbars.
    * @param floatable <tt>true</tt> to make this factory set {@link JToolBar#setFloatable floateable}
    * to <tt>true</tt>, <tt>false</tt> otherwise.
    */
   public void setFloatable(boolean floatable)
   {
      this.floatable = floatable;
   }

   /**
    * Checks if this factory is creating rollover enabled buttons.
    * @return <tt>true</tt> if this factory is setting {@link AbstractButton#setRolloverEnabled rollover enabled}
    * to <tt>true</tt>, <tt>false</tt> otherwise.
    */
   public boolean isRolloverEnabled()
   {
      return rolloverEnabled;
   }

   /**
    * Configures if this factory is creating rollover enabled buttons.
    * @param rolloverEnabled <tt>true</tt> to make this factory set {@link AbstractButton#setRolloverEnabled rollover enabled}
    * to <tt>true</tt>, <tt>false</tt> otherwise.
    */
   public void setRolloverEnabled(boolean rolloverEnabled)
   {
      this.rolloverEnabled = rolloverEnabled;
   }

   /**
    * Checks if the factory is building focusable toolbars.  The default is <code>false</code>.
    * @return <code>true</code> if the factory is building focusable toolbars, <code>false</code> otherwise.
    */
   public boolean isFocusable()
   {
      return focusable;
   }

   /**
    * Configure the factory to create focusable toolbars.  The default is <code>false</code>.
    * @param focusable <code>true</code> to create focusable toolbars, <code>false</code> otherwise.
    */
   public void setFocusable(boolean focusable)
   {
      this.focusable = focusable;
   }

}
