package org.pietschy.command;

import javax.swing.*;

/**
 * The MacFaceRender extends {@link DefaultFaceRenderer} to provide Mac specific rendering of
 * buttons and menus.  In particular this renderer ensures.
 * <ul>
 * <li>Ignores icons on menus</li>
 * <li>Ignores mnemonics on both buttons and menus</li>
 * <li>Configures the client properties as specified by faces</li>
 * <li>Configures the client properties of the buttons created by the members of ToggleGroups.</li>
 * </ul>
 */
public class
MacFaceRenderer
extends DefaultFaceRenderer
{

   private static final String TOOLBAR_STYLE = "Quaqua.Toolbar.style";

   private static boolean buttonMnemonicsEnabled = false;
   private static boolean menuMnemonicsEnabled = false;

   public void
   configureButton(AbstractButton button, RenderContext buttonContext)
   {
      super.configureButton(button, buttonContext);

      CommandGroup group = buttonContext.getGroup();

      // configureMenu quaqu settings
      if (group instanceof ToggleCommandGroup)
      {
         String quaquToolbarStyle = buttonContext.getGroup().getProperty(TOOLBAR_STYLE, "");

         if (group.getMemberCount() == 1)
         {
            button.putClientProperty("Quaqua.Button.style", "toggle");
         }
         else
         {
            if ("rounded".equals(quaquToolbarStyle))
            {

               if (buttonContext.isFirstMemeber())
                  button.putClientProperty("Quaqua.Button.style", "toggleWest");
               else if (buttonContext.isLastMember())
                  button.putClientProperty("Quaqua.Button.style", "toggleEast");
               else
                  button.putClientProperty("Quaqua.Button.style", "toggleCenter");
            }
            else if ("tabbed".equalsIgnoreCase(quaquToolbarStyle))
            {
               button.putClientProperty("Quaqua.Button.style", "toolBarTab");
            }
            else
            {
               // nothing has been configured at the group level so we set is
               // to the standard tabbed style...
               button.putClientProperty("Quaqua.Button.style", "toggle");
               button.putClientProperty("JButton.buttonType", "toolbar");

               // ..but let the face override it..
               configureClientProperties(buttonContext, button);

            }
         }
      }
      else
      {
         // configure any client properties specified by the face..
         configureClientProperties(buttonContext, button);
      }


   }

   private void
   configureClientProperties(RenderContext buttonContext, AbstractButton button)
   {
      // check the face properties for either Aqua or Quaqua properties.
      Face face = buttonContext.getFace();
      String[] names = face.getClientPropertyNames();
      for (int i = 0; i < names.length; i++)
      {
         if (names[i].startsWith("Quaqua."))
         {
            button.putClientProperty(names[i], face.getClientProperty(names[i]));
         }
         else if (names[i].startsWith("Aqua."))
         {
            // use the name after the 'Aqua.'
            String cpn = names[i].substring(names[i].indexOf('.') + 1);
            button.putClientProperty(cpn, face.getClientProperty(names[i]));
         }

      }
   }


   protected void
   configureMnemonic(AbstractButton button, Face face)
   {
      if (shouldDisplayMnemonic(button))
         super.configureMnemonic(button, face);
   }

   protected void
   configureIcons(AbstractButton button, Face face)
   {
      if (shouldDisplayIcon(button))
         super.configureIcons(button, face);
   }

   /**
    * Returns <tt>false</tt>.
    *
    * @param button the button to check.
    * @return <tt>false</tt>.
    */
   protected boolean
   shouldDisplayMnemonic(AbstractButton button)
   {
      if (button instanceof JMenuItem)
         return menuMnemonicsEnabled;
      else
         return buttonMnemonicsEnabled;
   }


   /**
    * Returns <tt>false</tt> if the button is an instance of {@link javax.swing.JMenuItem}.
    *
    * @param button the button to check.
    * @return false if the button is an instance of {@link javax.swing.JMenuItem}, <tt>true</tt>
    *         otherwise.
    */
   private boolean
   shouldDisplayIcon(AbstractButton button)
   {
      // mec menus don't have icons..
      return !(button instanceof JMenuItem);
   }


   /**
    * Checks the the current button context is rendering a toolbar.  This simply checks that
    * the parent groups isn't null and that the button face is {@link Face#TOOLBAR}.
    *
    * @param buttonContext the {@link RenderContext} for the current button.
    * @return <tt>true</tt> if the button has a toolbar face and is being rendered as part of
    *         a group.
    */
   public boolean
   isToolbar(RenderContext buttonContext)
   {
      return buttonContext.getGroup() != null && Face.TOOLBAR.equals(buttonContext.getFaceName());
   }


   /**
    * Checks if the renderer will be configuring mnemonics on buttons.
    *
    * @return <tt>true</tt> if the buttons mnemonics are enabled, <tt>false</tt> otherwise.
    */
   public static boolean
   isButtonMnemonicsEnabled()
   {
      return buttonMnemonicsEnabled;
   }

   /**
    * Sets if the renderer will be configuring mnemonics on buttons.
    *
    * @param buttonMnemonicsEnabled <tt>true</tt> if the buttons mnemonics are enabled, <tt>false</tt> otherwise.
    */
   public static void
   setButtonMnemonicsEnabled(boolean buttonMnemonicsEnabled)
   {
      MacFaceRenderer.buttonMnemonicsEnabled = buttonMnemonicsEnabled;
   }

   /**
    * Checks if the renderer will be configuring mnemonics on menu items.
    *
    * @return <tt>true</tt> if the menu mnemonics are enabled, <tt>false</tt> otherwise.
    */
   public static boolean
   isMenuMnemonicsEnabled()
   {
      return menuMnemonicsEnabled;
   }

   /**
    * Sets if the renderer will be configuring mnemonics on menu items.
    *
    * @param menuMnemonicsEnabled <tt>true</tt> if the menu mnemonics are enabled, <tt>false</tt> otherwise.
    */
   public static void
   setMenuMnemonicsEnabled(boolean menuMnemonicsEnabled)
   {
      MacFaceRenderer.menuMnemonicsEnabled = menuMnemonicsEnabled;
   }


}
