/**
 * GUI Commands
 * Copyright 2005 Andrew Pietsch
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * $Id: RenderContext.java,v 1.7 2006/02/26 00:59:06 pietschy Exp $
 */
package org.pietschy.command;

import javax.swing.*;
import java.util.HashMap;

/**
 * The RenderContext constains the information about buttons and menus that are
 * created by the library.  This includes the command that created the button and the
 * face that was used.  If the button or menu was created in the context of a group, then the
 * group is also specified.
 *
 * @author andrewp
 * @version $Revision: 1.7 $
 */
public class
RenderContext
{

   private static final String RENDER_CONTEXT_KEY = RenderContext.class.getName();

   /**
    * The command that created the button.
    */
   private Command command;

   /**
    * The name of the face the button was created with.
    */
   private String faceName;

   /**
    * The group that requested the creation of this button, or null the the button
    * was created independantly.
    */
   private CommandGroup group;


   private boolean firstMemeber = false;
   private boolean lastMember = false;


   private HashMap properties = new HashMap();


   /**
    * Gets the rendering context for the specified component, or <tt>null</tt> if it doesn't
    * have one.
    *
    * @param component the component with the renderering context.
    * @return the rendering context or <tt>null</tt> if the component doesn't have one.
    */
   public static RenderContext
   get(JComponent component)
   {
      return (RenderContext) component.getClientProperty(RENDER_CONTEXT_KEY);
   }


   /**
    * Binds this context to the specified component.
    *
    * @param component the component to bind to.
    * @see #get(javax.swing.JComponent)
    */
   static void
   bind(JComponent component, Command parent, String faceName)
   {
      component.putClientProperty(RENDER_CONTEXT_KEY, new RenderContext(parent, faceName));
   }


   static void
   unbind(JComponent component, Command command)
   {
      if (!command.isAttachedTo(component))
         throw new IllegalStateException("Command isn't the owner of the button");

      // reset the key
      component.putClientProperty(RENDER_CONTEXT_KEY, null);
   }


   /**
    * Creates a new RenderContext for the specified command and face.
    *
    * @param parent   the command that created the button.
    * @param faceName the name of the face the button was created with.
    */
   private RenderContext(Command parent, String faceName)
   {
      if (parent == null)
         throw new NullPointerException("parent is null");

      if (faceName == null)
         throw new NullPointerException("faceName is null");

      this.command = parent;
      this.faceName = faceName;
   }


   void
   setGroupContext(CommandGroup group, int groupIndex)
   {
      this.group = group;
      this.firstMemeber = groupIndex == 0;
      this.lastMember = groupIndex == group.getMemberCount() - 1;
   }

   public Command
   getCommand()
   {
      return command;
   }

   public String
   getFaceName()
   {
      return faceName;
   }

   public Face
   getFace()
   {
      return command.getFace(getFaceName());
   }

   public CommandGroup
   getGroup()
   {
      return group;
   }

   public boolean
   isFirstMemeber()
   {
      return firstMemeber;
   }

   public boolean
   isLastMember()
   {
      return lastMember;
   }

}
