/**
 * GUI Commands
 * Copyright 2004 Andrew Pietsch
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * $Id: ToggleGroupController.java,v 1.2 2005/09/18 00:37:18 pietschy Exp $
 */

package org.pietschy.command;

import org.pietschy.command.log.Logger;

import java.util.ArrayList;
import java.util.Iterator;

public class
ToggleGroupController
{

   private static final Logger log = CommandManager.getLogger(ToggleGroupController.class);
   private ArrayList commands = new ArrayList();

   private boolean emptySelectionAllowed = false;
   private boolean exclusive = false;

   public ToggleGroupController()
   {
   }

   public void add(ToggleCommand command)
   {
      log.enter("installFace");
      log.param("command", String.valueOf(command));
      if (!commands.contains(command))
      {
         commands.add(command);
      }
      log.exit("installFace()");
   }

   public void remove(ToggleCommand command)
   {
      log.enter("remove");
      log.param("command", String.valueOf(command));
      commands.remove(command);
      log.exit("remove()");
   }

   /**
    * Checks if this groups allows empty selection.
    *
    * @return <tt>true</tt> if the group allows empty selection, <tt>false</tt> if there must
    *         always be at least on selected command.
    */
   public boolean
   isEmptySelectionAllowed()
   {
      return emptySelectionAllowed;
   }

   /**
    * Sets if this groups allows empty selection.
    *
    * @param emptySelectionAllowed <tt>true</tt> if the group should allow empty
    *                              selection, <tt>false</tt> if there must always be at least on selected command.
    */
   public void
   setEmptySelectionAllowed(boolean emptySelectionAllowed)
   {
      this.emptySelectionAllowed = emptySelectionAllowed;
   }

   public boolean
   isExclusive()
   {
      return exclusive;
   }

   public void
   setExclusive(boolean exclusive)
   {
      this.exclusive = exclusive;
   }

   /**
    * Handles a particular selection request on a particular command.
    *
    * @param toggleCommand the command that has been selected
    * @param selected      the new state of the command.
    */
   public void handleSelectionRequest(ToggleCommand toggleCommand, boolean selected)
   {
      log.enter("requestSelection");
      log.param("toggleCommand", String.valueOf(toggleCommand.getId()));
      log.param("selected", String.valueOf(selected));

      if (exclusive)
      {
         handleExclusive(selected, toggleCommand);
      }
      else
      {
         try
         {
            toggleCommand.attemptSelection(selected);
         }
         catch (ToggleVetoException e)
         {
            log.info("Selection request Vetoed", e);
         }
      }
   }


   private void
   handleExclusive(boolean selected, ToggleCommand toggleCommand)
   {
      // we ignore deselections at the current time, I may make this
      // configurable in a later release..
      if (selected)
      {
         ToggleCommand originalSelection = null;

         for (Iterator iterator = commands.iterator(); iterator.hasNext();)
         {
            ToggleCommand command = (ToggleCommand) iterator.next();
            if (command.isSelected())
            {
               // this may fail if the command rejects the selection.
               originalSelection = command;
               break;
            }
         }

         log.param("selectedCommand", String.valueOf(originalSelection));

         // fullfill the request if there was no previous selection or if
         // the previous selection has deselected itself.
         if (originalSelection == null)
         {
            try
            {
               toggleCommand.attemptSelection(true);
            }
            catch (ToggleVetoException e)
            {
               log.info("Selection request Vetoed", e);
            }
         }
         else
         {
            // ask the command to deselect...  (passing in the original invoker)..
            Object oldInvoker = originalSelection.getHint(ActionCommand.HINT_INVOKER);
            originalSelection.putHint(ActionCommand.HINT_INVOKER, toggleCommand.getInvoker());

            boolean selectionVetoed = false;
            try
            {
               // update the new selection appropriately
               originalSelection.attemptSelection(false);

               try
               {
                  toggleCommand.attemptSelection(!originalSelection.isSelected());
               }
               catch (ToggleVetoException e)
               {
                  log.info(e);
                  try
                  {
                     originalSelection.attemptSelection(true);
                  }
                  catch (ToggleVetoException e2)
                  {
                     log.info(e2);
                  }
               }
            }
            catch (ToggleVetoException e)
            {
               selectionVetoed = true;
               log.info(e);
            }

            // restore the invoker...
            originalSelection.putHint(ActionCommand.HINT_INVOKER, oldInvoker);

         }
         log.exit("requestSelection()");
      }
      else // its a deselection..
      {
         if (emptySelectionAllowed)
         {
            try
            {
               toggleCommand.attemptSelection(false);
            }
            catch (ToggleVetoException e)
            {
               log.info(e);
            }
         }
         else
         {
            // make sure all the buttons are left in the correct state..
            toggleCommand.applySelection(true);
         }
      }
   }

}
