/* libinklevel.c
 *
 * (c) 2003, 2004, 2005, 2006 Markus Heinz
 *
 * This software is licensed under the terms of the GPL.
 * For details see file COPYING.
 */

#include "inklevel.h"

#include <string.h>

extern int get_device_id(int port, int portnumber, char *device_id);
extern int parse_device_id_new_hp(char tags[NR_TAGS][BUFLEN], int n, 
				  struct ink_level *level);
extern int parse_device_id_old_hp(char tags[NR_TAGS][BUFLEN], int n, 
				  struct ink_level *level);
/* extern int get_ink_level_epson(int port, int portnumber, int type,  */
/*                                struct ink_level *level); */
extern int get_ink_level_epson(int port, int portnumber, 
                               struct ink_level *level);
extern int get_ink_level_canon(int port, int portnumber, 
                               struct ink_level *level);

int get_ink_level(int port, int portnumber, struct ink_level *level);
int parse_device_id(int port, int portnumber, const char *device_id, 
                    struct ink_level *level);

int get_ink_level(int port, int portnumber, struct ink_level *level) {
  char device_id[BUFLEN];
  int ret;

  memset(level->model, 0, MODEL_NAME_LENGTH);
  memset(level->levels, 0, MAX_CARTRIDGE_TYPES * sizeof(unsigned short) * 2);
  level->status = RESPONSE_INVALID;

  if ((ret = get_device_id(port, portnumber, device_id)) == OK) {
    if ((ret = parse_device_id(port, portnumber, device_id, level)) == OK) {
      return OK;
    }
  }

  return ret;
}

/* This function parses the device id and calls the appropiate function */

int parse_device_id(int port, int portnumber, const char *device_id, 
                    struct ink_level *level) {
  int i = 0;
  int j = 0;
  int tag_mfg = -1;
  const char *c;
  char tags[NR_TAGS][BUFLEN];  

  /* init tags */

  memset(tags, 0, NR_TAGS*BUFLEN);

  /* Tokenize the device id */

  i = 0;
  j = 0;
  c = device_id;
  
  while ((*c != '\0') && (i < NR_TAGS)) {
    j = 0;
    while ((*c != '\0') && (*c != ';') && (j < BUFLEN)) {
      tags[i][j] = *c;
      c++;
      j++;
    }
    if (*c == ';') { /* Some printers do not terminate the last tag with ';' */
      c++; /* Skip the ';' */
    }
    i++;
  }

#ifdef DEBUG
  /* print all tags */

  for (i = 0; i < NR_TAGS; i++) {
    printf("%d: %s\n", i, tags[i]);
  }
#endif

  /* Check if we deal with a printer */

  /* Find the "CLS:" tag */

  for (i = 0; i < NR_TAGS; i++) {
    c = tags[i];
    if ((c[0] == 'C') && (c[1] == 'L') && (c[2] == 'S') && (c[3] == ':')) {
      break;
    }
  }

  if (i < NR_TAGS) {

    /* Check if it is "PRINTER" */

    if ((tags[i][4] != 'P') || (tags[i][5] != 'R') || (tags[i][6] != 'I') || 
	(tags[i][7] != 'N') || (tags[i][8] != 'T') || (tags[i][9] != 'E') || 
	(tags[i][10] != 'R')) {
      
#ifdef DEBUG
      printf("No printer found\n");
#endif
      
      return NO_PRINTER_FOUND;
    }
  } else { 
    /* Find the "CLASS:" tag */
    
    for (i = 0; i < NR_TAGS; i++) {
      c = tags[i];
      if ((c[0] == 'C') && (c[1] == 'L') && (c[2] == 'A') && (c[3] == 'S') && 
	  (c[4] == 'S') && (c[5] == ':')) {
	break;
      }
    }

    if (i < NR_TAGS) {
    
      /* Check if it is "PRINTER" */
      
      if ((tags[i][6] != 'P') || (tags[i][7] != 'R') || (tags[i][8] != 'I') || 
	  (tags[i][9] != 'N') || (tags[i][10] != 'T') || 
	  (tags[i][11] != 'E') || (tags[i][12] != 'R')) {
      
#ifdef DEBUG
	printf("No printer found\n");
#endif
	
	return NO_PRINTER_FOUND;
      }
    } else {

#ifdef DEBUG 
      printf("No device class found\n");
#endif

      return NO_DEVICE_CLASS_FOUND;
    }
  }

  /* Insert the name of the printer */

  /* Find the "MFG:" tag */

  for (i = 0; i < NR_TAGS; i++) {
    c = tags[i];
    if ((c[0] == 'M') && (c[1] == 'F') && (c[2] == 'G') && (c[3] == ':')) {
      break;
    }
  }

#ifdef DEBUG
  if (i < NR_TAGS) {
    printf("The \"MFG:\" tag has number %d\n", i);
  } else {
    printf("No \"MFG:\" tag found\n");
  }
#endif

  /* Null-terminate in case of a too large name */
  level->model[MODEL_NAME_LENGTH-1] = '\0';

  if (i < NR_TAGS) {
    tag_mfg = i;
    strncpy(level->model, c + 4, MODEL_NAME_LENGTH-1);
  }

  j = 0;
  while (level->model[j] != '\0') {
    j++;
  }

  /* Find the "MDL:" tag */

  for (i = 0; i < NR_TAGS; i++) {
    c = tags[i];
    if ((c[0] == 'M') && (c[1] == 'D') && (c[2] == 'L') && (c[3] == ':')) {
      break;
    }
  }

#ifdef DEBUG
  if(i < NR_TAGS) {
    printf("The \"MDL:\" tag has number %d\n", i);
  } else {
    printf("No \"MDL:\" tag found\n");
  }
#endif

  if (i < NR_TAGS) {
    level->model[j] = ' ';
    j++;
    strncpy(level->model + j, c + 4, MODEL_NAME_LENGTH-j-1);
  }

  /* Check for a new HP printer */

  /* Find the "S:" tag */

  for (i = 0; i < NR_TAGS; i++) {
    c = tags[i];
    if ((c[0] == 'S') && (c[1] == ':')) {
      break;
    }
  }

#ifdef DEBUG
  if (i < NR_TAGS) {
    printf("The \"S:\" tag has number %d\n", i);
  } else {
    printf("No \"S:\" tag found\n");
  }
#endif

  if (i<NR_TAGS) {
    return parse_device_id_new_hp(tags, i, level);
  }
  
  /* Check for an old HP printer */

  /* Find the "VSTATUS:" tag */

  for (i = 0; i < NR_TAGS; i++) {
    c = tags[i];
    if ((c[0] == 'V') && (c[1] == 'S') && (c[2] == 'T') && (c[3] == 'A') &&
	(c[4] == 'T') && (c[5] == 'U') && (c[6] == 'S') && (c[7] ==':')) {
      break;
    }
  }

#ifdef DEBUG
  if (i < NR_TAGS) {
    printf("The \"VSTATUS:\" tag has number %d\n", i);
  } else {
    printf("No \"VSTATUS:\" tag found\n");
  }
#endif

  if (i<NR_TAGS) {
    return parse_device_id_old_hp(tags, i, level);
  }

  /* Ckeck for a Epson Printer */

  if (tag_mfg >= 0) {
    /* Ckeck if it is "EPSON" */
    c = tags[tag_mfg];
    if ((c[4] == 'E') && (c[5] == 'P') && (c[6] == 'S') && (c[7] == 'O') &&
	(c[8] == 'N')) {

/*       /\* Check if we deal with an old or a new Epson printer *\/ */
/*       /\* A new Epson printer is characterized by "D4" in the "CMD:" tag *\/ */
/*       /\* I do not know if this is a viable means of distinction *\/ */

/*       /\* Find the "CMD:" tag *\/ */

/*       for (i = 0; i < NR_TAGS; i++) { */
/* 	c = tags[i]; */
/* 	if ((c[0] == 'C') && (c[1] == 'M') && (c[2] == 'D') && (c[3] == ':')) { */
/* 	  break; */
/* 	} */
/*       } */

/* #ifdef DEBUG */
/*       if (i < NR_TAGS) { */
/* 	printf("The \"CMD:\" tag has number %d\n", i); */
/*       } else { */
/* 	printf("No \"CMD:\" tag found\n"); */
/*       } */
/* #endif */

/*       if (i == NR_TAGS) { */
/* 	return NO_CMD_TAG_FOUND; */
/*       } */

/*       while (c[0] != '\0') { */
/* 	if ((c[0] == 'D') && (c[1] == '4')) { */
/* 	  break; */
/* 	} */
/* 	c++; */
/*       } */

/*       if (c[0] == '\0') { /\* old Epson printer *\/ */

/* #ifdef DEBUG */
/* 	printf("Old Epson printer detected\n"); */
/* #endif */

/* 	return get_ink_level_epson(port, portnumber, OLD, level); */
/*       } else { /\* new Epson printer *\/ */

/* #ifdef DEBUG */
/* 	printf("New Epson printer detected\n"); */
/* #endif */

/* 	return get_ink_level_epson(port, portnumber, NEW, level); */
/*       } */

      return get_ink_level_epson(port, portnumber, level);
    }
  }
  
  /* Ckeck for a Canon Printer */

  if (tag_mfg >= 0) {
    /* Ckeck if it is "Canon" */
    c = tags[tag_mfg];
    if ((c[4] == 'C') && (c[5] == 'a') && (c[6] == 'n') && (c[7] == 'o') &&
	(c[8] == 'n')) {

      return get_ink_level_canon(port, portnumber, level);
    }
  }
  
  /* Insert code to check for other printers here */
  
  return PRINTER_NOT_SUPPORTED; /* No matching printer was found */
}
