/* fixedDepthSearcher.tcc
 */
#ifndef OSL_CHECKMATE_FIXED_DEPTH_SERCHER_TCC
#define OSL_CHECKMATE_FIXED_DEPTH_SERCHER_TCC
#include "osl/checkmate/fixedDepthSearcher.h"
#include "osl/checkmate/immediateCheckmate.h"
#include "osl/checkmate/proofPieces.h"
#include "osl/checkmate/proofNumberTable.h"
#include "osl/state/numEffectState.h"
#include "osl/container/moveVector.h"
#include "osl/move_action/store.h"
#include "osl/move_action/count.h"
#include "osl/move_generator/addEffectWithEffect.h"
#include "osl/move_generator/escape_.h"
#include "osl/move_classifier/check_.h"
#include "osl/effect_util/effectUtil.h"
#include "osl/apply_move/applyMove.h"
#include "osl/neighboring8.h"
#include "osl/stat/ratio.h"
#include <boost/foreach.hpp>

namespace osl
{
  namespace checkmate
  {
    template<Player P, bool SetPieces>
    struct FixedAttackHelper{
      FixedDepthSearcher &searcher;
      Move move;
      int depth;
      ProofDisproof& pdp;
      PieceStand& pieces;
      FixedAttackHelper(FixedDepthSearcher &s,int d,ProofDisproof& p,
			PieceStand& pi)
	: searcher(s), depth(d), pdp(p), pieces(pi)
      {
      }
      void operator()(Position)
      {
	assert(move.isNormal());
	pdp=searcher.defense<P,SetPieces>(move,depth-1,pieces);
      }
    };
   /**
    * Pは動かす側ではなく攻撃側
    */
    template<Player P, bool SetPieces, bool MayUnsafe=false>
    struct FixedDefenseHelper{
      FixedDepthSearcher &searcher;
      int depth;
      ProofDisproof& pdp;
      PieceStand& pieces;
      Move best_move;
      FixedDefenseHelper(FixedDepthSearcher &s,int d,ProofDisproof& p,
			 PieceStand& pi) 
	: searcher(s), depth(d), pdp(p), pieces(pi)
      {
      }
      void operator()(Position)
      {
	if (MayUnsafe)
	  pdp=searcher.attackMayUnsafe<P,SetPieces,false>(depth-1, best_move, pieces);
	else
	  pdp=searcher.attack<P,SetPieces,false>(depth-1, best_move, pieces);
      }
    };
  }
}

template <osl::Player P, bool SetPieces, bool HasGuide>
const osl::checkmate::ProofDisproof 
osl::checkmate::FixedDepthSearcher::
attackMayUnsafe(int depth, Move& best_move, PieceStand& proof_pieces)
{
  assert(state->getTurn() == P);
  const Position target_king
    = state->template getKingPosition<PlayerTraits<P>::opponent>();
  if (state->hasEffectBy<P>(target_king))
    return ProofDisproof::NoEscape();
  return attack<P,SetPieces,HasGuide>(depth, best_move, proof_pieces);
}

template <osl::Player P, bool SetPieces, bool HasGuide>
const osl::checkmate::ProofDisproof 
osl::checkmate::FixedDepthSearcher::
attack(int depth, Move& best_move, PieceStand& proof_pieces)
{
  assert(state->getTurn() == P);
  assert ((! HasGuide)
	  || (state->isAlmostValidMove(best_move)
	      && move_classifier::
	      Check<P>::isMember(*state, best_move.ptype(), best_move.from(), 
				 best_move.to())));
  addCount();
  const Position target_king
    = state->template getKingPosition<PlayerTraits<P>::opponent>();
  assert(! state->hasEffectBy<P>(target_king));
  const King8Info info(state->Iking8Info(alt(P)));
  if ((! state->inCheck())
      && ImmediateCheckmate::hasCheckmateMove<P>(*state, info, target_king,
						 best_move))
  {
    if (SetPieces)
    {
      proof_pieces = PieceStand();
      if (best_move.isDrop())
	proof_pieces.add(best_move.ptype());
    }
    return ProofDisproof::Checkmate();
  }
  if (depth <= 0) 
  {
    const King8Info info_modified 
      = Edge_Table.resetEdgeFromLiberty(alt(P), target_king, info);
    return Proof_Number_Table.attackEstimation(*state, P, info_modified, target_king);
  }

  ProofDisproof pdp;
  typedef FixedAttackHelper<P,SetPieces> helper_t;
  helper_t helper(*this,depth,pdp,proof_pieces);
  int minProof = ProofDisproof::PROOF_MAX;
  int sumDisproof=0;
  if (HasGuide)
  {
    helper.move=best_move;
    ApplyMove<P>::doUndoMove(*state,best_move,helper);
    if (pdp.isCheckmateSuccess())
    {
      if (SetPieces)
	proof_pieces = ProofPieces::attack(proof_pieces, best_move, stand(P));
      return ProofDisproof::Checkmate();
    }
    minProof = pdp.proof();
    sumDisproof += pdp.disproof();
  }
  
  const Position targetKing
    = state->template getKingPosition<PlayerTraits<P>::opponent>();
  MoveVector moves;
  bool has_pawn_checkmate=false;
  move_generator::GenerateAddEffectWithEffect::generate<true>
    (P,*state,targetKing,moves,has_pawn_checkmate);

  if (moves.size()==0){
    if(has_pawn_checkmate)
      return ProofDisproof::PawnCheckmate();
    else
      return ProofDisproof::NoCheckmate();
  }
  if(has_pawn_checkmate)
    minProof=std::min(minProof,(int)ProofDisproof::PAWN_CHECK_MATE_PROOF);
  BOOST_FOREACH(Move move, moves) {
    if (HasGuide && move == best_move)
      continue;
    helper.move=move;
    ApplyMove<P>::doUndoMove(*state,move,helper);
    int proof=pdp.proof();
    if (proof<minProof){
      if (proof==0){
	best_move=move;
	if (SetPieces)
	{
	  proof_pieces = ProofPieces::attack(proof_pieces, best_move, stand(P));
	}
	return ProofDisproof::Checkmate();
      }
      minProof=proof;
    }
    sumDisproof+=pdp.disproof();
  }
  // depth >= 3 では PawnCheckmateの際にunpromoteを試す必要あり
  return ProofDisproof(minProof,sumDisproof);
}

template <osl::Player P, bool SetPieces>
inline
const osl::checkmate::ProofDisproof 
osl::checkmate::FixedDepthSearcher::
defenseEstimation(Move last_move, PieceStand& proof_pieces,
		  Piece attacker_piece, Position target_position) const
{
  assert(state->getTurn() == alt(P));
  const Player Opponent = PlayerTraits<P>::opponent;
  int count=King8Info(state->Iking8Info(Opponent)).libertyCount();
  // multiple checkなので，pawn dropではない
  if (attacker_piece.isEmpty())
  {
    if (count>0){
      return ProofDisproof(count,1);
    }
    return ProofDisproof::NoEscape();
  }
  const Position attack_from=attacker_piece.position();
  count += state->countEffect(alt(P), attack_from);
  if (Neighboring8::isNeighboring8(attack_from, target_position))
    --count;
  const EffectContent effect
    = Ptype_Table.getEffect(attacker_piece.ptypeO(), 
			    attack_from, target_position);
  if (! effect.hasUnblockableEffect())
  {
    // this is better approximation than naive enumeration of blocking moves,
    // for counting of disproof number in df-pn,
    ++count;
  }

  if (count==0){
    if (last_move.isValid() && last_move.isDrop() && last_move.ptype()==PAWN)
      return ProofDisproof::PawnCheckmate();
    if (SetPieces)
    {
      proof_pieces = ProofPieces::leaf(*state, P, stand(P));
    }
    return ProofDisproof::NoEscape();
  }
  return ProofDisproof(count, 1);
}

template <osl::Player Defense>
void osl::checkmate::FixedDepthSearcher::
generateBlockingWhenLiberty0(Piece defense_king, Position attack_from,
			     MoveVector& moves) const
{
  assert(state->getKingPiece(Defense) == defense_king);
  using namespace move_generator;
  using namespace move_action;
  MoveVector all_moves;
  {
    Store store(all_moves);
    Escape<Store>::
      generateBlockingKing<Defense,false>(*state, defense_king, attack_from,store);
  }

  BOOST_FOREACH(Move move, all_moves)
  {
    if (move.isDrop())
    {
      if (! state->hasEffectBy<Defense>(move.to()))
	continue;
    }
    else
    {
      // move
      if (! Neighboring8::isNeighboring8(move.from(), defense_king.position()))
      {
	if (! state->hasMultipleEffectBy(Defense, move.to()))
	  continue;
      }
    }
    moves.push_back(move);
  }
}

template <osl::Player Defense> inline
int osl::checkmate::FixedDepthSearcher::
blockEstimation(Position /*attack_from*/, Position /*defense_king*/) const
{
  // 利きのあるマスを数えようと思ったら効果がなかった
  return 1;
}

template <osl::Player P, bool SetPieces>
const osl::checkmate::ProofDisproof 
osl::checkmate::FixedDepthSearcher::
defense(Move last_move, int depth, PieceStand& proof_pieces)
{
  assert(state->getTurn() == alt(P));
  addCount();
  const Player Defense = PlayerTraits<P>::opponent;
  const Position attackerKing
    = state->template getKingPosition<P>();
  /**
   * 直前の攻め方の手が自殺手
   */
  if (attackerKing.isOnBoard() && state->hasEffectBy<Defense>(attackerKing))
    return ProofDisproof::NoCheckmate();
  const Piece target_king
    = state->template getKingPiece<Defense>();
  const Position target_position = target_king.position();
  assert(state->hasEffectBy<P>(target_position));
  Piece attacker_piece;
  state->template findCheckPiece<Defense>(attacker_piece);
  if (depth <= 0)
  {
    return defenseEstimation<P, SetPieces>
      (last_move, proof_pieces, attacker_piece, target_position);
  }

  assert(depth > 0);
  MoveVector moves;
  bool blockable_check = false;
  int nonblock_moves;
  {
    using namespace move_generator;
    using namespace move_action;
    if (attacker_piece.isEmpty()) {
      move_action::Store store(moves);
      Escape<Store>::template
	generateEscape<Defense,KING>(*state,target_king,store);
      nonblock_moves = moves.size();
    }
    else {
      const Position attack_from=attacker_piece.position();
      {
	move_action::Store store(moves);
	Escape<Store>::
	  generateCaptureKing<Defense>(*state, target_king, attack_from, store);
      }
      const int num_captures = moves.size();
      {
	move_action::Store store(moves);
	Escape<Store>::template
	  generateEscape<Defense,KING>(*state, target_king, store);
      }
      nonblock_moves = moves.size();
      blockable_check =
	! effect_util::UnblockableCheck::isMember(alt(P), *state);
      if ((depth <= 1) && num_captures && (nonblock_moves > 2))
      {
	if (nonblock_moves > 3)
	{
	  const int block_estimate = blockable_check 
	    ? blockEstimation<Defense>(attack_from, target_position) 
	    : 0;
	  return ProofDisproof(nonblock_moves + block_estimate, 1);
	}
      } 
      if (moves.empty())
	generateBlockingWhenLiberty0<Defense>(target_king, attack_from, moves);
    }
  }
  const size_t initial_moves = moves.size();
  if (moves.empty() && !blockable_check) {
    if (last_move.isValid() && last_move.isDrop() && last_move.ptype()==PAWN)
      return ProofDisproof::PawnCheckmate();
    if (SetPieces)
    {
      proof_pieces = ProofPieces::leaf(*state, P, stand(P));
    }
    return ProofDisproof::NoEscape();
  }
  const bool cut_candidate = (depth <= 1)
    && (nonblock_moves - (state->hasPieceOnStand<GOLD>(P)
			  || state->hasPieceOnStand<SILVER>(P)) > 4);
  if (cut_candidate)
    return ProofDisproof(nonblock_moves, 1);

  typedef FixedDefenseHelper<P,SetPieces> helper_t;
  if (SetPieces)
    proof_pieces = PieceStand();
  PieceStand child_proof;
  ProofDisproof pdp;
  helper_t helper(*this, depth, pdp, child_proof);
  int minDisproof = ProofDisproof::DISPROOF_MAX;
  int sumProof = 0;
  size_t i=0;
start_examine:
  for (;i<moves.size();i++){
    ApplyMove<PlayerTraits<P>::opponent>::doUndoMove(*state,moves[i],helper);
    const int disproof=pdp.disproof();
    if (disproof<minDisproof){
      if (disproof==0)
      {
	return pdp;		// maybe PawnCheckmate
      }
      minDisproof=disproof;
    }
    sumProof+=pdp.proof();
    if (sumProof == 0)
    {
      if (SetPieces)
	proof_pieces = proof_pieces.max(child_proof);
    }
    else
    {
      if (i+1 < moves.size())
      {
	minDisproof = 1;
	if ((int)i < nonblock_moves)
	  sumProof += nonblock_moves - (i+1);
	if (blockable_check)
	  ++sumProof;
      }
      return ProofDisproof(sumProof,minDisproof);
    }
  }
  if ((sumProof == 0) && blockable_check 
      && (moves.size() == initial_moves))
  {
    using namespace move_generator;
    using namespace move_action;
    const Position attack_from=attacker_piece.position();
    {
      move_action::Store store(moves);
      Escape<Store>::
	generateBlockingKing<Defense,false>(*state, target_king, attack_from,store);
    }
    if ((int)moves.size() > nonblock_moves)
      goto start_examine;
    if (moves.empty()) {
      assert(! (last_move.isValid() && last_move.isDrop() && last_move.ptype()==PAWN));
      if (SetPieces)
	proof_pieces = ProofPieces::leaf(*state, P, stand(P));
      return ProofDisproof::NoEscape();
    }
  }
  
  if (SetPieces && (sumProof == 0) && blockable_check)
  {
    ProofPiecesUtil::addMonopolizedPieces(*state, P, stand(P), proof_pieces);
  }

  // depth >= 2 では unprmote も試す必要あり
  return ProofDisproof(sumProof,minDisproof);
}

template <osl::Player P>
const osl::checkmate::ProofDisproof 
osl::checkmate::FixedDepthSearcher::
hasEscapeByMove(Move next_move, int depth, Move& check_move,
		PieceStand& proof_pieces)
{
  typedef FixedDefenseHelper<P,true,true> helper_t;
  proof_pieces = PieceStand();
  ProofDisproof pdp;
  helper_t helper(*this, depth+1, pdp, proof_pieces);
  ApplyMove<PlayerTraits<P>::opponent>::doUndoMove(*state,next_move,helper);
  check_move = helper.best_move;
  return pdp;
}

template <osl::Player P>
const osl::checkmate::ProofDisproof 
osl::checkmate::FixedDepthSearcher::
hasEscapeByMove(Move next_move, int depth)
{
  typedef FixedDefenseHelper<P,false,true> helper_t;
  PieceStand proof_pieces;
  ProofDisproof pdp;
  helper_t helper(*this, depth+1, pdp, proof_pieces);
  ApplyMove<PlayerTraits<P>::opponent>::doUndoMove(*state,next_move,helper);
  return pdp;
}

template <osl::Player P>
const osl::checkmate::ProofDisproof 
osl::checkmate::FixedDepthSearcher::
hasCheckmateWithGuide(int depth, Move& guide, PieceStand& proof_pieces)
{
  assert(guide.isNormal());
  if (! guide.isDrop())
  {
    const Piece p=state->getPieceOnBoard(guide.to());
    if (!p.isPtype<KING>())
      guide=guide.newCapture(p);
  }
  if (state->template isAlmostValidMove<false>(guide)
      && move_classifier::Check<P>
      ::isMember(*state, guide.ptype(), guide.from(), guide.to()))
    return attack<P,true,true>(depth, guide, proof_pieces);
  return attack<P,true,false>(depth, guide, proof_pieces);
}

#endif /* OSL_CHECKMATE_FIXED_DEPTH_SERCHER_TCC */
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
