/* queue.h
 */
#ifndef OSL_POINTERQUEUE_H
#define OSL_POINTERQUEUE_H

#include <deque>
#include <boost/thread/thread.hpp>
#include <boost/thread/condition.hpp>

namespace osl
{
  namespace misc
  {
    template <class T>
    class PointerQueue
    {
      typedef std::deque<boost::shared_ptr<T> > queue_t;
      queue_t data;
      typedef boost::mutex Mutex;
      mutable Mutex mutex;
      volatile bool finish;
      boost::condition condition;
    public:
      PointerQueue() : finish(false)
      {
      }
      ~PointerQueue()
      {
	if (! finish)
	  quit();
      }
      size_t size() const
      {
	Mutex::scoped_lock lk(mutex);
	return data.size();
      }
      void push_back(boost::shared_ptr<T>& ptr)
      {
	Mutex::scoped_lock lk(mutex);
	data.push_back(boost::shared_ptr<T>());
	data.back().swap(ptr);
	condition.notify_one();
      }
    private:
      boost::shared_ptr<T> pop_front_in_lock()
      {
	if (! data.empty())
	{
	  boost::shared_ptr<T> result = data.front();
	  data.pop_front();
	  return result;
	}
	return boost::shared_ptr<T>();
      }
    public:
      boost::shared_ptr<T> pop_front_non_block()
      {
	Mutex::scoped_lock lk(mutex);
	return pop_front_in_lock();
      }
      boost::shared_ptr<T> pop_front()
      {
	while (! finish)
	{
	  Mutex::scoped_lock lk(mutex);
	  boost::shared_ptr<T> result = pop_front_in_lock();
	  if (result.get() || finish)
	    return result;
	  condition.wait(lk);
	}
	return boost::shared_ptr<T>();
      }
      void quit(int seconds=0)
      {
	finish = true;
	if (seconds > 0)
	  boost::this_thread::sleep(boost::posix_time::seconds(seconds));
	Mutex::scoped_lock lk(mutex);
	condition.notify_one();
      }
    };
  }
}

#endif /* OSL_POINTERQUEUE_H */
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
