/* quiescenceSearch.t.cc
 */
// #define QSEARCH_DEBUG
#include "osl/search/quiescenceSearch2.h"
#include "osl/search/quiescenceSearch2.tcc"
#include "osl/search/simpleHashTable.h"
#include "osl/game_playing/historyToTable.h"
#include "osl/game_playing/gameState.h"
#include "osl/state/numEffectState.h"
#include "osl/record/csaString.h"
#include "osl/record/record.h"
#include "osl/record/csaRecord.h"
#include "osl/misc/milliSeconds.h"
#include "osl/oslConfig.h"

#include <cppunit/TestCase.h>
#include <cppunit/extensions/HelperMacros.h>

#include <fstream>
#include <iostream>

class QuiescenceSearchTest : public CppUnit::TestFixture 
{
  CPPUNIT_TEST_SUITE(QuiescenceSearchTest);
  CPPUNIT_TEST(test050313);
  CPPUNIT_TEST(testLose2);
  CPPUNIT_TEST(testLose);
  CPPUNIT_TEST(testForcedEscape);
  CPPUNIT_TEST(testNoCapture);
  CPPUNIT_TEST(testCaptureBlack);
  CPPUNIT_TEST(testCaptureWhite);
  CPPUNIT_TEST(testCaptureCheck);
  CPPUNIT_TEST(testCaptureCheckMate);
  CPPUNIT_TEST(testTakeBack);
  CPPUNIT_TEST(testThreat);
  CPPUNIT_TEST(testKnightAttack);
  CPPUNIT_TEST(testEquality);
  CPPUNIT_TEST(testCompare);
  CPPUNIT_TEST(testFocalPawn);
  CPPUNIT_TEST_SUITE_END();

  void testIsEvalValueConsistent(std::string const& fileName);
public:
  QuiescenceSearchTest()
  {
  }
  void test050313();
  void testLose2();
  void testLose();
  void testForcedEscape();
  void testNoCapture();
  void testCaptureBlack();
  void testCaptureWhite();
  void testCaptureCheck();
  void testCaptureCheckMate();
  /** 
   * 表を使うかどうかで同じ結果が出ることを確認する 
   */
  void testEquality();
  void testTakeBack();
  void testThreat();
  void testKnightAttack();

  bool betterThan(const char *l, const char *r);
  void testCompare();
  void testFocalPawn();
};

CPPUNIT_TEST_SUITE_REGISTRATION(QuiescenceSearchTest);

using namespace osl;
using namespace osl::search;
using namespace osl::game_playing;
using namespace osl::eval;
extern bool isShortTest;

const int Gold = PtypeEvalTraits<GOLD>::val;
const int Silver = PtypeEvalTraits<SILVER>::val;
const int Psilver = PtypeEvalTraits<PSILVER>::val;
const int Rook = PtypeEvalTraits<ROOK>::val;
const int Knight = PtypeEvalTraits<KNIGHT>::val;
const int Pawn = PtypeEvalTraits<PAWN>::val;
const int Pknight = PtypeEvalTraits<PKNIGHT>::val;
const int Bishop = PtypeEvalTraits<BISHOP>::val;
const int Pbishop = PtypeEvalTraits<PBISHOP>::val;

typedef NumEffectState state_t;
typedef QuiescenceSearch2<PieceEval> qsearch_t;

void QuiescenceSearchTest::test050313()
{
  osl::search::SearchState2::checkmate_t checkmate_searcher;
// 81RY 82FU PASS は68RY で詰ですよん
//P1-KY-KE *  *  *  * +RY * -KY
//P2 *  *  *  * +UM * +NK *  * 
//P3-FU-OU-GI-FU-FU-FU *  * -FU
//P4 *  * -FU *  *  *  *  *  * 
//P5 *  *  *  * +KA *  *  *  * 
//P6 *  *  *  *  *  *  *  *  * 
//P7+FU * +FU+FU+FU+FU+FU * +FU
//P8 *  * -NK * +OU *  *  *  * 
//P9+KY+KE * -HI * +KI+GI * +KY
//P+00KI00FU00FU
//P-00KI00KI00GI00GI00FU00FU00FU
//+

  // 王手を入れるとループがある問題
  {
    state_t state(CsaString(
		    "P1-KY+HI *  *  * -OU * -KE-KY\n"
		    "P2 *  *  *  *  *  * -KI *  * \n"
		    "P3 *  * -KE * -KI-FU * -FU * \n"
		    "P4-FU * +KI *  *  * -FU * -FU\n"
		    "P5 *  *  * -KE-GI+GI *  *  * \n"
		    "P6 *  *  *  *  *  * +FU * +FU\n"
		    "P7+FU+FU+KI * +KA+FU *  *  * \n"
		    "P8+KA *  *  *  *  *  * +HI * \n"
		    "P9+KY * +OU *  *  *  * +KE+KY\n"
		    "P+00FU00FU00FU00FU00FU00FU\n"
		    "P-00GI00GI00FU00FU\n"
		    "-\n").getInitialState());
    SearchState2Core sstate(state, checkmate_searcher);
    SimpleHashTable table(10000,-6);
    qsearch_t searcher(sstate, table);
    PieceEval ev(state);
    const int result = searcher.search(WHITE, ev, 
				       Move(Position(8,1),ROOK,BLACK));
    CPPUNIT_ASSERT(result < 1000000);
  }
  {
    state_t state(CsaString(
		    "P1 *  *  *  *  *  *  * -KE-KY\n"
		    "P2 * +RY *  *  *  * -KI-OU * \n"
		    "P3 *  *  *  *  * +FU-GI-KI * \n"
		    "P4-FU *  * -FU-FU *  *  * -FU\n"
		    "P5 * -UM *  *  * -FU-FU *  * \n"
		    "P6+FU+KI+FU+FU+FU *  *  * +FU\n"
		    "P7+OU *  *  * +GI *  *  *  * \n"
		    "P8 *  *  *  *  *  *  * -RY * \n"
		    "P9+KY+KE-GI *  *  *  *  *  * \n"
		    "P+00GI00FU00FU00FU00FU00FU\n"
		    "P-00KA00KI00KE00KE00KY00KY00FU\n"
		    "-\n").getInitialState());
    SearchState2Core sstate(state, checkmate_searcher);
    SimpleHashTable table(10000,-6);
    qsearch_t searcher(sstate, table);
    PieceEval ev(state);
    const int result = searcher.search(WHITE, ev, 
				       Move(Position(8,6),GOLD,BLACK));
    CPPUNIT_ASSERT(result < -5000000);
    CPPUNIT_ASSERT(searcher.nodeCount() < 5000); // 1手詰なんだけど...
  }
  {
    state_t state(CsaString(
		    "P1 *  *  *  *  *  *  * -KE-KY\n"
		    "P2 * +RY *  *  *  * -KI-OU * \n"
		    "P3 *  *  *  *  * +FU-GI-KI * \n"
		    "P4-FU *  * -FU-FU *  *  * -FU\n"
		    "P5 * -UM *  *  * -FU-FU *  * \n"
		    "P6+FU+KI+FU+FU+FU *  *  * +FU\n"
		    "P7+OU *  *  * +GI *  *  *  * \n"
		    "P8 * -GI *  *  *  *  * -RY * \n"
		    "P9+KY+KE+KE *  *  *  *  *  * \n"
		    "P+00GI00FU00FU00FU00FU00FU\n"
		    "P-00KA00KI00KE00KY00KY00FU\n"
		    "+\n").getInitialState());
    SearchState2Core sstate(state, checkmate_searcher);
    SimpleHashTable table(10000,-6);
    qsearch_t searcher(sstate, table);
    PieceEval ev(state);
    const int result = searcher.search(BLACK, ev, 
				       Move(Position(8,8),SILVER,WHITE));
    CPPUNIT_ASSERT(result < 0);
  }
}

void QuiescenceSearchTest::testLose2()
{
  osl::search::SearchState2::checkmate_t checkmate_searcher;
  // -8889TO が先に生成されると，負けを認識できないバグが以前あった
  // -0097FU から作った局面とは駒番号が違う?
  state_t state(CsaString(
		  "P1 *  *  *  *  *  * -KI-KE-OU\n"
		  "P2 *  *  *  *  *  * -KI-GI-KY\n"
		  "P3 *  *  *  *  *  * -FU-FU-FU\n"
		  "P4 *  *  *  *  *  *  *  *  * \n"
		  "P5 *  *  *  *  *  *  *  *  * \n"
		  "P6 *  *  *  *  *  *  *  *  * \n"
		  "P7-FU-GI *  *  *  *  *  *  * \n"
		  "P8 * -FU *  *  *  *  *  *  * \n"
		  "P9+OU *  *  *  *  *  *  *  * \n"
		  "P+00AL\n"
		  "+\n").getInitialState());
  SearchState2Core sstate(state, checkmate_searcher);
  SimpleHashTable table(1000,-6);
  qsearch_t searcher(sstate, table);
  PieceEval ev(state);
  const int result = searcher.search<BLACK>(ev, Move(Position(9,7),PAWN,WHITE));
  CPPUNIT_ASSERT(result < -100000);
}

void QuiescenceSearchTest::testLose()
{
  osl::search::SearchState2::checkmate_t checkmate_searcher;
  state_t state(CsaString(	// 色々とれるが取ると詰，実は必死
			 "P1-KY-KE-GI * -OU-KI-GI-KE-KY\n"
			 "P2 * -HI *  *  *  *  * -KA * \n"
			 "P3-FU+FU-FU-FU+FU-FU-FU+FU-FU\n"
			 "P4 *  *  *  *  *  *  *  *  * \n"
			 "P5 *  *  *  *  *  *  * -FU * \n"
			 "P6 * -FU *  *  * +FU *  *  * \n"
			 "P7+FU * +FU+FU-FU-KI+FU+HI+FU\n"
			 "P8 * +KA+KE * +KE *  * +KI * \n"
			 "P9+KY+KI+GI * +OU * +GI * +KY\n"
			 "+\n").getInitialState());
  SearchState2Core sstate(state, checkmate_searcher);
  SimpleHashTable table(0);
  qsearch_t searcher(sstate, table);
  PieceEval ev(state);
  const int val1 = searcher.search(BLACK, ev, Move(Position(5,7),PAWN,WHITE));
  // TODO: まだ必死は認識できない
  // CPPUNIT_ASSERT_EQUAL(FixedEval::winValue(WHITE), val1);
  // std::cerr << val1 << "\n";
  CPPUNIT_ASSERT(val1 < -1000);
}
void QuiescenceSearchTest::testForcedEscape()
{
  osl::search::SearchState2::checkmate_t checkmate_searcher;
  state_t state(CsaString(	// 色々とれるが取ると詰，桂損確定
			 "P1-KY-KE-GI * -OU-KI-GI-KE-KY\n"
			 "P2 * -HI *  *  *  *  * -KA * \n"
			 "P3-FU+FU-FU-FU+FU-FU-FU+FU-FU\n"
			 "P4 *  *  *  *  *  *  *  *  * \n"
			 "P5 *  *  *  *  *  *  * -FU * \n"
			 "P6 * -FU+KI *  * +FU *  *  * \n"
			 "P7+FU * +FU+FU-FU-KI+FU+HI+FU\n"
			 "P8 * +KA *  * +KE *  * +KI * \n"
			 "P9+KY+KE+GI * +OU * +GI * +KY\n"
			 "+\n").getInitialState());
  SearchState2Core sstate(state, checkmate_searcher);
  SimpleHashTable table(0);
  qsearch_t searcher(sstate, table);
  PieceEval ev(state);
  const int val1 = searcher.search(BLACK, ev, Move(Position(5,7),PAWN,WHITE));
  CPPUNIT_ASSERT(val1 <= -PtypeEvalTraits<KNIGHT>::val*2);
}

void QuiescenceSearchTest::testKnightAttack()
{
  osl::search::SearchState2::checkmate_t checkmate_searcher;
#if 0
  {
    state_t state(CsaString(
		    "P1-KY-KE-GI-KI-OU-KI-GI * -KY\n"
		    "P2 * -HI *  *  *  *  * -KA * \n"
		    "P3-FU-FU-FU-FU-FU-FU-FU-FU-FU\n"
		    "P4 *  *  *  *  *  * -KE *  * \n"
		    "P5 *  *  *  *  *  *  *  *  * \n"
		    "P6 *  *  *  *  *  * +FU *  * \n"
		    "P7+FU+FU+FU+FU+FU+FU * +FU+FU\n"
		    "P8 * +KA *  *  *  *  * +HI * \n"
		    "P9+KY+KE+GI+KI+OU+KI+GI+KE+KY\n"
		    "+\n").getInitialState());
    SearchState2Core sstate(state, checkmate_searcher);
    SimpleHashTable table(0);
    qsearch_t searcher(sstate, table);
    PieceEval ev(state);
    const int val1 = searcher.search(BLACK, ev, Move(Position(3,4),KNIGHT,WHITE));
    CPPUNIT_ASSERT(val1 > 0);
  }
  {
    state_t state(CsaString(
		    "P1-KY-KE-GI-KI-OU-KI-GI-KE-KY\n"
		    "P2 * -HI *  *  *  *  * -KA * \n"
		    "P3-FU-FU-FU-FU-FU-FU * -FU * \n"
		    "P4 *  *  *  *  *  *  *  *  * \n"
		    "P5 *  *  *  *  *  *  *  *  * \n"
		    "P6 *  *  *  *  *  * +KE * -FU\n"
		    "P7+FU * +FU+FU+FU+FU+FU+FU+FU\n"
		    "P8 * +KA *  *  *  *  * +HI * \n"
		    "P9+KY+KE+GI+KI+OU+KI+GI * +KY\n"
		    "P-00FU00FU\n"
		    "+\n").getInitialState());
    SearchState2Core sstate(state, checkmate_searcher);
    SimpleHashTable table(0);
    qsearch_t searcher(sstate, table);
    PieceEval ev(state);
    const int val1 = searcher.search(BLACK, ev, Move(Position(1,6),PAWN,WHITE));
    CPPUNIT_ASSERT(val1 < 0);
  }
#endif
}

void QuiescenceSearchTest::testNoCapture()
{
  osl::search::SearchState2::checkmate_t checkmate_searcher;
  {
    state_t state(CsaString(
		    "P1-KY-KE-GI-KI-OU-KI-GI-KE-KY\n"
		    "P2 * -HI *  *  *  *  * -KA * \n"
		    "P3-FU-FU-FU-FU-FU-FU-FU-FU-FU\n"
		    "P4 *  *  *  *  *  *  *  *  * \n"
		    "P5 *  *  *  *  *  *  *  *  * \n"
		    "P6 *  *  *  *  *  *  *  *  * \n"
		    "P7+FU+FU+FU+FU+FU+FU+FU+FU+FU\n"
		    "P8 * +KA *  *  *  *  * +HI * \n"
		    "P9+KY+KE+GI+KI+OU+KI+GI+KE+KY\n"
		    "+\n").getInitialState());
    SearchState2Core sstate(state, checkmate_searcher);
    SimpleHashTable table(0);
    qsearch_t searcher(sstate, table);
    PieceEval ev(state);
    const int val1 = searcher.search(BLACK, ev, Move(Position(8,2),ROOK,WHITE));
    CPPUNIT_ASSERT_EQUAL(0, val1);
  }
  {
    state_t state(CsaString(
      "P1-KY-KE-GI-KI-OU-KI-GI-KE-KY\n"
      "P2 *  *  *  *  *  *  * -KA * \n"
      "P3-FU-FU-FU-FU-FU-FU-FU-FU-FU\n"
      "P4 *  *  *  *  *  *  *  *  * \n"
      "P5 * -HI *  * +GI *  *  *  * \n"
      "P6 *  *  *  * +FU *  *  *  * \n"
      "P7+FU+FU+FU+FU * +FU+FU+FU+FU\n"
      "P8 *  * +KI *  * +KA * +HI * \n"
      "P9+KY+KE *  * +OU+KI+GI+KE+KY\n"
      "-\n").getInitialState());
    SearchState2Core sstate(state, checkmate_searcher);
    SimpleHashTable table(0);
    qsearch_t searcher(sstate, table);
    PieceEval ev(state);
    const int val1 = searcher.search(WHITE, ev, Move(Position(5,6),PAWN,BLACK));
    CPPUNIT_ASSERT_EQUAL(0, val1);
  }
}

void QuiescenceSearchTest::testCaptureBlack()
{
  osl::search::SearchState2::checkmate_t checkmate_searcher;
  SimpleHashTable table(0);
  {
    state_t state1(CsaString(
			    "P1-KY-KE-GI-KI-OU-KI-GI-KE-KY\n"
			    "P2 *  *  *  *  *  *  * -KA * \n"
			    "P3-FU-FU-FU-FU-FU-FU-FU-FU-FU\n"
			    "P4 *  *  *  *  *  *  *  *  * \n"
			    "P5 *  *  *  *  *  *  *  *  * \n"
			    "P6 * -HI *  *  *  *  *  *  * \n"
			    "P7+FU+FU+FU+FU+FU+FU+FU+FU+FU\n"
			    "P8 * +KA *  *  *  *  * +HI * \n"
			    "P9+KY+KE+GI+KI+OU+KI+GI+KE+KY\n"
			    "+\n").getInitialState());
    SearchState2Core sstate(state1, checkmate_searcher);
    qsearch_t searcher1(sstate, table);
    PieceEval ev(state1);
    const int val1 = searcher1.search(BLACK, ev, Move(Position(8,6),ROOK,WHITE));
    CPPUNIT_ASSERT_EQUAL(PtypeEvalTraits<ROOK>::val*2, val1);
  }
  {    
    // 飛車より角を取るのが得
    state_t state2(CsaString(
			    "P1-KY-KE * -KI-OU-KI-GI-KE-KY\n"
			    "P2 * -GI *  *  *  *  *  *  * \n"
			    "P3-FU * -FU-FU-FU-FU-FU-FU-FU\n"
			    "P4 * +FU *  *  *  *  *  *  * \n"
			    "P5 * -FU *  *  *  *  *  *  * \n"
			    "P6 * -HI *  *  *  *  * -KA * \n"
			    "P7+FU+HI+FU+FU+FU+FU+GI+FU+FU\n"
			    "P8 * +KA+GI *  *  * +FU *  * \n"
			    "P9+KY+KE * +KI+OU * +KI+KE+KY\n"
			    "+\n").getInitialState());
    SearchState2Core sstate(state2, checkmate_searcher);
    qsearch_t searcher2(sstate, table);
    PieceEval ev2(state2);
    const int val2 = searcher2.search(BLACK, ev2, 
				      Move(Position(8,6),ROOK,WHITE));
    CPPUNIT_ASSERT_EQUAL(PtypeEvalTraits<BISHOP>::val*2, val2); //  + 脅威
  }
}

void QuiescenceSearchTest::testCaptureWhite()
{
  osl::search::SearchState2::checkmate_t checkmate_searcher;
  SimpleHashTable table(0);
  {
    state_t state1(CsaString(
			    "P1-KY-KE-GI-KI-OU-KI-GI-KE-KY\n"
			    "P2 *  *  *  *  *  *  * -KA * \n"
			    "P3-FU-FU-FU-FU-FU-FU-FU-FU-FU\n"
			    "P4 *  *  *  *  *  *  *  *  * \n"
			    "P5 *  *  *  *  *  *  *  *  * \n"
			    "P6 * -HI *  *  *  *  *  *  * \n"
			    "P7+FU+FU+FU+FU+FU+FU+FU+FU+FU\n"
			    "P8 * +KA *  *  *  * +HI+GI+KI\n"
			    "P9+KY+KE *  * +OU+KI+GI+KE+KY\n"
			    "-\n").getInitialState());
    SearchState2Core sstate(state1, checkmate_searcher);
    qsearch_t searcher1(sstate, table);
    PieceEval ev(state1);
    const int val1 = searcher1.search(WHITE, ev, Move(Position(8,7),PAWN,BLACK));
    CPPUNIT_ASSERT_EQUAL(-PtypeEvalTraits<PROOK>::val
			 -PtypeEvalTraits<PAWN>::val*2
			 +PtypeEvalTraits<ROOK>::val,
			 val1);
  }
  {    
    // 飛車を先にとれば駒損なし
    state_t state2(CsaString(
			    "P1-KY-KE * -KI-OU-KI-GI-KE-KY\n"
			    "P2 * -GI *  *  *  *  *  *  * \n"
			    "P3-FU * -FU-FU-FU-FU-FU-FU-FU\n"
			    "P4 * +FU *  *  *  *  *  *  * \n"
			    "P5 * -FU *  *  *  *  *  * -KA\n"
			    "P6 * -HI+FU *  *  *  *  *  * \n"
			    "P7+FU+HI+KA+FU+FU+FU+GI+FU+FU\n"
			    "P8 *  * +GI * +OU * +FU *  * \n"
			    "P9+KY+KE * +KI * +KI * +KE+KY\n"
			    "-\n").getInitialState());
    SearchState2Core sstate(state2, checkmate_searcher);
    qsearch_t searcher2(sstate, table);
    PieceEval ev2(state2);
    const int val2 = searcher2.search(WHITE, ev2, Move(Position(7,8),SILVER,BLACK));
    CPPUNIT_ASSERT_EQUAL(0, val2); // 0 + 脅威
  }
}

void QuiescenceSearchTest::testCaptureCheck()
{
  osl::search::SearchState2::checkmate_t checkmate_searcher;
  SimpleHashTable table(0);
  {
    // 銀を取ると龍が素抜きなので歩を取る
    state_t state0(CsaString(
		     "P1 * -OU *  * -RY-GI * +RY * \n"
		     "P2 *  *  *  * -KA * +GI *  * \n"
		     "P3-FU-FU-FU-FU * -FU-FU-FU-FU\n"
		     "P4 *  *  *  * -FU * -GI-KI * \n"
		     "P5 *  *  *  * +FU *  *  *  * \n"
		     "P6 *  *  *  *  *  *  *  *  * \n"
		     "P7+FU+FU+FU+FU * +FU+FU+FU+FU\n"
		     "P8 * +KA *  *  *  * +KI *  * \n"
		     "P9+KY+KE+GI+KI+OU *  * +KE+KY\n"
		     "P-00AL\n"
		     "-\n").getInitialState());
    SearchState2Core sstate(state0, checkmate_searcher);
    qsearch_t searcher0(sstate, table);
    PieceEval ev0(state0);
    const int val0 = searcher0.search(WHITE, ev0, Move(Position(3,2),SILVER,BLACK));
    CPPUNIT_ASSERT_EQUAL(-Pawn*2+Psilver-Silver, val0);
  }
  {
    // 32金を取ると王が素抜き
    state_t state1(CsaString(
		     "P1-KY-OU *  *  * -KI * +RY * \n"
		     "P2-KY *  *  * -GI * +KI *  * \n"
		     "P3-FU-FU-FU-KA * -FU-FU-FU-FU\n"
		     "P4 *  *  *  * -FU *  * -RY * \n"
		     "P5 *  *  * -FU+FU *  *  *  * \n"
		     "P6 *  *  *  *  *  *  *  *  * \n"
		     "P7+FU+FU+FU+FU * +FU+FU+FU+FU\n"
		     "P8 * +KA *  *  *  * +GI *  * \n"
		     "P9+KY+KE+GI * +OU+KI * +KE+KY\n"
		     "P-00AL\n"
		     "-\n").getInitialState());
    SearchState2Core sstate(state1, checkmate_searcher);
    CPPUNIT_ASSERT_EQUAL(0, PieceEval(state1).value());
    qsearch_t searcher1(sstate, table);
    PieceEval ev(state1);
    const int val1 = searcher1.search(WHITE, ev, Move(Position(3,2),GOLD,BLACK));
    CPPUNIT_ASSERT_EQUAL(-PtypeEvalTraits<PAWN>::val*2, val1);
  }
  {
    state_t state2(CsaString(
		     "P1-KY-KE-KE-KI *  *  * +RY-KY\n"
		     "P2-KA-GI-OU-FU *  *  *  *  * \n"
		     "P3-FU-FU * -KI *  *  *  *  * \n"
		     "P4 *  *  *  * +FU *  * -FU-FU\n"
		     "P5 *  *  * +FU *  *  *  *  * \n"
		     "P6 *  * +FU *  *  * +GI+FU+FU\n"
		     "P7+FU *  *  * -GI * +KI *  * \n"
		     "P8 *  *  *  *  * -RY+FU+OU * \n"
		     "P9+KY+KE *  *  *  * -GI+KE+KY\n"
		     "P+00FU00FU00KI00KA\n"
		     "P-00FU00FU00FU00FU\n"
		     "+\n"
		     ).getInitialState());
    // +2818OU -6354KI +2111RY で損なので後手は2手目でパス
    SearchState2Core sstate(state2, checkmate_searcher);
    qsearch_t searcher2(sstate, table);
    PieceEval ev(state2);
    if (! isShortTest)
      std::cerr << ev.value() << "\n";
    const Move last_move = Move(Position(3,9),SILVER,WHITE);
    const int val2 = searcher2.search(BLACK, ev, last_move);
    const int val2n = searcher2.search<BLACK>(-899, -895, ev, last_move);
    CPPUNIT_ASSERT_EQUAL(val2, val2n);
  }
  {
    state_t state3(CsaString(
		     "P1-KY-KE * -KI *  *  * +RY-KY\n"
		     "P2 * -GI-OU-FU *  *  *  *  * \n"
		     "P3-FU-FU * -KI *  *  *  *  * \n"
		     "P4 *  *  *  * +FU *  *  * -FU\n"
		     "P5 *  *  * +FU *  *  *  *  * \n"
		     "P6 *  * +FU *  *  * +GI * +FU\n"
		     "P7+FU *  *  *  *  * +KI+FU * \n"
		     "P8 *  *  *  *  * -RY+FU+OU * \n"
		     "P9+KY+KE *  *  * -GI-KA+KE+KY\n"
		     "P+00FU00FU00KI00KA\n"
		     "P-00FU00FU00FU00FU00FU00KE00GI\n"
		     "+\n"
		     ).getInitialState());
    SearchState2Core sstate(state3, checkmate_searcher);
    qsearch_t searcher3(sstate, table);
    PieceEval ev(state3);
    if (! isShortTest)
      std::cerr << ev.value() << "\n";
    // +2818OU -4938NG PASS
    const Move last_move = Move(Position(3,9),BISHOP,WHITE);
    const int val3  = searcher3.search(BLACK, ev, last_move);
    const int val3n = searcher3.search<BLACK>(val3-1, val3+1, 
					      ev, last_move);
    // window を狭めても結果が同じ
    CPPUNIT_ASSERT_EQUAL(val3, val3n);
  }
}

void QuiescenceSearchTest::testCaptureCheckMate()
{
  osl::search::SearchState2::checkmate_t checkmate_searcher;
  SimpleHashTable table(0);
  {
    // 詰
    state_t state(CsaString(
		    "P1-KY-KE-GI-KI-KA-KI-GI-KE-KY\n"
		    "P2 * -HI *  *  *  *  * -OU * \n"
		    "P3-FU-FU-FU-FU-FU-FU-FU-FU-FU\n"
		    "P4 *  *  *  *  *  *  * +FU * \n"
		    "P5 *  *  *  *  *  *  *  *  * \n"
		    "P6 *  *  *  *  *  *  *  *  * \n"
		    "P7+FU+FU+FU+FU+FU+FU+FU * +FU\n"
		    "P8 * +KA *  *  *  *  * +HI * \n"
		    "P9+KY+KE+GI+KI+OU+KI+GI+KE+KY\n"
		    "+\n").getInitialState());
    SearchState2Core sstate(state, checkmate_searcher);
    qsearch_t searcher(sstate, table);
    PieceEval ev(state);
    const int val1 = searcher.search(BLACK, ev, Move(Position(2,3),PAWN,WHITE));
    CPPUNIT_ASSERT_EQUAL(FixedEval::winByCheckmate(BLACK), val1);
  }
  {
    // 王手じゃなければ飛車がとれるが，逃げている間に歩をとられる．
    state_t state(CsaString(
		    "P1-KY-KE-GI-KI * -KI-GI-KE-KY\n"
		    "P2 * -HI *  *  *  *  * -KA * \n"
		    "P3-FU-FU * -FU-FU-FU-OU-FU-FU\n"
		    "P4 *  *  *  *  *  * -FU *  * \n"
		    "P5 *  * -FU *  *  * +HI *  * \n"
		    "P6 *  * +FU *  *  *  *  *  * \n"
		    "P7+FU+FU * +FU+FU+FU+FU+FU+FU\n"
		    "P8 * +KA *  *  *  *  *  *  * \n"
		    "P9+KY+KE+GI+KI+OU+KI+GI+KE+KY\n"
		    "-\n").getInitialState());
    SearchState2Core sstate(state, checkmate_searcher);
    qsearch_t searcher(sstate, table);
    PieceEval ev(state);
    const int val1 = searcher.search(WHITE, ev, Move(Position(7,6),PAWN,BLACK));
    CPPUNIT_ASSERT_EQUAL(Pawn*2, val1);
  }
  {
    // 王で飛車を取って王手回避
    state_t state(CsaString(
		    "P1-KY-KE-GI-KI * -KI-GI-KE-KY\n"
		    "P2 * -HI *  *  *  *  * -KA * \n"
		    "P3-FU-FU-FU-FU-FU-FU-OU-FU-FU\n"
		    "P4 *  *  *  *  *  * -FU+HI * \n"
		    "P5 *  *  *  *  *  *  *  *  * \n"
		    "P6 *  * +FU *  *  *  *  *  * \n"
		    "P7+FU+FU * +FU+FU+FU+FU+FU+FU\n"
		    "P8 * +KA *  *  *  *  *  *  * \n"
		    "P9+KY+KE+GI+KI+OU+KI+GI+KE+KY\n"
		    "-\n").getInitialState());
    SearchState2Core sstate(state, checkmate_searcher);
    qsearch_t searcher(sstate, table);
    PieceEval ev(state);
    const int val1 = searcher.search(WHITE, ev, Move(Position(7,6),PAWN,BLACK));
    CPPUNIT_ASSERT_EQUAL(-PtypeEvalTraits<ROOK>::val*2, val1);
  }
  {
    // 送り金
    state_t state(CsaString(
		    "P1 *  *  *  *  *  *  *  *  * \n"
		    "P2 * +KI-OU-KI+RY *  *  *  * \n"
		    "P3-FU-FU-FU-FU-FU * -FU-FU-FU\n"
		    "P4 *  * -RY *  *  * -KI *  * \n"
		    "P5 *  *  *  *  *  *  *  *  * \n"
		    "P6 *  *  *  *  *  *  *  *  * \n"
		    "P7+FU+FU+FU+FU+FU+FU+FU+FU+FU\n"
		    "P8 * +KA * +GI *  *  *  *  * \n"
		    "P9+KY+KE *  * +OU+KI+GI+KE+KY\n"
		    "P-00AL\n"
		    "-\n").getInitialState());
    SearchState2Core sstate(state, checkmate_searcher);
    qsearch_t searcher(sstate, table);
    PieceEval ev(state);
    const int val1 = searcher.search(WHITE, ev, Move(Position(8,2),GOLD,BLACK));
    CPPUNIT_ASSERT_EQUAL(Pawn*2, val1);
  }
  {
    // バグの可能性があったが QuiescenceSearch のせいではないようだ
    state_t state(CsaString(
		    "P1-KY *  *  *  * +NK-OU * -KY\n"
		    "P2 *  *  *  * +TO-GI-KI *  * \n"
		    "P3-FU * -HI *  * -FU * -FU * \n"
		    "P4 *  *  *  * -GI * -FU-KY-FU\n"
		    "P5 * -FU *  *  * -KE * +FU * \n"
		    "P6+FU *  * -FU+FU *  *  *  * \n"
		    "P7+KA+FU+KE+GI * +FU+FU * +FU\n"
		    "P8-GI+KI+KI *  *  *  * +HI * \n"
		    "P9-UM * +OU *  *  *  * +KE+KY\n"
		    "P+00FU\n"
		    "P+00FU\n"
		    "P-00FU\n"
		    "P-00KI\n"
		    "-\n").getInitialState());
    SearchState2Core sstate(state, checkmate_searcher);
    qsearch_t searcher(sstate, table);
    PieceEval ev(state);
    const int val1 = searcher.search(WHITE, ev, 
				     Move(Position(5,1),Position(4,1),
					     PKNIGHT,PTYPE_EMPTY,false,BLACK));
    CPPUNIT_ASSERT(eval::isConsistentValueForNormalState<PieceEval>(val1));
    CPPUNIT_ASSERT(! FixedEval::isWinValue(BLACK, val1));
  }
}

static int error_count = 0;
void QuiescenceSearchTest::testIsEvalValueConsistent(std::string const& fileName)
{
  osl::search::SearchState2::checkmate_t checkmate_searcher;
  // QuiescenceLog::init("qsearch.log");
  
  Record rec=CsaFile(fileName).getRecord();
  const NumEffectState sstate(rec.getInitialState());
  const vector<osl::Move> moves=rec.getMoves();

  typedef QuiescenceSearch2<PieceEval> qsearch_t;

  PieceEval ev(sstate);
  GameState state(sstate);
  SimpleHashTable table(400000,-8,!isShortTest);
  SimpleHashTable null_table(400000,100,false);
  const int black_win = search::FixedEval::winByLoop(BLACK);
  const int white_win = search::FixedEval::winByLoop(WHITE);
  MilliSeconds started = MilliSeconds::now();
  size_t node_count = 0;
 for (unsigned int i=0;i<moves.size();i++)
  {
    if (i > 0)
    {
#if 0
      if ((i % 64) == 0)
	table.clear();
#endif
      NumEffectState state0(state.state());
      NumEffectState state1(state0);
      HistoryToTable::adjustTable(state, table, black_win, 0, white_win);
      HistoryToTable::adjustTable(state, null_table, black_win, 0, white_win);

      SearchState2Core sstate0(state0, checkmate_searcher);
      SearchState2Core sstate1(state1, checkmate_searcher);
      qsearch_t qs_null(sstate0, null_table);
      qsearch_t qs(sstate1, table);
      const Move last_move = moves[i-1];
      const int val_null = qs_null.search(state0.getTurn(), ev, last_move, 6);
      const int val = qs.searchIteratively(state1.getTurn(), ev, last_move, 6);
      node_count += qs.nodeCount() + qs_null.nodeCount();
      if (abs(val_null - val) > 1000)
      {
	if (! isShortTest)
	  std::cerr << fileName << " " << i << "\n" << state0
		    << "\n" << ev.value() << " " << val_null << " " << val << "\n";
	++error_count;
      }
    
      CPPUNIT_ASSERT(isConsistentValue(val));
      
      if (isShortTest)		// 最近はかなり異なる
	CPPUNIT_ASSERT(error_count < 50);
      // CPPUNIT_ASSERT_EQUAL(val_null, val);
      CPPUNIT_ASSERT(state0.isConsistent(true));
      CPPUNIT_ASSERT(state1.isConsistent(true));
    }

    const Move move = moves[i];
    state.pushMove(move);
    ev.update(state.state(), move);
  }
  if (! isShortTest)
  {
    std::cerr << node_count << " " << (MilliSeconds::now() - started).toSeconds() << "\n";
  }
}

void QuiescenceSearchTest::testEquality()
{
  std::ifstream ifs(OslConfig::testCsaFile("FILES"));
  CPPUNIT_ASSERT(ifs);
  std::string file_name;
  for (int i=0; i<(isShortTest ? 10: 100) && (ifs >> file_name); i++){
    if (file_name == "") 
      break;
    testIsEvalValueConsistent(OslConfig::testCsaFile(file_name));
  }
  CPPUNIT_ASSERT(error_count < 50);
  // CPPUNIT_ASSERT_EQUAL(0, error_count);
}

void QuiescenceSearchTest::testTakeBack()
{
  osl::search::SearchState2::checkmate_t checkmate_searcher;
  {
    state_t state(CsaString(
		    "P1-KY-KE-GI-KI-OU-KI-GI-KE-KY\n"
		    "P2 * -HI *  *  *  *  * -KA * \n"
		    "P3-FU-FU-FU-FU-FU-FU-FU-FU-FU\n"
		    "P4 *  *  *  *  *  *  *  *  * \n"
		    "P5 *  *  *  *  *  *  *  *  * \n"
		    "P6 *  *  *  *  *  *  * +FU * \n"
		    "P7+FU+FU+FU+FU+FU+FU+FU * +FU\n"
		    "P8 * +KA *  *  *  *  * +HI * \n"
		    "P9+KY+KE+GI+KI+OU+KI+GI+KE+KY\n"
		    "-\n").getInitialState());
    SearchState2Core sstate(state, checkmate_searcher);
    SimpleHashTable table(0);
    qsearch_t searcher(sstate, table);
    PieceEval ev(state);
    const int value = searcher.takeBackValue<WHITE>
      (1001, -1001, ev, Move(Position(2,6),PAWN,BLACK));
    CPPUNIT_ASSERT_EQUAL(0, value);
  }
  {
    state_t state(CsaString(
		    "P1-KY-KE-GI-KI-OU-KI-GI-KE-KY\n"
		    "P2 * -HI *  *  *  *  * -KA * \n"
		    "P3-FU-FU-FU-FU-FU-FU-FU-FU-FU\n"
		    "P4 *  *  *  *  *  *  * +FU * \n"
		    "P5 *  *  *  *  *  *  *  *  * \n"
		    "P6 *  *  *  *  *  *  *  *  * \n"
		    "P7+FU+FU+FU+FU+FU+FU+FU * +FU\n"
		    "P8 * +KA *  *  *  *  * +HI * \n"
		    "P9+KY+KE+GI+KI+OU+KI+GI+KE+KY\n"
		    "-\n").getInitialState());
    SearchState2Core sstate(state, checkmate_searcher);
    SimpleHashTable table(0);
    qsearch_t searcher(sstate, table);
    PieceEval ev(state);
    const int value = searcher.takeBackValue<WHITE>
      (1001, -1001, ev, Move(Position(2,4),PAWN,BLACK));
    CPPUNIT_ASSERT_EQUAL(0, value);
  }
  {
    state_t state(CsaString(
		    "P1-KY-KE-GI-KI-OU-KI-GI-KE-KY\n"
		    "P2 * -HI *  *  *  *  * -KA * \n"
		    "P3-FU-FU-FU-FU-FU-FU-FU-FU-FU\n"
		    "P4 *  *  *  *  *  *  * +FU * \n"
		    "P5 *  *  *  *  *  *  *  *  * \n"
		    "P6 *  *  *  *  *  *  *  *  * \n"
		    "P7+FU+FU+FU+FU+FU+FU+FU * +FU\n"
		    "P8 * +KA *  *  *  *  *  * +HI\n"
		    "P9+KY+KE+GI+KI+OU+KI+GI+KE+KY\n"
		    "-\n").getInitialState());
    SearchState2Core sstate(state, checkmate_searcher);
    SimpleHashTable table(0);
    qsearch_t searcher(sstate, table);
    PieceEval ev(state);
    const int value = searcher.takeBackValue<WHITE>
      (1001, -1001, ev, Move(Position(2,4),PAWN,BLACK));
    CPPUNIT_ASSERT_EQUAL(-Pawn*2, value);
  }
  {
    state_t state(CsaString(	// 取れない
		    "P1-KY-KE-GI-KI-OU-KI-GI * -KY\n"
		    "P2 * -HI *  *  *  *  *  *  * \n"
		    "P3-FU-FU-FU-FU-FU-FU-FU-FU-FU\n"
		    "P4 *  *  *  *  *  *  *  *  * \n"
		    "P5 *  *  *  *  * +FU+FU * -KA\n"
		    "P6 *  *  *  *  *  *  *  *  * \n"
		    "P7+FU+FU+FU+FU+FU-KE * +FU+FU\n"
		    "P8 * +KA *  *  * +GI * +HI * \n"
		    "P9+KY+KE+GI+KI+OU+KI * +KE+KY\n"
		    "+\n").getInitialState());
    SearchState2Core sstate(state, checkmate_searcher);
    SimpleHashTable table(0);
    qsearch_t searcher(sstate, table);
    PieceEval ev(state);
    const int value = searcher.takeBackValue<BLACK>
      (-1001, 1001, ev, Move(Position(4,7),KNIGHT,WHITE));
    CPPUNIT_ASSERT_EQUAL(0, value);
  }
  {				// 詰
    state_t state(CsaString(
		    "P1-KY-KE * -KI *  *  * +RY-KY\n"
		    "P2 * -GI-OU-FU *  *  *  *  * \n"
		    "P3-FU-FU * -KI *  *  *  *  * \n"
		    "P4 *  *  *  * +FU *  *  * -FU\n"
		    "P5 *  *  * +FU *  *  *  *  * \n"
		    "P6 *  * +FU *  *  * +GI * +FU\n"
		    "P7+FU *  *  *  *  * -NG+FU * \n"
		    "P8 *  *  *  *  * -RY *  * +OU\n"
		    "P9+KY+KE *  *  *  * -KA+KE+KY\n"
		    "P+00FU00FU00KI00KA\n"
		    "P-00FU00FU00FU00FU00FU00KE00GI00FU00KI\n"
		    "+\n"
		    ).getInitialState());
    SearchState2Core sstate(state, checkmate_searcher);
    SimpleHashTable table(0);
    qsearch_t searcher(sstate, table);
    PieceEval ev(state);
    const int value = searcher.takeBackValue<BLACK>
      (-10001,+10001,ev,Move(Position(3,8),Position(3,7),PSILVER,GOLD,false,WHITE));
    CPPUNIT_ASSERT_EQUAL(FixedEval::winByCheckmate(WHITE), value);
  }
}

void QuiescenceSearchTest::testThreat()
{
  osl::search::SearchState2::checkmate_t checkmate_searcher;
  {
    state_t state(CsaString(
		    "P1-KY-KE-GI-KI-OU-KI-GI-KE-KY\n"
		    "P2 * -HI *  *  *  *  * -KA * \n"
		    "P3-FU-FU-FU-FU-FU-FU-FU-FU-FU\n"
		    "P4 *  *  *  *  *  *  *  *  * \n"
		    "P5 *  *  *  *  *  *  *  *  * \n"
		    "P6 *  *  *  *  *  *  * +FU * \n"
		    "P7+FU+FU+FU+FU+FU+FU+FU * +FU\n"
		    "P8 * +KA *  *  *  *  * +HI * \n"
		    "P9+KY+KE+GI+KI+OU+KI+GI+KE+KY\n"
		    "-\n").getInitialState());
    SearchState2Core sstate(state, checkmate_searcher);
    SimpleHashTable table(0);
    qsearch_t searcher(sstate, table);
    PieceEval ev(state);
    const int value = searcher.staticValueWithThreat<WHITE>(ev);
    CPPUNIT_ASSERT_EQUAL(ev.value(), value);
  }
  {
    state_t state(CsaString(
		    "P1-KY-KE-GI-KI-OU-KI-GI-KE-KY\n"
		    "P2 * -HI *  *  *  *  * -KA * \n"
		    "P3-FU-FU-FU-FU-FU-FU-FU-FU-FU\n"
		    "P4 *  *  *  *  *  *  * +FU * \n"
		    "P5 *  *  *  *  *  *  *  *  * \n"
		    "P6 *  *  *  *  *  *  *  *  * \n"
		    "P7+FU+FU+FU+FU+FU+FU+FU * +FU\n"
		    "P8 * +KA *  *  *  *  * +HI * \n"
		    "P9+KY+KE+GI+KI+OU+KI+GI+KE+KY\n"
		    "-\n").getInitialState());
    SearchState2Core sstate(state, checkmate_searcher);
    SimpleHashTable table(0);
    qsearch_t searcher(sstate, table);
    PieceEval ev(state);
    const int value = searcher.staticValueWithThreat<WHITE>(ev);
    CPPUNIT_ASSERT(ev.value() < value); // 後手がパスすると危険
  }
  {
    state_t state(CsaString(
		    "P1-KY-KE-GI-KI-OU-KI-GI-KE-KY\n"
		    "P2 * -HI *  *  *  *  * -KA * \n"
		    "P3-FU-FU-FU-FU-FU-FU-FU-FU-FU\n"
		    "P4 *  *  *  *  *  *  * +FU * \n"
		    "P5 *  *  *  *  *  *  *  *  * \n"
		    "P6 *  *  *  *  *  *  *  *  * \n"
		    "P7+FU+FU+FU+FU+FU+FU+FU * +FU\n"
		    "P8 * +KA *  *  *  *  *  * +HI\n"
		    "P9+KY+KE+GI+KI+OU+KI+GI+KE+KY\n"
		    "-\n").getInitialState());
    SearchState2Core sstate(state, checkmate_searcher);
    SimpleHashTable table(0);
    qsearch_t searcher(sstate, table);
    PieceEval ev(state);
    const int value = searcher.staticValueWithThreat<WHITE>(ev);
    CPPUNIT_ASSERT(ev.value() < value);
    CPPUNIT_ASSERT(state.isConsistent());
  }
  {
    state_t state(CsaString(
		    "P1-KY-KE-GI-KI-OU * -GI-KE-KY\n"
		    "P2 * -HI *  *  *  * -KI-KA * \n"
		    "P3-FU-FU-FU-FU-FU-FU-FU-FU-FU\n"
		    "P4 *  *  *  *  *  *  * +FU * \n"
		    "P5 *  *  *  *  *  *  *  *  * \n"
		    "P6 *  *  *  *  *  *  *  *  * \n"
		    "P7+FU+FU+FU+FU+FU+FU+FU * +FU\n"
		    "P8 * +KA *  *  *  *  *  * +HI\n"
		    "P9+KY+KE+GI+KI+OU+KI+GI+KE+KY\n"
		    "-\n").getInitialState());
    SearchState2Core sstate(state, checkmate_searcher);
    SimpleHashTable table(0);
    qsearch_t searcher(sstate, table);
    PieceEval ev(state);
    const int value = searcher.staticValueWithThreat<WHITE>(ev);
    CPPUNIT_ASSERT_EQUAL(ev.value(), value);
    CPPUNIT_ASSERT(state.isConsistent());
  }
  {
    state_t state(CsaString(	// 取れない
		    "P1-KY-KE-GI-KI-OU-KE-GI * -KY\n"
		    "P2 * -HI *  *  *  *  *  *  * \n"
		    "P3-FU-FU-FU-FU-FU-FU-FU-FU-FU\n"
		    "P4 *  *  *  *  *  *  *  *  * \n"
		    "P5 *  *  *  *  * +FU+FU * -KA\n"
		    "P6 *  *  *  *  *  *  *  *  * \n"
		    "P7+FU+FU+FU+FU+FU-KI * +FU+FU\n"
		    "P8 * +KA * +GI * +KI * +HI * \n"
		    "P9+KY+KE * +KI+OU * +GI+KE+KY\n"
		    "+\n").getInitialState());
    SearchState2Core sstate(state, checkmate_searcher);
    SimpleHashTable table(0);
    qsearch_t searcher(sstate, table);
    PieceEval ev(state);
    const int value = searcher.staticValueWithThreat<BLACK>(ev);
    CPPUNIT_ASSERT_EQUAL(ev.value(), value);
    CPPUNIT_ASSERT(state.isConsistent());
  }
  {
    state_t state(CsaString(
      "P1-KY-KE-GI-KI-OU-KI-GI-KE-KY\n"
      "P2 *  *  *  *  *  *  * -KA * \n"
      "P3-FU-FU-FU-FU-FU-FU-FU-FU-FU\n"
      "P4 *  *  *  *  *  *  *  *  * \n"
      "P5 * -HI *  * +GI *  *  *  * \n"
      "P6 *  *  *  * +FU *  *  *  * \n"
      "P7+FU+FU+FU+FU * +FU+FU+FU+FU\n"
      "P8 *  * +KI *  * +KA * +HI * \n"
      "P9+KY+KE *  * +OU+KI+GI+KE+KY\n"
      "+\n").getInitialState());
    SimpleHashTable table(0);
    SearchState2Core sstate(state, checkmate_searcher);
    qsearch_t searcher(sstate, table);
    PieceEval ev(state);
    const int value = searcher.staticValueWithThreat<BLACK>(ev);
    CPPUNIT_ASSERT_EQUAL(0, value);
  }
  {
    state_t state(CsaString(
		    "P1-KY-KE * -KI *  *  * +RY-KY\n"
		    "P2 * -GI-OU-FU *  *  *  *  * \n"
		    "P3-FU-FU * -KI *  *  *  *  * \n"
		    "P4 *  *  *  * +FU *  *  * -FU\n"
		    "P5 *  *  * +FU *  *  *  *  * \n"
		    "P6 *  * +FU *  *  * +GI * +FU\n"
		    "P7+FU *  *  *  *  * +KI+FU * \n"
		    "P8 *  *  *  *  * -RY-NG * +OU\n"
		    "P9+KY+KE *  *  *  * -KA+KE+KY\n"
		    "P+00FU00FU00KI00KA\n"
		    "P-00FU00FU00FU00FU00FU00KE00GI00FU\n"
		    "+\n"
		    ).getInitialState());
    SearchState2Core sstate(state, checkmate_searcher);
    SimpleHashTable table(0);
    qsearch_t searcher(sstate, table);
    PieceEval ev(state);
    const int value = searcher.staticValueWithThreat<BLACK>(ev);
    CPPUNIT_ASSERT(value < -8000);
  }
}

bool QuiescenceSearchTest::betterThan(const char *l, const char *r)
{
  osl::search::SearchState2::checkmate_t checkmate_searcher;
  state_t state_l(CsaString(l).getInitialState());
  state_t state_r(CsaString(r).getInitialState());
  assert(state_l.getTurn() == state_r.getTurn());
  SimpleHashTable table(1000);
  int value_l, value_r;
  {
    // root ではない状態にするため、手番を変更しておいてからパス
    SearchState2Core sstate(state_l, checkmate_searcher);
    sstate.pushPass();
    qsearch_t searcher(sstate, table);
    sstate.pushPass();

    PieceEval ev(state_l);
    value_l = searcher.staticValueWithThreat(ev);
  }
  table.clear();
  {
    // root ではない状態にする
    SearchState2Core sstate(state_r, checkmate_searcher);
    sstate.pushPass();
    qsearch_t searcher(sstate, table);
    sstate.pushPass();

    PieceEval ev(state_r);
    value_r = searcher.staticValueWithThreat(ev);
  }
  return eval::betterThan(state_l.getTurn(), value_l, value_r);
}

void QuiescenceSearchTest::testCompare()
{
  if (QSearchTraits::FirstThreat >= 20)
  {
    const char *l = 
      "P1-KY-KE-GI-KI-OU * -GI-KE-KY\n"
      "P2 * -HI *  *  *  * -KI-KA * \n"
      "P3-FU * -FU-FU-FU-FU-FU-FU-FU\n"
      "P4 * -FU *  *  *  *  * +HI * \n"
      "P5 *  *  *  *  *  *  *  *  * \n"
      "P6 *  *  *  *  *  *  *  *  * \n"
      "P7+FU+FU+FU+FU+FU+FU+FU * +FU\n"
      "P8 * +KA *  *  *  *  *  *  * \n"
      "P9+KY+KE+GI+KI+OU+KI+GI+KE+KY\n"
      "P+00FU\n"
      "+\n";
    const char *r = 
      "P1-KY-KE-GI-KI-OU * -GI-KE-KY\n"
      "P2 * -HI *  *  *  * -KI-KA * \n"
      "P3-FU * -FU-FU-FU-FU-FU * -FU\n"
      "P4 * -FU *  *  *  *  * -FU * \n"
      "P5 *  *  *  *  *  *  *  *  * \n"
      "P6 *  *  *  *  *  *  *  *  * \n"
      "P7+FU+FU+FU+FU+FU+FU+FU * +FU\n"
      "P8 * +KA *  *  *  *  *  * +HI\n"
      "P9+KY+KE+GI+KI+OU+KI+GI+KE+KY\n"
      "P-00FU\n"
      "+\n";
    CPPUNIT_ASSERT(betterThan(l, r)); // 交換した方が良い
  }
  {
    // 二つの脅威
    // http://www31.ocn.ne.jp/~kfend/inside_kfend/quiescence.html#c5
    const char *l = 
      "P1-KY-KE-GI-KI-OU-KI-GI-KE-KY\n"
      "P2 *  *  *  *  *  *  * -KA * \n"
      "P3-FU-FU-FU-FU-FU-FU-FU-FU-FU\n"
      "P4 *  *  *  *  *  *  *  *  * \n"
      "P5 * -HI *  * +GI *  *  *  * \n"
      "P6 *  *  *  *  *  *  *  *  * \n"
      "P7+FU+FU+FU+FU+FU+FU+FU+FU+FU\n"
      "P8 *  * +KI *  * +KA * +HI * \n"
      "P9+KY+KE *  * +OU+KI+GI+KE+KY\n"
      "+\n";
    const char *r = 
      "P1-KY-KE-GI-KI-OU-KI-GI-KE-KY\n"
      "P2 *  *  *  *  *  *  * -KA * \n"
      "P3-FU-FU-FU-FU-FU-FU-FU-FU-FU\n"
      "P4 *  *  *  *  *  *  *  *  * \n"
      "P5 * -HI *  * +GI *  *  *  * \n"
      "P6 *  *  *  *  *  *  *  *  * \n"
      "P7+FU+FU+FU+FU+FU+FU+FU+FU+FU\n"
      "P8 *  *  *  *  * +KA * +HI * \n"
      "P9+KY+KE * +KI+OU+KI+GI+KE+KY\n"
      "+\n";
    CPPUNIT_ASSERT(betterThan(l, r));
  }
  {
    // 二つの脅威 同じptype
    const char *l = 
      "P1-KY-KE-GI-KI-OU-KI-GI-KE-KY\n"
      "P2 *  *  *  * -FU *  * -KA * \n"
      "P3-FU-FU-FU-FU * -FU-FU-FU-FU\n"
      "P4 * -HI *  * +GI *  *  *  * \n"
      "P5 *  *  *  *  *  *  *  *  * \n"
      "P6 * +GI *  *  *  *  *  *  * \n"
      "P7+FU+FU+FU+FU+FU+FU+FU+FU+FU\n"
      "P8 *  * +KI *  * +KA * +HI * \n"
      "P9+KY+KE *  * +OU+KI * +KE+KY\n"
      "+\n";
    const char *r = 
      "P1-KY-KE-GI-KI-OU-KI-GI-KE-KY\n"
      "P2 *  *  *  * -FU *  * -KA * \n"
      "P3-FU-FU-FU-FU * -FU-FU-FU-FU\n"
      "P4 * -HI *  * +GI *  *  *  * \n"
      "P5 *  *  *  *  *  *  *  *  * \n"
      "P6 * +GI *  *  *  *  *  *  * \n"
      "P7+FU * +FU+FU+FU+FU+FU+FU+FU\n"
      "P8 * +FU+KI *  * +KA * +HI * \n"
      "P9+KY+KE *  * +OU+KI * +KE+KY\n"
      "+\n";
    CPPUNIT_ASSERT(betterThan(l, r));
  }
  {
    // 成る脅威
    const char *l = 
      "P1-KY-KE-GI-KI-OU-KI-GI-KE-KY\n"
      "P2 *  *  *  *  *  *  * -KA * \n"
      "P3-FU-FU-FU-FU-FU-FU-FU-FU-FU\n"
      "P4 * -HI *  *  *  *  *  *  * \n"
      "P5 *  *  *  *  *  *  *  *  * \n"
      "P6 *  *  *  *  *  *  *  *  * \n"
      "P7+FU+FU+FU+FU+FU+FU+FU+FU+FU\n"
      "P8 *  * +GI *  * +KA * +HI * \n"
      "P9+KY+KE * +KI+OU+KI+GI+KE+KY\n"
      "+\n";
    const char *r = 
      "P1-KY-KE-GI-KI-OU-KI-GI-KE-KY\n"
      "P2 *  *  *  *  *  *  * -KA * \n"
      "P3-FU-FU-FU-FU-FU-FU-FU-FU-FU\n"
      "P4 * -HI *  *  *  *  *  *  * \n"
      "P5 *  *  *  *  *  *  *  *  * \n"
      "P6 *  *  *  *  *  *  *  *  * \n"
      "P7+FU * +FU+FU+FU+FU+FU+FU+FU\n"
      "P8 *  * +GI *  * +KA * +HI * \n"
      "P9+KY+KE * +KI+OU+KI+GI+KE+KY\n"
      "P+00FU\n"
      "+\n";
    CPPUNIT_ASSERT(betterThan(l, r));
  }
  {
    // 逃げられない脅威はr4103で廃止
  }
}

void QuiescenceSearchTest::testFocalPawn()
{
  osl::search::SearchState2::checkmate_t checkmate_searcher;
#if 0
  {
    // -2324FU+3524GI-4334GI で後手歩得で有利 とか思っていると
    // +0033FUで破滅
    state_t state(CsaString(
		    "P1-KY-KE * -KI *  *  * -KE * \n"
		    "P2 * -OU-GI * -KI * -HI-KA-KY\n"
		    "P3 * -FU-FU *  * -GI * -FU-FU\n"
		    "P4-FU *  * -FU-FU-FU+FU+FU * \n"
		    "P5 *  *  *  *  *  * +GI *  * \n"
		    "P6+FU * +FU * +FU *  *  * +FU\n"
		    "P7 * +FU * +FU * +FU *  *  * \n"
		    "P8 * +KA+OU * +KI * +HI *  * \n"
		    "P9+KY+KE+GI+KI *  *  * +KE+KY\n"
		    "P-00FU\n"
		    "-\n").getInitialState());
    SearchState2Core sstate(state, checkmate_searcher);
    SimpleHashTable table(10000,-6);
    qsearch_t searcher(sstate, table);
    PieceEval ev(state);
    const Move m24fu(Position(2,5),Position(2,4),PAWN,PTYPE_EMPTY,false,BLACK);
    const int val = searcher.search(WHITE, ev, m24fu);
    CPPUNIT_ASSERT(val > 0);	// 先手有利!
  }
  {
    // 角が引っ込んでいても同じ
    state_t state(CsaString(
		    "P1-KY-KE * -KI *  *  * -KE-KA\n"
		    "P2 * -OU-GI * -KI * -HI * -KY\n"
		    "P3 * -FU-FU * -FU-GI * -FU-FU\n"
		    "P4-FU *  * -FU * -FU+FU+FU * \n"
		    "P5 *  *  *  *  *  * +GI *  * \n"
		    "P6+FU * +FU * +FU *  *  * +FU\n"
		    "P7 * +FU * +FU * +FU *  *  * \n"
		    "P8 * +KA+OU * +KI * +HI *  * \n"
		    "P9+KY+KE+GI+KI *  *  * +KE+KY\n"
		    "P-00FU\n"
		    "-\n").getInitialState());
    SearchState2Core sstate(state, checkmate_searcher);
    SimpleHashTable table(10000,-6);
    qsearch_t searcher(sstate, table);
    PieceEval ev(state);
    const Move m24fu(Position(2,5),Position(2,4),PAWN,PTYPE_EMPTY,false,BLACK);
    const int val = searcher.search(WHITE, ev, m24fu);
    CPPUNIT_ASSERT(val > 0);	// 先手有利!
  }
#endif
}

/* ------------------------------------------------------------------------- */
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
