/* massXpert - the true massist's program.
   --------------------------------------
   Copyright(C) 2006,2007 Filippo Rusconi

   http://www.massxpert.org/massXpert

   This file is part of the massXpert project.

   The massxpert project is the successor to the "GNU polyxmass"
   project that is an official GNU project package(see
   www.gnu.org). The massXpert project is not endorsed by the GNU
   project, although it is released ---in its entirety--- under the
   GNU General Public License. A huge part of the code in massXpert
   is actually a C++ rewrite of code in GNU polyxmass. As such
   massXpert was started at the Centre National de la Recherche
   Scientifique(FRANCE), that granted me the formal authorization to
   publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License version 3, as published by the Free Software Foundation.
   

   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this software; if not, write to the

   Free Software Foundation, Inc.,

   51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA.
*/


/////////////////////// Local includes
#include "massSearchOligomerTreeViewModel.hpp"
#include "massSearchOligomerTreeViewItem.hpp"
#include "application.hpp"


namespace massXpert
{

  MassSearchOligomerTreeViewModel::MassSearchOligomerTreeViewModel 
 (QList <Oligomer *> *dataList, QObject *parent) 
    : QAbstractItemModel(parent)
  {
    QList<QVariant> rootData;

    Q_ASSERT(dataList);
    Q_ASSERT(parent);
  
    mp_parentDlg = static_cast<MassSearchDlg *>(parent);
  
    rootData << tr("Searched") << tr("Name") << tr("Coordinates")
	     << tr("Error")
	     << tr("Mono") << tr("Avg") << tr("Modif");
  
    mpa_rootItem = new MassSearchOligomerTreeViewItem(rootData);
  
    mp_list = dataList;
    setupModelData(mpa_rootItem);
  }


  MassSearchOligomerTreeViewModel::~MassSearchOligomerTreeViewModel()
  {
    delete mpa_rootItem;
  }


  MassSearchDlg *
  MassSearchOligomerTreeViewModel::getParentDlg()
  {
    return mp_parentDlg;
  }


  void 
  MassSearchOligomerTreeViewModel::setTreeView(QTreeView *treeView)
  {
    Q_ASSERT(treeView);
  
    mp_treeView = treeView;
  }


  QTreeView *
  MassSearchOligomerTreeViewModel::treeView()
  {
    return mp_treeView;
  }


  QVariant 
  MassSearchOligomerTreeViewModel::data(const QModelIndex &index, 
					 int role) const
  {
    if (!index.isValid())
      return QVariant();
  
    if (role != Qt::DisplayRole)
      return QVariant();
  
    MassSearchOligomerTreeViewItem *item = 
      static_cast<MassSearchOligomerTreeViewItem *>(index.internalPointer());
  
    return item->data(index.column());
  }


  // bool
  // MassSearchOligomerTreeViewModel::setData(const QModelIndex & index, 
  // 					      const QVariant & value, 
  // 					      int role)
  // {
  //   return true;
  // }


  QVariant 
  MassSearchOligomerTreeViewModel::headerData(int section, 
					       Qt::Orientation orientation,
					       int role) const
  {
    if (orientation == Qt::Horizontal && role == Qt::DisplayRole)
      return mpa_rootItem->data(section);
  
    return QVariant();
  }


  Qt::ItemFlags 
  MassSearchOligomerTreeViewModel::flags(const QModelIndex &index) const
  {
    if (!index.isValid())
      return Qt::ItemIsEnabled;
  
    return Qt::ItemIsEnabled | Qt::ItemIsSelectable;
  }


  QModelIndex 
  MassSearchOligomerTreeViewModel::index(int row, int column,
					  const QModelIndex &parent) const
  {
    MassSearchOligomerTreeViewItem *parentItem = 0;
  
    if (!parent.isValid())
      parentItem = mpa_rootItem;
    else
      parentItem = 
	static_cast<MassSearchOligomerTreeViewItem *> 
	(parent.internalPointer());
  
    MassSearchOligomerTreeViewItem *childItem = parentItem->child(row);
    if (childItem != 0)
      return createIndex(row, column, childItem);
    else
      return QModelIndex();
  }


  QModelIndex 
  MassSearchOligomerTreeViewModel::parent(const QModelIndex &index) const
  {
    if (!index.isValid())
      return QModelIndex();

    MassSearchOligomerTreeViewItem *childItem = 
      static_cast<MassSearchOligomerTreeViewItem *>(index.internalPointer());
  
    MassSearchOligomerTreeViewItem *parentItem = childItem->parent();

    if (parentItem == mpa_rootItem)
      return QModelIndex();
  
    return createIndex(parentItem->row(), 0, parentItem);
  }


  int 
  MassSearchOligomerTreeViewModel::rowCount(const QModelIndex &parent) const
  {
    MassSearchOligomerTreeViewItem *parentItem;
  
    if (!parent.isValid())
      parentItem = mpa_rootItem;
    else
      parentItem = 
	static_cast<MassSearchOligomerTreeViewItem *> 
	(parent.internalPointer());
  
    return parentItem->childCount();
  }


  int
  MassSearchOligomerTreeViewModel::columnCount(const QModelIndex &parent) const
  {
    if (parent.isValid())
      return static_cast<MassSearchOligomerTreeViewItem *> 
	(parent.internalPointer())->columnCount();
    else
      return mpa_rootItem->columnCount();
  }


  void
  MassSearchOligomerTreeViewModel::removeAll()
  {
    int count = rowCount();

    //   qDebug() << __FILE__ << __LINE__ 
    // 	    << "count items:" << count;

    if (!count)
      return;
    
    // On WinXP , the following beginRemoveRows crashes when count is 0,
    // because last>=first fails, due to having count-1 < than 0. This
    // is why we return immediately when count is 0.
    beginRemoveRows(QModelIndex(), 0, count - 1);

    while(!mp_list->isEmpty())
      {
	delete(mp_list->takeFirst());
            
	delete(mpa_rootItem->takeChild(0));
      
      }

    endRemoveRows();

    emit layoutChanged();
  }


  void 
  MassSearchOligomerTreeViewModel::addOligomer(Oligomer *oligomer)
  {
    Q_ASSERT(oligomer);
  
    Application *application = static_cast<Application *>(qApp);
    QLocale locale = application->locale();

    QList<QVariant> columnData;
  
    Prop *prop = oligomer->prop("SEARCHED_MZ");
    Q_ASSERT(prop);
  
    double value = 0;
    
    value  = *static_cast<const double *>(prop->data());
    QString mzSearched = locale.toString(value, 'f', MXP_OLIGOMER_DEC_PLACES);

    prop = oligomer->prop("ERROR_MZ");
    Q_ASSERT(prop);
  
    value = *static_cast<const double *>(prop->data());
    QString mzError = locale.toString(value, 'f', MXP_OLIGOMER_DEC_PLACES);

    QString coordinates(QString("[%1-%2]")
			 .arg(oligomer->startIndex() + 1)
			 .arg(oligomer->endIndex() + 1));
    
    // And now convert the masses from double to string, so
    // that we can use our conversion system.
    
    QString mono = oligomer->mono(locale, MXP_OLIGOMER_DEC_PLACES);
    // locale.toString(oligomer->mono(), 'f', 5);

    QString avg = oligomer->avg(locale, MXP_OLIGOMER_DEC_PLACES);
      // locale.toString(oligomer->avg(), 'f', 5);
    
    columnData
      << mzSearched << oligomer->name() 
      << coordinates << mzError 
      << mono << avg 
      << oligomer->isModified();
  
    mp_list->append(oligomer);

    // Create an item with those columnar strings. The parent of the
    // current item is the root item.
    MassSearchOligomerTreeViewItem *newItem = 
      new MassSearchOligomerTreeViewItem(columnData, mpa_rootItem);
  
    newItem->setOligomer(oligomer);

    int newRow = rowCount();

    mpa_rootItem->appendChild(newItem); 

    // There is no need to produce a model index for the parent, as we
    // know we are going to insert new row in the root of the tree(this
    // tree has no branches). Hence the QModelIndex() calls below.

    beginInsertRows(QModelIndex(), newRow, newRow);
    insertRow(newRow, QModelIndex());
    endInsertRows();

    emit layoutChanged();
  }


  bool 
  MassSearchOligomerTreeViewModel::setupModelData 
 (MassSearchOligomerTreeViewItem *parent)
  {
    Application *application = static_cast<Application *>(qApp);
    QLocale locale = application->locale();

    QList<MassSearchOligomerTreeViewItem *> parents;
    MassSearchOligomerTreeViewItem *currentItem = 0;
  
    Q_ASSERT(parent);
  
    // Start with populating the very first item of the treeviewmodel.
    parents << parent;
  
    // We have the mp_list pointer that points to a QList<Oligomer*>
    // list. 
  
    for (int iter = 0; iter < mp_list->size(); ++iter)
      {
	Oligomer *oligomer = mp_list->at(iter);
      
	QList<QVariant> columnData;
      
	Prop *prop = oligomer->prop("SEARCHED_MZ");
	Q_ASSERT(prop);
      
	double value = 0;
	
	value = *static_cast<const double *>(prop->data());
	QString mzSearched = 
	  locale.toString(value, 'f', MXP_OLIGOMER_DEC_PLACES);

	prop = oligomer->prop("ERROR_MZ");
	Q_ASSERT(prop);
      
	value = *static_cast<const double *>(prop->data());
	QString mzError = 
	  locale.toString(value, 'f', MXP_OLIGOMER_DEC_PLACES);

	QString coordinates(QString("[%1-%2]")
			     .arg(oligomer->startIndex() + 1)
			     .arg(oligomer->endIndex() + 1));
       
	// And now convert the masses from double to string, so
	// that we can use our conversion system.
	
	QString mono = oligomer->mono(locale, MXP_OLIGOMER_DEC_PLACES);
	// locale.toString(oligomer->mono(), 'f', 5);
	
	QString avg = oligomer->avg(locale, MXP_OLIGOMER_DEC_PLACES);
	// locale.toString(oligomer->avg(), 'f', 5);
	    
	columnData
	  << mzSearched << oligomer->name() 
	  << coordinates << mzError 
	  << mono << avg 
	  << oligomer->isModified();
      
	// Create an item with those columnar strings. The parent of the
	// current item is parents.
	currentItem = new MassSearchOligomerTreeViewItem(columnData,
							  parents [0]);
            
	currentItem->setOligomer(oligomer);

	// We should append that item right now.
	parents [0]->appendChild(currentItem);
      
	// At this point we have finished setting up the Model data.
      }
  
    return true;
  }

} // namespace massXpert
