/* massXpert - the true massist's program.
   --------------------------------------
   Copyright(C) 2006,2007 Filippo Rusconi

   http://www.massxpert.org/massXpert

   This file is part of the massXpert project.

   The massxpert project is the successor to the "GNU polyxmass"
   project that is an official GNU project package(see
   www.gnu.org). The massXpert project is not endorsed by the GNU
   project, although it is released ---in its entirety--- under the
   GNU General Public License. A huge part of the code in massXpert
   is actually a C++ rewrite of code in GNU polyxmass. As such
   massXpert was started at the Centre National de la Recherche
   Scientifique(FRANCE), that granted me the formal authorization to
   publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License version 3, as published by the Free Software Foundation.
   

   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this software; if not, write to the

   Free Software Foundation, Inc.,

   51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA.
*/


/////////////////////// Qt includes
#include <QMessageBox>
#include <QCloseEvent>
#include <QDebug>
#include <QInputDialog>

/////////////////////// Local includes
#include "mzLabInputOligomerTreeViewDlg.hpp"
#include "mzLabWnd.hpp"
#include "fragmentOligomer.hpp"


namespace massXpert
{

  MzLabInputOligomerTreeViewDlg::MzLabInputOligomerTreeViewDlg(QWidget *parent,
								const QString &name)
    : QDialog(parent), m_name(name)
  {
    Q_ASSERT(parent);

    mp_mzLabWnd = static_cast<MzLabWnd *>(parent);
        
    m_ui.setupUi(this);

    m_isFragment = false;
        
    m_ui.monoRadioButton->setChecked(true);
    m_massType = MXT_MASS_MONO;
    m_previousMassType = MXT_MASS_NONE;

    mpa_oligomerTreeViewModel = new MzLabInputOligomerTreeViewModel(this);
      
    m_ui.oligomerTreeView->setModel(mpa_oligomerTreeViewModel);
    mpa_oligomerTreeViewModel->setTreeView(m_ui.oligomerTreeView);

    m_ui.oligomerTreeView->setMzLabWnd(mp_mzLabWnd);
    m_ui.oligomerTreeView->setParentDlg(this);
    
    mpa_oligomerTreeViewModel->setMzLabWnd(mp_mzLabWnd);
    mpa_oligomerTreeViewModel->setParentDlg(this);

    // By default the oligomers we are going to deal with are not from
    // a fragmentation.
    m_ui.isFragmentCheckBox->setChecked(false);
    
    m_ui.nameLineEdit->setText(m_name);
    setWindowTitle(tr("massXpert: mz Lab - %1")
		    .arg(m_name));

    connect(m_ui.monoRadioButton,
	     SIGNAL(toggled(bool)),
	     this,
	     SLOT(monoMassRadioButtonToggled(bool)));

    connect(m_ui.avgRadioButton,
	     SIGNAL(toggled(bool)),
	     this,
	     SLOT(avgMassRadioButtonToggled(bool)));

    connect(m_ui.isFragmentCheckBox,
	    SIGNAL(stateChanged(int)),
	    this,
	    SLOT(isFragmentCheckBoxChanged(int)));
  }
  

  MzLabInputOligomerTreeViewDlg::~MzLabInputOligomerTreeViewDlg()
  {
    delete mpa_oligomerTreeViewModel;
  }


  void 
  MzLabInputOligomerTreeViewDlg::closeEvent(QCloseEvent *event)
  {
    QDialog::closeEvent(event);
  }
  

  void 
  MzLabInputOligomerTreeViewDlg::setName(const QString &name)
  {
    m_name = name;
    
    m_ui.nameLineEdit->setText(m_name);
    setWindowTitle(tr("massXpert: mz Lab - %1")
		    .arg(m_name));
  }
  

  QString
  MzLabInputOligomerTreeViewDlg::name()
  {
    return m_name;
  }


  const OligomerList *
  MzLabInputOligomerTreeViewDlg::oligomerList()
  {
    return mpa_oligomerTreeViewModel->oligomerList();
  }
  

  int 
  MzLabInputOligomerTreeViewDlg::duplicateOligomerData(const OligomerList *list)
  {
    Q_ASSERT(list);
    
    for (int iter = 0; iter < list->size(); ++iter)
      {
	if(m_isFragment)
	  {
	    FragmentOligomer * iterOligomer = 
	      dynamic_cast<FragmentOligomer *>(list->at(iter));
	    
	    FragmentOligomer * oligomer = new FragmentOligomer(*iterOligomer);
	    
	    mpa_oligomerTreeViewModel->addOligomer(oligomer);
	  }
	else
	  {
	    Oligomer *oligomer = new Oligomer(*list->at(iter));
	
	    mpa_oligomerTreeViewModel->addOligomer(oligomer);
	  }
      }
    
    return mpa_oligomerTreeViewModel->rowCount();
  }
  
    
  MassType
  MzLabInputOligomerTreeViewDlg::massType()
  {
    //    qDebug() << __FILE__ << __LINE__ << "m_massType:" << m_massType;
    return m_massType;
  }


  bool
  MzLabInputOligomerTreeViewDlg::isFragment()
  {
    return m_isFragment;
  }
  
  
  void
  MzLabInputOligomerTreeViewDlg::setFragment(bool fragment)
  {
    m_ui.isFragmentCheckBox->setChecked(fragment);
  }
  

  MassType
  MzLabInputOligomerTreeViewDlg::previousMassType()
  {
    return m_previousMassType;
  }


  void 
  MzLabInputOligomerTreeViewDlg::setPreviousMassType(MassType type)
  {
    m_previousMassType = type;
  }
  

  void
  MzLabInputOligomerTreeViewDlg::monoMassRadioButtonToggled(bool checked)
  {
    if (checked)
      m_massType = MXT_MASS_MONO;
    else
      m_massType = MXT_MASS_AVG;
  }
  
  
  void
  MzLabInputOligomerTreeViewDlg::avgMassRadioButtonToggled(bool checked)
  {
    if (checked)
      m_massType = MXT_MASS_AVG;
    else
      m_massType = MXT_MASS_MONO;
  }
  
  
  void
  MzLabInputOligomerTreeViewDlg::isFragmentCheckBoxChanged(int state)
  {
    if (state == Qt::Checked)
      m_isFragment = true;
    else
      m_isFragment = false;
  }
  

  void 
  MzLabInputOligomerTreeViewDlg::applyFormula(const Formula &formula,
					       bool inPlace)
  {
    const PolChemDef *polChemDef = mp_mzLabWnd->polChemDef();
    
    double mono = 0;
    double avg = 0;

    Formula local(formula);
        
    if (!local.accountMasses(polChemDef->atomList(), &mono, &avg))
      {
	QMessageBox::warning(this,
			      tr("massXpert: mz Lab"),
			      tr("%1@%2\n"
				  "Failed accounting for formula '%3'.")
			      .arg(__FILE__)
			      .arg(__LINE__)
			      .arg(formula.formula()));
	return;
      }
    
    if (inPlace)
      {
	// Just iterate in the oligomers and perform the computation.

	MzLabInputOligomerTreeViewModel *model =
	  static_cast<MzLabInputOligomerTreeViewModel *> 
	 (m_ui.oligomerTreeView->model());
	
	Q_ASSERT(model);
	
	int count = model->rowCount();

	if(!count)
	  return;
	
	for(int iter = 0; iter < count; ++iter)
	  {
	    // First the mass
	    QModelIndex index = 
	      model->index(iter, MZ_LAB_INPUT_OLIGO_MASS_COLUMN);

	    Q_ASSERT(index.isValid());
	    
	    MzLabInputOligomerTreeViewItem *item = 
	      static_cast<MzLabInputOligomerTreeViewItem *> 
	     (index.internalPointer());
	    
	    Oligomer *oligomer = item->oligomer();

	    FragmentOligomer *fragOligomer = 
	      dynamic_cast<FragmentOligomer *>(item->oligomer());
	    
	    int result = 0;
	    
	    if (fragOligomer)
	      result = fragOligomer->deionize();
	    else
	      result = oligomer->deionize();
	    
	    // The deionization does not change the oligomer's member
	    // IonizeRule, we do not need to store it. Further, we
	    // want to know if something happened during the
	    // deionize() call, as we'll have to reionize the oligomer
	    // after having incremented the mass.
	    
	    if (m_massType == MXT_MASS_MONO)
	      {
		oligomer->incrementMass(mono, m_massType);
		
		if(result == 1)
		  {
		    if (fragOligomer)
		      fragOligomer->ionize();
		    else
		      oligomer->ionize();
		  }
		
		item->setData(MZ_LAB_INPUT_OLIGO_MASS_COLUMN, 
			       QVariant(oligomer->mono()));
	      }
	    else
	      {
		oligomer->incrementMass(avg, m_massType);
		
		if(result == 1)
		  {
		    if (fragOligomer)
		      fragOligomer->ionize();
		    else
		      oligomer->ionize();
		  }
		
		item->setData(MZ_LAB_INPUT_OLIGO_MASS_COLUMN, 
			       QVariant(oligomer->avg()));
	      }
	    
	    model->dataChanged(index, index);
	  }
      }
    else
      {
	MzLabInputOligomerTreeViewModel *model =
	  static_cast<MzLabInputOligomerTreeViewModel *> 
	 (m_ui.oligomerTreeView->model());
	
	Q_ASSERT(model);
	
	MzLabInputOligomerTreeViewDlg *dlg = mp_mzLabWnd->newInputList();
	if(!dlg)
	  return;

	// The newly created dialog that it is opened to store
	// fragmentation oligomer data.
	if(m_isFragment)
	  dlg->setFragment(true);
		
	int count = dlg->duplicateOligomerData(model->oligomerList());
	
	if(count != model->oligomerList()->count())
	qFatal("Fatal error at %s@%d. Program aborted.",
		__FILE__, __LINE__);

	dlg->applyFormula(formula, true);
      }
  }
  
  void 
  MzLabInputOligomerTreeViewDlg::applyMass(double mass,
					    bool inPlace)
  {
    if (inPlace)
      {
	// Just iterate in the oligomers and perform the computation.
	
	MzLabInputOligomerTreeViewModel *model =
	  static_cast<MzLabInputOligomerTreeViewModel *> 
	 (m_ui.oligomerTreeView->model());
	
	Q_ASSERT(model);
	
	int count = model->rowCount();

	if(!count)
	  return;
	
	for(int iter = 0; iter < count; ++iter)
	  {
	    // First the mass
	    QModelIndex index = model->index(iter, 
					      MZ_LAB_INPUT_OLIGO_MASS_COLUMN);
	    Q_ASSERT(index.isValid());
	    
	    MzLabInputOligomerTreeViewItem *item = 
	      static_cast<MzLabInputOligomerTreeViewItem *> 
	     (index.internalPointer());
	    
	    Oligomer *oligomer = item->oligomer();
	    
	    FragmentOligomer *fragOligomer = 
	      dynamic_cast<FragmentOligomer *>(item->oligomer());
	    
	    int result = 0;
	    
	    if (fragOligomer)
	      result = fragOligomer->deionize();
	    else
	      result = oligomer->deionize();
	    
	    // The deionization does not change the oligomer's member
	    // IonizeRule, we do not need to store it. Further, we
	    // want to know if something happened during the
	    // deionize() call, as we'll have to reionize the oligomer
	    // after having incremented the mass.
	    
	    if (m_massType == MXT_MASS_MONO)
	      {
		oligomer->incrementMass(mass, m_massType);

		if(result == 1)
		  {
		    if (fragOligomer)
		      fragOligomer->ionize();
		    else
		      oligomer->ionize();
		  }
		
		item->setData(MZ_LAB_INPUT_OLIGO_MASS_COLUMN, 
			       QVariant(oligomer->mono()));
	      }
	    else
	      {
		oligomer->incrementMass(mass, m_massType);

		if(result == 1)
		  {
		    if (fragOligomer)
		      fragOligomer->ionize();
		    else
		      oligomer->ionize();
		  }

		item->setData(MZ_LAB_INPUT_OLIGO_MASS_COLUMN, 
			       QVariant(oligomer->avg()));
	      }
	    
	    model->dataChanged(index, index);
	  }
      }
    else
      {
	MzLabInputOligomerTreeViewModel *model =
	  static_cast<MzLabInputOligomerTreeViewModel *> 
	 (m_ui.oligomerTreeView->model());
	
	Q_ASSERT(model);
	
	MzLabInputOligomerTreeViewDlg *dlg = mp_mzLabWnd->newInputList();
	if(!dlg)
	  return;
	
	// The newly created dialog that it is opened to store
	// fragmentation oligomer data.
	if(m_isFragment)
	  dlg->setFragment(true);
		
	int count = dlg->duplicateOligomerData(model->oligomerList());
	
	if(count != model->oligomerList()->count())
	  qFatal("Fatal error at %s@%d. Program aborted.",
		  __FILE__, __LINE__);
	
	dlg->applyMass(mass, true);
      }
  }

  void 
  MzLabInputOligomerTreeViewDlg::applyThreshold(double threshold,
						 bool inPlace,
						 bool onMz)
  {
    // If onMz is false, then the threshold is to be applied on the
    // deionized oligomers' mass. Otherwise, the threshold is to be
    // applied to the m/z value of the ionized oligomer.
    
    if (inPlace)
      {
	// Just iterate in the oligomers and perform the computation.
	
	MzLabInputOligomerTreeViewModel *model =
	  static_cast<MzLabInputOligomerTreeViewModel *> 
	 (m_ui.oligomerTreeView->model());
	
	Q_ASSERT(model);
	
	if(!model->rowCount())
	  return;
	
	for(int iter = 0; iter < model->rowCount(); ++iter)
	  {
	    QModelIndex index = model->index(iter, 
					      MZ_LAB_INPUT_OLIGO_MASS_COLUMN);
	    Q_ASSERT(index.isValid());
	    
	    MzLabInputOligomerTreeViewItem *item = 
	      static_cast<MzLabInputOligomerTreeViewItem *> 
	     (index.internalPointer());
	    
	    Oligomer *oligomer = item->oligomer();
	    
	    if (onMz)
	      {
		// We do not have to deionize the oligomer, as we are
		// working on m/z values.
		
		if(oligomer->mass(m_massType) > threshold)
		  {
		    continue;
		  }
		else
		  {
		    // We have to remove the item.
		    model->removeOligomer(index);
		    --iter;
		  }
	      }
	    else
	      {
		// We first have to deionize the oligomer, because the
		// calculation is on M values. 
		
		// The deionization does not change the oligomer's
		// member IonizeRule, we do not need to store
		// it. Further, we want to know if something happened
		// during the deionize() call, as we'll have to
		// reionize the oligomer after having incremented the
		// mass.

		int result = oligomer->deionize();
		
		if(oligomer->mass(m_massType) > threshold)
		  {
		    if (result == 1)
		      oligomer->ionize();
		  }
		else
		  {
		    // We have to remove the item.
		    model->removeOligomer(index);
		    --iter;
		  }
	      }
	    // End 
	    // else(if(!onMz)
	  }
	// End
	// for (int iter = 0; iter < model->rowCount(); ++iter)
      }
    else
      {
	MzLabInputOligomerTreeViewModel *model =
	  static_cast<MzLabInputOligomerTreeViewModel *> 
	 (m_ui.oligomerTreeView->model());
	
	Q_ASSERT(model);
	
	MzLabInputOligomerTreeViewDlg *dlg = mp_mzLabWnd->newInputList();
	if(!dlg)
	  return;
	
	// The newly created dialog that it is opened to store
	// fragmentation oligomer data.
	if(m_isFragment)
	  dlg->setFragment(true);

	int count = dlg->duplicateOligomerData(model->oligomerList());
	
	if(count != model->oligomerList()->count())
	  qFatal("Fatal error at %s@%d. Program aborted.",
__FILE__, __LINE__);
	
	dlg->applyThreshold(threshold, true, onMz);
      }
  }
  
  
  void 
  MzLabInputOligomerTreeViewDlg::applyChargeIncrement(int increment, 
						       bool inPlace)
  {
    if (!increment)
      return;
    
    if (inPlace)
      {
	// Just iterate in the oligomers and perform the computation.
	
	MzLabInputOligomerTreeViewModel *model =
	  static_cast<MzLabInputOligomerTreeViewModel *> 
	 (m_ui.oligomerTreeView->model());
	
	Q_ASSERT(model);
	
	if(!model->rowCount())
	  return;
	
	for(int iter = 0; iter < model->rowCount(); ++iter)
	  {
	    QModelIndex index = model->index(iter, 
					      MZ_LAB_INPUT_OLIGO_MASS_COLUMN);
	    Q_ASSERT(index.isValid());
	    
	    MzLabInputOligomerTreeViewItem *item = 
	      static_cast<MzLabInputOligomerTreeViewItem *> 
	     (index.internalPointer());
	    
	    Oligomer *oligomer = item->oligomer();

	    FragmentOligomer *fragOligomer = 
	      dynamic_cast<FragmentOligomer *>(item->oligomer());
	    
	    int charge = 0;

	    if (fragOligomer)
	      charge = fragOligomer->charge();
	    else
	      charge = oligomer->charge();

	    if (charge == -1)
	      {
		QMessageBox::warning 
		 (this, 
		   tr("massXpert: mz Lab"),
		   tr("%1@%2\n"
		       "Failed to get oligomer's charge.")
		   .arg(__FILE__)
		   .arg(__LINE__),
		   QMessageBox::Ok);
		
		return;
	      }
	    
	    charge += increment;
	    
	    if (charge < 0)
	      charge = 0;
	    
	    int result = 0;
	    
	    if (fragOligomer)
	      result = fragOligomer->setCharge(charge);
	    else
	      result = oligomer->setCharge(charge);
	    		    

	    if (result == -1)
	      {
		QMessageBox::warning 
		 (this, 
		   tr("massXpert: mz Lab"),
		   tr("%1@%2\n"
		       "Failed to set oligomer's charge.")
		   .arg(__FILE__)
		   .arg(__LINE__),
		   QMessageBox::Ok);
		
		return;
	      }
	    
	    // Update the mass value depending on the mass type.
	    
	    if (m_massType == MXT_MASS_MONO)
	      {
		item->setData(MZ_LAB_INPUT_OLIGO_MASS_COLUMN, 
			       QVariant(oligomer->mono()));
	      }
	    else
	      {
		item->setData(MZ_LAB_INPUT_OLIGO_MASS_COLUMN, 
			       QVariant(oligomer->avg()));
	      }
	    
	    // Update the charge(irrespective of the mass type).

	    item->setData(MZ_LAB_INPUT_OLIGO_CHARGE_COLUMN, 
			   QVariant(oligomer->charge()));

	    model->dataChanged(index, index);
	  }
	// End
	// for (int iter = 0; iter < model->rowCount(); ++iter)
      }
    else
      {
	MzLabInputOligomerTreeViewModel *model =
	  static_cast<MzLabInputOligomerTreeViewModel *> 
	 (m_ui.oligomerTreeView->model());
	
	Q_ASSERT(model);
	
	MzLabInputOligomerTreeViewDlg *dlg = mp_mzLabWnd->newInputList();
	if(!dlg)
	  return;
	
	// The newly created dialog that it is opened to store
	// fragmentation oligomer data.
	if(m_isFragment)
	  dlg->setFragment(true);

	int count = dlg->duplicateOligomerData(model->oligomerList());
	
	if(count != model->oligomerList()->count())
	  qFatal("Fatal error at %s@%d. Program aborted.",
		  __FILE__, __LINE__);
	
	dlg->applyChargeIncrement(increment, true);
      }
  }    

    
  void 
  MzLabInputOligomerTreeViewDlg::applyIonizeRule(const IonizeRule &ionizeRule,
						  bool inPlace)
  {
    IonizeRule localIonizeRule = ionizeRule;
    
    PolChemDef *polChemDef = mp_mzLabWnd->polChemDef();

    if (!localIonizeRule.validate(polChemDef->atomList()))
      {
	QMessageBox::warning 
	 (this, 
	   tr("massXpert:mz Lab - Input list"),
	   tr("%1@%2\n"
	       "Failed to validate ionization rule.")
	   .arg(__FILE__)
	   .arg(__LINE__),
	   QMessageBox::Ok);

	return;
      }
    if (inPlace)
      {
	// Just iterate in the oligomers and perform the computation.
	
	MzLabInputOligomerTreeViewModel *model =
	  static_cast<MzLabInputOligomerTreeViewModel *> 
	 (m_ui.oligomerTreeView->model());
	
	Q_ASSERT(model);
	
	if(!model->rowCount())
	  return;
	
	for(int iter = 0; iter < model->rowCount(); ++iter)
	  {
	    QModelIndex index = model->index(iter, 
					      MZ_LAB_INPUT_OLIGO_MASS_COLUMN);
	    Q_ASSERT(index.isValid());
	    
	    MzLabInputOligomerTreeViewItem *item = 
	      static_cast<MzLabInputOligomerTreeViewItem *> 
	     (index.internalPointer());
	    
	    Oligomer *oligomer = item->oligomer();

	    // All we do is reionize the oligomer.

	    FragmentOligomer *fragOligomer = 
	      dynamic_cast<FragmentOligomer *>(item->oligomer());
	    
	    if (fragOligomer)
	      fragOligomer->ionize(localIonizeRule);
	    else
	      oligomer->ionize(localIonizeRule);

	    // Update the mass value depending on the mass type.
	    
	    if (m_massType == MXT_MASS_MONO)
	      {
		item->setData(MZ_LAB_INPUT_OLIGO_MASS_COLUMN, 
			       QVariant(oligomer->mono()));
	      }
	    else
	      {
		item->setData(MZ_LAB_INPUT_OLIGO_MASS_COLUMN, 
			       QVariant(oligomer->avg()));
	      }
	    
	    // Update the charge(irrespective of the mass type).

	    item->setData(MZ_LAB_INPUT_OLIGO_CHARGE_COLUMN, 
			   QVariant(oligomer->charge()));

	    model->dataChanged(index, index);
	  }
	// End
	// for (int iter = 0; iter < model->rowCount(); ++iter)
      }
    else
      {
	MzLabInputOligomerTreeViewModel *model =
	  static_cast<MzLabInputOligomerTreeViewModel *> 
	 (m_ui.oligomerTreeView->model());
	
	Q_ASSERT(model);
	
	MzLabInputOligomerTreeViewDlg *dlg = mp_mzLabWnd->newInputList();
	if(!dlg)
	  return;
	
	// The newly created dialog that it is opened to store
	// fragmentation oligomer data.
	if(m_isFragment)
	  dlg->setFragment(true);

	int count = dlg->duplicateOligomerData(model->oligomerList());
	
	if(count != model->oligomerList()->count())
	  qFatal("Fatal error at %s@%d. Program aborted.",
		  __FILE__, __LINE__);
	
	dlg->applyIonizeRule(localIonizeRule, true);
      }
  }    
  
} // namespace massXpert
