/* matchbox - a lightweight window manager

   Copyright 2002 Matthew Allum

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
*/

#include "client_common.h"

/*
   Common functions for use by all client types
*/

void
client_set_state(Client *c, int state)
{
   CARD32 data[2];

   data[0] = state;
   data[1] = None;
   
   XChangeProperty(c->wm->dpy, c->window, c->wm->wm_state, c->wm->wm_state,
		   32, PropModeReplace, (unsigned char *)data, 2);
}

long
client_get_state(Client *c)
{
    Atom real_type; int real_format;
    unsigned long items_read, items_left;
    long *data, state = WithdrawnState;

    if (XGetWindowProperty(c->wm->dpy, c->window,
			   c->wm->wm_state, 0L, 2L, False,
			   c->wm->wm_state, &real_type, &real_format,
			   &items_read, &items_left,
			   (unsigned char **) &data) == Success
	&& items_read) {
       state = *data;
       XFree(data);
    }
    return state;
}


void
client_deliver_config(Client *c)
{
   XConfigureEvent ce;
   
   ce.type = ConfigureNotify;
   ce.event = c->window;
   ce.window = c->window;
   
   ce.x = c->x;
   ce.y = c->y;
   
   ce.width  = c->width;
   ce.height = c->height;
   ce.border_width = 0;
   ce.above = None;
   ce.override_redirect = 0;
   
   XSendEvent(c->wm->dpy, c->window, False,
	      StructureNotifyMask, (XEvent *)&ce);
}

void
client_deliver_delete(Client *c)
{
    int i, n, found = 0;
    Atom *protocols;
    XEvent e;
    
    if (XGetWMProtocols(c->wm->dpy, c->window, &protocols, &n)) {
        for (i=0; i<n; i++)
	   if (protocols[i] == c->wm->wm_delete) found++;
        XFree(protocols);
    }
    if (found)
    {
       e.type = ClientMessage;
       e.xclient.window = c->window;
       e.xclient.message_type = c->wm->wm_protos;
       e.xclient.format = 32;
       e.xclient.data.l[0] = c->wm->wm_delete;
       e.xclient.data.l[1] = CurrentTime;
       XSendEvent(c->wm->dpy, c->window, False, NoEventMask, &e);
    }
    else XKillClient(c->wm->dpy, c->window);
}

/* ----- new bits ---------*/

int
client_want_focus(Client *c)
{
   /* TODO: check _WM protocols too ? */
   int ret = 1;
   XWMHints *hints = XGetWMHints(c->wm->dpy, c->window);
   if (hints != NULL)
      if (!(hints->input)) ret = 0; 
   XFree(hints);
   return ret;
}

Client*
client_get_next(Client* c, Client_type wanted)
{
   client *p;
   for (p=c->next; p != c; p = p->next)
      if (p->type == wanted && p->mapped) return p;
   return c;
}

Client*
client_get_prev(Client* c, Client_type wanted)
{
   client *p;
   for (p=c->prev; p != c; p = p->prev)
      if (p->type == wanted && p->mapped) return p;
   return c;
}

void
client_init_backing(Client* c, int width, int height)
{
   c->backing =
   XCreatePixmap(c->wm->dpy, c->wm->root, width, height ,
		 DefaultDepth(c->wm->dpy,DefaultScreen(c->wm->dpy)));

   /* todo check for error if pixmap cant be created */
#ifdef USE_XFT
   c->xftdraw = XftDrawCreate(c->wm->dpy, (Drawable) c->backing, 
			       DefaultVisual(c->wm->dpy,
					     DefaultScreen(c->wm->dpy)), 
			       DefaultColormap(c->wm->dpy,
					       DefaultScreen(c->wm->dpy)));
#endif
   
}

void
client_morph(Client *c, Client_type new_type)
{
   dbg("MIGHTY MORPHING RANGERS GO GO GO !!!!");
   switch(new_type)
   {
      case mainwin :
	 c->type = mainwin;
	 base_client_set_funcs(c);
	 c->reparent     = &main_client_reparent;
	 c->redraw       = &main_client_redraw;
	 c->button_press = &main_client_button_press;
	 c->move_resize  = &main_client_move_resize;
	 c->get_coverage = &main_client_get_coverage;
	 c->hide         = &main_client_hide;
	 c->show         = &main_client_show;
	 c->configure    = &main_client_configure;
	 c->destroy      = &main_client_destroy;
	 break;
      case toolbar :
	 c->type = toolbar;
	 base_client_set_funcs(c);
	 c->configure    = &toolbar_client_configure;
	 c->reparent     = &toolbar_client_reparent;
	 c->button_press = &toolbar_client_button_press;
	 c->redraw       = &toolbar_client_redraw;
	 c->hide         = &toolbar_client_hide;
	 c->show         = &toolbar_client_show;
	 c->move_resize  = &toolbar_client_move_resize;
	 c->get_coverage = &toolbar_client_get_coverage;
	 c->destroy      = &toolbar_client_destroy;
	 client_set_state(c,WithdrawnState); 
	 break;
      case docked:
	 dbg("BECOMIN A DOCK");
	 c->type = docked;
	 base_client_set_funcs(c);
	 c->reparent     = &docked_client_reparent;
	 c->configure    = &docked_client_configure;
	 c->show         = &docked_client_show;
	 c->hide         = &docked_client_hide;
	 c->destroy      = &docked_client_destroy;
	 c->move_resize  = &docked_client_move_resize;
	 break;
      case dialog :
      case override :
      case menu:
      case detached:
	 return;
   }
   c->configure(c);
   c->reparent(c);
   c->move_resize(c);
   client_deliver_config(c);
   c->show(c);
}
