""" The cookie manager manages all dynamic cookies. """



# Enthought libraries
from enthought.traits.api import SingletonHasTraits, Dict

# local imports
from cookie import Cookie
from enthought.envisage import get_application
from enthought.envisage.resource import ResourceManager


class CookieManager(SingletonHasTraits):
    """ The cookie manager manages all dynamic cookies. """

    #### 'CookieManager' interface ############################################

    # Cookies bound at the type level.
    type_cookies = Dict # {Any type : List(Class) cookies }

    # Cookies bound at the instance level.
    instance_cookies = Dict

    ###########################################################################
    # 'CookieManager' interface.
    ###########################################################################

    def get_cookie(self, cid, obj):
        """ Get a cookie of the specified type for an object or its type.

        Returns None if no such cookie is found.

        """
        

        resource_manager = get_application().service_registry.get_service(
            'enthought.envisage.resource.IResourceManager'
        )
        
        # Get instance-scope cookies.
        instance_cookies = self.instance_cookies.get(id(obj), {})
        if cid in instance_cookies:
            cookie = instance_cookies[cid]

        # Otherwise, we don't have an instance-scope cookie,  so look for a
        # type-scope cookie.
        else:
            # Determine the object's type.
            type = resource_manager.get_type_of(obj)
            if type is not None:
                # Get type-scope cookies.
                type_cookies = self.type_cookies.get(type, {})
                if cid in type_cookies:
                    cookie = type_cookies[cid]

                else:
                    cookie = None

            else:
                cookie = None


        return cookie
        
    def add_type_cookie(self, cid, cookie, type):
        """ Add a type-scope cookie.
            cid is a subclass of Cookie,
            cookie is a subclass of Cookie
            type is a subclass of ResourceType
        """
        
        type_to_cookie_map = self.type_cookies.setdefault(type, {})
        type_to_cookie_map[cid] = cookie

        return

    def add_instance_cookie(self, cid, cookie, obj):
        """ Add a instance specific cookie. """
        
        instance_to_cookie_map = self.instance_cookies.setdefault(id(obj), {})
        instance_to_cookie_map[cid] = cookie

        return

    def remove_type_cookie(self, cid, type):
        """ Remove a type-specific cookie.
            cid is a subclass of Cookie,
            type is a subclass of ResourceType
        """
        
        type_to_cookie_map = self.type_cookies.get(type, {})
        del type_to_cookie_map[cid]
        
        return
        
    def remove_instance_cookie(self, cid, obj):
        """ Remove an instance-specific cookie. """
        
        instance_to_cookie_map = self.instance_cookies.get(id(obj), {})
        del instance_to_cookie_map[cid]

        return
        

    ###########################################################################
    # Debugging interface.
    ###########################################################################

    def dump(self):
        """ Dumps the contents of the manager to stdout. """

        print 'Cookie Manager', id(self)

        for type, cookies in self.type_cookies.items():
            print 'Type:', type.id
            for cookie in cookies:
                print '    ', cookie.__name__

        print
        
        for obj_id, cookies in self.instance_cookies.items():
            print 'Instance:', obj_id
            for cookie in cookies:
                print '    ', cookie.__name__

        return

#### EOF ######################################################################
