#-----------------------------------------------------------------------------
#
#  Copyright (c) 2005, 2006 by Enthought, Inc.
#  All rights reserved.
#
#-----------------------------------------------------------------------------

"""
An action to save the current project.

"""

# Standard library imports.
import logging

# Enthought library imports.
from enthought.envisage.single_project.project_action import ProjectAction


# Setup a logger for this module.
logger=logging.getLogger(__name__)


class SaveAction(ProjectAction):
    """
    An action to save the current project.  This is only enabled when
    there is a current project and that project is saveable.

    """

    ##########################################################################
    # Attributes
    ##########################################################################

    #### public 'ObjectAction' interface #####################################

    # The universal object locator (UOL).
    uol = 'service://enthought.envisage.single_project.UiService'

    # The name of the method to invoke on the object.
    method_name = 'save'


    ##########################################################################
    # 'ProjectAction' interface
    ##########################################################################

    #### public interface ####################################################

    def refresh(self):
        """
        Refresh the enabled state of this action.

        This implementation enables the action when there is a current project
        which is marked as saveable.

        """

        self.enabled = self._refresh_project_exists() and \
            self._refresh_is_save_allowed()

        return


    #### trait handlers ######################################################

    def _on_project_changed(self, obj, trait_name, old, new):
        """
        Handle changes to the value of the current project.

        Extended to ensure that we listen for changes to the saveable flag on
        the current project.

        """

        if old is not None:
            self._update_project_listeners(old, remove=True)
        if new is not None:
            self._update_project_listeners(new, remove=False)

        super(SaveAction, self)._on_project_changed(obj, trait_name, old, new)


    ##########################################################################
    # 'SaveAction' interface
    ##########################################################################

    #### protected interface #################################################

    def _refresh_is_save_allowed(self):
        """
        Return the refresh state according to whether the current project is
        marked as saveable.

        Returns True if the action should be enabled and False otherwise.

        """

        return self.model_service.project.is_save_allowed


    def _update_project_listeners(self, project, remove):
        """
        Update listeners on the specified project.

        """

        logger.debug( (remove and 'Removing ' or 'Adding ') + \
            'listeners on project [%s] for SaveAction [%s]', project, self)

        project.on_trait_change(self._on_is_save_allowed, 'is_save_allowed',
            remove=remove)

        return


    #### trait handlers ######################################################

    def _on_is_save_allowed(self, obj, trait_name, old, new):
        """
        Handle changes to the value of the current project's is_save_allowed.

        """

        self.refresh()

        return


#### EOF #####################################################################
