/* ======================================================================
 * meshio.hh
 * 
 * This file is part of MeshIO, the general and extensible 3D mesh I/O
 * library.
 * Copyright (c) 1999, 2000 Niklas Elmqvist. All rights reserved.
 *
 * File created 1999-06-05 by Niklas Elmqvist <d97elm@dtek.chalmers.se>.
 * 
 * The 3Dwm Project
 *      <http://www.medialab.chalmers.se/projects/3dwm>
 * Chalmers Medialab
 *      <http://www.medialab.chalmers.se>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 *
 * ======================================================================
 */

#ifndef _MESHIO_HH_
#define _MESHIO_HH_

// Need to export functions on Win32
#if !defined(APIEXPORT)
#if !defined(WIN32)
#define APIEXPORT
#else
#define APIEXPORT __declspec(dllexport)
#endif 
#endif

// -- Defines
#define MIO_RAW_VERSION		"0.3"		// Raw file version number


// -- Namespace Definitions

namespace MeshIO {

    // -- Enumerated types
    
    /**
     * MeshIO result type.
     **/
    typedef enum _mioResult {
	MIO_NO_ERROR = 0,	// No error encountered
	MIO_FILE_NOT_FOUND,	// File cannot be found
	MIO_SYNTAX_ERROR,	// Something is wrong with the file format
	MIO_READ_ERROR,		// An error occurred while reading
	MIO_WRITE_ERROR,	// An error occurred while writing
	MIO_NO_FILETYPE		// No file type found
    } mioResult;

    // typedef enum { MIO_POLYMESH, MIO_LINEMESH, MIO_POINTMESH } MeshType;
    
    // -- Struct definitions
    
    /**
     * Simple 2D vertex struct.
     **/
    struct Vertex2D {

	/// Spatial coordinate anonymous union
	union {
	    
	    /// Named 2D coordinate form.
	    struct {
		float x, y;			
	    } coord;
	    
	    /// Array 2D coordinate form
	    float coords[2];		
	};
	
    };
    
    /**
     * Simple 3D vertex struct.
     **/
    struct Vertex3D {
	
	/// Spatial coordinate anonymous union
	union {

	    /// Named 3D coordinate form
	    struct {
		float x, y, z;
	    } coord;
	    
	    /// Array 3D coordinate form
	    float coords[3];
	};
    };
    
    /**
     * 3-point index struct.
     **/
    struct Index {

	/// Index anonymous union
	union {

	    /// Named index form
	    struct {
		int a, b, c;
	    } ndx;

	    /// Array index form
	    int ndxs[3];
	};

    };

    /**
     * Simple face (triangle) struct.
     **/
    struct Face {
	
	/// Vertex indices
	Index ndxVertex;
	
	/// Mapping indices
	Index ndxMapping;
	
	/// Material index
	int material;
    };

    /**
     * Mesh struct. One mesh represents a single object in a mesh set.
     **/
    struct Mesh {
    
	/// Number of vertices in mesh
	int vertexNum;
	
	/// Number of faces in mesh
	int faceNum;
	
	/// Number of mapping coords in mesh
	int mappingNum;
	
	/// Model vertices
	Vertex3D *vertexList;
	
	/// Model normals
	Vertex3D *normalList;

	/// Model mapping coordinates
	Vertex2D *mappingList;
	
	/// Model faces (triangles)
	Face *faceList;	
    };

    /**
     * Material data struct.
     **/
    struct Material {

	/// Material name
	char *name;
	
	/// Texture image filename
	char *texFile;
	
	/// Ambient material characteristics
	float ambient[3];

	/// Specular material characteristics
	float specular[3];

	/// Diffuse material characteristics
	float diffuse[3];
    };

    /**
     * Mesh list node.
     **/
    typedef struct _MeshNode {
	/// Contained mesh
	Mesh *mesh;
	
	/// Next node in list
	struct _MeshNode *next;	
    } MeshNode;

    /**
     * Mesh set (contains mesh data, materials and textures).
     **/
    struct MeshSet {

	/// List of meshes in set
	MeshNode *list;

	/// List of materials in set
	Material *materialList;	
	
	/// Number of meshes in set
	int meshNum;
	
	/// Number of materials in set
	int materialNum;
    };


    // -- Function Prototypes


    // == General MeshIO Functions ============

    /**
     * Retrieve a MeshIO error string given a result.
     **/
    APIEXPORT const char *mioError(mioResult result);


    // == Mesh Functions ============

    /// Mesh initialization.
    APIEXPORT void mioMeshInit(Mesh *mesh);

    /// Normal computation function.
    APIEXPORT void mioMeshComputeNormals(Mesh *mesh);

    /// Geometry clearing function.
    APIEXPORT void mioMeshClear(Mesh *mesh);

    /// Mesh copying.
    APIEXPORT Mesh *mioMeshCopy(Mesh *mesh);

    /// Dump the contents of the mesh into human-readable form.
    APIEXPORT void mioMeshDump(Mesh *mesh, bool verbose);

    /// Mesh vertex data scaling function.
    APIEXPORT void mioMeshScale(Mesh *mesh, 
				float xScale, float yScale, float zScale);
    
    // == MeshSet Functions ============

    /// Mesh set initialization.
    APIEXPORT void mioMSetInit(MeshSet *mset);

    /// Normal computation function.
    APIEXPORT void mioMSetComputeNormals(MeshSet *mset);

    /// Mesh addition function (adds a new mesh to the list)
    APIEXPORT void mioMSetAddMesh(MeshSet *mset, Mesh *mesh);

    /// Mesh removal function (removes a mesh from the list if found)
    APIEXPORT void mioMSetDelMesh(MeshSet *mset, Mesh *mesh);

    /// Set clearing function
    APIEXPORT void mioMSetClear(MeshSet *mset);

    /// Set copying function
    APIEXPORT MeshSet *mioMSetCopy(MeshSet *mset);

    /// Clear all meshes in set
    APIEXPORT void mioMSetClearMeshes(MeshSet *mset);

    /// Clear all materials in set
    APIEXPORT void mioMSetClearMaterials(MeshSet *mset);

    /// Find a material index in a mesh set given a name
    APIEXPORT int mioMSetFindMaterial(MeshSet *mset, char *name);

    /// Compute the number of faces found in a set
    APIEXPORT int mioMSetComputeFaceNum(MeshSet *mset);

    /// Compute the number of vertices found in a set
    APIEXPORT int mioMSetComputeVertexNum(MeshSet *mset);

    /// Dump the contents of the set into human-readable form
    APIEXPORT void mioMSetDump(MeshSet *mset, bool verbose = false);

    /// Scale the vertex data of the mesh set
    APIEXPORT void mioMSetScale(MeshSet *mset, 
				float fScaleX, float fScaleY, float fScaleZ);


    // == Material Functions ============

    /// Initialize material
    APIEXPORT void mioMaterialInit(Material *mat);

    /// Clear material
    APIEXPORT void mioMaterialClear(Material *mat);

    /// Copy material
    APIEXPORT void mioMaterialCopy(Material *dst, Material *src);


    // == General File Functions ========

    /// Load a 3D file (figure out filetype by the file extension)
    APIEXPORT mioResult mioLoad(const char *filename, MeshSet *mset);


    // == 3DS File Functions ============

    /// .3ds-file loading function (returns a pointer to a MeshList vector)
    APIEXPORT mioResult mio3DSLoad(const char *filename, MeshSet *mset);


    // == Raw File Functions ============

    /// Raw loading function
    APIEXPORT mioResult mioRawLoad(const char *filename, MeshSet *mset);

    /// Raw saving function
    APIEXPORT mioResult mioRawSave(const char *filename, const MeshSet *mset);

};

#endif /* meshio.hh */
