# -------------------------------------------------------------------------
#     This file is part of mMass - the spectrum analysis tool for MS.
#     Copyright (C) 2005-07 Martin Strohalm <mmass@biographics.cz>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 2 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE in the
#     main directory of the program
# -------------------------------------------------------------------------

# Function: Select MS scan from the list of all scans.

# load libs
import wx
import string

# load modules
from nucleus import mwx


class dlgSelectScan(wx.Dialog):
    """Select one scan from multiscan files"""

    # ----
    def __init__(self, parent, scans):
        wx.Dialog.__init__(self, parent, -1, "Select Scan", style=wx.DEFAULT_DIALOG_STYLE|wx.RESIZE_BORDER)

        self.scans = scans
        self.selectedScan = 0

        # make gui items
        self.makeScanList()
        buttons = self.makeButtonBox()

        # pack main frame
        mainSizer = wx.BoxSizer(wx.VERTICAL)
        if wx.Platform == '__WXMAC__':
            mainSizer.Add(self.scan_list, 1, wx.EXPAND|wx.TOP|wx.LEFT|wx.RIGHT, 20)
            mainSizer.Add(buttons, 0, wx.ALL|wx.ALIGN_CENTER, 10)
        else:
            mainSizer.Add(self.scan_list, 1, wx.EXPAND|wx.ALL, 3)
            mainSizer.Add(buttons, 0, wx.ALL|wx.ALIGN_CENTER, 3)

        # fit layout
        mainSizer.Fit(self)
        self.SetSizer(mainSizer)
        self.SetMinSize(self.GetSize())
        self.Centre()
    # ----


    # ----
    def makeScanList(self):
        """ Make list of imported scans. """

        # make list
        self.scan_list = scanList(self, self.scans)
        self.scan_list.SetToolTip(wx.ToolTip("Select scan to import"))

        # set events
        self.scan_list.Bind(wx.EVT_LIST_ITEM_SELECTED, self.onItemSelected)
        self.scan_list.Bind(wx.EVT_LIST_ITEM_ACTIVATED, self.onLMDC)
    # ----


    # ----
    def makeButtonBox(self):
        """ Make main dialog buttons. """

        # make items
        self.Import_button = wx.Button(self, wx.ID_OK, "Import")
        Cancel_button = wx.Button(self, wx.ID_CANCEL, "Cancel")

        # pack items
        buttonBox = wx.BoxSizer(wx.HORIZONTAL)
        buttonBox.Add(self.Import_button, 0, wx.ALL, 5)
        buttonBox.Add(Cancel_button, 0, wx.ALL, 5)

        # set defaults
        self.Import_button.Enable(False)

        return buttonBox
    # ----


    # ----
    def onItemSelected(self, evt):
        """ Enable 'Import' button if scan selected. """

        # get selected item index
        line = evt.m_itemIndex
        if line != -1:
            self.selectedScan = self.scans[line][0]
            self.Import_button.Enable(True)
        else:
            self.Import_button.Enable(False)
    # ----


    # ----
    def onLMDC(self, evt):
        """ Import scan on double-click. """

        # get selected item index
        line = evt.m_itemIndex
        if line != -1:
            self.selectedScan = self.scans[line][0]
            self.EndModal(wx.ID_OK)
        else:
            self.Import_button.Enable(False)
    # ----


class scanList(mwx.ListCtrl):
    """ Make virtual list of imported scans. """

    # ----
    def __init__(self, parent, data):

        # set style
        if wx.Platform == '__WXMAC__':
            style = wx.LC_REPORT|wx.LC_VIRTUAL|wx.LC_SINGLE_SEL|wx.LC_VRULES|wx.LC_HRULES|wx.SIMPLE_BORDER
        else:
            style = wx.LC_REPORT|wx.LC_VIRTUAL|wx.LC_SINGLE_SEL|wx.LC_VRULES|wx.LC_HRULES

        # init object
        mwx.ListCtrl.__init__(self, parent, -1, size=(500, 250), style=style)

        # init data
        self.data = data

        # make columns
        self.InsertColumn(0, "Scan", wx.LIST_FORMAT_RIGHT)
        self.InsertColumn(1, "Ret. Time", wx.LIST_FORMAT_RIGHT)
        self.InsertColumn(2, "Mass Range", wx.LIST_FORMAT_CENTER)
        self.InsertColumn(3, "Points", wx.LIST_FORMAT_RIGHT)
        self.InsertColumn(4, "MS Level", wx.LIST_FORMAT_CENTER)
        self.InsertColumn(5, "Precursor", wx.LIST_FORMAT_RIGHT)
        self.InsertColumn(6, "Charge", wx.LIST_FORMAT_RIGHT)
        self.InsertColumn(7, "Spectrum Type", wx.LIST_FORMAT_LEFT)

        self.SetItemCount(len(self.data))

        # set columns width
        for col in range(8):
            self.SetColumnWidth(col, wx.LIST_AUTOSIZE_USEHEADER)
        self.updateLastCol()
    # ----


    # ----
    def OnGetItemText(self, item, col):
        """ Paste data to virtual list. """

        return self.data[item][col]
    # ----