# -------------------------------------------------------------------------
#     This file is part of mMass - the spectrum analysis tool for MS.
#     Copyright (C) 2005-07 Martin Strohalm <mmass@biographics.cz>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 2 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE in the
#     main directory of the program
# -------------------------------------------------------------------------

# Function: Update application menubar, toolbar and statusbar.

# load libs
import wx

# load modules
from ids import *


class controls:
    """ Update application menubar, toolbar and statusbar. """

    # ----
    def __init__(self):
        self.menuBar = None
        self.toolBar = None
        self.statusBar = None
    # ----


    # ----
    def setAppStatus(self, status=False):
        """ Change cursor and show status info in main part of statusbar. """

        if status:
            self.statusBar.setMain(status)
            wx.BeginBusyCursor()
        else:
            self.statusBar.setMain(0)
            wx.EndBusyCursor()
    # ----


    # ----
    def setStatusBar(self, part, value):
        """ Set specified value to selected part of statusbar. """

        if part == 'main':
            self.statusBar.setMain(value)
        elif part == 'masstype':
            self.statusBar.setMassType(value)
        elif part == 'tolerance':
            self.statusBar.setTolerance(value)
        elif part == 'peaklist':
            self.statusBar.setPeaklistInfo(value)
        elif part == 'sequence':
            self.statusBar.setSequenceInfo(value)
        elif part == 'spectrum':
            self.statusBar.setSpectrumInfo(value)
    # ----


    # ----
    def enableAllItems(self, status=False):
        """ Enable/disable all menu and toolbar items. """

        # menu sections (not for mac)
        self.menuBar.EnableTop(1, status)
        self.menuBar.EnableTop(2, status)
        self.menuBar.EnableTop(3, status)
        self.menuBar.EnableTop(4, status)
        self.menuBar.EnableTop(5, status)
        self.menuBar.EnableTop(6, status)

        # file
        self.menuBar.Enable(ID_fileSave, status)
        self.menuBar.Enable(ID_fileSaveAs, status)
        self.menuBar.Enable(ID_fileClose, status)
        self.menuBar.Enable(ID_fileInfo, status)
        self.menuBar.Enable(ID_fileExport, status)
        self.menuBar.Enable(ID_filePrint, status)
        self.menuBar.Enable(ID_fileImportSequence, status)
        self.toolBar.EnableTool(ID_fileSave, status)
        self.toolBar.EnableTool(ID_fileClose, status)
        self.toolBar.EnableTool(ID_filePrintSpectrum, status)
        self.toolBar.EnableTool(ID_filePrintReport, status)
        self.toolBar.EnableTool(ID_fileInfo, status)
        self.toolBar.EnableTool(ID_fileImportSequence, status)

        # peaklist
        self.menuBar.Enable(ID_peaklistUndo, status)
        self.menuBar.Enable(ID_peaklistRedo, status)
        self.menuBar.Enable(ID_peaklistAddPeak, status)
        self.menuBar.Enable(ID_peaklistSelectAllPeaks, status)
        self.menuBar.Enable(ID_peaklistSelectAnnotatedPeaks, status)
        self.menuBar.Enable(ID_peaklistSelectByTreshold, status)
        self.menuBar.Enable(ID_peaklistInvertSelection, status)
        self.menuBar.Enable(ID_peaklistClearAllAnnots, status)
        self.menuBar.Enable(ID_peaklistDeleteAllPeaks, status)
        self.menuBar.Enable(ID_peaklistCopyToClipboard, status)
        self.menuBar.Enable(ID_peaklistRefresh, status)
        self.toolBar.EnableTool(ID_peaklistAddPeak, status)
        self.toolBar.EnableTool(ID_peaklistClearAllAnnots, status)
        self.toolBar.EnableTool(ID_peaklistDeleteAllPeaks, status)

        # spectrum
        self.menuBar.Enable(ID_spectrumLabelPoint, status)
        self.menuBar.Enable(ID_spectrumLabelPeak, status)
        self.menuBar.Enable(ID_spectrumDeleteLabel, status)
        self.menuBar.Enable(ID_spectrumMeasureDistance, status)
        self.menuBar.Enable(ID_spectrumAddToCompare, status)
        self.menuBar.Enable(ID_spectrumSetRange, status)
        self.menuBar.Enable(ID_spectrumViewOptions, status)
        self.menuBar.Enable(ID_spectrumRefresh, status)
        self.toolBar.EnableTool(ID_spectrumLabelPoint, status)
        self.toolBar.EnableTool(ID_spectrumLabelPeak, status)
        self.toolBar.EnableTool(ID_spectrumDeleteLabel, status)
        self.toolBar.EnableTool(ID_spectrumMeasureDistance, status)

        # processing
        self.menuBar.Enable(ID_processingUndo, status)
        self.menuBar.Enable(ID_processingInternalCalibration, status)
        self.menuBar.Enable(ID_processingStatisticalCalibration, status)
        # self.menuBar.Enable(ID_processingBaseline, status)
        # self.menuBar.Enable(ID_processingSmooth, status)
        self.menuBar.Enable(ID_processingCrop, status)
        self.toolBar.EnableTool(ID_processingInternalCalibration, status)
        self.toolBar.EnableTool(ID_processingStatisticalCalibration, status)

        # matching
        self.menuBar.Enable(ID_matchingMatchData, status)
        self.menuBar.Enable(ID_matchingMatchInfo, status)
        self.menuBar.Enable(ID_matchingAnnotate, status)
        self.toolBar.EnableTool(ID_matchingMatchData, status)
        self.toolBar.EnableTool(ID_matchingMatchInfo, status)
        self.toolBar.EnableTool(ID_matchingAnnotate, status)

        # sequence
        self.menuBar.Enable(ID_sequenceEditMod, status)
        self.menuBar.Enable(ID_sequenceSearchMass, status)
        self.menuBar.Enable(ID_sequenceSearchSeq, status)
        self.toolBar.EnableTool(ID_sequenceEditMod, status)
        self.toolBar.EnableTool(ID_sequenceSearchMass, status)
        self.toolBar.EnableTool(ID_sequenceSearchSeq, status)

        # tools
        self.menuBar.Enable(ID_toolsMSpec, status)
        self.menuBar.Enable(ID_toolsMSeq, status)
        self.menuBar.Enable(ID_toolsMCut, status)
        self.menuBar.Enable(ID_toolsMFrag, status)
        self.menuBar.Enable(ID_toolsMDiff, status)
        self.menuBar.Enable(ID_toolsMComp, status)
        self.menuBar.Enable(ID_toolsMMascotPMF, status)
        self.menuBar.Enable(ID_toolsMMascotSeq, status)
        self.menuBar.Enable(ID_toolsMMascotMSMS, status)
        self.menuBar.Enable(ID_toolsMConv, status)
        self.toolBar.EnableTool(ID_toolsMSpec, status)
        self.toolBar.EnableTool(ID_toolsMSeq, status)
        self.toolBar.EnableTool(ID_toolsMCut, status)
        self.toolBar.EnableTool(ID_toolsMFrag, status)
        self.toolBar.EnableTool(ID_toolsMDiff, status)
        self.toolBar.EnableTool(ID_toolsMComp, status)
        self.toolBar.EnableTool(ID_toolsMMascotPMF, status)
        self.toolBar.EnableTool(ID_toolsMMascotSeq, status)
        self.toolBar.EnableTool(ID_toolsMMascotMSMS, status)

        # config
        self.menuBar.Enable(ID_configMonoisotopic, status)
        self.menuBar.Enable(ID_configAverage, status)
        self.menuBar.Enable(ID_configTolerance, status)
        self.menuBar.Enable(ID_configShowPeaklist, status)
        self.menuBar.Enable(ID_configPeaklistBottom, status)
        self.toolBar.EnableTool(ID_configMonoisotopic, status)
        self.toolBar.EnableTool(ID_configAverage, status)
        self.toolBar.EnableTool(ID_configTolerance, status)
        self.toolBar.EnableTool(ID_configShowPeaklist, status)
        self.toolBar.EnableTool(ID_configPeaklistBottom, status)

        # clear statusbar
        if not status:
            self.setStatusBar('masstype', None)
            self.setStatusBar('tolerance', None)
            self.setStatusBar('peaklist', None)
            self.setStatusBar('sequence', None)
            self.setStatusBar('spectrum', None)
    # ----


    # ----
    def enableFileSave(self, status=False):
        """ Enable/disable save function. """

        self.menuBar.Enable(ID_fileSave, status)
        self.toolBar.EnableTool(ID_fileSave, status)
    # ----


    # ----
    def enablePrintingTools(self, spectrum=None, peaklist=None):
        """ Enable/disable printing. """

        if spectrum != None:
            self.menuBar.Enable(ID_filePrintSpectrum, spectrum)
            self.toolBar.EnableTool(ID_filePrintSpectrum, spectrum)
        if peaklist != None:
            self.menuBar.Enable(ID_filePrintReport, peaklist)
            self.toolBar.EnableTool(ID_filePrintReport, peaklist)
        if peaklist:
            self.menuBar.Enable(ID_filePrintSpectrum, True)
            self.toolBar.EnableTool(ID_filePrintSpectrum, True)
    # ----


    # ----
    def enablePeaklistTools(self, status=False):
        """ Enable/disable peaklist tools. """

        self.menuBar.Enable(ID_fileExportPeaklist, status)
        self.menuBar.Enable(ID_peaklistRefresh, status)
        self.menuBar.Enable(ID_peaklistClearAllAnnots, status)
        self.menuBar.Enable(ID_peaklistDeleteAllPeaks, status)
        self.menuBar.Enable(ID_peaklistSelectAllPeaks, status)
        self.menuBar.Enable(ID_peaklistSelectAnnotatedPeaks, status)
        self.menuBar.Enable(ID_peaklistSelectByTreshold, status)
        self.menuBar.Enable(ID_peaklistInvertSelection, status)
        self.menuBar.Enable(ID_peaklistCopyToClipboard, status)
        self.menuBar.Enable(ID_toolsMMascotPMF, status)
        self.menuBar.Enable(ID_toolsMMascotSeq, status)
        self.menuBar.Enable(ID_toolsMMascotMSMS, status)
        self.toolBar.EnableTool(ID_peaklistClearAllAnnots, status)
        self.toolBar.EnableTool(ID_peaklistDeleteAllPeaks, status)
        self.toolBar.EnableTool(ID_toolsMMascotPMF, status)
        self.toolBar.EnableTool(ID_toolsMMascotSeq, status)
        self.toolBar.EnableTool(ID_toolsMMascotMSMS, status)
    # ----


    # ----
    def enablePeaklistUndoRedo(self, undo=False, redo=False):
        """ Enable/disable peaklist undo/redo. """

        self.menuBar.Enable(ID_peaklistUndo, undo)
        self.menuBar.Enable(ID_peaklistRedo, redo)
    # ----


    # ----
    def enableSpectrumTools(self, spectrum=False, peaklist=False):
        """ Enable/disable spectrum tools. """

        if spectrum:
            export = True
            point = True
            peak = True
            delete = True
            distance = True
            compare = True
            setrange = True
            refresh = True
        elif peaklist:
            export = False
            point = False
            peak = False
            delete = True
            distance = True
            compare = False
            setrange = True
            refresh = True
        else:
            export = False
            point = False
            peak = False
            delete = False
            distance = False
            compare = False
            setrange = False
            refresh = False

        self.menuBar.Enable(ID_fileExportSpectrumData, export)
        self.menuBar.Enable(ID_spectrumLabelPoint, point)
        self.menuBar.Enable(ID_spectrumLabelPeak, peak)
        self.menuBar.Enable(ID_spectrumDeleteLabel, delete)
        self.menuBar.Enable(ID_spectrumMeasureDistance, distance)
        self.menuBar.Enable(ID_spectrumAddToCompare, compare)
        self.menuBar.Enable(ID_spectrumSetRange, setrange)
        self.menuBar.Enable(ID_spectrumRefresh, refresh)
        self.toolBar.EnableTool(ID_spectrumLabelPoint, point)
        self.toolBar.EnableTool(ID_spectrumLabelPeak, peak)
        self.toolBar.EnableTool(ID_spectrumDeleteLabel, delete)
        self.toolBar.EnableTool(ID_spectrumMeasureDistance, distance)
        self.toolBar.EnableTool(ID_spectrumRefresh, refresh)
    # ----


    # ----
    def enableProcessingUndo(self, processing):
        """ Enable/disable processing undo. """
        self.menuBar.Enable(ID_processingUndo, processing)
    # ----


    # ----
    def enableProcessingTools(self, spectrum=False, peaklist=False):
        """ Enable/disable processing tools. """

        if spectrum and peaklist:
            intcalib = True
            statcalib = True
            baseline = False
            smooth = False
            crop = True
        elif spectrum:
            intcalib = False
            statcalib = False
            baseline = False
            smooth = False
            crop = True
        elif peaklist:
            intcalib = True
            statcalib = True
            baseline = False
            smooth = False
            crop = False
        else:
            mancalib = False
            intcalib = False
            statcalib = False
            baseline = False
            smooth = False
            crop = False

        self.menuBar.Enable(ID_processingInternalCalibration, intcalib)
        self.menuBar.Enable(ID_processingStatisticalCalibration, statcalib)
        # self.menuBar.Enable(ID_processingBaseline, baseline)
        # self.menuBar.Enable(ID_processingSmooth, smooth)
        self.menuBar.Enable(ID_processingCrop, crop)
        self.toolBar.EnableTool(ID_processingInternalCalibration, intcalib)
        self.toolBar.EnableTool(ID_processingStatisticalCalibration, statcalib)
    # ----


    # ----
    def enableMatchingTools(self, module=False, data=False, matched=False):
        """ Enable/disable matching tools. """

        if not module in ('mCut', 'mFrag', 'mComp'):
            data = False
            match = False

        self.menuBar.Enable(ID_matchingMatchData, data)
        self.menuBar.Enable(ID_matchingMatchInfo, matched)
        self.menuBar.Enable(ID_matchingAnnotate, matched)
        self.toolBar.EnableTool(ID_matchingMatchData, data)
        self.toolBar.EnableTool(ID_matchingMatchInfo, matched)
        self.toolBar.EnableTool(ID_matchingAnnotate, matched)
    # ----


    # ----
    def enableSequenceTools(self, status=False):
        """ Enable/disable sequence tools. """

        self.menuBar.Enable(ID_sequenceEditMod, status)
        self.menuBar.Enable(ID_sequenceSearchMass, status)
        self.menuBar.Enable(ID_sequenceSearchSeq, status)
        self.toolBar.EnableTool(ID_sequenceEditMod, status)
        self.toolBar.EnableTool(ID_sequenceSearchMass, status)
        self.toolBar.EnableTool(ID_sequenceSearchSeq, status)
    # ----


    # ----
    def toggleModule(self, module=False):
        """ Toggle module button. """

        if module == 'mSpec':
            self.toolBar.ToggleTool(ID_toolsMSpec, True)
        elif module == 'mSeq':
            self.toolBar.ToggleTool(ID_toolsMSeq, True)
        elif module == 'mCut':
            self.toolBar.ToggleTool(ID_toolsMCut, True)
        elif module == 'mFrag':
            self.toolBar.ToggleTool(ID_toolsMFrag, True)
        elif module == 'mDiff':
            self.toolBar.ToggleTool(ID_toolsMDiff, True)
        elif module == 'mComp':
            self.toolBar.ToggleTool(ID_toolsMComp, True)
    # ----


    # ----
    def toggleSpectrumFce(self, fce):
        """ Toggle spectrum mouse functions. """

        if fce == 'point':
            self.menuBar.Check(ID_spectrumLabelPoint, True)
            self.toolBar.ToggleTool(ID_spectrumLabelPoint, True)
        elif fce == 'peak':
            self.menuBar.Check(ID_spectrumLabelPeak, True)
            self.toolBar.ToggleTool(ID_spectrumLabelPeak, True)
        elif fce == 'delete':
            self.menuBar.Check(ID_spectrumDeleteLabel, True)
            self.toolBar.ToggleTool(ID_spectrumDeleteLabel, True)
        elif fce == 'distance':
            self.menuBar.Check(ID_spectrumMeasureDistance, True)
            self.toolBar.ToggleTool(ID_spectrumMeasureDistance, True)
    # ----


    # ----
    def toggleMassType(self, massType):
        """ Toggle mass-type. """

        # set menubar and toolbar
        if massType == 'amass':
            self.menuBar.Check(ID_configAverage, True)
            self.toolBar.ToggleTool(ID_configAverage, True)
        else:
            self.menuBar.Check(ID_configMonoisotopic, True)
            self.toolBar.ToggleTool(ID_configMonoisotopic, True)

        # set statusbar
        self.statusBar.setMassType(massType)
    # ----


    # ----
    def toggleSplitWindow(self, status=True):
        """ Toggle window splitting. """

        self.menuBar.Check(ID_configShowPeaklist, status)
        self.menuBar.Enable(ID_configPeaklistBottom, status)
        self.toolBar.ToggleTool(ID_configShowPeaklist, status)
        self.toolBar.EnableTool(ID_configPeaklistBottom, status)
    # ----


    # ----
    def toggleSplittingType(self, splitting='v'):
        """ Toggle window splitting type. """

        if splitting == 'h':
            self.menuBar.Check(ID_configPeaklistBottom, True)
            self.toolBar.ToggleTool(ID_configPeaklistBottom, True)
        elif splitting == 'v':
            self.menuBar.Check(ID_configPeaklistBottom, False)
            self.toolBar.ToggleTool(ID_configPeaklistBottom, False)
        else:
            self.menuBar.Enable(ID_configPeaklistBottom, False)
            self.toolBar.EnableTool(ID_configPeaklistBottom, False)
    # ----


    # ----
    def updateFileHistory(self, history):
        """ Update file history in main menu. """

        IDs = (
                ID_fileRecentFiles0,
                ID_fileRecentFiles1,
                ID_fileRecentFiles2,
                ID_fileRecentFiles3,
                ID_fileRecentFiles4,
                ID_fileRecentFiles5,
                ID_fileRecentFiles6,
                ID_fileRecentFiles7,
                ID_fileRecentFiles8,
                ID_fileRecentFiles9
                )

        for x in range(10):
            if not history[x]:
                self.menuBar.Enable(IDs[x], False)
            else:
                label = '%d: %s' % (x, history[x])
                self.menuBar.SetLabel(IDs[x], label)
                self.menuBar.Enable(IDs[x], True)
    # ----
