###################################################################
# mmchat.tcl - MudMaster-compatible chat system
# 
# Copyright (C) 2003 Kurt Hutchinson
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
###################################################################

##
# This module implements a chat system compatible with the protocol
# used by the MudMaster mud client.
##

namespace eval mmchat {

    # These are our accessible commands.
    namespace export call chat chatcfg chatconn chatgroupset \
                     chatgroupremove chatlist chatsnoop dnd \
                     emote gchat gemote pchat peek pemote \
                     ping requestconnects sendfile unchat

    ##
    # These are the command constants MudMaster uses to identify different
    # chat commands. NOTE: Keep this array definition above everything else
    # because immediately after it is an array transform that depends on
    # only these constants being present in the array.
    ##
    if {[array exists MMChat]}  {unset MMChat}
    array set MMChat {
        cmd,NAME_CHANGE                1
        cmd,REQUEST_CONNECTIONS        2
        cmd,CONNECTION_LIST            3
        cmd,TEXT_EVERYBODY             4
        cmd,TEXT_PERSONAL              5
        cmd,TEXT_GROUP                 6
        cmd,MESSAGE                    7
        cmd,DO_NOT_DISTURB             8
        cmd,SEND_ACTION                9
        cmd,SEND_ALIAS                 10
        cmd,SEND_MACRO                 11
        cmd,SEND_VARIABLE              12
        cmd,SEND_EVENT                 13
        cmd,SEND_GAG                   14
        cmd,SEND_HIGHLIGHT             15
        cmd,SEND_LIST                  16
        cmd,SEND_ARRAY                 17
        cmd,SEND_BARITEM               18
        cmd,VERSION                    19
        cmd,FILE_START                 20
        cmd,FILE_DENY                  21
        cmd,FILE_BLOCK_REQUEST         22
        cmd,FILE_BLOCK                 23
        cmd,FILE_END                   24
        cmd,FILE_CANCEL                25
        cmd,PING_REQUEST               26
        cmd,PING_RESPONSE              27
        cmd,PEEK_CONNECTIONS           28
        cmd,PEEK_LIST                  29
        cmd,SNOOP                      30
        cmd,SNOOP_DATA                 31
        cmd,SNOOP_COLOR                32
        cmd,SEND_SUBSTITUTE            33

        cmd,END_OF_COMMAND             255
    }
    foreach constant [array names MMChat] {
        set MMChat($constant) [format %c $MMChat($constant)]
    }

    # File transfer constants
    array set MMChat {
        file,NONE       0
        file,SEND       1
        file,RECEIVE    2
        file,BLOCKSIZE  500
    }


    # Chat name: must be set before connections can be attempted
    set MMChat(cfg,chatname) ""

    # If you know your own IP address, and it isn't detected correctly
    # set MMChat(cfg,ip) [getip]
    # Note: This is actually set down below where [getip] is defined.

    # Port the chat server is listening on for connections
    set MMChat(cfg,listenport) 4050

    # Output the given connection's name before any text they send?
    set MMChat(cfg,paranoia) 0

    # Color used for notification messages: should follow the syntax
    # used by [color]
    set MMChat(cfg,notifycolor) {bold yellow}

    # Color used for chat messages: should follow the syntax used by [color]
    set MMChat(cfg,chatcolor) {bold red}

    # Path do place transferred files
    set MMChat(cfg,filepath) [pwd]

    # Line prefix for snoop data
    set MMChat(cfg,snoopprefix) [color {>>} {bold green}]
}


###### Accessory Commands ######
# These commands are not meant to be used directly, but are used as
# helpers to the main user-intended commands.
# Arranged alphabetically.

##
# Params: connection to add, name of remote chatter on this connection,
#         remote ip address, remote listen port
# Return: none
#
# Given a connection and name, sets up this connection in the chatlist,
# providing all the defaults for other values.
##
proc mmchat::addconn {conn name ip port} {
    variable MMChatList

    set MMChatList($conn,name)  $name
    set MMChatList($conn,ip)    $ip
    set MMChatList($conn,port)  $port

    foreach option {
        private ignore allowxfers serve version xferstate file \
        filelen byteswritten xferstart peek reqconn
    } {
        set MMChatList($conn,$option) 0
    }

    set MMChatList($conn,groups) ""
    set MMChatList($conn,filename) ""
}

##
# Params: notification message
# Return: none
#
# Takes a message and outputs it using the configured chat notification
# colors.
##
proc mmchat::chatnotify {mesg} {
    variable MMChat
    echo [color $mesg $MMChat(cfg,notifycolor)]
}

##
# Params: message to print
# Return: none
#
# Take a message and outputs it using the configured chat print colors.
# This differs from chatnotify in that chatnotify is meant for mainly
# informational messages, while this is meant for chat messages from
# other people.
##
proc mmchat::chatprint {mesg} {
    variable MMChat
    echo [color $mesg $MMChat(cfg,chatcolor)]
}

##
# Params: connection of transfer
# Return: none
#
# Performs cleanup of the transfer on the given connection.
##
proc mmchat::closexfer {conn} {
    variable MMChatList

    close $MMChatList($conn,file)
    set MMChatList($conn,filename) ""
    foreach option {file xferstate filelen byteswritten xferstart} {
        set MMChatList($conn,$option) 0
    }

    return
}

##
# Params: name of a connection
# Return: connection id
#
# Given the name of a connection, find and return the actual connection
# id. This gives direct access to the connection and lets connection
# information be easily looked up in the MMChatList array.
# An error will be raised if the name is not found.
##
proc mmchat::findname {name} {
    variable MMChatList

    foreach namekey [array names MMChatList *,name] {
        if {[string tolower $MMChatList($namekey)] eq [string tolower $name]} {
            return [lindex [split $namekey ","] 0]
        }
    }

    error "$name not found in chat list"
}

##
# Params: none
# Return: ip address
#
# Retrieve this machine's IP address and return it. This needs to be sent
# in the initial chat request.
##
proc mmchat::getip {} {
    set ip 0

    proc getip&close {sock ipin port} {
        upvar ip myip
        set myip $ipin
        close $sock
    }

    set ipserver [socket -server getip&close 0]
    set port [lindex [fconfigure $ipserver -sockname] 2]
    close [socket [info hostname] $port]
    while {$ip eq "0"} {
        update
    }
    close $ipserver

    return $ip
}
# Set the default IP address
namespace eval mmchat {
    set MMChat(cfg,ip) [getip]
}

##
# Params: none
# Return: none
#
# Notify all non-ignored connections of a name change.
##
proc mmchat::notifynamechange {} {
    variable MMChat
    variable MMChatList

    append mesg $MMChat(cmd,NAME_CHANGE) \
                $MMChat(cfg,chatname) \
                $MMChat(cmd,END_OF_COMMAND)

    foreach ignorekey [array names MMChatList *,ignore] {
        if {!$MMChatList($ignorekey)} {
            set conn [lindex [split $ignorekey ","] 0]
            puts -nonewline $conn $mesg
        }
    }
}

##
# Params: connection to remove
# Return: none
#
# Given a connection, removes that connection's info from the chatlist.
##
proc mmchat::remconn {conn} {
    variable MMChatList

    foreach key [array names MMChatList $conn*] {
        unset MMChatList($key)
    }
}

##
# Params: none
# Return: none
#
# Restarts the listen server. This will probably be called due to a change
# in the listenport.
# Note: As a special case, if the listenport is set to 0, the server will
# be disabled.
##
proc mmchat::restartserver {} {
    variable MMChat

    if {[info exists MMChat(server)]} {
        close $MMChat(server)
    }
    if {$MMChat(cfg,listenport) eq "0"} {
        chatnotify "Listen server disabled."
        catch {unset MMChat(server)}
        return
    }
    startserver
    chatnotify "Listen server restarted using port $MMChat(cfg,listenport)."

    return
}

##
# Params: none
# Return: none
#
# Attempt to bring up the listening server on the currently defined
# listenport.
##
proc mmchat::startserver {} {
    variable MMChat

    if {[
            catch {
                set MMChat(server) [
                    socket -server mmchat::accept $MMChat(cfg,listenport)
                ]
            } errormsg
        ]} {
        append errormsg "\ntry changing your listenport: " \
                        "[config set script_char]chatcfg listenport num"
        error $errormsg
    }
}

##
# Params: user message, chat message
# Return: none
#
# Displays the user message to the user (ie 'You chat to everybody...') and
# sends the chat message out to all non-ignored connections.
##
proc mmchat::text_to_all {usermesg chatmesg} {
    variable MMChat
    variable MMChatList

    chatprint "\n$usermesg"

    append chattext $MMChat(cmd,TEXT_EVERYBODY) "\n$chatmesg" \
                    $MMChat(cmd,END_OF_COMMAND)

    foreach ignorekey [array names MMChatList *,ignore] {
        if {!$MMChatList($ignorekey)} {
            set conn [lindex [split $ignorekey ","] 0]
            puts -nonewline $conn $chattext
        }
    }

    return
}

##
# Params: user message, chat group, chat message
# Return: none
#
# Displays the user message to the user (ie, 'You chat to <groupname>...')
# and sends the chat message to all connections in the chat group.
##
proc mmchat::text_to_group {usermesg group chatmesg} {
    variable MMChat
    variable MMChatList

    chatprint "\n$usermesg"

    append chattext $MMChat(cmd,TEXT_GROUP) [format "%-15s" $group] \
                    "\n$chatmesg" $MMChat(cmd,END_OF_COMMAND)
    foreach groupkey [array names MMChatList *,groups] {
        set conn [lindex [split $groupkey ","] 0]
        if {[string match "*$group,*" $MMChatList($groupkey)] && \
            !$MMChatList($conn,ignore)} {
            puts -nonewline $conn $chattext
        }
    }

    return
}

##
# Params: user message, name, chat message
# Return: none
#
# Displays the user message to the user (ie, 'You chat to Someone...') and
# sends the chat message to the named connection.
##
proc mmchat::text_to_name {usermesg name chatmesg} {
    variable MMChat
    variable MMChatList

    chatprint "\n$usermesg"

    append chattext $MMChat(cmd,TEXT_PERSONAL) "\n$chatmesg" \
                    $MMChat(cmd,END_OF_COMMAND)

    set conn [findname $name]
    puts -nonewline $conn $chattext

    return
}


###### User Commands ######
# These are the actual commands that are exported and intended to be used
# directly. They comprise the user interface for this module.
# Arranged alphabetically.

##
# Params: remote ip address or hostname, remote port
# Return: none
#
# Connect to a remote chatter, send initial request and wait for response.
##
proc mmchat::call {address {port 4050}} {
    variable MMChat

    if {$MMChat(cfg,chatname) eq ""} {
        error "You need to set your chatname first: [config set script_char]chatcfg chatname name"
        return
    }

    set request [
        format "CHAT:%s\n%s%-5u" $MMChat(cfg,chatname) \
               $MMChat(cfg,ip) $MMChat(cfg,listenport)
    ]

    chatnotify [format "Connecting to %s:%u..." $address $port]

    set conn [socket $address $port]
    fconfigure $conn -blocking 0 -buffering none -translation {binary binary}
    puts -nonewline $conn $request

    fileevent $conn readable [list mmchat::handshake_out $conn $address $port]

    return
}

##
# Params: message to chat
# Return: none
#
# Chats the given message to all non-ignored chat connections.
##
proc mmchat::chat {args} {
    variable MMChat

    set mesg [join $args]
    text_to_all "You chat to everybody, '$mesg'" \
                "$MMChat(cfg,chatname) chats to everybody, '$mesg'"
    return
}

##
# Params: none to get a printout of current option values
#         OR
#         option name alone to print current value or with a value
#         to set it
# Return: none
#
# With no arguments, a listing of the configuration options with their
# current values is printed. With just the name of an option, the current
# value of that option is printed. With an option name and a value, the
# named option is set to the given value.
##
proc mmchat::chatcfg {args} {
    variable MMChat

    if {[llength $args] == 0} {
        # List current options and values
        
        # Find column widths
        set maxoptlen 6
        set maxvallen 5
        foreach optkey [array names MMChat cfg*] {
            if {([string length $optkey] - 4) > $maxoptlen} {
                set maxoptlen [expr [string length $optkey] - 4]
            }
            if {[string length $MMChat($optkey)] > $maxvallen} {
                set maxvallen [string length $MMChat($optkey)]
            }
        }

        # Column headers
        chatnotify [
            format "%-${maxoptlen}s %-${maxvallen}s\n%s %s" "Option" \
                   "Value" [string repeat "=" $maxoptlen] \
                   [string repeat "=" $maxvallen]
        ]

        # Print out the options and values
        set options [array names MMChat cfg*]
        foreach optionkey [lsort -dictionary $options] {
            chatnotify [
                format "%-${maxoptlen}s %-s" \
                       [lindex [split $optionkey ","] 1] \
                       $MMChat($optionkey)
            ]
        }
        echo

    } elseif {[llength $args] == 1} {
        # List value for a given option

        set option [string tolower [lindex $args 0]]
        if {![info exists MMChat(cfg,$option)]} {
            error "no such chat configuration option"
        }
        chatnotify [
            format "%s: %s\n" $option $MMChat(cfg,$option)
        ]

    } elseif {[llength $args] == 2} {
        # Set the value of a given option

        set option [string tolower [lindex $args 0]]
        set value  [lindex $args 1]
        if {![info exists MMChat(cfg,$option)]} {
            error "no such chat configuration option"
        }
        set MMChat(cfg,$option) $value
        chatnotify "<CHAT> $option is now '$value'.\n"

        # Option-specific actions taken when value is changed
        switch -exact -- $option {
            chatname { notifynamechange }
            listenport { restartserver }
        }

    }

    return
}

##
# Params: name of connection, option name
# Return: none
#
# This is similar to [chatcfg] but for connection specific options.
# Toggles the option given for the named connectin.
#
# Options
# =======
# private - whether or not to show connection in peek or connections
#           requests
# ignore - ignore connection when receiving or sending text
# allowxfers - allow file transfers from this connection
# serve - any public chats received are echoed to served connections,
#         any public chats send by a served connection are echoed to all
##
proc mmchat::chatconn {name option} {
    variable MMChat
    variable MMChatList

    set conn [findname $name]
    set option [string tolower $option]
    if {![info exists MMChatList($conn,$option)]} {
        error "no such connection configuration option"
    }
    
    set MMChatList($conn,$option) [expr 1 - $MMChatList($conn,$option)]
    set value $MMChatList($conn,$option)

    # Build notification messages
    append mesg $MMChat(cmd,MESSAGE) "\n<CHAT> $MMChat(cfg,chatname) "
    append usermesg "$MMChatList($conn,name) is no" \
        [expr {$value ? "w" : " longer"}]
    
    switch -exact -- $option {
        private {
            append mesg "has marked you as " \
                [expr {$value ? "private." : "public."}]
            append usermesg " marked as private.\n"
        }

        ignore {
            append usermesg " being ignored."
        }

        allowxfers {
            append mesg "is no" [expr {$value ? "w" : " longer"}] \
                " accepting files from you."
            append usermesg " allowed to send files."
        }

        serve {
            append mesg "is no" [expr {$value ? "w" : " longer"}] \
                " serving you."
            append usermesg " being served."
        }
    }

    # Don't inform people when they're being ignored
    if {$option ne "ignore"} {
        append mesg $MMChat(cmd,END_OF_COMMAND)
        puts -nonewline $conn $mesg
    }
    
    chatnotify $usermesg

    return
}

##
# Params: name of connection, name of group
# Return: none
#
# Adds the given connection to the given chat group.
##
proc mmchat::chatgroupset {name group} {
    variable MMChatList

    set conn [findname $name]
    if {![string match "*$group,*" $MMChatList($conn,groups)]} {
        append MMChatList($conn,groups) "$group,"
    }

    chatnotify "$MMChatList($conn,name) added to <$group>.\n"

    return
}

##
# Params: name of connection, name of group
# Return: none
#
# Removes the given connection from the given chat group.
##
proc mmchat::chatgroupremove {name group} {
    variable MMChatList

    set conn [findname $name]
    regsub "$group," $MMChatList($conn,groups) "" MMChatList($conn,groups)

    chatnotify "$MMChatList($conn,name) removed from <$group>.\n"

    return
}

##
# Params: none
# Return: none
#
# Prints out a list of current connections and info.
##
proc mmchat::chatlist {} {
    variable MMChatList

    set maxnamelen 4
    foreach namekey [array names MMChatList *,name] {
        if {[string length $MMChatList($namekey)] > $maxnamelen} {
            set maxnamelen [string length $MMChatList($namekey)]
        }
    }
    set maxgrouplen 6
    foreach groupkey [array names MMChatList *,groups] {
        if {[string length $MMChatList($groupkey)] > $maxgrouplen} {
            set maxgrouplen [string length $MMChatList($groupkey)]
        }
    }

    set layout "%-${maxnamelen}s %-15s %-5s %-${maxgrouplen}s %-5s"
    chatnotify "Chat connections:"
    chatnotify [
        format $layout "Name" "Address" "Port" "Groups" "Flags"
    ]
    chatnotify [
        format $layout [string repeat "=" $maxnamelen] \
               "===============" "=====" \
               [string repeat "=" $maxgrouplen] "====="
    ]

    foreach namekey [array names MMChatList *,name] {
        set conn [lindex [split $namekey ","] 0]
        set flags ""
        append flags [expr {$MMChatList($conn,ignore)?"I":" "}] \
                     [expr {$MMChatList($conn,private)?"P":" "}] \
                     [expr {$MMChatList($conn,serve)?"S":" "}] \
                     [expr {$MMChatList($conn,xferstate)?"T": \
                            $MMChatList($conn,allowxfers)?"t":" "}]
        chatnotify [
            format $layout $MMChatList($namekey) $MMChatList($conn,ip) \
                   $MMChatList($conn,port) $MMChatList($conn,groups) $flags
        ]
    }

    chatnotify "\nFlags: I - Ignore, P - Private, S - Serving"
    chatnotify   "       t - Allow file transfers, T - Currently transferring"
    echo

    return
}

##
# Params: name of connection
# Return: none
#
# Toggle snooping the named connection.
##
proc mmchat::chatsnoop {name} {
    variable MMChat
    
    set conn [findname $name]
    append mesg $MMChat(cmd,SNOOP) $MMChat(cmd,END_OF_COMMAND)
    puts -nonewline $conn $mesg

    return
}

##
# Params: none
# Return: none
#
# Convenience function to toggle operation of the listen server.
##
proc mmchat::dnd {} {
    variable MMChat

    if {[info exists MMChat(server)]} {
        close $MMChat(server)
        catch {unset MMChat(server)}
        chatnotify "Listen server disabled.\n"
    } else {
        startserver
        chatnotify "Listen server enabled on port $MMChat(cfg,listenport).\n"
    }
}

##
# Params: message
# Return: none
#
# Emotes the message to all non-ignored connections.
##
proc mmchat::emote {args} {
    variable MMChat

    set mesg [join $args]
    text_to_all "You emote to everybody: $MMChat(cfg,chatname) $mesg" \
                "$MMChat(cfg,chatname) $mesg"
    return
}

##
# Params: name of group, chat message
# Return: none
#
# Chats the given message only to those connections in the given group.
##
proc mmchat::gchat {group args} {
    variable MMChat

    set mesg [join $args]
    text_to_group "You chat to <$group>, '$mesg'" $group \
                  "$MMChat(cfg,chatname) chats to <$group>, '$mesg'"
    return
}

##
# Params: name of group, message
# Return: none
#
# Emotes the message to connections in the given group.
##
proc mmchat::gemote {group args} {
    variable MMChat

    set mesg [join $args]
    text_to_group "You emote to <$group>: $MMChat(cfg,chatname) $mesg" \
                  $group "$MMChat(cfg,chatname) $mesg"
    return
}

##
# Params: name of connection, message
# Return: none
#
# Chats the given message only to the named connection.
##
proc mmchat::pchat {name args} {
    variable MMChat
    variable MMChatList

    set mesg [join $args]
    text_to_name "You chat to $MMChatList([findname $name],name), '$mesg'" \
                 $name "$MMChat(cfg,chatname) chats to you, '$mesg'"
    return
}

##
# Params: name of connection to peek
# Return: none
#
# Asks the remote client for a list of their public connections.
##
proc mmchat::peek {name} {
    variable MMChat

    set conn [findname $name]
    append request $MMChat(cmd,PEEK_CONNECTIONS) $MMChat(cmd,END_OF_COMMAND)
    puts -nonewline $conn $request

    set MMChatList($conn,peek) 1

    return
}

##
# Params: name of connections, message
# Return: none
#
# Emotes the message to the named connection.
##
proc mmchat::pemote {name args} {
    variable MMChat
    variable MMChatList

    set mesg [join $args]
    text_to_name \
        "You emote to $MMChatList([findname $name],name): $MMChat(cfg,chatname) $mesg" \
        $name "$MMChat(cfg,chatname) $mesg"
}

##
# Params: name of connection
# Return: none
#
# Pings the named connection. The data is just some measure of time that will
# be sent back and we can use to make a difference calculation later.
##
proc mmchat::ping {name} {
    variable MMChat

    set conn [findname $name]
    append request $MMChat(cmd,PING_REQUEST) \
                   [clock clicks -milliseconds] \
                   $MMChat(cmd,END_OF_COMMAND)
    puts -nonewline $conn $request

    return
}

##
# Params: name of connection
# Return: non
#
# Sends a request to the named connection for a list of their public
# connections.
##
proc mmchat::requestconnects {name} {
    variable MMChat

    set conn [findname $name]
    append request $MMChat(cmd,REQUEST_CONNECTIONS) $MMChat(cmd,END_OF_COMMAND)
    puts -nonewline $conn $request

    set MMChatList($conn,reqconn) 1

    return
}

##
# Params: name of connection, filename
# Return: none
#
# Sends a file-transfer-start request to the named connection. If successful,
# the remote client will begin sending file block requests back and the
# transfer will proceed.
##
proc mmchat::sendfile {name file} {
    variable MMChat
    variable MMChatList

    set conn [findname $name]
    if {$MMChatList($conn,xferstate) != $MMChat(file,NONE)} {
        chatnotify "There is already a tranfer in progress.\n"
        return
    }

    set filename [file join $MMChat(cfg,filepath) $file]
    set MMChatList($conn,file) [open $filename r]
    set MMChatList($conn,filename) $filename
    set MMChatList($conn,xferstate) $MMChat(file,SEND)
    set MMChatList($conn,filelen) [file size $filename]
    set MMChatList($conn,byteswritten) 0
    set MMChatList($conn,xferstart) [clock seconds]

    append mesg $MMChat(cmd,FILE_START) $file "," [file size $filename] \
                $MMChat(cmd,END_OF_COMMAND)
    puts -nonewline $conn $mesg

    return
}

##
# Params: name of connection
# Return: none
#
# Disconnects the named connection.
##
proc mmchat::unchat {name} {
    variable MMChatList

    set conn [findname $name]
    set name $MMChatList($conn,name)

    remconn $conn
    close $conn

    chatnotify "$name disconnected.\n"

    return
}


###### Auto-processing Commands ######
# These commands should be called automatically when a new connection is
# made or new data comes in on a connection.
# Except for [accept], [processchat], and the handshake commands, these
# are arranged in the order of the huge switch in [processchat] that
# handles chat command dispatch.

##
# Params: new connection, address, port
# Return: none
#
# Preliminary connection setup. Accept the connection and wait for the
# handshaking info to be sent. This prevents the app from locking when a
# connection is made and sends no data.
##
proc mmchat::accept {conn address port} {
    fconfigure $conn -blocking 0 -buffering none -translation {binary binary}
    fileevent $conn readable [list mmchat::handshake_in $conn]
    return
}

##
# Params: new connection
# Return: none
#
# Performs handshaking and setup of new incoming chat connections.
##
proc mmchat::handshake_in {conn} {
    variable MMChat

    chatnotify "\n<CHAT> Negotiating chat request..."

    if {[eof $conn]} {
        chatnotify "Chat failed: connection closed.\n"
        close $conn
        return
    }

    append request [read $conn 100]

    if {[string range $request 0 4] ne "CHAT:"} {
        chatnotify "Chat failed: invalid request.\n"
        close $conn
        return
    }

    set name [
        string trim [string range $request 5 [string first "\n" $request 5]]
    ]
    if {![string length $name]} {
        chatnotify "Chat failed: no chat name.\n"
        close $conn
        return
    }

    set remoteip [
        string range $request [expr [string first "\n" $request] + 1] end-5
    ]
    set remoteport [string trim [string range $request end-4 end]]

    chatnotify "Chat request from $name accepted.\n"
    puts -nonewline $conn "YES:$MMChat(cfg,chatname)\n"

    # Give them a client description
    puts -nonewline $conn [
        format "%sMmucl %s%s" \
               $MMChat(cmd,VERSION) [mmucl version] $MMChat(cmd,END_OF_COMMAND)
    ]

    addconn $conn $name $remoteip $remoteport
    fileevent $conn readable [list mmchat::processchat $conn]

    return
}

##
# Params: new connection, its address, and its port
# Return: none
#
# Performs handshaking and setup of new outgoing chat connections.
##
proc mmchat::handshake_out {conn address port} {
    variable MMChat

    set response [read $conn 2]
    if {[string range $response 0 1] eq "NO"} {
        chatnotify "Chat request denied.\n"
        close $conn
        return
    }

    append response [read $conn 2]
    if {$response ne "YES:"} {
        chatnotify "Unknown remote client. Connection closed."
        close $conn
        return
    }

    chatnotify "Chat request accepted.\n"

    set name ""
    while {[string range $name end end] ne "\n"} {
        append name [read $conn 1]
    }

    # Give them a client description
    puts -nonewline $conn [
        format "%sMmucl %s%s" \
               $MMChat(cmd,VERSION) [mmucl version] $MMChat(cmd,END_OF_COMMAND)
    ]

    addconn $conn [string trim $name] $address $port
    fileevent $conn readable [list mmchat::processchat $conn]
}

##
# Params: connection with new data
# Return: none
#
# Given a connection with new data in the socket, read the data in and process
# it as chat commands.
##
proc mmchat::processchat {conn} {
    variable MMChat
    variable MMChatList

    set chatbuf [read $conn 5000]

    if {[eof $conn]} {
        chatnotify "<CHAT> Connection with $MMChatList($conn,name) lost.\n"
        remconn $conn
        close $conn
        return
    }

    set endidx [string first $MMChat(cmd,END_OF_COMMAND) $chatbuf]
    while {$endidx >= 0} {
        set cmd [string index $chatbuf 0]
        set data [string range $chatbuf 1 [expr $endidx - 1]]

        switch -- $cmd \
            $MMChat(cmd,NAME_CHANGE) {
                namechange $conn $data
            } \
            $MMChat(cmd,PEEK_CONNECTIONS) {
                sendpeek $conn
            } \
            $MMChat(cmd,SNOOP) {
                incomingsnoop $conn
            } \
            $MMChat(cmd,SNOOP_DATA) {
                incomingsnoopdata $conn $data
            } \
            $MMChat(cmd,PEEK_LIST) {
                incomingpeek $conn $data
            } \
            $MMChat(cmd,REQUEST_CONNECTIONS) {
                sendconnections $conn
            } \
            $MMChat(cmd,CONNECTION_LIST) {
                incomingconnections $conn $data
            } \
            $MMChat(cmd,TEXT_EVERYBODY) {
                incomingchat "ALL" $conn $data
            } \
            $MMChat(cmd,TEXT_PERSONAL) {
                incomingchat "PRIVATE" $conn $data
            } \
            $MMChat(cmd,TEXT_GROUP) {
                incomingchat "GROUP" $conn $data
            } \
            $MMChat(cmd,MESSAGE) {
                incomingmessage $conn $data
            } \
            $MMChat(cmd,DO_NOT_DISTURB) {
                incomingdnd $conn $data
            } \
            $MMChat(cmd,VERSION) {
                incomingversion $conn $data
            } \
            $MMChat(cmd,PING_REQUEST) {
                incomingpingrequest $conn $data
            } \
            $MMChat(cmd,PING_RESPONSE) {
                incomingpingresponse $conn $data
            } \
            $MMChat(cmd,FILE_START) {
                incomingfilestart $conn $data
            } \
            $MMChat(cmd,FILE_DENY) {
                incomingfiledeny $conn $data
            } \
            $MMChat(cmd,FILE_BLOCK_REQUEST) {
                incomingfileblockrequest $conn $data
            } \
            $MMChat(cmd,FILE_BLOCK) {
                incomingfileblock $conn $data
            } \
            $MMChat(cmd,FILE_END) {
                incomingfileend $conn
            } \
            $MMChat(cmd,FILE_CANCEL) {
                incomingfilecancel $conn
            } \
            $MMChat(cmd,SEND_ACTION)    - \
            $MMChat(cmd,SEND_ALIAS)     - \
            $MMChat(cmd,SEND_MACRO)     - \
            $MMChat(cmd,SEND_VARIABLE)  - \
            $MMChat(cmd,SEND_EVENT)     - \
            $MMChat(cmd,SEND_GAG)       - \
            $MMChat(cmd,SEND_HIGHLIGHT) - \
            $MMChat(cmd,SEND_LIST)      - \
            $MMChat(cmd,SEND_ARRAY)     - \
            $MMChat(cmd,SEND_BARITEM)   - \
            $MMChat(cmd,SEND_SUBSTITUTE) {
                incomingcommand $conn $cmd $data
            } \
            default { }

        set chatbuf [string range $chatbuf [expr $endidx + 1] end]
        set endidx [string first $MMChat(cmd,END_OF_COMMAND) $chatbuf]
    }

    return
}

##
# Params: connection of name change, new name
# Return: none
#
# Update the name of that connection and lets the user know about the change.
##
proc mmchat::namechange {conn name} {
    variable MMChatList

    chatnotify "\n<CHAT> $MMChatList($conn,name) has changed names to $name."
    set MMChatList($conn,name) $name

    return
}

##
# Params: connection requesting peek
# Return: none
#
# Send a list of public connections to the remote client.
##
proc mmchat::sendpeek {conn} {
    variable MMChat
    variable MMChatList

    chatnotify "\n<CHAT> $MMChatList($conn,name) has peeked at your connections."

    foreach namekey [array names MMChatList *,name] {
        set myconn [lindex [split $namekey ","] 0]
        if {$myconn ne $conn && !$MMChatList($myconn,private)} {
            append connlist $MMChatList($myconn,ip) "~" \
                            $MMChatList($myconn,port) "~" \
                            $MMChatList($namekey) "~"
        }
    }

    if {![info exists connlist]} {
        append mesg $MMChat(cmd,MESSAGE) \
                    "\n<CHAT> $MMChat(cfg,chatname) doesn't have any other connections.\n" \
                    $MMChat(cmd,END_OF_COMMAND)
    } else {
        append mesg $MMChat(cmd,PEEK_LIST) \
                    $connlist \
                    $MMChat(cmd,END_OF_COMMAND)
    }

    puts -nonewline $conn $mesg

    return
}

##
# Params: connection of request
# Return: none
#
# Implementing snooping by remote clients would be a bit difficult because
# of the way the display routines work. Until a good solution is realized
# and implemented, just deny any requests.
##
proc mmchat::incomingsnoop {conn} {
    variable MMChat

    append mesg $MMChat(cmd,MESSAGE) \
                "\n<CHAT> Snooping is not available.\n" \
                $MMChat(cmd,END_OF_COMMAND)
    puts -nonewline $conn $mesg

    return
}

##
# Params: connection with data, snoop data
# Return: none
#
# Process incoming snoop data: Split the data on newlines so we can put the
# snoopprefix before each line. Keep track of all ANSI color sequences we've
# seen so that we can accurately reconstruct the color state before each
# line. This prevents color from being lost or messed up between lines.
##
proc mmchat::incomingsnoopdata {conn data} {
    variable MMChat
    
    # Strip off leading color spec and newline
    set data [string range $data 4 end]
    if {[string range $data 0 0] eq "\n"} {
        set data [string range $data 1 end]
    }
    set colorspec ""
    foreach line [split $data "\n"] {
        if {$line eq ""} continue
        echo "[color $MMChat(cfg,snoopprefix)] $colorspec$line"
        foreach seq [regexp -all -inline {(?:\33\[(?:\d{1,2}|;)*m)+} $line] {
            if {$seq ne ""} { append colorspec $seq }
        }
    }
}

##
# Params: connection peek data is on, peek data
# Return: none
#
# Displays the peek list sent from the remote client.
##
proc mmchat::incomingpeek {conn peekdata} {
    variable MMChatList

    # If we didn't try to peek this connection, don't do anything
    if {!$MMChatList($conn,peek)} {
        return
    }

    set MMChatList($conn,peek) 0
    
    # Split data by ~ and remove last item, which will be empty
    set peeklist [lreplace [split $peekdata "~"] end end]
    set count [expr [llength $peeklist] / 3]

    set maxnamelen 4
    for {set i 0} {$i < $count} {incr i} {
        set name [lindex $peeklist [expr $i * 3 + 2]]
        if {[string length $name] > $maxnamelen} {
            set maxnamelen [string length $name]
        }
    }

    set layout "%-${maxnamelen}s %-15s %-5s"
    chatnotify "\n<CHAT> Peek found $count connections:"
    chatnotify [
        format $layout "Name" "Address" "Port"
    ]
    chatnotify [
        format $layout [string repeat "=" $maxnamelen] \
               "===============" "====="
    ]
    for {set i 0} {$i < $count} {incr i} {
        chatnotify [
            format $layout [lindex $peeklist [expr $i*3+2]] \
                           [lindex $peeklist [expr $i*3+0]] \
                           [lindex $peeklist [expr $i*3+1]]
        ]
    }

    return
}

##
# Params: connnection
# Return: none
#
# Sends a list of public connections to the given connection.
##
proc mmchat::sendconnections {conn} {
    variable MMChat
    variable MMChatList

    chatnotify "\n<CHAT> $MMChatList($conn,name) has request your public connections."

    foreach namekey [array names MMChatList *,name] {
        set myconn [lindex [split $namekey ","] 0]
        if {$myconn ne $conn && !$MMChatList($myconn,private)} {
            lappend connlist $MMChatList($myconn,ip) $MMChatList($myconn,port)
        }
    }

    if {![info exists connlist]} {
        append mesg $MMChat(cmd,MESSAGE) \
                    "\n<CHAT> $MMChat(cfg,chatname) doesn't have any other connections.\n" \
                    $MMChat(cmd,END_OF_COMMAND)
    } else {
        append mesg $MMChat(cmd,CONNECTION_LIST) \
                    [join $connlist ","] \
                    $MMChat(cmd,END_OF_COMMAND)
    }

    puts -nonewline $conn $mesg

    return
}

##
# Params: connection, connection data
# Return:
#
# Takes the connection data sent by a remote client from the given connection,
# and attempts to connect to each of those connections.
##
proc mmchat::incomingconnections {conn conndata} {
    variable MMChat
    variable MMChatList

    # If we didn't request connections, don't do anything
    if {!$MMChatList($conn,reqconn)} {
        return
    }

    set MMChatList($conn,reqconn) 0

    set connlist [split $conndata ","]
    set count [expr [llength $connlist] / 2]

    chatnotify "\n<CHAT> $count connections returned."

    for {set i 0} {$i < $count} {incr i} {
        set thisip [lindex $connlist [expr $i*2]]
        set thisport [lindex $connlist [expr $i*2+1]]
        set connected 0
        foreach ipkey [array names MMChatList *,ip] {
            set myconn [lindex [split $ipkey ","] 0]
            if {$thisip eq $MMChatList($ipkey) && \
                $thisport eq $MMChatList($myconn,port)} {
                set connected 1
            }
        }
        if {$connected} {
            chatnotify "\nAlready connected to $thisip:$thisport."
        } else {
            # Don't connect to yourself
            if {!($thisip eq $MMChat(cfg,ip) && \
                  $thisport eq $MMChat(cfg,listenport))} {
                call $thisip $thisport
            }
        }
    }

    return
}

##
# Params: type of chat, connection chat is coming from, message
# Return: none
#
# Type of chat can be one of "ALL", "PRIVATE", "GROUP" and indicates the
# intended audience of the chat. This is needed because when the type is
# "ALL", those chats need to be sent out to those being chatserved.
proc mmchat::incomingchat {type conn mesg} {
    variable MMChat
    variable MMChatList

    if {$MMChatList($conn,ignore)} {
        return
    }

    # Strip off the group name
    if {$type eq "GROUP"} {
        set mesg [string range $mesg 15 end]
    }

    if {$MMChat(cfg,paranoia)} {
        append output "\[$MMChatList($conn,name)\] "
    }
    append output $mesg
    chatprint $output

    if {$type eq "ALL"} {
        # If serving this conn, echo to all public connections
        if {$MMChatList($conn,serve) && !$MMChatList($conn,ignore)} {
            foreach privkey [array names MMChatList *private] {
                set myconn [lindex [split $privkey ","] 0]
                if {!$MMChatList($privkey) && $conn ne $myconn} {
                    puts -nonewline $myconn \
                        "$MMChat(cmd,TEXT_EVERYBODY)$mesg$MMChat(cmd,END_OF_COMMAND)"
                }
            }
        } else {
            # Else, echo this chat too all served connections
            foreach servekey [array names MMChatList *serve] {
                set myconn [lindex [split $servekey ","] 0]
                if {$MMChatList($servekey) && \
                    !$MMChatList($conn,ignore) && \
                    $conn ne $myconn} {
                    puts -nonewline $myconn \
                        "$MMChat(cmd,TEXT_EVERYBODY)$mesg$MMChat(cmd,END_OF_COMMAND)"
                }
            }
        }
    }

    return
}

##
# Params: connection message is from, message
# Return: none
#
# Prints the message. This is usually an informational message from the
# remote client giving a status of a previous command.
##
proc mmchat::incomingmessage {conn mesg} {
    variable MMChat
    variable MMChatList

    if {$MMChat(cfg,paranoia)} {
        append output "\[$MMChatList($conn,name)\] "
    }
    append output $mesg
    chatprint $output

    return
}

proc mmchat::incomingdnd {conn dnd} {
}

##
# Params: connection of version change, version
# Return: none
#
# Updates the version of a given connection.
##
proc mmchat::incomingversion {conn version} {
    variable MMChatList
    set MMChatList($conn,version) $version

    return
}

##
# Params: connection of request, request data
# Return: none
#
# The incoming request data should be some measure of time significant to the
# requester. We don't have to care what it is, since all we do is send it back
# and the remote client can do its own difference calculation.
##
proc mmchat::incomingpingrequest {conn request} {
    variable MMChat

    append mesg $MMChat(cmd,PING_RESPONSE) $request \
                $MMChat(cmd,END_OF_COMMAND)
    puts -nonewline $conn $mesg

    return
}

##
# Params: connection of response, response
# Return: none
#
# The response should be an echo of what we sent as a ping request. Use it
# to find the time difference of our ping.
##
proc mmchat::incomingpingresponse {conn response} {
    variable MMChat
    variable MMChatList

    set now [clock clicks -milliseconds]
    set elapsed [expr $now - $response]
    chatnotify "\n<CHAT> Ping returned from $MMChatList($conn,name): ${elapsed}ms"

    return
}

##
# Params: connection of request, request data
# Return: none
#
# Does the setup of starting a file transfer when requested.
##
proc mmchat::incomingfilestart {conn filestart} {
    variable MMChat
    variable MMChatList

    if {!$MMChatList($conn,allowxfers)} {
        append mesg $MMChat(cmd,FILE_DENY) \
                    "Not allowing file transfers." \
                    $MMChat(cmd,END_OF_COMMAND)
        puts -nonewline $conn $mesg
        return
    }

    if {$MMChatList($conn,xferstate) != $MMChat(file,NONE)} {
        append mesg $MMChat(cmd,FILE_DENY) \
                    "Already transfering a file." \
                    $MMChat(cmd,END_OF_COMMAND)
        puts -nonewline $conn $mesg
        return
    }

    set filename [lindex [split $filestart ","] 0]
    set filesize [lindex [split $filestart ","] 1]
    if {$filename eq "" || $filesize eq ""} {
        append mesg $MMChat(cmd,FILE_DENY) \
                    "Invalid transfer start request." \
                    $MMChat(cmd,END_OF_COMMAND)
        puts -nonewline $conn $mesg
        return
    }

    set filename [file join $MMChat(cfg,filepath) $filename]
    if {[file exists $filename]} {
        append mesg $MMChat(cmd,FILE_DENY) \
                    "File already exists." \
                    $MMChat(cmd,END_OF_COMMAND)
        puts -nonewline $conn $mesg
        return
    }

    if {[catch {set MMChatList($conn,file) [open $filename w]}]} {
        append mesg $MMChat(cmd,FILE_DENY) \
                    "Could not create file." \
                    $MMChat(cmd,END_OF_COMMAND)
        puts -nonewline $conn $mesg
        return
    }
    fconfigure $MMChatList($conn,file) -translation {binary binary}

    append usermesg "\n<CHAT> Receiving file from $MMChatList($conn,name)" \
                    " -- Filename: $filename, Length: $filesize"
    chatnotify $usermesg

    set MMChatList($conn,filename) $filename
    set MMChatList($conn,xferstate) $MMChat(file,RECEIVE)
    set MMChatList($conn,filelen) $filesize
    set MMChatList($conn,byteswritten) 0
    set MMChatList($conn,xferstart) [clock seconds]

    append mesg $MMChat(cmd,FILE_BLOCK_REQUEST) $MMChat(cmd,END_OF_COMMAND)
    puts -nonewline $conn $mesg

    return
}

##
# Params: connection of denial, deny data
# Return: none
#
# Outputs the denial message received from the remote client.
##
proc mmchat::incomingfiledeny {conn filedeny} {
    variable MMChat
    variable MMChatList

    if {$MMChatList($conn,xferstate) != $MMChat(file,SEND)} {
        return
    }

    append mesg "\n<CHAT> File start denied from $MMChatList($conn,name): " \
                $filedeny
    chatnotify $mesg

    return
}

##
# Params: connection of request, request
# Return: none
#
# Accepts a block request from remote client and sends them the next block of
# the file.
##
proc mmchat::incomingfileblockrequest {conn request} {
    variable MMChat
    variable MMChatList

    if {$MMChatList($conn,xferstate) != $MMChat(file,SEND)} {
        return
    }

    append block $MMChat(cmd,FILE_BLOCK) \
                 [read $MMChatList($conn,file) $MMChat(file,BLOCKSIZE)] \
                 $MMChat(cmd,END_OF_COMMAND)
    puts -nonewline $conn $block

    if {[eof $MMChatList($conn,file)]} {
        set now [clock seconds]
        set timediff [expr $now - $MMChatList($conn,xferstart)]

        append mesg $MMChat(cmd,FILE_END) $MMChat(cmd,END_OF_COMMAND)
        puts -nonewline $conn $mesg

        append usermesg "\n<CHAT> File transfer complete. File: " \
                        "$MMChatList($conn,filename), Length: " \
                        "$MMChatList($conn,filelen), Time: " \
                        [format "%d:%02d" [expr $timediff / 60] \
                                          [expr $timediff % 60]]
        chatnotify $usermesg

        closexfer $conn
    }

    return
}

##
# Params: connection with data, file data
# Return: none
#
# Processes the next incoming file block from the remote client.
##
proc mmchat::incomingfileblock {conn block} {
    variable MMChat
    variable MMChatList

    if {$MMChatList($conn,xferstate) != $MMChat(file,RECEIVE)} {
        return
    }

    if {$MMChatList($conn,byteswritten) < $MMChatList($conn,filelen)} {
        # Figure out how many bytes the block should have
        set bytes [
            expr $MMChatList($conn,filelen) - \
                 $MMChatList($conn,byteswritten) - 1
        ]
        if {$bytes > $MMChat(file,BLOCKSIZE)} {
            set bytes $MMChat(file,BLOCKSIZE)
        }

        puts -nonewline $MMChatList($conn,file) [string range $block 0 $bytes]
        set MMChatList($conn,byteswritten) [
            expr $MMChatList($conn,byteswritten) + [string length $block]
        ]
    }

    append request $MMChat(cmd,FILE_BLOCK_REQUEST) $MMChat(cmd,END_OF_COMMAND)
    puts -nonewline $conn $request

    return
}

##
# Params: connection of notification
# Return: none
#
# Remote client has signified the end of the file, do any clean-up necessary.
##
proc mmchat::incomingfileend {conn} {
    variable MMChat
    variable MMChatList

    if {$MMChatList($conn,xferstate) != $MMChat(file,RECEIVE)} {
        return
    }

    set now [clock seconds]
    set timediff [expr $now - $MMChatList($conn,xferstart)]
    append usermesg "\n<CHAT> File transfer complete. File: " \
                    "$MMChatList($conn,filename), Length: " \
                    "$MMChatList($conn,byteswritten), Time: " \
                    [format "%d:%02d" [expr $timediff / 60] \
                                      [expr $timediff % 60]]
    chatnotify $usermesg

    closexfer $conn

    return
}

##
# Params: connection of cancellation
# Return: none
#
# Remote client has signalled a transfer cancellation. Clean-up.
proc mmchat::incomingfilecancel {conn} {
    variable MMChat
    variable MMChatList

    if {$MMChatList($conn,xferstate) == $MMChat(file,NONE)} {
        return
    }

    append usermesg "\n<CHAT> File transfer with $MMChatList($conn,name) " \
                    "cancelled. File: $MMChatList($conn,filename), " \
                    "$MMChatList($conn,byteswritten) of " \
                    "$MMChatList($conn,filelen) bytes written."
    chatnotify $usermesg

    closexfer $conn

    return
}

##
# Params: connection with data, command sent, command data
# Return: none
#
# This is intended as a way for MudMaster users to easily share commands.
# It doesn't really make sense here since there's no easy way to do
# conversions. So, just send a message back saying commands aren't accepted.
##
proc mmchat::incomingcommand {conn cmd data} {
    variable MMChat

    append mesg $MMChat(cmd,MESSAGE) \
                "\n<CHAT> $MMChat(cfg,chatname) is not accepting commands.\n" \
                $MMChat(cmd,END_OF_COMMAND)
    puts -nonewline $conn $mesg

    return
}


###### Some Default Setup ######
# These are just some general setup things it's nice to have handled automatically.

# Get all the commands we need into the current namespace
namespace import mmchat::*

# Start listening for chat connections
namespace eval mmchat {
    startserver
}
