/*

  Copyright (C) 2000, The MITRE Corporation

  Use of this software is subject to the terms of the GNU General
  Public License version 2.

  Please read the file LICENSE for the exact terms.

*/

/*
 *
 * Author: Mike Butler, mgb@mitre.org
 *
 * $Id: UtString.h,v 1.12 1999/08/27 20:59:48 mgb Exp mgb $
 */
/* Implements the ANSII cstring class,
 * but allows embedded nulls too...
 * String comparisons are now case sensitive...
 */

#ifndef _String_h_
#define _String_h_

#ifdef FORTIFY
  #include <stl.h>		// Appeasment...
  #include <Fortify.h>
#endif FORTIFY

#include <assert.h>
#include <iostream.h>
#include <ctype.h>
#include <stdlib.h>		/* for size_t */


class String {
  friend ostream &operator<<(ostream &os, const String &s);
  friend istream &operator>>(istream &is, String &s);
  friend istream &getline(istream &is, String &s, char term = '\n');
  friend String operator+(const char *lhs, const String &rhs) {
    return(String(lhs) + rhs);
  }
private:
  char *cBuf;
  size_t cLen;
  size_t cMax;
  void GetLine(istream &is, char t = '\n');
  int Compare(const char *s1, const char *s2, size_t l1, size_t l2, bool foldcase = false) const;

  // Why are these here?
  static size_t Copy(const char *from, char *to, int len);
  static size_t Strlen(const char *s);

  /* Always leaves room for a null terminator! */
  bool Realloc(size_t n);
  void Base64Init(unsigned char *encode, unsigned char *decode) const;

  String Chomper(const String &white, bool left = false) const;

public:

  static const size_t npos = static_cast<size_t>(-1);
  void Init();
  String &operator=(const String &o);
  String();
  String(const char *s, size_t start = 0, size_t n = npos);
  String(const String&o, size_t start = 0, size_t n = npos);
  explicit String(char c, size_t count = 1);
  explicit String(int v, const String &f="%d");
  explicit String(size_t v, const String &f="%d");
  explicit String(long v, const String &f="%ld");
  explicit String(double v, const String &f="%f");
  ~String();
  void reserve (size_t siz);

  size_t copy(char *cb, size_t n = npos, size_t pos = 0) const;
  String &append(const String &s, size_t start = 0, size_t n = npos);
  String &append(const char *s, size_t start = 0, size_t n = npos);
  String operator+(const char *o) const;
  String operator+(int i) const; 
  String operator+(const String&o) const;
  String &operator+=(const String&o);
  String &operator+=(const char *o);
  String &operator+=(const char o);
  String substr(size_t pos, size_t n = npos) const;
  char & operator[](size_t pos);
  char operator[](size_t pos) const;
  int operator==(const char *s) const;
  int operator==(const String&o) const;
  int operator!=(const char *s) const  { return(!(*this == s)); }
  int operator!=(const String&o) const { return(!(*this == o)); }
  int operator<(const char *s) const;
  int operator<(const String&o) const;
  size_t length() const;
  size_t find(const String &p, size_t base = 0) const;
  size_t find_first_of(const String &s, size_t base = 0) const;
  size_t find_first_not_of(const String &s, size_t base = 0) const;
  const char *c_str() const;

  // ----------------------------------------------------------------
  // Local extensions below here...  (not standard "string" class stuff)
  static String String::MakeAndPack(long data, size_t len, size_t offset = npos);
  void String::Pack(long data, size_t len, size_t offset = npos);
  long String::Unpack(size_t len, size_t offset = 0, bool extend = false);

  bool Getline(istream &is, char term = '\n');
  bool LoadFile(const String &path);
  bool ReadToEof(istream &is);

  // Base64 and Hex string conversions...
  bool Base64Validate() const;
  String HexDecode(bool strict = true) const;
  String HexEncode(bool white = true) const;
  String Base64Encode() const;
  String Base64Decode(bool strict = true) const;

  char *GetRawBuf();		// Ugh, breaks encapsulation, but very useful!

  // Chomp (ala perl) something off either end...
  String Chomp(const String &white=" \t\n") const { return(Chomper(white, false)); }
  String ChompL(const String &white=" \t\n") const { return(Chomper(white, true)); }
  // Determine if string contains only printable characters
  bool IsPrintable() const;
  String ToUpper() const;
  String ToLower() const;

  // Format string like a paragraph, break on white space so that lines
  // are not longer than len.  If force is true, break in the
  // middle of words if needed
  String BreakLines(int len = 80, bool force = true);

  // Why wasn't it this initially?
  const char *str() const { return(c_str()); }

  // Case foldable version of "=="
  int Compare(const String &rhs, bool fold = false) const {
    return(Compare(cBuf, rhs.cBuf, cLen, rhs.cLen, fold));
  }

  // Handy things...  Convert string to and from a number
  static String Convert(int    val, const String &fmt = "%d");
  static String Convert(long   val, const String &fmt = "%ld");
  static String Convert(double val, const String &fmt = "%f");
  int    Convert(int    *val, bool *ok = 0, const String &fmt = "%d") const;
  long   Convert(long   *val, bool *ok = 0, const String &fmt = "%ld")const;
  double Convert(double *val, bool *ok = 0, const String &fmt = "%lf") const;
  int    Int()    { return(Convert((int*)0)); }
  int    Long()   { return(Convert((long*)0)); }
  double Double() { return(Convert((double*)0)); }


  // Count occurences of a character in the string (e.g., to count lines)
  size_t Occurs(const char c, size_t base = 0) const;
  bool Contains(const String &pat, size_t base = 0) const 
    { return(find(pat, base) != npos); }

  // See if we're looking at specific pattern...
  // Used to be "Begins(pat)" which was the special case:
  // bool Begins(const String &pat) { return(LookingAt(pat)); }
  bool LookingAt(const String &pat, size_t base = 0) const { 
    if((base + pat.cLen) > cLen) return(false);
    return(!Compare(cBuf+base, pat.cBuf, pat.cLen, pat.cLen, false));
  }

  // Glob style pattern matching
  bool Match(const String &pat, bool fold = false) const;

  // Replace string "s" with string "r" upto "count" times...
  String Replace(const String &s, const String &r, size_t count = npos) const;

  // Split string on string pattern after base, return first piece
  // Split consumes the delimiting string
  String Split(const String &delim, String *rest = 0, size_t base = 0) const;
  // Return token (word) delimited by any character in delim...
  // Token skips leading delims, but does not comsume the delimiting token...
  String Token(String *rest = 0, size_t base = 0, const String &delim = " \t\n") const;

  // Copy the contents to a user buffer...
  size_t Export(char *to, size_t len = npos) const;

  // IsTrue() - see if a string has a value suggesting "true"
  bool IsTrue();
};

#endif // _String_h_
