/*
** Modular Logfile Analyzer
** Copyright 2000 Jan Kneschke <jan@kneschke.de>
**
** Homepage: http://www.kneschke.de/projekte/modlogan
**

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version, and provided that the above
    copyright and permission notice is included with all distributed
    copies of this or derived software.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA

**
** $Id: parse.c,v 1.13 2001/11/12 10:00:39 ostborn Exp $
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <ctype.h>
#include <errno.h>

#include "mlocale.h"
#include "mplugins.h"
#include "mrecord.h"
#include "mdatatypes.h"
#include "misc.h"

#include "plugin_config.h"

const char *short_month[] = {	"Jan", "Feb", "Mar", "Apr", "May", "Jun", 
			"Jul", "Aug", "Sep", "Oct", "Nov", "Dec", NULL};

int parse_timestamp(mconfig *ext_conf, const char *str, mlogrec *record) {
#define N 20 + 1
	int ovector[3 * N], n, i;
	char buf[10];
	struct tm tm;
	config_input *conf = ext_conf->plugin_conf;
	
	if ((n = pcre_exec(conf->match_timestamp, conf->match_timestamp_extra, str, strlen(str), 0, 0, ovector, 3 * N)) < 0) {
		if (n == PCRE_ERROR_NOMATCH) {
			fprintf(stderr, "%s.%d: string doesn't match: %s\n", __FILE__, __LINE__, str);
		} else {
			fprintf(stderr, "%s.%d: execution error while matching: %d\n", __FILE__, __LINE__, n);
		}
		return -1;
	}
	
	/* everything has matched, take the different pieces and be happy :) */
	pcre_copy_substring(str, ovector, n, 2, buf, sizeof(buf));
	tm.tm_mday = strtol(buf, NULL, 10);
	
	pcre_copy_substring(str, ovector, n, 1, buf, sizeof(buf));
	for (i = 0; short_month[i];i++) {
		if (!strcmp(buf, short_month[i])) {
			tm.tm_mon = i;
		}
	}
	
	pcre_copy_substring(str, ovector, n, 6, buf, sizeof(buf));
	tm.tm_year = strtol(buf, NULL, 10)-1900;
	
	pcre_copy_substring(str, ovector, n, 3, buf, sizeof(buf));
	tm.tm_hour = strtol(buf, NULL, 10);
	pcre_copy_substring(str, ovector, n, 4, buf, sizeof(buf));
	tm.tm_min = strtol(buf, NULL, 10);
	pcre_copy_substring(str, ovector, n, 5, buf, sizeof(buf));
	tm.tm_sec = strtol(buf, NULL, 10);
	
	record->timestamp = mktime (&tm);
#ifdef DEBUG_PCRE	
	pcre_get_substring_list(str, ovector, n, &list);
	
	for (i = 0; i < n; i++) {
		printf("%d: %s\n", i, list[i]);
	}
#endif	
	return 0;
#undef  N
}

int parse_record_pcre(mconfig *ext_conf, mlogrec *record, char *_buffer) {
#define N 20 + 1
	const char **list;
	int ovector[3 * N], n;
#ifdef DEBUG_PCRE
	int i;
#endif
	config_input *conf = ext_conf->plugin_conf;
	mlogrec_web *recweb = NULL; 
	mlogrec_web_ftp *recftp = NULL; 
	
	record->ext_type = M_RECORD_TYPE_WEB;
	record->ext = mrecord_init_web();
	
	recweb = record->ext;
	
	if (recweb == NULL) return -1;
	
	recweb->ext = mrecord_init_web_ftp();
	recweb->ext_type = M_RECORD_TYPE_WEB_FTP;
	
	recftp = recweb->ext;
	
	if ((n = pcre_exec(conf->match_clf, conf->match_clf_extra, _buffer, strlen(_buffer), 0, 0, ovector, 3 * N)) < 0) {
		if (n == PCRE_ERROR_NOMATCH) {
			fprintf(stderr, "%s.%d: string doesn't match: %s\n", __FILE__, __LINE__, _buffer);
		} else {
			fprintf(stderr, "%s.%d: execution error while matching: %d\n", __FILE__, __LINE__, n);
		}
		return -1;
	}
	
	/* everything has matched, take the different pieces and be happy :) */
	pcre_get_substring_list(_buffer, ovector, n, &list);
	/* 
	   1 - date
	   2 - duration
	   3 - requesting host
	   4 - xfersize
	   5 - filename
	   6 - mode 
	       a: ascii
	       b: binary
	   7 - special action
	       C: file was compressed
	       U: file was uncompressed
	       T: file was tar'ed
	       _: no action was taken
	   8 - direction
	       i: incomming
	       o: outgoing
	   9 - access-mode
	       a: anonymous
	       g: guest
	       r: real 
	   10 - user
	   11 - group
	   12 - authentication-method
	        0: none
	        1: RFC931 Authentication
	   13 - authenticated-user-id
	        *: not available
	*/
	
	parse_timestamp(ext_conf, list[1], record);
	
	if (recftp) 
		recftp->trans_duration = strtol(list[2], NULL,10);
	
	recweb->req_host = malloc(strlen((char *)list[3])+1);
	strcpy(recweb->req_host, (char *)list[3]);
	
	recweb->xfersize = strtol(list[4], NULL,10);
	
	recweb->req_url = malloc(strlen((char *)list[5])+1);
	strcpy(recweb->req_url, (char *)list[5]);
	
	if (recftp) {
		switch(*list[6]) {
		case 'a':
			recftp->trans_mode = M_RECORD_FTP_MODE_ASCII;
			break;
		case 'b':
			recftp->trans_mode = M_RECORD_FTP_MODE_BINARY;
			break;
		}
		
		switch(*list[8]) {
		case 'i':
			recftp->trans_command = M_RECORD_FTP_COMMAND_PUT;
			break;
		case 'o':
			recftp->trans_command = M_RECORD_FTP_COMMAND_GET;
			break;
		case 'd':
			recftp->trans_command = M_RECORD_FTP_COMMAND_DELETE;
			break;
		}
		
		recweb->req_user = malloc(strlen((char *)list[10])+1);
		strcpy(recweb->req_user, (char *)list[10]);
	}
	
	recweb->req_user = malloc(strlen((char *)list[10])+1);
	strcpy(recweb->req_user, (char *)list[10]);
	
	if (recftp) {
		recftp->req_group = malloc(strlen((char *)list[11])+1);
		strcpy(recftp->req_group, (char *)list[11]);
	}

#ifdef DEBUG_PCRE
	for (i = 0; i < n; i++) {
		printf("%d: %s\n", i, list[i]);
	}
#endif
	free(list);
	
	return 0;
#undef  N
}

int mplugins_input_wuftpd_get_next_record(mconfig *ext_conf, mlogrec *record) {
	int ret = 0;
	config_input *conf = ext_conf->plugin_conf;
	
	if (!fgets(conf->buffer, conf->buf_len-1,conf->inputfile)) {
		return M_RECORD_EOF;
	}

	while (conf->buffer[strlen(conf->buffer)-1] != '\n') {
		conf->buffer = realloc(conf->buffer, (conf->buf_len+conf->buf_inc) * sizeof(char));
		
		if (!fgets(conf->buffer+strlen(conf->buffer), conf->buf_inc-1,conf->inputfile)) {
			return M_RECORD_EOF;
		}
		
		conf->buf_len += conf->buf_inc;
	}
	ret = parse_record_pcre(ext_conf, record, conf->buffer);

	return ret == -1 ? M_RECORD_CORRUPT : M_RECORD_NO_ERROR;
}
