/*
 * Copyright (c) 2000, 2001, Amnon BARAK (amnon@cs.huji.ac.il).
 * All rights reserved.
 *
 *       MOSIX tune.c,v 1.1.1.1 1999/01/28 09:58:13 arielr Exp
 *
 * THIS SOFTWARE IS PROVIDED IN ITS "AS IS" CONDITION, WITH NO WARRANTY
 * WHATSOEVER. NO LIABILITY OF ANY KIND FOR ANY DAMAGES WHATSOEVER RESULTING
 * FROM THE USE OF THIS SOFTWARE WILL BE ACCEPTED.
 */
/*
 * Author(s): Amnon Shiloh.
 */

/*
 * tune.c:
 *
 * Produce an dscosts.h file to be #included by the kernel in order
 * to estimate communication overheads in MOSIX.
 *
 * All results are given in units of 1/1000000 of a second and include both
 * local and remote overheads.
 *
 * We distinguish between two kinds of penalties for communication overhead:
 *
 *	D: cost (system time) at deputy
 *	R: cost (system time) at remote
 *
 * The local communication overheads are measured for system calls with:
 *	1. Demand pages:
 *		PAGE_COST_{D/R}
 *	2. system calls:
 *		SYSCALL_COST_{D/R}
 *	3. output:
 *		COPYIN_COST_BASIC_{D/R}
 *		COPYIN_COST_PER_KB_{D/R}
 *	4. input:
 *		COPYOUT_COST_BASIC_{D/R}
 *		COPYOUT_COST_PER_KB_{D/R}
 *
 * Originally Written by A. Shiloh, March 1988.
 * Modified on February 1993.
 * Ported to Linux in 1998.
 * SMP enhancements in 1999.
 */

/*
 * - tune creates "mosix.costs" file. A file with the 14 costs, space
 *   separated. You can simply "cat mosix.costs > /proc/mosix/admin/overheads"
 *
 * - Cleaned-up the code to compile -Wall with no warnings
 *
 * 2/20/1999 - Manos Megagiannis manos@manos.com
*/

#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/wait.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <stdio.h>
#include <signal.h>
#include <a.out.h>
#include <fcntl.h>
#include <sys/time.h>
#include <setjmp.h>
#include <sys/mman.h>
#include <math.h>

/*
 * General definitions
*/
#define	MAXSMP	8
#define	MCASES	100
#define	DEFCASES 7
#define	STD_SPD	10000
#define	STD_TIME 7714000000LL
#define	REPS	10000
#define	PAGE1	10
#define	START	SIGUSR1
#define	END	SIGUSR2

#define	NEGALLOW	10000000
/*
 * Filename definitions
 */
#define HFILE "dscosts.h"
#define MFILE "mosix.costs"

/*
 * Error values
*/
#define NOT_MOSIX 1
#define TP_USAGE 2
#define MCASES_ERR 3
#define NO_SUID 4
#define NO_SINGLE 5
#define	NO_TUNEPASS 6
#define	BAD_TUNEPASS 7


off_t lseek();
jmp_buf env;
int ok;
double speed1, speed2;
int ips1, ips2;
int nhere, nthere;
int f1[MAXSMP], f2[MAXSMP];
int lfd;
volatile long long counter;
int cases = DEFCASES;
char buf[20480];

volatile int nsig;
int startpipe[2];
FILE *hfd, *hfd2;

int this, other;
char verbose;
char debug;
char assist;
char needy;
int even = 1, odd = 0;

int pgid;

int sizes[MCASES];

/* RESULTS */
 
int nl, nr, pl, pr;
int il[MCASES], is[MCASES], it[MCASES], ir[MCASES];
int ol[MCASES], os[MCASES], ot[MCASES], or[MCASES];

char *comment;

/*
 * function definitions
*/
extern int testloop();
void tunepass(int o);
int note(), jmp(), endit();
void rest_user();
void bestline(int x[MCASES], int y[MCASES], int *rm, int *rn);
void prdef(char *def, int val, int onneg);
void summarize();
int cpuspeed();
void measure(int fd, int ipsx);
int count();
void test(int empty, int paging, int syscall, int inf, int outf, int *lout, int *rout);
void run(int fd, int empty, int paging, int syscall, int inf, int outf);
void testall();
void prsizes();
void initsizes();
void forkpg(int where);
void repg();
void mkpg();
void initfiles();
void get_to(int where);
int migrate(int to);
void getsmps();
int howmany(int where);
void getspeeds();
int YES(int eol);
int readnum();
int whereami();
void setup();


/*
 * Error messages
*/
void error_msg(int x)
{
	switch (x) {
	case NOT_MOSIX :
		fprintf(stderr,"tune ignored: This is NOT a MOSIX system!\n");
		break;
	case TP_USAGE :
		fprintf(stderr, "Usage: tunepass other\n");
		break;
	case MCASES_ERR :
		fprintf(stderr, "Sorry, Number of cases must be between 2 and %d\n", MCASES);
		break;
	case NO_SUID :
		printf("Not Super-User\n");
		break;
	case NO_SINGLE :
		printf("Bring both nodes to single-user mode and come back.\n");
		break;
	case NO_TUNEPASS:
		perror("Failed running 'tunepass'\n");
		break;
	case BAD_TUNEPASS:
		fprintf(stderr, "Improper output from 'tunepass'\n");
		break;
	}
	exit(1);
}

int
main(na, argv)
char *argv[];
{
	char *getenv();

	while(na > 1 && argv[1][0] == '-')
	{
		switch(argv[1][1])
		{
			case 'o':
				ok = 1;
				break;
			case 'c':
				if(na > 2)
				{
					comment = argv[2];
					na--;
					argv++;
				}
				break;
			case 'a':
				ok = 1;
				assist = 1;
				break;
			case 'e':
				even = 0;
				break;
			case 'O':
				odd = 1;
				break;
			case 'n':
				if(argv[1][2])
					cases = atoi(&argv[1][2]);
				else if(na > 2)
				{
					na--;
					argv++;
					cases = atoi(argv[1]);
				}
				if(cases < 2 || cases > MCASES)
					error_msg(MCASES_ERR);
				break;
		}
		na--;
		argv++;
	}
	if(na > 1)
		other = atoi(argv[1]);
	if(getenv("VERBOSE"))
		verbose++;
	if(getenv("DTUNE"))
		debug++;
	setup();
	initsizes();
	getsmps();
	initfiles();
	getspeeds();
	rest_user();
	testall();
	summarize();

	return 0;
}

/*
 * Find involved processors, and make sure they have no other activities
 */
void setup()
{
	register int i;
	int lfd = open("/proc/self/lock", 1);

	printf("MOSIX communication tuning program:\n\n");
	if(getuid())
		error_msg(NO_SUID);
	if(lfd == -1)
		error_msg(NOT_MOSIX);
	write(lfd, "1\n", 2);
	close(lfd);
	get_to(0);
	this = whereami();
	while(other <= 0)
	{
		printf("Enter second node number: ");
		fflush(stdout);
		other = readnum();
	}
	if(!ok)
	{
		printf("Are both nodes in single-user mode, network enabled and in quiet-mode? ");
		if(!YES(1))
		{
			printf("Not being in single-user mode might produce inaccurate results. Continue? ");
			if(!YES(0))
				error_msg(NO_SINGLE);
		}
	}
	for(i = 1 ; i < NSIG ; i++)
		if(i != SIGKILL && i != SIGCHLD && i != SIGSTOP && i != SIGTSTP
								&& i != SIGCONT)
			signal(i, (sig_t)endit);
}

/*
 * Find processor speeds using both general loop and tight loop.
 * General loop is for finding the CPU speed as it is viewed by the kernel.
 * Tight loop is for exact measurments.
 */
void getspeeds()
{
	printf("Taking processor speeds:\n");
	get_to(0);
	speed1 = cpuspeed();
	get_to(other);
	speed2 = cpuspeed();
	get_to(0);
	if(assist)
		return;
	if(speed1 < speed2 * 0.99 || speed2 < speed1 * 0.99)
	{
		printf("\nThe two processors are of varying speeds:\n");
		printf("This makes accurate measurement slightly more complex\n");
		printf("and will require some manual assistance.  OK [Y/n]? ");
		needy = YES(1);
		if(!needy)
			printf("\nOK, I will continue automatically, but the results may not be as accurate!\n\n");
	}
	else
	{
		printf("\nThe two processors look similar: if their hardware is exactly the same,\n");
		printf("then tuning can proceed completely automatically.\n");
		printf("If, however, any of the following differs between the two computers:\n\n");
		printf("  1) the processor itself.\n");
		printf("  2) the type or speed of the memory.\n");
		printf("  3) the number of memory wait-states (if any).\n");
		printf("  4) the networking hardware.\n");
		printf("  5) one computer uses a kernel with SMP configured and the other does not.\n");
		printf("\nthen some manual assistance is required to produce more accurate results.\n");
		printf("\nAre the two computers identical [Y/n]? ");
		needy = !YES(1);
	}
}

void getsmps()
{
	nhere = howmany(this);
	nthere = howmany(other);
}

void get_to(int where)
{
	while(migrate(where))
	{
		printf("Cannot migrate to %d. Retry ? ", where);
		if(!YES(1))
			endit(1);
	}
}

void initfiles()
{
	register int i;

	if(access("/dev/fun", 0))
		system("/bin/mknod /dev/fun c 1 10");
	if((lfd = open("/dev/fun", 2)) == -1)
	{
		perror("/dev/fun");
		endit(1);
	}
	for(i = 0 ; i < nhere ; i++)
	{
		if((f1[i] = creat("/tmp/JuNkFiLe1", 0)) == -1)
		{
			fail:
			perror("failed to open scratch files");
			endit(1);
		}
		unlink("/tmp/JuNkFiLe1");
	}
	for(i = 0 ; i < nthere ; i++)
	{
		if((f2[i] = creat("/tmp/JuNkFiLe2", 0)) == -1)
			goto fail;
		unlink("/tmp/JuNkFiLe2");
	}
	if(pipe(startpipe))
		goto fail;
	fcntl(startpipe[0], F_SETFL, O_NONBLOCK);
}

void mkpg()
{
	FILE *asf;

	if((asf = fopen("pg.c", "w")) == NULL)
	{
		perror("pg.c");
		endit(1);
	}
	fprintf(asf, "#include <signal.h>\nonsig(){}\n");
	fprintf(asf, "char buf[4096*258] = {0};main(){register i;\n");
	fprintf(asf, "signal(SIGUSR1, (sig_t)onsig);pause();\n");
	fprintf(asf, "for(i=4096;i<4096*257;i+=4096)buf[i]++;_exit(0);}\n");
	fclose(asf);
	if(system("/usr/bin/cc -o pg -g pg.c"))
	{
		fprintf(stderr, "Tune: failed while preparing pg\n");
		unlink("pg.s");
		endit(1);
	}
	unlink("pg.c");
	fflush(stdout);
	forkpg(0);
}

void repg()
{
	system("/bin/touch pg");
	forkpg(other);
}

void fipg()
{
	unlink("pg");
}

void forkpg(int where)
{
	switch(pgid = fork())
	{
		case 0:
			get_to(where);
			execl("./pg", "pagetest", 0);
			perror("pg");
			exit(1);
		case -1:
			perror("Sorry: fork (for pg) failed\n");
			break;
	}
	sleep(2);
}

/*
 * unmount and mount the file system of the specified file on
 * specified processor to make sure all blocks of that file system
 * are out of cache.
 */

/*
 * Decide which funnel sizes to use:
 */
void initsizes()
{
	register int i;
	int def = 1;

	if(!ok)
	{
		printf("Avoid testing with odd sizes: OK? ");
		even = YES(1);
		if(!even)
		{
			printf("Test only odd sizes? ");
			odd = YES(0);
		}
		printf("I/O overhead will be tested with %d different sizes: OK ? ",
			cases);
		if(!YES(1))
		{
			def = 0;
			printf("By default, there are %d sizes\n", cases);
			do
			{
				printf("Enter number of sizes (2 - %d): ", MCASES);
				cases = readnum();
			}
			while(cases < 2 || cases > MCASES)
				;
		}
	}
	for(i = 0 ; i < cases ; i++)
		sizes[i] = 2 + 4094 * i / cases;
	if(odd || even)
	for(i = 0 ; i < cases ; i++)
	if((sizes[i] & 1) != odd)
		sizes[i]--;
	if(!def)
		prsizes();
}

void prsizes()
{
	register int line = 5, first = 1;
	register int i;

	printf("The following sizes will be tested: ");
	for(i = 0 ; i < cases ; i++)
	{
		if(line-- == 0)
		{
			printf(",\n");
			line = 12;
			first = 1;
		}
		if(first)
			first = 0;
		else
			printf(", ");
		printf("%d", sizes[i]);
	}
	printf("\n");
}

void testall()
{
	register int i;

	printf("\nTest begins: If you are not especially interested in");
	printf("\nperformance evaluation ignore the following output :\n\n");
	test(1, 0, 0, 0, 0, NULL, NULL);
	test(0, 1, 0, 0, 0, &pl, &pr);
	test(0, 0, 1, 0, 0, &nl, &nr);
	for(i = 0 ; i < cases ; i++)
		test(0, 0, 0, sizes[i], 0, &il[i], &ir[i]);
	for(i = 0 ; i < cases ; i++)
		test(0, 0, 0, 0, sizes[i], &ol[i], &or[i]);
}

void test(int empty, int paging, int syscall, int inf, int outf, int *lout, int *rout)
{
	int lref = 0;	/* just to pacify gcc */
	int local, remote;
	int s1[MAXSMP], s2[MAXSMP];
	register int i;

	signal(START, (sig_t)note);
	if(empty)
		printf("INITIAL CALIBRATION:\n");
	else if(paging)
		printf("PAGING:\n");
	else if(syscall)
		printf("SCALLS WITH NO I/O:\n");
	else if(inf)
		printf("INPUT of size %d:\n", inf);
	else if(outf)
		printf("OUTPUT of size %d:\n", outf);
	else
		printf("What Test???\n");
	fflush(stdout);
	if(paging)
	{
		mkpg();
#ifdef WARMUP
		/* an initial run to warm-up and bring it all to cache */
		run(lfd, empty, paging, syscall, inf, outf);
		forkpg(0);
#endif /* WARMUP */
	}
	if(empty)
		goto remote_test;

	/* LOCAL TEST */

	get_to(0);
	nsig = 0;
	for(i = 0 ; i < nhere ; i++)
	switch(s1[i] = fork())
	{
		case -1:
			for(i-- ; i >= 0 ; i--)
				kill(s1[i], SIGKILL);
			perror("Fork");
			endit(1);
		case 0:
			measure(f1[i], ips1);
	}
	while(nsig < nhere)
		sleep(1);
	sleep(3);
	run(lfd, empty, paging, syscall, inf, outf);
	for(i = 0 ; i < nhere ; i++)
		kill(s1[i], END);
	while(wait(0) > 0)
		;
	lref = 0;
if(debug)
for(i=0;i<nhere;i++)printf("lref[%d]=%ld\n", i, lseek(f1[i], 0, 1)-NEGALLOW);
	for(i = 0 ; i < nhere ; i++)
		lref += lseek(f1[i], (off_t)0, 1) - NEGALLOW;

	/* REMOTE TEST */

	remote_test:
	nsig = 0;
	get_to(0);
	if(paging)
		repg();
	for(i = 0 ; i < nhere ; i++)
	switch(s1[i] = fork())
	{
		case -1:
			for(i-- ; i >= 0 ; i--)
				kill(s1[i], SIGKILL);
			perror("Fork");
			endit(1);
		case 0:
			measure(f1[i], ips1);
	}
	while(migrate(other))
	{
		printf("Cannot migrate to %d", other);
		if(paging)
			printf(" (most likely reason is insufficient memory)");
		printf(". Retry ? ");
		if(!YES(1))
			goto kill_local_sons;
	}
	for(i = 0 ; i < nthere ; i++)
	switch(s2[i] = fork())
	{
		case -1:
			perror("Fork");
			kill_local_sons:
			for(i-- ; i >= 0 ; i--)
				kill(s2[i], SIGKILL);
			for(i = 0 ; i < nhere ; i++)
				kill(s1[i], SIGKILL);
			endit(1);
		case 0:
			measure(f2[i], ips2);
	}
	while(nsig < nhere + nthere)
		sleep(1);
	sleep(2);
	run(lfd, empty, paging, syscall, inf, outf);
	for(i = 0 ; i < nhere ; i++)
		kill(s1[i], END);
	for(i = 0 ; i < nthere ; i++)
		kill(s2[i], END);
	while(wait(0) >= 0)
		;
	local = remote = 0;
if(debug)
for(i=0;i<nhere;i++)printf("local[%d]=%ld\n", i, lseek(f1[i], 0, 1)-NEGALLOW);
	for(i = 0 ; i < nhere ; i++)
		local += lseek(f1[i], (off_t)0, 1) - NEGALLOW;
if(debug)
for(i=0;i<nthere;i++)printf("remote[%d]=%ld\n", i, lseek(f2[i], 0, 1) - NEGALLOW);
	for(i = 0 ; i < nthere ; i++)
		remote += lseek(f2[i], (off_t)0, 1) - NEGALLOW;
	if(empty)
	{
		ips1 = local / nhere;
		printf("%s of node %d count %d loops/second\n",
			nhere > 1 ? "Processors" : "The processor", this, ips1);
		ips2 = remote / nthere;
		printf("%s of node %d count %d loops/second\n",
			nthere > 1 ? "Processors" : "The processor",
			other, ips2);
		return;
	}

	printf("\tD = %.2f seconds (%.2f locally), R = %.2f seconds\n",
		(local-lref)/1000000.0, lref/1000000.0, remote/1000000.0);

	*lout = local - lref;
	*rout = remote;
	if(paging)
		fipg();
}

void run(int fd, int empty, int paging, int syscall, int inf, int outf)
{
	register int i;

	if(paging)
	{
		fflush(stdout);
		kill(pgid, SIGUSR1);
		if((i = wait(0)) != pgid)
		{
			printf("wait: not pgid ; %d instead of %d\n", i, pgid);
			exit(1);
		}
	}
	else if(inf)
	{
		if(inf < 0)
			return;
		for(i = 0 ; i < REPS ; i++)
			read(fd, buf, inf);
	}
	else if (outf)
		for(i = 0 ; i < REPS ; i++)
			write(fd, buf, outf);
	else if(syscall)
		for(i = 0 ; i < REPS ; i++)
			lseek(fd, (off_t)0, 0);
	else
		sleep(10);
	return;
}

//int last_second();

void measure(int fd, int ipsx)
{
	struct timeval t, t2;
	long usec;
	long sk;

	if(setjmp(env))
	{
		gettimeofday(&t2, 0);
		usec = (t2.tv_sec - t.tv_sec) * 1000000 + t2.tv_usec-t.tv_usec;
		if(ipsx)
			/*sk = ((long long)ipsx) * usec / 1000000 - counter;*/
			sk = usec - counter * 1000000LL / ipsx;
		else
			sk = counter * 1000000 / usec;
if(debug)
printf("%d-ended, usec=%ld, counter=%qd, seek to %ld\n",getpid(),usec, counter, sk);
		sk += NEGALLOW;
		if(sk < 0)
			sk = 0;
		lseek(fd, sk, 0);
		exit(0);
	}
if(debug)
printf("%d-Measure on %d.\n", getpid(), whereami());
	sleep(3); /* allow time for other SMP forks and parent migration */
	signal(END, (sig_t)jmp);
	write(startpipe[1], "x", 1);
	kill(getppid(), START);
	gettimeofday(&t, 0);
	count();
}

int count()
{
	int j = 3, k = 4;

	for(counter = 0 ;; counter++)
	{
		j += k;
		k += j;
		if(j|k)
			k++;
		;
	}
	return(k);
}

int note(int i)
{
	char x[2*MAXSMP];
	int n;

	signal(i, (sig_t)note);
	n = read(startpipe[0], x, sizeof(x));
	if(n > 0)
		nsig += n;
	return (0);
}

int jmp()
{
	longjmp(env, 1);
}

int cpuspeed()
{
	struct timeval t0, t1, t2;
	int loc = whereami();
	int speed;

	register int usec;

	sleep(1);
	gettimeofday(&t0, 0);
	gettimeofday(&t1, 0);
	testloop();
	gettimeofday(&t2, 0);
	usec = (t2.tv_sec-t1.tv_sec)*1000000+t2.tv_usec-t1.tv_usec;
	usec -= (t1.tv_sec-t0.tv_sec)*1000000+t1.tv_usec-t0.tv_usec;
	speed = STD_TIME / usec;
	if(speed == STD_SPD)
		printf("Processor %d's speed is like a Pentium-III at 1GHz\n", loc);
	else
		printf("Processor %d's speed is %d/%d of a Pentium-III at 1GHz\n",
			loc, speed, STD_SPD);
	return(speed);
}

void summarize()
{
	int m1, m2, m3, m4, n1, n2, n3, n4, extra;
	register int i;
	FILE *date;
	char now[100];
	char *c;

	if(needy)
	{
		printf("\nYour manual assistance is now required:\n\n");
		printf("Please type the exact following command on the other node's (%d) console:\n\n", other);
		printf("\ttune -a");
		if(!even)
			printf(" -e");
		if(odd)
			printf(" -O");
		if(cases != DEFCASES)
			printf(" -n%d", cases);
		printf(" %d\n\n", this);
		printf("Please wait for the results,\n");
		printf("then type those results (6 numbers) here: ");
		fflush(stdout);
		while(1)
		{
			fgets(now, sizeof(now), stdin);
			now[sizeof(now)-1] = '\0';
			for(c = now ; *c ; c++)
			if(*c == ',' || *c == ';')
				*c = ' ';
			if(sscanf(now, "%d %d %d %d %d %d %d",
				&pr, &nr, &m2, &n2, &m4, &n4, &extra) == 6)
				break;
			printf("\nPlease type the results (6 numbers) from the other node's (%d) console\n", other);
			printf("(separated by spaces) here: ");
			fflush(stdout);
		}
		printf("Thank you, test continues:\n\n");
	}

	/* Normalize the results to suit a P2/400 */

	if((date = popen("/bin/date", "r")))
	{
		if(fgets(now, 100, date))
		{
			now[99] = '\0';
			now[strlen(now)-1] = '\0';
		}
		else
			now[0] = '\0';
		pclose(date);
	}
	pl = pl * speed1 / (256*STD_SPD);
	if(!needy)
		pr = pr * speed2 / (256*STD_SPD);
	for(i = 0 ; i < cases ; i++)
	{
		il[i] = il[i] < nl ? 0 : (il[i] - nl) * speed1 / (REPS*STD_SPD);
		ir[i] = ir[i] < nr ? 0 : (ir[i] - nr) * speed2 / (REPS*STD_SPD);
		ol[i] = ol[i] < nl ? 0 : (ol[i] - nl) * speed1 / (REPS*STD_SPD);
		or[i] = or[i] < nr ? 0 : (or[i] - nr) * speed2 / (REPS*STD_SPD);
	}
	nl = nl * speed1 / (REPS*STD_SPD);
	if(!needy)
		nr = nr * speed2 / (REPS*STD_SPD);

	hfd = fopen(assist ? "/dev/null" : "dscosts.h", "w");
	if(hfd == NULL)
	{
		perror("dscosts.h");
		endit(1);
	}
/*
 * Open mosix.cost file
 */
	hfd2 = fopen(assist ? "/dev/null" : MFILE, "w");
	if(hfd == NULL)
	{
		perror(MFILE);
		endit(1);
	}

	fprintf(hfd, "/* dscosts.h -- MOSIX */\n");
	fprintf(hfd, "/* cost of remote operations (microseconds) */\n\n");
	fprintf(hfd, "/* measuring date: %s */\n", now);
	if(comment)
	{
		for(c = comment ; *c ; c++)
		if(*c == '*' && *(c+1) == '/')
			*c = '/';
		fprintf(hfd, "/* %s */\n", comment);
	}
	fprintf(hfd, "\n");

	prdef("PAGE_COST_D", pl, 0);
	prdef("PAGE_COST_R", pr, 0);

	fprintf(hfd, "\n");

	prdef("SYSCALL_COST_D", nl, 0);
	prdef("SYSCALL_COST_R", nr, 0);

	fprintf(hfd, "\n");

	bestline(sizes, il, &m1, &n1);
	prdef("COPYOUT_COST_BASE_D", n1, 0);
	prdef("COPYOUT_COST_PER_KB_D", m1, m1);
	if(!needy)
		bestline(sizes, ir, &m2, &n2);
	prdef("COPYOUT_COST_BASE_R", n2, 0);
	prdef("COPYOUT_COST_PER_KB_R", m2, m2);

	fprintf(hfd, "\n");

	bestline(sizes, ol, &m3, &n3);
	prdef("COPYIN_COST_BASE_D", n3, 0);
	prdef("COPYIN_COST_PER_KB_D", m3, m3);
	if(!needy)
		bestline(sizes, or, &m4, &n4);
	prdef("COPYIN_COST_BASE_R", n4, 0);
	prdef("COPYIN_COST_PER_KB_R", m4, m4);

	fprintf(hfd, "\n");

	if(assist)
	{
		printf("\nSubtest Complete:\n");
		printf("Please copy the exact following list of 6 numbers to the primary test:\n\n");
		printf("\t%d %d %d %d %d %d\n\n", pr, nr, m2, n2, m4, n4);
		exit(0);
	}

	printf("MIGRATIONS:\n");
	tunepass(other);

	printf("End Of Test");
	if(!ok)
		printf(": The measured results are in dscosts.h");
	printf("\n");
	exit(0);
}

void prdef(char *def, int val, int onneg)
{
	char com[20];

	if(val < 0 && onneg != val)
	{
		sprintf(com, "	/* (%d) */", val);
		val = onneg;
	}
	else
		com[0] = '\0';

	fprintf(hfd, "#define	%s	%d%s\n", def, val, com);
	fprintf(hfd2,"%d ",val);  // Manos - write values in alterate file
}

void bestline(int x[MCASES], int y[MCASES], int *rm, int *rn)
{
	register int i;
	double sx = 0, sy = 0, sx2 = 0, sy2 = 0, sxy = 0;
	double m, n;

	if(verbose)
	{
		printf("line=");
		for(i=0;i<cases;i++)printf("[%d,%d]",x[i],y[i]);
		printf("\n");
	}
	for(i = 0 ; i < cases ; i++)
	{
		sx += x[i];
		sy += y[i];
		sx2 += x[i] * x[i];
		sy2 += y[i] * y[i];
		sxy += x[i] * y[i];
	}
	if(sxy == sx * sy / cases)
		m = 104857689;
	else
		m = (sy2 - sy * sy / cases) / (sxy - sx * sy / cases);
	if(m <= 0)
	{
		fprintf(hfd, "Next slope too high, more then usec per KB\n");
		m = 0;
	}
	n = (sy - sx * m) / cases;
	if(verbose)
		printf("m=%g,n=%g", m, n);
	if(n < 0)
	{
		n = 0;
		m = sy / sx;
		if(verbose)
			printf(" Nay: n=0, m=%g", m);
	}
	if(verbose)
		printf("\n");
	*rm = m * 1024 + 0.5;
	*rn = n + 0.5;
}

int YES(int eol)
{
	char buf[1024];
	char *fgets();

	buf[0] = 'n';
	fgets(buf, 1024, stdin);
	return((buf[0] == 'y') || (eol && ((buf[0] == '\n') || (buf[0] == '\0'))));
}

int readnum()
{
	char buf[1024];
	char *fgets();
	int n = 0;

	if(!fgets(buf, 1024, stdin))
		endit(1);
	sscanf(buf, "%d", &n);
	return(n);
}

void rest_user()
{
	static int done = 0;

	if(done++)
		return;
	if(!ok)
		printf("Thank you, that is all\n\n");
}

int endit(val)
{
	exit(val);
}

int migrate(int to)
{
	int mfd = open("/proc/self/migrate", 1);
	char t[15];
	int result;
	
	sprintf(t, "%d\n", to);
	result = write(mfd, t, strlen(t));
	close(mfd);
	return(result < 0);
}

int whereami()
{
	FILE *f = fopen("/proc/self/where", "r");
	int w;

	if(fscanf(f, "%d", &w) != 1)
		error_msg(NOT_MOSIX);
	fclose(f);
	if(w == 0)
	{
		f = fopen("/proc/mosix/admin/mospe", "r");
		if(fscanf(f, "%d", &w) != 1)
			error_msg(NOT_MOSIX);
		fclose(f);
	}
	return(w);
}

int howmany(int where)
{
	char fn[50];
	FILE *f;
	int n;

	sprintf(fn, "/proc/mosix/nodes/%d/cpus", where);
	if(!(f = fopen(fn, "r")) || fscanf(f, "%d", &n) != 1)
		error_msg(NOT_MOSIX);
	fclose(f);
	if(n > MAXSMP)
	{
		fprintf(stderr, "Sorry, node %d is an %d-processor SMP,\n", where, n);
		fprintf(stderr, "which is more than the maximum of %d allowed by this program.\n", MAXSMP);
		exit(1);
	}
	if(n <= 0)
	{
		errno = -n;
		sprintf(fn, "Sorry, node %d is inaccessible", where);
		perror(fn);
		exit(1);
	}
	if(n > 1)
		printf("Node %d has %d processors\n", where, n);
	return(n);
}

void tunepass(int o)
{
	register int i;
	int m;
	char cmd[20], tline[80];
	FILE *tp = NULL;	/* just for pacify gcc */

	sprintf(cmd, "./tunepass %d", o);
	if(access("tunepass", 1) || !(tp = popen(cmd, "r")))
		error_msg(NO_TUNEPASS);
	for(i = 0 ; i < 2 ; i++)
	{
		if(!fgets(tline, sizeof(tline), tp) ||
			sscanf(tline, "#define %*s %d", &m) != 1)
			error_msg(BAD_TUNEPASS);
		fprintf(hfd, "%s", tline);
		fprintf(hfd2, "%d%c", m, i ? '\n' : ' ');
	}
	pclose(tp);
}
