
/*
 * @(#)option.c 1.5 01/01/25
 * 
 * Copyright (c) Steve Holden and Rich Burridge. All rights reserved.
 * 
 * This software is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.
 * 
 * You should have received a copy of the GNU Library General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "option.h"

#ifdef MIME_SUPPORT
char        *boundary;
int         paginate;
int         typesize;
#endif /* MIME_SUPPORT */

char        *prologue;            /* Name of PostScript prologue file. */
char        proname[MAXPATHLEN];  /* Full pathname of the prologue file. */
char        *ptype;               /* Prologue type. */

enum paper_type paper_size;       /* Paper size - default None. */

static char *addhdrs[MAXHDRS];    /* Additional headers to be printed. */
static char *aliasname;           /* User name alias on output. */
static char *appname;             /* Application (program) name. */
static char *configfile;          /* Alternate X resource file. */
static char *gsubject;            /* Global subject set from command line. */
static char *mpres[MAXRESOURCES]; /* Mp resource names. */
static char *namelist[MAXNAMES];  /* List of file names from argv. */
static char *printer_name;        /* Printer name (if any) for output. */
static char *remhdrs[MAXHDRS];    /* List of headers not to be printed. */

static int addhdr_num;    /* Number of additional header lines to print. */
static int allhdrs;       /* If set, print out all mail/news headers. */
static int article;       /* Set for news in "Article from " format. */
static int banners;       /* Print header/footer banners on each page? */
static int bsdprint;      /* Set if this is a BSD printer system. */
static int burstpage;     /* Print a burst page on the printer? */
static int content;       /* Set if Content-Length: has message length. */
static int copies;        /* Number of copies of each file to print. */
static int cpos;          /* Current column position for usage output. */
static int digest;        /* Are we are printing a mail digest. */
static int duplex;        /* Print on both sides of the paper? */
static int elm_if;        /* ELM mail frontend intermediate file format. */
static int flip;          /* Flip top and bottom banners. */
static int folder;        /* Set if we are printing a mail folder. */
static int forcepage;     /* Force next page to a new page or column? */
static int landscape;     /* Set if we are printing in landscape mode. */
static int llen;          /* Number of characters per line. */
static int modtime;       /* If set, use last modification time. */
static int namefields;    /* Number of "words" from passwd gecos. */
static int namelength;    /* Maximum # of chars. from passwd gecos. */
static int newpage;       /* If set, new page for each file if duplex. */
static int number;        /* If set, than number each output line. */
static int numcols;       /* Number of columns per page. */
static int numnames;      /* Number of file names from command line. */
static int plen;          /* Number of lines per page. */
static int print_orig;    /* Print From rather than To in mail header. */
static int print_ps;      /* Print PostScript files if set. */
static int remhdr_num;    /* # of additional header lines not to print. */
static int tabsize;       /* Number of spaces in a tab character. */
static int text_doc;      /* Printing normal text. */
static int toprinter;     /* Send output to the printer? */
static int tumble;        /* Tumble the duplex'ed page? */
static int wrapwords;     /* Wrap words of output. */

static int bmargin;       /* Bottom margin size (in points). */
static int lmargin;       /* Left margin size   (in points). */
static int rmargin;       /* Right margin size  (in points). */
static int tmargin;       /* Top margin size    (in points). */

static int cols_given;    /* "-columns    #" command line option given? */
static int llen_given;    /* "-linelength #" command line option given? */
static int plen_given;    /* "-pagelength #" command line option given? */

static XrmDatabase rDB;   /* Mp X resources database. */

static char    *set_bool(int);
static char    *get_resource(enum res_type);

static int     argcmp              (char *, char *, int);
static int     get_bool_resource   (enum res_type, int *);
static int     get_int_resource    (enum res_type, int *);
static int     get_str_resource    (enum res_type, char *);
static int     save_resources      (void);

static void    get_hdr_resource    (enum res_type, char *);
static void    load_mp_defs        (char *);
static void    put_hdr_resource    (enum res_type);
static void    put_resource        (enum res_type, char *);
static void    read_str            (char **, char *);
static void    usage               (void);
static void    usageoption         (char *);


/* Adjust page and line length depending upon paper size and orientation. */

void
adjust_paper_size()
{
    switch (paper_size) {
        case P_US:
            if (!llen_given) {
                llen = (landscape) ? 80 : 80;
            }
            if (!plen_given) {
                plen = (landscape) ? 68 : 72;
            }
            break;

        case P_A4:
            if (!llen_given) {
                llen = (landscape) ? 90 : 80;
            }
            if (!plen_given) {
                plen = (landscape) ? 66 : 78;
            }
            break;

        default:
            return;
    }

    llen_given = 1;
    plen_given = 1;
}


/*
 * Does a string compare between a1 and a2. To return '0', a1 and a2 must
 * match to the length of a2, and that length has to be at least 'minlen'.
 * Otherwise, return non-zero.
 */

static int
argcmp(char *a1, char *a2, int minlen)
{
    if ((int) strlen(a1) < minlen || (int) strlen(a2) < minlen) {
        return(1);
    }
    if ((int) strlen(a1) > (int) strlen(a2)) {
        return(1);
    }

    return(strncmp(a1, a2, strlen(a1)));
}


static char *
set_bool(int value)
{
    return((value) ? "true" : "false");
}


void
free_hdr(enum header_type htype, int n)
{
    if (htype == H_ADD) {
        FREE(addhdrs[n]);
    } else {
        FREE(remhdrs[n]);
    }
}


void
get_appname(int argc, char **argv)
{
    char *ptr;
    int  i;

    if ((ptr = strrchr(argv[0], '/')) != NULL) {
        read_str(&appname, ptr + 1);
    } else {
        read_str(&appname, argv[0]);
    }

/*
 * Search through all the command line arguments, looking for -name.
 * If it's present, then this name will be used, when looking for X
 * resources for this application. When the rest of the command line
 * arguments are checked later on, then the -name argument (if found)
 * is ignored.
 */

    for (i = 1; i < argc; i++) {
        if (ARG_EQUAL("-name", 3)) {
            if (++i < argc) {
                read_str(&appname, argv[i]);
            } else {
                usage();
            }
            break;
        }
    }
}


static int
get_bool_resource(enum res_type rtype, int *boolval)
{
    char *val, tempstr[MAXLINE];
    int  len, n;

    if ((val = get_resource(rtype)) == NULL) {
        return(0);
    }
    STRCPY(tempstr, val);
    len = strlen(tempstr);
    for (n = 0; n < len; n++) {
        if (isupper((int) tempstr[n])) {
            tempstr[n] = tolower(tempstr[n]);
        }
    }
    if (EQUAL(tempstr, "true")) {
        *boolval = 1;
    } else {
        *boolval = 0;
    }

    return(1);
}


char *
get_filename(int n)
{
    return(namelist[n]);
}


char *
get_hdr(enum header_type htype, int n)
{
    if (htype == H_ADD) {
        return(addhdrs[n]);
    } else {
        return(remhdrs[n]);
    }
}


/* Adds 'hdr' header argument to the given list if it doesn't already exist. */

void
get_hdr_arg(char *hdr, enum header_type htype, int *n)
{
    char arg[MAXLINE], **list;
    int  i, len;

    if (htype == H_ADD) {
        list = addhdrs;
    } else {
        list = remhdrs;
    }
    SSCANF(hdr, "%s", arg);
    len = strlen(arg);
    if (arg[len - 1] != ':') {
        STRCAT(arg, ":");
    }
    for (i = 0; i < *n; i++) {
        if (CASE_EQUAL(list[i], arg)) {
            return;
        }
    }
    list[(*n)++] = strdup(arg);
}


static void
get_hdr_resource(enum res_type rtype, char *str)
{
    char *ptr;

    if ((ptr = strtok(str, ", \t")) != NULL) {
        do {
            if (rtype == R_ADDHDRS) {
                get_hdr_arg(ptr, H_ADD, &addhdr_num);
            } else {
                get_hdr_arg(ptr, H_REMOVE, &remhdr_num);
            }
            ptr = strtok((char *) NULL, ", \t");
        } while (ptr);
    }
}


static int
get_int_resource(enum res_type rtype, int *intval)
{
    char *val;

    if ((val = get_resource(rtype)) == NULL) {
        return(0);
    }
    *intval = atoi(val);

    return(1);
}


int
get_margin(enum margin_type mtype)
{
    switch (mtype) {
        case M_BOTTOM:
            return(bmargin);

        case M_LEFT:
            return(lmargin);

        case M_RIGHT:
            return(rmargin);

        case M_TOP:
            return(tmargin);
    }

    return(0);
}


/* Read and process command line options. */

void
get_options(int argc, char **argv)
{
    int i;

    if (EQUAL(appname, "digestp")) {
        digest = 1;
    } else if (EQUAL(appname, "filep")) {
        text_doc = 1;
    } else if (EQUAL(appname, "filofaxp")) {
        ptype = PRO_FILOFAX;
    } else if (EQUAL(appname, "filofaxpp")) {
        ptype = PRO_FILOFAXP;
    } else if (EQUAL(appname, "franklinp")) {
        ptype = PRO_FRANKLIN;
    } else if (EQUAL(appname, "mailp")) {
        toprinter = 1;
    } else if (EQUAL(appname, "newsp")) {
        article = 1;
    } else if (EQUAL(appname, "proplanp")) {
        ptype = PRO_PRO_PLAN;
    } else if (EQUAL(appname, "timemanp")) {
        ptype = PRO_TIME_MANAGER;
    } else if (EQUAL(appname, "timesysip")) {
        ptype = PRO_TS_INTL;
    } else if (EQUAL(appname, "timesyspp")) {
        ptype = PRO_TS_PARTNER;
    } else {
        toprinter = 0;
    }

    for (i = 1; i < argc; i++) {
        if (argv[i][0] != '-') {
            if (numnames < MAXNAMES) {
                namelist[numnames++] = argv[i];
                if (numnames == MAXNAMES) {
                    FPRINTF(stderr,
                        "%s: too many filenames.  Only using first %d\n",
                        appname, MAXNAMES);
                }
            }
        } else if (                              /* A4 paper size. */
#ifdef OLD_OPTIONS
               OLD_ARG_EQ("-A4") ||
#endif /* OLD_OPTIONS */
               ARG_EQUAL("-a4", 3)) {
            paper_size = P_A4;
        } else if (ARG_EQUAL("-addhdr", 3)) {    /* Another header to print. */
            if (++i < argc) {
                get_hdr_arg(argv[i], H_ADD, &addhdr_num);
            }
        } else if (ARG_EQUAL("-alias", 4)) {    /* Users name on output. */
            if (++i < argc) {
                read_str(&aliasname, argv[i]);
            }
        } else if (ARG_EQUAL("-allhdrs", 4)) {  /* Print out all headers.. */
            allhdrs = 1;
        } else if (                             /* "Article from" format. */
#ifdef OLD_OPTIONS
             OLD_ARG_EQ("-a") ||
#endif /* OLD_OPTIONS */
             ARG_EQUAL("-article", 3)) {
            article = 1;
        } else if (ARG_EQUAL("-bottom", 3)) {      /* Bottom margin size. */
            if (++i < argc) {
                bmargin = atoi(argv[i]);
            }
        } else if (ARG_EQUAL("-bsdprint", 3)) {    /* BSD printer system. */
            bsdprint = 1;
        } else if (ARG_EQUAL("-chars", 3)) {       /* Gecos (chars). */
            if (++i < argc)
                namelength = atoi(argv[i]);
        } else if (ARG_EQUAL("-columns", 4)) {     /* # of columns of output. */
            if (++i < argc) {
                numcols = atoi(argv[i]);
            }
            cols_given = 1;
        } else if (ARG_EQUAL("config", 5)) {    /* Alternate resource file. */
            if (++i < argc) {
                read_str(&configfile, argv[i]);
            }
        } else if (ARG_EQUAL("-content", 5)) {  /* Use Content-Length: header */
            content = 1;
        } else if (ARG_EQUAL("-copies", 4)) {    /* # of copies to print. */
            if (++i < argc) {
                copies = atoi(argv[i]);
            }
        } else if (                              /* Print digest. */
#ifdef OLD_OPTIONS
               OLD_ARG_EQ("-d") ||
#endif /* OLD_OPTIONS */
               ARG_EQUAL("-digest", 2)) {
            digest = 1;

        } else if (                              /* ELM mail file format. */
#ifdef OLD_OPTIONS
             OLD_ARG_EQ("-e") ||
#endif /* OLD_OPTIONS */
             ARG_EQUAL("-elm", 2)) {
            elm_if = 1;
            folder = 1;    /* Kind of folder. */
        }
#ifdef OLD_OPTIONS
        else if (OLD_ARG_EQ("-f")) {             /* Filofax output. */
            ptype = PRO_FILOFAX;
        }
#endif /* OLD_OPTIONS */

        else if (ARG_EQUAL("-flip", 3)) {        /* Flip banners. */
            flip = 1;
        } else if (                              /* Print mail folder. */
#ifdef OLD_OPTIONS
             OLD_ARG_EQ("-m") ||
#endif /* OLD_OPTIONS */
             ARG_EQUAL("-folder", 4)) {
            folder = 1;
        } else if (ARG_EQUAL("-forcepage", 4)) {  /* Force new page. */
            forcepage = 1;
        } else if (                               /* Print originators name. */
#ifdef OLD_OPTIONS
             OLD_ARG_EQ("-F") ||
#endif /* OLD_OPTIONS */
             ARG_EQUAL("-from", 4)) {
            print_orig = 1;
        } else if (                               /* Print usage message. */
#ifdef OLD_OPTIONS
             OLD_ARG_EQ("-?") ||
#endif /* OLD_OPTIONS */
             ARG_EQUAL("-help", 2)) {
            usage();
        } else if (                               /* Landscape printing. */
#ifdef OLD_OPTIONS
             OLD_ARG_EQ("-l") ||
#endif /* OLD_OPTIONS */
             ARG_EQUAL("-landscape", 3)) {
            landscape = 1;
            ptype = PRO_LANDSCAPE;
        } else if (ARG_EQUAL("-left", 3)) {       /* Left margin size. */
            if (++i < argc) {
                lmargin = atoi(argv[i]);
            }
        } else if (ARG_EQUAL("-linelength", 3)) {    /* # of chars per line. */
            if (++i < argc) {
                llen = atoi(argv[i]);
            }
            llen_given = 1;
        } else if (ARG_EQUAL("-modtime", 2)) {    /* Print last mod. time. */
            modtime = 1;
        } else if (ARG_EQUAL("-name", 3)) {   /* Alternate application name. */
            if (++i < argc) {
                read_str(&appname, argv[i]);
            }
        } else if (ARG_EQUAL("-newpage", 3)) {    /* New page if duplex. */
            newpage = 1;
        } else if (ARG_EQUAL("-nobanners", 5)) {  /* No hdr/footer banners. */
            banners = 0;
        } else if (ARG_EQUAL("-noburstpage", 5)) {    /* No burst page. */
            burstpage = 0;
        } else if (ARG_EQUAL("-number", 3)) {         /* Number lines. */
            number = 1;
        } else if (ARG_EQUAL("-onesided", 3)) {       /* Simplex printing. */
            duplex = 0;
        } else if (ARG_EQUAL("-organiser", 3)) {      /* Organiser type. */
            if (++i < argc) {
                if (ARG_EQUAL("filofax", 7)) {        /* Filofax. */
                    ptype = PRO_FILOFAX;
                } else if (ARG_EQUAL("filofaxp", 8)) {  /* Filofaxp. */
                    ptype = PRO_FILOFAXP;
                } else if (ARG_EQUAL("franklin", 2)) {  /* Franklin. */
                    ptype = PRO_FRANKLIN;
                } else if (ARG_EQUAL("proplan", 2)) {   /* Proplan. */
                    ptype = PRO_PRO_PLAN;
                } else if (ARG_EQUAL("timeman", 2)) {   /* Time Manager. */
                    ptype = PRO_TIME_MANAGER;
                } else if (ARG_EQUAL("tsintl", 3)) {    /* Tm System (Intl). */
                    ptype = PRO_TS_INTL;
                } else if (ARG_EQUAL("tspartner", 3)) { /* Tm System (Prtnr). */
                    ptype = PRO_TS_PARTNER;
                }
            }
        } else if (ARG_EQUAL("-pagelength", 3)) {    /* # of lines per page. */
            if (++i < argc) {
                plen = atoi(argv[i]);
            }
            plen_given = 1;
        } else if (ARG_EQUAL("-portrait", 4)) {    /* Portrait printing. */
            landscape = 0;
            ptype = PRO_NORMAL;
        } else if (                               /* Print PostScript files. */
#ifdef OLD_OPTIONS
               OLD_ARG_EQ("-PS") ||
#endif /* OLD_OPTIONS */
               ARG_EQUAL("-postscript", 4)) {
            print_ps = 1;
        } else if (ARG_EQUAL("-printer", 4)) {    /* Printer name. */
            toprinter = 1;
            if (++i < argc) {
                if (argv[i][0] != '-') {
                    printer_name = argv[i];
                } else {
                    --i;
                }
            }
        } else if (                               /* New prologue file. */
#ifdef OLD_OPTIONS
               OLD_ARG_EQ("-p") ||
#endif /* OLD_OPTIONS */
               ARG_EQUAL("-prologue", 5)) {
            if (++i < argc) {
                STRCPY(proname, argv[i]);
            }
        } else if (ARG_EQUAL("-remhdr", 3)) {    /* Remove header to print. */
            if (++i < argc) {
                get_hdr_arg(argv[i], H_REMOVE, &remhdr_num);
            }
        } else if (ARG_EQUAL("-right", 3)) {     /* Right margin size. */
            if (++i < argc) {
                rmargin = atoi(argv[i]);
            }
        } else if (                              /* New subject line. */
#ifdef OLD_OPTIONS
               OLD_ARG_EQ("-s") ||
#endif /* OLD_OPTIONS */
               ARG_EQUAL("-subject", 2)) {
            if (++i < argc) {
                gsubject = argv[i];
            }
        } else if (ARG_EQUAL("-tab", 3)) {       /* # of spaces in a tab. */
            if (++i < argc) {
                tabsize = atoi(argv[i]);
            }
        } else if (                            /* Print ordinary text file. */
#ifdef OLD_OPTIONS
               OLD_ARG_EQ("-o") ||
#endif /* OLD_OPTIONS */
               ARG_EQUAL("-text", 3)) {
            text_doc = 1;

#ifdef OLD_OPTIONS
        } else if (OLD_ARG_EQ("-tm")) {            /* Time Manager. */
            ptype = PRO_TIME_MANAGER;
#endif /* OLD_OPTIONS */

        } else if (ARG_EQUAL("-top", 3)) {         /* Top margin size. */
            if (++i < argc) {
                tmargin = atoi(argv[i]);
            }
#ifdef OLD_OPTIONS
        } else if (OLD_ARG_EQ("-ts")) {            /* Time/System Intl. */
            ptype = PRO_TS_INTL;
#endif /* OLD_OPTIONS */
        } else if (ARG_EQUAL("-tumble", 3)) {      /* Tumble duplex page. */
            tumble = 1;
        } else if (ARG_EQUAL("-twosided", 3)) {    /* Duplex printing. */
            duplex = 1;
        } else if (                                /* US paper size. */
#ifdef OLD_OPTIONS
             OLD_ARG_EQ("-US") ||
#endif /* OLD_OPTIONS */
             ARG_EQUAL("-us", 2)) {
            paper_size = P_US;

        } else if (                                /* Print version number. */
#ifdef OLD_OPTIONS
             OLD_ARG_EQ("-v") ||
#endif /* OLD_OPTIONS */
             ARG_EQUAL("-version", 2)) {
            FPRINTF(stderr, "%s version %s\n\n", appname, MPDIST_VERSION);
            exit(1);
        } else if (ARG_EQUAL("-words", 3)) {       /* Gecos (words). */
            if (++i < argc) {
                namefields = atoi(argv[i]);
            }
        } else if (ARG_EQUAL("-wrap", 3)) {        /* Wrap words of output. */
            wrapwords = 1;
        }
    }

/*
 * If user specified a -column option of 1 and we are doing
 * landscape, then change to the correct prologue type.
 */

    if (cols_given && numcols == 1 && landscape) {
        ptype = PRO_LANDSCAPE_C1;
    }
}


static char *
get_resource(enum res_type rtype)
{
    char     app[MAXLINE], cstr[MAXLINE], nstr[MAXLINE], res[MAXLINE];
    char     *str_type[20];
    XrmValue value;

    STRCPY(app, appname);
    STRCPY(res, mpres[(int) rtype]);
    if (isupper((int) app[0])) {
        app[0] = tolower(app[0]);
    }
    SPRINTF(nstr, "%s.%s", app, res);

    if (islower((int) res[0])) {
        res[0] = toupper(res[0]);
    }
    if (islower((int) app[0])) {
        app[0] = toupper(app[0]);
    }
    SPRINTF(cstr, "%s.%s", app, res);

    if (XrmGetResource(rDB, nstr, cstr, str_type, &value) == 0) {
        return((char *) NULL);
    } else {
        return(value.addr);
    }
}


static int
get_str_resource(enum res_type rtype, char *strval)
{
    char *val;

    if ((val = get_resource(rtype)) == NULL) {
        return(0);
    }
    STRCPY(strval, val);

    return(1);
}


int
int_opt_val(enum opt_type otype)
{
    switch (otype) {
        case O_ADDHDRN:
            return(addhdr_num);
        case O_ALLHDRS:
            return(allhdrs);
        case O_ARTICLE:
            return(article);
        case O_BANNERS:
            return(banners);
        case O_BSDPRINT:
            return(bsdprint);
        case O_BURSTPAGE:
            return(burstpage);
        case O_CONTENT:
            return(content);
        case O_COPIES:
            return(copies);
        case O_DIGEST:
            return(digest);
        case O_DUPLEX:
            return(duplex);
        case O_ELM:
            return(elm_if);
        case O_FLIP:
            return(flip);
        case O_FOLDER:
            return(folder);
        case O_FORCEPAGE:
            return(forcepage);
        case O_LANDSCAPE:
            return(landscape);
        case O_LLEN:
            return(llen);
        case O_MODTIME:
            return(modtime);
        case O_NAMEF:
            return(namefields);
        case O_NAMELEN:
            return(namelength);
        case O_NEWPAGE:
            return(newpage);
        case O_NUMBER:
            return(number);
        case O_NUMCOLS:
            return(numcols);
        case O_NUMNAMES:
            return(numnames);
#ifdef MIME_SUPPORT
        case O_PAGIN:
            return(paginate);
#endif /* MIME_SUPPORT */
        case O_PLEN:
            return(plen);
        case O_PR_ORIG:
            return(print_orig);
        case O_PRINT_PS:
            return(print_ps);
        case O_REMHDRN:
            return(remhdr_num);
        case O_TABSIZE:
            return(tabsize);
        case O_TEXTDOC:
            return(text_doc);
        case O_TOPRINT:
            return(toprinter);
        case O_TUMBLE:
            return(tumble);
        case O_WRAPW:
            return(wrapwords);
        default:
            return(0);             /* Shouldn't happen. */
    }
}


int
is_opt_given(enum opt_type otype)
{
    switch (otype) {
        case O_LLEN:
            return(llen_given);
        case O_NUMCOLS:
            return(cols_given);
        case O_PLEN:
            return(plen_given);
        default:
            return(0);
    }
}


/*
 * Load the Mp resources database from the specified config file, or from
 * MPDEFAULTS environment variable (if set), or from $HOME/.mpdefaults.
 */

static void
load_mp_defs(char *file)
{
    char name[MAXPATHLEN], *ptr;

    XrmInitialize();
    if (file && *file != '\0') {
        ptr = file;
    } else if (getenv("HOME") == NULL) {
        rDB = NULL;
        return;
    } else if ((ptr = getenv("MPDEFAULTS")) == NULL) {
        SPRINTF(name, "%s/.mpdefaults", getenv("HOME"));
        ptr = name;
    }
    rDB = XrmGetFileDatabase(ptr);
}


void
Option(void)
{
    int i;

    addhdr_num = 0;         /* Number of additional header lines to print. */
    aliasname = NULL;       /* User name alias on output. */
    allhdrs = 0;            /* Just print "standard" headers by default. */
    appname = NULL;         /* Application name for resources. */
    article = 0;            /* Set for news in "Article from " format. */
    banners = 1;            /* Print header/footer banners on each page. */
    burstpage = 1;          /* Print a burst page by default. */
    configfile = NULL;      /* Alternate X resource file. */
    content = 0;            /* Set if Content-Length: has message length. */
    copies = DEFCOPIES;     /* Output one copy of each file by default. */
    cpos = 0;               /* Current column position for usage output. */
    digest = 0;             /* Set if printing a mail digest. */
    duplex = 1;             /* Print two sided by default. */
    elm_if = 0;             /* ELM mail frontend intermediate file format. */
    flip = 0;               /* If set, flip top and bottom banners. */
    folder = 0;             /* Set if we are printing a mail folder. */
    forcepage = 0;          /* A new page will be printed in next column. */
    gsubject = NULL;        /* Global subject set from command line. */
    landscape = 0;          /* Set if we are printing in landscape mode. */
    llen = LINELENGTH;      /* Number of characters per line. */
    modtime = 0;            /* If set, use last modification time. */
    newpage = 0;            /* Don't start new page if printing duplex. */
    number = 0;             /* Don't number lines by default. */
    numcols = DEFCOLUMNS;   /* Number of columns per page. */
    numnames = 0;           /* Number of file names from argv. */
    paper_size = P_NONE;    /* Paper size - default fill paper. */
    plen = PAGELENGTH;      /* Number of lines per page. */
    print_orig = 0;         /* Print From rather than To in mail header. */
    print_ps = 0;           /* Print PostScript files if set. */
    printer_name = NULL;    /* Default printer. */
    prologue = NULL;        /* Name of PostScript prologue file. */
    proname[0] = '\0';      /* Full pathname of the prologue file. */
    remhdr_num = 0;         /* No. of additional header lines not to print. */
    tabsize = TABSIZE;      /* Number of spaces in a tab character. */
    text_doc = 0;           /* Printing normal text. */
    toprinter = 1;          /* Output to a printer. */
    tumble = 0;             /* Don't tumble the duplex'ed page. */
    wrapwords = 0;          /* Wrap words of output if set. */

    bmargin = DEFMARGIN;    /* Default bottom margin size (in points). */
    lmargin = DEFMARGIN;    /* Default left margin size   (in points). */
    rmargin = DEFMARGIN;    /* Default right margin size  (in points). */
    tmargin = DEFMARGIN;    /* Default top margin size    (in points). */

    cols_given = 0;         /* Set if "-columns    #" on command line. */
    llen_given = 0;         /* Set if "-linelength #" on command line. */
    plen_given = 0;         /* Set if "-pagelength #" on command line. */

#ifdef MIME_SUPPORT
    boundary = NULL;
    paginate = 1;
    typesize = 10;
#endif /* MIME_SUPPORT */

/*
 * Let installer decide (in top-level Makefile) if BSD or System V
 * printer spooling should be used.
 */

#ifdef BSD_PRINT
    bsdprint = 1;
#else
    bsdprint = 0;
#endif /* BSD_PRINT */

/*
 * Try to get location of the mp prologue file from an environment
 * variable. If it's not found, then use the default value.
 */

    if ((prologue = getenv("MP_PROLOGUE")) == NULL) {
        prologue = PROLOGDIR;
    }
    ptype = PRO_NORMAL;

#ifdef GECOSFIELDS
    namefields = GECOSFIELDS;    /* New no. of "words" from passwd gecos. */
#else
    namefields = NAMEFIELDS;     /* Not supplied; use default. */
#endif /* GECOSFIELDS */

#ifdef GECOSLENGTH
    namelength = GECOSLENGTH;    /* New max. no. of chars. from passwd gecos. */
#else
    namelength = NAMELENGTH;     /* Not supplied; use default. */
#endif /* GECOSLENGTH */

    for (i = 0; i < MAXHDRS; i++) {
        addhdrs[i] = NULL;
    }
    for (i = 0; i < MAXHDRS; i++) {
        remhdrs[i] = NULL;
    }
    for (i = 0; i < MAXNAMES; i++) {
        namelist[i] = NULL;
    }
    for (i = 0; i < MAXRESOURCES; i++) {
        mpres[i] = NULL;
    }

    read_str(&mpres[(int) R_FTYPE],     "fileType");
    read_str(&mpres[(int) R_OTYPE],     "outputType");
    read_str(&mpres[(int) R_ORGANISER], "organiserType");
    read_str(&mpres[(int) R_PSIZE],     "paperSize");
    read_str(&mpres[(int) R_HDRPAGE],   "printHeaderPage");
    read_str(&mpres[(int) R_MAIL_FDR],  "mailFolder");
    read_str(&mpres[(int) R_MAIL_DIG],  "mailDigest");
    read_str(&mpres[(int) R_SUBJECT],   "subject");
    read_str(&mpres[(int) R_LLENGTH],   "lineLength");
    read_str(&mpres[(int) R_PLENGTH],   "pageLength");
    read_str(&mpres[(int) R_COPIES],    "copies");
    read_str(&mpres[(int) R_PNAME],     "printerName");
    read_str(&mpres[(int) R_DSIDED],    "doubleSided");
    read_str(&mpres[(int) R_ADDHDRS],   "addHeaders");
    read_str(&mpres[(int) R_REMHDRS],   "removeHeaders");
    read_str(&mpres[(int) R_ALIAS],     "aliasName");
    read_str(&mpres[(int) R_PROLOG],    "prologueName");
    read_str(&mpres[(int) R_NCOLUMNS],  "numberColumns");
    read_str(&mpres[(int) R_TABSIZE],   "tabSize");
    read_str(&mpres[(int) R_NAMEC],     "nameChars");
    read_str(&mpres[(int) R_NAMEW],     "nameWords");
    read_str(&mpres[(int) R_WRAP],      "wrapWords");
    read_str(&mpres[(int) R_FLIP],      "flipBanners");
    read_str(&mpres[(int) R_FORCEPAGE], "forcePage");
    read_str(&mpres[(int) R_ELM],       "elmMail");
    read_str(&mpres[(int) R_NUMBER],    "numberLines");
    read_str(&mpres[(int) R_FROM],      "useFrom");
    read_str(&mpres[(int) R_CONTENT],   "useContent");
    read_str(&mpres[(int) R_MOD],       "useModTime");
    read_str(&mpres[(int) R_TUMBLE],    "tumble");
    read_str(&mpres[(int) R_ALLHDRS],   "allHeaders");
    read_str(&mpres[(int) R_BSDPRINT],  "bsdPrinter");
    read_str(&mpres[(int) R_BMARGIN],   "bottomMargin");
    read_str(&mpres[(int) R_LMARGIN],   "leftMargin");
    read_str(&mpres[(int) R_RMARGIN],   "rightMargin");
    read_str(&mpres[(int) R_TMARGIN],   "topMargin");
    read_str(&mpres[(int) R_BANNERS],   "banners");
    read_str(&mpres[(int) R_NEWPAGE],   "newPage");

#ifdef MIME_SUPPORT
    read_str(&mpres[(int) R_BOUNDARY],  "boundary");
    read_str(&mpres[(int) R_TYPESIZE],  "typeSize");
    read_str(&mpres[(int) R_PAGINATE],  "paginate");
#endif /* MIME_SUPPORT */

    rDB = NULL;        /* X Resource database. */
}


static void
put_hdr_resource(enum res_type rtype)
{
    char **hdrs, strval[MAXLINE];
    int  i, n;

    if (rtype == R_ADDHDRS) {
        hdrs = addhdrs;
        n = addhdr_num;
    } else {
        hdrs = remhdrs;
        n = remhdr_num;
    }

    strval[0] = '\0';
    for (i = 0; i < n; i++) {
        STRCAT(strval, hdrs[i]);
        STRCAT(strval, ",");
    }
    strval[strlen(strval) - 1] = '\0';
    if (rtype == R_ADDHDRS) {
        put_resource(R_ADDHDRS, strval);
    } else {
        put_resource(R_REMHDRS, strval);
    }
}


static void
put_resource(enum res_type rtype, char *rval)
{
    char app[MAXLINE], resource[MAXLINE];

    STRCPY(app, appname);
    if (isupper((int) app[0])) {
        app[0] = tolower(app[0]);
    }
    SPRINTF(resource, "%s.%s", app, mpres[(int) rtype]);
    XrmPutStringResource(&rDB, resource, (rval) ? rval : "");
}


void
read_resources(void)
{
    int  boolval, intval;
    char str[MAXLINE];

    load_mp_defs(configfile);                 /* Get mp X resources. */
    if (get_str_resource(R_FTYPE, str)) {
        if (CASE_EQUAL(str, "news")) {
            article = 1;
        } else if (CASE_EQUAL(str, "postscript")) {
            print_ps = 1;
        } else if (CASE_EQUAL(str, "text")) {
            text_doc = 1;
        }
    }
    if (get_str_resource(R_OTYPE, str)) {
        if (CASE_EQUAL(str, "landscape")) {
            landscape = 1;
            ptype = PRO_LANDSCAPE;
        } else if (!CASE_EQUAL(str, "organiser")) {
            landscape = 0;
            ptype = PRO_NORMAL;
        }
    }
    if (get_str_resource(R_ORGANISER, str)) {
        if (CASE_EQUAL(str, "filofax")) {
            ptype = PRO_FILOFAX;
        } else if (CASE_EQUAL(str, "filofaxp")) {
            ptype = PRO_FILOFAXP;
        } else if (CASE_EQUAL(str, "franklin")) {
            ptype = PRO_FRANKLIN;
        } else if (CASE_EQUAL(str, "proplan")) {
            ptype = PRO_PRO_PLAN;
        } else if (CASE_EQUAL(str, "timeman")) {
            ptype = PRO_TIME_MANAGER;
        } else if (CASE_EQUAL(str, "tsintl")) {
            ptype = PRO_TS_INTL;
        } else if (CASE_EQUAL(str, "tspartner")) {
            ptype = PRO_TS_PARTNER;
        }
    }
    if (get_str_resource(R_PSIZE, str)) {
        if (CASE_EQUAL(str, "a4")) {
            paper_size = P_A4;
        } else if (CASE_EQUAL(str, "us")) {
            paper_size = P_US;
        }
    }
    if (get_bool_resource(R_HDRPAGE, &boolval)) {
        burstpage = boolval;
    }
    if (get_bool_resource(R_BANNERS, &boolval)) {
        banners = boolval;
    }
    if (get_bool_resource(R_MAIL_FDR, &boolval)) {
        folder = boolval;
    }
    if (get_bool_resource(R_MAIL_DIG, &boolval)) {
        digest = boolval;
    }
    if (get_bool_resource(R_DSIDED, &boolval)) {
        duplex = boolval;
    }
    if (get_bool_resource(R_NEWPAGE, &boolval)) {
        newpage = boolval;
    }

    if (get_str_resource(R_SUBJECT, str)) {
        read_str(&gsubject, str);
    }
    if (get_str_resource(R_PNAME, str)) {
        read_str(&printer_name, str);
    }

    if (get_int_resource(R_LLENGTH, &intval)) {
        llen_given = 1;
        llen = intval;
    }
    if (get_int_resource(R_PLENGTH, &intval)) {
        plen_given = 1;
        plen = intval;
    }
    if (get_int_resource(R_COPIES, &intval)) {
        copies = intval;
    }

    if (get_str_resource(R_ADDHDRS, str)) {
        get_hdr_resource(R_ADDHDRS, str);
    }
    if (get_str_resource(R_REMHDRS, str)) {
        get_hdr_resource(R_REMHDRS, str);
    }

    if (get_str_resource(R_ALIAS, str)) {
        read_str(&aliasname, str);
    }
    if (get_str_resource(R_PROLOG, str)) {
        STRCPY(proname, str);
    }

    if (get_int_resource(R_NCOLUMNS, &intval)) {
        cols_given = 1;
        numcols = intval;
    }
    if (get_int_resource(R_TABSIZE, &intval)) {
        tabsize = intval;
    }
    if (get_int_resource(R_NAMEC, &intval)) {
        namelength = intval;
    }
    if (get_int_resource(R_NAMEW, &intval)) {
        namefields = intval;
    }

    if (get_bool_resource(R_BSDPRINT, &boolval)) {
        bsdprint = boolval;
    }
    if (get_bool_resource(R_WRAP, &boolval)) {
        wrapwords = boolval;
    }
    if (get_bool_resource(R_FLIP, &boolval)) {
        flip = boolval;
    }
    if (get_bool_resource(R_FORCEPAGE, &boolval)) {
        forcepage = boolval;
    }
    if (get_bool_resource(R_ELM, &boolval)) {
        elm_if = boolval;
    }
    if (get_bool_resource(R_NUMBER, &boolval)) {
        number = boolval;
    }
    if (get_bool_resource(R_FROM, &boolval)) {
        print_orig = boolval;
    }
    if (get_bool_resource(R_CONTENT, &boolval)) {
        content = boolval;
    }
    if (get_bool_resource(R_MOD, &boolval)) {
        modtime = boolval;
    }
    if (get_bool_resource(R_TUMBLE, &boolval)) {
        tumble = boolval;
    }
    if (get_bool_resource(R_ADDHDRS, &boolval)) {
        allhdrs = boolval;
    }

    if (get_int_resource(R_BMARGIN, &intval)) {
        bmargin = intval;
    }
    if (get_int_resource(R_LMARGIN, &intval)) {
        lmargin = intval;
    }
    if (get_int_resource(R_RMARGIN, &intval)) {
        rmargin = intval;
    }
    if (get_int_resource(R_TMARGIN, &intval)) {
        tmargin = intval;
    }

#ifdef MIME_SUPPORT
    if (get_str_resource(R_BOUNDARY, str)) {
        read_str(&boundary, str);
    }
    if (get_int_resource(R_TYPESIZE, &intval)) {
        typesize = intval;
    }
    if (get_bool_resource(R_PAGINATE, &boolval)) {
        paginate = boolval;
    }
#endif /* MIME_SUPPORT */
}


static void
read_str(char **str, char *value)
{
    if (*str != NULL) {
        FREE(*str);
    }
    if (value != NULL) {
        *str = (char *) malloc((unsigned) (strlen(value) + 1));
        STRCPY(*str, value);
    } else {
        *str = NULL;
    }
}


static int
save_resources(void)
{
    char        *home, *filename;
    struct stat statbuf;

    if ((home = getenv("HOME")) == NULL) {
        return(1);
    }

    if ((filename = getenv("MPDEFAULTS")) == NULL) {
        filename = (char *) calloc(1, strlen(home) + 18);
        SPRINTF(filename, "%s/.mpdefaults", home);
    }

/* If file exists but user does not have access. */

    if (stat(filename, &statbuf) != -1 && access(filename, W_OK) != 0) {
        FREE(filename);
        return(1);
    }

/* If file does not exist this call will create it. */

    XrmPutFileDatabase(rDB, filename);
    FREE(filename);
    return(0);
}


void
set_int_opt_val(enum opt_type otype, int value)
{
    switch (otype) {
        case O_ALLHDRS:
            allhdrs = value;
            break;
        case O_ARTICLE:
            article = value;
            break;
        case O_BANNERS:
            banners = value;
            break;
        case O_BURSTPAGE:
            burstpage = value;
            break;
        case O_CONTENT:
            content = value;
            break;
        case O_COPIES:
            copies = value;
            break;
        case O_DIGEST:
            digest = value;
            break;
        case O_DUPLEX:
            duplex = value;
            break;
        case O_ELM:
            elm_if = value;
            break;
        case O_FLIP:
            flip = value;
            break;
        case O_FOLDER:
            folder = value;
            break;
        case O_FORCEPAGE:
            forcepage = value;
            break;
        case O_LANDSCAPE:
            landscape = value;
            break;
        case O_LLEN:
            llen = value;
            break;
        case O_MODTIME:
            modtime = value;
            break;
        case O_NEWPAGE:
            newpage = value;
            break;
        case O_NUMBER:
            number = value;
            break;
        case O_NUMCOLS:
            numcols = value;
            break;
#ifdef MIME_SUPPORT
        case O_PAGIN:
            paginate = value;
            break;
#endif /* MIME_SUPPORT */
        case O_PLEN:
            plen = value;
            break;
        case O_PR_ORIG:
            print_orig = value;
            break;
        case O_PRINT_PS:
            print_ps = value;
            break;
        case O_TABSIZE:
            tabsize = value;
            break;
        case O_TEXTDOC:
            text_doc = value;
            break;
        case O_TUMBLE:
            tumble = value;
            break;
        case O_WRAPW:
            wrapwords = value;
            break;
        default:;                    /* Do nothing. */
    }
}


void
set_margin(enum margin_type mtype, int value)
{
    switch (mtype) {
        case M_BOTTOM:
            bmargin = value;
            break;
        case M_LEFT:
            lmargin = value;
            break;
        case M_RIGHT:
            rmargin = value;
            break;
        case M_TOP:
            tmargin = value;
            break;
    }
}


void
set_str_opt_val(enum opt_type otype, char *value)
{
    switch (otype) {
        case O_GSUBJECT:
            gsubject = value;
            break;
        case O_PR_NAME:
            printer_name = value;
            break;
        default:;        /* Do nothing. */
    }
}


char           *
str_opt_val(enum opt_type otype)
{
    switch (otype) {
        case O_ALIAS:
            return (aliasname);
        case O_APPNAME:
            return (appname);
        case O_GSUBJECT:
            return (gsubject);
        case O_PR_NAME:
            return (printer_name);
        default:
            return(NULL);              /* Shouldn't happen. */
    }
}


static void
usage(void)               /* Print usage message and exit. */
{
    FPRINTF(stderr, "Usage:\n");
    usageoption(appname);
    usageoption("[-a4]");             /* A4 paper size. */
    usageoption("[-addhdr type]");    /* Add header to be printed. */
    usageoption("[-alias name]");     /* User name on output. */
    usageoption("[-allhdrs]");        /* Print out all headers. */
    usageoption("[-article]");        /* "Article from" format. */
    usageoption("[-bottom #]");       /* Bottom margin size. */
    usageoption("[-bsdprint]");       /* BSD printer system. */
    usageoption("[-chars #]");        /* Gecos (chars). */
    usageoption("[-columns #]");      /* Number of columns of output. */
    usageoption("[-config name]");    /* Alternate resource file. */
    usageoption("[-content]");        /* Use Content-Length: header. */
    usageoption("[-copies #]");       /* Number of copies to print. */
    usageoption("[-digest]");         /* Print digest. */
    usageoption("[-elm]");            /* ELM mail file format. */
    usageoption("[-flip]");           /* Flip banners. */
    usageoption("[-folder]");         /* Print mail folder. */
    usageoption("[-forcepage]");      /* ^L starts new physical page. */
    usageoption("[-from]");           /* Print originators name. */
    usageoption("[-help]");           /* Print usage message. */
    usageoption("[-landscape]");      /* Landscape printing. */
    usageoption("[-left #]");         /* Left margin size. */
    usageoption("[-linelength #]");   /* Number of chars per line output. */
    usageoption("[-modtime]");        /* Last modification time. */
    usageoption("[-name appname]");   /* Alternate application name. */
    usageoption("[-newpage]");        /* Start new page if duplex. */
    usageoption("[-nobanners]");      /* No header/footer page banners. */
    usageoption("[-noburstpage]");    /* No burst page on printer. */
    usageoption("[-number]");         /* Number lines of output. */
    usageoption("[-onesided]");       /* Simplex printing. */
    usageoption("[-organiser");       /* Organiser type. */
    usageoption("{filofax|franklin|proplan|timeman|tsintl|tspartner} ]");
    usageoption("[-pagelength #]");   /* Number of lines per page output. */
    usageoption("[-portrait]");       /* Portrait printing. */
    usageoption("[-postscript]");     /* Print PostScript files. */
    usageoption("[-printer]");        /* Send output to printer. */
    usageoption("[-prologue name]");  /* New prologue file. */
    usageoption("[-remhdr type]");    /* Remove header to print. */
    usageoption("[-right #]");        /* Right margin size. */
    usageoption("[-subject text]");   /* New subject line. */
    usageoption("[-tab #]");          /* # spaces in a tab. */
    usageoption("[-text]");           /* Print ordinary text file. */
    usageoption("[-tumble]");         /* Tumble duplex pages. */
    usageoption("[-twosided]");       /* Duplex printing. */
    usageoption("[-us]");             /* US paper size. */
    usageoption("[-version]");        /* Print version number. */
    usageoption("[-words #]");        /* Gecos (words). */
    usageoption("[-wrap]");           /* Wraps words of output. */
    usageoption("[filename ...]");
    FPRINTF(stderr, "\n\n");
#ifdef OLD_OPTIONS
    FPRINTF(stderr, "The following compatibility options ");
    FPRINTF(stderr, "are also supported:\nUsage:\n");
    cpos = 0;                /* Current column position for usage output */
    usageoption(appname);
    usageoption("[-A4]");             /* A4 paper size. */
    usageoption("[-F]");              /* Print originators name. */
    usageoption("[-PS]");             /* Print PostScript files. */
    usageoption("[-US]");             /* US paper size. */
    usageoption("[-a]");              /* "Article from" format. */
    usageoption("[-d]");              /* Print digest. */
    usageoption("[-e]");              /* ELM mail file format. */
    usageoption("[-f]");              /* Filofax output. */
    usageoption("[-l]");              /* Landscape printing. */
    usageoption("[-m]");              /* Print mail folder. */
    usageoption("[-o]");              /* Print ordinary text file. */
    usageoption("[-p prologue]");     /* New prologue file. */
    usageoption("[-s subject]");      /* New subject line. */
    usageoption("[-tm]");             /* Time Manager. */
    usageoption("[-ts]");             /* Time System (Intl) output. */
    usageoption("[-v]");              /* Print version number. */
    usageoption("[-?]");              /* Print usage message. */
    usageoption("[filename ...]");
    FPRINTF(stderr, "\n\n");
#endif /* OLD_OPTIONS */
    exit(1);
}


static void
usageoption(char *st)         /* Output usage option. */
{
    if ((int) strlen(st) + cpos > 78) {
        FPRINTF(stderr, "\n   ");
        cpos = 3;
    }
    FPRINTF(stderr, "%s ", st);
    cpos += strlen(st) + 1;
}


void
write_resources(void)
{
    char intval[MAXLINE], *value;

    load_mp_defs(configfile);

    if (article) {
        value = "news";
    } else if (print_ps) {
        value = "postscript";
    } else if (text_doc) {
        value = "text";
    } else {
        value = "mail";
    }
    put_resource(R_FTYPE, value);

    if (EQUAL(ptype, PRO_NORMAL)) {
        value = "portrait";
    } else if (EQUAL(ptype, PRO_LANDSCAPE)) {
        value = "landscape";
    } else {
        value = "organiser";
    }
    put_resource(R_OTYPE, value);

    value = "";
    if (EQUAL(ptype, PRO_FILOFAX)) {
        value = "filofax";
    } else if (EQUAL(ptype, PRO_FILOFAXP)) {
        value = "filofaxp";
    } else if (EQUAL(ptype, PRO_FRANKLIN)) {
        value = "franklin";
    } else if (EQUAL(ptype, PRO_PRO_PLAN)) {
        value = "proplan";
    } else if (EQUAL(ptype, PRO_TIME_MANAGER)) {
        value = "timeman";
    } else if (EQUAL(ptype, PRO_TS_INTL)) {
        value = "tsintl";
    } else if (EQUAL(ptype, PRO_TS_PARTNER)) {
        value = "tspartner";
    }
    put_resource(R_ORGANISER, value);

    if (paper_size == P_NONE) {
        value = "none";
    } else if (paper_size == P_A4) {
        value = "a4";
    } else if (paper_size == P_US) {
        value = "us";
    }
    put_resource(R_PSIZE, value);

    put_resource(R_HDRPAGE,  set_bool(burstpage));
    put_resource(R_BANNERS,  set_bool(banners));
    put_resource(R_MAIL_FDR, set_bool(folder));
    put_resource(R_MAIL_DIG, set_bool(digest));
    put_resource(R_DSIDED,   set_bool(duplex));
    put_resource(R_NEWPAGE,  set_bool(newpage));

    put_resource(R_SUBJECT, gsubject);
    put_resource(R_PNAME,   printer_name);

    SPRINTF(intval, "%d", llen);
    put_resource(R_LLENGTH, intval);
    SPRINTF(intval, "%d", plen);
    put_resource(R_PLENGTH, intval);
    SPRINTF(intval, "%d", copies);
    put_resource(R_COPIES, intval);

    put_hdr_resource(R_ADDHDRS);
    put_hdr_resource(R_REMHDRS);

    put_resource(R_ALIAS, aliasname);
    put_resource(R_PROLOG, proname);

    SPRINTF(intval, "%d", numcols);
    put_resource(R_NCOLUMNS, intval);
    SPRINTF(intval, "%d", tabsize);
    put_resource(R_TABSIZE, intval);
    SPRINTF(intval, "%d", namelength);
    put_resource(R_NAMEC, intval);
    SPRINTF(intval, "%d", namefields);
    put_resource(R_NAMEW, intval);

    put_resource(R_BSDPRINT,  set_bool(bsdprint));
    put_resource(R_WRAP,      set_bool(wrapwords));
    put_resource(R_FLIP,      set_bool(flip));
    put_resource(R_FORCEPAGE, set_bool(forcepage));
    put_resource(R_ELM,       set_bool(elm_if));
    put_resource(R_NUMBER,    set_bool(number));
    put_resource(R_FROM,      set_bool(print_orig));
    put_resource(R_CONTENT,   set_bool(content));
    put_resource(R_MOD,       set_bool(modtime));
    put_resource(R_TUMBLE,    set_bool(tumble));
    put_resource(R_ALLHDRS,   set_bool(allhdrs));

    SPRINTF(intval, "%d", bmargin);
    put_resource(R_BMARGIN, intval);
    SPRINTF(intval, "%d", lmargin);
    put_resource(R_LMARGIN, intval);
    SPRINTF(intval, "%d", rmargin);
    put_resource(R_RMARGIN, intval);
    SPRINTF(intval, "%d", tmargin);
    put_resource(R_TMARGIN, intval);

    if (save_resources()) {
        FPRINTF(stderr, "%s: Unable to save defaults.\n", appname);
    }
}
