/*

    mpv_win32.c

    Win32 Interface.

    mp - Programmer Text Editor

    Copyright (C) 1991-2001 Angel Ortega <angel@triptico.com>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

    http://www.triptico.com

*/

#include <stdio.h>
#include <string.h>
#include <windows.h>

#include "mp_core.h"
#include "mp_video.h"
#include "mp_iface.h"
#include "mp_res.h"


/*******************
	 Data
 *******************/

/* Win32 necessary globals */

/* the instance */
HINSTANCE hinst;

/* the windows */
HWND hwnd;
HWND hwtabs;

/* font handles */
HFONT _font_normal=NULL;
HFONT _font_italic=NULL;
HFONT _font_underline=NULL;

/* frame buffer with chars in document window */
static int _mpv_fb[1024][256];

/* cursor position */
static int _mpv_x_cursor;
static int _mpv_y_cursor;

/* temporal buffer */
static char _mpv_buffer[1024];

/* font size */
int _mpv_font_size=12;

/* font face */
char _mpv_font_face[80]="Lucida Console";

/* calculated font sizes in local units */
int _mpv_font_height=0;
int _mpv_font_width=0;

/* attributes */
static COLORREF _t_attrs[MP_COLOR_NUM+1];
static COLORREF _b_attrs[MP_COLOR_NUM+1];

/* win32.hlp path */
char _mpv_win32help[1024]="c:\\bin\\lcc\\bin\\win32.hlp";

/* dialog box vars */
static char * _mpv_dlg_prompt=NULL;
static char * _mpv_dlg_default=NULL;
static char * _mpv_list_title="";
static mp_txt * _mpv_list_text=NULL;

/* readline buffer */
static char _mpv_readline_buf[1024];

/* title and status line buffers */
static char _mpv_title_buffer[1024];
static char _mpv_status_line_buffer[1024];

/* menu and submenu handlers */
static HMENU _menu=NULL;
static HMENU _smenu=NULL;
static char _menu_label[40]="";

/* height of the tab of files */
int _tab_height=32;


/*******************
	 Code
 *******************/

/**
 * mpv_strcasecmp - Case ignoring string compare
 * @s1: first string
 * @s2: second string
 *
 * Case ignoring string compare. System dependent
 * (strcasecmp in Unix, stricmp in Win32)
 */
int mpv_strcasecmp(char * s1, char * s2)
{
	return(stricmp(s1,s2));
}


/**
 * mpv_goto - Positions the cursor to start drawing.
 * @x: the x position
 * @y: the y position
 *
 * Positions the cursor to start drawing the document window.
 */
void mpv_goto(int x, int y)
{
	/* just store the coords */
	_mpv_x_cursor=x;
	_mpv_y_cursor=y;
}


/**
 * mpv_char - Draws a char with color in the document window.
 * @c: the char
 * @color: the color (one of MP_COLOR_ constants)
 *
 * Draws a char with color in current cursor position of the
 * document window.
 */
void mpv_char(int c, int color)
{
	/* fill the frame buffer */
	_mpv_fb[_mpv_y_cursor][_mpv_x_cursor++]=(color<<8)|((unsigned char)c);
}


/**
 * mpv_str - Draws a string with color in the document window.
 * @str: the string
 * @color: the color (one of MP_COLOR_ constants)
 *
 * Draws a string, calling mpv_char() for each of their chars.
 */
void mpv_str(char * str, int color)
{
	while(*str)
	{
		_mpv_fb[_mpv_y_cursor][_mpv_x_cursor++]=(color<<8)|(*str);
		str++;
	}
}


/**
 * mpv_cursor - Positions the hardware cursor.
 * @x: the real x position
 * @y: the real y position
 *
 * Sets the hardware cursor to x, y position.
 */
void mpv_cursor(int x, int y)
{
	/* nothing; the Windows system caret could be used,
	   but I just hate it */
}


/**
 * mpv_refresh - Refresh the screen.
 *
 * Orders the underlying system to redraw the screen.
 */
void mpv_refresh(void)
{
	InvalidateRect(hwnd,NULL,TRUE);
}


/**
 * _mpv_title_status - Sets the window caption
 *
 * Sets the window caption with the title and the status bar.
 */
static void _mpv_title_status(void)
{
	char tmp[2048];

	strncpy(tmp,"mp " VERSION,sizeof(tmp));

	if(_mpv_title_buffer[0]!='\0')
	{
		strcat(tmp," - ");
		strcat(tmp,_mpv_title_buffer);
	}

	if(_mpv_status_line_buffer[0]!='\0')
		strcat(tmp,_mpv_status_line_buffer);

	SetWindowText(hwnd,tmp);
}


/**
 * mpv_title - Sets the string to be drawn as title
 * @str: the string
 *
 * Sets the string to be drawn as title of the document window.
 */
void mpv_title(char * str)
{
	if(str!=NULL)
		strncpy(_mpv_title_buffer,str,sizeof(_mpv_title_buffer));
	else
	if(_mp_active != NULL)
		strncpy(_mpv_title_buffer,_mp_active->name,sizeof(_mpv_title_buffer));
	else
		_mpv_title_buffer[0]='\0';

	_mpv_title_status();
}


/**
 * mpv_status_line - Sets the string to be drawn as status line
 * @str: the string
 *
 * Sets the string to be drawn as the status line.
 */
void mpv_status_line(char * str)
{
	if(str)
		strncpy(_mpv_status_line_buffer,str,sizeof(_mpv_status_line_buffer));
	else
		_mpv_status_line_buffer[0]='\0';

	_mpv_title_status();
}


/**
 * mpv_add_menu - Creates a new menu bar.
 * @label: the label
 *
 * Creates a new menu bar.
 */
void mpv_add_menu(char * label)
{
	if(_menu)
		AppendMenu(_menu,MF_STRING|MF_POPUP,
			(UINT)_smenu,_menu_label);
	else
		_menu=CreateMenu();

	strncpy(_menu_label,label,sizeof(_menu_label));
	_smenu=CreatePopupMenu();
}


/**
 * mpv_add_menu_item - Adds a menu item to current menu bar.
 * @label: the label
 * @key: the associated MPK_ key.
 * @toggle: a pointer to int with the toggle variable status.
 *
 * Adds a menu item to current menu bar. If toggle is defined,
 * it is a pointer to an integer to toggle if the item is
 * activated.
 */
void mpv_add_menu_item(char * label, int key, int * toggle)
{
	if(*label=='-')
		AppendMenu(_smenu,MF_SEPARATOR,0,NULL);
	else
		AppendMenu(_smenu,MF_STRING,1000+key,label);
}


/**
 * mpv_menu - Manages the menu.
 *
 * Manages the menu (drawing it, if applicable).
 * Returns the key associated to the activated menu item,
 * or 0 if none was. 
 */
int mpv_menu(void)
{
	/* dummy */
	return(0);
}


/**
 * mpv_alert - Alerts the user.
 * @msg: the alert message
 * @msg2: a possible second message.
 *
 * Alerts the user by showing the message and
 * asking for validation.
 */
void mpv_alert(char * msg, char * msg2)
{
	char tmp[4096];

	if(msg2==NULL)
		strncpy(tmp,msg,sizeof(tmp));
	else
		sprintf(tmp,msg,msg2);

	MessageBox(hwnd,tmp,"mp " VERSION,
		MB_ICONWARNING|MB_OK);
}


/**
 * mpv_confirm - Asks for confirmation.
 * @prompt: the question to ask
 *
 * Asks for confirmation.
 * Returns 1 if choosen 'yes'.
 */
int mpv_confirm(char * prompt)
{
	if(MessageBox(hwnd,prompt,"mp " VERSION,
		MB_ICONQUESTION|MB_YESNO)==IDYES)
		return(1);
	else
		return(0);
}


/**
 * TextDlgProc - Procedure for text input dialog
 * @hwnd: the window handler
 * @msg: the message sent
 * @wparam: the word param
 * @lparam: the long word param
 *
 * Procedure for text input dialog. Called from inside MS Windows.
 */
BOOL CALLBACK TextDlgProc(HWND hwnd, UINT msg, WPARAM wparam, LPARAM lparam)
{
	int ret;

	switch(msg)
	{
	case WM_INITDIALOG:

		SetWindowText(hwnd,"mp " VERSION);

		SetDlgItemText(hwnd,WMP_1STR_LABEL,
			_mpv_dlg_prompt);

		if(_mpv_dlg_default!=NULL)
			SetDlgItemText(hwnd,WMP_1STR_EDIT,
			_mpv_dlg_default);

		return(TRUE);

	case WM_COMMAND:

		switch(LOWORD(wparam))
		{
		case WMP_OK:
		case WMP_CANCEL:

			if(LOWORD(wparam)==WMP_OK)
			{
				ret=1;
				GetDlgItemText(hwnd,WMP_1STR_EDIT,
					_mpv_readline_buf,
					sizeof(_mpv_readline_buf));
			}
			else
				ret=0;

			EndDialog(hwnd,ret);

			return(TRUE);
		}
	}

	return(FALSE);
}


/**
 * ListDlgProc - Procedure for the list input dialog
 * @hwnd: the window handler
 * @msg: the message sent
 * @wparam: the word param
 * @lparam: the long word param
 *
 * Procedure for list input dialog. Called from inside MS Windows.
 */
BOOL CALLBACK ListDlgProc(HWND hwnd, UINT msg, WPARAM wparam, LPARAM lparam)
{
	int ret;
	HWND lst;
	char line[1024];

	switch(msg)
	{
	case WM_INITDIALOG:

		SetWindowText(hwnd, _mpv_list_title);

		lst=GetDlgItem(hwnd, WMP_LIST);
		SendMessage(lst, WM_SETFONT, 
			(WPARAM) GetStockObject(ANSI_FIXED_FONT), 0);

		/* traverses the list, filling the listbox */
		mp_move_bof(_mpv_list_text);

		while(mp_peek_char(_mpv_list_text)!='\0')
		{
			mp_get_str(_mpv_list_text,line,
				sizeof(line),'\n');

			SendMessage(lst, LB_ADDSTRING, 0,
				(LPARAM) line);
		}

		return(TRUE);

	case WM_COMMAND:

		switch(LOWORD(wparam))
		{
		case WMP_OK:
		case WMP_CANCEL:

			if(LOWORD(wparam)==WMP_OK)
				ret=SendDlgItemMessage(hwnd, WMP_LIST,
					LB_GETCURSEL, 0, 0);
			else
				ret=-1;

			EndDialog(hwnd,ret);

			return(TRUE);
		}
	}

	return(FALSE);
}


/**
 * AboutDlgProc - Procedure for the about box
 * @hwnd: the window handler
 * @msg: the message sent
 * @wparam: the word param
 * @lparam: the long word param
 *
 * Procedure for the about box. Called from inside MS Windows.
 */
BOOL CALLBACK AboutDlgProc(HWND hwnd, UINT msg, WPARAM wparam, LPARAM lparam)
{
	char tmp[4096];
	char * ptr;
	int n;

	switch(msg)
	{
	case WM_INITDIALOG:

		SetWindowText(hwnd, MSG_ABOUT);

		/* move the license text converting \n to \r\n */
		for(n=0,ptr=tmp;MP_LICENSE[n];n++)
		{
			if(MP_LICENSE[n]=='\n')
				*(ptr++)='\r';
			*(ptr++)=MP_LICENSE[n];
		}
		*ptr='\0';

		SetDlgItemText(hwnd,WMP_LICENSE,tmp);

		SendDlgItemMessage(hwnd,WMP_LICENSE,
			EM_SETSEL, -1, 0);
		SendDlgItemMessage(hwnd,WMP_LICENSE,
			EM_SETREADONLY, 1, 0);

		return(TRUE);

	case WM_COMMAND:

		SendDlgItemMessage(hwnd,WMP_LICENSE,
			EM_SETSEL, -1, 0);

		switch(LOWORD(wparam))
		{
		case WMP_OK:

			EndDialog(hwnd,0);

			return(TRUE);
		}
	}

	return(FALSE);
}


/**
 * mpv_list - Manages a selection list
 * @title: the title or caption of the list
 * @txt: the text containing the list to show
 *
 * Shows a unique element selection list to the user.
 * The list must be previously built into txt.
 * Returns the selected element (the number of the
 * line where the element is) or -1 on cancellation.
 */
int mpv_list(char * title, mp_txt * txt)
{
	_mpv_list_title=title;
	_mpv_list_text=txt;

	return(DialogBox(hinst,"DLGLIST",NULL,ListDlgProc));
}


/**
 * mpv_readline - Ask for user input.
 * @type: the type of input asked (one of MPR_ constants)
 * @prompt: the prompt
 * @def: the default value
 *
 * Asks for user input. The expected data type is
 * described in type.
 * Returns a pointer to a static buffer with the
 * data entered by the user, or NULL if user cancelled.
 */
char * mpv_readline(int type, char * prompt, char * def)
{
	OPENFILENAME ofn;

	if(def==NULL)
		_mpv_readline_buf[0]='\0';
	else
		strncpy(_mpv_readline_buf,def,sizeof(_mpv_readline_buf));

	if(type==MPR_OPEN || type==MPR_SAVE)
	{
		_mpv_readline_buf[0]='\0';

		memset(&ofn,'\0',sizeof(OPENFILENAME));
		ofn.lStructSize=sizeof(OPENFILENAME);
		ofn.hwndOwner=hwnd;
		ofn.lpstrFilter="*.*\0*.*\0";
		ofn.nFilterIndex=1;
		ofn.lpstrFile=_mpv_readline_buf;
		ofn.nMaxFile=sizeof(_mpv_readline_buf);
		ofn.lpstrTitle=prompt;
		ofn.lpstrDefExt=(def==NULL ? "" : def);

		if(type==MPR_OPEN)
		{
			ofn.Flags=OFN_PATHMUSTEXIST|OFN_HIDEREADONLY|
				OFN_NOCHANGEDIR|OFN_FILEMUSTEXIST;

			if(GetOpenFileName(&ofn))
				return(_mpv_readline_buf);
		}
		else
		{
			ofn.Flags=OFN_HIDEREADONLY;

			if(GetSaveFileName(&ofn))
				return(_mpv_readline_buf);
		}
	}
	else
	if(type!=MPR_EXEC)
	{
		_mpv_dlg_prompt=prompt;
		_mpv_dlg_default=def;

		if(DialogBox(hinst,"DLG1STRING",NULL,TextDlgProc))
			return(_mpv_readline_buf);
	}

	return(NULL);
}


/**
 * mpv_about - Shows the 'About Minimum Profit...' information.
 *
 * Shows a text or dialog box showing the information
 * about the program, version and such.
 */
void mpv_about(void)
{
	DialogBox(hinst,"ABOUTBOX",NULL,AboutDlgProc);
}


/**
 * mpv_help - Shows the available help for a term
 * @term: the term
 * @synhi: the syntax highlighter
 *
 * Shows the available help for the term. The argument
 * filetype is a string describing the file type,
 * taken from the syntax hilighter (this allows to
 * retrieve the help from different sources for C
 * or Perl code, for example).
 * Returns 0 on error or if no help is available
 * for this term.
 */
int mpv_help(char * term, int synhi)
{
	WinHelp(hwnd,_mpv_win32help,
		HELP_KEY,(DWORD) term);

	return(0);
}


/**
 * _mpv_init_fonts - Starts up font stuff.
 * @hdc: the device context
 *
 * Starts up font stuff.
 */
static void _mpv_init_fonts(HDC hdc)
{
	TEXTMETRIC tm;
	RECT rect;
	int n;
	static int f=1;

	/* create fonts */
	n=-MulDiv(_mpv_font_size,GetDeviceCaps(hdc, LOGPIXELSY),72);

	_font_normal=CreateFont(n,0,0,0,0,0,0,
		0,0,0,0,0,0,_mpv_font_face);

	_font_italic=CreateFont(n,0,0,0,0,1,0,
		0,0,0,0,0,0,_mpv_font_face);

	_font_underline=CreateFont(n,0,0,0,0,0,1,
		0,0,0,0,0,0,_mpv_font_face);

	SelectObject(hdc, _font_normal);

	GetTextMetrics(hdc, &tm);

	/* store sizes */
	_mpv_font_height=tm.tmHeight;
	_mpv_font_width=tm.tmAveCharWidth;

	GetClientRect(hwnd, &rect);

	/* calculate the size in chars */
	_mpv_x_size=((rect.right-rect.left)/_mpv_font_width)+1;
	_mpv_y_size=((rect.bottom-rect.top-_tab_height)/_mpv_font_height)+1;

	if(f)
	{
		f=0;
		GetWindowRect(hwnd,&rect);
		SetCursorPos(rect.left+200,rect.top+6);
	}
}


/**
 * mpv_zoom - Zooms the document window.
 * @inc: the increment (+1 or -1)
 *
 * Increases / decreases font size of the document window,
 * if applicable.
 */
int mpv_zoom(int inc)
{
	HDC hdc;

	hdc=GetDC(hwnd);

	SelectObject(hdc,GetStockObject(SYSTEM_FONT));

	DeleteObject(_font_normal);
	DeleteObject(_font_italic);
	DeleteObject(_font_underline);

	if(inc>0)
	{
		if(_mpv_font_size<32)
			_mpv_font_size+=2;
	}
	else
	{
		if(_mpv_font_size>6)
			_mpv_font_size-=2;
	}

	_mpv_init_fonts(hdc);

	mpi_draw_all(_mp_active);

	ReleaseDC(hwnd,hdc);

	return(0);
}


/**
 * _mpv_paint - Dump the frame buffer to screen.
 * @hwnd: the window handler
 *
 * Dumps the document window frame buffer to the window.
 */
static void _mpv_paint(HWND hwnd)
{
	HDC hdc;
	PAINTSTRUCT ps;
	RECT rect;
	RECT r2;
	int n,m,i;
	int c, color;

	GetClientRect(hwnd, &rect);
	r2=rect;

	hdc=BeginPaint(hwnd, &ps);

	if(_font_normal==NULL)
		_mpv_init_fonts(hdc);

	SelectObject(hdc, _font_normal);

	SetTextColor(hdc, _t_attrs[MP_COLOR_SELECTED]);
	SetBkColor(hdc, _b_attrs[MP_COLOR_SELECTED]);

	r2.top+=_tab_height;
	r2.bottom=r2.top + _mpv_font_height;

	for(n=0;n<_mpv_y_size;n++)
	{
		r2.left=r2.right=rect.left;

		for(m=0;m<_mpv_x_size;)
		{
			/* get first color */
			color=(_mpv_fb[n][m]&0xff00)>>8;

			/* writes into _mpv_buffer while
			   color is the same */
			for(i=0;m<_mpv_x_size &&
				color==((_mpv_fb[n][m]&0xff00)>>8);
				i++,m++)
			{
				c=_mpv_fb[n][m]&0xff;
				_mpv_buffer[i]=c;
				r2.right+=_mpv_font_width;
			}

			_mpv_buffer[i]='\0';

			SetTextColor(hdc,_t_attrs[color]);
			SetBkColor(hdc,_b_attrs[color]);

			SelectObject(hdc, color==MP_COLOR_COMMENT ?
				_font_italic :
				color==MP_COLOR_LOCAL ? _font_underline :
				_font_normal);

			DrawText(hdc,_mpv_buffer,-1,&r2,DT_SINGLELINE|DT_NOPREFIX);

			r2.left=r2.right;
		}

		r2.top+=_mpv_font_height;
		r2.bottom+=_mpv_font_height;
	}

	EndPaint(hwnd, &ps);
}


/**
 * _windows_to_clpbrd - Copies MS Windows clipboard to MP
 *
 * Copies MS Windows clipboard to MP's one
 */
static void _windows_to_clpbrd(void)
{
	HGLOBAL hclp;
	char * ptr;

	OpenClipboard(NULL);
	hclp=GetClipboardData(CF_TEXT);
	CloseClipboard();

	if(hclp)
	{
		if((ptr=(char *)GlobalLock(hclp))!=NULL)
		{
			/* destroy previous clipboard */
			if(_mp_clipboard!=NULL)
				mp_delete_sys_txt(_mp_clipboard);

			if((_mp_clipboard=mp_create_sys_txt(NULL))==NULL)
				return;

			/* transfer */
			while(*ptr)
			{
				if(*ptr!='\r')
					mp_insert_char(_mp_clipboard, *ptr);

				ptr++;
			}

			GlobalUnlock(hclp);
		}
	}
}


/**
 * _clpbrd_to_windows - Copies MP clipboard to MS Windows
 *
 * Copies MP clipboard to MS Windows's one
 */
static void _clpbrd_to_windows(void)
{
	int n,c;
	HGLOBAL hclp;
	char * ptr;

	if(_mp_clipboard==NULL) return;

	/* traverses clipboard counting chars */
	mp_move_bof(_mp_clipboard);
	for(n=0;(c=mp_get_char(_mp_clipboard))!='\0';n++)
	{
		if(c=='\n') n++;
	}

	mp_move_bof(_mp_clipboard);

	/* alloc and transfer */
	hclp=GlobalAlloc(GHND, n+1);
	ptr=(char *)GlobalLock(hclp);

	while(n > 0)
	{
		c=mp_get_char(_mp_clipboard);

		if(c=='\n')
		{
			*ptr++='\r';
			n--;
		}

		*ptr++=c;
		n--;
	}

	*ptr='\0';

	GlobalUnlock(hclp);

	OpenClipboard(NULL);
	EmptyClipboard();
	SetClipboardData(CF_TEXT, hclp);
	CloseClipboard();
}


/**
 * mpv_vkey - Converts Windows keys to MPK_ constants.
 * @c: the key
 *
 * Converts Windows virtual keys to MPK_ constants.
 */
static int mpv_vkey(int c)
{
	static int _maximized=0;
	int ctrl=0;

	if(GetKeyState(VK_CONTROL)&0x8000 ||
	   GetKeyState(VK_MENU)&0x8000)
		ctrl=1;

	if(_mpi_shift)
		_mpi_shift=0;

	switch(c)
	{
	case VK_UP:	c=MPK_UP; break;
	case VK_DOWN:	c=MPK_DOWN; break;
	case VK_LEFT:	c=ctrl? MPK_WORDLEFT : MPK_LEFT; break;
	case VK_RIGHT:	c=ctrl? MPK_WORDRIGHT : MPK_RIGHT; break;
	case VK_PRIOR:	c=ctrl? MPK_FNUP : MPK_PGUP; break;
	case VK_NEXT:	c=ctrl? MPK_FNDN : MPK_PGDN; break;
	case VK_HOME:	c=ctrl? MPK_BOF : MPK_BOL; break;
	case VK_END:	c=ctrl? MPK_EOF : MPK_EOL; break;

	case VK_BACK:	c=MPK_BACKSPC; break;
	case VK_DELETE: c=MPK_DELCHAR; break;
	case VK_INSERT: c=MPK_INSMODE; break;

	case VK_F1:	c=MPK_HELP; break;
	case VK_F2:	c=MPK_SAVE; break;
	case VK_F3:	c=MPK_OPEN; break;
	case VK_F4:	c=MPK_CLOSE; break;
	case VK_F5:	c=MPK_NEW; break;
	case VK_F6:	c=MPK_NEXT; break;
	case VK_F7:	c=MPK_PLAYMACRO; break;
	case VK_F8:	c=MPK_UNMARK; break;
	case VK_F9:	c=MPK_MARK; break;
	case VK_F10:	c=MPK_RECORDMACRO; break;

	case VK_F11:	c=MPK_UNZOOM; break;
	case VK_F12:

		if(ctrl)
		{
			c=0;

			SendMessage(hwnd,
			WM_SYSCOMMAND,
			_maximized ? SC_RESTORE : SC_MAXIMIZE,
			0);

			_maximized^=1;
		}
		else
			c=MPK_ZOOM;
		break;

	default:	c=0; break;
	}

	return(c);
}


#define _ctrl(c) ((c)&63)

/**
 * mpv_akey - Converts ascii keys to MPK_ constants.
 * @k: the key
 *
 * Converts ascii keys to MPK_ constants.
 */
static int mpv_akey(int k)
{
	if(_mpi_shift)
		_mpi_shift=0;

	switch(k)
	{
	case _ctrl('W'): k=MPK_BOF; break;
	case _ctrl('E'): k=MPK_EOF; break;
	case _ctrl('J'): k=MPK_WORDLEFT; break;
	case _ctrl('K'): k=MPK_WORDRIGHT; break;

	case _ctrl('B'): k=MPK_SEEK; break;
	case _ctrl('L'): k=MPK_SEEKNEXT; break;
	case _ctrl('R'): k=MPK_REPLACE; break;
	case _ctrl('G'): k=MPK_GOTO; break;

	case _ctrl('Y'): k=MPK_DELLINE; break;

	case _ctrl('D'):
	case _ctrl('C'): k=MPK_COPY; break;

	case _ctrl('Q'):
	case _ctrl('T'): k=MPK_CUT; break;
	case _ctrl('P'):
	case _ctrl('V'): k=MPK_PASTE; break;

	case _ctrl('O'): k=MPK_OPEN2; break;
	case _ctrl('U'): k=MPK_TEMPLATE; break;
	case _ctrl('F'): k=MPK_TAG; break;

	case _ctrl('N'): k=MPK_SHIFT; break;

	case _ctrl('X'): k=MPK_EXIT; break;
	}

	return(k);
}


/**
 * _mpv_action - Process the action.
 * @c: the key to act upon
 *
 * Calls mpi_action() and takes into account another
 * Windows-dependent actions.
 */
static void _mpv_action(int c)
{
	/* pre actions */
	if(c==MPK_PASTE)
		_windows_to_clpbrd();

	if(mpi_action(_mp_active,c))
		mpi_draw_all(_mp_active);

	/* post actions */
	if(c==MPK_EXIT)
		SendMessage(hwnd,WM_CLOSE,0,0);
	else
	if(c==MPK_COPY || c==MPK_CUT)
		_clpbrd_to_windows();
}


#define CHECKMENUITEM(k,v) CheckMenuItem(_menu, k + 1000, \
	v ? MF_CHECKED : MF_UNCHECKED);

/**
 * _check_menus - (Un)Checks menu item marks
 *
 * Updates the check/uncheck status of toggleable
 * menu items.
 */
static void _check_menus(void)
{
	CHECKMENUITEM(MPK_SAVETABS, _mp_save_tabs);
	CHECKMENUITEM(MPK_CRLF, _mp_cr_lf);
	CHECKMENUITEM(MPK_AUTOINDENT, _mp_auto_indent);
	CHECKMENUITEM(MPK_COL80, _mpi_mark_column_80);
	CHECKMENUITEM(MPK_TOGGLECASE, _mp_case_cmp);
}


#ifndef STDCALL
#define STDCALL __stdcall
#endif

/**
 * WndProc - Main window callback.
 * @hwnd: the window handler
 * @msg: the message
 * @wparam: the word param
 * @lparam: the long word param
 *
 * Main window callback.
 */
long STDCALL WndProc(HWND hwnd, UINT msg, UINT wparam, LONG lparam)
{
	int c;
	int x,y;
	LPNMHDR p;
	mp_txt * t;

	switch(msg)
	{
	case WM_KEYDOWN:

		if(c=mpv_vkey(wparam))
			_mpv_action(c);

		return(0);

	case WM_CHAR:

		c=mpv_akey(wparam);
		_mpv_action(c);

		return(0);

	case WM_VSCROLL:

		switch(LOWORD(wparam))
		{
		case SB_PAGEUP: c=MPK_PGUP; break;
		case SB_PAGEDOWN: c=MPK_PGDN; break;
		case SB_LINEUP: c=MPK_UP; break;
		case SB_LINEDOWN: c=MPK_DOWN; break;
		default: c=0; break;
		}

		if(c) _mpv_action(c);

		return(0);

	case WM_PAINT:
		_mpv_paint(hwnd);
		return(0);

	case WM_SIZE:

		if(_mpv_font_width && _mpv_font_height)
		{
			_mpv_x_size=(LOWORD(lparam)/_mpv_font_width)+1;
			_mpv_y_size=((HIWORD(lparam)-_tab_height)/_mpv_font_height)+1;

			mpi_draw_all(_mp_active);
		}

		MoveWindow(hwtabs,0,0,LOWORD(lparam),_tab_height,TRUE);

		return(0);

	case WM_LBUTTONDOWN:

		x=LOWORD(lparam);
		y=HIWORD(lparam) - _tab_height;

		x/=_mpv_font_width;
		y/=_mpv_font_height;

		mp_move_xy(_mp_active,x,y+_mp_active->vy);
		mpi_draw_all(_mp_active);

		return(0);

	case WM_COMMAND:

		c=0;

		c=LOWORD(wparam)-1000;

		if(c) _mpv_action(c);

		_check_menus();

		return(0);

	case WM_CLOSE:
		while(_mp_active)
			mpi_close(_mp_active);

		DestroyWindow(hwnd);
		return(0);

	case WM_DESTROY:
		PostQuitMessage(0);
		return(0);

	case WM_NOTIFY:
		p=(LPNMHDR)lparam;

		if(p->code==TCN_SELCHANGE)
		{
			y=TabCtrl_GetCurSel(hwtabs);

			for(t=_mp_txts,x=0;t!=NULL;t=t->next,x++)
			{
				if(x==y)
				{
					_mp_active=t;
					break;
				}
			}

			mpi_draw_all(_mp_active);
		}

		return(0);
	}

	return(DefWindowProc(hwnd,msg,wparam,lparam));
}


/**
 * mpv_scrollbar - Draws / updates the scrollbar.
 * @pos: current position
 * @size: vertical size
 * @max: maximum value
 *
 * Draws / updates the scrollbar. pos is current position
 * (the line where the cursor is), size is the number of
 * lines of the document window and max is the maximum
 * value (the total number of lines in current text).
 */
void mpv_scrollbar(int pos, int size, int max)
{
	SCROLLINFO si;

	si.cbSize=sizeof(si);
	si.fMask=SIF_ALL;
	si.nMin=1;
	si.nMax=max;
	si.nPage=size;
	si.nPos=pos;

	SetScrollInfo(hwnd, SB_VERT, &si, TRUE);
}


/**
 * mpv_filetabs - Draws the tab set containing the file names
 *
 * Draws the names of the opened files in the form of a tab set.
 */
void mpv_filetabs(void)
{
	TC_ITEM ti;
	mp_txt * t;
	int n;
	char * ptr;

	TabCtrl_DeleteAllItems(hwtabs);

	for(t=_mp_txts,n=0;t!=NULL;t=t->next,n++)
	{
		ti.mask=TCIF_TEXT;

		if((ptr=strrchr(t->name,'\\'))==NULL)
			ptr=t->name;
		else
			ptr++;

		ti.pszText=ptr;

		TabCtrl_InsertItem(hwtabs,n,&ti);

		if(_mp_active==t)
			TabCtrl_SetCurSel(hwtabs,n);
	}
}


/**
 * mpv_startup - Starts up the system dependent driver.
 *
 * Starts up the system dependent driver.
 * Returns 1 if succesful.
 */
int mpv_startup(void)
{
	WNDCLASS wc;
	unsigned long int n;
	HKEY hkey;
	RECT r;

	/* fill a usable default template file */
	strcpy(_mpi_template_file,"C:\\mp_templates");

	/* reads the possible default values in the registry */
	if(RegOpenKeyEx(HKEY_CURRENT_USER,
		"Software\\Minimum Profit",0,
		KEY_QUERY_VALUE, &hkey)==ERROR_SUCCESS)
	{
		char tmp[16];

		n=sizeof(_mpi_template_file);
		RegQueryValueEx(hkey,"TemplateFile",NULL,
		NULL,_mpi_template_file,&n);

		n=sizeof(_mpv_font_face);
		RegQueryValueEx(hkey,"FontFace",NULL,
		NULL,_mpv_font_face,&n);

		n=sizeof(_mpv_win32help);
		RegQueryValueEx(hkey,"Win32HelpPath",NULL,
		NULL,_mpv_win32help,&n);

		n=sizeof(tmp);
		RegQueryValueEx(hkey,"FontSize",NULL,
		NULL,tmp,&n);
		_mpv_font_size=atoi(tmp);

		RegCloseKey(hkey);
	}

	/* register the window */
	wc.style=CS_HREDRAW|CS_VREDRAW;
	wc.lpfnWndProc=WndProc;
	wc.cbClsExtra=0;
	wc.cbWndExtra=0;
	wc.hInstance=hinst;
	wc.hIcon=LoadIcon(hinst,"MP_ICON");
	wc.hCursor=LoadCursor(NULL,IDC_ARROW);
	wc.hbrBackground=NULL;
	wc.lpszMenuName="WMP_MENU";
	wc.lpszClassName="minimumprofit3.x";

	if(!RegisterClass(&wc))
		return(0);

	mpv_add_menu("");

	/* create the window */
	hwnd=CreateWindow("minimumprofit3.x","mp " VERSION,
		WS_OVERLAPPEDWINDOW | WS_CLIPCHILDREN | WS_VSCROLL,
		CW_USEDEFAULT,CW_USEDEFAULT,
		CW_USEDEFAULT,CW_USEDEFAULT,
		NULL,_menu,hinst,NULL);

	if(hwnd==NULL)
		return(0);

	DrawMenuBar(hwnd);

	ShowWindow(hwnd,SW_SHOW);
	UpdateWindow(hwnd);

	GetClientRect(hwnd,&r);

	hwtabs=CreateWindow(WC_TABCONTROL, "tab",
		WS_CHILD | TCS_TABS | TCS_SINGLELINE | TCS_FOCUSNEVER,
		0, 0, r.right-r.left, _tab_height, hwnd, NULL, hinst, NULL);

	ShowWindow(hwtabs,SW_SHOW);
	UpdateWindow(hwtabs);

	/* colors */
	_t_attrs[MP_COLOR_NORMAL]=0x00000000;
	_b_attrs[MP_COLOR_NORMAL]=0x00ffffff;
	_t_attrs[MP_COLOR_SELECTED]=0x00ffffff;
	_b_attrs[MP_COLOR_SELECTED]=0x000000ff;
	_t_attrs[MP_COLOR_COMMENT]=0x00aaaa00;
	_b_attrs[MP_COLOR_COMMENT]=0x00ffffff;
	_t_attrs[MP_COLOR_STRING]=0x00ff0000;
	_b_attrs[MP_COLOR_STRING]=0x00ffffff;
	_t_attrs[MP_COLOR_TOKEN]=0x0000aa00;
	_b_attrs[MP_COLOR_TOKEN]=0x00ffffff;
	_t_attrs[MP_COLOR_VAR]=0x006666ff;
	_b_attrs[MP_COLOR_VAR]=0x00ffffff;
	_t_attrs[MP_COLOR_CURSOR]=0x00ffffff;
	_b_attrs[MP_COLOR_CURSOR]=0x00000000;
	_t_attrs[MP_COLOR_CAPS]=0x0000dddd;
	_b_attrs[MP_COLOR_CAPS]=0x00ffffff;
	_t_attrs[MP_COLOR_LOCAL]=0x00ff8888;
	_b_attrs[MP_COLOR_LOCAL]=0x00ffffff;

	return(1);
}


/**
 * mpv_mainloop - Main message processing loop.
 *
 * Main message processing loop.
 */
void mpv_mainloop(void)
{
	MSG msg;

	_check_menus();

	while(GetMessage(&msg, NULL, 0, 0))
	{
		TranslateMessage(&msg);
		DispatchMessage(&msg);
	}
}


/**
 * mpv_shutdown - Shuts down the system dependent driver.
 *
 * Shuts down the system dependent driver.
 */
void mpv_shutdown(void)
{
	SendMessage(hwnd,WM_CLOSE,0,0);
}


int STDCALL WinMain(HINSTANCE hi, HINSTANCE hpi, LPSTR cmd, int cm)
{
	char * f;
	hinst=hi;

	mp_startup();
	mps_startup();
	mpi_startup();
	mpv_startup();

	/* load everything sent to command line, splitting it */
	for(f=strtok(cmd," ");f!=NULL;f=strtok(NULL," "))
		mpi_open(f,0);

	/* force the reading of the tags file, if exists */
	mpi_find_tag(NULL);

	if(_mp_active==NULL)
		mp_create_txt(MSG_UNNAMED);

	mpv_title(_mp_active->name);
	mpi_draw_all(_mp_active);

	mpv_mainloop();

	mpv_shutdown();
	mp_shutdown();

	return(0);
}

