/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2001 CodeFactory AB
 * Copyright (C) 2001 Anders Carlsson <andersca@codefactory.se>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Anders Carlsson
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <unistd.h>
#include <gal/e-table/e-table-scrolled.h>
#include <gal/e-table/e-table-simple.h>
#include <gal/widgets/e-unicode.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>
#include "util/type-utils.h"
#include "template-dialog.h"


#define TABLE_SPEC "\
<ETableSpecification cursor-mode=\"line\" no-headers=\"true\"> \
<ETableColumn model_col=\"0\" _title=\"Type\" minimum_width=\"18\" expansion=\"0.0\" resizable=\"false\" cell=\"pixbuf\" compare=\"string\"/> \
<ETableColumn model_col=\"1\" _title=\"Name\" expansion=\"1.0\" resizable=\"true\" cell=\"string\" compare=\"string\"/> \
</ETableSpecification>"

#define TABLE_STATE "\
<ETableState> \
  <column source=\"0\"/> \
  <column source=\"1\"/> \
  <grouping> \
  </grouping> \
</ETableState> "

#if 0
/* XPM */
static const char * blank_xpm[] = {
"15 16 6 1",
" 	c None",
".	c #000000",
"+	c #FFFFFF",
"@	c #F6F6F6",
"#	c #B2B2B2",
"$	c #EEEEEE",
" ...........   ",
" .++++++++.+.  ",
" .+@@@@@@@.@+. ",
" .+@@@@@@@.... ",
" .+@@@@@@@@@#. ",
" .+@@@@@@@@@#. ",
" .+@@@@@@@@@#. ",
" .+@@@@@@@@@#. ",
" .+@@@@@@@@@#. ",
" .+@@@@@@@@@#. ",
" .+@@@@@@@@@#. ",
" .+@@@@@@@@@#. ",
" .+@@@@@@@@@#. ",
" .+@@@@@@@@@#. ",
" .$##########. ",
" ............. "};

/* XPM */
static const char * template_xpm[] = {
"15 16 7 1",
" 	c None",
".	c #000000",
"+	c #FFFFFF",
"@	c #989898",
"#	c #F6F6F6",
"$	c #B2B2B2",
"%	c #EEEEEE",
" ...........   ",
" .++++++++.+.  ",
" .+@@@@@@@.#+. ",
" .+#######.... ",
" .+@@@@@@@@@$. ",
" .+#########$. ",
" .+@@@@@@@@@$. ",
" .+#########$. ",
" .+@@@@@@@@@$. ",
" .+#########$. ",
" .+@@@@@@@@@$. ",
" .+#########$. ",
" .+@@@@@@@@@$. ",
" .+#########$. ",
" .%$$$$$$$$$$. ",
" ............. "};
#endif

static void     template_dialog_init        (GtkObject      *object);
static void     template_dialog_class_init  (GtkObjectClass *object_class);

enum {
	COL_PIXBUF,
	COL_NAME,
	LAST_COL
};

GNOME_CLASS_BOILERPLATE (TemplateDialog, template_dialog, 
			 GnomeDialog, gnome_dialog);


struct _TemplateDialogPriv {
	GPtrArray *file_names;
	gint num_files;
	
	gchar *path;

	ETable *table;
};

#if 0
static gint
template_table_col_count (ETableModel *model, void *data)
{
	return 2;
}

static gint
template_table_row_count (ETableModel *model, void *data)
{
	return (TEMPLATE_DIALOG (data)->priv->num_files + 1);
}

static void *
template_table_value_at (ETableModel *etc, int col, int row, void *data)
{
	static GdkPixbuf *blank_pixbuf = NULL;
	static GdkPixbuf *template_pixbuf = NULL;
		
	if (blank_pixbuf == NULL) {
		blank_pixbuf = gdk_pixbuf_new_from_xpm_data (blank_xpm);
		template_pixbuf = gdk_pixbuf_new_from_xpm_data (template_xpm);
	}

	switch (col) {
	case COL_PIXBUF:
		if (row == 0) {
			return blank_pixbuf;
		}
		else {
			return template_pixbuf;
		}
		break;
	case COL_NAME:
		if (row == 0) {
			return _("(Blank Project)");
		}
		
		if (row - 1 >= TEMPLATE_DIALOG (data)->priv->num_files)
			return NULL;

		
		return g_ptr_array_index (TEMPLATE_DIALOG (data)->priv->file_names, row - 1);

		break;
	default:
	}

	return NULL;
}
#endif

static void
template_dialog_destroy (GtkObject *object)
{
	TemplateDialog *dialog = TEMPLATE_DIALOG(object);

	if (dialog->priv->file_names)
		g_ptr_array_free (dialog->priv->file_names, TRUE);

	if (dialog->priv->path)
		g_free (dialog->priv->path);
	
	g_free (dialog->priv);
}

static void
template_dialog_construct (TemplateDialog *dialog)
{
#if 0
	GnomeDialog *gnome_dialog;
	GtkWidget *table, *label;
	ETableModel *model;
	
	gnome_dialog = GNOME_DIALOG (dialog);

	gnome_dialog_append_button (gnome_dialog, GNOME_STOCK_BUTTON_OK);
	gnome_dialog_append_button (gnome_dialog, GNOME_STOCK_BUTTON_CANCEL);

	model = e_table_simple_new (template_table_col_count,
				    template_table_row_count,
				    template_table_value_at,
				    NULL, NULL, 
				    NULL, NULL,
				    NULL, NULL, NULL, dialog);

	table = e_table_scrolled_new (model, NULL, TABLE_SPEC, TABLE_STATE);
	e_table_set_cursor_row (E_TABLE_SCROLLED (table)->table, 0);

	label = gtk_label_new (_("Choose a template:"));
	gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
	
	gtk_box_pack_start (GTK_BOX (gnome_dialog->vbox), label, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (gnome_dialog->vbox), table, TRUE, TRUE, 0);

	dialog->priv->table = E_TABLE_SCROLLED (table)->table;
#endif
}

static void
template_dialog_init (GtkObject *object)
{
	TemplateDialog *dialog = TEMPLATE_DIALOG (object);
	
	dialog->priv = g_new0 (TemplateDialogPriv, 1);


}

static void
template_dialog_class_init (GtkObjectClass *object_class)
{
	object_class->destroy = template_dialog_destroy;
}

gint
template_dialog_run (TemplateDialog *dialog, gchar **filename)
{
	gint status;
	gint cursor_row;

	g_return_val_if_fail (filename != NULL, -1);

	gtk_widget_show_all (GTK_WIDGET (dialog));
	status = gnome_dialog_run_and_close (GNOME_DIALOG (dialog));

	cursor_row = e_table_get_cursor_row (dialog->priv->table);

	if (cursor_row == 0) {
		*filename = NULL;
	}
	else {
		gchar *complete_name, *tmp_str;
		
		tmp_str = e_utf8_to_locale_string (g_ptr_array_index (dialog->priv->file_names, cursor_row - 1));

		complete_name = g_strdup_printf ("%s/%s.mrproject", dialog->priv->path, tmp_str);

		*filename = complete_name;
		g_free (tmp_str);
	}
	
	
	return status;
}

static DIR *
open_directory (TemplateDialog *dialog)
{
	GList *lang_list;
	gchar *dir_name;
	DIR *opened_dir;

	lang_list = gnome_i18n_get_language_list ("LC_MESSAGE");
	
	while (lang_list) {
		dir_name = g_strdup_printf ("%stemplates/%s", MRPROJECT_DATADIR, (gchar *)lang_list->data);

		opened_dir = opendir (dir_name);

		if (opened_dir) {
			dialog->priv->path = dir_name;
			return opened_dir;
		}

		g_free (dir_name);
		lang_list = lang_list->next;
	}

	/* Try to open the C directory */
	dir_name = g_strdup_printf ("%s/templates/C", MRPROJECT_DATADIR);
	
	opened_dir = opendir (dir_name);
	
	if (opened_dir) {
		dialog->priv->path = dir_name;
		return opened_dir;
	}

	g_free (dir_name);
	return NULL;
}

static void
add_files_to_array (TemplateDialog *dialog)
{
	GPtrArray *array;
	DIR *dir;
	struct dirent *dirent;
	gint num_files;
	
	dir = open_directory (dialog);

	if (dir == NULL) {
		return;
	}

	array = g_ptr_array_new ();
	num_files = 0;
	
	while (1) {
		gchar *tmp_str;
		
		dirent = readdir (dir);

		if (dirent == NULL)
			break;

		/* Only add files with the suffix .mrproject */
		tmp_str = strstr (dirent->d_name, ".mrproject");
		
		if (tmp_str != NULL &&
		    strcmp (tmp_str, ".mrproject") == 0) {
			g_ptr_array_add (array, e_utf8_from_locale_string_sized (dirent->d_name, (tmp_str - dirent->d_name)));
			num_files++;
		}
	}

	closedir (dir);

	dialog->priv->num_files = num_files;
	dialog->priv->file_names = array;
}

GtkWidget *
template_dialog_new (void)
{
	TemplateDialog                     *dialog;

	dialog = gtk_type_new (TEMPLATE_TYPE_DIALOG);
	gtk_window_set_title (GTK_WINDOW (dialog), _("New Project"));
	gtk_window_set_policy (GTK_WINDOW (dialog), FALSE, TRUE, FALSE);
	gtk_window_set_default_size (GTK_WINDOW (dialog), 300, 300);

	gtk_window_set_wmclass (GTK_WINDOW (dialog),
				"MrProject",
				"TemplateDialog");
	
	/* FIXME: check that the dialog really gets freed somewhere. */
	gnome_dialog_close_hides (GNOME_DIALOG (dialog), TRUE);
	
	add_files_to_array (dialog);
	template_dialog_construct (dialog);
	
	return GTK_WIDGET (dialog);
}
