/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2001 CodeFactory AB
 * Copyright (C) 2001 Mikael Hallendal <micke@codefactory.se>
 * Copyright (C) 2001 Richard Hult <rhult@codefactory.se>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Mikael Hallendal <micke@codefactory.se>
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <bonobo.h>
#include <gtk/gtkobject.h>
#include "resource-manager.h"
#include "util/corba-utils.h"
#include "util/id-map.h"
#include "util/type-utils.h"

/* GtkObject. */
static void resource_manager_init        (ResourceManager        *manager);
static void resource_manager_class_init  (ResourceManagerClass   *klass);
static void rm_destroy                   (GtkObject              *object);
static void rm_get_arg                   (GtkObject              *object, 
					  GtkArg                 *arg, 
					  guint                   arg_id);
static void rm_set_arg                   (GtkObject              *object, 
					  GtkArg                 *arg, 
					  guint                   arg_id);

#define PARENT_TYPE BONOBO_X_OBJECT_TYPE
static GtkObjectClass  *parent_class;

struct _ResourceManagerPriv {
	BonoboEventSource              *event_source;
	GNOME_MrProject_ProjectState    state;
};

/* Object arguments */
enum {
	ARG_0,
	ARG_STATE
};

/* Resource Management */

static void
impl_ResourceManager__set_state (PortableServer_Servant  servant,
				 const GM_ProjectState   state,
				 CORBA_Environment      *ev)
{
	ResourceManager          *manager;

	manager = RESOURCE_MANAGER (bonobo_x_object (servant));
	manager->priv->state = state;
}

static GM_ProjectState
impl_ResourceManager__get_state (PortableServer_Servant  servant,
				 CORBA_Environment      *ev)
{
	ResourceManager          *manager;

	manager = RESOURCE_MANAGER (bonobo_x_object (servant));
	return manager->priv->state;
}

static GNOME_MrProject_Resource *
impl_ResourceManager_createResource (
	PortableServer_Servant              servant,
	CORBA_Environment                  *ev)
{
	ResourceManager                 *rm;
	ResourceManagerClass            *klass;
	ResourceManagerPriv             *priv;
	GNOME_MrProject_Resource        *resource;
	GNOME_MrProject_ResourceGroup   *group;
	GNOME_MrProject_Id               default_group;
	
	rm    = RESOURCE_MANAGER (bonobo_x_object (servant));
 	klass = RESOURCE_MANAGER_CLASS (GTK_OBJECT (rm)->klass);
	priv  = rm->priv;
	
	/* Create a new resource */
	resource              = GNOME_MrProject_Resource__alloc ();
	resource->resourceId  = -1;
	resource->name        = CORBA_string_dup ("");
	resource->type        = GNOME_MrProject_RESOURCE_TYPE_PEOPLE;
	resource->units       = 100; /* percent */
	resource->email       = CORBA_string_dup ("");
	resource->stdRate     = 100.0;
	resource->ovtRate     = 200.0;
	
 	default_group = klass->get_default_group (rm); 

	/* Check to see if a defeault group is choosen */
 	if (default_group >= 0) { 
		group = klass->get_group (rm, default_group);

		if (group) {
			resource->groupId = default_group;
		} else {
			resource->groupId = -1;
		}
		
	} else {
		resource->groupId = -1;
	}

	return resource;
}

static GNOME_MrProject_Id
impl_ResourceManager_insertResource (
	PortableServer_Servant              servant,
	const GNOME_MrProject_Resource     *resource,
	CORBA_Environment                  *ev)
{
	ResourceManager            *rm;
	ResourceManagerClass       *klass;
	ResourceManagerPriv        *priv;
	CORBA_any                  *any;
	GNOME_MrProject_Resource   *copy;
	CORBA_Environment           e;
	gint                        id;
	
	rm    = RESOURCE_MANAGER (bonobo_x_object (servant));
	klass = RESOURCE_MANAGER_CLASS (GTK_OBJECT (rm)->klass);
	priv  = rm->priv;
	id    = klass->insert_resource (rm, resource);

	/* Create an object to send with the inserted signal */
	copy         = corba_util_resource_duplicate (resource);
	any          = CORBA_any__alloc ();
	any->_type   = TC_GNOME_MrProject_Resource;
	any->_value  = (GNOME_MrProject_Resource *) copy;

	/* Makes CORBA_free (any) free any->_value */
	CORBA_any_set_release (any, CORBA_TRUE);
	
	copy->resourceId = id;

	CORBA_exception_init (&e);
	
	if (priv->event_source) {
		bonobo_event_source_notify_listeners (
			priv->event_source,
			"GNOME/MrProject:resource:inserted",
			any,
			&e);
	} else {
		g_warning ("Could not get EventSource interface.");
	}
	
	CORBA_exception_free (&e);
	CORBA_free (any);

	priv->state = GNOME_MrProject_PROJECT_STATE_DIRTY;
	
	return id;
}

static GNOME_MrProject_Resource *
impl_ResourceManager_getResource (
	PortableServer_Servant              servant,
	const GNOME_MrProject_Id            resourceId,
	CORBA_Environment                  *ev)
{
	ResourceManager            *rm;
	ResourceManagerClass       *klass;
	ResourceManagerPriv        *priv;
	GNOME_MrProject_Resource   *resource;
	GNOME_MrProject_Resource   *copy;
	
	rm    = RESOURCE_MANAGER (bonobo_x_object (servant));
	priv  = rm->priv;
	klass = RESOURCE_MANAGER_CLASS (GTK_OBJECT (rm)->klass);
	
	resource = klass->get_resource (rm, resourceId);

	if (!resource) {
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
				     ex_GNOME_MrProject_ResourceManager_NoSuchResource,
				     NULL);
		return NULL;
	}

	copy = corba_util_resource_duplicate (resource);
	
	return copy;
}

static GNOME_MrProject_ResourceSeq *
impl_ResourceManager_getAllResources (
	PortableServer_Servant              servant,
	CORBA_Environment                  *ev)
{
	ResourceManager               *rm;
	ResourceManagerClass          *klass;
	ResourceManagerPriv           *priv;
	GNOME_MrProject_ResourceSeq   *resources;
	GSList                        *list;
 	
	rm    = RESOURCE_MANAGER (bonobo_x_object (servant));
	klass = RESOURCE_MANAGER_CLASS (GTK_OBJECT (rm)->klass);
	priv  = rm->priv;
	list  = klass->get_all_resources (rm);

	resources = corba_util_resource_seq_from_list (list);

	g_slist_free (list);
	
	return resources;
}

static void
impl_ResourceManager_updateResource (
	PortableServer_Servant              servant,
	const GNOME_MrProject_Resource     *resource,
	CORBA_Environment                  *ev)
{
	ResourceManager            *rm;
	ResourceManagerClass       *klass;
	ResourceManagerPriv        *priv;
	CORBA_any                  *any;
	CORBA_Environment           e;
	GNOME_MrProject_Resource   *updated_resource;
	GNOME_MrProject_Resource   *copy;
	
	g_return_if_fail (resource != NULL);

	rm    = RESOURCE_MANAGER (bonobo_x_object (servant));
	klass = RESOURCE_MANAGER_CLASS (GTK_OBJECT (rm)->klass);
	priv  = rm->priv;

	if (klass->update_resource (rm, resource)) {

		updated_resource = klass->get_resource (rm, 
							resource->resourceId);

		copy        = corba_util_resource_duplicate (updated_resource);
		any         = CORBA_any__alloc ();
		any->_type  = TC_GNOME_MrProject_Resource;
		any->_value = copy;
	
		CORBA_any_set_release (any, CORBA_TRUE);
	
		CORBA_exception_init (&e);
	
		if (priv->event_source) {
			bonobo_event_source_notify_listeners (
				priv->event_source,
				"GNOME/MrProject:resource:updated",
				any,
				&e);
		} else {
			g_warning ("Could not get EventSource interface.");
		}
		
		priv->state = GNOME_MrProject_PROJECT_STATE_DIRTY;
		
		CORBA_exception_free (&e);
		CORBA_free (any);
	}
}

static void
impl_ResourceManager_removeResources (
	PortableServer_Servant    servant,
	const GM_IdSeq           *ids,
	CORBA_Environment        *ev)
{
	ResourceManager      *rm;
	ResourceManagerClass *klass;
	ResourceManagerPriv  *priv;
	BonoboArg            *arg;
	gchar                *id_str;
	CORBA_Environment     e;
	GSList               *res_list = NULL;
	
	rm    = RESOURCE_MANAGER (bonobo_x_object (servant));
 	klass = RESOURCE_MANAGER_CLASS (GTK_OBJECT (rm)->klass);
	priv  = rm->priv;

	res_list = corba_util_id_seq_to_list (ids);
	
	if (!res_list) {
		/* FIX: Set exception ? */
		return;
	}

	klass->remove_resources (rm, res_list);
	
	/* Notify clients. */
	arg    = bonobo_arg_new (BONOBO_ARG_STRING);
	id_str = corba_util_id_string_from_list (res_list);
	
	BONOBO_ARG_SET_STRING (arg, id_str);
	
	if (priv->event_source) {
		CORBA_exception_init (&e);

		bonobo_event_source_notify_listeners (priv->event_source,
						      "GNOME/MrProject:resource:removed_seq",
						      arg,
						      &e);
		CORBA_exception_free (&e);
	} 

	CORBA_free (id_str);
	bonobo_arg_release (arg);
	
	g_slist_free (res_list);
	
	priv->state = GNOME_MrProject_PROJECT_STATE_DIRTY;
}

/* Group Management */

static GNOME_MrProject_ResourceGroup *
impl_ResourceManager_createGroup (
	PortableServer_Servant              servant,
	CORBA_Environment                  *ev)
{
	GNOME_MrProject_ResourceGroup   *group;
	
	group              = GNOME_MrProject_ResourceGroup__alloc ();
	group->groupId     = -1;
	group->name        = CORBA_string_dup ("Uninitialized");
	group->adminName   = CORBA_string_dup ("");
	group->adminPhone  = CORBA_string_dup ("");
	group->adminEmail  = CORBA_string_dup ("");

	return group;
}

static GNOME_MrProject_Id
impl_ResourceManager_insertGroup (
	PortableServer_Servant                 servant,
	const GNOME_MrProject_ResourceGroup   *group,
	CORBA_Environment                     *ev)
{
	ResourceManager                 *rm;
	ResourceManagerClass            *klass;
	ResourceManagerPriv             *priv;
	gint                             id;
	GNOME_MrProject_ResourceGroup   *copy;
	CORBA_any                       *any;	
	CORBA_Environment                e;	
	gchar                           *str;

	rm    = RESOURCE_MANAGER (bonobo_x_object (servant));
 	klass = RESOURCE_MANAGER_CLASS (GTK_OBJECT (rm)->klass);
	priv  = rm->priv;
	id    = klass->insert_group (rm, group);
	
	copy           = corba_util_resource_group_duplicate (group);
	copy->groupId  = id;

	if (!strcmp (copy->name, "Uninitialized")) {
		CORBA_free (copy->name);
		str = g_strdup_printf (_("Group %02d"), copy->groupId);
		copy->name = CORBA_string_dup (str);
		g_free (str);
	}

	klass->update_group (rm, copy);

  	any          = CORBA_any__alloc ();
	any->_type   = TC_GNOME_MrProject_ResourceGroup;
	any->_value  = copy;

	CORBA_exception_init (&e);
	
	if (priv->event_source) {
		bonobo_event_source_notify_listeners (
			priv->event_source,
			"GNOME/MrProject:resource:group_inserted",
			any,
			&e);
	} else {
		g_warning ("Could not get EventSource interface");
	}

	priv->state = GNOME_MrProject_PROJECT_STATE_DIRTY;

	CORBA_exception_free (&e);
	CORBA_free (any);
	
	return id;
}

static GNOME_MrProject_ResourceGroup *
impl_ResourceManager_getGroup (
	PortableServer_Servant     servant,
	const GNOME_MrProject_Id   groupId,
	CORBA_Environment          *ev)
{
	ResourceManager                 *rm;
	ResourceManagerClass            *klass;
	ResourceManagerPriv             *priv;
	GNOME_MrProject_ResourceGroup   *group;
	GNOME_MrProject_ResourceGroup   *copy;
	
	rm    = RESOURCE_MANAGER (bonobo_x_object (servant));
 	klass = RESOURCE_MANAGER_CLASS (GTK_OBJECT (rm)->klass);
	priv  = rm->priv;

	group = klass->get_group (rm, groupId);
	
	if (!group) {
		CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
				     ex_GNOME_MrProject_ResourceManager_NoSuchGroup,
				     NULL);
		return NULL;
	}
	
	copy = corba_util_resource_group_duplicate (group);
	
	return copy;
}

static GNOME_MrProject_ResourceGroupSeq *
impl_ResourceManager_getAllGroups (
	PortableServer_Servant                   servant,
	CORBA_Environment                       *ev)
{
	ResourceManager                    *rm;
	ResourceManagerClass               *klass;
	ResourceManagerPriv                *priv;
	GNOME_MrProject_ResourceGroupSeq   *groups;
	GSList                             *list;
	
	rm     = RESOURCE_MANAGER (bonobo_x_object (servant));
 	klass  = RESOURCE_MANAGER_CLASS (GTK_OBJECT (rm)->klass);
	priv   = rm->priv;
	list   = klass->get_all_groups (rm);
	groups = corba_util_resource_group_seq_from_list (list);

	g_slist_free (list);

	return groups;
}

static void
impl_ResourceManager_updateGroup (
	PortableServer_Servant                 servant,
	const GNOME_MrProject_ResourceGroup   *group,
	CORBA_Environment                     *ev)
{
	ResourceManager                 *rm;
	ResourceManagerClass            *klass;
	ResourceManagerPriv             *priv;
	CORBA_any                       *any;	
	CORBA_Environment                e;	
	GNOME_MrProject_ResourceGroup   *updated_group;
	GNOME_MrProject_ResourceGroup   *copy;

	g_return_if_fail (group != NULL);

	rm    = RESOURCE_MANAGER (bonobo_x_object (servant));
 	klass = RESOURCE_MANAGER_CLASS (GTK_OBJECT (rm)->klass);
	priv  = rm->priv;

	if (klass->update_group (rm, group)) {
		updated_group = klass->get_group (rm, group->groupId);
		copy          = corba_util_resource_group_duplicate (updated_group);
		any           = CORBA_any__alloc ();
		any->_type    = TC_GNOME_MrProject_ResourceGroup;
		any->_value   = copy;

		/* Makes CORBA_free (any) free any->_value */
		CORBA_any_set_release (any, CORBA_TRUE);

		CORBA_exception_init (&e);
		
		if (priv->event_source) {
			bonobo_event_source_notify_listeners (
				priv->event_source,
				"GNOME/MrProject:resource:group_updated",
				any,
				&e);
		} else {
			g_warning ("Could not get EventSource interface");
		}
		
		priv->state = GNOME_MrProject_PROJECT_STATE_DIRTY;
		
		CORBA_exception_free (&e);
		CORBA_free (any);
	}
}

static void
impl_ResourceManager_removeGroup (
	PortableServer_Servant    servant,
	GNOME_MrProject_Id        groupId,
	CORBA_Environment        *ev)
{
	ResourceManager                             *rm;
	ResourceManagerClass                        *klass;
	ResourceManagerPriv                         *priv;
	CORBA_any                                   *any;	
	CORBA_Environment                            e;	
	GNOME_MrProject_EventResourceGroupRemoved   *data;

	rm    = RESOURCE_MANAGER (bonobo_x_object (servant));
	priv  = rm->priv;
 	klass = RESOURCE_MANAGER_CLASS (GTK_OBJECT (rm)->klass);
	
	klass->remove_group (rm, groupId);
	
	data         = GNOME_MrProject_EventResourceGroupRemoved__alloc ();
	data->id     = groupId;
	any          = CORBA_any__alloc ();
	any->_type   = TC_GNOME_MrProject_EventResourceGroupRemoved;
	any->_value  = data;

	/* Makes CORBA_free (any) free any->_value */
	CORBA_any_set_release (any, CORBA_TRUE);

	CORBA_exception_init (&e);
	
	if (priv->event_source) {
		bonobo_event_source_notify_listeners (
			priv->event_source,
			"GNOME/MrProject:resource:group_removed",
			any,
			&e);
	} else {
		g_warning ("Could not get EventSource interface");
	}

	CORBA_exception_free (&e);
	CORBA_free (any);

	priv->state = GNOME_MrProject_PROJECT_STATE_DIRTY;
}

static GNOME_MrProject_Id
impl_ResourceManager_getDefaultGroup (
	PortableServer_Servant              servant,
	CORBA_Environment                  *ev)
{
	ResourceManager        *rm;
	ResourceManagerClass   *klass;
	
	rm    = RESOURCE_MANAGER (bonobo_x_object (servant));
 	klass = RESOURCE_MANAGER_CLASS (GTK_OBJECT (rm)->klass);

	return klass->get_default_group (rm);
}

static void
impl_ResourceManager_setDefaultGroup (
	PortableServer_Servant    servant,
	GNOME_MrProject_Id        groupId,
	CORBA_Environment        *ev)
{
	ResourceManager                                *rm;
	ResourceManagerClass                           *klass; 
	ResourceManagerPriv                            *priv;
	CORBA_any                                      *any;	
	CORBA_Environment                               e;	
	GNOME_MrProject_EventResourceGroupDefaultSet   *data;
	
	rm    = RESOURCE_MANAGER (bonobo_x_object (servant));
 	klass = RESOURCE_MANAGER_CLASS (GTK_OBJECT (rm)->klass);
	priv  = rm->priv;

	klass->set_default_group (rm, groupId);

	data         = GNOME_MrProject_EventResourceGroupDefaultSet__alloc ();
	data->id     = groupId;
	any          = CORBA_any__alloc ();
	any->_type   = TC_GNOME_MrProject_EventResourceGroupDefaultSet;
	any->_value  = data;

	/* Makes CORBA_free (any) free any->_value */
	CORBA_any_set_release (any, CORBA_TRUE);

	CORBA_exception_init (&e);
	
	if (priv->event_source) {
		bonobo_event_source_notify_listeners (
			priv->event_source,
			"GNOME/MrProject:resource:group_default_set",
			any,
			&e);
	} else {
		g_warning ("Could not get EventSource interface");
	}

	CORBA_exception_free (&e);
	CORBA_free (any);

	priv->state = GNOME_MrProject_PROJECT_STATE_DIRTY;
}

static void
resource_manager_init (ResourceManager   *rm)
{
        ResourceManagerPriv   *priv;

        priv                 = g_new0 (ResourceManagerPriv, 1);
        rm->priv             = priv;
        
	priv->state          = GNOME_MrProject_PROJECT_STATE_EMPTY;
}

static void
resource_manager_class_init (ResourceManagerClass *klass)
{
	POA_GNOME_MrProject_ResourceManager__epv   *epv;
        GtkObjectClass                             *object_class;
        
	epv           = &klass->epv;
        object_class  = (GtkObjectClass *) klass;
        parent_class  = gtk_type_class (gtk_object_get_type ());
        
        object_class->destroy  = rm_destroy;
	object_class->get_arg  = rm_get_arg;
	object_class->set_arg  = rm_set_arg;
	
	/* Resource Management */
	epv->_set_state          = impl_ResourceManager__set_state;
	epv->_get_state          = impl_ResourceManager__get_state;
	epv->createResource      = impl_ResourceManager_createResource;
	epv->insertResource      = impl_ResourceManager_insertResource;
	epv->getResource         = impl_ResourceManager_getResource;
	epv->getAllResources     = impl_ResourceManager_getAllResources;
	epv->updateResource      = impl_ResourceManager_updateResource;
	epv->removeResources     = impl_ResourceManager_removeResources;
	
	/* Group Management */
	epv->createGroup         = impl_ResourceManager_createGroup;
	epv->insertGroup         = impl_ResourceManager_insertGroup;
	epv->getGroup            = impl_ResourceManager_getGroup;
	epv->getAllGroups        = impl_ResourceManager_getAllGroups;

	epv->getDefaultGroup     = impl_ResourceManager_getDefaultGroup;
	epv->setDefaultGroup     = impl_ResourceManager_setDefaultGroup;
	epv->updateGroup         = impl_ResourceManager_updateGroup;
	epv->removeGroup         = impl_ResourceManager_removeGroup;

	/* Object arguments */
	gtk_object_add_arg_type ("ResourceManager::state", 
				 GTK_TYPE_INT, 
				 GTK_ARG_READWRITE, 
				 ARG_STATE);
}

static void
rm_get_arg (GtkObject   *object, 
	    GtkArg      *arg, 
	    guint        arg_id)
{
	ResourceManager       *rm;
	ResourceManagerPriv   *priv;
	
	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_RESOURCE_MANAGER (object));

	rm  = RESOURCE_MANAGER (object);
	priv     = rm->priv;

	switch (arg_id) {
	case ARG_STATE:
		GTK_VALUE_INT (*arg) = priv->state;
		break;
		
	default:
		arg->type = GTK_TYPE_INVALID;
		break;
	}
}

static void
rm_set_arg (GtkObject   *object, 
	    GtkArg      *arg, 
	    guint        arg_id)
{
	ResourceManager       *rm;
	ResourceManagerPriv   *priv;
	
	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_RESOURCE_MANAGER (object));

	rm  = RESOURCE_MANAGER (object);
	priv     = rm->priv;

	switch (arg_id) {
	case ARG_STATE:
		priv->state = GTK_VALUE_INT (*arg);
		break;
	default:
		return;
		break;
	}
}

void
resource_manager_construct (ResourceManager     *rm, 
			    BonoboEventSource   *event_source)
{
	ResourceManagerPriv *priv;

	priv = rm->priv;
	priv->event_source = event_source;
}

static void 
rm_destroy (GtkObject *object)
{
        ResourceManager        *rm;
        ResourceManagerPriv    *priv;
        
        g_return_if_fail (object != NULL);
        g_return_if_fail (IS_RESOURCE_MANAGER (object));

        rm   =  RESOURCE_MANAGER (object);
        priv =  rm->priv;
        
	if (priv->event_source) {
		priv->event_source = NULL;
	}

        GNOME_CALL_PARENT_HANDLER (GTK_OBJECT_CLASS, destroy, (object));
}

void
resource_manager_foreach_resource (ResourceManager              *rm,
				   ResourceManagerForeachFunc    func,
				   gpointer                      user_data)
{
	ResourceManagerClass *klass;
	
	g_return_if_fail (rm != NULL);
	g_return_if_fail (IS_RESOURCE_MANAGER (rm));
	g_return_if_fail (func != NULL);

	klass = RESOURCE_MANAGER_CLASS (GTK_OBJECT (rm)->klass);
	if (klass->foreach_resource) {
		klass->foreach_resource (rm, func, user_data);
	}
}

void
resource_manager_foreach_group (ResourceManager   *rm,
				GHFunc             func,
				gpointer           user_data)
{
	ResourceManagerClass *klass;

	g_return_if_fail (rm != NULL);
	g_return_if_fail (IS_RESOURCE_MANAGER (rm));
	g_return_if_fail (func != NULL);

	klass = RESOURCE_MANAGER_CLASS (GTK_OBJECT (rm)->klass);
	if (klass->foreach_group) {
		klass->foreach_group (rm, func, user_data);
	}
}

BONOBO_X_TYPE_FUNC_FULL (ResourceManager,
			 GNOME_MrProject_ResourceManager,
			 PARENT_TYPE,
			 resource_manager);









