 /******************************************************************
 
        MUSCLE SmartCard Development ( http://www.linuxnet.com )
            Title  : muscleTool.c
            Package: MuscleTools
            Author : David Corcoran
            Date   : 03/14/02
            License: Copyright (C) 2002 David Corcoran
                     <corcoran@linuxnet.com>
            Purpose: A shell for the MuscleCard framework 
 
********************************************************************/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#ifndef WIN32
#include <unistd.h>
#endif

#include <sys/stat.h>
#include <pthread.h>
#ifndef __APPLE__
#include <musclecard.h>
#else
#include <PCSC/musclecard.h>
#endif


#define MUSCLETOOL_VERSION  "0.9.2"


static MSCLPTokenConnection pConnection = 0;
static MSCLPTokenInfo tokenList = 0;
static MSCULong32 tokenSize     = 0;
static MSCString connectToken = 0;
static int quitStat = 0;

void doRelease();
void doPrintAUT(MSCUShort16);


#define CHECK_ERR(cond, msg) { if (cond) { \
  printf("ERR: %s (0x%lX %s)\n", msg, rv, msc_error(rv)); return; } }

#define PRINT_SUPPORT(cond) { if (cond) { \
  printf("X\n"); } else { printf("\n"); } }

void printStatStars() {
  while (1) {
#ifndef WIN32
    usleep(500000);
#else
    Sleep(500);
#endif
    putc('*', stdout);
    fflush(stdout);
    if (quitStat == 1) {
      break;
    }
  }
  
  printf("] : ");
  quitStat = 0;
}


MSCULong32 tokenCallback(MSCLPTokenInfo tokInfo, MSCULong32 tokSize,
			 MSCPVoid32 data) {
 

  doRelease();

  printf("Token Removed - Press Enter to Continue\n");

  return 0;
 }
 
static MSCUChar8 defaultCFlexKey[8] = {0x2C, 0x15, 0xE5, 0x26, 0xE9, 0x3E, 0x8A, 0x19};
static MSCUChar8 muscleCFlexKey[8]  = {'M', 'u', 's', 'c', 'l', 'e', '0', '0'};

int textToBytes(MSCString inStr, MSCPUChar8 Buffer, MSCPULong32 Length)
{

	int i;
	int j;
	int inLen;

	j = 0;
	inLen = 0;

	inLen = strlen(inStr);

	if (inLen > MSC_MAXSIZE_AID)
	{
		return -1;
	}

	for (i = 0; i < inLen; i += 2)
	{
		if (inStr[i] <= '9' && inStr[i] >= '0')
		{
			Buffer[j] = (inStr[i] - '0') * 16;
		} else if (inStr[i] <= 'F' && inStr[i] >= 'A')
		{
			Buffer[j] = (inStr[i] - 'A' + 10) * 16;
		}

		if (inStr[i + 1] <= '9' && inStr[i + 1] >= '0')
		{
			Buffer[j] += inStr[i + 1] - '0';
		} else if (inStr[i + 1] <= 'F' && inStr[i + 1] >= 'A')
		{
			Buffer[j] += inStr[i + 1] - 'A' + 10;
		}

		j += 1;
	}

	*Length = j;

	return 0;
}


void binToHex(MSCUChar8 *data, MSCULong32 data_len, MSCUChar8 *out)
{
  MSCULong32 i;

  for (i = 0; i < data_len; i++)
    sprintf(&out[i * 2], "%.2X", data[i]);

  out[data_len * 2] = 0;
}

MSCULong32 hexToBin(MSCUChar8 *data, MSCUChar8 *out)
{
  MSCULong32 i;
  MSCULong32 count = 0;
  MSCUShort16 c;
  MSCUChar8 shortData[3];

  for (i = 0; i < strlen(data); i += 2)
  {
    shortData[0] = data[i];
    shortData[1] = data[i+1];
    shortData[2] = 0x00;
    sscanf(shortData, "%hX", &c);
    out[count] = c;
    count++;
  }

  return count;
}

void padData(MSCUChar8 *data, MSCULong32 inSize, MSCULong32 outSize)
{
  MSCULong32 i;

  for (i = inSize; i < outSize; i++)
    data[i] = 0;
}

void doHelp() {
   
  MSCString arg2;

  arg2 = strtok(NULL, " ");

  if (arg2 != NULL) {
    if (strcmp(arg2, "resume") == 0) {
      printf(" Resume related functions \n");
      printf("----------------------------------------------------\n");
      printf("     crypt     - Lists available crypto algorithms\n");
      printf("     rsa       - Lists RSA capabilities\n");
      printf("     dsa       - Lists DSA capabilities\n");
      printf("     des       - Lists DES capabilities\n");
      printf("     3des      - Lists 3DES capabilities\n");
      printf("     keys      - Lists Key capabilities\n");
      printf("     pins      - Lists PIN capabilities\n");
      printf("     objects   - Lists Object capabilities\n");
    } else if (strcmp(arg2, "challenge") == 0) {
      printf("challenge [number]    - returns number of random\n");
    } else if (strcmp(arg2, "verify") == 0) {
      printf("verify [number] [value] - PIN #, PIN Value\n");
    } else if (strcmp(arg2, "create") == 0) {
      printf("create [id] [size]      - Create object\n");
    } else if (strcmp(arg2, "read") == 0) {
      printf("read [id] [file]        - Read from token to file\n");
    } else if (strcmp(arg2, "write") == 0) {
      printf("write [file] [id]       - Write to token from file\n");
    } else if (strcmp(arg2, "delete") == 0) {
      printf("delete [id]             - Delete object\n");
    } 

    return;
  }

  
  printf("HELP - type help [topic] for more information\n");
  printf("Topics which contain more information will contain a +\n\n");
  
  printf(" Object related functions \n");
  printf("----------------------------------------------------\n");
  printf("     create    - Create an object +\n");
  printf("     delete    - Delete an object +\n");
  printf("     write     - Write from file to object +\n");
  printf("     read      - Read from object to file +\n");
  printf("     list      - List objects on token\n");
  printf("\n");
  printf(" PIN related functions \n");
  printf("----------------------------------------------------\n");
  printf("     verify    - Verify a PIN +\n");
  printf("     makepin   - Create a PIN\n"); 
  printf("     changepin - Change a PIN\n");
  printf("     listpins  - List the PINs\n");
  printf("\n");
  printf(" Key related functions \n");
  printf("----------------------------------------------------\n");
  printf("     listkeys    - List the keys\n");
  printf("     exportkey   - Export a key\n");
  printf("     importkey   - Import a key\n");
  printf("     genkeys     - Generate keys\n");
  printf("     crypt       - Perform a cipher operation\n");

  printf("\n");
  printf(" General functions \n");
  printf("----------------------------------------------------\n");
  printf("     format    - Format the token\n");
  printf("     tokens    - Lists tokens available\n");
  printf("     connect   - Connect to a token\n");
  printf("     resume    - Get the token's capabilities + \n");
  printf("     release   - Release a token\n");
  printf("     status    - Gets the token's status\n");
  printf("     challenge - Gets a random number + \n");
  printf("     logout    - Logs out AUT\n");
  printf("     version   - Print the version\n");
  printf("     exit      - Exits this program\n");

}

void doListTokens() {
  
  MSCLong32 rv;
  int i;
  
  if (tokenList) { 
    free(tokenList); tokenList = 0; tokenSize = 0; 
  }
  
  rv = MSCListTokens( MSC_LIST_KNOWN, tokenList, &tokenSize );
  CHECK_ERR(rv != MSC_SUCCESS, "ListTokens Failed");
  
  if (tokenSize == 0) {
    printf("No Valid Tokens Found\n");
    return;
  }

  tokenList = (MSCLPTokenInfo)malloc(sizeof(MSCTokenInfo)*tokenSize);
  CHECK_ERR(tokenList == 0, "Malloc Failed");
  
  rv = MSCListTokens( MSC_LIST_KNOWN, tokenList, &tokenSize );
  CHECK_ERR(rv != MSC_SUCCESS, "ListTokens Failed");
  
  for (i=0; i < tokenSize; i++) {
    printf("   %d.    %s\n", i+1, tokenList[i].tokenName);
  } printf("\n");
  
}

void doConnect() {

  MSCLong32 rv;
  MSCString arg2;
  MSCULong32 tokenNumber;

  CHECK_ERR(tokenList == 0, "List Tokens First !");

  arg2 = strtok(NULL, " ");
  if (arg2 == NULL) {
    printf("ERR: Invalid Use: connect [number]\n");
    return;
  }

  tokenNumber = strtoul(arg2, 0, 10);

  if (tokenNumber < 1 || tokenNumber > tokenSize ) {
    printf("ERR: Invalid choice made !\n");
    return;
  }

  if ( pConnection == 0 ) {
    pConnection = (MSCLPTokenConnection)malloc(sizeof(MSCTokenConnection));
  } else {
    printf("ERR: Must Release Other Connection !\n");
    return;
  }

  rv = MSCEstablishConnection( &tokenList[tokenNumber-1], 
			       MSC_SHARE_SHARED, 0, 
                               0, pConnection );
  if ( rv != MSC_SUCCESS ) {
    free(pConnection); pConnection = 0;
    printf("ERR: EstablishConnection Failed !");
    return;
  }

  //  rv = MSCCallbackForTokenEvent(&tokenList[tokenNumber-1], 1, 
  //				tokenCallback, 0 );

  connectToken = strdup(tokenList[tokenNumber-1].tokenName);

}

void doRelease() {

  MSCLong32 rv;

  CHECK_ERR(pConnection == 0, "Must Connect First !");

  //  rv = MSCCallbackCancelEvent();

  rv = MSCReleaseConnection(pConnection, MSC_RESET_TOKEN);
 
  free(pConnection); pConnection = 0;
  free(connectToken); connectToken = 0;
   
  CHECK_ERR(rv != MSC_SUCCESS, "ReleaseConnection Failed !");

}


void doList() {

  MSCLong32 rv;
  MSCObjectInfo objInfo;

  CHECK_ERR(pConnection == 0, "Must Connect First !");

  rv = MSCListObjects( pConnection, MSC_SEQUENCE_RESET, &objInfo );
  
  printf("%20s %12s %8s%8s %8s\n", "Object ID", "Object Size", 
	 "READ", "WRITE", "DELETE");
  printf("   -----------------  -----------    ------  ------  ------\n");
  
  if ( rv == MSC_SUCCESS ) {
   printf("%20s   %10ld  ", objInfo.objectID, 
	   objInfo.objectSize);
	   doPrintAUT(objInfo.objectACL.readPermission), 
	   doPrintAUT(objInfo.objectACL.writePermission),
	   doPrintAUT(objInfo.objectACL.deletePermission);
           printf("\n");  
  } else {
    CHECK_ERR(rv != MSC_SUCCESS, "ListObjects Failed");
    return;
  }  

  do {
    rv = MSCListObjects( pConnection, MSC_SEQUENCE_NEXT, &objInfo );
    if ( rv == MSC_SUCCESS ) {
   printf("%20s   %10ld  ", objInfo.objectID, 
	   objInfo.objectSize);
	   doPrintAUT(objInfo.objectACL.readPermission), 
	   doPrintAUT(objInfo.objectACL.writePermission),
	   doPrintAUT(objInfo.objectACL.deletePermission);
           printf("\n");  

    } else {
      if ( rv != MSC_SEQUENCE_END ) {
        CHECK_ERR(rv != MSC_SUCCESS, "ListObjects Failed");
      }
      break;
    }
    
  } while (1);
    
}

void doPrintAUT(MSCUShort16 aut) {

  if (aut == MSC_AUT_NONE) {
    printf("%8s", "NEVER");
    return;
  } else if (aut == MSC_AUT_ALL) {
    printf("%8s", "ALWAYS");
    return;
  }

  if (aut & MSC_AUT_PIN_0) {
    printf("%8s", "PIN #0");
  }

  if (aut & MSC_AUT_PIN_1) {
    printf("%8s", "PIN #1");
  }

  if (aut & MSC_AUT_PIN_2) {
    printf("%8s", "PIN #2");
  }

  if (aut & MSC_AUT_PIN_3) {
    printf("%8s", "PIN #3");
  }

  if (aut & MSC_AUT_PIN_4) {
    printf("%8s", "PIN #4");
  }

  if (aut & MSC_AUT_KEY_0) {
    printf("%8s", "KEY #0");
  }

  if (aut & MSC_AUT_KEY_1) {
    printf("%8s", "KEY #1");
  }

  if (aut & MSC_AUT_KEY_2) {
    printf("%8s", "KEY #2");
  }

  if (aut & MSC_AUT_KEY_3) {
    printf("%8s", "KEY #3");
  }

  if (aut & MSC_AUT_KEY_4) {
    printf("%8s", "KEY #4");
  }

  if (aut & MSC_AUT_KEY_5) {
    printf("%8s", "KEY #5");
  }

}

void doLoggedID(MSCULong32 aut) {

  if (aut == 0) {
    printf("NONE");
    return;
  }

  if (aut & MSC_AUT_PIN_0) {
    printf("PIN #0 ");
  }

  if (aut & MSC_AUT_PIN_1) {
    printf("PIN #1 ");
  }

  if (aut & MSC_AUT_PIN_2) {
    printf("PIN #2 ");
  }

  if (aut & MSC_AUT_PIN_3) {
    printf("PIN #3 ");
  }

  if (aut & MSC_AUT_PIN_4) {
    printf("PIN #4 ");
  }

  if (aut & MSC_AUT_KEY_0) {
    printf("KEY #0 ");
  }

  if (aut & MSC_AUT_KEY_1) {
    printf("KEY #1 ");
  }

  if (aut & MSC_AUT_KEY_2) {
    printf("KEY #2 ");
  }

  if (aut & MSC_AUT_KEY_3) {
    printf("KEY #3 ");
  }

  if (aut & MSC_AUT_KEY_4) {
    printf("KEY #4 ");
  }

  if (aut & MSC_AUT_KEY_5) {
    printf("KEY #5 ");
  }


}

void doStatus() {

  MSCLong32 rv;
  MSCStatusInfo pStatusStruct;

  CHECK_ERR(pConnection == 0, "Must Connect First !");

  rv = MSCGetStatus(pConnection, &pStatusStruct);
  CHECK_ERR(rv != MSC_SUCCESS, "GetStatus Failed !");

  printf("         Version: %04X %04X\n", pStatusStruct.appVersion,
	 pStatusStruct.swVersion);
  printf("     Free Memory: %ld\n", pStatusStruct.freeMemory);
  printf("    Total Memory: %ld\n", pStatusStruct.totalMemory);
  printf("       PINs Used: %d\n", pStatusStruct.usedPINs);
  printf("       Keys Used: %d\n", pStatusStruct.usedKeys);
  printf("      Logged IDs: "); 
  doLoggedID(pStatusStruct.loggedID);
  printf("\n");
}

void doResume() {

  MSCLong32 rv;
  MSCULong32 retLength;
  MSCULong32 ulValue;
  MSCUChar8 ucValue;
  MSCUShort16 usValue;
  MSCString arg2;
  
  CHECK_ERR(pConnection == 0, "Must Connect First !");
  
  ulValue = 0;
  rv = MSCGetCapabilities(pConnection, MSC_TAG_SUPPORT_FUNCTIONS, 
			  (MSCPUChar8)&ulValue, &retLength);
  if ( rv != MSC_SUCCESS ) {
    printf("ERR: No TAG_SUPPORT_FUNCTIONS !\n");
    return;
  }
  

  arg2 = strtok(NULL, " ");


  if ( arg2 != NULL ) {
    if (strcmp(arg2, "crypt") == 0) {
      
      ulValue = 0;
      rv = MSCGetCapabilities(pConnection, MSC_TAG_SUPPORT_CRYPTOALG, 
			      (MSCPUChar8)&ulValue, &retLength);
      if ( rv != MSC_SUCCESS ) {
	printf("ERR: No TAG_SUPPORT_FUNCTIONS !\n");
	return;
      }
      
      printf("Crypto Algorithm    Supported\n");
      printf("-------------------------------\n");
      printf("RSA                    ");
      PRINT_SUPPORT(ulValue & MSC_SUPPORT_RSA);
      printf("DSA                    ");
      PRINT_SUPPORT(ulValue & MSC_SUPPORT_DSA);
      printf("ELGAMAL                ");
      PRINT_SUPPORT(ulValue & MSC_SUPPORT_ELGAMAL);
      printf("DES                    ");
      PRINT_SUPPORT(ulValue & MSC_SUPPORT_DES);
      printf("3DES                   ");
      PRINT_SUPPORT(ulValue & MSC_SUPPORT_3DES);
      printf("IDEA                   ");
      PRINT_SUPPORT(ulValue & MSC_SUPPORT_IDEA);
      printf("AES                    ");
      PRINT_SUPPORT(ulValue & MSC_SUPPORT_AES);
      printf("BLOWFISH               ");
      PRINT_SUPPORT(ulValue & MSC_SUPPORT_BLOWFISH);
      printf("TWOFISH                ");
      PRINT_SUPPORT(ulValue & MSC_SUPPORT_TWOFISH);
      printf("SHA1                   ");
      PRINT_SUPPORT(ulValue & MSC_SUPPORT_SHA1);
      printf("MD5                    ");
      PRINT_SUPPORT(ulValue & MSC_SUPPORT_MD5); 
      return;
    } else if (strcmp(arg2, "rsa") == 0) {
      
      ulValue = 0;
      rv = MSCGetCapabilities(pConnection, MSC_TAG_CAPABLE_RSA, 
			      (MSCPUChar8)&ulValue, &retLength);
      if ( rv != MSC_SUCCESS ) {
	printf("ERR: No TAG_SUPPORT_FUNCTIONS !\n");
	return;
      }
      
      printf("RSA Capabilities     Supported\n");
      printf("-------------------------------\n");
      printf("512 Bit                ");
      PRINT_SUPPORT(ulValue & MSC_CAPABLE_RSA_512);
      printf("768 Bit                ");
      PRINT_SUPPORT(ulValue & MSC_CAPABLE_RSA_768);
      printf("1024 Bit               ");
      PRINT_SUPPORT(ulValue & MSC_CAPABLE_RSA_1024);
      printf("2048 Bit               ");
      PRINT_SUPPORT(ulValue & MSC_CAPABLE_RSA_2048);
      printf("4096 Bit               ");
      PRINT_SUPPORT(ulValue & MSC_CAPABLE_RSA_4096);
      printf("Key Generation         ");
      PRINT_SUPPORT(ulValue & MSC_CAPABLE_RSA_KEYGEN);
      printf("No Padding             ");
      PRINT_SUPPORT(ulValue & MSC_CAPABLE_RSA_NOPAD);
      printf("PKCS#1 Padding         ");
      PRINT_SUPPORT(ulValue & MSC_CAPABLE_RSA_PKCS1);
      return;
   
      
    } else if (strcmp(arg2, "dsa") == 0) {
      
      ulValue = 0;
      rv = MSCGetCapabilities(pConnection, MSC_TAG_CAPABLE_DSA, 
			      (MSCPUChar8)&ulValue, &retLength);
      if ( rv != MSC_SUCCESS ) {
	printf("ERR: Not Supported !\n");
	return;
      }
      
      printf("DSA Capabilities     Supported\n");
      printf("-------------------------------\n");
      printf("512 Bit                ");
      PRINT_SUPPORT(ulValue & MSC_CAPABLE_DSA_512);
      printf("768 Bit                ");
      PRINT_SUPPORT(ulValue & MSC_CAPABLE_DSA_768);
      printf("1024 Bit               ");
      PRINT_SUPPORT(ulValue & MSC_CAPABLE_DSA_1024);
      printf("2048 Bit               ");
      PRINT_SUPPORT(ulValue & MSC_CAPABLE_DSA_2048);
      printf("4096 Bit               ");
      PRINT_SUPPORT(ulValue & MSC_CAPABLE_DSA_4096);
      printf("Key Generation         ");
      PRINT_SUPPORT(ulValue & MSC_CAPABLE_DSA_KEYGEN);
      return;

    }  else if (strcmp(arg2, "des") == 0) {
      
      ulValue = 0;
      rv = MSCGetCapabilities(pConnection, MSC_TAG_CAPABLE_DES, 
			      (MSCPUChar8)&ulValue, &retLength);
      if ( rv != MSC_SUCCESS ) {
	printf("ERR: Not Supported !\n");
	return;
      }
      
      printf("DES Capabilities     Supported\n");
      printf("-------------------------------\n");
      printf("CBC Mode               ");
      PRINT_SUPPORT(ulValue & MSC_CAPABLE_DES_CBC);
      printf("EBC Mode               ");
      PRINT_SUPPORT(ulValue & MSC_CAPABLE_DES_EBC);
      printf("ECB Mode               ");
      PRINT_SUPPORT(ulValue & MSC_CAPABLE_DES_ECB);
      printf("Key Generation         ");
      PRINT_SUPPORT(ulValue & MSC_CAPABLE_DES_KEYGEN);
      return;

    } else if (strcmp(arg2, "3des") == 0) {
      
      ulValue = 0;
      rv = MSCGetCapabilities(pConnection, MSC_TAG_CAPABLE_3DES, 
			      (MSCPUChar8)&ulValue, &retLength);
      if ( rv != MSC_SUCCESS ) {
	printf("ERR: Not Supported !\n");
	return;
      }
      
      printf("3DES Capabilities     Supported\n");
      printf("-------------------------------\n");
      printf("CBC Mode               ");
      PRINT_SUPPORT(ulValue & MSC_CAPABLE_3DES_CBC);
      printf("EBC Mode               ");
      PRINT_SUPPORT(ulValue & MSC_CAPABLE_3DES_EBC);
      printf("ECB Mode               ");
      PRINT_SUPPORT(ulValue & MSC_CAPABLE_3DES_ECB);
      printf("3 Key Mode             ");
      PRINT_SUPPORT(ulValue & MSC_CAPABLE_3DES_3KEY);
      printf("Key Generation         ");
      PRINT_SUPPORT(ulValue & MSC_CAPABLE_3DES_KEYGEN);
      return;


    }  else if (strcmp(arg2, "keys") == 0) {
      
      printf("AUT needed to import/generate keys:\n    ");
      
      usValue = 0;
      rv = MSCGetCapabilities(pConnection, MSC_TAG_CAPABLE_KEY_AUTH, 
			      (MSCPUChar8)&usValue, &retLength);
      if ( rv != MSC_SUCCESS ) {
	printf("ERR: Not Supported !\n");
	return;
      }
	
      doPrintAUT(usValue);
      printf("\n");

      return;

    } else if (strcmp(arg2, "pins") == 0) {
      
      ucValue = 0;
      rv = MSCGetCapabilities(pConnection, MSC_TAG_CAPABLE_PIN_MAXSIZE, 
			      (MSCPUChar8)&ucValue, &retLength);
	
      printf("PIN Policies\n");
      printf("-------------------------------\n");
      printf("Maximum PIN size         %03d\n", ucValue);
    
      ucValue = 0;
      rv = MSCGetCapabilities(pConnection, MSC_TAG_CAPABLE_PIN_MINSIZE, 
			      (MSCPUChar8)&ucValue, &retLength);
  
      printf("Minimum PIN size         %03d\n", ucValue);

      ucValue = 0;
      rv = MSCGetCapabilities(pConnection, MSC_TAG_CAPABLE_PIN_MAXNUM, 
			      (MSCPUChar8)&ucValue, &retLength);
      
      printf("Maximum number of PINs   %03d\n", ucValue);
      
      ulValue = 0;
      rv = MSCGetCapabilities(pConnection, MSC_TAG_CAPABLE_PIN_ATTR, 
			      (MSCPUChar8)&ulValue, &retLength);

      printf("Unblock PIN resets PIN   ");
      if ( ulValue & MSC_CAPABLE_PIN_RESET ) {
	printf("YES\n");
      } else {
	printf(" NO\n");
      }

      printf("Unblock PIN leaves PIN   ");
      if ( ulValue & MSC_CAPABLE_PIN_LEAVE ) {
	printf("YES\n");
      } else {
	printf(" NO\n");
      }      

      printf("AUT needed to create pins:\n    ");
      
      usValue = 0;
      rv = MSCGetCapabilities(pConnection, MSC_TAG_CAPABLE_PIN_AUTH, 
			      (MSCPUChar8)&usValue, &retLength);
      if ( rv != MSC_SUCCESS ) {
	printf("ERR: Not Supported !\n");
	return;
      }
	
      doPrintAUT(usValue);
      printf("\n");

      return;

    } else if (strcmp(arg2, "objects") == 0) {
 
     
      ucValue = 0;
      rv = MSCGetCapabilities(pConnection, MSC_TAG_CAPABLE_OBJ_IDSIZE, 
			      (MSCPUChar8)&ucValue, &retLength);
	
      printf("Object Capabilities\n");
      printf("-------------------------------\n");
      printf("Maximum Object ID size   %03d\n", ucValue);
      
      ulValue = 0;
      rv = MSCGetCapabilities(pConnection, MSC_TAG_CAPABLE_OBJ_MAXNUM, 
			      (MSCPUChar8)&ucValue, &retLength);
  
      printf("Maximum number objects   %03ld\n", ulValue);

      printf("AUT needed to create objects:\n    ");
      
      usValue = 0;
      rv = MSCGetCapabilities(pConnection, MSC_TAG_CAPABLE_OBJ_AUTH, 
			      (MSCPUChar8)&usValue, &retLength);
      if ( rv != MSC_SUCCESS ) {
	printf("ERR: Not Supported !\n");
	return;
      }
	
      doPrintAUT(usValue);
      printf("\n");

      return;
    }



  }


  printf("Functions             Supported\n");
  printf("-------------------------------\n");
  printf("MSCGenerateKeys          ");
  PRINT_SUPPORT(ulValue & MSC_SUPPORT_GENKEYS);
  printf("MSCImportKey             ");
  PRINT_SUPPORT(ulValue & MSC_SUPPORT_IMPORTKEY);
  printf("MSCExportKey             ");
  PRINT_SUPPORT(ulValue & MSC_SUPPORT_EXPORTKEY);
  printf("MSCComputeCrypt          ");
  PRINT_SUPPORT(ulValue & MSC_SUPPORT_COMPUTECRYPT);
  printf("MSCExternalAuthenticate  ");
  PRINT_SUPPORT(ulValue & MSC_SUPPORT_EXTAUTH);
  printf("MSCListKeys              ");
  PRINT_SUPPORT(ulValue & MSC_SUPPORT_LISTKEYS);
  printf("MSCCreatePIN             ");
  PRINT_SUPPORT(ulValue & MSC_SUPPORT_CREATEPIN);
  printf("MSCVerifyPIN             ");
  PRINT_SUPPORT(ulValue & MSC_SUPPORT_VERIFYPIN);
  printf("MSCChangePIN             ");
  PRINT_SUPPORT(ulValue & MSC_SUPPORT_CHANGEPIN);
  printf("MSCUnblockPIN            ");
  PRINT_SUPPORT(ulValue & MSC_SUPPORT_UNBLOCKPIN);
  printf("MSCListPINs              ");
  PRINT_SUPPORT(ulValue & MSC_SUPPORT_LISTPINS);
  printf("MSCCreateObject          ");
  PRINT_SUPPORT(ulValue & MSC_SUPPORT_CREATEOBJECT);
  printf("MSCDeleteObject          ");
  PRINT_SUPPORT(ulValue & MSC_SUPPORT_DELETEOBJECT);
  printf("MSCWriteObject           ");
  PRINT_SUPPORT(ulValue & MSC_SUPPORT_WRITEOBJECT);
  printf("MSCReadObject            ");
  PRINT_SUPPORT(ulValue & MSC_SUPPORT_READOBJECT);
  printf("MSCListObjects           ");
  PRINT_SUPPORT(ulValue & MSC_SUPPORT_LISTOBJECTS);
  printf("MSCLogoutAll             ");
  PRINT_SUPPORT(ulValue & MSC_SUPPORT_LOGOUTALL);
  printf("MSCGetChallenge          ");
  PRINT_SUPPORT(ulValue & MSC_SUPPORT_GETCHALLENGE);


  
}

void doLogout() {

  MSCLong32 rv;

  rv = MSCLogoutAll(pConnection);

  if (rv == MSC_UNSUPPORTED_FEATURE) {
    printf("ERR: Feature Unsupported\n");
  } else if ( rv == MSC_SUCCESS ) {
    printf("Logged out identities\n");
  } else {
    printf("ERR: LogoutAll Failed !\n");
  }

}

void doChallenge() {

  MSCLong32 rv;
  MSCString arg2;
  MSCUShort16 challengeLength;
  MSCUChar8 randomData[1024];
  //MSCUChar8 pSeed[25];
  int i;

  challengeLength = 8;

  CHECK_ERR(pConnection == 0, "Must Connect First !");

  arg2 = strtok(NULL, " ");

  if ( arg2 != NULL) {
    challengeLength = strtoul(arg2, 0, 10);
    
    if (challengeLength > sizeof(randomData) ) {
      printf("ERR: Invalid choice made !\n");
      return;
    }
  }

  rv = MSCGetChallenge(pConnection, 0, 0, randomData, 
		       challengeLength);

  CHECK_ERR(rv != MSC_SUCCESS, "GetChallenge Failed !");

  for (i=0; i < challengeLength; i++) {
    printf("%02X ", randomData[i]);
  } printf("\n");

  return;
}

void doVerify() {

  MSCLong32 rv;
  MSCString arg2;
  MSCString arg3;
  MSCUChar8 pinNumber;
  MSCULong32 pinSize;
  MSCUChar8 pinData[128];
  //int i;

  CHECK_ERR(pConnection == 0, "Must Connect First !");

  arg2 = strtok(NULL, " ");

  if ( arg2 ) {
    pinNumber = strtoul(arg2, 0, 10);
    
    arg3 = strtok(NULL, " ");
	if (arg3 == NULL) {
		printf("ERR: Invalid Use: verify [number] [value]\n");
		return;
	}

    pinSize = strlen(arg3);

    CHECK_ERR(pinSize > sizeof(pinData), "ERR: Invalid PIN Code !\n");
    memcpy(pinData, arg3, pinSize);
  } else {
    printf("ERR: Invalid Use: verify [number] [value]\n");
    return;
  }

  rv = MSCVerifyPIN(pConnection, pinNumber, pinData, pinSize);
  CHECK_ERR(rv != MSC_SUCCESS, "VerifyPIN Failed !");
  printf("PIN Verify Successful\n");


  return;
}

void doMakePIN() {

  MSCLong32 rv;
  MSCString arg2;
  MSCString arg3;
  MSCString arg4;
  MSCUChar8 pinNumber;
  MSCULong32 pinSize;
  MSCULong32 pinUnblockSize;
  MSCUChar8 pinData[128];
  MSCUChar8 pinUnblockData[128];
  //int i;



  CHECK_ERR(pConnection == 0, "Must Connect First !");

  arg2 = strtok(NULL, " ");

  if ( arg2 ) {
    pinNumber = strtoul(arg2, 0, 10);
    
    arg3 = strtok(NULL, " ");
    pinSize = strlen(arg3);

    arg4 = strtok(NULL, " ");
    pinUnblockSize = strlen(arg4);

    CHECK_ERR(pinSize > sizeof(pinData), "ERR: Invalid PIN Code !\n");
    memcpy(pinData, arg3, pinSize);
    memcpy(pinUnblockData, arg4, pinUnblockSize);
  } else {
    printf("ERR: Invalid Use: makepin [number] [value] [unblockvalue]\n");
    return;
  }

  /* Max tries is hardcoded to 8 */
  rv = MSCCreatePIN(pConnection, pinNumber, 8, pinData, pinSize, pinUnblockData, pinUnblockSize);

  CHECK_ERR(rv != MSC_SUCCESS, "CreatePIN Failed !");
  printf("PIN Create Successful\n");

  return;
}

void doChangePIN() {

  MSCLong32 rv;
  MSCString arg2;
  MSCString arg3;
  MSCString arg4;
  MSCUChar8 pinNumber;
  MSCULong32 pinSize;
  MSCULong32 newPinSize;

  MSCUChar8 pinData[128];
  MSCUChar8 newPinData[128];

  //int i;

  CHECK_ERR(pConnection == 0, "Must Connect First !");

  arg2 = strtok(NULL, " ");

  if ( arg2 ) {
    pinNumber = strtoul(arg2, 0, 10);
    
    arg3 = strtok(NULL, " ");
    pinSize = strlen(arg3);

    arg4 = strtok(NULL, " ");
    newPinSize = strlen(arg4);

    CHECK_ERR(pinSize > sizeof(pinData), "ERR: Invalid PIN Code !\n");
    CHECK_ERR(newPinSize > sizeof(newPinData), "ERR: Invalid PIN Code !\n");

    memcpy(pinData, arg3, pinSize);
    memcpy(newPinData, arg4, newPinSize);
  } else {
    printf("ERR: Invalid Use: changepin [number] [old value] [new value]\n");
    return;
  }

  rv = MSCChangePIN(pConnection, pinNumber, pinData, pinSize, 
                    newPinData, newPinSize);

  CHECK_ERR(rv != MSC_SUCCESS, "ChangePIN Failed !");
  printf("PIN Change Successful\n");

  return;
}

void doListPIN() {

  MSCLong32 rv;
  MSCUShort16 pinStack;

  CHECK_ERR(pConnection == 0, "Must Connect First !");

  rv = MSCListPINs(pConnection, &pinStack);
  CHECK_ERR(rv != MSC_SUCCESS, "ListPINs Failed !");
  printf("   ");
  doLoggedID(pinStack);
  printf("\n");
  
}

void doCreate() {

  MSCLong32 rv;
  MSCString arg2;
  MSCString arg3;
  MSCString objectID;
  MSCULong32 objectSize;
  MSCObjectACL objectACL;
  //int i;

  CHECK_ERR(pConnection == 0, "Must Connect First !");

  arg2 = strtok(NULL, " ");

  if ( arg2 == NULL ) {
    printf("ERR: Invalid Use: create [id] [size]\n");
    return;
  }
  
  objectID = strdup(arg2);  
  arg3 = strtok(NULL, " ");
  
  if (arg3 == NULL) {
    printf("ERR: Invalid Use: create [id] [size]\n");
    return;
  }
  
  objectSize = strtoul(arg3, 0, 10);
  
  /* World read/write/delete permissions hardcoded for now */
  objectACL.readPermission   = MSC_AUT_ALL;
  objectACL.writePermission  = MSC_AUT_PIN_1;
  objectACL.deletePermission = MSC_AUT_PIN_1;

  rv = MSCCreateObject(pConnection, objectID, objectSize, &objectACL);

  free(objectID);
  CHECK_ERR(rv != MSC_SUCCESS, "CreateObject Failed !");
  printf("Create Object Successful\n");
  printf("Object created with world read, pin #1 for delete/write\n");

  return;
}

void doDelete() {
  MSCLong32 rv;
  MSCString arg2;
  //MSCString arg3;
  MSCString objectID;
  //MSCULong32 objectSize;
  //MSCObjectACL objectACL;
  //int i;

  CHECK_ERR(pConnection == 0, "Must Connect First !");

  arg2 = strtok(NULL, " ");

  if ( arg2 == NULL ) {
    printf("ERR: Invalid Use: delete [id] \n");
    return;
  }
  
  objectID = strdup(arg2);  

  rv = MSCDeleteObject(pConnection, objectID, MSC_ZF_DEFAULT);
  CHECK_ERR(rv != MSC_SUCCESS, "DeleteObject Failed !");
  printf("Delete Object Successful\n");

  return;
}

void myCallback(void *randValue, int pctDone) {

  int intPct;
  double dbPct;

  dbPct = (double)pctDone / (double)1000;

  intPct = (int)(dbPct * 100);

  putc('\b', stdout);
  putc('\b', stdout);
  putc('\b', stdout);
  putc('\b', stdout);
  printf("%03d%%", intPct);


  fflush(stdout);
}


void doRead() {

  MSCLong32 rv;
  MSCString arg2;
  MSCString arg3;
  MSCString objectID;
  MSCString fileName;
  MSCPUChar8 objectData;
  MSCULong32 dataSize;
  FILE *fp;

  CHECK_ERR(pConnection == 0, "Must Connect First !");

  arg2 = strtok(NULL, " ");

  if ( arg2 == NULL ) {
    printf("ERR: Invalid Use: read [id] [file]\n");
    return;
  }
  
  objectID = strdup(arg2);  
  arg3 = strtok(NULL, " ");
  
  if (arg3 == NULL) {
    printf("ERR: Invalid Use: read [id] [file]\n");
    return;
  }
  
  fileName = strdup(arg3);

  printf("Percentage read:     ");

  rv = MSCReadAllocateObject(pConnection, objectID, 
			     &objectData, &dataSize,
			     (LPRWEventCallback) myCallback,
			     NULL);

  printf("\n");
  CHECK_ERR(rv != MSC_SUCCESS, "ReadObject Failed !");
  
  fp = fopen(fileName, "w+");

  if (fp == NULL) {
    printf("Open file %s failed\n", fileName);
    return;
  }

  fwrite(objectData, 1, dataSize, fp);
  fclose(fp);

  printf("Read Object Successful\n");
  return;
}

void doWrite() {

  MSCLong32 rv;
  MSCString arg2;
  MSCString arg3;
  MSCString objectID;
  MSCString fileName;
  MSCPUChar8 objectData;
  MSCULong32 dataSize;
  struct stat fileStat;
  FILE *fp;

  CHECK_ERR(pConnection == 0, "Must Connect First !");

  arg2 = strtok(NULL, " ");

  if ( arg2 == NULL ) {
    printf("ERR: Invalid Use: write [file] [id]\n");
    return;
  }
  
  fileName = strdup(arg2);  
  arg3 = strtok(NULL, " ");
  
  if (arg3 == NULL) {
    printf("ERR: Invalid Use: write [file] [id]\n");
    return;
  }
  
  objectID = strdup(arg3);


  fp = fopen(fileName, "r");
  
  if (fp == NULL) {
    printf("Open file %s failed\n", fileName);
    return;
  }

  if (stat(fileName, &fileStat) != 0) {
    printf("ERR: Failed to stat file\n");
    return;
  }

  dataSize = fileStat.st_size;
  objectData = (MSCPUChar8)malloc(dataSize);
  if (!objectData) { printf("ERR: Malloc failed\n"); return; }

  fread(objectData, 1, dataSize, fp);
  fclose(fp);

  printf("Percentage written:     ");

  rv = MSCWriteObject(pConnection, objectID, 
		      0, objectData, dataSize,
		      (LPRWEventCallback) myCallback,
		      NULL);

  printf("\n");
  CHECK_ERR(rv != MSC_SUCCESS, "WriteObject Failed !");
  
  printf("Write Object Successful\n");
  return;
}

void doCrypt() {

  MSCLong32 rv;
  MSCString arg2;
  MSCString arg3;
  MSCULong32 keyNum;
  MSCString hexString;
  MSCUChar8 inCryptData[4096];
  MSCUChar8 outCryptData[4096];
  MSCULong32 inDataSize;
  MSCULong32 outDataSize;
  MSCCryptInit cryptInit;
  MSCKeyInfo keyInfo;
  MSCULong32 keySize;
  //MSCUChar8 cipherMode;
  //MSCUChar8 cipherDir;
  int i;

  CHECK_ERR(pConnection == 0, "Must Connect First !");

  arg2 = strtok(NULL, " ");

  if ( arg2 == NULL ) {
    printf("ERR: Invalid Use: crypt [keynum] [hexstring]\n");
    return;
  }
  
  keyNum = atoi(strdup(arg2));  
  arg3 = strtok(NULL, " ");
  
  if (arg3 == NULL) {
    printf("ERR: Invalid Use: crypt [keynum] [hexstring]\n");
    return;
  }
  
  hexString = strdup(arg3);

  inDataSize = hexToBin(hexString, inCryptData);

  rv = MSCGetKeyAttributes( pConnection, keyNum, &keyInfo );
  CHECK_ERR(rv != MSC_SUCCESS, "GetKeyAttribtues Failed !");

  keySize = keyInfo.keySize / 8;

  cryptInit.keyNum = keyNum;
  cryptInit.cipherMode = MSC_MODE_RSA_NOPAD;
  cryptInit.cipherDirection = MSC_DIR_SIGN;
  cryptInit.optParams = 0;
  cryptInit.optParamsSize = 0;

  padData(inCryptData, inDataSize, keySize);
  inDataSize = outDataSize = keySize;

  binToHex(inCryptData, inDataSize, outCryptData);

  rv = MSCComputeCrypt( pConnection, &cryptInit,
			inCryptData, inDataSize, 
			outCryptData, &outDataSize);


  CHECK_ERR(rv != MSC_SUCCESS, "ComputeCrypt Failed !");

  printf("Result    : ");
 
  for (i=0; i < outDataSize; i++) {
    printf("%02X", outCryptData[i]);
  } printf("\n");
  
  printf("ComputeCrypt Successful\n");
  return;
}

void doListKeys() {

  MSCLong32 rv;
  MSCKeyInfo keyInfo;

  CHECK_ERR(pConnection == 0, "Must Connect First !");

  rv = MSCListKeys( pConnection, MSC_SEQUENCE_RESET, &keyInfo );
  
  printf("%20s %12s %6s %8s%8s%8s\n", "Key Type", "Key Num", 
	 "SIZE", "READ", "WRITE", "USE");
  printf("   -----------------  -----------   -----   ------  ------  ------\n");

    switch(keyInfo.keyType) {
        case MSC_KEY_RSA_PUBLIC:
            printf("%20s", "RSA PUBLIC");
            break;
        case MSC_KEY_RSA_PRIVATE:
            printf("%20s", "RSA PRIVATE");
            break;
        case MSC_KEY_RSA_PRIVATE_CRT:
            printf("%20s", "RSA PRIVATE CRT");
            break;    
        case MSC_KEY_DES:
            printf("%20s", "DES");
            break;
        case MSC_KEY_3DES:
            printf("%20s", "Triple DES");
            break;
    }

  if ( rv == MSC_SUCCESS ) {
    printf("%13d   %04d  ", keyInfo.keyNum, 
	   keyInfo.keySize);
	   doPrintAUT(keyInfo.keyACL.readPermission), 
	   doPrintAUT(keyInfo.keyACL.writePermission),
	   doPrintAUT(keyInfo.keyACL.usePermission);
           printf("\n");
  } 
  
  do {
    rv = MSCListKeys( pConnection, MSC_SEQUENCE_NEXT, &keyInfo );
    if ( rv == MSC_SUCCESS ) {

    switch(keyInfo.keyType) {
        case MSC_KEY_RSA_PUBLIC:
            printf("%20s", "RSA PUBLIC");
            break;
        case MSC_KEY_RSA_PRIVATE:
            printf("%20s", "RSA PRIVATE");
            break;
        case MSC_KEY_RSA_PRIVATE_CRT:
            printf("%20s", "RSA PRIVATE CRT");
            break;    
        case MSC_KEY_DES:
            printf("%20s", "DES");
            break;
        case MSC_KEY_3DES:
            printf("%20s", "Triple DES");
            break;
    }
    
    printf("%13d   %04d  ", keyInfo.keyNum, 
	   keyInfo.keySize);
	   doPrintAUT(keyInfo.keyACL.readPermission), 
	   doPrintAUT(keyInfo.keyACL.writePermission),
	   doPrintAUT(keyInfo.keyACL.usePermission);
           printf("\n");

    } else {
      break;
    }
    
  } while (1);
    
}

void doExportKey() {

  MSCLong32 rv;
  MSCString arg2;
  MSCString arg3;
  //MSCString objectID;
  MSCUChar8 keyNumber;
  MSCString fileName;
  MSCUChar8 keyData[2000];
  MSCULong32 keySize;
  FILE *fp;

  CHECK_ERR(pConnection == 0, "Must Connect First !");

  arg2 = strtok(NULL, " ");

  if ( arg2 == NULL ) {
    printf("ERR: Invalid Use: exportkey [id] [file]\n");
    return;
  }
  
  keyNumber = atoi(strdup(arg2));
  
  arg3 = strtok(NULL, " ");
  
  if (arg3 == NULL) {
    printf("ERR: Invalid Use: exportkey [id] [file]\n");
    return;
  }
  
  keySize  = sizeof(keyData);
  fileName = strdup(arg3);

  rv = MSCExportKey(pConnection, keyNumber, 
                    keyData, &keySize, 0, 0);

  printf("\n");
  CHECK_ERR(rv != MSC_SUCCESS, "ExportKey Failed !");
  
  fp = fopen(fileName, "w+");

  if (fp == NULL) {
    printf("Open file %s failed\n", fileName);
    return;
  }

  fwrite(keyData, 1, keySize, fp);
  fclose(fp);

  printf("Export Key Successful\n");
  return;
}

void doImportKey() {

  MSCLong32 rv;
  MSCString arg2;
  MSCString arg3;
  //MSCString objectID;
  MSCUChar8 keyNumber;
  MSCKeyACL keyACL;
  MSCKeyPolicy keyPolicy;
  MSCString fileName;
  MSCPUChar8 objectData;
  MSCULong32 dataSize;
  struct stat fileStat;
  FILE *fp;

  CHECK_ERR(pConnection == 0, "Must Connect First !");

  arg2 = strtok(NULL, " ");

  if ( arg2 == NULL ) {
    printf("ERR: Invalid Use: importkey [id] [file]\n");
    return;
  }
  
  keyNumber = atoi(strdup(arg2));
  
  arg3 = strtok(NULL, " ");
  
  if (arg3 == NULL) {
    printf("ERR: Invalid Use: importkey [id] [file]\n");
    return;
  }
  
  fileName = strdup(arg3);

  fp = fopen(fileName, "rib");

  if (fp == NULL) {
    printf("Open file %s failed\n", fileName);
    return;
  }

  if (stat(fileName, &fileStat) != 0) {
    printf("ERR: Failed to stat file\n");
    return;
  }

  dataSize = fileStat.st_size;
  objectData = (MSCPUChar8)malloc(dataSize);
  if (!objectData) { printf("ERR: Malloc failed\n"); return; }

  fread(objectData, 1, dataSize, fp);
  fclose(fp);

  keyPolicy.cipherDirection = MSC_KEYPOLICY_DIR_SIGN | MSC_KEYPOLICY_DIR_DECRYPT;
  keyPolicy.cipherMode = MSC_KEYPOLICY_MODE_RSA_NOPAD;
  keyACL.readPermission = MSC_AUT_NONE;
  keyACL.writePermission = MSC_AUT_PIN_1;
  keyACL.usePermission = MSC_AUT_PIN_1;

  printf("import\n");

  rv = MSCImportKey(pConnection, keyNumber,
                    &keyACL, objectData, dataSize,
                    &keyPolicy, 0, 0);

  printf("\n");

  if (objectData)
    free(objectData);

  CHECK_ERR(rv != MSC_SUCCESS, "ImportKey Failed !");

  printf("Import Key Successful\n");
  return;
}

void doGenKeys() {
  MSC_RV rv;
  MSCUChar8 prvKeyNum, pubKeyNum;
  MSCGenKeyParams keyParams;
  pthread_t pthThread;
  int prv, pub;
  char myChoice[5];
  
  keyParams.algoType =  MSC_GEN_ALG_RSA_CRT;
  keyParams.keySize   = 1024;
  keyParams.privateKeyACL.readPermission = MSC_AUT_NONE;
  keyParams.privateKeyACL.writePermission = MSC_AUT_PIN_1;
  keyParams.privateKeyACL.usePermission = MSC_AUT_PIN_1;
  
  keyParams.publicKeyACL.readPermission = MSC_AUT_ALL;
  keyParams.publicKeyACL.writePermission = MSC_AUT_PIN_1;
  keyParams.publicKeyACL.usePermission = MSC_AUT_PIN_1;

  /* Ignore this
  keyParams.publicKeyPolicy
  keyParams.privateKeyPolicy
  */
  
  keyParams.keyGenOptions  = 0;
  keyParams.pOptParams     = 0;
  keyParams.optParamsSize  = 0;

  printf("***************** Key Generation Routine *****************\n");
  printf("Note: Keys will be generated with default permissions\n");
  printf("       using the RSA CRT algorithm for generation.\n");
  printf("**********************************************************\n");
  printf("Enter the private key number: ");
  scanf("%d", &prv);
  prvKeyNum = prv;

  printf("Enter the public key number: ");
  scanf("%d", &pub);
  pubKeyNum = pub;

  printf("Are you sure ? (Y/N): ");
  fflush(stdout); fflush(stdin);

  scanf("%s", myChoice);
  
  if (myChoice[0] != 'Y' && myChoice[0] != 'y') {
    printf("Key generation cancelled\n");
    return;
  }
  
  printf("Generating keys [");
  
  rv = pthread_create(&pthThread, NULL, (void *)&printStatStars, 0);
  
  fflush(stdout); fflush(stdin);

  rv = MSCGenerateKeys(pConnection, prv, pub, &keyParams);
  
  quitStat = 1;
  while (quitStat == 1);  
  
  CHECK_ERR(rv != MSC_SUCCESS, "GenerateKeys Failed !");
  printf("Success\n");

}

void doVersion() {
  printf("\n");
  printf("MuscleTool Version: %s\n", MUSCLETOOL_VERSION);
  printf("Copyright (C) 1998-2002 David Corcoran\n");
  printf("Copyright (C) 2003 Ludovic Rousseau\n");
  printf("\n");
}

void doFormat() {
  MSC_RV rv;
  MSCString arg2;
  MSCULong32 tokenNumber;
  MSCULong32 transportKeyChoice;
  MSCInitTokenParams initParams;
  MSCUChar8 userTransportKey[40];
  MSCChar8 userStringKey[80];

  MSCUChar8 newAdmin[25];
  MSCUChar8 userPIN[25];
  MSCUChar8 userUnblock[25];
  MSCULong32 getTries;
  MSCUChar8 numTries;

  //MSCULong32 transportKeySize;
  MSCULong32 userKeyLength;
  MSCULong32 memoryAlloc;
  MSCULong32 sureChoice;
  MSCPUChar8 adminKey; 
  MSCULong32 adminKeyLength; 
  pthread_t pthThread;    
  //int i, j;
  
  CHECK_ERR(tokenList == 0, "List Tokens First !");

  arg2 = strtok(NULL, " ");
  if (arg2 == NULL) {
    printf("ERR: Invalid Use: format [number]\n");
    return;
  }

  tokenNumber = strtoul(arg2, 0, 10);

  if (tokenNumber < 1 || tokenNumber > tokenSize ) {
    printf("ERR: Invalid choice made !\n");
    return;
  }

  if ( pConnection == 0 ) {
    pConnection = (MSCLPTokenConnection)malloc(sizeof(MSCTokenConnection));
  } else {
    printf("ERR: Must Release Other Connection !\n");
    return;
  }

  printf("Would you like to: \n");
  printf("	1. Use the default factory key: 2C15E526E93E8A19\n");
  printf(" 	2. Use the MUSCLE default key : 4D7573636C653030\n");
  printf("	3. Enter your own transport key\n");
  printf("\n");
  
  printf("Choose (1-3): ");
  scanf("%ld", &transportKeyChoice);
  
  if (transportKeyChoice < 1 || transportKeyChoice > 3) {
      printf("Invalid choice made\n");
      return;
  } 
  
  if (transportKeyChoice == 3) {
    printf("Enter your transport key below (2 chars per byte, uppercase)\n");
    printf("Example: 4D7573636C653030\n");
    printf("Enter key: ");
    scanf("%s", userStringKey);
    rv = textToBytes(userStringKey, userTransportKey, &userKeyLength);    
    if (rv != 0) {
      printf("Invalid transport key\n");
      return;
    }
  }

  memset(newAdmin, 0x00, sizeof(newAdmin));
  memset(userPIN, 0x00, sizeof(userPIN));
  memset(userUnblock, 0x00, sizeof(userUnblock));

  printf("\n");
  printf("How much object memory would you like to allocate ?\n");
  printf("Example: 7096 (7k) : ");
  scanf("%ld", &memoryAlloc);


  printf("Please enter a new auth pin value (pin 0) : ");
  scanf("%s", newAdmin);

  printf("Please enter your user pin value (pin 1)  : ");
  scanf("%s", userPIN);

  printf("Please enter your user pin unblock value  : ");
  scanf("%s", userUnblock);


  printf("Please enter your number of pin tries     : ");
  scanf("%ld", &getTries);

  numTries = (MSCUChar8)getTries;


  printf("\n");
  printf("********************* WARNING ! *********************\n");
  printf("You are about to destroy all data on this token.\n");
  printf("*****************************************************\n");
  printf("\n");

  
  if (transportKeyChoice == 1) {
    adminKey = defaultCFlexKey;
    adminKeyLength = 8;
  } else if (transportKeyChoice == 2) {
    adminKey = muscleCFlexKey;  
    adminKeyLength = strlen(adminKey);
  } else if (transportKeyChoice == 3) {
    adminKey = userTransportKey;
    adminKeyLength = userKeyLength;
  }

  adminKey[8] = 0x00; /* Add the NULL so it prints OK */

  printf("Current admin pin      : %s\n", adminKey);
  printf("New admin pin          : %s\n", newAdmin);
  printf("New user pin           : %s\n", userPIN);
  printf("New user unblock pin   : %s\n", userUnblock);
  printf("Number of tries        : %d\n", numTries);
  printf("Object memory          : %ld\n", memoryAlloc);
  printf("\n");

  printf("Are you sure you want to continue ? (1-YES, 2-NO): ");
  
  scanf("%ld", &sureChoice);
  
  if (sureChoice != 1) {
    printf("User aborted format - exiting\n");
    return;
  }
  
  printf("\n");

  rv = MSCEstablishConnection( &tokenList[tokenNumber-1], 
			       MSC_SHARE_DIRECT, 0, 
                               0, pConnection );
  if ( rv != MSC_SUCCESS ) {
    free(pConnection); pConnection = 0;
    printf("ERR: EstablishConnection Failed !");
    return;
  }

  printf("Formatting token [");
  
  rv = pthread_create(&pthThread, NULL, (void *)&printStatStars, 0);

  initParams.transportBehavior = MSC_INIT_USE_KEY;  
  memcpy(initParams.transportKey, adminKey, adminKeyLength);
  initParams.transportKeyLen   = adminKeyLength;

  memcpy(initParams.newTransportKey, newAdmin, strlen(newAdmin));
  initParams.newTransportKeyLen   = strlen(newAdmin);

  memcpy(initParams.defaultCHV, userPIN, strlen(userPIN));
  initParams.defaultCHVLen = strlen(userPIN);

  memcpy(initParams.defaultCHVUnblock, userUnblock, strlen(userUnblock));
  initParams.defaultCHVUnblockSize = strlen(userUnblock);


  initParams.defaultCHVTries        = (MSCUChar8)getTries;
  initParams.defaultCHVUnblockTries = 2;
  initParams.objectMemory           = memoryAlloc;

  rv = MSCWriteFramework(pConnection, &initParams);
  MSCReleaseConnection( pConnection, MSC_RESET_TOKEN );

  free(pConnection); pConnection = 0;


  quitStat = 1;
  while (quitStat == 1);

  if (rv != MSC_SUCCESS) {
    printf("Failed (%s)\n", msc_error(rv));
    return;
  } else {
    printf("Success\n");
  }
  
}

int main(int argc, char **argv) {

  //MSCLong32 rv;
  MSCChar8 cmd[1024];
  MSCString arg1;
  MSCUChar8 firstFlag;
  MSCULong32 cmdLen;

  printf("\nMuscleCard Shell - type help for help\n\n");

  firstFlag = 0;

  do {
    fflush(stdin); fflush(stdout);

    if (connectToken) {
      printf("muscle [%s] > ", connectToken);
    } else {
      printf("muscle > ");
    }

    if (fgets(cmd, sizeof(cmd), stdin) == NULL) { continue; }
    if (cmd[0] == '\n') { continue; }
    if (cmd[0] == ' ') { continue; }

    cmdLen = strlen(cmd);
    cmd[cmdLen-1] = 0; /* NULL the EOL */

    arg1 = strtok(cmd, " ");
    
    if (strcmp(arg1, "help") == 0) {
      doHelp();
    } else if (strcmp(arg1, "tokens") == 0) {
      doListTokens();
    } else if (strcmp(arg1, "connect") == 0) {
      doConnect();
    } else if (strcmp(arg1, "release") == 0) {
      doRelease();
    } else if (strcmp(arg1, "status") == 0) {
      doStatus();
    } else if (strcmp(arg1, "list") == 0) {
      doList();
    } else if (strcmp(arg1, "resume") == 0) {
      doResume();
    } else if (strcmp(arg1, "challenge") == 0) {
      doChallenge();
    } else if (strcmp(arg1, "logout") == 0) {
      doLogout();
    } else if (strcmp(arg1, "verify") == 0) {
      doVerify();
    } else if (strcmp(arg1, "makepin") == 0) {
      doMakePIN();
    } else if (strcmp(arg1, "listpins") == 0) {
      doListPIN();
    } else if (strcmp(arg1, "changepin") == 0) {
      doChangePIN();
    } else if (strcmp(arg1, "create") == 0) {
      doCreate();
    } else if (strcmp(arg1, "delete") == 0) {
      doDelete();
    } else if (strcmp(arg1, "read") == 0) {
      doRead();
    } else if (strcmp(arg1, "write") == 0) {
      doWrite();
    } else if (strcmp(arg1, "crypt") == 0) {
      doCrypt();
    } else if (strcmp(arg1, "listkeys") == 0) {
      doListKeys();
    } else if (strcmp(arg1, "exportkey") == 0) {
      doExportKey();
    } else if (strcmp(arg1, "importkey") == 0) {
      doImportKey();
    } else if (strcmp(arg1, "genkeys") == 0) {
      doGenKeys();
    } else if (strcmp(arg1, "format") == 0) {
      doFormat();
    } else if (strcmp(arg1, "version") == 0) {
      doVersion();
    } else if (strcmp(arg1, "exit") == 0) {
      printf("Bye\n");
    } else
		printf("Unknown command. Try help\n");


  } while (strcmp(cmd, "exit") != 0);


  if (pConnection) { doRelease(); };

  return 0;
 }

