/***************************************************************************
                          asyncsocket.cpp  -  description
                             -------------------
    begin                : Tue May 29 2001
    copyright            : (C) 2001 by
    email                : maksik@gmx.co.uk
 ***************************************************************************/
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <sys/time.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/poll.h>
#include <unistd.h>
#include <fcntl.h>
#include <typeinfo>


#include "mutella.h"
#include "asyncsocket.h"

#ifndef INADDR_NONE
#define INADDR_NONE -1
#endif

bool is_fd_ok(SOCKET fd);

MAsyncSocket::MAsyncSocket()
{
	m_hSocket = INVALID_SOCKET;
	m_bConnecting = false;
	m_bListening = false;
	m_bNewSocket = true;
	m_nSelectFlags = 0;
	m_bSelectActive = false;
}

MAsyncSocket::~MAsyncSocket()
{
	if (m_hSocket != INVALID_SOCKET)
		Close();
}

BOOL MAsyncSocket::Create(UINT nSocketPort /*= 0*/, int nSocketType /*=SOCK_STREAM*/,
		long lEvent /*= FD_READ | FD_WRITE | FD_OOB | FD_ACCEPT | FD_CONNECT | FD_CLOSE*/,
		LPCTSTR lpszSocketAddress /*=NULL*/)
{
#ifdef _DEBUG
	if (lEvent & FD_WRITE)
		TRACE2("WARNING: creating a socket with FD_WRITE on for ", typeid(*this).name());
#endif
	if (Socket(nSocketType, lEvent))
	{
		if (Bind(nSocketPort,lpszSocketAddress))
			return TRUE;
		int nResult = GetLastError();
		Close();
		//WSASetLastError(nResult);
		errno = nResult;
	}
	return FALSE;
}

BOOL MAsyncSocket::Attach(SOCKET hSocket,
		long lEvent /*= FD_READ | FD_WRITE | FD_OOB | FD_ACCEPT | FD_CONNECT | FD_CLOSE*/)
{
#ifdef _DEBUG
	if (lEvent & FD_WRITE)
		TRACE2("WARNING: attaching a socket with FD_WRITE on for ", typeid(*this).name());
#endif
	ASSERT(hSocket != INVALID_SOCKET);

	m_hSocket = hSocket;
	MAsyncSocket::AttachHandle(hSocket, this);
	
	m_bSelectActive = true;
	return AsyncSelect(lEvent);
}
	
SOCKET MAsyncSocket::Detach()
{
	SOCKET hSocket = m_hSocket;
	if (AsyncSelect(0))
	{
		MAsyncSocket::KillSocket(hSocket, this);
		m_hSocket = INVALID_SOCKET;
		return hSocket;
	}
	return INVALID_SOCKET;
}

BOOL MAsyncSocket::GetPeerName(CString& rPeerAddress, UINT& rPeerPort)
{
	SOCKADDR_IN sockAddr;
	memset(&sockAddr, 0, sizeof(sockAddr));
	MLock lock(g_LibcMutex);

	socklen_t nSockAddrLen = sizeof(sockAddr);
	BOOL bResult = GetPeerName((SOCKADDR*)&sockAddr, &nSockAddrLen);
	if (bResult)
	{
		rPeerPort = ntohs(sockAddr.sin_port);
		rPeerAddress = ::inet_ntoa(sockAddr.sin_addr);
	}
	return bResult;
}

BOOL MAsyncSocket::GetPeerName(SOCKADDR* lpSockAddr, socklen_t* lpSockAddrLen)
{
	return 0==getpeername(m_hSocket, lpSockAddr, lpSockAddrLen);
}

BOOL MAsyncSocket::GetSockName(CString& rSocketAddress, UINT& rSocketPort)
{
	SOCKADDR_IN sockAddr;
	memset(&sockAddr, 0, sizeof(sockAddr));
	MLock lock(g_LibcMutex);

	socklen_t nSockAddrLen = sizeof(sockAddr);
	BOOL bResult = GetSockName((SOCKADDR*)&sockAddr, &nSockAddrLen);
	if (bResult)
	{
		rSocketPort = ntohs(sockAddr.sin_port);
		rSocketAddress = ::inet_ntoa(sockAddr.sin_addr);
	}
	return bResult;
}

BOOL MAsyncSocket::GetSockName(SOCKADDR* lpSockAddr, socklen_t* lpSockAddrLen)
{
	return 0 == getsockname(m_hSocket, lpSockAddr, lpSockAddrLen);
}

BOOL MAsyncSocket::Accept(MAsyncSocket& rConnectedSocket,
		SOCKADDR* lpSockAddr /*=NULL*/, socklen_t* lpSockAddrLen /*=NULL*/)
{
	ASSERT(rConnectedSocket.m_hSocket == INVALID_SOCKET);
	//ASSERT(MAsyncSocket::FromHandle(INVALID_SOCKET) == NULL);

	MAsyncSocket::AttachHandle(INVALID_SOCKET, &rConnectedSocket);

	SOCKET hTemp = accept(m_hSocket, lpSockAddr, lpSockAddrLen);

	if (hTemp == INVALID_SOCKET)
	{
		DWORD dwProblem = GetLastError();
		MAsyncSocket::DetachHandle(rConnectedSocket.m_hSocket);
		rConnectedSocket.m_hSocket = INVALID_SOCKET;
		//SetLastError(dwProblem);
	}
	else /*if (MAsyncSocket::FromHandle(INVALID_SOCKET) != NULL)*/
	{
		rConnectedSocket.m_hSocket = hTemp;
		MAsyncSocket::DetachHandle(INVALID_SOCKET);
		MAsyncSocket::AttachHandle(hTemp, &rConnectedSocket);
		rConnectedSocket.m_bSelectActive = true;
		
		int val = 1;
		setsockopt(hTemp,SOL_SOCKET,SO_REUSEADDR,(char*)&val,sizeof(val));
		fcntl(hTemp,F_SETFL,O_NONBLOCK);
	}

	return (hTemp != INVALID_SOCKET);
}

BOOL MAsyncSocket::Bind(UINT nSocketPort, LPCTSTR lpszSocketAddress /*=NULL*/)
{
	USES_CONVERSION;

	SOCKADDR_IN sockAddr;
	memset(&sockAddr,0,sizeof(sockAddr));

	LPSTR lpszAscii = T2A((LPTSTR)lpszSocketAddress);
	sockAddr.sin_family = AF_INET;

	if (lpszAscii == NULL)
		sockAddr.sin_addr.s_addr = htonl(INADDR_ANY);
	else
	{
		MLock lock(g_LibcMutex);
		DWORD lResult = ::inet_addr(lpszAscii);
		if (lResult == INADDR_NONE)
		{
			//WSASetLastError(WSAEINVAL);
			return FALSE;
		}
		sockAddr.sin_addr.s_addr = lResult;
	}

	sockAddr.sin_port = htons((u_short)nSocketPort);

	return Bind((SOCKADDR*)&sockAddr, sizeof(sockAddr));
}

BOOL MAsyncSocket::Bind(const SOCKADDR* lpSockAddr, socklen_t nSockAddrLen)
{
	return 0 == bind(m_hSocket, lpSockAddr, nSockAddrLen);
}

void MAsyncSocket::Close()
{
	m_bConnecting = false;
	m_bListening = false;
	if (m_hSocket != INVALID_SOCKET)
	{
		//VERIFY(SOCKET_ERROR != closesocket(m_hSocket));
		if (0 != close(m_hSocket))
		{
			TRACE2("WARNING: closing closed socket ", m_hSocket);
		}
		MAsyncSocket::KillSocket(m_hSocket, this);
		m_hSocket = INVALID_SOCKET;
	}
}

BOOL MAsyncSocket::Connect(LPCTSTR lpszHostAddress, UINT nHostPort)
{
	USES_CONVERSION;

	ASSERT(lpszHostAddress != NULL);
	
	MLock lock(g_LibcMutex);

	SOCKADDR_IN sockAddr;
	memset(&sockAddr,0,sizeof(sockAddr));

	LPSTR lpszAscii = T2A((LPTSTR)lpszHostAddress);
	sockAddr.sin_family = AF_INET;
	sockAddr.sin_addr.s_addr = ::inet_addr(lpszAscii);

	if (sockAddr.sin_addr.s_addr == INADDR_NONE)
	{
		LPHOSTENT lphost;
		lphost = ::gethostbyname(lpszAscii);
		if (lphost != NULL)
			sockAddr.sin_addr.s_addr = ((LPIN_ADDR)lphost->h_addr)->s_addr;
		else
		{
			//WSASetLastError(WSAEINVAL);
			return FALSE;
		}
	}

	sockAddr.sin_port = htons((u_short)nHostPort);

	return Connect((SOCKADDR*)&sockAddr, sizeof(sockAddr));
}

BOOL MAsyncSocket::Connect(const SOCKADDR* lpSockAddr, socklen_t nSockAddrLen)
{
    SOCKADDR sockAddr;
	memcpy(&sockAddr,lpSockAddr,nSockAddrLen);
	int retval;
    bool bRes = (0 == (retval = ::connect(m_hSocket, &sockAddr, nSockAddrLen)));
	m_bConnecting = bRes ? true : (errno == EINPROGRESS);
	m_bSelectActive = m_bConnecting;
	//printf("connect %s, errno=%d\n", m_bConnecting?"succeeded":"failed", errno);
	return bRes;
}

//BOOL MAsyncSocket::IOCtl(long lCommand, DWORD* lpArgument)
//{
//	return (SOCKET_ERROR != ioctl(m_hSocket, lCommand, lpArgument));
//}

BOOL MAsyncSocket::Listen(int nConnectionBacklog /*=5*/)
{
	m_bListening = (0 == ::listen(m_hSocket, nConnectionBacklog));
	m_bSelectActive = m_bListening;
	return m_bListening;
}

int MAsyncSocket::Receive(void* lpBuf, int nBufLen, int nFlags /*=0*/)
{
	return ::recv(m_hSocket, (LPSTR)lpBuf, nBufLen, nFlags);
}

BOOL MAsyncSocket::ShutDown(int nHow /*=sends*/)
{
	return 0 == ::shutdown(m_hSocket,nHow);
}

int MAsyncSocket::Send(const void* lpBuf, int nBufLen, int nFlags /*=0*/)
{
	if (is_fd_ok(m_hSocket))
		return ::send(m_hSocket, (LPSTR)lpBuf, nBufLen, nFlags);
	//printf("MAsyncSocket::Send: trying to use invalid socket descriptor\n");
	return 0;
}

//static MMutex mutexASelect(true);

BOOL MAsyncSocket::AsyncSelect(long lEvent /*= FD_READ | FD_WRITE | FD_OOB | FD_ACCEPT | FD_CONNECT | FD_CLOSE*/)
{
	//mutexASelect.lock();
	m_nSelectFlags = lEvent;
	//mutexASelect.unlock();
	return true;
	//return MAsyncSocket::AsyncSelect(m_hSocket, lEvent);
}

BOOL MAsyncSocket::ModifySelectFlags(long lEventAdd, long lEventRemove)
{
	m_nSelectFlags &= ~lEventRemove;
	m_nSelectFlags |= lEventAdd;
	return true;
}

void MAsyncSocket::OnReceive(int nErrorCode)
{
}

void MAsyncSocket::OnSend(int nErrorCode)
{
}

void MAsyncSocket::OnOutOfBandData(int nErrorCode)
{
}

void MAsyncSocket::OnAccept(int nErrorCode)
{
}

void MAsyncSocket::OnConnect(int nErrorCode)
{
}

void MAsyncSocket::OnClose(int nErrorCode)
{
}

BOOL MAsyncSocket::Socket(int nSocketType /*=SOCK_STREAM*/,
		long lEvent /*= FD_READ | FD_WRITE | FD_OOB | FD_ACCEPT | FD_CONNECT | FD_CLOSE*/,
		int nProtocolType /*=0*/,
		int nAddressFormat /*=PF_INET*/)
{
#ifdef _DEBUG
	if (lEvent & FD_WRITE)
		TRACE2("WARNING: calling 'socket' with FD_WRITE on for ",typeid(*this).name());
#endif
	ASSERT(m_hSocket == INVALID_SOCKET);

	m_hSocket = socket(nAddressFormat,nSocketType,nProtocolType);
	if (m_hSocket != INVALID_SOCKET)
	{
		MAsyncSocket::AttachHandle(m_hSocket, this);
		
		int val = 1;
		setsockopt(m_hSocket,SOL_SOCKET,SO_REUSEADDR,(char*)&val,sizeof(val));
		fcntl(m_hSocket,F_SETFL,O_NONBLOCK);

		return AsyncSelect(lEvent);
	}
	return FALSE;

}

// statics
int MAsyncSocket::GetLastError()
{
	return errno;
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// static socket maps and message scheduling
//

typedef std::map<SOCKET,MAsyncSocket*> MapSocket_MAS;

static MapSocket_MAS socket_map;
static MMutex mapMutex;

MAsyncSocket* MAsyncSocket::FromHandle(SOCKET hSocket)
{
	return MAsyncSocket::LookupHandle(hSocket);
}

void MAsyncSocket::KillSocket(SOCKET hSocket, MAsyncSocket* pSocket)
{
	//MAsyncSocket::AsyncSelect(hSocket, 0); //deselect
	ASSERT(hSocket == INVALID_SOCKET || MAsyncSocket::LookupHandle(hSocket) == pSocket);
	MAsyncSocket::DetachHandle(hSocket);
}

void MAsyncSocket::AttachHandle(SOCKET hSocket, MAsyncSocket* pSocket)
{
	mapMutex.lock();
	socket_map[hSocket] = pSocket;
	mapMutex.unlock();
	ASSERT(MAsyncSocket::LookupHandle(hSocket) == pSocket);
}

void MAsyncSocket::DetachHandle(SOCKET hSocket)
{
	MLock lock(mapMutex);
	socket_map.erase(hSocket);
}

MAsyncSocket* MAsyncSocket::LookupHandle(SOCKET hSocket)
{
	MLock lock(mapMutex);
	MapSocket_MAS::iterator it = socket_map.find(hSocket);
	if (it != socket_map.end())
		return (*it).second;
	return NULL;
}

#ifndef _DEBUG

#define CALL_WITH_BLOCKING_CHECK(_object, _function)\
	_object->_function;

#else //_DEBUG

#define CALL_WITH_BLOCKING_CHECK(_object, _function)\
{\
	clock_t _TimeBefore = clock();\
	int _nTimeBefore = xtime();\
	CString _sClsName = typeid(*_object).name();\
	_object->_function;\
	double _dt = ((double)(clock() - _TimeBefore))/CLOCKS_PER_SEC;\
	if (_dt>0.2)\
		printf("blocking in %s::%s detected (%g ms)\n", _sClsName.c_str(), #_function, _dt*1000);\
	else if (_nTimeBefore + 1 < xtime())\
		printf("blocking in %s::%s detected with xtime\n", _sClsName.c_str(), #_function);\
}

#endif //_DEBUG

void MAsyncSocket::Heartbeat(int nSleep)
{
	// called by the "main event loop" thread
	int retval;
	int nLastError;
	int nSockFlags;
	int nFDs;
	long lEvent;
	MAsyncSocket* pSocket;
	MapSocket_MAS::iterator ih;
	struct pollfd * aPollFDs;
	struct pollfd * pPFD;
	//
	mapMutex.lock();
	// first fill pollfds
	nFDs = socket_map.size();
	if (!nFDs)
	{
		// the map is empty
		mapMutex.unlock();
		usleep(1000*nSleep);
		return;
	}
	aPollFDs = new pollfd[nFDs];
	pPFD = aPollFDs;
	//
	for (ih = socket_map.begin(); ih!=socket_map.end(); ++ih )
	{
		pSocket = (*ih).second;
		ASSERT(pSocket);
		if (pSocket->m_bSelectActive)
		{
			pPFD->fd = (*ih).first;
			pPFD->events = 0;
			pPFD->revents = 0;
		
			ASSERT(pSocket->m_hSocket == pPFD->fd);
			lEvent = pSocket->m_nSelectFlags;
			if ( lEvent & (FD_READ) )
				pPFD->events |= POLLIN;
			if ( lEvent & (FD_OOB) )
				pPFD->events |= POLLPRI;	
			if ( lEvent & (FD_WRITE) )
				pPFD->events |= POLLOUT;
			// connecting
			if ( pSocket->m_bConnecting && (lEvent & (FD_CONNECT)) )
				pPFD->events |= POLLOUT;
			// listening
			if ( pSocket->m_bListening && (lEvent & (FD_ACCEPT)) )
				pPFD->events |= POLLIN;
			++pPFD;
		}
		else
			nFDs--;
	}
	mapMutex.unlock();
	
	retval = poll(aPollFDs, nFDs, nSleep);
	
    nLastError = MAsyncSocket::GetLastError();
    if (retval>0)
    {
    	for ( pPFD = aPollFDs; retval>0 && nFDs>0; ++pPFD, --nFDs)
    	{
    		if (pPFD->revents)
    		{
    			--retval; // to save looping
    			pSocket = MAsyncSocket::FromHandle(pPFD->fd);
    			if (pSocket)
    			{
    				ASSERT(pSocket->m_hSocket==pPFD->fd);
    				// ensure that the socket is non-blocking
    				if (pSocket->m_bNewSocket)
    				{
    					nSockFlags = fcntl(pPFD->fd,F_GETFL,0);
    					fcntl(pPFD->fd,F_SETFL,nSockFlags|O_NONBLOCK);	
    					pSocket->m_bNewSocket = false;
    				}
    				// call appropriate notifiers
    				if ( pPFD->revents & (POLLERR|POLLHUP|POLLNVAL) )
    				{
    					// error condition
    					if ( (pSocket->m_nSelectFlags & FD_CLOSE) && (pPFD->revents & POLLHUP) )
    					{
    						//printf("poll-hangup\n");
    						//pSocket->OnClose(nLastError);
    						CALL_WITH_BLOCKING_CHECK(pSocket,OnClose(nLastError))
    						if ( pSocket != MAsyncSocket::FromHandle(pPFD->fd) )
    							continue;
    					}
    					pSocket->Close();
    					break;
    				}
    				if ( pPFD->revents & POLLOUT )
    				{
						//TRACE2(">>> POLLOUT event for ", typeid(*pSocket).name());
    					if (pSocket->m_bConnecting)
    					{
    						pSocket->m_bConnecting = false;
    						//pSocket->OnConnect(nLastError);
    						CALL_WITH_BLOCKING_CHECK(pSocket,OnConnect(nLastError))
    						if ( pSocket != MAsyncSocket::FromHandle(pPFD->fd) )
    							continue;
    					}
    					else
    					{
							//TRACE2(">>> Calling OnSend() for ", typeid(*pSocket).name());
    						//pSocket->OnSend(nLastError);
    						CALL_WITH_BLOCKING_CHECK(pSocket,OnSend(nLastError))
    						if ( pSocket != MAsyncSocket::FromHandle(pPFD->fd) )
    							continue;
    					}
    				}
    				if ( pPFD->revents & POLLPRI )
    				{
    					//pSocket->OnOutOfBandData(nLastError);
    					CALL_WITH_BLOCKING_CHECK(pSocket,OnOutOfBandData(nLastError))
    					TRACE("OnOutOfBandData occured");
    					if ( pSocket != MAsyncSocket::FromHandle(pPFD->fd) )
    						continue;
    				}
    				if ( pPFD->revents & POLLIN )
    				{
    					if (pSocket->m_bListening)
    					{
    						//pSocket->OnAccept(nLastError);
    						CALL_WITH_BLOCKING_CHECK(pSocket,OnAccept(nLastError))
    						if ( pSocket != MAsyncSocket::FromHandle(pPFD->fd) )
    							continue;
    					}
    					else
    					{
    						//pSocket->OnReceive(nLastError);
    						CALL_WITH_BLOCKING_CHECK(pSocket,OnReceive(nLastError))
    						if ( pSocket != MAsyncSocket::FromHandle(pPFD->fd) )
    							continue;
    					}
    				}
    			}
    			else
    			{
    				// TODO: proper analysis why it happens, close FD if required, etc
    				TRACE("AsyncSocket dissapeared while in poll");
    				::close(pPFD->fd);
    			}
    		}
    	}
    }
#ifdef _DEBUG
    else if (retval<0)
    {
		//printf("poll error: retwal=%d, errno=%d serving %d sockets\n", retval, errno, nFDs);
    }
#endif //_DEBUG

    // cleanup
    delete [] aPollFDs;
}

bool is_fd_ok(SOCKET fd)
{
	int nError = 0;
	socklen_t nSize = sizeof(nError);
	int nReturn = getsockopt(fd, SOL_SOCKET,SO_ERROR,(char*)&nError,&nSize);
	ASSERT(nReturn || nSize == sizeof(int));
	if (nError && nError!=EWOULDBLOCK && nError!=EAGAIN && nError!=EINPROGRESS)
	{
		//printf("possibly bad socket %d, error= %d\n",fd,nError);
		return false;
	}
	return 0 == nReturn;
}

