/***************************************************************************
                          gnudirector.cpp  -  description
                             -------------------
    begin                : Tue May 29 2001
    copyright            : (C) 2001 by
    email                : maksik@gmx.co.uk
 ***************************************************************************/

// the original version of this file was taken from Gnucleus (http://gnucleus.sourceforge.net)

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "mutella.h"
#include "controller.h"

#include "preferences.h"
#include "gnuhash.h"
#include "gnusock.h"
#include "gnunode.h"
#include "gnuupload.h"
#include "gnudownload.h"
#include "gnudirector.h"
#include "gnucache.h"
#include "gnushare.h"
#include "gnusearch.h"
#include "event.h"

static bool MatchIP(IP ipTest, ExIP &ipCompare)
{
	// -1 is a wildcard

	if(ipTest._ip.a == ipCompare.a || ipCompare.a == -1)
		if(ipTest._ip.b == ipCompare.b || ipCompare.b == -1)
			if(ipTest._ip.c == ipCompare.c || ipCompare.c == -1)
				if(ipTest._ip.d == ipCompare.d || ipCompare.d == -1)
					return true;

	return false;
}

/////////////////////////////////////////////////////////////////////////////
// MGnuDirector

MGnuDirector::MGnuDirector(MController* pController) : m_listmutex(true) // resurcive
{
	m_pController = pController;
	m_pController->Attach(this);
	m_pPrefs = pController->GetPrefs();
	
	m_pCache = NULL;
	m_pShare = NULL;
	new MGnuCache(this);
	new MGnuShare(this);	

	CreateGuid(&m_ClientID);
	m_nSecsAboveLimitConn = 0;
	m_nReSearchCounter = 0;
	m_nReSearchTimer = 0;

	m_nPort = 0;
	
	m_nQGood = 0;
	m_nQRepeat = 0;
	m_nQError = 0;
	
	m_dwConnBytesRecv = 0;
	m_dwConnBytesSend = 0;
	m_dwConnRecvRate = 0;
	m_dwConnSendRate = 0;
	
	m_dDownloadRate = 0;
	m_dUploadRate = 0;
	
	m_nLastConnCount = 0;
}

MGnuDirector::~MGnuDirector()
{
	MLock lock(m_listmutex);
	//
	MGnuNode* pNode;
	while( m_NodeList.size() )
	{
		pNode = m_NodeList.back();
		m_NodeList.pop_back();
		delete pNode;
	}

	while( m_UploadList.size() )
	{
		delete m_UploadList.back();
		m_UploadList.pop_back();
	}

	while( m_DownloadList.size() )
	{
		delete m_DownloadList.back();
		m_DownloadList.pop_back();
	}
	
	m_PendingQueries.clear();
	while( m_SearchList.size() )
	{
		delete m_SearchList.back();
		m_SearchList.pop_back();
	}

	// TODO: pending searches
	
	if (m_pCache)
		delete m_pCache;
	if (m_pShare)
		delete m_pShare;

	m_pController->Detach(this);
}

void MGnuDirector::AttachCache(MGnuCache* pCache)
{
	ASSERT(pCache);
	ASSERT(m_pCache==NULL);
	m_pCache = pCache;
}

void MGnuDirector::DetachCache(MGnuCache* pCache)
{
	ASSERT(m_pCache==pCache);
	m_pCache = NULL;
}

void MGnuDirector::AttachShare(MGnuShare* pShare)
{
	ASSERT(pShare);
	ASSERT(m_pShare==NULL);
	m_pShare = pShare;
}

void MGnuDirector::DetachShare(MGnuShare* pShare)
{
	ASSERT(m_pShare==pShare);
	m_pShare = NULL;
}

bool MGnuDirector::IsOkForDirectConnect(const IP& host)
{
/*
192.168/16,
172.16/12,
10/8,
169.254/16,
0/8,
240/4
*/
	static std::vector<ExIP> privSubnets;
	static const char* Subnets[] = {
		"192.168.*.*:DENY",
		"172.16.*.*:DENY",
		"169.254.*.*:DENY",
		"127.*.*.*:DENY",
		"10.*.*.*:NENY",
		"0.*.*.*:DENY"};
	static bool bInit = true;
	if (bInit)
	{
		for (int i=0; i<sizeof(Subnets)/sizeof(char*); ++i)
		{
			privSubnets.push_back(StrtoExIP(Subnets[i]));
		}
		bInit = false;
	}
	std::vector<ExIP>::iterator itIP;
	bool allowIP = true;
	for (itIP = privSubnets.begin(); itIP != privSubnets.end(); itIP++)
	{
		if( (*itIP).mode == DENY && MatchIP(host, (*itIP)) )
			allowIP = false;

		if( (*itIP).mode == ALLOW && MatchIP(host, (*itIP)) )
			allowIP = true;
	}

	return allowIP;
}

bool MGnuDirector::CheckIP(const IP& ipTest)
{
	// Default is allow
	// NOTE: Denied IPs are always at the front of the list, allowed are at the end

	bool allowIP = true;

	// Check with screened nodes
	std::vector<ExIP>::iterator itIP;
	for (itIP = m_pPrefs->m_ScreenedNodes.begin(); itIP != m_pPrefs->m_ScreenedNodes.end(); itIP++)
	{
		if( (*itIP).mode == DENY && MatchIP(ipTest, (*itIP)) )
			allowIP = false;

		if( (*itIP).mode == ALLOW && MatchIP(ipTest, (*itIP)) )
			allowIP = true;
	}

	return allowIP;
}

bool MGnuDirector::NotLocal(Node TestNode)
{
	IP RemoteIP = TestNode.Host, LocalIP;
	LocalIP._ip.a = 127; LocalIP._ip.b = 0; LocalIP._ip.c = 0; LocalIP._ip.d = 1;

	UINT LocalPort  = m_nPort;//(m_pPrefs->m_ForcedPort) ? m_pPrefs->m_ForcedPort : m_pPrefs->m_LocalPort;
	UINT RemotePort = TestNode.Port;

	if(LocalPort == RemotePort)
	{
		if(RemoteIP.S_addr == LocalIP.S_addr)
			return false;

		LocalIP.S_addr = (m_pPrefs->m_ipForcedHost.S_addr) ? m_pPrefs->m_ipForcedHost.S_addr : m_pPrefs->m_ipLocalHost.S_addr;

		if(RemoteIP.S_addr == LocalIP.S_addr)
			return false;
	}

	return true;
}

bool MGnuDirector::BehindFirewall()
{
	ASSERT(m_pPrefs);
	return m_pPrefs->m_bBehindFirewall;
}

/////////////////////////////////////////////////////////////////////////////
// LISTEN CONTROL

bool MGnuDirector::StartListening()
{
	m_nPort = m_pPrefs->m_dwLocalPort;

	bool Success  = false;
	int	 Attempts = 0;

	while(!Success && Attempts < 10)
	{
		StopListening();

		if(Create(m_nPort, SOCK_STREAM, FD_ACCEPT))
		{
			if(Listen())
			{
				Success = true;
				m_pPrefs->m_dwLocalPort = m_nPort;
				return true;
			}
			else
				POST_ERROR(ES_IMPORTANT, CString("Failed to switch socket to listen mode; error = ") + DWrdtoStr(GetLastError()));
		}
		else
			POST_ERROR(ES_IMPORTANT, CString("Failed to create listening socket; error = ") + DWrdtoStr(GetLastError()));
	
		m_nPort += 1;//rand() % 99 + 0;
		POST_ERROR(ES_IMPORTANT, CString("trying port ") + DWrdtoStr(m_nPort));
		Attempts++;
	}

	return false;
}

void MGnuDirector::StopListening()
{
	Close();
}


/////////////////////////////////////////////////////////////////////////////
// CAsyncSocket OVERRIDES

void MGnuDirector::OnAccept(int nErrorCode)
{
	// maybe i should start off with a cgnusock.. ? no idea
	MAsyncSocket New;
	bool Safe = Accept(New);

	if(Safe)
	{
		//
		if (m_pPrefs->m_bBehindFirewall)
		{
			m_pPrefs->m_bBehindFirewall = false; //we can accept
			POST_MESSAGE(ES_UNIMPORTANT, "Accepted incoming connection -- resetting 'Firewall' flag");
		}
		//
		if(AllowIncomingConnections())
		{
			MGnuSock* Incomming = new MGnuSock(this);

			SOCKET Free = New.Detach();
			if (Free != INVALID_SOCKET)
			{
				Incomming->Attach(Free, FD_READ | FD_CLOSE);
				AddSock(Incomming);
				return;
			}
		}
		New.Close();
	}
	else
	{
		POST_ERROR(ES_IMPORTANT, CString("Node Accept Error: ") + DWrdtoStr(GetLastError()));
	}
		
	MAsyncSocket::OnAccept(nErrorCode);
}


/////////////////////////////////////////////////////////////////////////////
// TRAFFIC CONTROL

void MGnuDirector::Post_QueryHit(QueryComp* pQuery, const BYTE* pHit, DWORD ReplyLength, BYTE ReplyCount)
{
	int packetLength = sizeof(packet_QueryHit) + ReplyLength + sizeof(packet_QueryHitEx) + sizeof(GUID);
    char* pPacket = new char[packetLength];
	packet_QueryHit*  pQueryHit = (packet_QueryHit*)pPacket;
	packet_QueryHitEx* pQHD = (packet_QueryHitEx*) (pPacket+sizeof(packet_QueryHit)+ReplyLength);
	GUID* pClientID = (GUID*) (pPacket+sizeof(packet_QueryHit)+ReplyLength+sizeof(packet_QueryHitEx));
	//
	memcpy(pPacket+sizeof(packet_QueryHit), pHit, ReplyLength);
	// Build Query Packet
	pQueryHit->Header.Guid	    = pQuery->QueryGuid;
	pQueryHit->Header.Function  = 0x81;
	pQueryHit->Header.TTL	    = pQuery->nHops;
	//TRACE2("Posting QueryHit with TTL = ", (int) pQueryHit->Header.TTL);
	pQueryHit->Header.Hops	    = 0;
	pQueryHit->Header.le_Payload = packetLength - sizeof(packet_Header);

	pQueryHit->TotalHits		    = ReplyCount;
	pQueryHit->le_Port		    = (WORD) m_pPrefs->m_dwForcedPort ? m_pPrefs->m_dwForcedPort : GetLocalPort();
	pQueryHit->Host			    = m_pPrefs->m_ipForcedHost.S_addr ? m_pPrefs->m_ipForcedHost : m_pPrefs->m_ipLocalHost;
	pQueryHit->le_Speed		    = m_pPrefs->m_dwSpeedStat ? m_pPrefs->m_dwSpeedStat : m_pPrefs->m_dwSpeedDyn;

	// Add Query Hit Descriptor
	bool Busy = m_pPrefs->m_nMaxUploads >= 0 && CountUploading() >= m_pPrefs->m_nMaxUploads;
	
	memcpy(pQHD->VendorID, "MUTE", 4);
	pQHD->Length		= 2;
	pQHD->FlagPush		= 0;//true;
	pQHD->FlagBad		= 1;//true;
	pQHD->FlagBusy		= 1;//true;
	pQHD->FlagStable	= 1;//true;
	pQHD->FlagSpeed		= 1;//true;
	pQHD->FlagTrash		= 0;
	pQHD->Push			= BehindFirewall();
	pQHD->Bad			= 0;//false;
	pQHD->Busy			= 0;//false;//Busy;
	pQHD->Stable		= 1;//false;//true;//m_pDirector->HaveUploaded();
	pQHD->Speed			= 0;//false;//true;//m_pDirector->GetRealSpeed() ? true : false;
	pQHD->Trash			= 0;
	// Add our ID
	memcpy(pClientID, GetClientID(), sizeof(GUID));
	// post the packet
	QueuedHit qh;
	qh.size = packetLength;
	qh.pPacket = pPacket;
	qh.pNode = pQuery->Origin;
	m_hitqueuemutex.lock();
	m_hitQueue.push(qh);
	m_hitqueuemutex.unlock();
	/*{
		MLock lock(m_listmutex);
		for(int i = 0; i < m_NodeList.size(); i++)	
		{
			MGnuNode *p = m_NodeList[i];
			if(p == pQuery->Origin)
			{
				p->Send(pPacket, packetLength);
				delete [] pPacket;
				return;
			}
		}
	}*/
}

void MGnuDirector::Broadcast_Ping(packet_Ping* Ping, int length, MGnuNode* exception)
{
	MLock lock(m_listmutex);
	//
	for(int i = 0; i < m_NodeList.size(); i++)	
	{
		MGnuNode *p = m_NodeList[i];
		if(p != exception && p->IsConnected())
			p->Send((BYTE*) Ping, length);
	}
}

void MGnuDirector::Broadcast_Query(packet_Query* Query, int length, MGnuNode* exception)
{
	MLock lock(m_listmutex);
	//printf("broadcasting query %s\n",((char*)Query)+25);
	//
	for(int i = 0; i < m_NodeList.size(); i++)	
	{
		MGnuNode *p = m_NodeList[i];

		if(p != exception && p->IsConnected())
			p->Send((BYTE*) Query, length);
	}
}

GUID MGnuDirector::Broadcast_LocalQuery(BYTE* Packet, WORD Speed, int length)
{
	GUID Guid = GUID_NULL;
	CreateGuid(&Guid);
	if (Guid == GUID_NULL)
	{
		POST_ERROR(ES_UNIMPORTANT, "Failed to create a GUID to broadcast query");
		return Guid;
	}

	packet_Query* Query = (packet_Query*) Packet;
	
	Query->Header.Guid = Guid;
	Query->Header.Function = 0x80;
	Query->Header.Hops = 0;
	Query->Header.TTL = MAX_TTL_SELF;
	Query->Header.le_Payload = length - 23;
	Query->le_Speed = (WORD) Speed;

	m_TableLocal.Insert(&Guid, NULL);

	{
		MLock lock(m_listmutex);
    	// Send Query
    	for(int i = 0; i < m_NodeList.size(); i++)	
    	{
    		MGnuNode *p = m_NodeList[i];

    		if(p->IsConnected())
    			p->Send(Packet, length);
    	}
    }	
	return Guid;
}

void MGnuDirector::SendAllLocalQueries(MGnuNode* pNode)
{
	ASSERT(pNode);
	MLock lock(m_listmutex);
	for(int i = 0; i < m_SearchList.size(); i++)	
	{
		MGnuSearch *pS = m_SearchList[i];
		if (pS->m_Packet != NULL && pS->m_nPacketLength != 0)
			pNode->Send(pS->m_Packet, pS->m_nPacketLength);
    }
}

void MGnuDirector::Route_Pong(packet_Pong* Pong, int length, key_Value* key)
{
	MLock lock(m_listmutex);
	//
	for(int i = 0; i < m_NodeList.size(); i++)	
	{
		MGnuNode *p = m_NodeList[i];

		if(p == key->Origin && p->IsConnected())
			p->Send((BYTE*) Pong, length);
	}
}

void MGnuDirector::Route_QueryHit(packet_QueryHit* QueryHit, DWORD length, key_Value* key)
{
	MLock lock(m_listmutex);
	//
	DWORD BytestoRead = length;
	for(int i = 0; i < m_NodeList.size(); i++)	
	{
		MGnuNode *p = m_NodeList[i];

		if(p == key->Origin && p->IsConnected())
		{
			//while(0 < BytestoRead && BytestoRead <= length)
			//	BytestoRead -= p->Send((BYTE*) QueryHit, (30000 < BytestoRead) ? 30000 : BytestoRead);
			p->Send((BYTE*) QueryHit, length);
			return;
		}
	}
}

void MGnuDirector::Route_Push(packet_Push* Push, int length, key_Value* key)
{
	MLock lock(m_listmutex);
	//
	for(int i = 0; i < m_NodeList.size(); i++)	
	{
		MGnuNode *p = m_NodeList[i];
		if(p == key->Origin && p->IsConnected())
		{
			p->Send((BYTE*) Push, length);
			return;
		}
	}
}

bool MGnuDirector::Route_LocalPush(const Result& Download)
{
	GUID Guid = GUID_NULL;
	CreateGuid(&Guid);
	if (Guid == GUID_NULL)
	{
		POST_ERROR(ES_UNIMPORTANT, "Failed to create a GUID to route local push");
		return false;
	}

	// Create packet
	packet_Push Push;

	Push.Header.Guid		= Guid;
	Push.Header.Function	= 0x40;
	Push.Header.TTL			= Download.Distance;
	Push.Header.Hops		= 0;
	Push.Header.le_Payload	= 26;
	Push.ServerID			= Download.PushID;
	Push.le_Index			= Download.FileIndex;
	Push.Host				= m_pPrefs->m_ipForcedHost.S_addr ? m_pPrefs->m_ipForcedHost : m_pPrefs->m_ipLocalHost;
	Push.le_Port			= m_pPrefs->m_dwForcedPort ? m_pPrefs->m_dwForcedPort : m_nPort;

	{	
		MLock lock(m_listmutex);
		// Send Push
		for(int i = 0; i < m_NodeList.size(); i++)	
		{
			MGnuNode *p = m_NodeList[i];

			if(p == Download.Origin && p->IsConnected())
			{
				p->Send(&Push, 49);
				return true;
			}
		}
		return false;
	}
}

////////////////////////////////////////////////////////////////////////////
// Node control

void MGnuDirector::AddNode(CString Host, UINT Port, bool bForceV4)
{
	if(!bForceV4 && FindNode(Host, Port) != NULL)
		return;

	MGnuNode* Node = new MGnuNode(this, Host, Port, false, bForceV4);

	//Attempt to connect to node
	if(!Node->Create(0, SOCK_STREAM, FD_READ | FD_CONNECT | FD_CLOSE))
	{
		POST_ERROR(ES_IMPORTANT, CString("Node Create Error: ") + DWrdtoStr(Node->GetLastError()));
		return;
	}

	if( !Node->Connect(Host.c_str(), Port) )
	{
		if (Node->GetLastError() != EINPROGRESS)
		{
			delete Node;
			return;
		}
    }
	// Add node to list
	{
		MLock lock(m_listmutex);
		//
		Node->m_dwID = GenID();
		m_NodeList.push_back(Node);
	}
	//SendAllLocalQueries(Node);//this is no good because it gets sent before handshake
	NodeMessage(SOCK_UPDATE, 0);
}

void MGnuDirector::AddNode(CString url)
{
	url = StripWhite(url);
	UINT nPort = 6346;
	int nPortPos = url.find(':');	
	if (nPortPos < 0)
		nPortPos = url.find(' ');
	if ( nPortPos > 0)
	{
		nPort = atoi(url.substr(nPortPos+1).c_str());
		url[nPortPos]='\0';
	}
	if (url.length())
	{
		TRACE4("opening connection to '", url, "' port ", nPort);
		AddNode(url, nPort);
	}
}


void MGnuDirector::RemoveNode(MGnuNode* pNode)
{
	MLock lock(m_listmutex);
	//
	{
		std::vector<MGnuNode*>::iterator itNode;
		for(itNode = m_NodeList.begin(); itNode != m_NodeList.end(); itNode++)
			if(*itNode == pNode)
			{
				// Make sure socket is closed
				if(pNode->m_hSocket != INVALID_SOCKET)
				{
					pNode->AsyncSelect(FD_CLOSE);
					pNode->ShutDown(2);
				}
				pNode->Close();
				m_NodeList.erase(itNode);
				delete pNode;
				NodeMessage(SOCK_UPDATE, 0);
				return;
			}
	}
}

bool MGnuDirector::IsInTheList(MGnuNode* pNode)
{
	MLock lock(m_listmutex);
	for (int i=0; i<m_NodeList.size();++i)
		if (pNode == m_NodeList[i])
			return true;
	return false;
}

MGnuNode* MGnuDirector::FindNode(CString Host, UINT Port)
{
	MLock lock(m_listmutex);
	//
	for(int i = 0; i < m_NodeList.size(); i++)
		if(m_NodeList[i]->GetHost() == Host)
			return m_NodeList[i];
	return NULL;
}


/////////////////////////////////////////////////////////////////////////////
// TRANSFER CONTROL

int MGnuDirector::CountConnections()
{
	MLock lock(m_listmutex);
	//
	int NumConnected = 0;
	for(int i = 0; i < m_NodeList.size(); i++)	
	{
		MGnuNode *p = m_NodeList[i];

		if(p->IsConnected())
			NumConnected++;
	}
	return NumConnected;
}

int MGnuDirector::CountConnPerSubnet(int SubnetBits, IP ipHost)
{
	MLock lock(m_listmutex);
	//
	int NumConnected = 0;
	u_long ulSubnetMask = (1<<SubnetBits) - 1; // OK for little-endian mashines
	MByteOrderTest bot;
	if (bot.IsBigEndian())
		ulSubnetMask = VBO(ulSubnetMask, true);
	u_long ulSubnet = ulSubnetMask & ipHost.S_addr;
	for(int i = 0; i < m_NodeList.size(); i++)	
	{
		MGnuNode *p = m_NodeList[i];
		IP ip = p->GetSGnuNode().m_ipHost;

		if( p->IsConnected() &&
		    ((p->GetSGnuNode().m_ipHost.S_addr & ulSubnetMask) == ulSubnet) )
			NumConnected++;
	}
	return NumConnected;
}

int MGnuDirector::CountUploading()
{
	MLock lock(m_listmutex);
	//
	int NumUploading = 0;
	for(int i = 0; i < m_UploadList.size(); i++)	
	{
		MGnuUpload *p = m_UploadList[i];

		if(p->m_nStatus == TRANSFER_SENDING)
			NumUploading++;
	}

	return NumUploading;
}

int MGnuDirector::CountDownloading()
{
	MLock lock(m_listmutex);
	//
	int NumDownloading = 0;
	for(int i = 0; i < m_DownloadList.size(); i++)	
	{
		MGnuDownload *p = m_DownloadList[i];

		if(p->m_nStatus != TRANSFER_QUEUED && p->m_nStatus != TRANSFER_CLOSED && p->m_nStatus != TRANSFER_COMPLETED && p->m_nStatus != TRANSFER_NEW)
			NumDownloading++;
	}

	return NumDownloading;
}

int MGnuDirector::CountDownloads(IP Host, MGnuDownload* Origin)
{
	MLock lock(m_listmutex);
	int nCount = 0;
	//
	for(int i = 0; i < m_DownloadList.size(); i++)	
	{
		MGnuDownload *p = m_DownloadList[i];
		if(p->m_nStatus == TRANSFER_CONNECTED || p->m_nStatus == TRANSFER_RECEIVING || p->m_nStatus == TRANSFER_CONNECTING)
			if(p->GetCurrentHost().S_addr == Host.S_addr)
				if(p != Origin)
					nCount++;
	}
	return nCount;
}

int MGnuDirector::GetUploadsCount(const IP& host, MGnuUpload* pExcept)
{
	MLock lock(m_listmutex);
	//
	int nCount = 0;
	for(int i = 0; i < m_UploadList.size(); i++)
		if( (m_UploadList[i]->m_nStatus == TRANSFER_CONNECTED ||
		     m_UploadList[i]->m_nStatus == TRANSFER_SENDING ) &&
		     m_UploadList[i] != pExcept                      &&
		     m_UploadList[i]->m_ipHost.S_addr == host.S_addr      )
			++nCount;
	return nCount;
}

MGnuDownload* MGnuDirector::LookUpDownload(CString FileName, int Index)
{
	MLock lock(m_listmutex);
	//
	MakeLower(FileName);
	for(int i = 0; i < m_DownloadList.size(); i++)	
	{
		MGnuDownload *p = m_DownloadList[i];
		int j;
		for(j = 0; j < p->m_Queue.size(); j++)
			if(p->m_bActive && p->m_Queue[j].FileIndex == Index && p->m_Queue[j].NameLower == FileName)
				return p;
	}
	return NULL;
}


bool MGnuDirector::CheckIfPushung(packet_Push* Push)
{
	MLock lock(m_listmutex);
	//
	for(int i = 0; i < m_UploadList.size(); i++)
	{
		MGnuUpload* p = m_UploadList[i];
		if(p->m_ipHost.S_addr == Push->Host.S_addr &&
		   p->m_nPort		== Push->le_Port &&
		   p->m_nFileIndex	== Push->le_Index )
			return true;
	}
	return false;
}

void MGnuDirector::AddSock(MGnuSock* pSock)
{
	MLock lock(m_listmutex);
	//
	m_SockList.push_back(pSock);
}

void MGnuDirector::AddNode(MGnuNode* pNode)
{
	MLock lock(m_listmutex);
	//
	pNode->m_dwID = GenID();
	m_NodeList.push_back(pNode);	
}

void MGnuDirector::AddUpload(MGnuUpload* pUpload)
{
	MLock lock(m_listmutex);
	//
	pUpload->m_dwID = GenID();
	m_UploadList.push_back(pUpload);
}

void MGnuDirector::AddDownload(const ResultVec& results)
{
	MGnuDownload* pD =  new MGnuDownload(this);
	for (int i=0; i<results.size();++i)
	{
		Result result = results[i];
		// check if we already downloading this
		if ( LookUpDownload(result.Name, result.FileIndex) )
		{
			delete pD;
			return; // todo: something more appropriate
		}
		result.Status     = TRANSFER_QUEUED;
		result.ChangeTime = xtime();
		pD->m_Queue.push_back(result);
	}
	pD->m_nStatus = TRANSFER_QUEUED;
	pD->m_sName = results[0].Name;
	pD->m_dwFileLength = results[0].Size;
	// add to download queue
	AddDownload(pD);	
}

void MGnuDirector::AddDownload(Result& result)
{
	// check if we already downloading this
	if ( LookUpDownload(result.Name, result.FileIndex) )
		return;
	// construct the download object and add it to the list
	MGnuDownload* pD =  new MGnuDownload(this);
	result.Status     = TRANSFER_QUEUED;
	result.ChangeTime = xtime();
	pD->m_Queue.push_back(result);
	pD->m_nStatus = TRANSFER_QUEUED;
	pD->m_sName = result.Name;
	pD->m_dwFileLength = result.Size;
	// add to download queue
	AddDownload(pD);
}

void MGnuDirector::AddDownload(MGnuDownload* pDownload)
{
	MLock lock(m_listmutex);
	//
	pDownload->m_dwID = GenID();
	m_DownloadList.push_back(pDownload);
}

void MGnuDirector::ReplaceDownload(MGnuDownload* what, MGnuDownload* withwhat)
{
	MLock lock(m_listmutex);
	//
	for(int i = 0; i < m_DownloadList.size(); i++)	
		if (m_DownloadList[i] == what)
		{
			withwhat->m_dwID = what->m_dwID;
			what->m_dwID = 0;
			m_DownloadList[i] = withwhat;
			return;
		}
	withwhat->m_dwID = GenID();
	m_DownloadList.push_back(withwhat);
}

MGnuSearch* MGnuDirector::LookUpSearch(const CString& search, int size)
{
	MLock lock(m_listmutex);
	for (int i = 0; i<m_SearchList.size(); ++i)
		if ( LIMIT_EXACTLY == m_SearchList[i]->m_SizeFilterMode &&
		     size == m_SearchList[i]->m_SizeFilterValue         &&
		     search == m_SearchList[i]->m_Search                )
		{
			return m_SearchList[i];
		}
	return NULL;
}

MGnuSearch* MGnuDirector::AddSearch(const CString& search, int type, int size, int sizeMode)
{
	// check for repeating or overlaping
	MLock lock(m_listmutex);
	if (m_SearchList.size()>=m_pPrefs->m_nMaxSearches)
		return NULL;
	for (int i = 0; i<m_SearchList.size(); ++i)
		if ( sizeMode == m_SearchList[i]->m_SizeFilterMode &&
			 size == m_SearchList[i]->m_SizeFilterValue    &&
			 QueryMatch(search, m_SearchList[i]->m_Search) &&
			 QueryMatch(m_SearchList[i]->m_Search, search) )
			 return NULL; // mutualy matching searches are redundant
			 // TODO: above check conflicts slightly with exclusive searches -- fix it
	MGnuSearch* pS = new MGnuSearch(this, search, type, size, sizeMode);
	// add to the list
	pS->m_dwID = GenID();
	m_SearchList.push_back( pS );
	// queue sends and do them on timer
	m_PendingQueries.push_back(pS);
	return pS;
}

bool MGnuDirector::AddSearch(MGnuSearch* pSearch)
{
	// check for repeating or overlaping
	MLock lock(m_listmutex);
	if (m_SearchList.size()>=m_pPrefs->m_nMaxSearches)
		return false;
	// add to the list
	pSearch->m_dwID = GenID();
	m_SearchList.push_back( pSearch );
	// queue sends and do them on timer
	m_PendingQueries.push_back(pSearch);
	return true;
}


void MGnuDirector::ForEachSearch(void* pData, tEachSearch callback)
{
	MLock lock(m_listmutex);
	SGnuSearch gs;
	for (int i = 0; i<m_SearchList.size(); ++i)
	{
		gs = *m_SearchList[i];
		(*callback)(pData, &gs);
	}
}

MGnuSearch* MGnuDirector::GetSearchByID(DWORD dwID)
{
	if (dwID == 0)
		return false;
	m_listmutex.lock();
	for (std::vector<MGnuSearch*>::iterator it = m_SearchList.begin() ; it!=m_SearchList.end(); ++it)
		if ( (*it)->m_dwID == dwID )
		{
			MGnuSearch* pSearch = (*it);
			m_listmutex.unlock();
			return pSearch;
		}
	m_listmutex.unlock();
	return NULL;
}

bool MGnuDirector::GetSearchByID(DWORD dwID, SGnuSearch& gs, std::vector<Result>& rv, std::vector<ResultGroup>& gv)
{
	if (dwID == 0)
		return false;
	m_listmutex.lock();
	for (std::vector<MGnuSearch*>::iterator it = m_SearchList.begin() ; it!=m_SearchList.end(); ++it)
		if ( (*it)->m_dwID == dwID )
		{
			MGnuSearch* pSearch = (*it);
			pSearch->m_mutex.lock();
			gs = *pSearch;
			rv = pSearch->m_CurrentList;
			gv = pSearch->m_GroupList;
			pSearch->m_mutex.unlock();
			m_listmutex.unlock();
			return true;
		}
	m_listmutex.unlock();
	return false;
}

bool MGnuDirector::GetResultsByID(DWORD dwID, std::vector<Result>& rv)
{
	if (dwID == 0)
		return false;
	rv.clear();
	m_listmutex.lock();
	for (std::vector<MGnuSearch*>::iterator it = m_SearchList.begin() ; it!=m_SearchList.end(); ++it)
	{
		MGnuSearch* pSearch = (*it);
		pSearch->m_mutex.lock();
		for (std::vector<ResultGroup>::iterator itg = pSearch->m_GroupList.begin() ; itg!=pSearch->m_GroupList.end(); ++itg)
		{
			if ( (*itg).dwID == dwID )
			{
				if ( 0 == (*itg).ResultList.size() )
				{
					// should never happen
					pSearch->m_mutex.unlock();
					m_listmutex.unlock();
					return false;
				}
				for (int i = 0; i<(*itg).ResultList.size(); ++i )
					rv.push_back(pSearch->m_CurrentList[(*itg).ResultList[i]]);
				//
				pSearch->m_mutex.unlock();
				m_listmutex.unlock();
				return true;
			}
		}
		for (std::vector<Result>::iterator itr = pSearch->m_CurrentList.begin() ; itr!=pSearch->m_CurrentList.end(); ++itr)
		{
			if ( (*itr).dwID == dwID )
			{
				rv.push_back(*itr);
				//
				pSearch->m_mutex.unlock();
				m_listmutex.unlock();
				return true;
			}
		}
		pSearch->m_mutex.unlock();
	}
	m_listmutex.unlock();
	return false;
}

bool MGnuDirector::ClearSearchByID(DWORD dwID)
{
	if (dwID == 0)
		return false;
	m_listmutex.lock();
	for (std::vector<MGnuSearch*>::iterator it = m_SearchList.begin() ; it!=m_SearchList.end(); ++it)
		if ( (*it)->m_dwID == dwID )
		{
			MGnuSearch* pSearch = (*it);
			pSearch->Clear();
			m_listmutex.unlock();
			return true;
		}
	m_listmutex.unlock();
	return false;
}

bool MGnuDirector::RemoveSearchByID(DWORD dwID)
{
	if (dwID == 0)
		return false;
	m_listmutex.lock();
	for (std::vector<MGnuSearch*>::iterator it = m_SearchList.begin() ; it!=m_SearchList.end(); ++it)
		if ( (*it)->m_dwID == dwID )
		{
			MGnuSearch* pSearch = (*it);
			m_SearchList.erase(it);
			m_PendingQueries.remove(pSearch);
			m_listmutex.unlock();
			delete pSearch;
			return true;
		}
	m_listmutex.unlock();
	return false;
}

void MGnuDirector::ForEachConnection(void* pData, tEachConnection callback)
{
	MLock lock(m_listmutex);
	SGnuNode gn;
	for (int i = 0; i<m_NodeList.size(); ++i)
	{
		gn = m_NodeList[i]->GetSGnuNode();
		(*callback)(pData, &gn);
	}
}

bool MGnuDirector::CloseConnectionByID(DWORD dwID)
{
	if (dwID == 0)
		return false;
	m_listmutex.lock();
	for (std::vector<MGnuNode*>::iterator it = m_NodeList.begin() ; it!=m_NodeList.end(); ++it)
		if ( (*it)->GetID() == dwID )
		{
			MGnuNode* pNode = (*it);
			m_listmutex.unlock();
			//RemoveNode(pNode);
			pNode->ForceDisconnect();
			return true;
		}
	m_listmutex.unlock();
	return false;
}

void MGnuDirector::ForEachUpload(void* pData, tEachUpload callback)
{
	MLock lock(m_listmutex);
	SGnuUpload gu;
	for (int i = 0; i<m_UploadList.size(); ++i)
	{
		m_UploadList[i]->m_mutex.lock();
		gu = *m_UploadList[i];
		m_UploadList[i]->m_mutex.unlock();
		(*callback)(pData, &gu);
	}
}

void MGnuDirector::ForEachDownload(void* pData, tEachDownload callback)
{
	MLock lock(m_listmutex);
	SGnuDownload gd;
	for (int i = 0; i<m_DownloadList.size(); ++i)
	{
		m_DownloadList[i]->m_mutex.lock();
		gd = *m_DownloadList[i];
		m_DownloadList[i]->m_mutex.unlock();
		ASSERT(gd.m_nQueuePos < gd.m_Queue.size());
		(*callback)(pData, &gd);
	}
}

bool MGnuDirector::RemoveTransferByID(DWORD dwID, bool bDelPart)
{
	if (dwID == 0)
		return false;
	m_listmutex.lock();
	for (int i = 0; i<m_DownloadList.size(); ++i)
		if ( m_DownloadList[i]->m_dwID == dwID )
		{
			MGnuDownload* pD = m_DownloadList[i];
			m_listmutex.unlock();
			pD->StopDownload(bDelPart,true);
			return true;
		}
	//	
	for (int i = 0; i<m_UploadList.size(); ++i)
		if ( m_UploadList[i]->m_dwID == dwID )
		{
			MGnuUpload* pU = m_UploadList[i];
			m_listmutex.unlock();
			pU->ForceDisconnect();
			return true;
		}
	m_listmutex.unlock();
	return false;
}

void MGnuDirector::OnQueryHit(packet_QueryHit* QueryHit, MGnuNode* pOrigin, bool bOur)
{
	if (!bOur && m_pPrefs->m_bStrictSearch)
		return;
	if (QueryHit->Header.Function != 0x81)
		return;
	if (!m_pPrefs->m_bReachableForPush && !IsOkForDirectConnect(QueryHit->Host))
		return;

	BYTE*			 Packet   = (BYTE*) QueryHit;

	bool ExtendedPacket = false;
	bool Firewall		= false;
	bool Busy			= false;
	bool Stable			= false;
	bool ActualSpeed	= false;

	int    HitsLeft = QueryHit->TotalHits, i, j;
	DWORD  NextPos  = 34;
	DWORD  Length   = QueryHit->Header.le_Payload + 23;

	// Find start of QHD
	int ItemCount = 0;
	bool bNoZero = true;
	for(i = 42; i < Length - 16; i++)
	{
		if(Packet[i] == 0)
			if (bNoZero)
				bNoZero = false;
			else
			{
				ItemCount++;
				bNoZero = true;
				if(ItemCount != QueryHit->TotalHits)
					i += 8;
				else
					break;
			}
	}

	packet_QueryHitEx* QHD = (packet_QueryHitEx*) (Packet+i+1);
	CString Vendor( (char*) QHD->VendorID, 4);
	if( ValidVendor(Vendor) )
	{
		ExtendedPacket = true;

		if(QHD->Length == 1)
			if(QHD->Push == 1)
				Firewall = true;

		if(QHD->Length == 2)
		{
			if(QHD->FlagPush)
				Firewall = QHD->Push;

			if(QHD->FlagBusy)
				Busy = QHD->Busy;

			if(QHD->FlagStable)
				Stable = QHD->Stable;

			if(QHD->FlagSpeed)
				ActualSpeed = QHD->Speed;
		}
	}

	// Extract results from the packet
	while(HitsLeft > 0 && NextPos < Length - 16)
	{
		Result Item;
		packet_QueryHitItem* pQHI = (packet_QueryHitItem*) (Packet+NextPos);

		Item.Size = pQHI->le_Size;
		Item.Port = QueryHit->le_Port;
		if ( (m_pPrefs->m_nMinFileKSize<=0 || Item.Size >= m_pPrefs->m_nMinFileKSize*1024) &&
			 (m_pPrefs->m_nMaxFileKSize<=0 || Item.Size <= m_pPrefs->m_nMaxFileKSize*1024) &&
			 Item.Port != 99 ) // spam protection
		{
			Item.FileIndex = pQHI->le_Index;
			Item.Host      = QueryHit->Host;
			Item.Speed	   = QueryHit->le_Speed;

			Item.Firewall	 = Firewall;
			Item.Busy		 = Busy;
			Item.Stable		 = Stable;
			Item.ActualSpeed = ActualSpeed;
			Item.Vendor = GetVendor(Vendor);

			Item.Origin = pOrigin;
			memcpy(&Item.PushID, Packet+(Length - 16), 16);
			Item.Distance = QueryHit->Header.Hops;

			// Get Filename
			for(i = NextPos + 8; Packet[i] != '\0' && i < Length - 16; ++i);
			
			Item.Name = CString(((char*)Packet)+NextPos+8, 0, i-NextPos-8);
			if ( i<Length-16 && Packet[i+1] != '\0')
			{
				// we have extra info at the hit tail
				Item.Extra = CString(((char*)Packet)+i+1, 0, Length-17-i);
				for( ; Packet[i+1] != '\0' && i < Length - 16; ++i);
			}
			/*// TODO: get rid of this rubbush
			for(i = NextPos + 8; !(Packet[i] == '\0' && Packet[i + 1] == '\0'); i++)
				if(i < Length - 16)
					Item.Name += (char) Packet[i];
				else
					break;*/

			Item.NameLower = Item.Name;
			MakeLower(Item.NameLower);
			// now check if this result matches some of the queries
			//cout << "query hit: " << Item.NameLower << endl;
			m_listmutex.lock();
			for (j=0; j<m_SearchList.size(); ++j)
			{
				if (m_SearchList[j]->CheckAgainstResult(Item))
					m_pController->UpdateHits();
			}
			m_listmutex.unlock();
		}
		else
		{
			// skip the result
			for(i = NextPos + 8; Packet[i] != '\0' && i < Length - 16; ++i);
			for( ; Packet[i+1] != '\0' && i < Length - 16; ++i);
		}
		
		// Check for end of reply packet
		if(i + 2 >= Length - 16)
			HitsLeft = 0;
		else
		{
			HitsLeft--;
			NextPos = i + 2;
		}
	}
}

/////////////////////////////////////////////////////////////////////////////
// COMMUNICATIONS

void MGnuDirector::NodeMessage(UINT dwMessID, WPARAM packet)

{
    /*switch(dwMessID)
	{
	case SOCK_UPDATE:
		if(m_pController->m_pViewConnect)
			CWnd::FromHandle(m_pController->m_pViewConnect)->SendMessage(WM_COMMAND, dwMessID, NULL);

		std::vector<HWND>::iterator itStat;
		for(itStat = m_pController->m_pViewStatistics.begin(); itStat != m_pController->m_pViewStatistics.end(); itStat++)
			CWnd::FromHandle(*itStat)->SendMessage(WM_COMMAND, dwMessID, NULL);

		return;
		break;
	case PACKET_INCOMING:
		if(m_pController->m_pViewConnect && ((packet_Log*)packet)->Header->Function == 0x01)
			CWnd::FromHandle(m_pController->m_pViewConnect)->SendMessage(dwMessID, packet, NULL);

		if(((packet_Log*)packet)->Header->Function == 0x81)
		{
			std::vector<HWND>::iterator itSearch;
			for(itSearch = m_pController->m_pViewSearch.begin(); itSearch != m_pController->m_pViewSearch.end(); itSearch++)
				if( ((packet_Log*) packet)->Header->Guid == ((CViewSearch*) CWnd::FromHandle(*itSearch))->m_QueryID)
					CWnd::FromHandle(*itSearch)->SendMessage(dwMessID, packet, NULL);
		}
		break;
	}	

	std::vector<HWND>::iterator itStat;
	for(itStat = m_pController->m_pViewStatistics.begin(); itStat != m_pController->m_pViewStatistics.end(); itStat++)
		CWnd::FromHandle(*itStat)->SendMessage(dwMessID, packet, NULL);*/
}

void MGnuDirector::TransferMessage(UINT dwMessID, WPARAM pTransfer)
{	
	/*if(m_pController->m_StopPost)
		return;

	CSingleLock ViewAccess(&m_pController->m_TransferAccess);
	ViewAccess.Lock();
			
	if(ViewAccess.IsLocked())
	{
		if(m_pController->m_pViewTransfers)
		{
			ViewAccess.Unlock();

			switch(dwMessID)
			{
			case UPLOAD_UPDATE:
				
					CWnd::FromHandle(m_pController->m_pViewTransfers)->PostMessage(dwMessID, pTransfer, NULL);
				break;
			case DOWNLOAD_UPDATE:
				if(m_pController->m_pViewTransfers)
					CWnd::FromHandle(m_pController->m_pViewTransfers)->PostMessage(dwMessID, pTransfer, NULL);
				break;
			case PARTIAL_UPDATE:
				if(m_pController->m_pViewTransfers)
					CWnd::FromHandle(m_pController->m_pViewTransfers)->SendMessage(WM_COMMAND, dwMessID, NULL);
				break;
			}
		}
		else
			ViewAccess.Unlock();
	}*/
}

void MGnuDirector::OnTimer() // called once per second
{
	POST_DMESSAGE("MGnuDirector::OnTimer()");
	//double BytesReceived = 0,  BytesSent = 0;
	int    nNumConnected = 0;
	int    nNumUploads   = CountUploading();
	
	if (xtime()%60 == 0)
	{
		m_TableQuery.ClearUpTable(180);// 3 min is enough to route packets
		m_TablePing.ClearUpTable(180); // 3 min is enough to route packets
		//m_TablePush.ClearUpTable(600);   // these tables doesnt seem to overflow
		//m_TableLocal.ClearUpTable(600);  //
	}
	

	// Null socks - used to interogate incomming connects
	int i;
	for(i = 0; i<m_SockList.size(); i++)
	{
		MGnuSock *pSock = m_SockList[i];
    	if(pSock->m_nSecsAlive > 15 || pSock->m_bDestroy)
		{
			delete pSock;
			m_SockList.erase(&m_SockList[i]);
			i--;
		}
		else
			pSock->Timer();
	}

	// Node Connections
	for(i = 0; i < m_NodeList.size(); i++)
	{
		MGnuNode *pNode = m_NodeList[i];
    	if(pNode->IsConnected())
		{
			//BytesReceived += pNode->m_nSecBytes[0];
			//BytesSent     += pNode->m_nSecBytes[1];
			nNumConnected++;
		}
		pNode->OnTimer();
	}
	
	// Conection bandwidth
	m_dwConnRecvRate *= 15;
	m_dwConnRecvRate += m_dwConnBytesRecv;
	m_dwConnRecvRate >>= 4;
	m_dwConnBytesRecv = 0;
	m_dwConnSendRate *= 15;
	m_dwConnSendRate += m_dwConnBytesSend;
	m_dwConnSendRate >>= 4;
	m_dwConnBytesSend = 0;
	
    //
	ManageBandwidth((m_dwConnRecvRate+m_dwConnSendRate) / 1024, nNumConnected, nNumUploads);
	ManageConnects(nNumConnected);
	//cout << "connection state: connections " << nNumConnected << "  bandwidth " << BytesReceived + BytesSent << endl;

	long BytesPerUpload = -1;
	if (nNumUploads)
	{
		if(m_pPrefs->m_dBandwidthTotal >= 0 && nNumUploads)
		{
			BytesPerUpload = (int) ((m_pPrefs->m_dBandwidthTotal * 1024 - (m_dwConnRecvRate+m_dwConnSendRate)) / nNumUploads);

			if(BytesPerUpload < 0)
				BytesPerUpload = 0;
		}
		if ( m_pPrefs->m_dBandwidthTransfer >=0 &&
		     ( BytesPerUpload < 0 ||
		       BytesPerUpload > (m_pPrefs->m_dBandwidthTransfer * 1024)/nNumUploads) )
		{
			BytesPerUpload = long((m_pPrefs->m_dBandwidthTransfer * 1024)/nNumUploads);
		}
	}

	m_listmutex.lock();
	// Uploads
	std::vector<MGnuUpload*>::iterator itUpload;
	m_dUploadRate = 0;
	for(i = 0; i < m_UploadList.size(); i++)
	{
		MGnuUpload *pUp = m_UploadList[i];

		if(pUp->m_nStatus == TRANSFER_CLOSED || pUp->m_nStatus == TRANSFER_COMPLETED)
		{
			delete pUp;
			m_UploadList.erase(&m_UploadList[i]);
			i--;
		}
		else
		{
			pUp->BandwidthTimer();
			
			if(TRANSFER_SENDING == pUp->m_nStatus)
			{
				m_dUploadRate += pUp->m_dRate;
			}
			pUp->SetMaxRate(BytesPerUpload);
		}
	}

	// Downloads
	m_dDownloadRate = 0;
	for(i = 0; i < m_DownloadList.size(); i++)
	{
		MGnuDownload *pDown = m_DownloadList[i];
		pDown->BandwidthTimer();
		if(TRANSFER_RECEIVING == pDown->m_nStatus)
			m_dDownloadRate += pDown->m_dRate;
	}
	// Bandwidth
	int BitSpeed = (int) ((m_dwConnRecvRate+m_dwConnSendRate+m_dDownloadRate+m_dUploadRate) / 1024 * 8);
	if(BitSpeed > m_pPrefs->m_dwSpeedDyn)
		m_pPrefs->m_dwSpeedDyn = BitSpeed;
		
	// renew searches
	if (GetPrefs()->m_nResubmitSearches && m_SearchList.size())
	{
		m_nReSearchTimer++;
		if ( m_nReSearchTimer >= (2 /*let the min time be 2 sec*/ + GetPrefs()->m_nResubmitSearches/m_SearchList.size()) )
		{
			m_nReSearchTimer = 0;
			m_nReSearchCounter %= m_SearchList.size();
			if (!m_SearchList[m_nReSearchCounter]->IsFull()) //food protection for searches like ".mp3"
				m_PendingQueries.push_back(m_SearchList[m_nReSearchCounter]);
			m_nReSearchCounter++;
		}
	}

    // clean up downloads list
	for( i = 0; i < m_DownloadList.size(); i++)
	{
		MGnuDownload* pDown = m_DownloadList[i];
		if (pDown->m_nSecInactive>30)
		{
			// TODO: messageID
			POST_MESSAGE(ES_GOODTOKNOW, CString("Removing download `") + pDown->m_sName + "'\n  status:" + MGnuDownload::GetErrorString(pDown->m_nDisconnectReason));
			// add auto-getting search if appropriate
			if (pDown->m_nDisconnectReason != REASON_COMPLETED && pDown->m_nDisconnectReason != REASON_STOPPED)
				pDown->AddSearch(true);
			else if (pDown->m_bDeleteSearch && pDown->m_dwSearchID)
			{
				// remove "autoget" search
				// m_listmutex is locked here
				for (vector<MGnuSearch*>::iterator is = m_SearchList.begin(); is != m_SearchList.end(); ++is)
				{
					if ( (*is)->IsAutomatic() && pDown->m_dwSearchID == (*is)->m_dwID )
					{
						delete (*is);
						m_SearchList.erase(is);
						break;
					}
				}
			}
			//
			m_DownloadList.erase(&m_DownloadList[i]);
			i--;
			delete pDown;
		}
	}
	
	// auto-get support
	for (int i=0; i<m_SearchList.size(); i++)
	{
		MGnuSearch* pS = m_SearchList[i];
		pS->m_mutex.lock();
		if (pS->m_bAutoget && pS->m_bUpdated && pS->m_CurrentList.size())
		{
			ASSERT(pS->m_SizeFilterMode == LIMIT_EXACTLY);
			POST_MESSAGE(ES_GOODTOKNOW, CString("auto-getting ") + pS->m_Filename);
			//
			MGnuDownload* pD =  new MGnuDownload(this);
			pS->m_bUpdated = false;
			for (int i=0; i<pS->m_CurrentList.size();++i)
			{
				Result result = pS->m_CurrentList[i];
				// check if we already downloading this
				if ( LookUpDownload(result.Name, result.FileIndex) )
				{
					delete pD;
					pD = NULL;
					break;
				}
				result.Status     = TRANSFER_QUEUED;
				result.ChangeTime = xtime();
				pD->m_Queue.push_back(result);
			}
			if (pD)
			{
				pD->m_nStatus = TRANSFER_QUEUED;
				if (pS->m_Filename.size())
					pD->m_sName = pS->m_Filename;
				else
					pD->m_sName = pS->m_CurrentList[0].Name;
				pD->m_sSearch = pS->m_Search;
				pD->m_dwSearchID = pS->m_dwID;
				pD->m_dwFileLength = pS->m_CurrentList[0].Size;
				// add to download queue
				AddDownload(pD);
			}
			//
			pS->m_bAutoget = FALSE;
		}
		pS->m_mutex.unlock();
	}
	// send pending queries
	while (!m_PendingQueries.empty())
	{
		m_PendingQueries.front()->SendQuery();
		m_PendingQueries.pop_front();
	}
	// carefull, potential dead-lock
	m_hitqueuemutex.lock();
	while (!m_hitQueue.empty())
	{
		QueuedHit qh = m_hitQueue.front();
		m_hitQueue.pop();
		for(int i = 0; i < m_NodeList.size(); i++)	
		{
			MGnuNode *p = m_NodeList[i];
			if(p == qh.pNode)
			{
				p->Send(qh.pPacket, qh.size); // TODO: get rig of data copiing and memory re-allocation
				delete [] qh.pPacket;
				break;
			}
		}
	}
	m_hitqueuemutex.unlock();
	m_listmutex.unlock();
	
	m_nLastConnCount = CountConnections();
	
	if (xtime()%60==0)
	{
		//printf("query stats: good: %d \t repeats: %.3f%% \t errors: %.3f%%\n",	m_nQGood, 100.0*m_nQRepeat/(m_nQGood+0.1), 100.0*m_nQError/(m_nQGood+0.1));
		m_nQGood = 0;
		m_nQRepeat = 0;
		m_nQError = 0;
	}
}

void MGnuDirector::ManageBandwidth(float CurrentSpeed, int ConnectNum, int UploadNum)
{
	float BandwidthLimit = -1;

	if(m_pPrefs->m_dBandwidthTotal >= 0 && m_pPrefs->m_dBandwidthConnects >= 0)
		BandwidthLimit = min(m_pPrefs->m_dBandwidthTotal,m_pPrefs->m_dBandwidthConnects);
	else if(m_pPrefs->m_dBandwidthTotal >= 0)
		BandwidthLimit = m_pPrefs->m_dBandwidthTotal;
	else if(m_pPrefs->m_dBandwidthConnects >= 0)
		BandwidthLimit = m_pPrefs->m_dBandwidthConnects;

	if(BandwidthLimit >= 0)
	{
		if(CurrentSpeed > BandwidthLimit)
			m_nSecsAboveLimitConn++;
		else
			m_nSecsAboveLimitConn = 0;

		if(m_nSecsAboveLimitConn > 10)
		{
			RemoveNode_MostRecv();
			m_nSecsAboveLimitConn = 0;
		}
	}
}

bool MGnuDirector::AllowMakingConnections()
{
	return !m_pPrefs->m_bQuietMode;
}

void MGnuDirector::ManageConnects(int nCurrent)
{
	if(m_pPrefs->m_nMaxConnects > 0 && nCurrent >= m_pPrefs->m_nMaxConnects)
	{
		RemoveNode_MaxDrop();
	}
	else
	{
		if(rand()%30 == 0)
			RemoveNode_MaxDrop();
		if(rand()%120 == 0)
			RemoveNode_LeastRate();
	}
	
	if(AllowMakingConnections())
	{
		if(m_pPrefs->m_nMinConnects > 0 &&
		   nCurrent < m_pPrefs->m_nMinConnects &&
		   (m_pPrefs->m_nMaxConnects<0 || nCurrent < m_pPrefs->m_nMaxConnects))
		{
			//printf("ManageConnects: making connection\n");
			AddConnect();
		}
	}
}

// Node management functions
void MGnuDirector::AddConnect()
{
	Node node;
	if (m_pCache->GetRandomNode(node))
	{
		if (CheckIP(node.Host) && IsOkForDirectConnect(node.Host) && NotLocal(node))
			AddNode( IPtoStr(node.Host), node.Port );
	}
	else
	{
		if (strlen(m_pPrefs->m_szAutoConnectSrv1))
			AddNode(m_pPrefs->m_szAutoConnectSrv1);
		if (strlen(m_pPrefs->m_szAutoConnectSrv2))
			AddNode(m_pPrefs->m_szAutoConnectSrv2);
		if (strlen(m_pPrefs->m_szAutoConnectSrv3))
			AddNode(m_pPrefs->m_szAutoConnectSrv3);
		if (strlen(m_pPrefs->m_szAutoConnectSrv4))
			AddNode(m_pPrefs->m_szAutoConnectSrv4);
	}
}

//
void MGnuDirector::RemoveNode_MaxDrop()
{
	u_long nMaxDrop = 0;
	if(!m_NodeList.size())
		return;

	std::vector<MGnuNode*>::iterator itNode, itRemove;
	for(itNode = m_NodeList.begin(); itNode != m_NodeList.end(); itNode++)
	{
		MGnuNode *p = *itNode;

		if(p->GetSGnuNode().m_nDroppedPackets > nMaxDrop)
		{
			itRemove   = itNode;
			nMaxDrop = p->GetSGnuNode().m_nDroppedPackets;
		}
	}
	if (nMaxDrop >0)
		RemoveNode(*itRemove);
}

// Primitive also.. disconnecting the node wich sends us the least data
void MGnuDirector::RemoveNode_LeastRate()
{
	double LeastRate = -1;

	if(!m_NodeList.size())
		return;

	std::vector<MGnuNode*>::iterator itNode, itRemove;
	for(itNode = m_NodeList.begin(); itNode != m_NodeList.end(); itNode++)
	{
		MGnuNode *p = *itNode;

		if(p->GetSGnuNode().m_dRate[0] < LeastRate)
		{
			itRemove   = itNode;
			LeastRate = p->GetSGnuNode().m_dRate[0];
		}
	}

	if (LeastRate>0)
		RemoveNode(*itRemove);
}

void MGnuDirector::RemoveNode_MostRecv()
{
	double MostBytes = -1;

	if(!m_NodeList.size())
		return;
	
	std::vector<MGnuNode*>::iterator itNode, itRemove;
	for(itNode = m_NodeList.begin(); itNode != m_NodeList.end(); itNode++)
	{
		MGnuNode *p = *itNode;

		if(p->GetSGnuNode().m_dRate[0] > MostBytes)
		{
			itRemove  = itNode;
			MostBytes = p->GetSGnuNode().m_dRate[0];
		}
	}

	if (MostBytes > 0)
	{
		// this is kinda hack, but in principle it is right thing to do
		if (m_dwConnRecvRate > (*itRemove)->GetSGnuNode().m_dRate[0])
			m_dwConnRecvRate -= (u_int)(*itRemove)->GetSGnuNode().m_dRate[0];
		if (m_dwConnSendRate > (*itRemove)->GetSGnuNode().m_dRate[1])
			m_dwConnSendRate -= (u_int)(*itRemove)->GetSGnuNode().m_dRate[1];
		// remove
		RemoveNode(*itRemove);
	}
}

void MGnuDirector::GetNetStats(int& nHosts, int& nSharingHosts, int& nFiles, int& nSize)
{
	MLock lock(m_listmutex);
	nHosts = 0;
	nSharingHosts = 0;
	nFiles = 0;
	nSize = 0;
	for (int i=0; i< m_NodeList.size(); ++i)
	{
		MGnuNode* pNode = m_NodeList[i];
		if ( pNode->IsConnected() )
		{
			nHosts++;
			nHosts        += pNode->GetSGnuNode().m_dwFriendsTotal;
			nSharingHosts += pNode->GetSGnuNode().m_dwSharingTotal;
			nSize         += pNode->GetSGnuNode().m_llLibraryTotal/1024;
			nFiles        += pNode->GetSGnuNode().m_dwFilesTotal;
		}
	}
}

