/***************************************************************************
 Mutella - A commandline/HTTP client for the Gnutella filesharing network.

 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation; either version 2
 of the License, or (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

                          uitextmode.cpp  -  description
                             -------------------
    begin                : Thu Mar 13 2003
    copyright            : (C) 2003 by Max Zaitsev
    email                : maksik@gmx.co.uk
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <ctype.h>
#include <time.h>
#include <sys/types.h>	/* waitpid () */
#include <sys/wait.h>	/* waitpid () */
#include <unistd.h>	/* fnctl () */
#include <fcntl.h>	/* fnctl () */

#include "mutella.h"
#include "structures.h"

#include "controller.h"
#include "mui.h"
#include "uiterminal.h"
#include "property.h"
#include "preferences.h"
#include "mprintf.h"
#include "ansicolor.h"
#include "event.h"
#include "messages.h"
#include "lineinput.h"
#include "conversions.h"
#include "common.h"
#include "term_help.h"
#include "gnumarkedfiles.h"

#include "uitextmode.h"

#define _isblank isspace

MUITextModePriv::MUITextModePriv(MController* pC) /*: m_more(80,24), m_input(this, pC)*/
{
	m_pController = pC;
	m_pOutput = NULL;
}

MUITextModePriv::~MUITextModePriv()
{
}

MUITextModePriv::ComEntry MUITextModePriv::commands[] = {
  { "help",   &MUITextModePriv::com_help,   "Displays a help message. Type `help <command>' for more specific info", NULL},
  { "?",      &MUITextModePriv::com_help,   "Synonym for `help'", NULL},
  { "info"   ,&MUITextModePriv::com_info,   "Displays various information regarding current network activities",
  									"info [network] [connections] [transfers] [uploads] [downloads] [loop]\n"
  									"    Displays different bits of information regarding current status of the\n"
  									"    client and its activities. Use parameters to select specific types of\n"
  									"    info, e.g. 'info downloads' will only display status of current downloads.\n"
  									"    Parameter 'loop' repeats the info command each 2 seconds until key is\n"
  									"    pressed. 'info loop' can be combined with any other 'info' parameter.\n"
  									"    Parameters can be abbreviated and multiple parameters can be given. Without\n"
  									"    parameters displays all status information.\n"
  									"    NOTE: rates displayed for individual Gnutella-net connecvtions do not\n"
  									"    reflect the bandwidth consumed and are given only for the reference.\n"
  									"    Total connection speed however is quite reliable parameter."},
  { "hosts"  ,&MUITextModePriv::com_hosts,  "Displays current content of the hosts cache", NULL },
  { "find",   &MUITextModePriv::com_find,   "Adds a query to the search list",
  									"find <list of keywords> [options]\n"
  									"    Adds new query to the current list of searches. All the keywords are\n"
  									"    matched as a boolean AND ignoring upper/lower case.\n"
  									"    NEW: exclusive search is supported: `find hook .avi -hookers' will only\n"
  									"    look for a movie, filtering out porno-crap.\n"
  									"    NEW: Sha1 searches can be created by replacing the search string with\n"
  									"    the string starting from 'sha1:', e.g. 'sha1:XX..XX', where 'XX..XX' is\n"
  									"    base32-encoded sha1 hash. Sha1 searches can be combined with other\n"
  									"    options and '/autoget' in particular."
  									"    Additional search options allow one to specify exact file size with\n"
  									"    `size:XXXX' or minimum file size with `min:XXXX' or approximate size\n"
  									"    within 10% tolerance with `around:XXXX'. Auto-get searches can be created\n"
  									"    by means of adding `/auto' or `/autoget' flag (to be used with care)."
  									"    With no parameters equivalent to `list'" },
  { "list",   &MUITextModePriv::com_list,   "Lists current searches",
  									"list [pattern] [options]\n"
  									"    Displays current list of searches. Optionally takes pattern as a\n"
  									"    parameter. When pattern is given only searches that match it are\n"
  									"    listed. This is useful when search list grows above 5-10 entries.\n"
  									"    Additionally it is possible to filter out searches with no hits by\n"
  									"    '-empty' option and automatically generated searches with '-auto'" },
  { "ls",     &MUITextModePriv::com_ls,     "Synonym for 'list -empty'", NULL},
  { "edit"   ,&MUITextModePriv::com_edit,   "Modifies keyword string for existing search",
  									"edit <search_ID> <new_search_string>\n"
  									"    Enables user to modify existing searches. Useful when automatic keyword\n"
  									"    extraction fails to generate proper search string when searching for\n"
  									"    alternative locations. NOTE that this does not change the file name of\n"
  									"    the file being downloaded"},
  { "delete", &MUITextModePriv::com_delete, "Deletes a query or queries from the list of searches",
  									"delete <search_ID(s)>\n"
  									"    Deletes a query or queries from the current list of searches. Numeric\n"
  									"    ID(s) should be taken from last output of the `list' command. IDs can\n"
  									"    be given in fairly relaxed way. For example `delete 2,5,8-10,1' will\n"
  									"    delete searches listed under numbers 1,2,5,8,9,10; `delete 1-' deletes\n"
  									"    all the searches listed by the last `list' command\n" },
  { "erase", &MUITextModePriv::com_erase,   "Deletes query(ies) from the search list and erases the partial file(s)",
  									"erase <search_ID(s)>\n"
  									"    Deletes a query or queries from the current list of searches and deletes\n"
  									"    the partial file(s) for the case of automatically generated auto-get\n"
  									"    searches. The partial files for the active downloads will not be deleted.\n"
  									"    You should use 'kill' command instead. Similarly to 'delete', numeric\n"
  									"    ID(s) should be taken from last output of the `list' command. IDs can\n"
  									"    be given in fairly relaxed way. For example `delete 2,5,8-10,1' will\n"
  									"    delete searches listed under numbers 1,2,5,8,9,10; `delete 1-' deletes\n"
  									"    all the searches listed by the last `list' command\n" },

  { "clear",  &MUITextModePriv::com_clear,  "Clears results list for the query or query list",
  									"clear <search_ID(s)>\n"
  									"    Clears results of each the queries referenced by numeric IDs. Numeric\n"
  									"    ID(s) are to be taken from last output of the `list' command. IDs can\n"
  									"    be given in fairly relaxed way. For example `clear 2,5,8-10,1' will\n"
  									"    clear up searches listed under numbers 1,2,5,8,9,10; `clear 1-' clears\n"
  									"    all the searches listed by the last `list' command\n"},
  { "results",&MUITextModePriv::com_results,"Displays search results",
  									"results [pattern | search_ID(s)]\n"
  									"    Displays results of the search. Searches can be selected by either\n"
  									"    pattern in the same way as for `list' command, or numerical search ID(s)" },
  { "get"    ,&MUITextModePriv::com_get,    "Get files from the net",
  									"get <result_ID(s)>\n"
  									"    Initiates download of the file(s) for given result ID(s). Result ID(s)\n"
  									"    should reference search result list as it was produced by the last\n"
  									"    `results' command. IDs can be given in fairly relaxed way. For example\n"
  									"    `get 10, 15, 22-25' will start download of the search results listed\n"
  									"    under numbers 10, 15, 21, 22, 23, 24, 25.\n"
  									"    The progress of download procedure can be examined with `info' command.\n"
  									"    Once download is started Mutella tries to get requested file forever,\n"
  									"    until the transfer is successful or stopped with the `stop' command.\n"
  									"    When download fails to start immediately Mutella initiates the search for\n"
  									"    alternative locations for the file. If host is not responding for last\n"
  									"    25 trials 'auto-get' search is added and download failure is reported.\n"
  									"    When the file appears on the Gnutella horizon Mutella automatically\n"
  									"    initiates the transfer. Auto-get searches are also created for all\n"
  									"    partial files found in the download directory on the client start" },
  { "stop"   ,&MUITextModePriv::com_stop,   "Stops the transfer",
  									"stop <transfer_ID(s)>\n"
  									"    Stops transfers corresponding to the given numeric ID(s). Transfer IDs\n"
  									"    are to be taken from the last `info' command output\n"
  									"    NOTE that 'stop' doesn't remove partial file from the download directory\n"
  									"    use `kill' instead" },
  { "kill"   ,&MUITextModePriv::com_kill,   "Same as stop, but erases partial file in case of download",
  									"kill <transfer_ID(s)>\n"
  									"    Stops transfers corresponding to the given numeric ID(s) and deletes\n"
  									"    appropriate patrial files in case of download. Transfer IDs are to be taken\n"
  									"    from the last `info' command output"},
  { "move"   ,&MUITextModePriv::com_move,   "Modifies filename of the file being downloaded",
  									"move <download_ID> <new_file_name>\n"
  									"    Useful to remove 'SHARE-THIS' rubbish frim the file name so that auto-search\n"
  									"    feature produces more results and finished file look nicer. If the automatic\n"
  									"    search for alternative locations has already been created it will be modified\n"
  									"    SEE ALSO: 'edit'"},
  { "open"   ,&MUITextModePriv::com_open,   "Opens new Gnutella-net connection",
  									"open <host> [port]\n"
  									"    Opens Gnutella-net connection to the specified host. When 'port'\n"
  									"    parameter is omitted 6346 is assumed" },
  { "close"  ,&MUITextModePriv::com_close,  "Closes specified Gnutella connection(s)",
  									"close <connection_ID(s)>\n"
  									"    Closes connection(s) which correspond to the given numeric ID(s).\n"
  									"    Connection IDs are to be taken from the last `info' command output" },
  { "scan"   ,&MUITextModePriv::com_scan,   "(Re)Scans shared directory", NULL },
  { "library",&MUITextModePriv::com_library,"Lists currently shared files", NULL },
  { "load"   ,&MUITextModePriv::com_load,   "Loads and executes Mutella terminal-mode script", NULL },
  { "set"    ,&MUITextModePriv::com_set,    "Accesses Mutella options",
  									"set [Variable [new_value]]\n"
  									"    Used to access Mutella options. Without parameters displays the entire\n"
  									"    list of options. When called up with one parameter displays the value\n"
  									"    of the appropriate variable. When called with two parameters treats\n"
  									"    the first parameter as variable name and the second one as a new value\n"
  									"    To empty the string variable type `set <variable> \"\"'\n"
  									"    All mutella options are stored in `~/.mutella/mutellarc' file" },
  { "set+"   ,&MUITextModePriv::com_set_add,"Adds a value to the list-type option",
  									"set+ [Variable [new_value]]\n"
  									"    Enables to add a value to the list. Without parameters displays all the\n"
  									"    list-type options. When called with one parameter displays the value\n"
  									"    of the appropriate variable. When called with two parameters treats\n"
  									"    the first parameter as variable name and the second as a value to be\n"
  									"    added to the list" },
  { "set-"   ,&MUITextModePriv::com_set_remove,"Removes a value from the list-type option",
  									"set- [Variable [value]]\n"
  									"    Enables to remove a given value from the list. Without parameters displays\n"
  									"    all the list-type options. When called with one parameter displays the value\n"
  									"    of the appropriate variable. When called with two parameters treats\n"
  									"    the first parameter as variable name and the second as a value to be\n"
  									"    removed from the list" },
  { "color"  ,&MUITextModePriv::com_color,  "Sets terminal colors",
  									"color [field [new_ASCI_code]]\n"
  									"    Used to set colors for terminal UI. Syntax is similar to `set'. The\n"
  									"    color scheme is stored in the location ponted by ColorScheme variable\n"
  									"    (default is `~/.mutella/termclr')" },
  { "system" ,&MUITextModePriv::com_system, "Executes a shell command given by argument", NULL },
  { "!"      ,&MUITextModePriv::com_system, "synonym for 'system'", NULL },
  { "version",&MUITextModePriv::com_version,"Displays program version", NULL },
  { "exit",   &MUITextModePriv::com_exit,   "Exits Mutella",
  									"    Initiates exit sequence. Cannot be abbreviated to one letter"},
  { "leave",  &MUITextModePriv::com_leave,  "Stops the current user interface",
  									"    Stops the current user interface but lets Mutella continue. Cannot\n"
  									"    be abbreviated to one letter"},
  
  { NULL, NULL, NULL, NULL }
};

/* Look up NAME as the name of a command, and return a pointer to that
   command.  Return a NULL pointer if NAME isn't a command name. */
MUITextModePriv::ComEntry * MUITextModePriv::find_command ( char *name )
{
	register int i;
	for (i = 0; commands[i].name; i++)
		if (strncmp (name, commands[i].name, strlen(name)) == 0)
		{
			if (strcmp (name, "e") == 0)
				return NULL;
			return (&commands[i]);
		}
	return NULL;
}

bool MUITextModePriv::execute_line (const CString& sLine)
{
	register int i;
	ComEntry *command;
	char *word;
	//
	char *line = (char*) alloca(sLine.size()+1);
	ASSERT(line);
	strncpy(line, sLine.c_str(), sLine.size());
	line[sLine.size()] = '\0';
	//
	/* Isolate the command word. */
	i = 0;
	while (line[i] && _isblank(line[i]))
		i++;
	word = line + i;

	while (line[i] && !_isblank(line[i]))
		i++;

	if (line[i])
		line[i++] = '\0';

	command = find_command (word);

	if (!command)
	{
		printf ("%s: No such command.\n", word);
		return false;
	}

	/* Get argument to command, if any. */
	while (_isblank(line[i]))
		i++;

	word = line + i;

	return execute_command(command, word);
}

bool MUITextModePriv::execute_command(ComEntry *command, LPSTR word)
{
	return ((*this).*(command->handler))(word);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////
// couple of useful parsers
//

bool ParseNumRanges(intSet& seq, char* str, int nMax /*for n- combination*/)
{
	bool bAfterDash = false;
	int n = -1;
	int m = -1;
	int len = strlen(str);
	char* p = NULL;
	for( int i = 0; i<=len; ++i )
	{
		if (str[i]>='0' && str[i]<='9')
		{
			// it is a digit -- set the pointer to it if it is the first one
			if (!p)
				p=str+i;
		}
		else if ( str[i]==',' || str[i]==';' || str[i]=='-' || str[i]==' ' || str[i]=='\t' || str[i]=='\0' )
		{
			// achieved space or separator or range symbol (dash);
			if (p)
				// if we have the pointer to the begining of the number
				if (bAfterDash)
				{
					// there was a dash before -- treat as a range
					if ( str[i]=='-' )
						// nn-nn-
						return false;
					str[i] = '\0';
					m = atoi(p);
					p = NULL;
					if (n<0 && m<n)
						return false;
					if (m>nMax)
						m = nMax;
					for(n++;n<=m;++n)
						seq.insert(n);
					bAfterDash = false;
				}
			 	else
				{
					// just a number -- count it, and if current char is not a separator --
					// leave n untouched for the case of a dash
					bool bComa = ( str[i]==',' || str[i]==';' );
					bAfterDash = ( str[i]=='-' );
					str[i] = '\0';
					n = atoi(p);
					p = NULL;
					seq.insert(n);
					if (bComa)
						n = -1;
				}
			else
			{
				if ( str[i]=='-' )
				{
					if ( n<0 || bAfterDash )
						//   -n or -- situation
						return false;
					bAfterDash = true;
				}
				else if ( str[i]==',' || str[i]==';' )
				{
					if (bAfterDash)
					// n-, situation
					{
						if (n<0)
							return false;
						for (n++;n<=nMax;++n)
							seq.insert(n);
						bAfterDash = false;
					}
					n = -1;
				}
			}
		}
		else
			// unexpected symbol
			return false;
	}
	if ( bAfterDash )
	{
		// n-  situation
		if (n<0)
			return false;
		for (n++;n<=nMax;++n)
			seq.insert(n);
	}
	// sort and compress would be needed by the list, set is already sorted and
	//seq.sort();
	//seq.unique();
	return true;
}

// not quite a parser, but still... useful
static CString FormatSearch(SGnuSearch* pSearch)
{
	CString res = pSearch->m_Search;
	for (set<SHA1Hash>::iterator it = pSearch->m_setSha1.begin(); it != pSearch->m_setSha1.end(); ++it)
	{
		if (!res.empty())
			res += ' ';
		res += "sha1:";
		res += it->toStr();
	}
	return res;
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////
// command implementation
//

bool MUITextModePriv::com_help ( char *arg )
{
	register int i;
	int printed = 0;
	if ( !arg || 0==strlen(arg) )
	{
		m_pOutput->print("??header>Available comands:\n");
		m_pOutput->print("??header>==================\n");
		for (i = 0; commands[i].name; i++)
		{
			m_pOutput->print ("??name>%s\t??%s\n", commands[i].name, commands[i].doc_short);
		}
		m_pOutput->print("??message>\nAll commands accept abbrevations, e.g. `results' can be replaced\n"
		             "with `res' or even `r'\n");
		m_pOutput->print("??flag1>NEW: ??message>All commands support Unix pipes, that is output of the mutella\n"
		             "command can be redirected to the system command using standard Unix\n"
		             "pipleline syntax, e.g. `res 1-3 | grep something'\n");
		m_pOutput->print("??message>Type `help <command>' to get more detailed help on specific command\n");
		m_pOutput->print("??flag1>NEW: ??message>type `help <variable_name>' to get description of the specific\n"
		             "variable. List of mutella variables (options) can be acquired with\n"
		             "`set' command with no parameters\n");
		return true;
	}
	MProperty* pP;
	if (pP = m_pController->GetProperty(arg))
	{
		if (pP->GetPropertyDescriptionFull() && strlen(pP->GetPropertyDescriptionFull()))
			m_pOutput->print ("\n%s:	%s\n\n%s\n", arg, pP->GetPropertyDescriptionShort(), pP->GetPropertyDescriptionFull());
		else if (pP->GetPropertyDescriptionShort() && strlen(pP->GetPropertyDescriptionShort()))
			m_pOutput->print ("\n%s:	%s\n", arg, pP->GetPropertyDescriptionShort());
		else
			m_pOutput->print ("\n??message>Sorry, variable `%s' is not documented yet\n", arg);
		printed++;
	}
	else
	{
		for (i = 0; commands[i].name; i++)
		{
			if ( strncmp (arg, commands[i].name, strlen(commands[i].name) ) == 0 )
			{
				if (commands[i].doc_long)
					m_pOutput->print ("\n%s\n", commands[i].doc_long);
				else
					m_pOutput->print ("\n%s\n    %s.\n", commands[i].name, commands[i].doc_short);
				printed++;
			}
		}
	}
	if (!printed)
	{
		if ( arg && strlen(arg) )
			m_pOutput->print("??message>No commands match `%s'.  Possibilties are:\n", arg);
		for (i = 0; commands[i].name; i++)
		{
			m_pOutput->print ("%s\t", commands[i].name);
			// Print in six columns.
			if ( i%6 == 5)
				m_pOutput->print ("\n");
		}
		if ( i%6 != 0)
			m_pOutput->print("\n");
		m_pOutput->print("\n??message>List of mutella variables can be acquired with `set' command\n");
	}
	return true;
}

bool MUITextModePriv::com_exit ( char *arg )
{
	// TODO: ask for safe exit
	m_pOutput->print("??message>bye\n");
	stop(false);
	m_pController->ForceStop();
	return true;
}

bool MUITextModePriv::com_leave ( char *arg )
{
	m_pOutput->print("??message>closing UI, but mutella will go on ...\n");
	stop(false);
	return true;
}

bool MUITextModePriv::com_system(char * arg)
{
	return vcom_system(arg);
}

bool MUITextModePriv::vcom_system(char * arg)
{
	CString s = "sh -c \'";
	s += arg;
	s += "\'";
	return -1 != system(s.c_str());
}

bool MUITextModePriv::com_load(char * arg)
{
	CString s = ExpandPath(arg);
	FILE* f = fopen(s.c_str(), "r");
	if (f == NULL)
	{
		m_pOutput->print("??message>failed to open `??name>%s??message>'\n", s.c_str());
		return false;
	}
	char tmp[1024];
	char * t;
	while (!feof(f) && !ferror(f))
	{
		if (NULL!=fgets(tmp,1024,f))
		{
			tmp[1023] = '\0';
			t = StripWhite(tmp);
			if (strlen(t) && *t != '#')
				m_command_queue.push(t);
		}
	}
	fclose(f);
	return true;
}

bool parse_num_param(CString& s, int& num, int nStatrAt = 0)
{
	// find the firs 'token' -- something isolated by the spaces
	int nBegin = nStatrAt;
	while (_isblank(s[nBegin]) && nBegin<s.length())
		++nBegin;
	if (nBegin == s.length())
		return false;
	int nEnd = nBegin+1;
	while (!_isblank(s[nEnd]) && nEnd<s.length())
		nEnd++;
	char* tmp = (char*) alloca(nEnd-nBegin+2);
	ASSERT(tmp);
	strncpy(tmp, s.substr(nBegin,nEnd-nBegin).c_str(), nEnd-nBegin+1);
	if (asc2num( tmp, &num))
	{
		s = s.substr(0, nStatrAt) + s.substr(nEnd);
		return true;
	}
	return false;
}

bool MUITextModePriv::com_find(char * arg)
{
	if ( arg==NULL || strlen(arg)==0 )
		return com_list("");
	CString search = StripWhite(arg);
	// check for options
	int nSizeMode = LIMIT_NONE;
	int nSize = 0;
	int nPos = 0;
	// size filters
	if ( (nPos=search.find("min:")) >= 0 )
	{
		if (!parse_num_param(search, nSize, nPos+4))
		{
			m_pOutput->print("??error>find: syntax error after 'min:'\n");
			return false;
		}
		nSizeMode = LIMIT_MORE;
		search = search.substr(0,nPos)+search.substr(nPos+4);
	}
	else if ( (nPos=search.find("size:")) >= 0 )
	{
		if (!parse_num_param(search, nSize, nPos+5))
		{
			m_pOutput->print("??error>find: syntax error after 'size:'\n");
			return false;
		}
		nSizeMode = LIMIT_EXACTLY;
		search = search.substr(0,nPos)+search.substr(nPos+5);
	}
	else if ( (nPos=search.find("around:")) >= 0 )
	{
		if (!parse_num_param(search, nSize, nPos+7))
		{
			m_pOutput->print("??error>find: syntax error after 'around:'\n");
			return false;
		}
		nSizeMode = LIMIT_APPROX;
		search = search.substr(0,nPos)+search.substr(nPos+7);
	}
	// autoget flag "/autoget"
	bool bAutoGet = false;
	if ( (nPos=search.find("/autoget")) >= 0 )
	{
		bAutoGet = true;
		search = search.substr(0,nPos)+search.substr(nPos+8);
	}
	else if ( (nPos=search.find("/auto")) >= 0 )
	{
		bAutoGet = true;
		search = search.substr(0,nPos)+search.substr(nPos+5);
	}
	// add a query to the search list
	if (m_pController->AddSearchUsr(search.c_str(), nSize, nSizeMode, bAutoGet))
		return true;
	m_pOutput->print("??error>adding search `??name>%s??error>' failed.\nprobably there are similar searches already \nor maximum number of searches has been reached.\n", search.c_str());
	return false;
}

bool MUITextModePriv::com_edit(char * arg)
{
	CString sArg = StripWhite(arg);
	// check if we have 2 parameters and extract the first one
	int nPos = sArg.find(' ');
	if (nPos<0)
	{
		m_pOutput->print("??error>'edit' requires 2 parameters\n");
		return false;
	}

	DWORD dwID = 0;
	int nSNum = atol(sArg.substr(0, nPos).c_str());
	if (nSNum>0 && nSNum<=m_vecSearchIDs.size())
		dwID = m_vecSearchIDs[nSNum-1];
	if (dwID==0)
	{
		m_pOutput->print("??error>'edit' requires valid search ID\n");
		return false;
	}
	CString sSearch = StripWhite(sArg.substr(nPos+1));
	if (!sSearch.length() || !m_pController->ModifySearch(dwID, sSearch.c_str()))
	{
		m_pOutput->print("??error>Editing search failed, may be the ID is invalid or the search string is too short?\n");
		return false;
	}
	//m_pOutput->print("??message>Ok\n");
	return true;
}

struct resParams{
	MUITextModePriv* pThis;
	CString  sFilter;
	int      nMaxResultsDisplayed;
	dwordSet seq;
	dwordSet retSet;
};

CString makePrintable(CString str)
{
	for (int i=0; i<str.length(); ++i)
		if (BYTE(str[i])<32                ||
			BYTE(str[i])>127 && BYTE(str[i])<160 )
				str[i]='#';
	return str;
}

int compRes(const void* p1, const void* p2)
{
	Result* pr1 = *(Result**)p1;
	Result* pr2 = *(Result**)p2;
	// sord decending by size, than desending by speed, than accending by name
	if (pr1->Size>pr2->Size)
		return -1;
	if (pr1->Size<pr2->Size)
		return 1;
	if (pr1->Speed>pr2->Speed)
		return -1;
	if (pr1->Speed<pr2->Speed)
		return 1;
	if (pr1->Name>pr2->Name)
		return 1;
	if (pr1->Name<pr2->Name)
		return -1;
	return 0;
}

bool resCallback(void* pData, SGnuSearch* pSearch)
{
	resParams* pRP = (resParams*) pData;
	if ( (pSearch->m_nHits != 0 && pRP->sFilter.length()!=0 && QueryMatch(pSearch->m_Search, pRP->sFilter)) ||
		 (pSearch->m_nHits != 0 && pRP->sFilter.length()==0 && pRP->seq.size()==0) ||
		 (pRP->seq.end() != pRP->seq.find(pSearch->m_dwID)) )
	{
		pRP->retSet.insert(pSearch->m_dwID);
	}
	return true;
}

int compGroups(const void* p1, const void* p2)
{
	ResultGroup* pg1 = *(ResultGroup**)p1;
	ResultGroup* pg2 = *(ResultGroup**)p2;
	// sort decending by size, than desending by speed, than acending by name
	if (pg1->Size>pg2->Size)
		return -1;
	if (pg1->Size<pg2->Size)
		return 1;
	if (pg1->AvgSpeed>pg2->AvgSpeed)
		return -1;
	if (pg1->AvgSpeed<pg2->AvgSpeed)
		return 1;
	if (pg1->Name>pg2->Name)
		return 1;
	if (pg1->Name<pg2->Name)
		return -1;
	return 0;
}

static CString GetKnownFlagStr(int nFlags)
{
	if (!nFlags)
		return "New";
	CString s;
	if (MFT_LibraryActual & nFlags)
		s = "Lib";
	else if (MFT_LibraryStored & nFlags)
		s = "LibOld";
	if (MFT_InDownload & nFlags)
	{
		if (s.length())
			s += "+";
		s += "Dwnl";
	}
	else if (MFT_Attempted & nFlags)
	{
		if (s.length())
			s += "+";
		s += "Att";
	}
	if (MFT_Partial & nFlags)
	{
		if (s.length())
			s += "+";
		s += "Part";
	}
	if (MFT_User & nFlags)
	{
		if (s.length())
			s += "+";
		s += "Usr";
	}
	//
	if (MFR_Possible & nFlags)
		s = "?" + s + "?";
	else if (MFR_Probable & nFlags)
		s = "~" + s + "~";
	else if (MFR_Exact & nFlags)
		s = "=" + s + "=";
	//
	if (s.empty())
		return "Unknown";
	return s;
}

bool MUITextModePriv::com_results(char * arg)
{
	resParams rp;
	rp.pThis = this;
	if (strlen(arg)==0 ||
	    strpbrk(arg, "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvqxyz") )
	{
		// use string filter approach
		rp.sFilter = arg;
	}
	else
	{
		// try numbersint
		intSet is;
		if (!ParseNumRanges(is, arg, m_vecSearchIDs.size()))
		{
			m_pOutput->print("??error>results: failed to parse parameters\n");
			return false;
		}
		// transform numbers into IDs
		for (intSet::iterator it=is.begin();it!=is.end();++it)
		{
			if (*it<=m_vecSearchIDs.size())
				rp.seq.insert(m_vecSearchIDs[(*it)-1]);
			else
			{
				m_pOutput->print("??error>results: no searches with ID = %d\n", *it);
			}
		}
	}
	rp.nMaxResultsDisplayed = 1000;
	MProperty* pMaxRes = m_pController->GetProperty("MaxResultsDisplayed");
	if (pMaxRes)
		rp.nMaxResultsDisplayed = pMaxRes->GetIntValue();
	m_pController->ForEachSearch((void*)&rp, resCallback);
	//
	m_nCounter = 1;
	m_vecResIDs.clear();
	m_pOutput->print("Current search results, matching your criteria\n");
	for (dwordSet::iterator it = rp.retSet.begin(); it!=rp.retSet.end(); ++it)
	{
		if (rp.nMaxResultsDisplayed < m_nCounter)
			break;
		//
		SGnuSearch gs;
		vector<Result> rv;
		vector<ResultGroup> gv;
		if (!m_pController->GetSearchByID(*it,gs,rv,gv))
			continue;
		//
		m_pOutput->print("  ??header>QUERY:??\"??name>%s??header>\" \t HITS:??num>%d\n", FormatSearch(&gs).c_str(), gs.m_nHits);
		if (m_bShowGroups)
		{
			// now copy result groups to our array so that we could sort it
			int nCount = gv.size();
			ResultGroup** arrpRGps = new ResultGroup*[nCount];
			ASSERT(arrpRGps);
			for (int i = 0; i<nCount; ++i)
			{
				arrpRGps[i]= &gv[i];
			}
			// we dont need a quick sort for a list of 10-20 results
			// but we have it so we use it
			qsort(arrpRGps, nCount, sizeof(Result*),compGroups); // TODO: different ways of sorting
			for (int i=0; i<nCount; i++)
			{
				ResultGroup* pgrp=arrpRGps[i];
				m_vecResIDs.push_back(pgrp->dwID);
				if ( pgrp->ResultSet.size() > 1 )
				{
					m_pOutput->print("??id>%3d)?? `??name>%s??' ??size>%s??\n    LOCATIONS:??num>%-3d??  avg.speed:??speed>%-10s??  known:??known>%s\n",
						m_vecResIDs.size(), makePrintable(pgrp->Name).c_str(), FormatSize(pgrp->Size).c_str(),
						pgrp->ResultSet.size(), GetSpeedString(pgrp->AvgSpeed).c_str(),
						GetKnownFlagStr(pgrp->nKnownFlags).c_str());
				}
				else if (pgrp->ResultSet.size() == 1)
				{
					Result* pres= &rv[*pgrp->ResultSet.begin()];
					m_pOutput->print("??id>%3d)?? `??name>%s??' ??size>%s?? known:??known>%s\n",
						m_vecResIDs.size(), makePrintable(pgrp->Name).c_str(), FormatSize(pres->Size).c_str(), GetKnownFlagStr(pgrp->nKnownFlags).c_str());
				}
				if (pgrp->ResultSet.size()<4)
				{
					int j = 0;
					for ( set<int>::iterator iti = pgrp->ResultSet.begin(); iti != pgrp->ResultSet.end(); ++iti)
					{
						Result* pres= &rv[*iti];
						if (pres->Sha1.isValid())
							m_pOutput->print("     sha1: ??sha1>%s\n", pres->Sha1.toStr().c_str());
						for ( int k = 0; k < pres->Extra.size(); k++)
						{
							if (pres->Extra[k].size())
								m_pOutput->print("    extra: ??extra>%s\n", makePrintable(pres->Extra[k]).c_str());
						}
						m_pOutput->print("    ??ip>%16s:%-4d?? speed:??speed>%-12s ??flag1>%-12s?? time:??time>%s\n",
							Ip2Str(pres->Host).c_str(), pres->Port, GetSpeedString(pres->Speed).c_str(), pres->Vendor.c_str(),
							FormatTime(xtime() - pres->ChangeTime).c_str() );
						if (++j>2) break;
					}
				}
			}
			delete [] arrpRGps;
		}
		else
		{
			// now copy references to our array so that we could sort it
			int nCount = rv.size();
			Result** arrpRes = new Result*[nCount];
			ASSERT(arrpRes);
			for (int i=0; i<nCount; i++)
			{
				arrpRes[i]=&rv[i];
			}
			// we dont need a quick sort for a list of 10-20 results
			// but we have it so we use it
			qsort(arrpRes, nCount, sizeof(Result*),compRes); // TODO: different ways of sorting
			for (int i=0; i<nCount; i++)
			{
				Result* pres=arrpRes[i];
				m_vecResIDs.push_back(pres->dwID);
				m_pOutput->print(" ??id>%d)?? `??name>%s??' ??size>%s?? REF:??num>%d??\n    IP:??ip>%s:%d?? \tSPEED:??speed>%s \t??flag1>%s\n",
					m_vecResIDs.size(), makePrintable(pres->Name).c_str(), FormatSize(pres->Size).c_str(), pres->FileIndex,
					Ip2Str(pres->Host).c_str(), pres->Port, GetSpeedString(pres->Speed).c_str(), pres->Vendor.c_str());
			}
			delete [] arrpRes;
		}
	}
	m_pOutput->print("count: ??num>%d\n", m_nCounter-1);
	return true;
}

struct clrParams{
	MUITextModePriv* pThis;
	CString  sFilter;
	int      nMaxResultsDisplayed;
	dwordSet seq;
	dwordSet retSet;
};


bool clearCallback(void* pData, SGnuSearch* pSearch)
{
	clrParams* pRP = (clrParams*) pData;
	if (pRP->nMaxResultsDisplayed < pRP->pThis->m_nCounter)
		return false;
	if ( (pRP->sFilter.length()!=0 && QueryMatch(pSearch->m_Search, pRP->sFilter)) ||
		 (pRP->sFilter.length()==0 && pRP->seq.size()==0) ||
		 (pRP->seq.end() != pRP->seq.find(pSearch->m_dwID)) )
	{
		pRP->pThis->m_pOutput->print("\"??name>%s??\"\n", pSearch->m_Search.c_str());
		pRP->retSet.insert(pSearch->m_dwID);
	}
	return true;
}

bool MUITextModePriv::com_clear(char * arg)
{
	if (strlen(arg)==0)
	{
		m_pOutput->print("??error>clear: argument is required\n");
		return false;
	}
	clrParams cp;
	cp.pThis = this;
	if (strpbrk(arg, "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvqxyz"))
	{
		// use string filter approach
		cp.sFilter = arg;
	}
	else
	{
		intSet seq;
		// try numbersint
		if (!ParseNumRanges(seq, arg, m_vecSearchIDs.size()))
		{
			m_pOutput->print("??error>clear: failed to parse parameters\n");
			return false;
		}
		// convert to IDs
		for(intSet::iterator it = seq.begin(); it != seq.end(); it++)
		{
			if (*it <= m_vecSearchIDs.size())
				cp.seq.insert(m_vecSearchIDs[(*it)-1]);
		}
	}
	cp.nMaxResultsDisplayed = 1000;
	MProperty* pMaxRes = m_pController->GetProperty("MaxResultsDisplayed");
	if (pMaxRes)
		cp.nMaxResultsDisplayed = pMaxRes->GetIntValue();
	cp.retSet.clear();
	m_nCounter = 1;
	m_pOutput->print("??message>Clearing search results of following searches\n");
	m_pController->ForEachSearch((void*)&cp, clearCallback);
	if (!cp.retSet.empty())
	{
		//pSearch->Clear();
		for (dwordSet::iterator it = cp.retSet.begin();it != cp.retSet.end(); ++it)
			m_pController->ClearSearchByID(*it);
	}
	return true;
}

struct listParams{
	MUITextModePriv* pThis;
	CString sFilter;
	bool bNoEmpty;
	bool bNoAutoSearches;
};

bool listCallback(void* pData, SGnuSearch* pSearch)
{
	listParams* pLP = (listParams*) pData;
	if ( (pLP->sFilter.length()==0 || QueryMatch(pSearch->m_Search, pLP->sFilter)) &&
		 (!pLP->bNoEmpty || pSearch->m_nHits) &&
		 (!pLP->bNoAutoSearches || !pSearch->IsAutomatic()) )
	{
		pLP->pThis->m_vecSearchIDs.push_back(pSearch->m_dwID);
		ASSERT(pLP->pThis->m_vecSearchIDs.size()==pLP->pThis->m_nCounter);
		//
		pLP->pThis->m_pOutput->print("??id>%2d) ??name>`%s'\n    ", pLP->pThis->m_nCounter, FormatSearch(pSearch).c_str());
		if (pSearch->m_SizeFilterMode == LIMIT_NONE)
			pLP->pThis->m_pOutput->print("??flag1>no size filter ");
		else if (pSearch->m_SizeFilterMode == LIMIT_MORE)
			pLP->pThis->m_pOutput->print("MIN:??size>%-11s",FormatSize(pSearch->m_SizeFilterValue).c_str());
		else if (pSearch->m_SizeFilterMode == LIMIT_EXACTLY)
			pLP->pThis->m_pOutput->print("SIZE:??size>%-10d",pSearch->m_SizeFilterValue);
		else if (pSearch->m_SizeFilterMode == LIMIT_LESS)
			pLP->pThis->m_pOutput->print("MAX:??size>%-11s",FormatSize(pSearch->m_SizeFilterValue).c_str());
		else if (pSearch->m_SizeFilterMode == LIMIT_APPROX)
			pLP->pThis->m_pOutput->print("AROUND:??size>%-11s",FormatSize(pSearch->m_SizeFilterValue).c_str());
		if (pSearch->m_bAutoget)
			pLP->pThis->m_pOutput->print("??flag2>  AUTOGET");
		else
			pLP->pThis->m_pOutput->print("         ");
		if (pSearch->m_nHits)
			pLP->pThis->m_pOutput->print("  HITS:??num>%d\n", pSearch->m_nHits);
		else
			pLP->pThis->m_pOutput->print("??flag1>  NO HITS\n");
		pLP->pThis->m_nCounter++;
	}
	return true;
}

bool MUITextModePriv::com_list(char * arg)
{
	m_nCounter = 1;
	listParams lp;
	lp.pThis = this;
	lp.sFilter = arg;
	lp.bNoEmpty = false;
	lp.bNoAutoSearches = false;
	// switches
	MakeLower(lp.sFilter);
	lp.sFilter = " " + lp.sFilter + " ";
	int n;
	n = lp.sFilter.find(" -empty ");
	if (0<=n)
	{
		lp.sFilter = lp.sFilter.substr(0, n)+ lp.sFilter.substr(n+7);
		lp.bNoEmpty = true;
	}
	n = lp.sFilter.find(" -auto ");
	if (0<=n)
	{
		lp.sFilter = lp.sFilter.substr(0, n)+ lp.sFilter.substr(n+6);
		lp.bNoAutoSearches = true;
	}
	lp.sFilter = StripWhite(lp.sFilter);
	//
	m_vecSearchIDs.clear();
	//
	m_pOutput->print("??message>Searches in progress, matching your criteria\n");
	m_pController->ForEachSearch((void*)&lp, listCallback);
	m_pOutput->print("??message>count: ??num>%d\n", m_nCounter-1);
	return true;
}

bool MUITextModePriv::com_ls(char * arg)
{
	if (NULL!=strstr(arg, "-empty"))
		return com_list(arg);
	char* tmp = (char*) alloca(strlen(arg)+16); // to lazy to count letters
	ASSERT(tmp);
	sprintf(tmp, "%s -empty", arg);
	return com_list(tmp);
}

bool MUITextModePriv::com_delete(char * arg)
{
	intSet seq;
	if (!ParseNumRanges(seq, arg, m_vecSearchIDs.size()))
	{
		m_pOutput->print("??error>delete: failed to parse parameters\n");
		return false;
	}
	for(intSet::iterator it = seq.begin(); it != seq.end(); it++)
	{
		if (*it<=m_vecSearchIDs.size())
			m_pController->RemoveSearchByID(m_vecSearchIDs[(*it)-1], false);
	}
	return true;
}

bool MUITextModePriv::com_erase(char * arg)
{
	intSet seq;
	if (!ParseNumRanges(seq, arg, m_vecSearchIDs.size()))
	{
		m_pOutput->print("??error>erase: failed to parse parameters\n");
		return false;
	}
	for(intSet::iterator it = seq.begin(); it != seq.end(); it++)
	{
		if (*it<=m_vecSearchIDs.size())
			m_pController->RemoveSearchByID(m_vecSearchIDs[(*it)-1], true);
	}
	return true;
}

static const char* FormatPeerMode(SGnuNode* pNode)
{
	if (pNode->m_nStatus != SOCK_CONNECTED)
		return "-";
	switch (pNode->m_nPeerMode) {
		case CM_NORMAL:    return "N";
		case CM_LEAF:      return "L";
		case CM_ULTRAPEER: return "U";
	}
	return "?";
}

bool infoConnCallback(void* pData, SGnuNode* pNode)
{
	MUITextModePriv* pThis = (MUITextModePriv*) pData;
	pThis->m_vecConnIDs.push_back(pNode->m_dwID);
	ASSERT(pThis->m_vecConnIDs.size()==pThis->m_nCounter);
	pThis->m_pOutput->print("??id>%2d)?? ??ip>%18s:%-5d??  ??status>%s??  ??speed>%5s??:??speed>%-5s?? ??num>%5s??/??num>%-6s?? ??percent>%3d%%??   ??time>%6s??   ??time>%6s\n",
			pThis->m_nCounter, pNode->m_sHost.c_str(), pNode->m_nPort,
			FormatPeerMode(pNode),
			FormatSize((int)pNode->m_dStatisticsRate[NR_BYTE_IN]).c_str(),FormatSize((int)pNode->m_dStatisticsRate[NR_BYTE_OUT]).c_str(),
			FormatNumber(pNode->m_dwFriendsTotal).c_str(),
			FormatKSizeLL(pNode->m_llLibraryTotal).c_str(),
			pNode->m_nEfficiency,
			FormatTime(xtime()-pNode->m_nPeerUptime).c_str(),
			FormatTime(xtime()-pNode->m_nUptime).c_str());
	//m_pOutput->print("      version 0.%d\n",pNode->m_dwVersion);
	pThis->m_nCounter++;
	return true;
}

bool infoUploadCallback(void* pData, SGnuUpload* pUpload)
{
	MUITextModePriv* pThis = (MUITextModePriv*) pData;
	pThis->m_vecTransIDs.push_back(pUpload->m_dwID);
	ASSERT(pThis->m_vecTransIDs.size()==pThis->m_nCounter);
	pThis->m_pOutput->print("??id>%2d)??name> %s\n    ??ip>%16s:%-4d??  ??percent>%5s  ??speed>%5s/s  ",
		pThis->m_nCounter,
		InsertElypsis(pUpload->m_sFileName, 75).c_str(),
		Ip2Str(pUpload->m_ipHost).c_str(), pUpload->m_nPort,
		FormatPercent(pUpload->m_nBytesCompleted,pUpload->m_nFileLength).c_str(),
		FormatSize((int)pUpload->m_dRate).c_str());
	if (pUpload->m_dRate > 0)
		pThis->m_pOutput->print("ETA:??time>%s  ",
			FormatTime((int) ((pUpload->m_nFileLength-pUpload->m_nBytesCompleted)/pUpload->m_dRate)).c_str());
	pThis->m_pOutput->print("??status>%s\n",
		SGnuUpload::GetErrorString(pUpload->m_nError));
	pThis->m_nCounter++;
	return true;
}

/*struct SGnuDownloadItem
{
	// Node info
	IP		  ipHost;
	WORD      nPort;
	CString	  sVendor;
	// Download info
	int       nStatus;
	int       nLastStatusChange;
	int       nLastActive;
	int       nDisconnectReason;
	DWORD     dwRate;
	//
	bool      bServerIsUp;   // true if we recieved anything from it ever
	int       nConErrors;    // counter of consequitive connection errors
	int       nPushTimeouts; // counter of consequitive push timeouts
};

struct SGnuDownload
{
	// Download Properties
	CString m_sName;
	DWORD   m_dwSearchID;
	// File info
	DWORD   m_dwFileLength;
	DWORD   m_dwBytesCompleted;
	// source addresses, etc
	double m_dRate;          // real download rate in bytes per second
	//
	int    m_nActiveConnections;
	int    m_nCreationTime;
	// status
	int    m_nLastDisconnectReason;
	int    m_nStatus;
	// ID for UI
	DWORD  m_dwID;
	// Download items -- individual hosts
	DownloadItemVec m_vecItems;
	// few static methods
	static LPCSTR GetErrorString(int nCode);
	static LPCSTR GetStatusString(int nStatus);
};*/


bool infoDownloadCallback(void* pData, SGnuDownload* pDownload)
{
	MUITextModePriv* pThis = (MUITextModePriv*) pData;
	pThis->m_vecTransIDs.push_back(pDownload->m_dwID);
	ASSERT(pThis->m_vecTransIDs.size()==pThis->m_nCounter);
	pThis->m_pOutput->print("??id>%2d)??name> %s\n", pThis->m_nCounter, InsertElypsis(pDownload->m_sName,75).c_str());
	if (pDownload->m_nStatus != TRANSFER_CLOSED && pDownload->m_nStatus != TRANSFER_CLOSING)
	{
		pThis->m_pOutput->print("    ??status>%-7s  A:??num>%2d/%-2d ",
				SGnuDownload::GetStatusString(pDownload->m_nStatus),
				pDownload->m_vecItems.size(),
				pDownload->m_nBadResults);
		pThis->m_pOutput->print("??size>%5s/%-5s  ??percent>%5s  ",
				FormatSize(pDownload->m_dwBytesCompleted).c_str(),
				FormatSize(pDownload->m_dwFileLength).c_str(),
				FormatPercent(pDownload->m_dwBytesCompleted,pDownload->m_dwFileLength).c_str());
		if (pDownload->m_dRate > 0)
			pThis->m_pOutput->print("??speed>%5s/s?? ETA:??time>%s\n",
					FormatSize((int)pDownload->m_dRate).c_str(),
					FormatTime((int) ((pDownload->m_dwFileLength-pDownload->m_dwBytesCompleted)/pDownload->m_dRate)).c_str());
		else
			pThis->m_pOutput->print("\n");
			/*pThis->m_pOutput->print("??status>%s\n",
					SGnuDownload::GetErrorString(pDownload->m_nLastDisconnectReason));*/
		//
		if (pDownload->m_vecItems.size())
		{
		  for (int i=0; i<min((int)pDownload->m_vecItems.size(),3); ++i)
		  {
			pThis->m_pOutput->print("    ??status>%-7s ", SGnuDownload::GetStatusString(pDownload->m_vecItems[i].nStatus));
			pThis->m_pOutput->print("??ip>%16s:%-5d??  %-10s  ??speed>%5s/s??  ConErr:%-2d  PushTO:%-2d\n",
					Ip2Str(pDownload->m_vecItems[i].ipHost).c_str(),
					pDownload->m_vecItems[i].nPort,
					pDownload->m_vecItems[i].bServerIsUp ? "ServerIsUp" : "NoReplyYet",
					FormatSize(pDownload->m_vecItems[i].dwRate).c_str(),
					pDownload->m_vecItems[i].nConErrors,
					pDownload->m_vecItems[i].nPushTimeouts
					);
		  }
		}
	}
	else
		pThis->m_pOutput->print("    INACTIVE , reason: ??status>%s\n", SGnuDownload::GetErrorString(pDownload->m_nLastDisconnectReason));

	pThis->m_nCounter++;
	return true;
}

bool MUITextModePriv::com_info(char * arg)
{
	bool bNet=true, bConn=true, bUpl=true, bDownl=true, bRealtime = false;
	// parse arg here
	if (arg && strlen(arg))
	{
		bNet=bConn=bUpl=bDownl=false; // start from reseting everything
		CString sArg = StripWhite(arg);
		CString sPar;
		int nPos;
		sArg += " ";// to make parsing simplier
		while (sArg.length()>1) /* this is because of the space at the tail */
		{
			nPos = sArg.find(" ");
			sPar = sArg.substr(0,nPos);
			sArg = StripWhite(sArg.substr(nPos+1));
			sArg += " ";// ah, yeah im to lazy
			//
			if (0==strncmp("network", sPar.c_str(), sPar.length()))
				bNet = true;
			if (0==strncmp("connections", sPar.c_str(), sPar.length()))
				bConn = true;
			if (0==strncmp("uploads", sPar.c_str(), sPar.length()))
				bUpl = true;
			if (0==strncmp("downloads", sPar.c_str(), sPar.length()))
				bDownl = true;
			if (0==strncmp("transfers", sPar.c_str(), sPar.length()))
			{
				bUpl = true;
				bDownl = true;
			}
			if (0==strncmp("loop", sPar.c_str(), sPar.length()))
				bRealtime = true;
		}
		// here we have a little problem: if 'i r' was specified this will
		// reset all the flags -- just fix it manually now
		if (bRealtime && !bNet && !bConn && !bUpl &&!bDownl)
			bNet=bConn=bUpl=bDownl=true;
	}
	//
	do {
		if (bNet)
		{
			m_pOutput->print("??header>host mode: ??name>%s??  ??header>uptime: ??time>%s\n\n",
					m_pController->IsUltrapeer() ? "ULTRAPEER" : "LEAF",
					FormatTime(xtime() - m_pController->GetClientStartTime()).c_str() );
			m_pOutput->print("??header>network horizon:\n");
			m_pOutput->print("??header>----------------\n");
			int nHosts, nSharingHosts, nFiles, nSize;
			m_pController->GetNetStats(nHosts, nSharingHosts, nFiles, nSize);
			//m_pOutput->print("%d %d %d\n",nHosts, nFiles, nSize);
			m_pOutput->print("ReachebleHosts: ??num>%s??  SharingHosts: ??num>%s??  Files: ??num>%s??  Capacity:??num> %s\n\n",
					FormatSize(nHosts).c_str(),
					FormatSize(nSharingHosts).c_str(),
					FormatSize(nFiles).c_str(),
					FormatMSize(nSize).c_str());

			m_pOutput->print("??header>caches:\n");
			m_pOutput->print("??header>-------\n");
			int nCacher, nUltraCatcher, nStore, nGWeb;
			m_pController->GetCacheStats(nCacher, nUltraCatcher, nStore, nGWeb);
			m_pOutput->print("CachedHosts: ??num>%d??  UltraPeers: ??num>%d??  PersistentHosts: ??num>%d??  WebCacheURLs: ??num>%d\n\n",nCacher, nUltraCatcher, nStore, nGWeb);

			m_pOutput->print("??header>sockets:\n");
			m_pOutput->print("??header>--------\n");
			int nConn, nUpl, nDownlA, nDownlT;
			m_pController->GetConnStats(nConn, nUpl, nDownlA, nDownlT);
			m_pOutput->print("Network: ??num>%d??  Uploads: ??num>%d??  Downloads: ??num>%d?? / ??num>%d\n\n",nConn, nUpl, nDownlA, nDownlT);
			m_pOutput->print("??header>bandwidth:\n");
			m_pOutput->print("??header>----------\n");
			m_pController->GetBandwidthStats(nConn, nUpl, nDownlA);
			m_pOutput->print("Network:??speed>%s/s??  Uploads:??speed>%s/s??  Downloads:??speed>%s/s??  Total:??speed>%s/s??\n\n",
					FormatSize(nConn).c_str(),
					FormatSize(nUpl).c_str(),
					FormatSize(nDownlA).c_str(),
					FormatSize(nConn+nUpl+nDownlA).c_str());
		}
		//
		if (bConn)
		{
			m_pOutput->print("??header>gnutella network connections:\n");
			m_pOutput->print("??header>-----------------------------\n");
			m_pOutput->print("??header>no.??|           ??header>address??:??header>port?? | ??header>T?? |  ??header>rate i:o??  | ??header>horizon??  | ??header>eff.?? | ??header>uptime?? | ??header>c.time??\n");
			m_nCounter = 1;
			m_vecConnIDs.clear();
			m_pController->ForEachConnection((void*)this, infoConnCallback);
			m_pOutput->print("total connections: ??num>%d??  rate: [??speed>%5s??|??speed>%-5s??]/sec\n\n",
				   m_nCounter-1,
				   FormatSize(m_pController->GetConnRateRecv()).c_str(),
				   FormatSize(m_pController->GetConnRateSend()).c_str());
		}
		//
		m_nCounter = 1;
		if (bUpl)
		{
			m_pOutput->print("??header>uploads:\n");
			m_pOutput->print("??header>--------\n");
			m_vecTransIDs.clear();
			m_pController->ForEachUpload((void*)this, infoUploadCallback);
		}
		//
		if (bDownl)
		{
			m_pOutput->print("??header>downloads:\n");
			m_pOutput->print("??header>----------\n");
			if (!bUpl)
				m_vecTransIDs.clear();
			m_pController->ForEachDownload((void*)this, infoDownloadCallback);
		}
		if (bUpl && bDownl)
			m_pOutput->print("total transfers: ??num>%d\n", m_nCounter-1);
		else if (bUpl)
			m_pOutput->print("total uploads: ??num>%d\n", m_nCounter-1);
		else if (bDownl)
			m_pOutput->print("total downloads: ??num>%d\n", m_nCounter-1);
		if (bRealtime)
		{
			bRealtime = on_realtime();
		}
	} while (bRealtime);
	return true;
}

bool MUITextModePriv::on_realtime()
{
	return false;
}

bool MUITextModePriv::com_hosts(char * arg)
{
	list<Node> hosts;
	m_pController->GetNodeCache(hosts);
	if (hosts.size())
	{
		m_pOutput->print("??header>discovered hosts\n");
		m_pOutput->print("??header>----------------\n");
		for (list<Node>::iterator it = hosts.begin(); it != hosts.end(); ++it)
		{
			m_pOutput->print("??ip>%16s:%-4d ??num>%6d  ??size>%5s  ??speed>%s\n",
				Ip2Str(it->Host).c_str(), it->Port,
				it->ShareCount,
				FormatKSize(it->ShareSize).c_str(),
				GetSpeedString(it->Speed).c_str());
		}
		m_pOutput->print("total: ??num>%d\n", hosts.size());
	}
	else
	{
		m_pOutput->print("??message>hosts cache is empty\n");
	}
}

bool MUITextModePriv::com_close(char * arg)
{
	intSet seq;
	int nConn, nUpl, nDownlA, nDownlT;
	m_pController->GetConnStats(nConn, nUpl, nDownlA, nDownlT);
	if (!ParseNumRanges(seq, arg, nConn))
	{
		m_pOutput->print("??error>close: failed to parse parameters\n");
		return false;
	}
	for(intSet::iterator it = seq.begin(); it != seq.end(); it++)
	{
		if (m_vecConnIDs.size()<*it)
			break;
		m_pController->CloseConnectionByID(m_vecConnIDs[(*it)-1]);
	}
	return true;
}

bool MUITextModePriv::stop_help(LPCSTR szCom,bool bDelPart, char * arg)
{
	intSet seq;
	int nConn, nUpl, nDownlA, nDownlT;
	m_pController->GetConnStats(nConn, nUpl, nDownlA, nDownlT);
	if (!ParseNumRanges(seq, arg, nUpl+nDownlT))
	{
		m_pOutput->print("??error>%s: failed to parse parameters\n", szCom);
		return false;
	}
	for(intSet::iterator it = seq.begin(); it != seq.end(); it++)
	{
		if (m_vecTransIDs.size()<*it)
			break;
		if ((*it)>0)
			m_pController->RemoveTransferByID(m_vecTransIDs[(*it)-1], bDelPart);
	}
	return true;
}

bool MUITextModePriv::com_stop(char * arg)
{
	return stop_help("stop", false, arg);
}

bool MUITextModePriv::com_kill(char * arg)
{
	return stop_help("kill", true, arg);
}

bool MUITextModePriv::com_open(char * arg)
{
	int nPort = 6346;
	arg = StripWhite ( arg );
	char* pPort = strchr(arg,':');
	if (pPort == NULL)
		pPort = strchr(arg,' ');
	if ( pPort )
	{
		*pPort='\0';
		nPort = atoi(pPort+1);
		if (nPort<1024)
			nPort = 6346;
	}
	if (strlen(arg))
	{
		m_pOutput->print("??message>opening connection to '??ip>%s??message>' port ??num>%d\n", arg, nPort);
		m_pController->OpenConnection(arg, nPort);
		return true;
	}
	return false;
}

bool MUITextModePriv::com_set(char * arg)
{
	return set_helper(m_pController->GetPropertyContainer(), SET_SET, arg);
}

bool MUITextModePriv::com_set_add(char * arg)
{
	return set_helper(m_pController->GetPropertyContainer(), SET_ADD, arg);
}

bool MUITextModePriv::com_set_remove(char * arg)
{
	return set_helper(m_pController->GetPropertyContainer(), SET_REMOVE, arg);
}

bool MUITextModePriv::com_color(char * arg)
{
	return vcom_color(arg);
}

bool MUITextModePriv::set_helper(MPropertyContainer* pPC, SetMode mode, char * arg)
{
	//
	char tmp[1024];
	CString sArg = StripWhite(arg);
	sArg += ' ';
	int nSpacePos = sArg.find(" ");
	CString sPropName = sArg.substr(0,nSpacePos);
	CString sPropVal = StripWhite(sArg.substr(nSpacePos+1));
	if (sPropName.length())
	{
		MProperty* pP = pPC->FindProperty(sPropName.c_str());
		if (pP && (mode == SET_SET || pP->IsSet()))
		{
			if (sPropVal.length())
			{
				if (!is_set_permitted(sPropName))
				{
					m_pOutput->print("??error>error: set operation is no permitted\n");
					return false;
				}
				//
				strncpy(tmp,sPropVal.c_str(),1024);
				switch (mode)
				{
					case SET_SET:    if (pP->SetFromStr(tmp)) return true;
						break;
					case SET_ADD:    if (pP->GetSet()->InsertStr(tmp)) return true;
						break;
					case SET_REMOVE: if (pP->GetSet()->RemoveStr(tmp)) return true;
						break;
				}
				m_pOutput->print("??error>failed to parse `%s'\n", sPropVal.c_str());
				return false;
			}
			else
			{
		        m_pOutput->print("??variable>%s?? = ??value>%s\n", pP->GetPropertyName(), pP->Format().c_str());
			}
			return true;
		}
		if (mode == SET_SET)
			m_pOutput->print("??error>there is no `%s' variable\n", sPropName.c_str());
		else
			m_pOutput->print("??error>there is no `%s' variable of type 'list'\n", sPropName.c_str());
		return false;
	}
	//
	set<CString> propset;
	MPropertyContainer::iterator it;
	for (it = pPC->begin(); it!= pPC->end(); it++)
	{
		if (mode == SET_SET || (pPC->GetProperty(it)->IsSet()))
			propset.insert(pPC->GetPropertyName(it));;
	}
	int s = propset.size();
	MPropertyContainer::sec_iterator its;
	for (its=pPC->sec_begin(); its!= pPC->sec_end(); its++)
	{
		MPropertySection* pS = pPC->GetSection(its);
		bool bSectionPrinted = false;
		for (it = pS->begin();it!=pS->end();it++)
		{
			MProperty* pP = pPC->GetProperty(it);
			if (propset.find(pPC->GetPropertyName(it))!=propset.end())
			{
				if (!bSectionPrinted)
				{
					m_pOutput->print("[ %s ]\n",pPC->GetSectionName(its));
					bSectionPrinted = true;
				}
				m_pOutput->print("??variable>%32s?? = ??value>%s\n", pPC->GetPropertyName(it), pP->Format().c_str());
				propset.erase(pPC->GetPropertyName(it));
				//cout << propset.size() << ") " << pPC->GetPropertyName(it) << endl;
			}
		}
	}
	if (propset.size())
	{
		if (pPC->HasSections())
			m_pOutput->print("*** no section assigned ***\n");
		for (set<CString>::iterator itn = propset.begin(); itn!=propset.end();itn++)
		{
			MProperty* pP = pPC->FindProperty(itn->c_str());
			ASSERT(pP);
			if (pP)
			{
				m_pOutput->print("??variable>%32s?? = ??value>%s\n", itn->c_str(), pP->Format().c_str());
			}
		}
	}
	return true;
}


struct getParams{
	MUITextModePriv* pThis;
	intSet   seq;
};


bool MUITextModePriv::com_get(char * arg)
{
	getParams gp;
	gp.pThis = this;

	if (!ParseNumRanges(gp.seq, arg, m_vecResIDs.size()))
	{
		m_pOutput->print("??error>get: failed to parse parameters\n");
		return false;
	}
	if (gp.seq.empty())
	{
		m_pOutput->print("??error>get requires numeric parameters\n");
		return false;
	}
	//
	ResultVec resVec;
	std::queue<int> toErase;
	for (intSet::iterator it=gp.seq.begin();it!=gp.seq.end(); ++it)
	{
		if (*it <= m_vecResIDs.size() && m_pController->GetResultsByID(m_vecResIDs[(*it)-1],resVec))
		{
			ASSERT(resVec.size());
			if (m_pController->AddDownload(resVec))
				m_pOutput->print("??message>starting download of ??name>%s\n", resVec[0].Name.c_str());
			else
				m_pOutput->print("??error>failed to start download of ??name>%s\n", resVec[0].Name.c_str());
			//
			resVec.clear();
			toErase.push(*it);
		}
	}
	while (toErase.size())
	{
		gp.seq.erase(toErase.front());
		toErase.pop();
	}
	if (gp.seq.empty())
		return true;
	m_pOutput->print("??error>no results with number(s)");
	for (intSet::iterator it=gp.seq.begin(); it!=gp.seq.end(); it++)
	{
		m_pOutput->print(" ??error>%d", (*it));
	}
	m_pOutput->print("\n");
	return false;
}

bool MUITextModePriv::com_move(char * arg)
{
	CString sArg = StripWhite(arg);
	// check if we have 2 parameters and extract the first one
	int nPos = sArg.find(' ');
	if (nPos<0)
	{
		m_pOutput->print("??error>'move' requires 2 parameters\n");
		return false;
	}

	DWORD dwID = 0;
	int nSNum = atol(sArg.substr(0, nPos).c_str());
	if (nSNum>0 && nSNum<=m_vecTransIDs.size())
		dwID = m_vecTransIDs[nSNum-1];
	if (dwID==0)
	{
		m_pOutput->print("??error>'move' requires valid download ID\n");
		return false;
	}
	CString sFileName = StripWhite(sArg.substr(nPos+1));
	if (!sFileName.length() || !m_pController->RenameDownload(dwID, sFileName.c_str()))
	{
		m_pOutput->print("??error>Renaming file failed, may be ID or file name is invalid?\n");
		return false;
	}
	//m_pOutput->print("??message>Ok\n");
	return true;
}

bool MUITextModePriv::com_scan(char * arg)
{
	m_pController->Rescan();
	return true;
}

bool MUITextModePriv::com_library(char * arg)
{
	vector<SharedFile> SharedFiles;
	m_pController->GetSharedFiles(SharedFiles);
	if (SharedFiles.size()==0)
	{
		m_pOutput->print("??message>no files shared\n");
		return true;
	}
	m_pOutput->print("??header>shared files\n");
	m_pOutput->print("??header>============\n");
	int nCounter = 1;
	for (vector<SharedFile>::iterator itFile = SharedFiles.begin(); itFile != SharedFiles.end(); itFile++, nCounter++)
	{
		if (itFile->Hidden)
			continue;
		m_pOutput->print("??id>%4d)?? ??name>%-42s ??size>%5s??  hits:??num>%-5d?? uploads:??num>%-4d\n",nCounter,
			   InsertElypsis(itFile->Name, 42).c_str(),
			   FormatSize(itFile->Size).c_str(),
			   itFile->Matches,itFile->Uploads);
		if (!itFile->Sha1Hash.empty())
			m_pOutput->print("      sha1: ??sha1>%s\n", itFile->Sha1Hash.c_str());
	}
	return true;
}

bool MUITextModePriv::com_version(char * arg)
{
	m_pOutput->print("??message>You are currently using Mutella v%s\n  ??flag1>Enjoy!\n", VERSION);
}

