/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/**
 * @file myx_gc_svgparser.h
 * @brief Parser for SVG elements, which are converted to OpenGL calls.
 * @note This parser does not handle a full svg description but only single svg elements.
 * 
 */

#ifndef __GC_GL_SVGPARSER_H__
#define __GC_GL_SVGPARSER_H__

#include <string>
#include <libxml/xmlmemory.h>

#include "myx_gc_datatypes.h"
#include "myx_gc_model.h"

//----------------------------------------------------------------------------------------------------------------------

/**
 * Simple tokenizer class that works similar as Java's StringTokenizer.
 */
class StringTokenizer
{
  string FInput, FDelimiters;
  string::size_type FPosition;
public:
  StringTokenizer(string Text, string Delimiters): FInput(Text), FDelimiters(Delimiters)
  {
    FPosition = FInput.find_first_not_of(FDelimiters);
  };
  
  bool HasMoreTokens()
  {
    return FPosition != string::npos;
  };
  
  string NextToken()
  {
    string::size_type Endposition = FInput.find_first_of(FDelimiters, FPosition);
    string Token = FInput.substr(FPosition, Endposition - FPosition);
    FPosition = FInput.find_first_not_of(FDelimiters, Endposition);

    return Token;
  };

  float NextTokenAsFloat()
  {
    return (float) atof(NextToken().c_str());
  };

  int NextTokenAsInt()
  {
    return atoi(NextToken().c_str());
  };
};

//----------------------------------------------------------------------------------------------------------------------

/**
 * CSVGParser is the main SVG parser class. It converts an <svg> element description into OpenGL calls.
 *
 * @note Not all possible subelements/attributes can be parsed by this class. If they are specified then they will be
 *       ignored. See Generic Canvas documentation for more details.
 */
class CSVGParser
{
protected:
  void ParseCircle(xmlNodePtr SVG, bool DoFill, GLubyte* FillColor, bool DoStroke, GLubyte* StrokeColor, float StrokeWidth,
    CBoundingBoxComputer* BB);
  GLuint ParseElement(xmlNodePtr SVG, bool DoFill, GLubyte* FillColor, bool DoStroke, GLubyte* StrokeColor, float StrokeWidth, 
    float MasterAlpha, CBoundingBoxComputer* BB);
  GLuint ParseGroup(xmlNodePtr SVG, bool DoFill, GLubyte* FillColor, bool DoStroke, GLubyte* StrokeColor, float StrokeWidth, 
    float MasterAlpha, CBoundingBoxComputer* BB);
  void ParseImage(xmlNodePtr SVG, CBoundingBoxComputer* BB);
  void ParseLine(xmlNodePtr SVG, bool DoStroke, GLubyte* StrokeColor, float StrokeWidth, CBoundingBoxComputer* BB);
  void ParsePolygon(xmlNodePtr SVG, bool DoFill, GLubyte* FillColor, bool DoStroke, GLubyte* StrokeColor, float StrokeWidth,
    CBoundingBoxComputer* BB);
  void ParsePolyline(xmlNodePtr SVG, bool DoFill, GLubyte* FillColor, bool DoStroke, GLubyte* StrokeColor, float StrokeWidth,
    CBoundingBoxComputer* BB);
  void ParseRectangle(xmlNodePtr SVG, bool DoFill, GLubyte* FillColor, bool DoStroke, GLubyte* StrokeColor, float StrokeWidth,
    CBoundingBoxComputer* BB);
  GLuint ParseText(xmlNodePtr SVG, bool DoFill, GLubyte* FillColor, bool DoStroke, GLubyte* StrokeColor, float StrokeWidth, 
    float MasterAlpha, CBoundingBoxComputer* BB);
  void ParseTransformation(char* Transformation);
  void RenderVertices(bool DoFill, GLubyte* FillColor, bool DoStroke, GLubyte* StrokeColor, const CVertexVector& Vertices, 
    CVertexVector* TextureCoordinates, float StrokeWidth, bool CloseShape, CBoundingBoxComputer* BB);
public:
  CSVGParser(void);
  virtual ~CSVGParser(void);

  void Convert(xmlNodePtr SVG, GLuint DisplayList, CBoundingBoxComputer* BB);
  void ParseDefinition(xmlNodePtr Definition, CGCModel* Model);
};

//----------------------------------------------------------------------------------------------------------------------

#endif // #ifndef __GC_GL_SVGPARSER_H__