unit ExOpenDialog;

// Copyright (C) 2003, 2004 MySQL AB
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  VirtualExplorerTree, VirtualTrees, VirtualExplorerListviewEx,
  VirtualShellToolBar, ExtCtrls, StdCtrls, VirtualUnicodeControls, VirtualShellUtilities,
  ImgList, VirtualShellHistory, Dialogs,
  VirtualShellAutoComplete, VirtualWideStrings,
  TntStdCtrls, TntForms, ToolWin, Menus, TntMenus, ComCtrls, TntExtCtrls,
  AuxFuncs;

type
  TExOpenDialog = class(TTntForm)
    PlacesPanel: TPanel;
    FileNameEdit: TTntEdit;
    Label2: TLabel;
    Label3: TLabel;
    FilterCombobox: TTntComboBox;
    ViewerPanel: TPanel;
    VSHistory: TVirtualShellHistory;
    OpenButton: TButton;
    CancelButton: TButton;
    FileListView: TVirtualExplorerListviewEx;
    VirtualShellLink1: TVirtualShellLink;
    NavigationImagelistNormal: TImageList;
    ViewsMenu: TTntPopupMenu;
    ile1: TTntMenuItem;
    Symbols1: TTntMenuItem;
    List1: TTntMenuItem;
    Details1: TTntMenuItem;
    VirtualSpecialFolderToolbar1: TVirtualSpecialFolderToolbar;
    ThumbnailsItem: TTntMenuItem;
    FilesPnl: TPanel;
    SearchPnl: TTntPanel;
    Label1: TLabel;
    DirCombo: TVirtualExplorerCombobox;
    NavigationToolbar: TToolBar;
    PrevBtn: TToolButton;
    UpBtn: TToolButton;
    NewFolderBtn: TToolButton;
    ViewToolButton: TToolButton;
    procedure UpBtnClick(Sender: TObject);
    procedure ViewItemClick(Sender: TObject);
    procedure FileListViewEnumFolder(Sender: TCustomVirtualExplorerTree; Namespace: TNamespace;
      var AllowAsChild: Boolean);
    procedure PrevBtnClick(Sender: TObject);
    procedure FileListViewChange(Sender: TBaseVirtualTree; Node: PVirtualNode);
    procedure DirComboPathChange(Sender: TCustomVirtualExplorerCombobox; SelectedNamespace: TNamespace);
    procedure NewFolderBtnClick(Sender: TObject);
    procedure FilterComboboxKeyPress(Sender: TObject; var Key: Char);
    procedure FileNameEditChange(Sender: TObject);
    procedure FileListViewDblClick(Sender: TObject);
    procedure CancelButtonClick(Sender: TObject);
    procedure FileListViewEditing(Sender: TBaseVirtualTree; Node: PVirtualNode; Column: TColumnIndex;
      var Allowed: Boolean);
    procedure FileListViewEdited(Sender: TBaseVirtualTree; Node: PVirtualNode; Column: TColumnIndex);
    procedure FileListViewEditCancelled(Sender: TBaseVirtualTree; Column: TColumnIndex);
    procedure FilterComboboxSelect(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure OpenButtonClick(Sender: TObject);
    procedure FileListViewRootChange(Sender: TCustomVirtualExplorerTree);
  private
    FCurrentFile: WideString;
    FIsEditing: Boolean;
    FFiles: TWideStringList;
    FFilter: WideString;
    FFilterIndex: Integer;
    FInitialDir: WideString;
    FOptions: TOpenOptions;
    FOptionsEx: TOpenOptionsEx;
    FTitle: WideString;

    FOnSelectionChange: TNotifyEvent;
    FOnFolderChange: TNotifyEvent;
    FOnTypeChange: TNotifyEvent;
    FOnCanClose: TCloseQueryEvent;
    FOnIncludeItem: TIncludeItemEvent;

    procedure SetInitialDir(Folder: WideString);
  protected
    procedure ApplyOptions;
    procedure ClearFilter;
    procedure ParseFilterString;
    procedure ParseFileEditEntries;
  public
    function Execute: Boolean; virtual;
    property Files: TWideStringList read FFiles;
  published
    property FileName: WideString read FCurrentFile write FCurrentFile;
    property Filter: WideString read FFilter write FFilter;
    property FilterIndex: Integer read FFilterIndex write FFilterIndex default 1;
    property InitialDir: WideString read FInitialDir write SetInitialDir;
    property Options: TOpenOptions read FOptions write FOptions default [ofHideReadOnly, ofEnableSizing];
    property OptionsEx: TOpenOptionsEx read FOptionsEx write FOptionsEx default [];
    property Title: WideString read FTitle write FTitle;
    
    property OnCanClose: TCloseQueryEvent read FOnCanClose write FOnCanClose;
    property OnFolderChange: TNotifyEvent read FOnFolderChange write FOnFolderChange;
    property OnSelectionChange: TNotifyEvent read FOnSelectionChange write FOnSelectionChange;
    property OnTypeChange: TNotifyEvent read FOnTypeChange write FOnTypeChange;
    property OnIncludeItem: TIncludeItemEvent read FOnIncludeItem write FOnIncludeItem;
  end;

//----------------------------------------------------------------------------------------------------------------------

implementation

{$R *.dfm}

uses
  TntSysUtils;
  
function PathMatchSpec(const pszFileParam, pszSpec: PChar): Bool; stdcall; external 'shlwapi.dll' name 'PathMatchSpecA';
function PathMatchSpecA(const pszFileParam, pszSpec: PAnsiChar): Bool; stdcall; external 'shlwapi.dll';
function PathMatchSpecW(const pszFileParam, pszSpec: PWideChar): Bool; stdcall; external 'shlwapi.dll';

//----------------------------------------------------------------------------------------------------------------------

procedure TExOpenDialog.ApplyOptions;

// Parses the dialog options and sets the appropriate options in the dialog.

begin
  PlacesPanel.Visible := not (ofExNoPlacesBar in FOptionsEx);

  if ofReadOnly in FOptions then
    FileListView.TreeOptions.VETMiscOptions := FileListView.TreeOptions.VETMiscOptions + [toVETReadOnly]
  else
    FileListView.TreeOptions.VETMiscOptions := FileListView.TreeOptions.VETMiscOptions - [toVETReadOnly];

  // ofOverwritePrompt is only used in save dialogs.
  // ofHideReadOnly is not supported.
  // ofNoChangeDir is handled on close.
  // ofShowHelp is not supported.
  // ofNoValidate is not supported

  // ofAllowMultiSelect is not fully supported.
  if ofAllowMultiSelect in FOptions then
    FileListView.TreeOptions.SelectionOptions := FileListView.TreeOptions.SelectionOptions + [toMultiSelect]
  else
    FileListView.TreeOptions.SelectionOptions := FileListView.TreeOptions.SelectionOptions - [toMultiSelect];

  // ofExtensionDifferent is not supported.
  // ofPathMustExist is not supported.
  // ofFileMustExist is not supported.
  // ofCreatePrompt is not supported.
  // ofShareAware is not supported.
  // ofNoReadOnlyReturn is not supported.
  // ofNoTestFileCreate is not supported.
  // ofNoNetworkButton is not supported.
  // ofNoLongNames is not supported.
  // ofOldStyleDialog is not supported.
  // ofNoDereferenceLinks is not supported.
  // ofEnableIncludeNotify is not supported.
  // ofEnableSizing is not supported.
  // ofDontAddToRecent is not supported.
  	
  if ofForceShowHidden in FOptions then
    FileListView.FileObjects := FileListView.FileObjects + [foHidden]
  else
    FileListView.FileObjects := FileListView.FileObjects - [foHidden];
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TExOpenDialog.UpBtnClick(Sender: TObject);

begin
  FileListView.BrowseToPrevLevel;
  UpBtn.Enabled := not FileListView.RootFolderNamespace.IsDesktop;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TExOpenDialog.ViewItemClick(Sender: TObject);

var
  Tag: Integer;

begin
  //get sender's tag: 0-Icons, 1-smallicons, 2-list, 3-details, 4-thumbnails
  Tag := TToolButton(Sender).Tag;
  FileListView.ViewStyle := TViewStyleEx(Tag);
  if FileListView.ViewStyle <> vsxReport then
  begin
    FileListView.SyncSelectedItems(True);
    FileListView.SyncOptions;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TExOpenDialog.FileListViewEnumFolder(Sender: TCustomVirtualExplorerTree; Namespace: TNamespace;
  var AllowAsChild: Boolean);

var
  Name: WideString;
  Filter: PWideChar;

begin
  if FilterCombobox.ItemIndex > -1 then
  begin
    Name := NameSpace.NameForParsingInFolder;
    Filter := PWideChar(FilterCombobox.Items.Objects[FilterCombobox.ItemIndex]);
    AllowAsChild := Namespace.Folder or PathMatchSpecW(PWideChar(Name), Filter);
  end
  else
    AllowAsChild := False;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TExOpenDialog.PrevBtnClick(Sender: TObject);

begin
  VSHistory.Back;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TExOpenDialog.FileListViewChange(Sender: TBaseVirtualTree; Node: PVirtualNode);

var
  NS: TNamespace;

begin
  if (Sender as TCustomVirtualExplorerListViewEx).ValidateNamespace(Node, NS) then
    if not NS.Folder then
    begin
      if NS.Link then
      begin
        VirtualShellLink1.ReadLink(NS.NameParseAddress);
        FCurrentFile := VirtualShellLink1.TargetPath;
        FileNameEdit.Text := ExtractFileName(VirtualShellLink1.TargetPath);
      end
      else
      begin
        FCurrentFile := FileListView.SelectedPath;
        FileNameEdit.Text := FileListView.SelectedFile;
      end;
    end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TExOpenDialog.DirComboPathChange(Sender: TCustomVirtualExplorerCombobox; SelectedNamespace: TNamespace);

var
  Name: string;

begin
  Name := DirCombo.Path;
  if FTitle = '' then
  begin
    if Pos('::{', Name) = 1 then
      Caption := SelectedNameSpace.NameForEditing
    else
      Caption := Name;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TExOpenDialog.NewFolderBtnClick(Sender: TObject);

begin
  with FileListview do
    CreateNewFolder(RootFolderNamespace.NameForParsing);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TExOpenDialog.FilterComboboxKeyPress(Sender: TObject; var Key: Char);

begin
  if Key = #13 then
    FileListView.RefreshTree;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TExOpenDialog.FileNameEditChange(Sender: TObject);

begin
  OpenButton.Enabled := FileNameEdit.Text <> '';
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TExOpenDialog.FileListViewDblClick(Sender: TObject);

begin
  // If we cannot browse to the next level then a file was double clicked.
  if not FileListView.BrowseToNextLevel then
  begin
    FFiles.Clear;
    FCurrentFile := FileListView.SelectedPath;
    FFiles.Add(FCurrentFile);
    if not (ofNoChangeDir in FOptions) then
      WideSetCurrentDir(WideExtractFilePath(FCurrentFile));
    ModalResult := mrOk;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TExOpenDialog.CancelButtonClick(Sender: TObject);

begin
  if not FIsEditing then
    ModalResult := mrCancel;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TExOpenDialog.ClearFilter;

// Removes all stored filters from the filter combobox.

var
  I: Integer;

begin
  for I := 0 to FilterCombobox.Items.Count - 1 do
    StrDisposeW(PWideChar(FilterCombobox.Items.Objects[I]));
  FilterCombobox.Clear;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TExOpenDialog.FileListViewEditing(Sender: TBaseVirtualTree; Node: PVirtualNode; Column: TColumnIndex;
  var Allowed: Boolean);

begin
  FIsEditing := Allowed;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TExOpenDialog.FileListViewEdited(Sender: TBaseVirtualTree; Node: PVirtualNode; Column: TColumnIndex);

begin
  FIsEditing := False;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TExOpenDialog.FileListViewEditCancelled(Sender: TBaseVirtualTree; Column: TColumnIndex);

begin
  FIsEditing := False;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TExOpenDialog.FilterComboboxSelect(Sender: TObject);

begin
  FFilterIndex := FilterCombobox.ItemIndex + 1;
  FileListView.RefreshTree;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TExOpenDialog.FormCreate(Sender: TObject);

begin
  FFiles := TWideStringList.Create;
  FFilterIndex := 1;

  InitForm(self);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TExOpenDialog.FormDestroy(Sender: TObject);

begin
  FFiles.Free;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TExOpenDialog.FormShow(Sender: TObject);

var
  FileNameSave: WideString;

begin
  Caption := FTitle;
  FileNameSave := FCurrentFile;

  ApplyOptions;
  FFiles.Clear;
  ClearFilter;
  if FFilter <> '' then
    ParseFilterString;
  FilterCombobox.ItemIndex := FFilterIndex - 1;
  FileListView.RefreshTree();

  FCurrentFile := FileNameSave;
  if FCurrentFile <> '' then
  begin
    FFiles.Add(FCurrentFile);
    FileNameEdit.Text := WideExtractFileName(FCurrentFile);
  end;

  if WideDirectoryExists(FInitialDir) then
    FileListView.BrowseTo(FInitialDir, False)
  else
  begin
    FInitialDir := WideExtractFilePath(FCurrentFile);
    if WideDirectoryExists(FInitialDir) then
      FileListView.BrowseTo(FInitialDir, False)
  end;

  UpBtn.Enabled := not FileListView.RootFolderNamespace.IsDesktop;
end;

//----------------------------------------------------------------------------------------------------------------------

function TExOpenDialog.Execute: Boolean;

begin
  Result := ShowModal = mrOk;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TExOpenDialog.ParseFilterString;

// Split the current filter string into single entries and fill the filter box with them.

var
  Head, Tail: PWideChar;
  Description,
  FilterValue: WideString;

begin
  Head := PWideChar(FFilter);
  while Head^ <> #0 do
  begin
    // Scan until first vertical bar.
    Tail := Head;
    while not (Tail^ in [WideChar('|'), WideNull]) do
      Inc(Tail);
    SetString(Description, Head, Tail - Head);
    FilterValue := '';
    if Tail^ <> #0 then
    begin
      Inc(Tail);
      Head := Tail;

      // Scan until second vertical bar.
      Tail := Head;
      while not (Tail^ in [WideChar('|'), WideNull]) do
        Inc(Tail);
      SetString(FilterValue, Head, Tail - Head);
      if (Description <> '') and (FilterValue <> '') then
      begin
        FilterCombobox.AddItem(Description, Pointer(StrNewW(PWideChar(FilterValue))));
      end;
      if Tail^ = '|' then
        Inc(Tail);
    end;
    Head := Tail;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TExOpenDialog.SetInitialDir(Folder: WideString);

begin
  FInitialDir := Folder;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TExOpenDialog.ParseFileEditEntries;

// Splits the content of the file edit control into single strings and fills the file list with them.

var
  S: WideString;

begin
  FFiles.Clear;
  S := Trim(FileNameEdit.Text);
  if S = '' then
    FileNameEdit.Text := ''
  else
    FFiles.CommaText := S;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TExOpenDialog.OpenButtonClick(Sender: TObject);

var
  CurrentRootPath: WideString;
  NewRootPath: WideString;
  I: Integer;

begin
  // Check if we can browse to the given targets. If any of the given targets is a folder we can browse into
  // then do this and keep the current file edit content. If we cannot find a folder then return what we have as new files.
  // Note: An empty edit or only white spaces does not close the dialog.
  ParseFileEditEntries;
  if FFiles.Count = 0 then
    ModalResult := mrNone
  else
  begin
    CurrentRootPath := WideIncludeTrailingBackslash(FileListView.RootFolderNamespace.NameForParsing);
    for I := 0 to FFiles.Count - 1 do
    begin
      if FileListView.BrowseTo(CurrentRootPath + FFiles[I], False, False, False, False) then
      begin
        // Target exists. Now see if it is a folder.
        NewRootPath := WideIncludeTrailingBackslash(FileListView.RootFolderNamespace.NameForParsing);
        if NewRootPath <> CurrentRootPath then
        begin
          // We changed the folder so stop here and let the user decide what to do.
          ModalResult := mrNone;
          Break;
        end;
      end;
      // If ModalResult is still mrOk here then we have a list without any matching folder name.
      // Only files names of existing or non-existing files. Go ahead and do what a file dialog must do.
    end;

    if ModalResult <> mrNone then
    begin
      // If we are going to close the dialog do some housekeeping.
      for I := 0 to FFiles.Count - 1 do
        FFiles[I] := CurrentRootPath + FFiles[I];
      FCurrentFile := FFiles[0];

      if not (ofNoChangeDir in FOptions) then
        WideSetCurrentDir(WideExtractFilePath(FCurrentFile));
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TExOpenDialog.FileListViewRootChange(Sender: TCustomVirtualExplorerTree);

begin
  UpBtn.Enabled := not FileListView.RootFolderNamespace.IsDesktop;
  PrevBtn.Enabled := VSHistory.HasBackItems;
end;

//----------------------------------------------------------------------------------------------------------------------

end.

