/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/**
 * @file myx_gc_layer.cpp 
 * @brief Implementation of the GC layer class.
 * 
 */

#ifdef _WINDOWS
  #include <windows.h>
#else
#endif // ifdef _WINDOWS
#include <gl/gl.h>
#include <gl/glu.h>
#include <math.h>

#include "myx_gc_figure.h"
#include "myx_gc_layer.h"
#include "myx_gc_canvas.h"
#include "myx_gc_gl_helper.h"

//----------------- CLayer ---------------------------------------------------------------------------------------------

CLayer::CLayer(CGenericCanvas* Owner)
{
  FUpdateCount = 0;
  FCanvas = Owner;
  FList = -1;

  // Initialize with useful values.
  FScaling[0] = 1;
  FScaling[1] = 1;
  FScaling[2] = 1;

  FTranslation[0] = 0;
  FTranslation[1] = 0;
  FTranslation[2] = 0;

  FDirty = true;
  FVisible = true;
  FEnabled = true;
}

//----------------------------------------------------------------------------------------------------------------------

CLayer::~CLayer(void)
{
  FUpdateCount++;

  if ((FCanvas != NULL) && (!FCanvas->IsUpdating()))
    FCanvas->RemoveLayer(this);
  Clear();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Applies the layer's transformations for rendering, feedback etc.
 */
void CLayer::ApplyTransformations()
{
  glTranslated(FTranslation[0], FTranslation[1], FTranslation[2]);
  glScaled(FScaling[0], FScaling[1], FScaling[2]);
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Triggers the error checking of the canvas.
 */
void CLayer::CheckError(void)
{
  FCanvas->CheckError();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Marks the display list for this layer as invalid, hence it will be recreated next time ValidateDisplayList is called.
 * If a list already exists then it is freed.
 */
void CLayer::MakeDirty(void)
{
  if (!FDirty)
  {
    glDeleteLists(FList, 1);
    FList = -1;
    FDirty = true;
    if ((FCanvas != NULL) && (!FCanvas->IsUpdating()))
      FCanvas->Invalidate();
  };
}

//----------------------------------------------------------------------------------------------------------------------

/*
 * Helper method to determine the transformed vertices of the given figure instance. The layer applies its own
 * transformations and only renders the figure instance.
 *
 * @param Instance The figure instance for which feedback data is requested.
 */
void CLayer::RenderFeedback(CFigureInstance* Instance)
{
  ApplyTransformations();
  Instance->RenderFeedback();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Renders layer content that is not determined by figure instances. This method might be overridden by descendants.
 */
void CLayer::RenderLayerContent(void)
{
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Creates the display list of this figure (and all child figures) if necessary.
 */
void CLayer::ValidateDisplayList(void)
{
  if (FDirty)
  {
    FDirty = false;

    ValidateLayerContent();

    for (CInstanceIterator Iterator = FInstances.begin(); Iterator != FInstances.end(); Iterator++)
      (*Iterator)->ValidateDisplayList();

    // Create the display list. Make sure not to change the current transformation matrix.
    FList = glGenLists(1);
    glNewList(FList, GL_COMPILE);
    glPushMatrix();

    ApplyTransformations();
    RenderLayerContent();

    // At this point figure instances have valid display lists.
    for (CInstanceIterator Iterator = FInstances.begin(); Iterator != FInstances.end(); Iterator++)
      (*Iterator)->Render();

    glPopMatrix();
    glEndList();

    CheckError();
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Prepares layer content that is not determined by figure instances. This method might be overridden by descendants.
 */
void CLayer::ValidateLayerContent(void)
{
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Adds the given figure instance to the end of the instance list. If Instance belongs to another layer currently 
 * it is removed from the other's instance list first.
 *
 * @param Instance The figure instance to add.
 */
void CLayer::AddInstance(CFigureInstance* Instance)
{
  if (Instance->FLayer != NULL)
    Instance->FLayer->RemoveInstance(Instance);
  FInstances.insert(Instance);
  Instance->FLayer = this;
  MakeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * @see CGenericCanvas.BeginUpdate()
 */
void CLayer::BeginUpdate(void)
{
  FUpdateCount++;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Removes all figure instances from this layer.
 */
void CLayer::Clear(void)
{
  for (CInstanceIterator Iterator = FInstances.begin(); Iterator != FInstances.end(); Iterator++)
  {
    CFigureInstance* Instance = *Iterator;
    Instance->FLayer = NULL;
    if (Instance->Selected())
      FCanvas->RemoveFromSelection(Instance);
    delete Instance;
  };
  FInstances.clear();
  MakeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Creates a new instance for the given figure and adds it to this layer.
 *
 * @param Figure The figure for which the instance is to be created.
 * @return A new figure instance.
 */
CFigureInstance* CLayer::CreateInstance(CFigure* Figure)
{
  CFigureInstance* Instance = new CFigureInstance(this, Figure);

  return Instance;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * @see CGenericCanvas.EndUpdate()
 */
void CLayer::EndUpdate(void)
{
  if (FUpdateCount > 0)
    FUpdateCount++;
  if (FUpdateCount == 0)
    MakeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

CGenericCanvas* CLayer::GetCanvas(void)
{
  return FCanvas;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Returns the visibility state of the layer.
 *
 * @return true if the layer is visible in the canvas, otherwise false.
 */
bool CLayer::GetVisible(void)
{
  return FVisible;
}

//----------------------------------------------------------------------------------------------------------------------

bool CLayer::IsUpdating(void)
{
  return FUpdateCount != 0;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Release function for external access to free this layer. It will just delete itself.
 */
void CLayer::Release(void)
{
  delete this;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Removes the given figure instance from the instance list if it is currently there.
 * No error is raised if the instance does not belong to this layer.
 *
 * @param Instance The instance to be removed.
 */
void CLayer::RemoveInstance(CFigureInstance* Instance)
{
  for (CInstanceIterator Iterator = FInstances.begin(); Iterator != FInstances.end(); Iterator++)
    if (*Iterator == Instance)
    {
      FInstances.erase(Iterator);
      Instance->FLayer = NULL;
      MakeDirty();
      break;
    };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Checks the validity of the figure display list and executes it.
 */
void CLayer::Render(void)
{
  if (FVisible && (FUpdateCount == 0)) 
  {
    if (FDirty)
      ValidateDisplayList();

    glCallList(FList);
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Scales the layer by the amount given in Factor. If Accumulative is true then the new scale factors are multiplied
 * with the existing values. This version of Scale uses single float values as parameters.
 *
 * @param Sx Scale factor for the x axis.
 * @param Sy Scale factor for the y axis.
 * @param Sz Scale factor for the z axis.
 * @param Accumulative If true then the given values are added to any existing values otherwiese they are used as given.
 */
void CLayer::Scale(double Sx, double Sy, double Sz, bool Accumulative)
{
  if (Accumulative)
  {
    FScaling[0] += Sx;
    FScaling[1] += Sy;
    FScaling[2] += Sz;
  }
  else
  {
    FScaling[0] = Sx;
    FScaling[1] = Sy;
    FScaling[2] = Sz;
  };
  MakeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

/** 
 * Scales the layer by the amount given in Factor. If Accumulative is true then the new scale factors are multiplied
 * with the existing values. This version of Scale uses an array of values in the parameter list.
 *
 * @param Factor An array of 3 scale values, one for each axis.
 * @param Accumulative If true then the given values are added to any existing values otherwiese they are used as given.
 */
void CLayer::ScaleV(const double Factor[3], bool Accumulative)
{
  if (Accumulative)
  {
    FScaling[0] += Factor[0];
    FScaling[1] += Factor[1];
    FScaling[2] += Factor[2];
  }
  else
  {
    FScaling[0] = Factor[0];
    FScaling[1] = Factor[1];
    FScaling[2] = Factor[2];
  };
  MakeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Moves the layer by the amount given in Tx, Ty and Tz. If Accumulative is true then the new translation factors 
 * are multiplied with the existing values. This version of Translate uses an array for the values in the parameter list.
 *
 * @param Tx Scale factor for the x axis.
 * @param Ty Scale factor for the y axis.
 * @param Tz Scale factor for the z axis.
 * @param Accumulative If true then the given values are added to any existing values otherwiese they are used as given.
 */
void CLayer::Translate(double Tx, double Ty, double Tz, bool Accumulative)
{
  if (Accumulative)
  {
    FTranslation[0] += Tx;
    FTranslation[1] += Ty;
    FTranslation[2] += Tz;
  }
  else
  {
    FTranslation[0] = Tx;
    FTranslation[1] = Ty;
    FTranslation[2] = Tz;
  };
  MakeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Moves the layer by the amount given in Factor. If Accumulative is true then the new translation factors are multiplied
 * with the existing values. This version of Translate uses an array for the values in the parameter list.
 *
 * @param Factor An array of translation values, for each axis one.
 * @param Accumulative If true then the given values are added to any existing values otherwiese they are used as given.
 */
void CLayer::TranslateV(const double Factor[3], bool Accumulative)
{
  if (Accumulative)
  {
    FTranslation[0] += Factor[0];
    FTranslation[1] += Factor[1];
    FTranslation[2] += Factor[2];
  }
  else
  {
    FTranslation[0] = Factor[0];
    FTranslation[1] = Factor[1];
    FTranslation[2] = Factor[2];
  };
  MakeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Sets the layer's enabled state.
 *
 * @param IsEnabled Set it to true if you want the layer to be visible.
 */
void CLayer::SetEnabled(bool IsEnabled)
{
  if (FEnabled != IsEnabled)
  {
    FEnabled = IsEnabled;
    FCanvas->Invalidate();
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Sets the layer's visibility state.
 *
 * @param IsVisible Set it to true if you want the layer to be visible.
 */
void CLayer::SetVisible(bool IsVisible)
{
  if (FVisible != IsVisible)
  {
    FVisible = IsVisible;
    FCanvas->Invalidate();
  };
}

//----------------- CGridLayer -----------------------------------------------------------------------------------------

CGridLayer::CGridLayer(CGenericCanvas* Owner): CLayer(Owner)
{
}

//----------------------------------------------------------------------------------------------------------------------

void CGridLayer::RenderLayerContent(void)
{
  GLint ViewInfo[4];
  glGetIntegerv(GL_VIEWPORT, ViewInfo);
  double Left = ViewInfo[0];
  double Right = ViewInfo[0] + ViewInfo[2];
  double Top = ViewInfo[1];
  double Bottom = ViewInfo[1] + ViewInfo[3];

  // Disable antialiasing temporarily.
  glPushAttrib(GL_ENABLE_BIT);
  glDisable(GL_LINE_SMOOTH);
  glColor4d(0, 0, 0, 0.2);

  glBegin(GL_LINES);
  for (double Y = Top; Y < Bottom; Y += 30)
  {
    glVertex2d(Left, Y);
    glVertex2d(Right, Y);
  };

  for (double X = Left; X < Right; X += 30)
  {
    glVertex2d(X, Top);
    glVertex2d(X, Bottom);
  };

  glEnd();
  glPopAttrib();
}

//----------------- CSelectionLayer ------------------------------------------------------------------------------------

CSelectionLayer::CSelectionLayer(CGenericCanvas* Owner): CLayer(Owner)
{
  FHandleSize = 5;
}

//----------------------------------------------------------------------------------------------------------------------

CSelectionLayer::~CSelectionLayer(void)
{
  ClearSelection();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Creates the decoration display list for the figure instance given by the iterator.
 *
 * @param Iterator The iterator pointing to the figure instance in the selection map that needs a new display list.
 */
void CSelectionLayer::CreateDecoration(CSelectionIterator Iterator)
{
  // Update the bounding box of the figure instance.
  TBounds Bounds;
  Iterator->second.Instance->GetBounds(&Bounds);

  // Give a small border around the figure instance.
  Bounds.Left -= 0.1;
  Bounds.Top += 0.1;
  Bounds.Right += 0.1;
  Bounds.Bottom -= 0.1;
  Iterator->second.Bounds = Bounds;

  GLubyte Color[4];
  if (!ColorByName("Highlight", Color))
  {
    // If the system's highlight color could not be found then use a default one.
    Color[0] = 64;
    Color[1] = 64;
    Color[2] = 128;
  };

  Iterator->second.DisplayList = glGenLists(1);
  glNewList(Iterator->second.DisplayList, GL_COMPILE);

  // The interior.
  Color[3] = 30;
  glColor4ubv(Color);
  glRectd(Bounds.Left, Bounds.Top, Bounds.Right, Bounds.Bottom);

  glPushAttrib(GL_LINE_BIT);
  glDisable(GL_LINE_SMOOTH);

  // The border.
  Color[3] = 200;
  glColor4ubv(Color);
  glBegin(GL_LINE_LOOP);
    glVertex2d(Bounds.Left, Bounds.Top);
    glVertex2d(Bounds.Right, Bounds.Top); 
    glVertex2d(Bounds.Right, Bounds.Bottom);
    glVertex2d(Bounds.Left, Bounds.Bottom); 
  glEnd();

  // The handles.
  Color[3] = 100;
  glPointSize(FHandleSize);
  glBegin(GL_POINTS);
    glVertex2d(Bounds.Left, Bounds.Top);
    glVertex2d((Bounds.Right + Bounds.Left) / 2, Bounds.Top);
    glVertex2d(Bounds.Right, Bounds.Top);

    glVertex2d(Bounds.Left, (Bounds.Top + Bounds.Bottom) / 2);
    glVertex2d(Bounds.Right, (Bounds.Top + Bounds.Bottom) / 2);

    glVertex2d(Bounds.Left, Bounds.Bottom);
    glVertex2d((Bounds.Right + Bounds.Left) / 2, Bounds.Bottom);
    glVertex2d(Bounds.Right, Bounds.Bottom);
  glEnd();

  glPopAttrib();

  glEndList();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Renders the decorations for all figure instances that are currently selected.
 */
void CSelectionLayer::RenderLayerContent(void)
{
  for (CSelectionIterator Iterator = FSelection.begin(); Iterator != FSelection.end(); Iterator++)
    glCallList(Iterator->second.DisplayList);
  CheckError();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Creates display lists for all invalid decorations.
 */
void CSelectionLayer::ValidateLayerContent(void)
{
  for (CSelectionIterator Iterator = FSelection.begin(); Iterator != FSelection.end(); Iterator++)
    if (Iterator->second.Dirty)
    {
      CreateDecoration(Iterator);
      Iterator->second.Dirty = false;
    };
  CheckError();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Adds the given figure instance to the current selection.
 *
 * @param Instance The instance to be added to the selection. If it is already in the set it won't be added again.
 */
void CSelectionLayer::AddToSelection(CFigureInstance* Instance)
{
  // Be error tolerant. If someone removes the selected flag manually then the instance might still already be in our list.
  Instance->FSelected = true;
  CSelectionIterator Iterator = FSelection.find(Instance);
  if (Iterator == FSelection.end())
  {
    TSelectionEntry Entry;
    Entry.Instance = Instance;
    Entry.Dirty = true;
    
    FSelection[Instance] = Entry;
    GetCanvas()->Change(Instance, GC_CHANGE_SELECTION_ADD);
    if (!IsUpdating())
      MakeDirty();
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Removes all figure instances fromt the selection set, making it empty.
 */
void CSelectionLayer::ClearSelection(void)
{
  // Delete all display lists first.
  InvalidateBounds(NULL);

  for (CSelectionIterator Iterator = FSelection.begin(); Iterator != FSelection.end(); Iterator++)
    Iterator->second.Instance->FSelected = false;
  FSelection.clear();
  GetCanvas()->Change(NULL, GC_CHANGE_SELECTION_CLEAR);
  if (!IsUpdating())
    MakeDirty();
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Retrieves the value of the given property, if it is a property of this class.
 *
 * @param Property The property to retrieve.
 * @param Value [out] The value of the property.
 * @return True if the asked property is known in this class, false otherwise (in this case Value is not changed).
 * @note Implicit datatype conversion is performed if datatypes of property and Value parameter do not match.
 *       This can result in lost precision or even conversion errors. So make sure to use the right datatype for the call.
 */
bool CSelectionLayer::GetProperty(TProperty Property, double& Value)
{
  bool Result = true;
  switch (Property)
  {
    case GC_PROPERTY_HANDLE_SIZE:
      {
        Value = FHandleSize;
        break;
      };
    default:
      {
        Result = false;
        break;
      };
  };

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Retrieves the value of the given property, if it is a property of this class.
 *
 * @param Property The property to retrieve.
 * @param Value [out] The value of the property.
 * @return True if the asked property is known in this class, false otherwise (in this case Value is not changed).
 * @note Implicit datatype conversion is performed if datatypes of property and Value parameter do not match.
 *       This can result in lost precision or even conversion errors. So make sure to use the right datatype for the call.
 */
bool CSelectionLayer::GetProperty(TProperty Property, int& Value)
{
  bool Result = true;
  switch (Property)
  {
    case GC_PROPERTY_HANDLE_SIZE:
      {
        Value = ROUND(FHandleSize);
        break;
      };
    default:
      {
        Result = false;
        break;
      };
  };

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Determines whether the given position corresponds to any of the parts (body, handles) of a selection decoration.
 * This test is quite fast and can be used for cursor feedback and such.
 * The method needs correctly set modelview and projection matrices on enter in order to convert mouse coordinates
 * correctly.
 *
 * @param X The horizontal mouse coordinate in layer coordinates.
 * @param Y The vertical mouse coordinate in layer coordinates.
 * @return One of the selection info flags.
 */
TGCSelectionInfo CSelectionLayer::GetSelectionInfo(double X, double Y)
{
  TGCSelectionInfo Result = GC_SI_NONE;

  // Convert window coordinates into object (scene) coordinates.
  ApplyTransformations();

  GLint Viewport[4];
  glGetIntegerv(GL_VIEWPORT, Viewport);

  GLdouble ModelviewMatrix[16];
  glGetDoublev(GL_MODELVIEW_MATRIX, ModelviewMatrix);

  GLdouble ProjectionMatrix[16];
  glGetDoublev(GL_PROJECTION_MATRIX, ProjectionMatrix);

  double LocalX, LocalY, LocalZ;
  gluUnProject(X, Viewport[3] - Y, 0, ModelviewMatrix, ProjectionMatrix, Viewport, &LocalX, &LocalY, &LocalZ);

  // Now try to find a decoration that is located at the given position.
  // We examine only those that are not dirty. Dirty decorations cannot be in view.
  // Iteration happens backwards to find decorations on top earlier than others.
  for (CSelectionIteratorReverse Iterator = FSelection.rbegin(); Iterator != FSelection.rend(); Iterator++)
  {
    if (!Iterator->second.Dirty)
    {
      TBounds Bounds = Iterator->second.Bounds;
      if ((LocalX >= Bounds.Left - 0.4) && (LocalX <= Bounds.Right + 0.4) &&
        (LocalY <= Bounds.Top + 0.4) && (LocalY >= Bounds.Bottom - 0.4))
      {
        // Found a decoration. Check if a handle is hit.
        Result = GC_SI_ON_OBJECT;

        bool LeftColumn = abs(Bounds.Left - LocalX) <= 0.8;
        bool MiddleColumn = abs((Bounds.Left + Bounds.Right) / 2 - LocalX) <= 0.8;
        bool RightColumn = abs(Bounds.Right - LocalX) <= 0.8;
        bool TopRow = abs(Bounds.Top - LocalY) <= 0.8;
        bool MiddleRow = abs((Bounds.Top + Bounds.Bottom) / 2 - LocalY) <= 0.8;
        bool BottomRow = abs(Bounds.Bottom - LocalY) <= 0.8;

        if (LeftColumn)
        {
          if (TopRow)
            Result = GC_SI_NORTH_WEST;
          else
            if (MiddleRow)
              Result = GC_SI_WEST;
            else
              if (BottomRow)
                Result = GC_SI_SOUTH_WEST;
        }
        else
          if (MiddleColumn)
          {
            if (TopRow)
              Result = GC_SI_NORTH;
            else
              if (BottomRow)
                Result = GC_SI_SOUTH;
          }
          else
            if (RightColumn)
            {
              if (TopRow)
                Result = GC_SI_NORTH_EAST;
              else
                if (MiddleRow)
                  Result = GC_SI_EAST;
                else
                  if (BottomRow)
                    Result = GC_SI_SOUTH_EAST;
            };

        break;
      };
    };
  };

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Invalidates the selection decoration of the given Instance (or all instances if Instance is NULL) so they are recomputed
 * next time the selection layer draws them.
 *
 * @param Instance The figure instance whose bounds need recomputation. If this parameter is NULL then all bounds are invalidated.
 */
void CSelectionLayer::InvalidateBounds(CFigureInstance* Instance)
{
  if (Instance == NULL)
  {
    for (CSelectionIterator Iterator = FSelection.begin(); Iterator != FSelection.end(); Iterator++)
    {
      if (!Iterator->second.Dirty)
      {
        Iterator->second.Dirty = true;
        glDeleteLists(Iterator->second.DisplayList, 1);
      };
    };
  }
  else
    if (Instance->FSelected)
    {
      CSelectionIterator Iterator = FSelection.find(Instance);
      if ((Iterator != FSelection.end()) && (!Iterator->second.Dirty))
      {
        Iterator->second.Dirty = true;
        glDeleteLists(Iterator->second.DisplayList, 1);
      };
    };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Removes the given figure instance from the current selection.
 *
 * @param Instance The instance to be removed. If it isn't actually selected then nothing happens.
 */
void CSelectionLayer::RemoveFromSelection(CFigureInstance* Instance)
{
  Instance->FSelected = false;
  CSelectionIterator Iterator = FSelection.find(Instance);
  if (Iterator != FSelection.end())
  {
    FSelection.erase(Iterator);
    GetCanvas()->Change(Instance, GC_CHANGE_SELECTION_REMOVE);
    if (!IsUpdating())
      MakeDirty();
  };
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Sets the the value of the given property, if it is a property of this class.
 *
 * @param Property The property to set.
 * @param Value The new value of the property.
 * @return True if the asked property is known in this class, false otherwise (in this case the property is not changed).
 * @note Implicit datatype conversion is performed if datatypes of property and Value parameter do not match.
 *       This can result in lost precision or even conversion errors. So make sure to use the right datatype for the call.
 */
bool CSelectionLayer::SetProperty(TProperty Property, double Value)
{
  bool Result = true;
  switch (Property)
  {
    case GC_PROPERTY_HANDLE_SIZE:
      {
        // If point antialising is enabled then only certain point sizes are allowed. So we simply set the new value
        // and read back what OpenGL will use instead.
        GLfloat LastPointSize;
        glGetFloatv(GL_POINT_SIZE, &LastPointSize);
        glPointSize((float) Value);
        glGetFloatv(GL_POINT_SIZE, &FHandleSize);
        glPointSize(LastPointSize);

        break;
      };
    default:
      {
        Result = false;
        break;
      };
  };

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

/**
 * Sets the the value of the given property, if it is a property of this class.
 *
 * @param Property The property to set.
 * @param Value The new value of the property.
 * @return True if the asked property is known in this class, false otherwise (in this case the property is not changed).
 * @note Implicit datatype conversion is performed if datatypes of property and Value parameter do not match.
 *       This can result in lost precision or even conversion errors. So make sure to use the right datatype for the call.
 */
bool CSelectionLayer::SetProperty(TProperty Property, int Value)
{
  bool Result = true;
  switch (Property)
  {
    case GC_PROPERTY_HANDLE_SIZE:
      {
        // If point antialising is enabled then only certain point sizes are allowed. So we simply set the new value
        // and read back what OpenGL will use instead.
        GLfloat LastPointSize;
        glGetFloatv(GL_POINT_SIZE, &LastPointSize);
        glPointSize((float) Value);
        glGetFloatv(GL_POINT_SIZE, &FHandleSize);
        glPointSize(LastPointSize);

        break;
      };
    default:
      {
        Result = false;
        break;
      };
  };

  return Result;
}

//----------------------------------------------------------------------------------------------------------------------

