/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */


#ifdef ENABLE_JAVA_MODULES

#include "myx_grt_java.h"
#include "myx_shared_aux_functions.h"

#ifdef __GNUC__
#define JAVA_PATH_SEPARATOR ":"
#else
#define JAVA_PATH_SEPARATOR ";"
#endif

#if defined(__WIN__) || defined(_WIN32) || defined(_WIN64)
#include <Basetsd.h>
#else
#define LongToPtr(l) ((void*)(l))
#define PtrToLong(p) ((long)(p))
#endif

#define ENABLE_JNI_DEBUG


#define GREF(env, r) (*env)->NewGlobalRef(env, r)
#define GRTCLASSPREFIX "com/mysql/grt/"

static MYX_GRT_ERROR java_call_function(MYX_GRT_FUNCTION *function, MYX_GRT_VALUE *value, MYX_GRT_VALUE **retval);
static MYX_GRT_ERROR java_init_module(MYX_GRT_MODULE_LOADER *loader, const char *file, MYX_GRT_MODULE **retmodule);
static MYX_GRT_ERROR java_fetch_messages(MYX_GRT_MODULE_LOADER *loader, MYX_GRT_MSGS *msgs);

static jobject j_xml_object_from_grt_value(JNIEnv *env, MYX_GRT_VALUE *value);
static MYX_GRT_VALUE *j_grt_value_from_java_xml_object(MYX_GRT *grt, JNIEnv *env, jobject object);
static jobject j_call_method(JNIEnv *env, jclass theclass, jobject object, const char *name, const char *signature, jvalue *args);
static jobject j_call_static_method(JNIEnv *env, jclass theclass, const char *name, const char *signature, jvalue *args);
static char *j_get_string(JNIEnv *env, jobject strobj);
static MYX_GRT_LIST *j_get_string_list(MYX_JAVA_LOADER *jloader, jobject lobj);

#if defined(__WIN__) || defined(_WIN32) || defined(_WIN64)
typedef jint (JNICALL *CreateJavaVM_t)(JavaVM **pvm, void **env, void *args);

static int get_java_runtimelib_filename(char **runtimelib_filename);
#endif

MYX_GRT_MODULE_LOADER *myx_java_init_loader(MYX_GRT *grt, const char *class_path, MYX_GRT_ERROR *error)
{
  MYX_GRT_MODULE_LOADER *loader= g_new0(MYX_GRT_MODULE_LOADER, 1);
  MYX_JAVA_LOADER *priv= g_new0(MYX_JAVA_LOADER, 1);
  JavaVMInitArgs vm_init_args;
#ifdef ENABLE_JNI_DEBUG
  JavaVMOption options[6];
#else
  JavaVMOption options[1];
#endif
  jclass grt_class;
  static char *file_extensions[]= {
    ".class"
  };
  int res;
  const char *grt_class_path= "java";
  const char *grt_class_path_libraries_dir= "java/lib";
  char *grt_class_path_libraries= NULL;
  GDir *dir;
  const char *entry;
  jmethodID getInstance_methodID;
  jmethodID setMyxGrtPointer_methodID;
  jmethodID setCallback_methodID;
  jobject grt_object;
  jvalue *args= NULL;
#if defined(__WIN__) || defined(_WIN32) || defined(_WIN64)
  char *runtimelib_filename;
  HINSTANCE handle;
  CreateJavaVM_t create_java_vm;
#endif

  *error= MYX_GRT_NO_ERROR;

  loader->grt= grt;
  loader->loader_type= MYX_JAVA_MODULE_TYPE;
  loader->priv= priv;
  loader->init_module= java_init_module;
  loader->call_function= java_call_function;
  loader->fetch_messages= java_fetch_messages;
  loader->extensions_num= 1;
  loader->extensions= file_extensions;

  vm_init_args.version = JNI_VERSION_1_4;
#ifdef ENABLE_JNI_DEBUG
  vm_init_args.nOptions = 5;
#else
  vm_init_args.nOptions = 1;
#endif

#if defined(__WIN__) || defined(_WIN32) || defined(_WIN64)
  //try to locate the JVM .dll
  *error= get_java_runtimelib_filename(&runtimelib_filename);
  if (*error != MYX_GRT_NO_ERROR)
    goto errorfree;
#endif

  //scan for .jars in the lib directory
  dir= g_dir_open(grt_class_path_libraries_dir, 0, NULL);
  if (!dir)
  {
    *error= MYX_GRT_BAD_PATH;
    goto errorfree;
  }

  while ((entry= g_dir_read_name(dir)) != NULL)
  {
    char *path= g_build_filename(grt_class_path_libraries_dir, entry, NULL);

    grt_class_path_libraries= str_g_append_and_free(
        grt_class_path_libraries, g_strdup_printf("%s%s", 
          path, JAVA_PATH_SEPARATOR));

    g_free(path);
  }
  g_dir_close(dir);

  //base class path
  options[0].optionString= g_strdup_printf("-Djava.class.path=%s%s", grt_class_path, JAVA_PATH_SEPARATOR);

  //add library path to class path
  if (grt_class_path_libraries)
    options[0].optionString= str_g_append_and_free(options[0].optionString,
      grt_class_path_libraries);

  //add user defined class_path to class path
  if (class_path)
    options[0].optionString= str_g_append(options[0].optionString,
      class_path);

#ifdef ENABLE_JNI_DEBUG
  options[1].optionString= "-Xdebug";
  options[2].optionString= strdup("-Xrunjdwp:transport=dt_socket,server=y,address=8000,suspend=n");
  options[3].optionString= "-Xnoagent";
  options[4].optionString= "-Djava.compiler=NONE";
  options[5].optionString= "-Xss200m";
#endif

  vm_init_args.options = options;
  vm_init_args.ignoreUnrecognized = JNI_FALSE;

  // Try launch the Java VM
#if defined(__WIN__) || defined(_WIN32) || defined(_WIN64)
  // Try to load the jvm.dll
  if ((handle= LoadLibrary(runtimelib_filename)) == 0)
  {
    *error= MYX_GRT_JAVA_JRE_CANNOT_BE_LOADED;
	  return NULL;
  }

  // Locate the JNI_CreateJavaVM function
  create_java_vm=	(void *)GetProcAddress(handle, "JNI_CreateJavaVM");
  if (create_java_vm == 0)
  {
    *error= MYX_GRT_JAVA_JRE_CANNOT_BE_LOADED;
	  return NULL;
  }

  // Call JNI_CreateJavaVM to get the Java environment
  res= create_java_vm(&priv->jvm, (void **)&priv->env, &vm_init_args);
#else
  res= JNI_CreateJavaVM(&priv->jvm, (void **)&priv->env, &vm_init_args);
#endif

  g_free(options[0].optionString);

  if (res < 0)
    goto error;

  // --------------------------------------------------------------------------------------------------------------------
  // Search class reference for the java Grt class
  grt_class= (*priv->env)->FindClass(priv->env, GRTCLASSPREFIX "Grt");
  if (!grt_class)
  {
    if (getenv("GRT_VERBOSE"))
      g_warning("Could not load java class " GRTCLASSPREFIX "Grt.");
    goto error;
  }
  priv->grt_class= (*priv->env)->NewGlobalRef(priv->env, grt_class);

  // Search for the function callModuleFunction in the Grt class
  priv->grt_call_func= (*priv->env)->GetStaticMethodID(priv->env, 
     priv->grt_class, "callModuleFunction", "(Ljava/lang/Class;Ljava/lang/String;Ljava/lang/String;Ljava/lang/String;)Ljava/lang/String;");
  if (!priv->grt_call_func)
  {
    if (getenv("GRT_VERBOSE"))
      g_warning("The java function callModuleFunction cannot be found in the java class " GRTCLASSPREFIX "Grt.");
    goto error;
  }

  // --------------------------------------------------------------------------------------------------------------------
  //  Create Grt Java Object
  getInstance_methodID= (*priv->env)->GetStaticMethodID(priv->env, priv->grt_class, 
    "getInstance", "()L" GRTCLASSPREFIX "Grt;");
  if (!getInstance_methodID)
  {
    if (getenv("GRT_VERBOSE"))
      g_warning("The method Grt.getInstance() cannot be found.");
    goto error;
  }
  grt_object= (*priv->env)->CallStaticObjectMethodA(priv->env, priv->grt_class, getInstance_methodID, NULL);
  if (!grt_object)
  {
    if (getenv("GRT_VERBOSE"))
      g_warning("The creation of the Java Grt object failed.");
    goto error;
  }

  // --------------------------------------------------------------------------------------------------------------------
  // Set the grt pointer in the Java
  setMyxGrtPointer_methodID= (*priv->env)->GetMethodID(priv->env, priv->grt_class, 
    "setMyxGrtPointer", "(J)V");
  if (!setMyxGrtPointer_methodID)
  {
    if (getenv("GRT_VERBOSE"))
      g_warning("The method Grt.setMyxGrtPointer() cannot be found.");
    goto error;
  }

  args= g_new0(jvalue, 1);
  args[0].j= PtrToLong(grt);

  (*priv->env)->CallVoidMethodA(priv->env, grt_object, setMyxGrtPointer_methodID, args);

  g_free(args);
  args= NULL;

  // --------------------------------------------------------------------------------------------------------------------
  // Set callback object
  setCallback_methodID= (*priv->env)->GetMethodID(priv->env, priv->grt_class, 
    "setCallback", "(Ljava/lang/String;Ljava/lang/String;)V");
  if (!setCallback_methodID)
  {
    if (getenv("GRT_VERBOSE"))
      g_warning("The method Grt.setCallback() cannot be found.");
    goto error;
  }

  args= g_new0(jvalue, 2);
  args[0].l= (*priv->env)->NewStringUTF(priv->env, "GrtCallbackNative");
  {
    char *path= str_g_append(g_get_current_dir(), "/");
    args[1].l= (*priv->env)->NewStringUTF(priv->env, path);
    g_free(path);
  }

  (*priv->env)->CallVoidMethodA(priv->env, grt_object, setCallback_methodID, args);

  g_free(args);
  args= NULL;

  // Cache java objects and methods
  if (java_cache_java_objects_and_methods(priv) != MYX_GRT_NO_ERROR)
    goto error;


  return loader;
  
error:
  *error= MYX_GRT_MODULE_INIT_ERROR;
errorfree:
  g_free(priv);
  g_free(loader);
  g_free(args);
  return NULL;
}

static MYX_GRT_ERROR java_cache_java_objects_and_methods(MYX_JAVA_LOADER *priv)
{
  // ---------------------------------------
  // Find Class class
  priv->java_class= (*priv->env)->FindClass(priv->env, "java/lang/Class");
  if (!priv->java_class)
  {
    if (getenv("GRT_VERBOSE"))
      g_warning("Could not load java class java.lang.Class.");
    return MYX_GRT_MODULE_INIT_ERROR;
  }
  else
    priv->java_class= (*priv->env)->NewGlobalRef(priv->env, priv->java_class);

  // Find Class's getName method
  priv->java_class_getName= (*priv->env)->GetMethodID(priv->env, priv->java_class, 
    "getName", "()Ljava/lang/String;");
  if (!priv->java_class_getName)
  {
    if (getenv("GRT_VERBOSE"))
      g_warning("The method Class.getName()Ljava/lang/String; cannot be found.");
    return MYX_GRT_MODULE_INIT_ERROR;
  }

  // Find Class's getMethods method
  priv->java_class_getMethods= (*priv->env)->GetMethodID(priv->env, priv->java_class, 
    "getMethods", "()[Ljava/lang/reflect/Method;");
  if (!priv->java_class_getMethods)
  {
    if (getenv("GRT_VERBOSE"))
      g_warning("The method Class.getMethods()[Ljava/lang/Object; cannot be found.");
    return MYX_GRT_MODULE_INIT_ERROR;
  }

  // ---------------------------------------
  // Find Method class
  priv->java_method= (*priv->env)->FindClass(priv->env, "java/lang/reflect/Method");
  if (!priv->java_method)
  {
    if (getenv("GRT_VERBOSE"))
      g_warning("Could not load java class java.lang.reflect.Method.");
    return MYX_GRT_MODULE_INIT_ERROR;
  }
  else
    priv->java_method= (*priv->env)->NewGlobalRef(priv->env, priv->java_method);

  // Find Method's getName method
  priv->java_method_getName= (*priv->env)->GetMethodID(priv->env, priv->java_method, 
    "getName", "()Ljava/lang/String;");
  if (!priv->java_method_getName)
  {
    if (getenv("GRT_VERBOSE"))
      g_warning("The method Method.getName()Ljava/lang/String; cannot be found.");
    return MYX_GRT_MODULE_INIT_ERROR;
  }

  // Find Method's getReturnType method
  priv->java_method_getReturnType= (*priv->env)->GetMethodID(priv->env, priv->java_method, 
    "getReturnType", "()Ljava/lang/Class;");
  if (!priv->java_method_getReturnType)
  {
    if (getenv("GRT_VERBOSE"))
      g_warning("The method Method.getReturnType()Ljava/lang/Class; cannot be found.");
    return MYX_GRT_MODULE_INIT_ERROR;
  }

  // ---------------------------------------
  // Find String class
  priv->java_string= (*priv->env)->FindClass(priv->env, "java/lang/String");
  if (!priv->java_string)
  {
    if (getenv("GRT_VERBOSE"))
      g_warning("Could not load java class java.lang.String.");
    return MYX_GRT_MODULE_INIT_ERROR;
  }
  else
    priv->java_string= (*priv->env)->NewGlobalRef(priv->env, priv->java_string);


  // Find Integer class
  priv->java_integer= (*priv->env)->FindClass(priv->env, "java/lang/Integer");
  if (!priv->java_integer)
  {
    if (getenv("GRT_VERBOSE"))
      g_warning("Could not load java class java.lang.Integer.");
    return MYX_GRT_MODULE_INIT_ERROR;
  }
  else
    priv->java_integer= (*priv->env)->NewGlobalRef(priv->env, priv->java_integer);

  // Find Integer's <init> constructor
  priv->java_integer_init= (*priv->env)->GetMethodID(priv->env, priv->java_integer, 
    "<init>", "(I)V");
  if (!priv->java_integer_init)
  {
    if (getenv("GRT_VERBOSE"))
      g_warning("The constructor Integer(I) cannot be found.");
    return MYX_GRT_MODULE_INIT_ERROR;
  }

  // Find Integer's intValue method
  priv->java_integer_intValue= (*priv->env)->GetMethodID(priv->env, priv->java_integer, 
    "intValue", "()I");
  if (!priv->java_integer_intValue)
  {
    if (getenv("GRT_VERBOSE"))
      g_warning("The constructor Integer.intValue()I cannot be found.");
    return MYX_GRT_MODULE_INIT_ERROR;
  }


  // ---------------------------------------
  // Find Double class
  priv->java_double= (*priv->env)->FindClass(priv->env, "java/lang/Double");
  if (!priv->java_double)
  {
    if (getenv("GRT_VERBOSE"))
      g_warning("Could not load java class java.lang.Integer.");
    return MYX_GRT_MODULE_INIT_ERROR;
  }
  else
    priv->java_double= (*priv->env)->NewGlobalRef(priv->env, priv->java_double);

  // Find Double's <init> constructor
  priv->java_double_init= (*priv->env)->GetMethodID(priv->env, priv->java_double, 
    "<init>", "(D)V");
  if (!priv->java_integer_init)
  {
    if (getenv("GRT_VERBOSE"))
      g_warning("The constructor Double(D) cannot be found.");
    return MYX_GRT_MODULE_INIT_ERROR;
  }

  // Find Double's intValue method
  priv->java_double_doubleValue= (*priv->env)->GetMethodID(priv->env, priv->java_double, 
    "doubleValue", "()D");
  if (!priv->java_double_doubleValue)
  {
    if (getenv("GRT_VERBOSE"))
      g_warning("The method Double.doubleValue()D cannot be found.");
    return MYX_GRT_MODULE_INIT_ERROR;
  }

  
  // ---------------------------------------
  // Find GrtObject class
  priv->java_grtobject= (*priv->env)->FindClass(priv->env, GRTCLASSPREFIX "GrtObject");
  if (!priv->java_grtobject)
  {
    if (getenv("GRT_VERBOSE"))
      g_warning("Could not load java class " GRTCLASSPREFIX "GrtObject.");
    return MYX_GRT_MODULE_INIT_ERROR;
  }
  else
    priv->java_grtobject= (*priv->env)->NewGlobalRef(priv->env, priv->java_grtobject);

  // Find GrtObject's get_id method
  priv->java_grtobject_get_id= (*priv->env)->GetMethodID(priv->env, priv->java_grtobject, 
    "get_id", "()Ljava/lang/String;");
  if (!priv->java_grtobject_get_id)
  {
    if (getenv("GRT_VERBOSE"))
      g_warning("The method GrtObject.get_id()Ljava/lang/String; cannot be found.");
    return MYX_GRT_MODULE_INIT_ERROR;
  }

  // ---------------------------------------
  // Find GrtList class
  priv->java_grtlist= (*priv->env)->FindClass(priv->env, GRTCLASSPREFIX "GrtList");
  if (!priv->java_grtlist)
  {
    if (getenv("GRT_VERBOSE"))
      g_warning("Could not load java class " GRTCLASSPREFIX "GrtList.");
    return MYX_GRT_MODULE_INIT_ERROR;
  }
  else
    priv->java_grtlist= (*priv->env)->NewGlobalRef(priv->env, priv->java_grtlist);

  // Find GrtList's size method
  priv->java_grtlist_size= (*priv->env)->GetMethodID(priv->env, priv->java_grtlist, 
    "size", "()I");
  if (!priv->java_grtlist_size)
  {
    if (getenv("GRT_VERBOSE"))
      g_warning("The method GrtList.size()I cannot be found.");
    return MYX_GRT_MODULE_INIT_ERROR;
  }

  // Find GrtList's getObject method
  priv->java_grtlist_getObject= (*priv->env)->GetMethodID(priv->env, priv->java_grtlist, 
    "getObject", "(I)Ljava/lang/Object;");
  if (!priv->java_grtlist_getObject)
  {
    if (getenv("GRT_VERBOSE"))
      g_warning("The method GrtList.getObject(I)Ljava/lang/Object; cannot be found.");
    return MYX_GRT_MODULE_INIT_ERROR;
  }

  // Find GrtList's java_grtlist_getContentType method
  priv->java_grtlist_getContentType= (*priv->env)->GetMethodID(priv->env, priv->java_grtlist, 
    "getContentType", "()Ljava/lang/String;");
  if (!priv->java_grtlist_getContentType)
  {
    if (getenv("GRT_VERBOSE"))
      g_warning("The method GrtList.getContentType()Ljava/lang/String; cannot be found.");
    return MYX_GRT_MODULE_INIT_ERROR;
  }

  // Find GrtList's java_grtlist_getContentStructName method
  priv->java_grtlist_getContentStructName= (*priv->env)->GetMethodID(priv->env, priv->java_grtlist, 
    "getContentStructName", "()Ljava/lang/String;");
  if (!priv->java_grtlist_getContentStructName)
  {
    if (getenv("GRT_VERBOSE"))
      g_warning("The method GrtList.getContentStructName()Ljava/lang/String; cannot be found.");
    return MYX_GRT_MODULE_INIT_ERROR;
  }


  // ---------------------------------------
  // Find GrtHashMap class
  priv->java_grthashmap= (*priv->env)->FindClass(priv->env, GRTCLASSPREFIX "GrtHashMap");
  if (!priv->java_grthashmap)
  {
    if (getenv("GRT_VERBOSE"))
      g_warning("Could not load java class " GRTCLASSPREFIX "GrtHashMap.");
    return MYX_GRT_MODULE_INIT_ERROR;
  }
  else
    priv->java_grthashmap= (*priv->env)->NewGlobalRef(priv->env, priv->java_grthashmap);

  // Find GrtHashMap's getKeys method
  priv->java_grthashmap_getKeys= (*priv->env)->GetMethodID(priv->env, priv->java_grthashmap, 
    "getKeys", "()[Ljava/lang/String;");
  if (!priv->java_grthashmap_getKeys)
  {
    if (getenv("GRT_VERBOSE"))
      g_warning("The method GrtHashMap.getKeys()[Ljava/lang/String; cannot be found.");
    return MYX_GRT_MODULE_INIT_ERROR;
  }

  // Find GrtHashMap's getKeys method
  priv->java_grthashmap_getObject= (*priv->env)->GetMethodID(priv->env, priv->java_grthashmap, 
    "getObject", "(Ljava/lang/String;)Ljava/lang/Object;");
  if (!priv->java_grthashmap_getObject)
  {
    if (getenv("GRT_VERBOSE"))
      g_warning("The method GrtHashMap.getObject(Ljava/lang/String;)Ljava/lang/Object; cannot be found.");
    return MYX_GRT_MODULE_INIT_ERROR;
  }

  // Find GrtHashMap's getContentType method
  priv->java_grthashmap_getContentType= (*priv->env)->GetMethodID(priv->env, priv->java_grthashmap, 
    "getContentType", "()Ljava/lang/String;");
  if (!priv->java_grthashmap_getContentType)
  {
    if (getenv("GRT_VERBOSE"))
      g_warning("The method GrtHashMap.getContentType()Ljava/lang/String; cannot be found.");
    return MYX_GRT_MODULE_INIT_ERROR;
  }

  // Find GrtHashMap's java_grthashmap_getContentStructName method
  priv->java_grthashmap_getContentStructName= (*priv->env)->GetMethodID(priv->env, priv->java_grthashmap, 
    "getContentStructName", "()Ljava/lang/String;");
  if (!priv->java_grthashmap_getContentStructName)
  {
    if (getenv("GRT_VERBOSE"))
      g_warning("The method GrtHashMap.getContentStructName()Ljava/lang/String; cannot be found.");
    return MYX_GRT_MODULE_INIT_ERROR;
  }

  return MYX_GRT_NO_ERROR;
}


static MYX_GRT_ERROR java_init_module(MYX_GRT_MODULE_LOADER *loader, const char *file, MYX_GRT_MODULE **retmodule)
{
  MYX_JAVA_LOADER *jloader= loader->priv;
  MYX_GRT_MODULE *module;
  MYX_JAVA_MODULE *jmodule;
  jclass theclass;
  jobject module_info;
  unsigned int i;
  char *name, *ptr, *class_name;
  char *module_name= NULL;
  MYX_GRT_VALUE *module_functions= NULL;
  char *extends= NULL;

  class_name = g_path_get_basename(file);
  
  ptr= strchr(class_name, '.');
  if (ptr)
    *ptr= 0;

  name= g_strdup_printf(GRTCLASSPREFIX "modules/%s", class_name);
  g_free(class_name);

  theclass= (*jloader->env)->FindClass(jloader->env, name);
  if (!theclass)
  {
    (*jloader->env)->ExceptionDescribe(jloader->env);
    (*jloader->env)->ExceptionClear(jloader->env);

    if (getenv("GRT_VERBOSE"))
      g_warning("Could not load java class %s", name);
    g_free(name);
    return MYX_GRT_MODULE_INIT_ERROR;
  }

  // fetch info about the module
  module_info= j_call_static_method(jloader->env, theclass,
                                    "getModuleInfo", "()Ljava/lang/String;", NULL);

  if (module_info)
  {
    MYX_GRT_VALUE *grt_info;

    grt_info= j_grt_value_from_java_xml_object(NULL, jloader->env, module_info);

    if (!grt_info || myx_grt_value_get_type(grt_info) != MYX_DICT_VALUE)
      g_warning("could not parse xml response data from %s",
                file);
    else
    {
      char *name_start= g_strrstr(myx_grt_dict_item_get_as_string(grt_info, "name"), ".");
      const char *extends= myx_grt_dict_item_get_as_string(grt_info, "extends");

      if (name_start)
        module_name= g_strdup(name_start+1);
      else
        module_name= g_strdup(myx_grt_dict_item_get_as_string(grt_info, "name"));

      module_functions= myx_grt_dict_item_get_value(grt_info, "functions");
      if (module_functions && myx_grt_value_get_type(module_functions)==MYX_LIST_VALUE)
        myx_grt_value_retain(module_functions);
      else
        module_functions= NULL;

      if ((extends) && (extends[0]))
        extends= g_strdup(extends);
    }
    (*jloader->env)->DeleteLocalRef(jloader->env, module_info);
    
    if (!grt_info)
      return MYX_GRT_BAD_MODULE;

    myx_grt_value_release(grt_info);
  }
  else
  {
    // No exception handling needed here because it is handled
    // directly in j_call_static_method

    if (getenv("GRT_VERBOSE"))
      g_warning("Module %s doesn't implement getModuleInfo", file);
    return MYX_GRT_BAD_MODULE;
  }

  if (!module_name || !module_functions)
  {
    if (getenv("GRT_VERBOSE"))
    {
      if (!module_name)
        g_warning("Module info from %s doesn't contain 'name'", file);
      if (!module_functions)
        g_warning("Module info from %s doesn't contain 'functions'", file);
    }
    g_free(module_name);
    g_free(extends);
    if (module_functions)
      myx_grt_value_release(module_functions);
    g_free(name);
    return MYX_GRT_BAD_MODULE;
  }

  // init internal module descriptor
  module= g_new0(MYX_GRT_MODULE, 1);
  jmodule= g_new0(MYX_JAVA_MODULE, 1);

  module->loader= loader;
  module->priv= jmodule;
  module->name= module_name;
  module->path= name;
  module->functions_num= myx_grt_list_item_count(module_functions);
  module->functions= g_new0(MYX_GRT_FUNCTION, module->functions_num);
  for (i= 0; i < module->functions_num; i++)
  {
    MYX_GRT_FUNCTION *func= module->functions+i;
    MYX_JAVA_FUNCTION *jfunc= g_new0(MYX_JAVA_FUNCTION, 1);
    char *tmp= g_strdup(myx_grt_value_as_string(myx_grt_list_item_get(module_functions, i)));
    char *return_type;
    
    func->module= module;

    // do not use myx_grt_parse_function_spec here
    // since we need special handling for the java signature
    func->name= g_strdup(strtok(tmp, ":"));
    jfunc->java_signature= g_strdup(strtok(NULL, ":"));
    func->param_struct_name= NULL;
    return_type= strtok(NULL, ":");
    if ((return_type) && (return_type[0]))
      func->return_struct_name= g_strdup(return_type);
  
    func->priv= jfunc;

    g_free(tmp);
  }
  myx_grt_value_release(module_functions);
  module->extends= extends;

  // java specific module info
  jmodule->class_ref= (*jloader->env)->NewGlobalRef(jloader->env, theclass);

  *retmodule= module;
  
  if (getenv("GRT_VERBOSE"))
    g_message("Initialized module %s", name);

  return MYX_GRT_NO_ERROR;
}

static MYX_GRT_ERROR java_call_function(MYX_GRT_FUNCTION *function, MYX_GRT_VALUE *value, MYX_GRT_VALUE **retval)
{
  MYX_JAVA_LOADER *jloader= function->module->loader->priv;
  MYX_JAVA_MODULE *jmodule= function->module->priv;
  MYX_JAVA_FUNCTION *jfunc= function->priv;
  JNIEnv *env= jloader->env;
  jobject res;
  jvalue args[4];

  args[0].l= jmodule->class_ref;
  args[1].l= (*env)->NewStringUTF(env, function->name);
  args[2].l= (*env)->NewStringUTF(env, jfunc->java_signature);
  args[3].l= j_xml_object_from_grt_value(env, value);

  res= (*env)->CallStaticObjectMethodA(env, jloader->grt_class, jloader->grt_call_func, args);
  if (res)
  {
    *retval= j_grt_value_from_java_xml_object(NULL, env, res);
    (*env)->DeleteLocalRef(env, res);
  }
  else
    *retval= NULL;
  (*env)->DeleteLocalRef(env, args[1].l);
  (*env)->DeleteLocalRef(env, args[2].l);
  (*env)->DeleteLocalRef(env, args[3].l);

  if (!res)
  {
    (*env)->ExceptionDescribe(env);
    (*env)->ExceptionClear(env);

    return MYX_GRT_FUNCTION_CALL_ERROR;
  }

  return MYX_GRT_NO_ERROR;
}

static MYX_GRT_ERROR java_fetch_messages(MYX_GRT_MODULE_LOADER *loader, MYX_GRT_MSGS *msgs)
{
  MYX_GRT_ERROR error;
  MYX_GRT_VALUE *res;

  // fetch messages here and put them in msgs  

  //call getMessages from the GrtBase module
  res= myx_grt_function_get_and_call(loader->grt, "Grt", "getMessagesAsList", 0, NULL, &error);

  //check if the result is ok
  if ((res) && (error == MYX_GRT_NO_ERROR) && (myx_grt_value_get_type(res) == MYX_DICT_VALUE))
  {
    MYX_GRT_VALUE *value= myx_grt_dict_item_get_value(res, "value");

    //check if the "value" is a list of the correct type
    if (value && (myx_grt_value_get_type(value) == MYX_LIST_VALUE) && 
      (strcmp2(myx_grt_list_content_get_struct_name(value), "GrtMessage") == 0))
    {
      unsigned int i;

      //allocate message list
      msgs->msgs_num= myx_grt_list_item_count(value);
      msgs->msgs= g_new0(MYX_GRT_MSG, msgs->msgs_num);

      //loop over all items of the returned list
      for (i= 0; i<msgs->msgs_num; i++)
      {
        MYX_GRT_VALUE *val_msg= myx_grt_list_item_get(value, i);
        MYX_GRT_MSG *msg= msgs->msgs+i;

        msg->msg_type= myx_grt_dict_item_get_as_int(val_msg, "msg_type");
        msg->msg= g_strdup(myx_grt_dict_item_get_as_string(val_msg, "msg"));
        msg->msg_detail= myx_grt_list_as_stringlist(
          myx_grt_dict_item_get_value(val_msg, "details"));
      }
    }
    else
      return MYX_GRT_INTERNAL_ERROR;
  }
  else
    return MYX_GRT_INTERNAL_ERROR;

  return MYX_GRT_NO_ERROR;
}

//-----------------------------------------------------------------------------
// Private Stuff
//-----------------------------------------------------------------------------


static jobject j_call_method(JNIEnv *env, jclass theclass, jobject object, const char *name, const char *signature, jvalue *args)
{
  jmethodID methodID;
  jobject res= NULL;

  methodID= (*env)->GetMethodID(env, theclass, name, signature);
  if (methodID)
    res= (*env)->CallObjectMethodA(env, object, methodID, args);
  else
  {
    if (getenv("GRT_VERBOSE"))
      g_warning("Method '%s' / %s not found in class", name, signature);
  }
    
  return res;
}


static jobject j_call_static_method(JNIEnv *env, jclass theclass, const char *name, const char *signature, jvalue *args)
{
  jobject res = NULL;
  jmethodID methodID;

  methodID= (*env)->GetStaticMethodID(env, theclass, name, signature);
  if (methodID)
    res= (*env)->CallStaticObjectMethodA(env, theclass, methodID, args);
  else
  {
    //Clear the exception if the Method cannot be found
    (*env)->ExceptionClear(env);

    if (getenv("GRT_VERBOSE"))
      g_warning("Static method '%s' / %s not found in class", name, signature);
  }

  return res;
}

static char *j_get_string(JNIEnv *env, jobject strobj)
{
  const char *jstr= (*env)->GetStringUTFChars(env, strobj, 0);
  char *str= NULL;

  if (jstr)
  {
    str= g_strdup(jstr);

    (*env)->ReleaseStringUTFChars(env, strobj, jstr);
  }
  else
    str= g_strdup("");

  return str;
}


static jobject j_xml_object_from_grt_value(JNIEnv *env, MYX_GRT_VALUE *value)
{
  jobject jstr= NULL;
  char *data;

  data= myx_grt_value_to_xml(value);
  if (data)
  {
    jstr= (*env)->NewStringUTF(env, data);

    g_free(data);
  }

  return jstr;
}

static jobject j_xml_global_object_from_grt_value(JNIEnv *env, const char *objectPath, MYX_GRT_VALUE *value)
{
  jobject jstr= NULL;
  char *data;

  data= myx_grt_value_to_xml_global_object(objectPath, value);
  if (data)
  {
    jstr= (*env)->NewStringUTF(env, data);

    g_free(data);
  }

  return jstr;
}


static MYX_GRT_VALUE *j_grt_value_from_java_xml_object(MYX_GRT *grt, JNIEnv *env, jobject strobj)
{
  MYX_GRT_VALUE *value= NULL;
  const char *jstr= (*env)->GetStringUTFChars(env, strobj, 0);

  if (jstr)
  {
    /*FILE *file= myx_fopen("C:\\test.xml", "w");
    fprintf(file, jstr);
    fclose(file);*/

    if (grt)
      value= myx_grt_value_from_xml_global_object(grt, jstr, strlen(jstr));
    else
      value= myx_grt_value_from_xml(jstr, strlen(jstr));

    (*env)->ReleaseStringUTFChars(env, strobj, jstr);
  }
  return value;
}

static MYX_JAVA_LOADER *find_java_loader(MYX_GRT *grt)
{
  MYX_JAVA_LOADER *jloader= NULL;
  unsigned int i;

  // get Java loader
  for (i= 0; i < grt->loaders_num; i++)
  {
    MYX_GRT_MODULE_LOADER *loader= grt->loaders[i];
    if (loader->loader_type == MYX_JAVA_MODULE_TYPE)
    {
      jloader= loader->priv;
      break;
    }
  }

  return jloader;
}

static jobject java_global_object_from_grt_value(MYX_GRT *grt, JNIEnv *env, const char *objectPath, MYX_GRT_VALUE *value)
{
  jobject jobj= NULL;
  MYX_JAVA_LOADER *jloader= find_java_loader(grt);

  if ( (value) && (jloader) )
  {
    if (myx_grt_value_get_type(value) == MYX_STRING_VALUE)
    {
      jobj= (*env)->NewStringUTF(env, myx_grt_value_as_string(value));
    }
    else if (myx_grt_value_get_type(value) == MYX_INT_VALUE)
    {
      jvalue *args= g_new0(jvalue, 1);

      args[0].i= myx_grt_value_as_int(value);
      jobj= (*env)->NewObjectA(env, jloader->java_integer, jloader->java_integer_init, args);
      g_free(args);
    }
    else if (myx_grt_value_get_type(value) == MYX_REAL_VALUE)
    {
      jvalue *args= g_new0(jvalue, 1);

      args[0].d= myx_grt_value_as_real(value);
      jobj= (*env)->NewObjectA(env, jloader->java_double, jloader->java_double_init, args);
      g_free(args);
    }
    else if (myx_grt_value_get_type(value) == MYX_LIST_VALUE)
    {
      jvalue *args= g_new0(jvalue, 2);
      jclass j_class;
      jmethodID j_init;
      char *j_class_name= g_strdup(GRTCLASSPREFIX);

      if ( (myx_grt_list_content_get_type(value) == MYX_STRING_VALUE) && 
        (!myx_grt_list_content_get_struct_name(value)) )
        j_class_name= str_g_append(j_class_name, "GrtStringList");
      else if (myx_grt_list_content_get_struct_name(value))
      {
        j_class_name= str_g_append_and_free(j_class_name, 
          str_g_replace(g_strdup(myx_grt_list_content_get_struct_name(value)), ".", "/"));
        j_class_name= str_g_append(j_class_name, "List");
      }
      else
        j_class_name= str_g_append(j_class_name, "GrtList");
        

      // Find List class
      j_class= (*env)->FindClass(env, j_class_name);
      if (!j_class)
        return NULL;

      // Find List's <init> constructor
      j_init= (*env)->GetMethodID(env, j_class, 
        "<init>", "(Ljava/lang/String;Ljava/lang/String;)V");
      if (!j_init)
        return NULL;

      if (myx_grt_list_content_get_struct_name(value))
        args[0].l= (*env)->NewStringUTF(env, myx_grt_list_content_get_struct_name(value));
      else
        args[0].l= (*env)->NewStringUTF(env, "");
      args[1].l= (*env)->NewStringUTF(env, objectPath);
      jobj= (*env)->NewObjectA(env, j_class, j_init, args);
      g_free(args);

      g_free(j_class_name);
    }
    else if (myx_grt_value_get_type(value) == MYX_DICT_VALUE)
    {
      jvalue *args;
      jclass j_class;
      jmethodID j_init;
      char *j_class_name= g_strdup(GRTCLASSPREFIX);
      
      if (myx_grt_dict_struct_get_name(value))
      {
        // this is a struct instance
        j_class_name= str_g_append_and_free(j_class_name, 
          str_g_replace(g_strdup(myx_grt_dict_struct_get_name(value)), ".", "/"));

        // Find Dict class
        j_class= (*env)->FindClass(env, j_class_name);
        if (!j_class)
          return NULL;
          
        // Find Dict's <init> constructor
        j_init= (*env)->GetMethodID(env, j_class, "<init>", "(Ljava/lang/Object;)V");
        if (!j_init)
          return NULL;

        /// Create new object instance
        args= g_new0(jvalue, 1);
        args[0].l= (*env)->NewStringUTF(env, objectPath);
        jobj= (*env)->NewObjectA(env, j_class, j_init, args);
        g_free(args);
      }
      else
      {
        if ( (myx_grt_dict_content_get_type(value) == MYX_STRING_VALUE) && (!myx_grt_dict_struct_get_name(value)) )
          j_class_name= str_g_append(j_class_name, "GrtStringHashMap");
        else
          j_class_name= str_g_append(j_class_name, "GrtHashMap");

        // Find List class
        j_class= (*env)->FindClass(env, j_class_name);
        if (!j_class)
          return NULL;
          
        // Find List's <init> constructor
        j_init= (*env)->GetMethodID(env, j_class, "<init>", "(Ljava/lang/String;Ljava/lang/String;)V");
        if (!j_init)
          return NULL;

        /// Create new object instance
        args= g_new0(jvalue, 2);
        args[0].l= (*env)->NewStringUTF(env, "");
        args[1].l= (*env)->NewStringUTF(env, objectPath);
        jobj= (*env)->NewObjectA(env, j_class, j_init, args);
        g_free(args);
      }

      g_free(j_class_name);
    }

    return jobj;
  }
  else
    return NULL;
}

static MYX_GRT_VALUE *java_object_to_global_grt_value(MYX_GRT *grt, JNIEnv *env, jobject value)
{
  MYX_GRT_VALUE *res= NULL;
  MYX_JAVA_LOADER *jloader= find_java_loader(grt);

  //FILE *f= myx_fopen("C:\\log.txt", "a");

  // check the class of the value
  if ((*env)->IsInstanceOf(env, value, jloader->java_grtlist))
  {
    // List
    jstring j_content_type= (*env)->CallObjectMethodA(env, value, jloader->java_grtlist_getContentType, NULL);
    jstring j_content_struct_name= (*env)->CallObjectMethodA(env, value, jloader->java_grtlist_getContentStructName, NULL);
    char *content_type_name= j_get_string(env, j_content_type);
    char *content_struct_name;
    MYX_GRT_ERROR error;
    MYX_GRT_VALUE_TYPE content_type= myx_get_value_type_from_string(content_type_name, &error);
    jint item_count= (*env)->CallIntMethodA(env, value, jloader->java_grtlist_size, NULL);
    int i;

    if (j_content_struct_name)
      content_struct_name= j_get_string(env, j_content_struct_name);

    /*fprintf(f, "LIST" _br);
    fprintf(f, "content_type_name: %s" _br, content_type_name);
    fprintf(f, "content_struct_name: %s" _br, content_struct_name);
    fprintf(f, "item-count: %d" _br, item_count);*/

    // create list of correct content_type and content_struct_name
    if (error == MYX_GRT_NO_ERROR)
      res= myx_grt_list_new(content_type, content_struct_name);

    g_free(content_type_name);
    g_free(content_struct_name);

    if (error != MYX_GRT_NO_ERROR)
      return NULL;

    // now convert the list's items
    for (i= 0; i < item_count; i++)
    {
      jobject j_item;
      MYX_GRT_VALUE *item_value;
      jvalue *args= g_new0(jvalue, 1);
      
      args[0].i= i;
      j_item= (*env)->CallObjectMethodA(env, value, jloader->java_grtlist_getObject, args);
      g_free(args);

      if (j_item)
      {
        item_value= java_object_to_global_grt_value(grt, env, j_item);
        myx_grt_list_item_add(res, item_value);
        myx_grt_value_release(item_value);
      }
    }
  } 
  else if ((*env)->IsInstanceOf(env, value, jloader->java_grthashmap))
  {
    // Hashmap
    jstring j_content_type= (*env)->CallObjectMethodA(env, value, jloader->java_grthashmap_getContentType, NULL);
    jstring j_content_struct_name= (*env)->CallObjectMethodA(env, value, jloader->java_grthashmap_getContentStructName, NULL);
    char *content_type_name= j_get_string(env, j_content_type);
    char *content_struct_name;
    MYX_GRT_ERROR error;
    MYX_GRT_VALUE_TYPE content_type= myx_get_value_type_from_string(content_type_name, &error);
    jobjectArray keys= (*env)->CallObjectMethodA(env, value, jloader->java_grthashmap_getKeys, NULL);
    int i;

    if (j_content_struct_name)
      content_struct_name= j_get_string(env, j_content_struct_name);

    /*fprintf(f, "HASHMAP" _br);
    fprintf(f, "content_type_name: %s" _br, content_type_name);
    fprintf(f, "content_struct_name: %s" _br, content_struct_name);
    fprintf(f, "key-count: %s" _br, (*env)->GetArrayLength(env, keys));*/

    // create hashmap of correct content_type and content_struct_name
    if (error == MYX_GRT_NO_ERROR)
      res= myx_grt_dict_new_typed(content_type, content_struct_name);

    g_free(content_type_name);
    g_free(content_struct_name);

    if (error != MYX_GRT_NO_ERROR)
      return NULL;

    // now convert the hashmap's items
    for (i= 0; i < (*env)->GetArrayLength(env, keys); i++)
    {      
      jstring j_key= (*env)->GetObjectArrayElement(env, keys, i);
      char *key= j_get_string(env, j_key);
      jobject j_map_value;
      MYX_GRT_VALUE *map_value;
      jvalue *args= g_new0(jvalue, 1);

      args[0].l= j_key;
      j_map_value= (*env)->CallObjectMethodA(env, value, jloader->java_grthashmap_getObject, args);
      g_free(args);

      map_value= java_object_to_global_grt_value(grt, env, j_map_value);
      myx_grt_dict_item_set_value(res, key, map_value);
      myx_grt_value_release(map_value);

      g_free(key);
    }
  } 
  else if ((*env)->IsInstanceOf(env, value, jloader->java_grtobject))
  {
    // Object
    jclass java_obj_class= (*env)->GetObjectClass(env, value);
    jstring j_class_name= (*env)->CallObjectMethodA(env, java_obj_class, jloader->java_class_getName, NULL);
    char *class_name= j_get_string(env, j_class_name);
    char *struct_name;
    MYX_GRT_STRUCT *gstruct;
    unsigned int member_count;

    jobjectArray methods= (*env)->CallObjectMethodA(env, java_obj_class, jloader->java_class_getMethods, NULL);
    unsigned int i;

    // Convert struct name to class name
    struct_name= g_strdup(class_name + strlen(GRTCLASSPREFIX));

    /*fprintf(f, "DICT" _br);
    fprintf(f, "class_name: %s" _br, class_name);
    fprintf(f, "struct_name: %s" _br, struct_name);*/

    // Create object
    res= myx_grt_dict_new(struct_name);

    gstruct= myx_grt_struct_get(grt, struct_name);

    member_count= myx_grt_struct_get_member_count_total(grt, gstruct);

    //fprintf(f, "member_count: %d" _br, member_count);

    // Set object's members
    for (i= 0; i < member_count; i++)
    {
      MYX_GRT_STRUCT_MEMBER *member= myx_grt_struct_get_member_by_index_total(grt, gstruct, i);
      char *method_name= g_strdup("get");
      char *method_sig= g_strdup("()");
      int is_int= 0;
      int is_double= 0;
      jmethodID j_getter;

      method_name= str_g_append(method_name, myx_grt_struct_get_member_name(member));
      method_name[3]= g_ascii_toupper(method_name[3]);

      // build java method signature
      if (myx_grt_struct_member_get_type(member) == MYX_STRING_VALUE)
      {
        method_sig= str_g_append(method_sig, "Ljava/lang/String;");

        if (myx_grt_struct_member_get_is_ref(member) == 1)
          method_name= str_g_append(method_name, "ById");
      }
      else if (myx_grt_struct_member_get_type(member) == MYX_INT_VALUE)
      {
        method_sig= str_g_append(method_sig, "I");
        is_int= 1;
      }
      else if (myx_grt_struct_member_get_type(member) == MYX_REAL_VALUE)
      {
        method_sig= str_g_append(method_sig, "D");
        is_double= 1;
      }
      else if (myx_grt_struct_member_get_type(member) == MYX_LIST_VALUE)
      {
        if (myx_grt_struct_member_get_content_struct_name(member) != NULL)
        {
          char *class_name= str_g_replace(g_strdup(myx_grt_struct_member_get_content_struct_name(member)), ".", "/");
          method_sig= str_g_append(method_sig, "L" GRTCLASSPREFIX);
          method_sig= str_g_append_and_free(method_sig, class_name);
          method_sig= str_g_append(method_sig, "List;");
        }
        else if (myx_grt_struct_member_get_content_type(member) == MYX_STRING_VALUE)
          method_sig= str_g_append(method_sig, "L" GRTCLASSPREFIX "GrtStringList;");
        else
          method_sig= str_g_append(method_sig, "L" GRTCLASSPREFIX "GrtList;");
      }
      else if (myx_grt_struct_member_get_type(member) == MYX_DICT_VALUE)
      {
        if (myx_grt_struct_member_get_struct_name(member) != NULL)
        {
          char *class_name= str_g_replace(g_strdup(myx_grt_struct_member_get_struct_name(member)), ".", "/");
          method_sig= str_g_append(method_sig, "L" GRTCLASSPREFIX); 
          method_sig= str_g_append_and_free(method_sig, class_name);
          method_sig= str_g_append(method_sig, ";");
        }
        else
        {
          if (myx_grt_struct_member_get_content_type(member) == MYX_STRING_VALUE)
            method_sig= str_g_append(method_sig, "L" GRTCLASSPREFIX "GrtStringHashMap;");
          else if ( (myx_grt_struct_member_get_content_type(member) == MYX_DICT_VALUE) &&
            (myx_grt_struct_member_get_content_struct_name(member) != NULL) )
          {
            char *class_name= str_g_replace(g_strdup(myx_grt_struct_member_get_content_struct_name(member)), ".", "/");
            method_sig= str_g_append(method_sig, "L" GRTCLASSPREFIX);
            method_sig= str_g_append_and_free(method_sig, class_name);
            method_sig= str_g_append(method_sig, "HashMap;");
          }
          else
            method_sig= str_g_append(method_sig, "L" GRTCLASSPREFIX "GrtHashMap;");
        }
      }

      //fprintf(f, "method_name: %s - %s" _br, method_name, method_sig);

      j_getter= (*env)->GetMethodID(env, java_obj_class, method_name, method_sig);

      if (j_getter)
      {
        if (is_int)
        {
          jint j_int= (*env)->CallIntMethodA(env, value, j_getter, NULL);
          MYX_GRT_VALUE *member_value= myx_grt_value_from_int(j_int);
          myx_grt_dict_item_set_value(res, myx_grt_struct_get_member_name(member), member_value);
        }
        else if (is_double)
        {
          jdouble j_double= (*env)->CallDoubleMethodA(env, value, j_getter, NULL);
          MYX_GRT_VALUE *member_value= myx_grt_value_from_real(j_double);
          myx_grt_dict_item_set_value(res, myx_grt_struct_get_member_name(member), member_value);
        }
        else
        {
          jobject j_member= (*env)->CallObjectMethodA(env, value, j_getter, NULL);
          if (j_member)
          {
            if ( (myx_grt_struct_member_get_content_struct_name(member) != NULL)
                && (myx_grt_struct_member_get_content_type(member) == MYX_STRING_VALUE) )
            {
              //deal with reference lists
              jint item_count= (*env)->CallIntMethodA(env, j_member, jloader->java_grtlist_size, NULL);
              MYX_GRT_VALUE *ref_list= myx_grt_list_new(MYX_STRING_VALUE, 
                  myx_grt_struct_member_get_content_struct_name(member));
              int j;

              for (j= 0; j < item_count; j++)
              {
                jobject j_item;
                jvalue *args= g_new0(jvalue, 1);
                
                args[0].i= j;
                j_item= (*env)->CallObjectMethodA(env, j_member, jloader->java_grtlist_getObject, args);
                g_free(args);

                if (j_item)
                {
                  jstring j_ref_id= (*env)->CallObjectMethodA(env, j_item, jloader->java_grtobject_get_id, NULL);
                  char *ref_id= j_get_string(env, j_ref_id);
                  myx_grt_list_item_add_as_string(ref_list, ref_id);
                }
              }

              myx_grt_dict_item_set_value(res, myx_grt_struct_get_member_name(member), ref_list);
              myx_grt_value_release(ref_list);
            }
            else
            {
              MYX_GRT_VALUE *member_value= java_object_to_global_grt_value(grt, env, j_member);
              if (member)
              {
                myx_grt_dict_item_set_value(res, myx_grt_struct_get_member_name(member), member_value);
                myx_grt_value_release(member_value);
              }
            }
          }
        }
      }
      else
        (*env)->ExceptionClear(env);

      g_free(method_sig);
      g_free(method_name);
    }

    g_free(class_name);
  }
  else if ((*env)->IsInstanceOf(env, value, jloader->java_integer))
  {
    // Integer
    jint i= (*env)->CallIntMethodA(env, value, jloader->java_integer_intValue, NULL);
    res= myx_grt_value_from_int(i);
  }
  else if ((*env)->IsInstanceOf(env, value, jloader->java_double))
  {
    // Real
    jdouble d= (*env)->CallDoubleMethodA(env, value, jloader->java_double_doubleValue, NULL);
    res= myx_grt_value_from_real(d);
  }
  else if ((*env)->IsInstanceOf(env, value, jloader->java_string))
  {
    // String
    res= myx_grt_value_from_string(j_get_string(env, value));
  }

  //fclose(f);
  return res;
}

// ------------------------------------------------------------------------------------------------------------------
// Function call
//

JNIEXPORT jstring JNICALL Java_com_mysql_grt_GrtCallbackNative_callGrtFunction
  (JNIEnv *env, jobject obj, jlong grtMyxGrtPointer, jstring module, jstring name, jstring arguments)
{
  MYX_GRT *grt= LongToPtr(grtMyxGrtPointer);
  MYX_GRT_ERROR error;
  const char *c_str_module= (*env)->GetStringUTFChars(env, module, 0);
  const char *c_str_name= (*env)->GetStringUTFChars(env, name, 0);
  MYX_GRT_VALUE *result= NULL;
  

  if (c_str_module && c_str_name)
  {
    result= myx_grt_function_get_and_call(grt, c_str_module, c_str_name, 0, 
      j_grt_value_from_java_xml_object(grt, env, arguments), &error);

    if (error != MYX_GRT_NO_ERROR)
    {
      //return an error
    }
  }

  if (c_str_module)
    (*env)->ReleaseStringUTFChars(env, module, c_str_module);
  if (c_str_name)
    (*env)->ReleaseStringUTFChars(env, name, c_str_name);

  return j_xml_object_from_grt_value(env, result);
}

// ------------------------------------------------------------------------------------------------------------------
// getGrtGlobalById
//


JNIEXPORT jobject JNICALL Java_com_mysql_grt_GrtCallbackNative_getGrtGlobalById
  (JNIEnv *env, jobject obj, jlong grtMyxGrtPointer, jstring id)
{
  MYX_GRT *grt= LongToPtr(grtMyxGrtPointer);
  const char *c_str_id= (*env)->GetStringUTFChars(env, id, 0);
  MYX_GRT_VALUE *value= myx_grt_reference_cache_lookup(grt, c_str_id);
  jobject res= NULL;

  if (value)
    res= java_global_object_from_grt_value(grt, env, myx_grt_dict_get_object_path(value), value);

  if (c_str_id)
    (*env)->ReleaseStringUTFChars(env, id, c_str_id);

  return res;
}

// ------------------------------------------------------------------------------------------------------------------
// Getters
//

JNIEXPORT jobject JNICALL Java_com_mysql_grt_GrtCallbackNative_getGrtGlobalAsObject
  (JNIEnv *env, jobject obj, jlong grtMyxGrtPointer, jstring objectPath)
{
  MYX_GRT *grt= LongToPtr(grtMyxGrtPointer);
  const char *c_str_objectPath= (*env)->GetStringUTFChars(env, objectPath, 0);
  MYX_GRT_VALUE *value= myx_grt_value_get_by_path(myx_grt_get_root(grt), c_str_objectPath);
  jobject res;

  //if the value doesn't exist yet, create it
  if (!value)
  {
    MYX_GRT_VALUE *parent_value;
    char *parent_path= g_strdup(c_str_objectPath);
    char *ptr= strrchr(parent_path, '/'); 
    char *member_name= g_strdup(ptr+1);
    *ptr= 0;

    //fetch parent value
    parent_value= myx_grt_value_get_by_path(myx_grt_get_root(grt), parent_path);

    //if the parent is a dict
    if ((parent_value) && (myx_grt_value_get_type(parent_value) == MYX_DICT_VALUE))
    {
      //get the assigned struct
      MYX_GRT_STRUCT *parent_struct= myx_grt_dict_struct_get(grt, parent_value);

      if (parent_struct)
      {
        //get the struct's member with the correct name
        MYX_GRT_STRUCT_MEMBER *member= myx_grt_struct_get_member_by_name(grt, parent_struct, member_name, 1);

        //create a list with the correct content_type and content_struct_name
        if ((member) && (member->value_type == MYX_LIST_VALUE))
        {
          if (member->overrides)
            value= myx_grt_list_new(member->content_type, member->overrides);
          else
            value= myx_grt_list_new(member->content_type, member->content_struct_name);
          myx_grt_dict_item_set_value(parent_value, member_name, value);
        }
      }
    }

    g_free(parent_path);
    g_free(member_name);
  }

  res= java_global_object_from_grt_value(grt, env, c_str_objectPath, value);

  if (c_str_objectPath)
    (*env)->ReleaseStringUTFChars(env, objectPath, c_str_objectPath);

  return res;
}

JNIEXPORT jstring JNICALL Java_com_mysql_grt_GrtCallbackNative_getGrtGlobalAsString
  (JNIEnv *env, jobject obj, jlong grtMyxGrtPointer, jstring objectPath)
{
  MYX_GRT *grt= LongToPtr(grtMyxGrtPointer);
  const char *c_str_objectPath= (*env)->GetStringUTFChars(env, objectPath, 0);
  MYX_GRT_VALUE *value= myx_grt_value_get_by_path(myx_grt_get_root(grt), c_str_objectPath);

  if (c_str_objectPath)
    (*env)->ReleaseStringUTFChars(env, objectPath, c_str_objectPath);

  if (value)
    return (*env)->NewStringUTF(env, myx_grt_value_as_string(value));
  else
    return NULL;
}

JNIEXPORT jint JNICALL Java_com_mysql_grt_GrtCallbackNative_getGrtGlobalAsInt
  (JNIEnv *env, jobject obj, jlong grtMyxGrtPointer, jstring objectPath)
{
  MYX_GRT *grt= LongToPtr(grtMyxGrtPointer);
  const char *c_str_objectPath= (*env)->GetStringUTFChars(env, objectPath, 0);
  MYX_GRT_VALUE *value= myx_grt_value_get_by_path(myx_grt_get_root(grt), c_str_objectPath);

  if (c_str_objectPath)
    (*env)->ReleaseStringUTFChars(env, objectPath, c_str_objectPath);

  if (value)
    return myx_grt_value_as_int(value);
  else
    return 0;
}

JNIEXPORT jdouble JNICALL Java_com_mysql_grt_GrtCallbackNative_getGrtGlobalAsReal
  (JNIEnv *env, jobject obj, jlong grtMyxGrtPointer, jstring objectPath)
{
  MYX_GRT *grt= LongToPtr(grtMyxGrtPointer);
  const char *c_str_objectPath= (*env)->GetStringUTFChars(env, objectPath, 0);
  MYX_GRT_VALUE *value= myx_grt_value_get_by_path(myx_grt_get_root(grt), c_str_objectPath);

  if (c_str_objectPath)
    (*env)->ReleaseStringUTFChars(env, objectPath, c_str_objectPath);

  if (value)
    return myx_grt_value_as_real(value);
  else
    return 0;
}

// ------------------------------------------------------------------------------------------------------------------
// Setters
//

JNIEXPORT void JNICALL Java_com_mysql_grt_GrtCallbackNative_setGrtGlobalFromObject
  (JNIEnv *env, jobject obj, jlong grtMyxGrtPointer, jstring objectPath, jobject value)
{
  MYX_GRT *grt= LongToPtr(grtMyxGrtPointer);
  const char *c_str_objectPath= (*env)->GetStringUTFChars(env, objectPath, 0);
  //MYX_GRT_VALUE *value_grt= myx_grt_value_from_xml(c_str_xml, strlen(c_str_xml));
  MYX_GRT_VALUE *value_grt= java_object_to_global_grt_value(grt, env, value);

  myx_grt_value_set_by_path(myx_grt_get_root(grt), c_str_objectPath, value_grt);

  if (c_str_objectPath)
    (*env)->ReleaseStringUTFChars(env, objectPath, c_str_objectPath);
}


JNIEXPORT void JNICALL Java_com_mysql_grt_GrtCallbackNative_setGrtGlobalFromString
  (JNIEnv *env, jobject obj, jlong grtMyxGrtPointer, jstring objectPath, jstring value)
{
  MYX_GRT *grt= LongToPtr(grtMyxGrtPointer);
  const char *c_str_objectPath= (*env)->GetStringUTFChars(env, objectPath, 0);
  const char *c_value_str= (*env)->GetStringUTFChars(env, value, 0);
  MYX_GRT_VALUE *value_grt= myx_grt_value_from_string(c_value_str);

  myx_grt_value_set_by_path(myx_grt_get_root(grt), c_str_objectPath, value_grt);

  if (c_str_objectPath)
    (*env)->ReleaseStringUTFChars(env, objectPath, c_str_objectPath);
  if (c_value_str)
    (*env)->ReleaseStringUTFChars(env, value, c_value_str);
}

JNIEXPORT void JNICALL Java_com_mysql_grt_GrtCallbackNative_setGrtGlobalFromInt
  (JNIEnv *env, jobject obj, jlong grtMyxGrtPointer, jstring objectPath, jint value)
{
  MYX_GRT *grt= LongToPtr(grtMyxGrtPointer);
  const char *c_str_objectPath= (*env)->GetStringUTFChars(env, objectPath, 0);
  MYX_GRT_VALUE *value_grt= myx_grt_value_from_int(value);

  myx_grt_value_set_by_path(myx_grt_get_root(grt), c_str_objectPath, value_grt);

  if (c_str_objectPath)
    (*env)->ReleaseStringUTFChars(env, objectPath, c_str_objectPath);
}

JNIEXPORT void JNICALL Java_com_mysql_grt_GrtCallbackNative_setGrtGlobalFromReal
  (JNIEnv *env, jobject obj, jlong grtMyxGrtPointer, jstring objectPath, jdouble value)
{
  MYX_GRT *grt= LongToPtr(grtMyxGrtPointer);
  const char *c_str_objectPath= (*env)->GetStringUTFChars(env, objectPath, 0);
  MYX_GRT_VALUE *value_grt= myx_grt_value_from_real(value);

  myx_grt_value_set_by_path(myx_grt_get_root(grt), c_str_objectPath, value_grt);

  if (c_str_objectPath)
    (*env)->ReleaseStringUTFChars(env, objectPath, c_str_objectPath);
}


// ------------------------------------------------------------------------------------------------------------------
// List
//

JNIEXPORT jint JNICALL Java_com_mysql_grt_GrtCallbackNative_getGrtGlobalListSize
  (JNIEnv *env, jobject obj, jlong grtMyxGrtPointer, jstring objectPath)
{
  MYX_GRT *grt= LongToPtr(grtMyxGrtPointer);
  const char *c_str_objectPath= (*env)->GetStringUTFChars(env, objectPath, 0);
  MYX_GRT_VALUE *value= myx_grt_value_get_by_path(myx_grt_get_root(grt), c_str_objectPath);

  if (c_str_objectPath)
    (*env)->ReleaseStringUTFChars(env, objectPath, c_str_objectPath);

  
  if (myx_grt_value_get_type(value) == MYX_LIST_VALUE)
    return myx_grt_list_item_count(value);
  else
    return -1;
}

JNIEXPORT jobject JNICALL Java_com_mysql_grt_GrtCallbackNative_getGrtGlobalListItem
  (JNIEnv *env, jobject obj, jlong grtMyxGrtPointer, jstring objectPath, jint index)
{
  MYX_GRT *grt= LongToPtr(grtMyxGrtPointer);
  const char *c_str_objectPath= (*env)->GetStringUTFChars(env, objectPath, 0);
  MYX_GRT_VALUE *list= myx_grt_value_get_by_path(myx_grt_get_root(grt), c_str_objectPath);
  jobject jobj = NULL;

  if (myx_grt_value_get_type(list) == MYX_LIST_VALUE)
  {
    MYX_GRT_VALUE *list_item= myx_grt_list_item_get(list, index);
    char *item_path= g_strdup_printf("%s/%d", c_str_objectPath, index);

    jobj= java_global_object_from_grt_value(grt, env, item_path, list_item);
    g_free(item_path);
  }

  if (c_str_objectPath)
    (*env)->ReleaseStringUTFChars(env, objectPath, c_str_objectPath);

  return jobj;
}

JNIEXPORT void JNICALL Java_com_mysql_grt_GrtCallbackNative_addGrtGlobalListItem
  (JNIEnv *env, jobject obj, jlong grtMyxGrtPointer, jstring objectPath, jobject value)
{
  MYX_GRT *grt= LongToPtr(grtMyxGrtPointer);
  const char *c_str_objectPath= (*env)->GetStringUTFChars(env, objectPath, 0);
  MYX_GRT_VALUE *list= myx_grt_value_get_by_path(myx_grt_get_root(grt), c_str_objectPath);
  MYX_GRT_VALUE *list_item= java_object_to_global_grt_value(grt, env, value);

  if ( (list) && (list_item) )
    myx_grt_list_item_add(list, list_item);

  if (c_str_objectPath)
    (*env)->ReleaseStringUTFChars(env, objectPath, c_str_objectPath);
}

JNIEXPORT void JNICALL Java_com_mysql_grt_GrtCallbackNative_removeGrtGlobalListItem
  (JNIEnv *env, jobject obj, jlong grtMyxGrtPointer, jstring objectPath, jint value)
{

}


// ------------------------------------------------------------------------------------------------------------------
// Dict
//

JNIEXPORT jobject JNICALL Java_com_mysql_grt_GrtCallbackNative_getGrtGlobalDictItem
  (JNIEnv *env, jobject obj, jlong grtMyxGrtPointer, jstring objectPath, jstring key)
{
  MYX_GRT *grt= LongToPtr(grtMyxGrtPointer);
  const char *c_str_objectPath= (*env)->GetStringUTFChars(env, objectPath, 0);
  const char *c_str_key= (*env)->GetStringUTFChars(env, key, 0);
  MYX_GRT_VALUE *dict= myx_grt_value_get_by_path(myx_grt_get_root(grt), c_str_objectPath);
  jobject jobj= NULL;

  if (myx_grt_value_get_type(dict) == MYX_DICT_VALUE)
  {
    char *value_path= g_strdup_printf("%s/%s", c_str_objectPath, c_str_key);
    MYX_GRT_VALUE *dict_value= myx_grt_dict_item_get_value(dict, c_str_key);

    jobj= java_global_object_from_grt_value(grt, env, value_path, dict_value);
    g_free(value_path);
  }

  if (c_str_objectPath)
    (*env)->ReleaseStringUTFChars(env, objectPath, c_str_objectPath);
  if (c_str_key)
    (*env)->ReleaseStringUTFChars(env, key, c_str_key);

  return jobj;
}

JNIEXPORT void JNICALL Java_com_mysql_grt_GrtCallbackNative_addGrtGlobalDictItem
  (JNIEnv *env, jobject obj, jlong grtMyxGrtPointer, jstring objectPath, jstring key, jobject value)
{
  MYX_GRT *grt= LongToPtr(grtMyxGrtPointer);
  const char *c_str_objectPath= (*env)->GetStringUTFChars(env, objectPath, 0);
  const char *c_str_key= (*env)->GetStringUTFChars(env, key, 0);
  MYX_GRT_VALUE *dict= myx_grt_value_get_by_path(myx_grt_get_root(grt), c_str_objectPath);
  MYX_GRT_VALUE *result= NULL;

  if (c_str_objectPath)
    (*env)->ReleaseStringUTFChars(env, objectPath, c_str_objectPath);

  if (value)
    myx_grt_dict_item_set_value(dict, c_str_key, java_object_to_global_grt_value(grt, env, value));

  if (c_str_key)
    (*env)->ReleaseStringUTFChars(env, key, c_str_key);
}

JNIEXPORT void JNICALL Java_com_mysql_grt_GrtCallbackNative_removeGrtGlobalDictItem
  (JNIEnv *env, jobject obj, jlong grtMyxGrtPointer, jstring objectPath, jstring key)
{

}

// -------------------------------------------------------------------------------------------------

#if defined(__WIN__) || defined(_WIN32) || defined(_WIN64)

static int registry_get_value_as_string(HKEY key, const char *name, char *buffer, int buffer_size)
{
    DWORD reg_value_type;
    DWORD reg_value_size;

    if (((RegQueryValueEx(key, name, 0, &reg_value_type, 0, &reg_value_size) == 0) && 
        (reg_value_type == REG_SZ) && (reg_value_size < (unsigned int)buffer_size))) 
    {
	    if (RegQueryValueEx(key, name, 0, 0, buffer, &reg_value_size) == 0) 
	      return 1;
    }

    return 0;
}

static int get_java_runtimelib_filename(char **runtimelib_filename)
{
    HKEY java14key;
    char buffer[MAXPATHLEN];

    /* Find the current version of the JRE */
    if (RegOpenKeyEx(HKEY_LOCAL_MACHINE, "Software\\JavaSoft\\Java Runtime Environment\\1.4", 
      0, KEY_READ, &java14key) != 0)
	    return MYX_GRT_JAVA_NOT_FOUND;

    if (!registry_get_value_as_string(java14key, "RuntimeLib", buffer, sizeof(buffer))) 
	    return MYX_GRT_JAVA_REGISTRY_CORRUPTED;

    *runtimelib_filename= g_strdup(buffer);

    RegCloseKey(java14key);

    return MYX_GRT_NO_ERROR;
}

#endif

#endif
