/* Copyright (C) 2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */


#ifdef ENABLE_PYTHON_MODULES

#include "myx_grt_python.h"
#include <Python.h>


typedef struct MYX_GRT_MODULE_LOADER_PRIVATE
{
  int status;
} MYX_PYTHON_PRIVATE;


typedef struct MYX_GRT_MODULE_PRIVATE
{
  PyThreadState *state;
  PyObject *module;
} MYX_PYMOD_PRIVATE;



MYX_GRT_MODULE_LOADER *myx_python_init_loader(MYX_GRT *grt, MYX_GRT_ERROR *error)
{
  MYX_GRT_MODULE_LOADER *loader= g_new0(MYX_GRT_MODULE_LOADER, 1);
  MYX_PYTHON_PRIVATE *priv= g_new0(MYX_PYTHON_PRIVATE, 1);
  static char *file_extensions[]= {
    ".py", ".pyc", ".pyo"
  };

  Py_Initialize();
  
  *error= MYX_GRT_NO_ERROR;

  loader->grt= grt;
  loader->loader_type= MYX_PYTHON_MODULE_TYPE;
  loader->priv= priv;
  loader->init_module= py_init_module;
  loader->call_function= py_call_function;
  loader->extensions_num= 3;
  loader->extensions= file_extensions;

  return loader;
}


static MYX_GRT_ERROR py_init_module(MYX_GRT_MODULE_LOADER *loader, const char *file, MYX_GRT_MODULE **retmodule)
{
  MYX_GRT_MODULE *module= NULL;
  MYX_PYMOD_PRIVATE *priv= NULL;
  PyThreadState *pystate;
  PyObject *mod, *pyval, *pyfunc= NULL, *info= NULL;
  unsigned int i;
  FILE *f;

  
  f= fopen(file, "r");
  if (!f)
    return MYX_GRT_CANT_OPEN_FILE;

//  PyEval_AcquireLock();
  pystate= Py_NewInterpreter();

  if (PyRun_SimpleFile(f, (char*)file) != 0)
  {
    fclose(f);
    goto error;
  }
  fclose(f);

  mod= PyDict_GetItemString(PyImport_GetModuleDict(), "__main__");
  if (!mod)
    goto error;

  pyfunc= PyObject_GetAttrString(mod, "getModuleInfo");
  if (!pyfunc || !PyCallable_Check(pyfunc))
  {
    goto error;
  }

  info= PyObject_Call(pyfunc, NULL, NULL);
  if (!info)
    goto error;

  module= g_new0(MYX_GRT_MODULE, 1);
  module->loader= loader;
  module->path= g_strdup(file);
  
  // process name
  pyval= PyDict_GetItemString(info, "name");
  if (!pyval || !PyString_Check(pyval))
  {
    g_warning("moduleInfo for Python module %s contains invalid or no name attribute", file);
    goto error;
  }
  module->name= g_strdup(PyString_AsString(pyval));
  
  // process list of functions
  pyval= PyDict_GetItemString(info, "functions");
  if (!pyval || !PyList_Check(pyval))
  {
    g_warning("moduleInfo for Python module %s contains invalid or no functions attribute", file);
    goto error;
  }
  module->functions_num= PyList_Size(pyval);
  module->functions= g_new0(MYX_GRT_FUNCTION, module->functions_num);
  for (i= 0; i < module->functions_num; i++)
  {
    PyObject *v= PyList_GetItem(pyval, i);
    module->functions[i].module= module;
    myx_grt_parse_function_spec(PyString_AsString(v), &module->functions[i]);
    module->functions[i].priv= NULL;    
  }

  // get extended module name
  pyval= PyDict_GetItemString(info, "extends");
  if (!pyval || !PyString_Check(pyval))
  {
    g_warning("moduleInfo for Python module %s contains invalid or no implements attribute", file);
    goto error;
  }
  module->extends= g_strdup(PyString_AsString(pyval));

  Py_DECREF(pyfunc);
  Py_DECREF(info);

  priv= g_new0(MYX_PYMOD_PRIVATE, 1);
  priv->state= pystate;
  priv->module= mod;

  PyEval_ReleaseThread(pystate);

  // private data
  module->priv= priv;

  *retmodule= module;

  return MYX_GRT_NO_ERROR;
  
error:
  if (module)
  {
    g_free(module->path);
    g_free(module->name);
    for (i= 0; i < module->functions_num; i++)
      g_free(module->functions[i].name);
    g_free(module->functions);
    g_free(module->extends);
    g_free(module);
  }

  Py_XDECREF(pyfunc);
  Py_XDECREF(info);

  Py_EndInterpreter(pystate);

//  PyEval_ReleaseLock();

  return MYX_GRT_MODULE_INIT_ERROR;
}



PyObject *py_value_to_object(MYX_GRT_VALUE *value)
{
  PyObject *obj= NULL;
  unsigned int i;

  switch (value->type)
  {
  case MYX_INT_VALUE:
    obj= PyLong_FromLong(value->value.i);
    break;
  case MYX_REAL_VALUE:
    obj= PyFloat_FromDouble(value->value.r);
    break;
  case MYX_STRING_VALUE:
    obj= PyString_FromString(value->value.s);
    break;
  case MYX_LIST_VALUE:
    obj= PyList_New(value->value.l->items_num);
    for (i= 0; i < value->value.l->items_num; i++)
    {
      PyObject *ch= py_value_to_object(value->value.l->items[i]);
      if (ch)
      {
        PyList_Append(obj, ch);
        Py_DECREF(ch);
      }
    }
    break;
  case MYX_DICT_VALUE:
    obj= PyDict_New();
    for (i= 0; i < value->value.d->items_num; i++)
    {
      PyObject *ch= py_value_to_object(value->value.d->items[i].value);
      if (ch)
      {
        PyDict_SetItemString(obj, value->value.d->items[i].key, ch);
        Py_DECREF(ch);
      }
    }
    break;
  }

  return obj;
}



MYX_GRT_VALUE *py_object_to_value(PyObject *object)
{
  if (PyInt_Check(object))
  {
    return myx_grt_value_from_int(PyInt_AsLong(object));
  }
  else if (PyFloat_Check(object))
  {
    return myx_grt_value_from_real(PyFloat_AsDouble(object));
  }
  else if (PyString_Check(object))
  {
    return myx_grt_value_from_string(PyString_AsString(object));
  }
  else if (PyList_Check(object))
  {
    unsigned int i, count= PyList_GET_SIZE(object);
    MYX_GRT_VALUE_TYPE content_type, ctype;
    MYX_GRT_VALUE *value;

    for (i= 0; i < count; i++)
    {
      PyObject *item= PyList_GET_ITEM(object, i);

      if (PyInt_Check(item))
        ctype= MYX_INT_VALUE;
      else if (PyFloat_Check(item))
        ctype= MYX_REAL_VALUE;
      else if (PyString_Check(item))
        ctype= MYX_STRING_VALUE;
      else if (PyList_Check(item))
        ctype= MYX_LIST_VALUE;
      else if (PyDict_Check(item))
        ctype= MYX_DICT_VALUE;
      else
      {
        PyObject *repr= PyObject_Repr(item);
        g_warning("received invalid data from python function: %s",
                  PyString_AsString(repr));
        Py_DECREF(repr);
        return NULL;
      }

      if (i == 0)
        content_type= ctype;
      else
        if (ctype != content_type)
        {
          g_warning("received value from python has a list with multiple type contents");
          return NULL;
        }
    }

    value= myx_grt_list_new(content_type, NULL);
    if (!value)
      return NULL;

    for (i= 0; i < count; i++)
    {
      PyObject *item= PyList_GET_ITEM(object, i);
      MYX_GRT_VALUE *gitem;

      gitem= py_object_to_value(item);
      myx_grt_list_add_item(value, gitem);
      myx_grt_value_release(gitem);
    }
    return value;
  }
  else if (PyDict_Check(object))
  {
    unsigned int i;
    MYX_GRT_VALUE *value;
    PyObject *pykey, *pyvalue;

    i= 0;
    value= myx_grt_dict_new(NULL);
    while (PyDict_Next(object, &i, &pykey, &pyvalue))
    {
      MYX_GRT_VALUE *tmp;
      
      if (!PyString_Check(pykey))
      {
        PyObject *repr= PyObject_Repr(pykey);
        g_warning("received data from python has non-string key: %s",
                  PyString_AsString(repr));
        Py_DECREF(repr);
        myx_grt_value_release(value);
        return NULL;
      }
      
      tmp= py_object_to_value(pyvalue);
      if (!tmp)
      {
        myx_grt_value_release(value);
        return NULL;
      }

      myx_grt_dict_set_item(value, PyString_AsString(pykey), tmp);
    }

    return value;
  }
  else
  {
    PyObject *repr= PyObject_Repr(object);
    g_warning("received invalid data from python function: %s",
              PyString_AsString(repr));
    Py_DECREF(repr);

    return NULL;
  }
}




MYX_GRT_ERROR py_call_function(MYX_GRT_FUNCTION *function, MYX_GRT_VALUE *value, MYX_GRT_VALUE **retval)
{
  MYX_PYMOD_PRIVATE *priv= function->module->priv;
  PyObject *pyfunc, *pyarg, *pyret;

  PyEval_AcquireThread(priv->state);

  pyfunc= PyObject_GetAttrString(priv->module, function->name);
  if (!pyfunc)
  {
    g_warning("Function %s not found in module %s", function->name,
              function->module->name);
    goto error;
  }

  if (value)
    pyarg= py_value_to_object(value);
  else
    pyarg= NULL;

  pyret = PyObject_CallFunction(pyfunc, "(O)", pyarg);
  if (pyret == NULL) // there was an exception in the function
  {
    *retval= NULL;
  }
  else
  {
    *retval= py_object_to_value(pyret);
  }

  Py_DECREF(pyfunc);
  Py_XDECREF(pyarg);
  Py_XDECREF(pyret);
  
  PyEval_ReleaseThread(priv->state);
  
  return MYX_GRT_NO_ERROR;

error:
  PyEval_ReleaseThread(priv->state);
  
  return MYX_GRT_NO_ERROR;
}


#endif
