unit ScriptPanel;

// Copyright (C) 2003, 2004 MySQL AB
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

interface

uses
  Windows, Messages, SysUtils, Classes, Contnrs, Graphics, Controls,
  VirtualTrees, TntExtCtrls, TntComCtrls, TntClasses, TntStdCtrls,
  gnugettext, auxfuncs, PNGImage,
  MySQLConnection, MySQLResultSet, AuxLists, TntMenus, Forms,
  UCESQLHighlighter, UCEHighlighter, UCEEditorKeyCommands,
  UCEHTMLHighlighter, UCEShared, UniCodeEditor, LexicalTools,
  TextSearch, Options, CommonTypes, SQLErrorGrid;

type
  TScriptPanel = class;
  TScriptEditor = class;
  TScriptErrorGridOld = class;

  TScriptTabSheet = class(TTntPanel)
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  protected
    function GetActiveScriptPanel: TScriptPanel;
    procedure SetActiveScriptPanel(ActiveScriptPanel: TScriptPanel);
  private
    FActiveScriptPanel: TScriptPanel;

    FOnBeforeActiveScriptPanelChanged,
    FOnActivateScriptPanel: TNotifyEvent;
    FTabIndex: Integer;
  public
    ScriptPanels: TList;

    function AddScriptPanel(MySQLConn: TMySQLConn; OptionProvider: IOptionProvider): TScriptPanel;

    property ActiveScriptPanel: TScriptPanel read GetActiveScriptPanel write SetActiveScriptPanel;
    property OnBeforeActiveScriptPanelChanged: TNotifyEvent read FOnBeforeActiveScriptPanelChanged write
      FOnBeforeActiveScriptPanelChanged;
    property OnActivateScriptPanel: TNotifyEvent read FOnActivateScriptPanel write FOnActivateScriptPanel;
    property TabIndex: Integer read FTabIndex write FTabIndex;
  end;

  TScriptPanel = class(TTntPanel)
  private
    FActive: Boolean;
    FCurrentExecutionLine: Integer;
    FLastExecutionLine: Integer;
    FMySQLConn: TMySQLConn;
    FHighlighter: TUCESQLHighlighter;

    FScriptEditor: TScriptEditor;
    FScriptErrorGrid: TSQLErrorGrid;
    FErrorMessages: TObjectList;
    FProgressMonitor: IProgressMonitor;

    function GetActive: Boolean;
    procedure SetActive(Value: Boolean);
  protected
    function GetRunning: Boolean;
    function GetMySQLConn: TMySQLConn;
    function DetermineNextStatement: Integer;
    procedure DoChildControlEnter(Sender: TObject);
    function ExecuteCurrentStatement: Boolean;
    function HasNextStatement: Boolean;
    procedure InitProgress;
    procedure ProcessUserCommand(Sender: TCustomUnicodeEdit; var Command: TEditorCommand; var AChar: WideChar;
      Data: Pointer);
    procedure ResetExecution;
    procedure SetMySQLConn(MySQLConn: TMySQLConn);
    function StatementHasBreakPoint: Boolean;
    procedure UpdateProgress;
    procedure UpdateExecutionLineDisplay(NewLine: Integer);
    procedure DoSelectLine(Sender: TSQLErrorGrid; Line, Position: Integer);
  public
    constructor Create(AOwner: TComponent; OptionProvider: IOptionProvider); reintroduce;
    destructor Destroy; override;

    procedure Continue;
    procedure ExecuteScript;
    procedure StepInto;
    procedure StepOver;
    procedure Stop;

    property Active: Boolean read GetActive write SetActive;
    property MySQLConn: TMySQLConn read GetMySQLConn write SetMySQLConn;
    property ProgressMonitor: IProgressMonitor read FProgressMonitor write FProgressMonitor;
    property Running: Boolean read GetRunning;
    property ScriptEditor: TScriptEditor read FScriptEditor;
  end;

  TScriptBreakPoint = class(TObject)
    Line: Integer;
    constructor Create(Line: Integer);
  end;

  TScriptEditor = class(TUniCodeEdit, IOptionChangeListener)
  private
    FFileName: WideString;
    FBreakpoints: TObjectList;
    FBreakPointLineStyle: IUCELineStyle;
    FCurrentPositionLineStyle: IUCELineStyle;
    FBreakPointMarker: IUCELineMarker;
    FDebugInfoMarker: IUCELineMarker;
    FCurrentPositionMarker: IUCELineMarker;
    FEmptyMarker: IUCELineMarker;
    FOptionProvider: IOptionProvider;
    FTokenizer: TSQLTokenizer;
  protected
    procedure AddExecutionMarker(Line: Integer);
    procedure AnalyzeLine(CommandEndPending: Boolean; Line: TUCELine);
    procedure DeleteLine(Sender: TObject; Line: TUCELine);
    procedure GutterMouseDown(Sender: TCustomUnicodeEdit; Button: TMouseButton; Shift: TShiftState; X, Y, Line: Integer);
    procedure RemoveExecutionMarker(Line: Integer);
    function Unquote(const S: WideString): WideString;
    procedure ValidateLine(Sender: TObject; Line: TUCELine);
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    function DoSearch(Sender: TObject;
      SearchText: WideString; ReplaceText: WideString;
      SearchOptions: TTextSearchOptions): Integer;
    procedure DoEnter; override;
    procedure OptionChanged;
  public
    constructor Create(AOwner: TComponent; OptionProvider: IOptionProvider); reintroduce;
    destructor Destroy; override;

    procedure ChangeLine(Sender: TObject; Line: TUCELine);

    procedure DoDisplaySearch(ShowReplacePage: Boolean = False);

    procedure ToggleBreakpoint(Line: Integer);
    procedure ClearBreakpoints;

    property FileName: WideString read FFileName write FFileName;
  end;

  TScriptErrorGridOld = class(TVirtualStringTree, IOptionChangeListener)
    constructor Create(AOwner: TComponent; OptionProvider: IOptionProvider); reintroduce;
    destructor Destroy; override;

    procedure DoMessagesChanged;

    procedure ClearMessages(Sender: TObject);
  private
    FScriptPanel: TScriptPanel;
    RSErrorPNGImg,
    RSWarningPNGImg: TPNGObject;
    ErrorPopupMenu: TTntPopupMenu;
    FOptionProvider: IOptionProvider;
  protected
    procedure DoGetText(Node: PVirtualNode; Column: TColumnIndex; TextType: TVSTTextType; var Text: WideString); override;
    procedure DoAfterCellPaint(Canvas: TCanvas; Node: PVirtualNode; Column: TColumnIndex; CellRect: TRect); override;
    procedure DblClick; override;
    procedure OptionChanged;
  end;

  PScriptMsg = ^TScriptMsg;
  TScriptMsg = record
    MsgIndex: Integer;
  end;

  TScriptErrorOld = class(TObject)
  public
    Msg: WideString;
    MsgType: TScriptMessageType;
    MsgNr: Integer;
    Line: Integer;
    CharPos: Integer;
    constructor Create(Msg: WideString; MsgType: TScriptMessageType; MsgNr: Integer; Line: Integer; CharPos: Integer);
  end;

//----------------------------------------------------------------------------------------------------------------------

implementation

uses
  Math, PNGTools,
  myx_public_interface,
  StringContainers;

const
  DefaultDelimiter = ';';
  
type
  // Data that is stored at line level.
  // TODO: Line data needs rework once a Code DOM is available.
  PStatement = ^TStatement;
  TStatement = record
    Start,                  // -1 if the statement does not start on this line, otherwise its char position (0-based).
    Stop: Integer;          // -1 if the statement does not end on this line, otherwise the position (0-based) of the
                            // last character of this statement.
    FNewSchema: WideString; // If non-empty then a new database is to be used from now on. No need to extract
                            // the commmand again when executing. 
  end;

  TLineData = class
  private
    FAnalyzed: Boolean;                // True if command start and end have been determined.
    FStatements: array of TStatement;
    FEndState: TTokenizerState;        // The state in which the lexer stopped when the line was parsed.
    FHasBreakPoint: Boolean;           // True if there is currently a break point set for this line, otherwise False.
    FCommandPending: Boolean;          // If True then there is an unfinished command on this or a previous line.
    FDelimiter: WideString;            // The statement delimiter, which is active for this line and all following.
    FNextStatementCounter: Integer;    // For use with GetFirstStatement/GetNextStatement.
  protected
    procedure AddStatement(X1, X2: Integer; Schema: WideString = '');
    function GetFirstStatement: PStatement;
    function GetFirstStatementEnd: Integer;
    function GetFirstStatementStart: Integer;
    function GetLastStatementEnd: Integer;
    function GetNextStatement: PStatement;
    function HasStatementStart: Boolean;

    property CommandPending: Boolean read FCommandPending write FCommandPending;
    property EndState: TTokenizerState read FEndState write FEndState;
  public
    constructor Create; virtual;
  end;

  // Style for our break point lines.
  TBreakPointLineStyle = class(TInterfacedObject, IUCELineStyle)
  protected
    function GetBackground: TColor;
    function GetFontStyles: TFontStyles;
    function GetForceFontStyles: Boolean;
    function GetForeground: TColor;
  end;

  // Style for current debug position.
  TCurrentPositionLineStyle = class(TInterfacedObject, IUCELineStyle)
  protected
    function GetBackground: TColor;
    function GetFontStyles: TFontStyles;
    function GetForceFontStyles: Boolean;
    function GetForeground: TColor;
  end;

  // The marker for break point lines.
  TBreakPointMarker = class(TInterfacedObject, IUCELineMarker)
  private
    FImage: TPngObject;
  protected
    procedure Draw(Index: Integer; Canvas: TCanvas; X, Y: Integer);
    function GetSize(Index: Integer): TSize;
  end;

  // The marker for lines with debug info.
  TDebugInfoMarker = class(TInterfacedObject, IUCELineMarker)
  private
    FImage: TPngObject;
  protected
    procedure Draw(Index: Integer; Canvas: TCanvas; X, Y: Integer);
    function GetSize(Index: Integer): TSize;
  end;

  // The marker for lines with debug info.
  TCurrentPositionMarker = class(TInterfacedObject, IUCELineMarker)
  private
    FImage: TPngObject;
  protected
    procedure Draw(Index: Integer; Canvas: TCanvas; X, Y: Integer);
    function GetSize(Index: Integer): TSize;
  end;

  // The marker for lines without debug info.
  TEmptyMarker = class(TInterfacedObject, IUCELineMarker)
  protected
    procedure Draw(Index: Integer; Canvas: TCanvas; X, Y: Integer);
    function GetSize(Index: Integer): TSize;
  end;

  TLineParseThread = class(TThread)
  private
    FEdit: TScriptEditor;
    FLine: TUCELine;
  protected
    procedure Execute; override;
  public
    constructor Create(Edit: TScriptEditor; Line: TUCELine);
  end;
  
//----------------------------------------------------------------------------------------------------------------------

constructor TScriptTabSheet.Create(AOwner: TComponent);

begin
  inherited Create(AOwner);
  Parent := TWinControl(AOwner);
  BevelOuter := bvNone;
  Caption := '';
  Width := 650;
  Height := 500;

  ScriptPanels := TObjectList.Create;
end;

//----------------------------------------------------------------------------------------------------------------------

destructor TScriptTabSheet.Destroy;

begin
  ScriptPanels.Free;

  inherited;
end;

//----------------------------------------------------------------------------------------------------------------------

function TScriptTabSheet.AddScriptPanel(MySQLConn: TMySQLConn; OptionProvider: IOptionProvider): TScriptPanel;

var
  ScriptPanel, MasterScriptPanel: TScriptPanel;
  Splitter: TTntSplitter;
  ParentForm: TCustomForm;

begin
  ScriptPanel := TScriptPanel.Create(Self, OptionProvider);
  ScriptPanel.Parent := self;

  ScriptPanel.MySQLConn := MySQLConn;

  ScriptPanels.Add(ScriptPanel);

  if (ScriptPanels.Count > 1) then
  begin
    MasterScriptPanel := TScriptPanel(ScriptPanels[ScriptPanels.Count - 2]);

    ScriptPanel.Height := MasterScriptPanel.Height div 2;
    MasterScriptPanel.Height := MasterScriptPanel.Height div 2;
    ScriptPanel.Align := alBottom;

    //Copy events & properties
    ScriptPanel.FScriptEditor.PopupMenu := MasterScriptPanel.FScriptEditor.PopupMenu;

    Splitter := TTntSplitter.Create(self);
    Splitter.Parent := self;
    Splitter.Align := alBottom;
    Splitter.Height := 9;
    Splitter.Top := MasterScriptPanel.Top + MasterScriptPanel.Height + 10;
    Splitter.MinSize := 40;
    Splitter.AutoSnap := False;

    ScriptPanel.Top := MasterScriptPanel.Top + MasterScriptPanel.Height + 10;
  end
  else
    ScriptPanel.Align := alClient;

  ActiveScriptPanel := ScriptPanel;

  //Set Focus to new RSPanel
  ParentForm := GetParentForm(ScriptPanel);
  if (ParentForm <> nil) then
    if (ParentForm.Visible and ParentForm.Enabled) then
      ScriptPanel.SetFocus;
  ScriptPanel.Active := True;

  Result := ScriptPanel;
end;

//----------------------------------------------------------------------------------------------------------------------

function TScriptTabSheet.GetActiveScriptPanel: TScriptPanel;

begin
  Result := FActiveScriptPanel;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TScriptTabSheet.SetActiveScriptPanel(ActiveScriptPanel: TScriptPanel);

var
  i: Integer;

begin
  if (FActiveScriptPanel <> ActiveScriptPanel) then
  begin
    if (Assigned(FOnBeforeActiveScriptPanelChanged)) then
      FOnBeforeActiveScriptPanelChanged(FActiveScriptPanel);

    FActiveScriptPanel := ActiveScriptPanel;

    for i := 0 to ScriptPanels.Count - 1 do
      if (ScriptPanels[i] <> ActiveScriptPanel) then
        TScriptPanel(ScriptPanels[i]).Active := False;

    if (Assigned(FOnActivateScriptPanel)) then
      FOnActivateScriptPanel(FActiveScriptPanel);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

constructor TScriptPanel.Create(AOwner: TComponent; OptionProvider: IOptionProvider);

var
  Stroke: TKeyStroke;
  
begin
  inherited Create(AOwner);

  BevelOuter := bvNone;
  Caption := '';

  FActive := True;
  FErrorMessages := TObjectList.Create;

  // Create and setup SQL syntax highlighter.
  FHighlighter := TUCESQLHighlighter.Create(Self);
  with FHighlighter do
  begin
    CommentAttributes.Foreground := clGray;
    CommentAttributes.Style := [fsItalic];
    KeyAttributes.Foreground := clBlue;
    StringAttributes.Foreground := $0080FF;
    IdentifierAttributes.Foreground := clBlack;
    NumberAttributes.Foreground := clFuchsia;
    SpaceAttributes.Foreground := clWindow;
    SymbolAttributes.Foreground := clBlack;

    CommentWithCommandAttributes.Foreground := clGray;
    CommentWithCommandAttributes.Style := [fsItalic];
    EmbeddedCommandAttributes.Foreground := clNavy;
    EmbeddedCommandAttributes.Background := $F0F0F0;

    SystemVariableAttributes.Foreground := $808000;
    SystemVariableAttributes.Style := [fsBold];
    UserVariableAttributes.Foreground := $C08080;
    UserVariableAttributes.Style := [fsBold];
  end;
  
  // Create and setup the actual SQL script editor.
  FScriptEditor := TScriptEditor.Create(Self, OptionProvider);
  FScriptEditor.Parent := Self;
  with FScriptEditor do
  begin
    Font.Name := OptionProvider.OptionAsString['CodeFontName'];
    Font.Height := OptionProvider.OptionAsInteger['CodeFontHeight'];
    Align := alClient;
    Highlighter := FHighlighter;
    OnEnter := DoChildControlEnter;
    Options := [eoAutoIndent, eoAutoUnindent, eoGroupUndo, eoInserting, eoLineNumbers, eoScrollPastEOL, eoShowScrollHint,
      {eoSmartTabs, }eoTripleClicks, eoUndoAfterSave, eoUseSyntaxHighlighting, eoWantTabs, eoUseUndoRedo];

    GutterWidth := 60;
    RightMargin := 80;
    MaxRightChar := 10000;
    ScrollHintColor.Foreground := clHighlightText;
    ScrollHintColor.Background := clAppWorkSpace;
    SelectedColor.Foreground := clHighlightText;
    SelectedColor.Background := clHighlight;
    LineNumberFont.Name := 'Terminal';
    LineNumberFont.Size := 5;
    OnProcessUserCommand := ProcessUserCommand;
    OnGutterMouseDown := GutterMouseDown;

    // Add own keystrokes.
    // F5, to toggle break points.
    Stroke := Keystrokes.Add;
    Stroke.Command := ecUserFirst;
    Stroke.Key := VK_F5;
    // Shift+Bk like Bk alone
    Stroke := Keystrokes.Add;
    Stroke.Command := ecDeleteLastChar;
    Stroke.Key := VK_BACK;
    Stroke.Shift := [ssShift];
  end;

  FLastExecutionLine := -1;

  // Create error grid.
  FScriptErrorGrid := TSQLErrorGrid.Create(Self, OptionProvider);
  with FScriptErrorGrid do
  begin
    Parent := Self;
    OnEnter := DoChildControlEnter;
    OnSelectLine := DoSelectLine;
  end;

  ResetExecution;
end;

//----------------------------------------------------------------------------------------------------------------------

destructor TScriptPanel.Destroy;

begin
  ResetExecution;
  FErrorMessages.Free;

  inherited;
end;

//----------------------------------------------------------------------------------------------------------------------

function TScriptPanel.GetActive: Boolean;

begin
  Result := FActive;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TScriptPanel.SetActive(Value: Boolean);

begin
  if Value <> FActive then
  begin
    FActive := Value;

    if Value then
    begin
      FScriptEditor.Color := clWindow;
      if FCurrentExecutionLine > -1 then
        InitProgress;
    end
    else
    begin
      FScriptEditor.Color := $00EEEEEE;
      if Assigned(FProgressMonitor) then
        FProgressMonitor.ProgressFinish;
    end;

    // Deactivate other ScriptPanels if this one got active.
    if Value and Parent.InheritsFrom(TScriptTabSheet) then
      TScriptTabSheet(Parent).ActiveScriptPanel := Self;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

function TScriptPanel.GetRunning: Boolean;

begin
  Result := (FCurrentExecutionLine <> -1);
end;

//----------------------------------------------------------------------------------------------------------------------

function TScriptPanel.GetMySQLConn: TMySQLConn;

begin
  Result := FMySQLConn;
end;

//----------------------------------------------------------------------------------------------------------------------

function TScriptPanel.DetermineNextStatement: Integer;

// Iterates through all lines and tries to find one on which a command starts.
// If FCurrentExecutionLine is > 0 then search is started from there.
// Returns the index of the line found or -1 if there is no more.
// Note: Multiple statements on one line are handle as one statement.

var
  I: Integer;
  Data: TLineData;

begin
  Result := -1;
  if FCurrentExecutionLine >= 0 then
    I := FCurrentExecutionLine + 1
  else
    I := 0;
  while I < FScriptEditor.Content.Count do
  begin
    Data := FScriptEditor.Content[I].Data as TLineData;
    if Data.GetFirstStatementStart > -1 then
    begin
      Result := I;
      Break;
    end;
    Inc(I);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TScriptPanel.SetMySQLConn(MySQLConn: TMySQLConn);

begin
  FMySQLConn := MySQLConn;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TScriptPanel.DoChildControlEnter(Sender: TObject);

begin
  Active := True;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TScriptPanel.ExecuteScript;

var
  Finished: Boolean;
  
begin
  try
    FErrorMessages.Clear;
    FScriptErrorGrid.Clear;

    ResetExecution;
    Finished := True;
    InitProgress;
    while HasNextStatement do
    begin
      if StatementHasBreakPoint or not ExecuteCurrentStatement then
      begin
        UpdateExecutionLineDisplay(FCurrentExecutionLine);
        Finished := False;
        UpdateProgress;
        Break;
      end
      else
        FLastExecutionLine := FCurrentExecutionLine;
      UpdateProgress;
    end;

    if Finished then
      ResetExecution;
  except
    ResetExecution;
    raise;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

function TScriptPanel.HasNextStatement: Boolean;

// Determines if there is a non yet executed statment in the script and returns True if so.

begin
  if FLastExecutionLine = FCurrentExecutionLine then
    FCurrentExecutionLine := DetermineNextStatement;
  Result := FCurrentExecutionLine > -1;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TScriptPanel.InitProgress;

begin
  if Assigned(FProgressMonitor) then
  begin
    FProgressMonitor.ProgressInit(FScriptEditor.Content.Count);
    FProgressMonitor.ProgressPosition(FCurrentExecutionLine);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TScriptPanel.ProcessUserCommand(Sender: TCustomUnicodeEdit; var Command: TEditorCommand; var AChar: WideChar;
  Data: Pointer);

begin
  with FScriptEditor do
  begin
    case Command of
      ecUserFirst:
        ToggleBreakPoint(CaretY);
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TScriptPanel.ResetExecution;

// Sets the active statement to none. So on next execution the first statement is used.

begin
  UpdateExecutionLineDisplay(-1);
  FLastExecutionLine := -1;
  if Assigned(FProgressMonitor) then
    FProgressMonitor.ProgressFinish;
  FScriptErrorGrid.Clear;
end;

//----------------------------------------------------------------------------------------------------------------------

function TScriptPanel.StatementHasBreakPoint: Boolean;

// Determines whether the next to execute statement has a break point set.

var
  Data: TLineData;

begin
  Result := FCurrentExecutionLine > -1;
  if Result then
  begin
    Data := FScriptEditor.Content[FCurrentExecutionLine].Data as TLineData;
    Result := Data.FHasBreakPoint;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TScriptPanel.UpdateProgress;

begin
  if Assigned(FProgressMonitor) then
    FProgressMonitor.ProgressPosition(FCurrentExecutionLine);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TScriptPanel.UpdateExecutionLineDisplay(NewLine: Integer);

begin
  FScriptEditor.RemoveExecutionMarker(FCurrentExecutionLine);
  FCurrentExecutionLine := NewLine;
  FScriptEditor.AddExecutionMarker(FCurrentExecutionLine);
  FScriptEditor.CaretY := FCurrentExecutionLine;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TScriptPanel.Continue;

var
  Marker: Integer;
  Finished: Boolean;

begin
  try
    if FCurrentExecutionLine = -1 then
    begin
      // Debugging has not yet started.
      FErrorMessages.Clear;
      FScriptErrorGrid.Clear;
      InitProgress;
    end
    else
    begin
      // Remove the last execution line marker.
      Marker := FCurrentExecutionLine;
      UpdateExecutionLineDisplay(-1);
      FCurrentExecutionLine := Marker;
    end;
    Finished := True;
    while HasNextStatement do
    begin
      if not ExecuteCurrentStatement then
      begin
        UpdateExecutionLineDisplay(FCurrentExecutionLine);
        Finished := False;
        Break;
      end
      else
      begin
        FLastExecutionLine := FCurrentExecutionLine;
        FCurrentExecutionLine := DetermineNextStatement;
        if StatementHasBreakPoint then
        begin
          UpdateExecutionLineDisplay(FCurrentExecutionLine);
          UpdateProgress;
          Finished := False;
          Break;
        end;
      end;
      UpdateProgress;
    end;

    if Finished then
      ResetExecution;
  except
    ResetExecution;
    raise;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TScriptPanel.StepInto;

begin
  StepOver;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TScriptPanel.StepOver;

begin
  // The first step is special.
  if (FCurrentExecutionLine = -1) and HasNextStatement then
  begin
    // Set marker to the first line that gets executed next time.
    UpdateExecutionLineDisplay(FCurrentExecutionLine);
    InitProgress;
  end
  else
    if HasNextStatement then
    begin
      // Only advance execution point if there was no error;
      if ExecuteCurrentStatement then
      begin
        FLastExecutionLine := FCurrentExecutionLine;
        UpdateExecutionLineDisplay(DetermineNextStatement);
        if FCurrentExecutionLine > -1 then
          UpdateProgress
        else
          ResetExecution;
      end;
    end
    else
      ResetExecution; // Finished debugging.
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TScriptPanel.Stop;

begin
  ResetExecution;
end;

//----------------------------------------------------------------------------------------------------------------------

function TScriptPanel.ExecuteCurrentStatement: Boolean;

// Executes the statement that starts at the current statement position line and ends on the same or a following line.
// Returns False if either the statement could not be executed.

var
  StatementString: WideString;
  Start,
  Stop: TPoint;
  Data: TLineData;
  Statement: PStatement;
  
begin
  Result := FCurrentExecutionLine > -1;
  if Result then
  begin
    Start.Y := FCurrentExecutionLine;
    Data := FScriptEditor.Content[Start.Y].Data as TLineData;
    Statement := Data.GetFirstStatement;

    // Anything to do at all?
    if Assigned(Statement) then
    begin
      repeat
        // First check for special case "use db".
        if Length(Statement.FNewSchema) > 0 then
          FMySQLConn.DefaultSchema := Statement.FNewSchema
        else
        begin
          Start.X := Statement.Start;
          Stop.Y := FCurrentExecutionLine;
          Stop.X := Statement.Stop;
          if Stop.X = -1 then
          begin
            // Statement ends on a line somewhere later. Look for it.
            while Stop.Y < FScriptEditor.Content.Count - 1 do
            begin
              Inc(Stop.Y);
              Data := FScriptEditor.Content[Stop.Y].Data as TLineData;
              if not Data.FAnalyzed then
                FScriptEditor.AnalyzeLine(True, FScriptEditor.Content[Stop.Y]);
              Stop.X := Data.GetFirstStatementEnd;
              if Stop.X > -1 then
                Break;
            end;
            Data := nil;
            // If no command end could be found then Stop.Y has the last line as command end line.
            // Use the entire line in this case for the statement.
            if Stop.Y = FScriptEditor.Content.Count - 1 then
              Stop.X := Length(FScriptEditor.Content[Stop.Y].Text);
          end;

          StatementString := FScriptEditor.Content.CollectText(Start, Stop);

          if not (FMySQLConn.ExecuteDirect(StatementString, False)) then
          begin
            FScriptErrorGrid.AddError(myx_mysql_error(FMySQLConn.MySQL), smtError, myx_mysql_errno(FMySQLConn.MySQL),
              Start.Y + 1, Start.X + 1);

            Result := False;
          end;
        end;

        if Data = nil then
          Statement := nil
        else
          Statement := Data.GetNextStatement;
      until (Statement = nil) or not Result;
    end;
  end;
end;

//----------------- TScriptBreakPoint ----------------------------------------------------------------------------------

constructor TScriptBreakPoint.Create(Line: Integer);

begin
  Self.Line := Line;
end;

//----------------- TScriptEditor --------------------------------------------------------------------------------------

constructor TScriptEditor.Create(AOwner: TComponent; OptionProvider: IOptionProvider);

var
  Temp: Integer;

begin
  inherited Create(AOwner);
  FOptionProvider := OptionProvider;

  OptionProvider.AddListener(Self);
  Font.Name := OptionProvider.OptionAsString['CodeFontName'];
  Font.Height := OptionProvider.OptionAsInteger['CodeFontHeight'];
  Temp := OptionProvider.OptionAsInteger['CodeFontWidth'];
  if Temp > 0 then
    CharWidth := Temp;
  Content.OnValidateLine := ValidateLine;
  Content.OnDeleteLine := DeleteLine;
  Content.OnChangeLine := ChangeLine;
  
  FBreakpoints := TObjectList.Create;
  FFileName := '';

  FBreakPointLineStyle := TBreakPointLineStyle.Create;
  FCurrentPositionLineStyle := TCurrentPositionLineStyle.Create;
  FBreakPointMarker := TBreakPointMarker.Create;
  FDebugInfoMarker := TDebugInfoMarker.Create;
  FCurrentPositionMarker := TCurrentPositionMarker.Create;
  FEmptyMarker := TEmptyMarker.Create;
  TabSize := 2;

  MaxRightChar := 128;
  MaxUndo := 32000;

  FTokenizer := TSQLTokenizer.Create;
end;

//----------------------------------------------------------------------------------------------------------------------

destructor TScriptEditor.Destroy;

begin
  FOptionProvider.RemoveListener(Self);
  FBreakpoints.Free;
  FTokenizer.Free;

  inherited;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TScriptEditor.AddExecutionMarker(Line: Integer);

begin
  if (Line > - 1) and (Line < Content.Count) then
  begin
    InvalidateLine(Line);
    Content[Line].PushStyle(FCurrentPositionLineStyle);
    Content[Line].AddMarker(FCurrentPositionMarker);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TScriptEditor.AnalyzeLine(CommandEndPending: Boolean; Line: TUCELine);

// Scans the given line for certain information. It tries to locate a command and stores the current state in the line
// data. Additionally, text is examined to learn when the delimiter is changed.

var
  Data: TLineData;
  Stream: TWideStringStream;
  Token: WideChar;
  I: Integer;
  StartState: TTokenizerState;
  Start: Integer;
  Stop: Integer;
  LastStop: Integer;
  NewSchema: WideString;

begin
  Stream := TWideStringStream.Create(Line.Text);
  if Line.Index = 0 then
    StartState := tsNormal
  else
  begin
    Data := Content[Line.Index - 1].Data as TLineData;
    StartState := Data.FEndState;
    Data.FDelimiter := Data.FDelimiter;
  end;
  Data := Line.Data as TLineData;
  FTokenizer.Initialize(Stream, StartState);
  Start := -1;
  NewSchema := '';
  try
    repeat
      Token := FTokenizer.NextToken;
      case Token of
        toEOF:
          Break;
        toWhiteSpace,
        toSLComment,
        toMLComment:
          Continue;
      else
        if Token = Data.FDelimiter[1] then
        begin
          // Potential delimiter found. Scan further to know for sure.
          Stop := FTokenizer.TokenPosition - 1;
          I := 2;
          while I <= Length(Data.FDelimiter) do
          begin
            Token := FTokenizer.NextToken;
            if Token = Data.FDelimiter[I] then
              Inc(I)
            else
              Break;
          end;

          if (I - 1) =  Length(Data.FDelimiter) then
          begin
            // A statement ends here. Add it to the list, but without the delimiter.
            // If Start is -1 then there is statement, that might contain special comments.
            if (Start = -1) and not CommandEndPending then
            begin
              LastStop := Data.GetLastStatementEnd;
              if LastStop > -1 then
                Start := LastStop + 1
              else
                Start := 0;
            end;
            Data.AddStatement(Start, Stop, NewSchema);
            Start := -1;
            CommandEndPending := False;
          end;
        end
        else
        begin
          // Any other input is simply noticed as part of a command.
          // if command end is True then we have found already a command on this line in a previous run.
          // In this case we have now a new command start and mark this accordingly while the command end becomes open again.
          if (Start = -1) and not CommandEndPending then
          begin
            // If this is a new command text then check if it is the "delimiter" command.
            // In this case we have to update our delimiter variable.
            if FTokenizer.TokenSymbolIs('delimiter') then
            begin
              // Skip white space and comments.
              repeat
                Token := FTokenizer.NextToken;
              until not (Token in [toWhiteSpace, toSLComment, toMLComment]);

              if Token <> toEOF then
              begin
                Data.FDelimiter := Token;
                repeat
                  Token := FTokenizer.NextToken;
                  if Token in [toEOF, toWhiteSpace, toSLComment, toMLComment] then
                    Break;
                  Data.FDelimiter := Data.FDelimiter + Token;
                until False;

                // The delimiter statement ends here. Don't add it to the list.
              end
              else
                Break;
            end
            else
              if FTokenizer.TokenSymbolIs('use') then
              begin
                // Database is about to change. Extract the name and store it in the data.
                // This way we don't need to check every line later for a use command.

                // Skip white spaces and comments.
                repeat
                  Token := FTokenizer.NextToken;
                until not (Token in [toWhiteSpace, toSLComment, toMLComment]);
                if Token <> toEOF then
                begin
                  NewSchema := FTokenizer.TokenString;
                  if Token = toString then
                    NewSchema := Unquote(NewSchema);

                  // Continue main loop until the terminator appears.
                end;
              end
              else
                Start := FTokenizer.TokenPosition;
          end;
        end;
      end;
    until False;

    // If there still a command pending then add it to the line too.
    if Start > -1 then
    begin
      Data.AddStatement(Start, -1);
      CommandEndPending := True;
    end;

    Data.CommandPending := CommandEndPending;
    Data.FEndState := FTokenizer.State;

    Data.FAnalyzed := True;
    if Data.GetFirstStatementStart > -1 then
    begin
      Line.AddMarker(FDebugInfoMarker);
      Line.AddMarker(FEmptyMarker);
    end
    else
    begin
      Line.RemoveMarker(FDebugInfoMarker);
      Line.RemoveMarker(FEmptyMarker);
    end;
  finally
    Stream.Free;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TScriptEditor.DeleteLine(Sender: TObject; Line: TUCELine);

begin
  Line.Data.Free;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TScriptEditor.GutterMouseDown(Sender: TCustomUnicodeEdit; Button: TMouseButton; Shift: TShiftState; X, Y, Line: Integer);

begin
  ToggleBreakpoint(Line);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TScriptEditor.RemoveExecutionMarker(Line: Integer);

begin
  if (Line > - 1) and (Line < Content.Count) then
  begin
    Content[Line].RemoveStyle(FCurrentPositionLineStyle);
    Content[Line].RemoveMarker(FCurrentPositionMarker);
    InvalidateLine(Line);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TScriptEditor.ToggleBreakpoint(Line: Integer);

// Switches the break point for that line or the line that has the command start for this line on or off.

var
  Data: TLineData;

begin
  repeat
    Data := Content[Line].Data as TLineData;
    if Data.GetFirstStatementStart > -1 then
      Break;
    Dec(Line);
  until Line < 0;

  if Line >= 0 then
  begin
    Data.FHasBreakPoint := not Data.FHasBreakPoint;

    if not Data.FHasBreakPoint then
    begin
      // Empty marker for alignment.
      Content[Line].ReplaceMarker(FBreakPointMarker, FEmptyMarker);
      Content[Line].RemoveStyle(FBreakpointLineStyle);
      InvalidateLine(Line);
    end
    else
    begin
      Content[Line].ReplaceMarker(FEmptyMarker, FBreakPointMarker);
      Content[Line].PushStyle(FBreakpointLineStyle);
      InvalidateLine(Line);
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TScriptEditor.ClearBreakpoints;

// Clear all breakpoints

var
  Data: TLineData;
  Line: Integer;

begin
  Line := Content.Count-1;

  repeat
    Data := Content[Line].Data as TLineData;
    if Data.GetFirstStatementStart > -1 then
    begin
      Data.FHasBreakPoint := False;

      Content[Line].ReplaceMarker(FBreakPointMarker, FEmptyMarker);
      Content[Line].RemoveStyle(FBreakpointLineStyle);
      InvalidateLine(Line);
    end;
    Dec(Line);
  until Line < 0;

end;

//----------------------------------------------------------------------------------------------------------------------

function TScriptEditor.Unquote(const S: WideString): WideString;

begin
  SetString(Result, PWideChar(S) + 1, Length(S) - 2);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TScriptEditor.ValidateLine(Sender: TObject; Line: TUCELine);

var
  LineData: TLineData;
  PrevLineData: TLineData;

begin
  LineData := TLineData.Create;
  Line.Data.Free;
  Line.Data := LineData;
  if Line.Index = 0 then
  begin
    // This is the first line in the editor. Validate this here so we always have a safe anchor where
    // iterations can stop later.
    AnalyzeLine(False, Line);
  end
  else
  begin
    // Check if the previous line has been analyzed already. In this case we can quickly analyze this line too.
    PrevLineData := Content[Line.Index - 1].Data as TLineData;
    if Assigned(PrevLineData) and PrevLineData.FAnalyzed then
      AnalyzeLine(PrevLineData.CommandPending, Line);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TScriptEditor.ChangeLine(Sender: TObject; Line: TUCELine);

var
  Data: TLineData;
  CommandWasPending: Boolean;
  LastState: TTokenizerState;
  LineIndex: Integer;

begin
  // Line changed, so we have to reanalyze it.
  Data := Line.Data as TLineData;
  if Assigned(Data) then
  begin
    CommandWasPending := Data.CommandPending;
    LastState := Data.EndState;
    ValidateLine(Sender, Line);
    LineIndex := Line.Index;

    Data := Line.Data as TLineData;
    if (Data.CommandPending <> CommandWasPending) or (Data.EndState <> LastState) then
    begin
      // When either a command is now finished that was not before (or vice versa) or
      // a string/comment was closed that was not before (or vice versa) then revalidate all following lines.
      Inc(LineIndex);
      while LineIndex < Content.Count do
      begin
        Data := Content[LineIndex].Data as TLineData;
        // Stop invalidating if this line has never been validated.
        if Data = nil then
          Break;
        Data.FAnalyzed := False;
        Content.RevalidateLine(Line);
        InvalidateLine(LineIndex);
        Inc(LineIndex);                      
      end;
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TScriptEditor.KeyDown(var Key: Word; Shift: TShiftState);

begin
  if (Key=Ord('F')) and (Shift=[ssCtrl]) then
    DoDisplaySearch
  else
    if (Key=Ord('R')) and (Shift=[ssCtrl]) then
      DoDisplaySearch(True)
    else
      inherited KeyDown(Key, Shift);
end;

//------------------------------------------------------------------------------

procedure TScriptEditor.DoDisplaySearch(ShowReplacePage: Boolean);

var
  Intf: IAdaptable;
  Component: TComponent;
  Provider: IOptionProvider;

begin
  if TextSearchForm=nil then
  begin
    Component := Self;
    while Component.Owner <> nil do
    begin
      if (Component is TCustomForm) and Supports(Component, IAdaptable, Intf) then
      begin
        Provider := Intf.GetAdapter(IOptionProvider) as IOptionProvider;

        TextSearchForm := TTextSearchForm.Create(nil, Provider);
        break;
      end;

      Component := Component.Owner;
    end;
  end;

  if ShowReplacePage then
    TextSearchForm.PageControl.ActivePage:=
      TextSearchForm.ReplaceTabSheet;


  TextSearchForm.Show;
  TextSearchForm.OnSearch := DoSearch;
end;

//------------------------------------------------------------------------------

function TScriptEditor.DoSearch(Sender: TObject;
  SearchText: WideString; ReplaceText: WideString;
  SearchOptions: TTextSearchOptions): Integer;

var
  SOptions: TSearchOptions;
  //ParentForm: TCustomForm;

begin
  //Translate Options
  SOptions := [];
  if tsoBackwards in SearchOptions then
    include(SOptions, soBackwards);
  if tsoEntireScope in SearchOptions then
    include(SOptions, soEntireScope);
  if tsoIgnoreNonSpacing in SearchOptions then
    include(SOptions, soIgnoreNonSpacing);
  if tsoMatchCase in SearchOptions then
    include(SOptions, soMatchCase);
  if tsoPrompt in SearchOptions then
    include(SOptions, soPrompt);
  if tsoRegularExpression in SearchOptions then
    include(SOptions, soRegularExpression);
  if tsoReplace in SearchOptions then
    include(SOptions, soReplace);
  if tsoReplaceAll in SearchOptions then
    include(SOptions, soReplaceAll);
  if tsoSelectedOnly in SearchOptions then
    include(SOptions, soSelectedOnly);
  if tsoSpaceCompress in SearchOptions then
    include(SOptions, soSpaceCompress);
  if tsoWholeWord in SearchOptions then
    include(SOptions, soWholeWord);

  Result := SearchReplace(SearchText, ReplaceText, SOptions);


  {ParentForm := GetParentForm(self);
  if ParentForm<>nil then
    ParentForm.BringToFront;}
end;

//------------------------------------------------------------------------------

procedure TScriptEditor.DoEnter;

begin
  if TextSearchForm<>nil then
  begin
    TextSearchForm.Show;
    TextSearchForm.OnSearch := DoSearch;
  end;

  inherited DoEnter;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TScriptEditor.OptionChanged;

begin
  Font.Name := FOptionProvider.OptionAsString['CodeFontName'];
  Font.Height := FOptionProvider.OptionAsInteger['CodeFontHeight'];
  CharWidth := FOptionProvider.OptionAsInteger['CodeFontWidth'];
end;

//----------------- TScriptErrorGrid -----------------------------------------------------------------------------------

constructor TScriptErrorGridOld.Create(AOwner: TComponent; OptionProvider: IOptionProvider);

var
  TVColumn: TVirtualTreeColumn;
  PopupItem: TTntMenuItem;

begin
  inherited Create(AOwner);

  FOptionProvider := OptionProvider;
  OptionProvider.AddListener(Self);
  FScriptPanel := TScriptPanel(AOwner);
  NodeDataSize := SizeOf(TScriptMsg);

  Header.Options := Header.Options + [hoDblClickResize, hoVisible];
  Header.Font.Name := OptionProvider.OptionAsString['DataFontName'];
  Header.Font.Size := -8;
  Font.Name := OptionProvider.OptionAsString['CodeFontName'];
  Font.Height := OptionProvider.OptionAsInteger['CodeFontHeight'];
  TVColumn := Header.Columns.Add;
  TVColumn.Text := '!';
  TVColumn.Width := 16;
  TVColumn := Header.Columns.Add;
  TVColumn.Text := 'Line';
  TVColumn.Width := 50;
  TVColumn := Header.Columns.Add;
  TVColumn.Text := 'Description';
  TVColumn := Header.Columns.Add;
  TVColumn.Text := 'ErrorNr.';
  TVColumn.Width := 70;
  Header.AutoSizeIndex := 2;
  Header.Options := Header.Options + [hoAutoResize] - [hoColumnResize];
  ParentBackground := False;
  TreeOptions.MiscOptions := TreeOptions.MiscOptions + [toGridExtensions];
  TreeOptions.PaintOptions := TreeOptions.PaintOptions - [toShowRoot, toShowTreeLines];
  TreeOptions.SelectionOptions := TreeOptions.SelectionOptions + [toFullRowSelect];
  TreeOptions.StringOptions := TreeOptions.StringOptions - [toSaveCaptions];
  HintMode := hmTooltip;
  ShowHint := True;
  DefaultNodeHeight := 16;
  Align := alBottom;
  Height := 0;

  ErrorPopupMenu := TTntPopupMenu.Create(AOwner);
  PopupMenu := ErrorPopupMenu;

  PopupItem := TTntMenuItem.Create(ErrorPopupMenu);
  PopupItem.Caption := _('Clear Messages');
  PopupItem.OnClick := ClearMessages;
  ErrorPopupMenu.Items.Add(PopupItem);

  //Load PNG Images
  RSErrorPNGImg := LoadPNGImageFromResource('rs_error');
  RSWarningPNGImg := LoadPNGImageFromResource('rs_warning');
end;

//----------------------------------------------------------------------------------------------------------------------

destructor TScriptErrorGridOld.Destroy;

begin
  FOptionProvider.RemoveListener(Self);
  
  RSErrorPNGImg.Free;
  RSWarningPNGImg.Free;

  inherited;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TScriptErrorGridOld.ClearMessages(Sender: TObject);

begin
  FScriptPanel.FErrorMessages.Clear;
  DoMessagesChanged;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TScriptErrorGridOld.DoMessagesChanged;

var
  Node: PVirtualNode;
  Nodedata: PScriptMsg;

begin
  if (FScriptPanel.FErrorMessages.Count = 0) then
  begin
    Height := 0;
    Clear;
  end
  else
  begin
    Clear;

    Canvas.Font := Font;
    RootNodeCount := FScriptPanel.FErrorMessages.Count;
    Node := GetFirst;
    while Assigned(Node) do
    begin
      NodeData := GetNodeData(Node);
      NodeData.MsgIndex := Node.Index;
      Multiline[Node] := True;
      NodeHeight[Node] := ComputeNodeHeight(Canvas, Node, 2);

      Node := GetNextSibling(Node);
    end;

    FocusedNode := Node;
                         
    if RootNodeCount = 1 then
     ClientHeight := 1; // Client height bug. The first setting seems not to be honoured.
    ClientHeight := Min(RootNode.TotalHeight - DefaultNodeHeight, 200);

    FScriptPanel.FScriptEditor.EnsureCursorPosVisible;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TScriptErrorGridOld.DoGetText(Node: PVirtualNode; Column: TColumnIndex; TextType: TVSTTextType; var Text: WideString);

var
  NodeData: PScriptMsg;

begin
  if (Assigned(OnGetText)) then
    OnGetText(Self, Node, Column, TextType, Text)
  else
  begin
    if (Column = 0) then
    begin
      Text := '';
    end
    else
    begin
      NodeData := GetNodeData(Node);

      if (NodeData <> nil) then
        if (NodeData.MsgIndex < FScriptPanel.FErrorMessages.Count) then
          case Column of
            1:
              Text := IntToStr(TScriptErrorOld(FScriptPanel.FErrorMessages[NodeData.MsgIndex]).Line);
            2:
              Text := TScriptErrorOld(FScriptPanel.FErrorMessages[NodeData.MsgIndex]).Msg;
            3:
              if (TScriptErrorOld(FScriptPanel.FErrorMessages[NodeData.MsgIndex]).MsgNr <> 0) then
                Text := IntToStr(TScriptErrorOld(FScriptPanel.FErrorMessages[NodeData.MsgIndex]).MsgNr)
              else
                Text := '';
          end;
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TScriptErrorGridOld.DoAfterCellPaint(Canvas: TCanvas; Node: PVirtualNode; Column: TColumnIndex; CellRect: TRect);

var
  NodeData: PScriptMsg;

begin
  if (Column = 0) then
  begin
    NodeData := nil;
    if (Node <> nil) then
      NodeData := GetNodeData(Node);

    if (NodeData <> nil) then
      if (NodeData.MsgIndex < FScriptPanel.FErrorMessages.Count) then
      begin
        case TScriptErrorOld(FScriptPanel.FErrorMessages[NodeData.MsgIndex]).MsgType of
          smtError:
            RSErrorPNGImg.Draw(Canvas, Rect(4, 2,
              RSErrorPNGImg.Width + 1, RSErrorPNGImg.Height + 1));
          smtWarning:
            RSWarningPNGImg.Draw(Canvas, Rect(4, 2,
              RSWarningPNGImg.Width + 1, RSWarningPNGImg.Height + 1));
        end;
      end;
  end;

  if (Assigned(OnAfterCellPaint)) then
    OnAfterCellPaint(Self, Canvas, Node, Column, CellRect);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TScriptErrorGridOld.DblClick;

var
  NodeData: PScriptMsg;

begin
  if (Assigned(OnDblClick)) then
    OnDblClick(Self)
  else
  begin
    NodeData := nil;
    if (FocusedNode <> nil) then
      NodeData := GetNodeData(FocusedNode);

    if (NodeData <> nil) then
    begin
      FScriptPanel.FScriptEditor.CaretY :=
        TScriptErrorOld(FScriptPanel.FErrorMessages[NodeData.MsgIndex]).Line-1;
      FScriptPanel.FScriptEditor.SetFocus;
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TScriptErrorGridOld.OptionChanged;

begin
  Font.Name := FOptionProvider.OptionAsString['CodeFontName'];
  Font.Height := FOptionProvider.OptionAsInteger['CodeFontHeight'];
  Header.Font.Name := FOptionProvider.OptionAsString['DataFontName'];
  Header.Font.Size := -8;
end;

//----------------- TScriptError ---------------------------------------------------------------------------------------

constructor TScriptErrorOld.Create(Msg: WideString; MsgType: TScriptMessageType; MsgNr: Integer; Line: Integer;
  CharPos: Integer);

begin
  Self.Msg := Msg;
  Self.MsgType := MsgType;
  Self.MsgNr := MsgNr;
  Self.Line := Line;
  Self.CharPos := CharPos;
end;

//----------------- TLineData ------------------------------------------------------------------------------------------

constructor TLineData.Create;

begin
  FDelimiter := DefaultDelimiter;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TLineData.AddStatement(X1, X2: Integer; Schema: WideString = '');

var
  Count: Integer;

begin
  Count := Length(FStatements);
  SetLength(FStatements, Count + 1);
  FStatements[Count].Start := X1;
  FStatements[Count].Stop := X2;
  FStatements[Count].FNewSchema := Schema;
end;

//----------------------------------------------------------------------------------------------------------------------

function TLineData.GetFirstStatement: PStatement;

begin
  Result := nil;
  FNextStatementCounter := 0;
  if Length(FStatements) > 0 then
  begin
    Result := @FStatements[FNextStatementCounter];
    Inc(FNextStatementCounter);
    if Result.Start = -1 then
    begin
      if FNextStatementCounter < Length(FStatements) then
      begin
        Result := @FStatements[FNextStatementCounter];
        Inc(FNextStatementCounter);
      end
      else
        Result := nil;
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

function TLineData.GetFirstStatementEnd: Integer;

// Returns the position of the end of the first command on this line. The command does not necessarily start on this line.

begin
  Result := -2;
  if Length(FStatements) > 0 then
    Result := FStatements[0].Stop
end;

//----------------------------------------------------------------------------------------------------------------------

function TLineData.GetLastStatementEnd: Integer;

// Returns the end position of the last command that ends on this line. The command does not necessarily start on this line
// but cannot have any other complete command after it. An incomplete command could start there, though.

var
  Count: Integer;

begin
  Result := -2;
  Count := High(FStatements);
  if Count > -1 then
  begin
    Result := FStatements[Count].Stop;
    if (Result = -1) and (Count > 0) then
      Result := FStatements[Count - 1].Stop;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

function TLineData.GetFirstStatementStart: Integer;

// Returns the position of the first command that starts on this line. The command does not necessarily start on this line
// and may also have parts of a previous command that started on another line before it.

begin
  Result := -2;
  if Length(FStatements) = 1 then
    Result := FStatements[0].Start
  else
    if Length(FStatements) > 1 then
      Result := FStatements[1].Start;
end;

//----------------------------------------------------------------------------------------------------------------------

function TLineData.GetNextStatement: PStatement;

// Returns the next statement in the list of statements. In order to work correctly you *must* call
// GetFirstStatement once before you can iterate further with GetNextStatement.

begin
  Result := nil;
  if FNextStatementCounter < Length(FStatements) then
  begin
    Result := @FStatements[FNextStatementCounter];
    Inc(FNextStatementCounter);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

function TLineData.HasStatementStart: Boolean;

// Determines if this line contains the start of a command.

begin
  // If there is more than one statement then there is for sure one that starts on this line.
  Result := Length(FStatements) > 1;
  if not Result and (Length(FStatements) > 0) then
  begin
    // Only one statement.
    Result := FStatements[0].Start > -1;
  end;
end;

//----------------- TBreakPointLineStyle -------------------------------------------------------------------------------

function TBreakPointLineStyle.GetBackground: TColor;

begin
  Result := $0052FF;
end;

//----------------------------------------------------------------------------------------------------------------------

function TBreakPointLineStyle.GetFontStyles: TFontStyles;

begin
  Result := [];
end;

//----------------------------------------------------------------------------------------------------------------------

function TBreakPointLineStyle.GetForceFontStyles: Boolean;

begin
  Result := False;
end;

//----------------------------------------------------------------------------------------------------------------------

function TBreakPointLineStyle.GetForeground: TColor;

begin
  Result := clWhite;
end;

//----------------- TCurrentpositionLineStyle --------------------------------------------------------------------------

function TCurrentPositionLineStyle.GetBackground: TColor;

begin
  Result := $C69A2F;
end;

//----------------------------------------------------------------------------------------------------------------------

function TCurrentPositionLineStyle.GetFontStyles: TFontStyles;

begin
  Result := [];
end;

//----------------------------------------------------------------------------------------------------------------------

function TCurrentPositionLineStyle.GetForceFontStyles: Boolean;

begin
  Result := False;
end;

//----------------------------------------------------------------------------------------------------------------------

function TCurrentPositionLineStyle.GetForeground: TColor;

begin
  Result := clWhite;
end;

//----------------- TBreakPointMarker ----------------------------------------------------------------------------------

procedure TBreakPointMarker.Draw(Index: Integer; Canvas: TCanvas; X, Y: Integer);

begin
  if FImage = nil then
    FImage := LoadPNGImageFromResource('gutter_breakpoint');

  FImage.Draw(Canvas, Rect(X, Y, X + FImage.Width, Y + FImage.Height));
end;

//----------------------------------------------------------------------------------------------------------------------

function TBreakPointMarker.GetSize(Index: Integer): TSize;

begin
  if FImage = nil then
    FImage := LoadPNGImageFromResource('gutter_breakpoint');

  Result.cx := FImage.Width;
  Result.cy := FImage.Height;
end;

//----------------- TDebugInfoMarker -----------------------------------------------------------------------------------

procedure TDebugInfoMarker.Draw(Index: Integer; Canvas: TCanvas; X, Y: Integer);

begin
  if FImage = nil then
    FImage := LoadPNGImageFromResource('gutter_query_start');

  FImage.Draw(Canvas, Rect(X, Y, X + FImage.Width, Y + FImage.Height));
end;

//----------------------------------------------------------------------------------------------------------------------

function TDebugInfoMarker.GetSize(Index: Integer): TSize;

begin
  if FImage = nil then
    FImage := LoadPNGImageFromResource('gutter_query_start');

  Result.cx := FImage.Width;
  Result.cy := FImage.Height;
end;

//----------------- TCurrentPositionMarker -----------------------------------------------------------------------------

procedure TCurrentPositionMarker.Draw(Index: Integer; Canvas: TCanvas; X, Y: Integer);

begin
  if FImage = nil then
    FImage := LoadPNGImageFromResource('gutter_current_pos');

  FImage.Draw(Canvas, Rect(X, Y, X + FImage.Width, Y + FImage.Height));
end;

//----------------------------------------------------------------------------------------------------------------------

function TCurrentPositionMarker.GetSize(Index: Integer): TSize;

begin
  if FImage = nil then
    FImage := LoadPNGImageFromResource('gutter_current_pos');

  Result.cx := FImage.Width;
  Result.cy := FImage.Height;
end;

//----------------- TEmptyMarker ---------------------------------------------------------------------------------------

procedure TEmptyMarker.Draw(Index: Integer; Canvas: TCanvas; X, Y: Integer);

begin
  // Nothing to draw.
end;

//----------------------------------------------------------------------------------------------------------------------

function TEmptyMarker.GetSize(Index: Integer): TSize;

begin
  Result.cx := 12;
  Result.cy := 16;
end;

//----------------- TLineParseThread -----------------------------------------------------------------------------------

constructor TLineParseThread.Create(Edit: TScriptEditor; Line: TUCELine);

begin
  inherited Create(True);

  FEdit := Edit;
  FLine := Line;
  FreeOnTerminate := True;
  Resume;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TLineParseThread.Execute;

var
  Data: TLineData;
  Index: Integer;

begin
  Data := FLine.Data as TLineData;

  // Check again if the data is not already parsed.
  if not Data.FAnalyzed then
  begin
    // If not already analyzed then scan back and find the last parsed line.
    Index := FLine.Index;
    while Index >= 0 do
    begin
      Data := FEdit.Content[Index].Data as TLineData;
      if Assigned(Data) and Data.FAnalyzed then
        Break;
      Dec(Index);
    end;

    // The previous loop safely stops at line 0 if all fails.
    // Now go forward and validate all lines from between the last parsed and the one needed.
    Inc(Index);
    while Index < FLine.Index do
    begin
      Data := FEdit.Content[Index].Data as TLineData;
      FEdit.AnalyzeLine(Data.CommandPending, FEdit.Content[Index]);
      Inc(Index);
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TScriptPanel.DoSelectLine(Sender: TSQLErrorGrid; Line, Position: Integer);

begin
  FScriptEditor.CaretXY := Point(Position, Line);
  FScriptEditor.SetFocus; 
end;

//----------------------------------------------------------------------------------------------------------------------

end.

