#!@PERL@
#
# This test is for testing how long it takes to create tables,
# make a count(*) from them and finally drop the tables. These
# commands will be done in different ways in this test.
# Using option --fast will drop all the tables in the end
# of this test with one command instead of making own
# 'drop' command for each and every table.
# By changing the variable '$table_amount' value you can make
# this test a lot harder/easier for your computer to drive.
# Note that when using value bigger than 64 for this variable
# will do 'drop table'-command	in totally different way because of that
# how MySQL handles these commands.

##################### Standard benchmark inits ##############################

use DBI;
use Benchmark;

$opt_loop_count=1000; # Change this to make test harder/easier
# This is the default value for the amount of tables used in this test.

chomp($pwd = `pwd`); $pwd = "." if ($pwd eq '');
require "$pwd/bench-init.pl" || die "Can't read Configuration file: $!\n";

if ($opt_small_test)
{
  $opt_loop_count/=10;
}

print "Testing the speed of creating and droping tables\n";
print "All tests are done $opt_loop_count times\n\n";

####
####  Connect and start timeing
####

$dbh = $server->connect();

### Here we'll make a loop $opt_loop_count times, which includes 1 create table,
### 1 count * from table and 1 drop table. Then we'll check how much
### time did it take.

if ($opt_force) # If tables used in this test exist, drop 'em
{
  print "Okay..Let's make sure that our tables don't exist yet.\n\n";
  for ($i=1 ; $i<=$opt_loop_count ; $i++)
  {
    $dbh->do("drop table bench_$i");
  }
}

if ($opt_fast && $opt_server eq "pg")
{
  $server->vacuum($dbh);
}

print "Testing create of tables\n";

$loop_time=$start_time=new Benchmark;

for ($i=1 ; $i <= $opt_loop_count ; $i++)
{
  do_many($dbh,$server->create("bench_$i",
			       ["i int NOT NULL",
				"d double",
				"f float",
				"s char(10)",
				"v varchar(100)"],
			       ["primary key (i)"]));
}

$end_time=new Benchmark;
print "Time for create_table ($opt_loop_count): " .
  timestr(timediff($end_time, $loop_time),"all") . "\n\n";

if ($opt_fast && $opt_server eq "pg")
{
  $server->vacuum($dbh);
}

#### Here comes $opt_loop_count couples of cont(*) to the tables.
#### We'll check how long it will take...
####

print "Accessing tables\n";

if ($limits->{'group_functions'})
{
  $query="select count(*) from ";
  $type="select_group";
}
else
{
  $query="select * from ";
  $type="select";
}

$loop_time=new Benchmark;
for ($i=1 ; $i <= $opt_loop_count ; $i++)
{
  $sth = $dbh->do("$query bench_$i") or die $DBI::errstr;
}

$end_time=new Benchmark;
print "Time to $type ($opt_loop_count): " .
    timestr(timediff($end_time, $loop_time),"all") . "\n\n";

####
#### Now we are going to drop $opt_loop_count tables;
####

print "Testing drop\n";

$loop_time=new Benchmark;

if ($opt_fast && $server->{'limits'}->{'multi_drop'} &&
    $server->{'limits'}->{'query_size'} > 11+$opt_loop_count*10)
{
  my $query="drop table bench_1";
  for ($i=2 ; $i <= $opt_loop_count ; $i++)
  {
    $query.=",bench_$i";
  }
  $sth = $dbh->do($query) or die $DBI::errstr;
}
else
{
  for ($i=1 ; $i <= $opt_loop_count ; $i++)
  {
    $sth = $dbh->do("drop table bench_$i")
      or die $DBI::errstr;
  }
}


$end_time=new Benchmark;
print "Time for drop_table ($opt_loop_count): " .
    timestr(timediff($end_time, $loop_time),"all") . "\n\n";

if ($opt_fast && $opt_server eq "pg")
{
  $server->vacuum($dbh);
}

#### We'll do first one 'create table' and then we'll drop it
#### away immediately. This loop shall be executed $opt_loop_count
#### times.

print "Testing create+drop\n";

$loop_time=new Benchmark;

for ($i=1 ; $i <= $opt_loop_count ; $i++)
{
  do_many($dbh,$server->create("bench_$i",
			       ["i int NOT NULL",
				"d double",
				"f float",
				"s char(10)",
				"v varchar(100)"],
			       ["primary key (i)"]));
  $sth = $dbh->do("drop table bench_$i") or die $DBI::errstr;
}

$end_time=new Benchmark;
print "Time for create+drop ($opt_loop_count): " .
    timestr(timediff($end_time, $loop_time),"all") . "\n";

if ($opt_fast && $opt_server eq "pg")
{
  $server->vacuum($dbh);
}

#
# Same test, but with a table with many keys
#

my @fields=(); my @keys=();
$keys=min($limits->{'max_index'},16);		# 16 is more than enough
$seg= min($limits->{'max_index_parts'},$keys,16);	# 16 is more than enough

# Make keys on the most important types
@types=(0,0,0,1,0,0,0,1,1,1,1,1,1,1,1,1,1);	# A 1 for each char field
push(@fields,"field1 tinyint not null");
push(@fields,"field2 mediumint not null");
push(@fields,"field3 smallint not null");
push(@fields,"field4 char(16) not null");
push(@fields,"field5 integer not null");
push(@fields,"field6 float not null");
push(@fields,"field7 double not null");
for ($i=8 ; $i <= $keys ; $i++)
{
  push(@fields,"field$i char(5) not null");	# Should be relatively fair
}

# Let first key contain many segments
my $query="primary key (";
for ($i= 1 ; $i <= $seg ; $i++)
{
  $query.= "field$i,";
}
substr($query,-1)=")";
push (@keys,$query);

#Create other keys
for ($i=2 ; $i <= $keys ; $i++)
{
  push(@keys,"index index$i (field$i)");
}

$loop_time=new Benchmark;
for ($i=1 ; $i <= $opt_loop_count ; $i++)
{
  do_many($dbh,$server->create("bench_$i", \@fields, \@index));
  $dbh->do("drop table bench_$i") or die $DBI::errstr;
}

$end_time=new Benchmark;
print "Time for create_key+drop ($opt_loop_count): " .
    timestr(timediff($end_time, $loop_time),"all") . "\n";

if ($opt_fast && $opt_server eq "pg")
{
  $server->vacuum($dbh);
}

####
#### End of benchmark
####

$dbh->disconnect;				# close connection
end_benchmark($start_time);
