
/*

Copyright (C) 2000, 2001 Christian Kreibich <kreibich@aciri.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/param.h>
#include <dirent.h>
#include <glib.h>

#include <nd_globals.h>
#include <nd_debug.h>
#include <nd_misc.h>
#include <nd_plugin.h>
#include <callbacks.h>
#include <interface.h>
#include <support.h>

char global_plugin_dir[MAXPATHLEN];
char user_plugin_dir[MAXPATHLEN];

static GList *plugins = NULL;  /* list<ND_Plugin*> */

static ND_Plugin *nd_plugin_new(char *filename);
static void       nd_plugin_scan(char *dirname);
static void       nd_plugin_hook_up(ND_Plugin *p);
static void       nd_plugin_load(void);
static void       nd_plugin_register(void);
static int        nd_plugin_cmp(ND_Plugin *p1, char *filename);

static int        nd_plugin_dummy(void);
static char      *nd_plugin_dummyname(void);
static int        nd_plugin_dummyop(ND_Packet *p);

static int
nd_plugin_dummy(void)
{
  return (0);
}


static char      *
nd_plugin_dummyname(void)
{
  return ("The Unknown Plugin.");
}


static char      *
nd_plugin_dummydescription(void)
{
  return ("No description available.");
}


static char      *
nd_plugin_dummyauthor(void)
{
  return ("The Unknown Stranger.");
}


static int        
nd_plugin_dummyop(ND_Packet *p)
{
  return (0);
  p = NULL;
}


static ND_Plugin *
nd_plugin_new(char *filename)
{
  ND_Plugin *p;

  p = (ND_Plugin*)malloc(sizeof(ND_Plugin));
  bzero(p, sizeof(ND_Plugin));

  if (filename)
    p->filename = strdup(filename);

  p->name        = nd_plugin_dummyname;
  p->description = nd_plugin_dummydescription;
  p->author      = nd_plugin_dummyauthor;
  p->init        = nd_plugin_dummy;
  p->cleanup     = nd_plugin_dummy;
  p->pre_op      = nd_plugin_dummy;
  p->op          = nd_plugin_dummyop;
  p->post_op     = nd_plugin_dummy;

  return (p);
}


static int    
nd_plugin_cmp(ND_Plugin *p1, char *filename)
{
  if (!p1 || !filename)
    return (-1);

  return (strcmp(p1->filename, filename));
}


static void
nd_plugin_scan(char *dir)
{
  DIR             *dirp;
  struct dirent   *de;
  char            *dot;

  if ((dirp = opendir(dir)) != NULL)
    {
      for (de = readdir(dirp); de; de = readdir(dirp))
	{
	  if (de->d_name[0] != '.')
	    {
	      dot = strrchr(de->d_name, '.');

	      if (dot)
		*dot = '\0';

	      if (!g_list_find_custom(plugins, de->d_name, (GCompareFunc)nd_plugin_cmp))
		plugins = g_list_append(plugins, nd_plugin_new(de->d_name));
	    }
	}      
      closedir(dirp);
    }
}


static void       
nd_plugin_hook_up(ND_Plugin *p)
{
  lt_ptr_t     sym;

  if ((sym = lt_dlsym(p->lt, "name")) != NULL)
    p->name = sym;
  if ((sym = lt_dlsym(p->lt, "description")) != NULL)
    p->description = sym;
  if ((sym = lt_dlsym(p->lt, "author")) != NULL)
    p->author = sym;

  if ((sym = lt_dlsym(p->lt, "init")) != NULL)
    p->init = sym;
  if ((sym = lt_dlsym(p->lt, "cleanup")) != NULL)
    p->cleanup = sym;
  
  if ((sym = lt_dlsym(p->lt, "pre_op")) != NULL)
    p->pre_op = sym;
  if ((sym = lt_dlsym(p->lt, "op")) != NULL)
    p->op = sym;
  if ((sym = lt_dlsym(p->lt, "post_op")) != NULL)
    p->post_op = sym;
}


static void       
nd_plugin_load(void)
{
  GList       *l;
  ND_Plugin   *p;
  char         s[MAXPATHLEN];

  for (l = g_list_first(plugins); l; l = g_list_next(l))
    {
      p = (ND_Plugin*)l->data;

      snprintf(s, MAXPATHLEN, "%s/%s", user_plugin_dir, p->filename);

      if ((p->lt = lt_dlopenext(s)))
	{
	  nd_plugin_hook_up(p);
	}
      else
	{
	  snprintf(s, MAXPATHLEN, "%s/%s", global_plugin_dir, p->filename);

	  if ((p->lt = lt_dlopenext(s)))
	    {
	      nd_plugin_hook_up(p);
	    }
	}
    }
}


static void       
nd_plugin_register(void)
{
  GtkWidget   *menu, *item;
  GList       *l;
  ND_Plugin   *p;

  /* Hook plugin names into "Plugins" menu ... */

  menu = gtk_object_get_data(GTK_OBJECT(toplevel), "plugins1_menu");
  D_ASSERT(menu);

  for (l = g_list_first(plugins); l; l = g_list_next(l))
    {
      p = (ND_Plugin*)l->data;

      item = gtk_menu_item_new_with_label(p->name());
      gtk_widget_ref(item);
      gtk_widget_show(item);
      gtk_object_set_data(GTK_OBJECT(item), "plugin", p);
      gtk_container_add(GTK_CONTAINER(menu), item);
      gtk_signal_connect (GTK_OBJECT (item), "activate",
			  GTK_SIGNAL_FUNC (on_plugin_activate),
			  NULL);
    }

  /* Hook plugin names into "About Plugins" menu ... */

  menu = gtk_object_get_data(GTK_OBJECT(toplevel), "about_plugins1_menu");
  D_ASSERT(menu);

  for (l = g_list_first(plugins); l; l = g_list_next(l))
    {
      p = (ND_Plugin*)l->data;

      item = gtk_menu_item_new_with_label(p->name());
      gtk_widget_ref(item);
      gtk_widget_show(item);
      gtk_object_set_data(GTK_OBJECT(item), "plugin", p);
      gtk_container_add(GTK_CONTAINER(menu), item);
      gtk_signal_connect (GTK_OBJECT (item), "activate",
			  GTK_SIGNAL_FUNC (on_aboutdummyitem_activate),
			  NULL);
    }
}


void 
nd_plugin_init(void)
{
  if (lt_dlinit() != 0)
    {
      fprintf(stderr, "Could not initialize libltdl -- exiting.\n");
      exit(-1);
    }

  snprintf(global_plugin_dir, MAXPATHLEN, 
	   "%s%s", PACKAGE_DATA_DIR, "/plugins");
  snprintf(user_plugin_dir, MAXPATHLEN,
	   "%s%s", getenv("HOME"), "/.nd");

  nd_plugin_scan(user_plugin_dir);
  nd_plugin_scan(global_plugin_dir);
  nd_plugin_load();
  nd_plugin_register();
}


void    
nd_plugin_activate(ND_Plugin *pl)
{
  ND_Packet     *p;

  if (pl)
    {
      pl->pre_op();

      for (nd_misc_pbar_reset(0), p = nd_trace_sel(); p;
	   p = nd_packet_sel_next(p), nd_misc_pbar_inc())
	{
	  if (pl->op(p))
	    nd_trace_set_dirty(TRUE);
	}

      pl->post_op();
      
      nd_misc_pbar_clear();  
      if (trace.p)
	nd_packet_set(trace.p);
    }
}


char   *
nd_plugin_get_description(ND_Plugin *p)
{
  if (!p)
    return (NULL);

  return (p->description());
}


char   *
nd_plugin_get_author(ND_Plugin *p)
{
  if (!p)
    return (NULL);

  return (p->author());
}


void    
nd_plugin_show_about(ND_Plugin *p)
{
  GtkWidget *w;
  GtkWidget *label;

  if (!p)
    return;


  w = create_plugin_about_dialog();

  if ((label = gtk_object_get_data(GTK_OBJECT(w), "plugin_name_label")))
    {
      if (p->name)
	gtk_label_set_text(GTK_LABEL(label), p->name());
      else
	gtk_label_set_text(GTK_LABEL(label), _("Unnamed Plugin"));
    }
  
  if ((label = gtk_object_get_data(GTK_OBJECT(w), "plugin_author_label")))
    {
      if (p->author)
	gtk_label_set_text(GTK_LABEL(label), p->author());
      else
	gtk_label_set_text(GTK_LABEL(label), _("No author(s) specified."));
    }
  
  if ((label = gtk_object_get_data(GTK_OBJECT(w), "plugin_description_label")))
    {
      if (p->description)
	gtk_label_set_text(GTK_LABEL(label), p->description());
      else
	gtk_label_set_text(GTK_LABEL(label), _("No description given."));
    }
  
  gtk_widget_show(w);  
}

