/*  GtkExText	
*  Copyright (C) 1999, 2000 
*Mikael Hermansson <mikeh@bahnhof.se>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU Library General Public License for more details. 
*
*  You should have received a copy of the GNU Library General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#ifdef WIN32
#include <windows.h> 
#else
#include <unistd.h>
#endif
#include <ctype.h>
#include <string.h>
#include <stdio.h>
#include <fcntl.h>
#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <gdk/gdkkeysyms.h>

#include "line-wrap.xbm"
#include "line-arrow.xbm"
#include "gtkextext.h"

#define TEXT_BORDER_ROOM 2
#define CURSOR_TIMEOUT 1000
#define LINE_WRAP_WIDTH ((line_wrap_width*3))
#define MAX_SCROLL_WIDTH 4000
#define MIN_TEXT_WIDTH_LINES     20
#define MIN_TEXT_HEIGHT_LINES    10
#define MIN_TEXT_HEIGHT 5
#define MAX_LINE_LEN 300
#define MAX_STYLE_PER_LINE 64
#define TEXT_BUFFERT_DEFAULT_SIZE 0x8000
#define GAP_DEFAULT_SIZE TEXT_BUFFERT_DEFAULT_SIZE 
#define SCROLL_TIME 100
#define DEFAULT_TAB_STOP  5
#define DEBUG 
#ifdef WIN32
#define TEXT_STYLE_HEIGHT(style) style ? style->ascent *1.2 : 10
#else
#define TEXT_STYLE_HEIGHT(style) style ? ((int )((style->ascent + style->descent)*1.2)) : 10
#endif
#define LINE_DATA_FLAGS(cur) cur->lineptr->width
#define LINE_DATA_PREV(cur) cur->lineptr->prev
#define LINE_DATA_NEXT(cur) cur->lineptr->next

typedef struct _InternalStyleCache
{
  GtkExTextStyle style;
  gpointer property_user_data;
  gint len;	/* length of text */
  gint width;	/* text width */
  gint height;	/* text height */
} InternalStyleCache;

typedef struct _InternalPaintCache
{
  InternalStyleCache styles[MAX_STYLE_PER_LINE];
  gint style_count;
  gint height;
  gint width;
  gint startpos;
  gint length;	/*	text length */
  gint tabcount;
  gint tabspaces;
  gint cursor_h;
  gint cursor_x;
} InternalPaintCache;

typedef gint (*TextDataCallback)(GtkExText *text,GtkExTextLineData *lp,InternalPaintCache *pcache,gpointer userdata);

void recompute_text_segment(GtkExText *text,gint start,gint end);

static InternalPaintCache *
create_paint_cache(GtkExText *text,GtkExTextLineData *linedataptr,gint line_offset,gint stop,gint max_width,InternalPaintCache *pcache);

void 
gtk_extext_get_text_data_foreach_line( GtkExText *text,
                            GtkExTextLineData *lp,
                          TextDataCallback st,
                           gpointer user_data);

static gint
text_data_set_line_data(GtkExText *text,GtkExTextLineData *lp,InternalPaintCache *pcache,gpointer data);

static gint 
extext_get_line_width(GtkExText *text,GtkExTextLineData *linedataptr,gint pos);

/*****************************************************************************/
/*                                    Utils                                      */
/*****************************************************************************/

gboolean is_value_ok(gint value,gint start,gint end);
gint get_next_return(gchar *text,gint len);
gint get_tab_count(gchar *txt,gint len);
gint get_return_count(gchar *txt,gint len);
gint get_real_text_height(GdkFont *font);

/*****************************************************************************/
/*                               InsertDelete characters API                                 */
/*****************************************************************************/

static gint text_insert_line_wrap(GtkExText *text,gint pos,char *txt,gint len);
static gint text_insert(GtkExText *text,gint pos,char *txt,gint len,gboolean changed_signal);
static gint text_remove(GtkExText *text,gint startpos,gint len);
static void make_room_for(GtkExText *text,gint length);
void gap_to(GtkExText *text,gint length);
static gint text_set_pos(GtkExText *class,gint pos);
void gtk_extext_set_position(GtkEditable *text,gint pos);
guchar * text_get_text_with_expanded_tabs(GtkExText *text, gint start, gint end, gint *tabindex);
guchar * text_get_text(GtkExText *text,gint start,gint end);
gint move_cursor(GtkExText *text,gint moveto, gboolean select);


/*****************************************************************************/
/*                              Undo buffertcode                                */
/*****************************************************************************/

void extext_undo_foreach(gpointer data,gpointer user_data);
void extext_undo_insert(GtkExText *text,gint type, gint start, gint end);

/*****************************************************************************/
/*                           Insert Delete Line code                             */
/*****************************************************************************/

static gint line_get_height(GtkExText *text,gint lstartpos,gint lendpos,GtkExTextProperty *prop);
static void line_set_len(GtkExText *text,GtkExTextLineData *linedataptr,gint len);
static GtkExTextLineData* line_insert(GtkExText *textext,GtkExTextLineData *cur,gint count);
static GtkExTextLineData* line_remove(GtkExText *textext,GtkExTextLineData *cur);
static GtkExTextLineData* line_set_by_ptr(GtkExText *class,GtkExTextLineData *data,gboolean forward);
static Lines* line_set(GtkExText *text,gint line);
static gint line_set_cursor_pos(GtkExText *text,gint pos);

static GtkExTextLineData* line_get_by_char_pos(GtkExText *text,gint pos);
static GtkExTextLineData* line_get_by_offset(GtkExText *text,gint y,gint *newoffsety);
static gint column_get_by_offset(GtkExText *text,GtkExTextLineData *data,gint x,gint *newoffsetx);

/*****************************************************************************/
/*                        Insert/Delete Style Cache API                          */
/*****************************************************************************/

typedef struct _FontRef	{
  GdkFont *font;
  gboolean exist;	/* is TRUE if font already exist in stylellist */
} FontRef;

void gtk_extext_style_remove_real(GtkExText *text ,gchar *key);
void propertys_destroy(GtkExText *text);


/*****************************************************************************/
/*                                 WIDGET API                                   */
/*****************************************************************************/

static void gtk_extext_set_arg (GtkObject        *object,
		  GtkArg           *arg,
		  guint             arg_id);
static void gtk_extext_get_arg (GtkObject        *object,
		  GtkArg           *arg,
		  guint             arg_id);


gint  expose_line_text(GtkExText *text,GdkDrawable *draw,
					GtkExTextLineData *linedataptr,gint x, gint y);
gint  expose_line_text_wrap(GtkExText *text,GdkDrawable *draw,
					GtkExTextLineData *linedataptr,gint x, gint y);

static void  gtk_extext_adjustment_changed     (GtkAdjustment  *adjustment,
				      GtkExText        *text);
static void  gtk_extext_adjustment_value_changed     (GtkAdjustment  *adjustment,
				      GtkExText        *text);
static void  gtk_extext_disconnect     (GtkAdjustment  *adjustment,
				      GtkExText        *text);

static void gtk_extext_class_init(GtkExTextClass *text);
static void gtk_extext_init(GtkExText *text);
static void gtk_extext_destroy(GtkObject *text);

static void gtk_extext_style_set(GtkWidget *style,GtkStyle *old);
static void  gtk_extext_finalize       (GtkObject      *object);
static void  gtk_extext_realize        (GtkWidget      *widget);
static void  gtk_extext_unrealize      (GtkWidget      *widget);
static gint  gtk_extext_cursor_blink(gpointer data);
static gint  gtk_extext_cursor_blink_start(gpointer data);
static gint gtk_extext_get_tab_width(GtkExText *text,gint index);

/* Event handlers */

static void  recompute_geometry (GtkExText* text);
static void  recompute_scroll_line_width (GtkExText* text,GtkExTextLineData *lp,gint count);
static void recompute_scroll_line_height(GtkExText *text);

static void recompute_scroll_line_data(GtkExText *text,gint pos,gboolean resetprop);

static void gtk_extext_size_allocate (GtkWidget     *widget,
			GtkAllocation *allocation);

static void  gtk_extext_size_request   (GtkWidget      *widget,
				      GtkRequisition *requisition);

static void  gtk_extext_draw              (GtkWidget         *widget,
					 GdkRectangle      *area);
static gint  gtk_extext_expose            (GtkWidget         *widget,
					 GdkEventExpose    *event);
static gint  gtk_extext_button_press      (GtkWidget         *widget,
					 GdkEventButton    *event);
static gint  gtk_extext_button_release    (GtkWidget         *widget,
					 GdkEventButton    *event);
static gint  gtk_extext_motion_notify     (GtkWidget         *widget,
					 GdkEventMotion    *event);
static gint  gtk_extext_key_press         (GtkWidget         *widget,
					 GdkEventKey       *event);
static gint  gtk_extext_focus_in          (GtkWidget         *widget,
					 GdkEventFocus     *event);
static gint  gtk_extext_focus_out(GtkWidget *widget, GdkEventFocus *event);
static gint gtk_extext_motion_notify(GtkWidget *widget, GdkEventMotion *event);

static void gtk_extext_draw_focus(GtkWidget *widget);

static gint gtk_extext_real_property_text_insert(GtkExText *text,GtkExTextProperty *prop ,gint startpos,gint endpos);
static gint gtk_extext_real_property_text_remove(GtkExText *text,GtkExTextProperty *prop ,gint startpos,gint endpos);


/*****************************************************************************/
/*                                 New signals                                   */
/*****************************************************************************/



/*****************************************************************************/
/*                       Overrided GtkEditable functions                         */
/*****************************************************************************/

static void gtk_extext_update_text(GtkEditable *editable,gint startpos,gint endpos);
static void gtk_extext_insert_text       (GtkEditable       *editable,
					const gchar       *new_text,
					gint               new_text_length,
					gint               *position);

static void gtk_extext_delete_text       (GtkEditable       *editable,
					gint              startpos,
					gint               endpos);
static void gtk_extext_select_region(GtkEditable *editable,gint start,gint end);
static gchar *gtk_extext_real_get_chars(GtkEditable *editable,gint start,gint end);

static void gtk_extext_delete_to_line_end(GtkExText *text);
static void gtk_extext_delete_backward_word(GtkExText *text);
static void gtk_extext_delete_line(GtkExText *text);

static Lines *goto_line(GtkExText *text,gint lineto,gboolean sel,gboolean killselect);
void view_to_scroll(GtkExText *text);
void scroll_to_view(GtkExText *text,gint line);
static void find_cursor(GtkExText *text,gdouble x,gdouble y,gboolean select);
static void update_select(GtkExText *text,gint oldpos,gboolean select,gboolean killselect);
static void gtk_extext_real_set_editable(GtkEditable *editable,gboolean b);
 
enum {
  ARG_0,
  ARG_HADJUSTMENT,
  ARG_VADJUSTMENT,
};

enum {
  PROPERTY_TEXT_INSERT,
  PROPERTY_TEXT_REMOVE,
  PROPERTY_DESTROY,
  UNDO_CHANGED,
  PROPERTY_MARK,
  LINE_INSERT,
  LINE_REMOVE,
  LAST_SIGNAL
};

static GtkWidgetClass *parent_class = NULL;
static guint extext_signals[LAST_SIGNAL]={ 0 };

typedef void (*GtkExTextFunction)(GtkEditable *text, gint time);
static const GtkExTextFunction control_keys[26] =
{
  NULL,    /* a */
  NULL,   /* b */
  (GtkExTextFunction)gtk_editable_copy_clipboard,        /* c */
  NULL,  /* d */
  NULL,          /* e */
  NULL,    /* f */
  NULL,                                                /* g */
  NULL, /* h */
  NULL,                                                /* i */
  NULL,                                                /* j */
  (GtkExTextFunction)gtk_extext_delete_to_line_end,        /* k */
  NULL,                                                /* l */
  NULL,                                                /* m */
  NULL,            /* n */
  NULL,                                                /* o */
  NULL,        /* p */
  NULL,                                                /* q */
  NULL,                                                /* r */
  NULL,                                                /* s */
  NULL,                                                /* t */
  (GtkExTextFunction)gtk_extext_delete_line,               /* u */
  (GtkExTextFunction)gtk_editable_paste_clipboard,       /* v */
  (GtkExTextFunction)gtk_extext_delete_backward_word,      /* w */
  (GtkExTextFunction)gtk_editable_cut_clipboard,         /* x */
  NULL,                                                /* y */
  (GtkExTextFunction)gtk_extext_undo,                                                /* z */
};

static const GtkExTextFunction alt_keys[26] =
{
  NULL,                                                /* a */
  NULL,       /* b */
  NULL,	/* c */                      
	NULL,                          /* d */
  NULL,                                           /* e */
  NULL,         /* f */
  NULL,                                           /* g */
  NULL,                                           /* h */
  NULL,                                           /* i */
  NULL,                                           /* j */
  NULL,                                           /* k */
  NULL,                                           /* l */
  NULL,                                           /* m */
  NULL,                                           /* n */
  NULL,                                           /* o */
  NULL,                                           /* p */
  NULL,                                           /* q */
  NULL,                                           /* r */
  NULL,                                           /* s */
  NULL,                                           /* t */
  NULL,                                           /* u */
  NULL,                                           /* v */
  NULL,                                           /* w */
  NULL,                                           /* x */
  NULL,                                           /* y */
  NULL,                                           /* z */
};


/*****************************************************************************/
/*                          hmm lets start coding :-)                            */
/*****************************************************************************/

gint
get_next_return(gchar *text,gint len)
{
  gint index;
  for(index=0;index<len;index++)	{
  	if(text[index]=='\n')
  		return index;
  }
  return -1;
}

gint
get_tab_count(gchar *txt,gint len)
{
  gint tabs;
  gint i;
  tabs=i=0;
  if(len==-1)
  	len=strlen(txt);

  while(i<len)
  {	
  	if(txt[i]=='\t')
  		tabs++;

  	i++;
  };
  return tabs;
}

gint
get_return_count(gchar *txt,gint len)
{
  gint ret;
  gint i;
  ret=i=0;
  while(i<len)
  {	
  	if(txt[i]=='\n')
  		ret++;

  	i++;
  };
  return ret;
}

gint
is_value_ok(gint pos,gint start,gint end)
{
  if(pos >= start && pos < end)
  	return 0;
  if(pos < start)
  	return -1;

  return 1;
}


/*****************************************************************************/
/*              gtk_extext_get_current_word return TRUE if word                 */
/*****************************************************************************/

gboolean 
gtk_extext_get_current_word(GtkExText *text,gint *startpos, gint *endpos)
{
  gint key;
  gint pos;

  pos=*startpos;
  key=GTK_EXTEXT_INDEX(text,pos);
  if(!isalnum(key))	
  	return FALSE;

  key=GTK_EXTEXT_INDEX(text,pos);
  while(pos >= 0 && isalnum(key))	{
  	pos--;
   	key=GTK_EXTEXT_INDEX(text,pos);
  };
	
  if(pos < -1 ) /* no word */
  	return FALSE;

  *startpos=pos+1;
  if(!endpos)
  	return TRUE;

  pos++;
  key=GTK_EXTEXT_INDEX(text,pos);
  while(pos < text->length && isalnum(key))	{
  	pos++;
  	key=GTK_EXTEXT_INDEX(text,pos);
  };
  *endpos=pos;

  return TRUE;
}

gboolean 
gtk_extext_get_previous_word(GtkExText *text,gint *startpos, gint *endpos)
{
  guchar key;
  gint pos;
  if(!startpos )
    return FALSE;

   pos=*startpos;
  key=GTK_EXTEXT_INDEX(text,pos);
  if(isalnum(key))	{
    if(!gtk_extext_get_current_word(text,&pos,NULL))
      return FALSE;

      pos--;
  }

  while(pos >= 0)
  {
    key=GTK_EXTEXT_INDEX(text,pos);
    if(isalnum(key))		
      break;
	
    pos--;
  };

  if(!isalnum(key))		
    return FALSE;
	
  if(endpos)
    *endpos=pos+1;

  while(pos >= 0)
  {
    key=GTK_EXTEXT_INDEX(text,pos);
    if(!isalnum(key))		
      break;

    pos--;
  };
  *startpos=pos+1;
	
  return TRUE;
}

gboolean 
gtk_extext_get_next_word(GtkExText *text,gint *startpos, gint *endpos)
{
  guchar key;
  gint start,e;
  if(!startpos || !endpos)
    return FALSE;

  start= *startpos > 0 ? *startpos : 0;	
  e=start;
  key=GTK_EXTEXT_INDEX(text,start);
  if(isalnum(key))
    gtk_extext_get_current_word(text,&start,&e);
	
  start=e;
  while(start < text->length)
  {
    key=GTK_EXTEXT_INDEX(text,start);
    if(isalnum(key))
      break;

    start++;
  };
  if(start >= text->length)
    return FALSE;

  *startpos=start;
  if(!endpos)
    return TRUE;

  *endpos=start+1;
  while(start < text->length && start >= 0)
  {
    key=GTK_EXTEXT_INDEX(text,start);
    if(!isalnum(key))
      break;

    start++;
  }
  *endpos=start;

  return TRUE;
}

void 
gap_to(GtkExText *text,gint pos)
{
  gint i=0;
  if(pos==text->part1len)
  	return;

  if(!text->use_wchar)	{	
  	if(pos<text->part1len)	{
  		for(i=0;i<text->part1len-pos;i++)
  			text->text.ch[text->part1len+text->gap_len - i -1]=text->text.ch[text->part1len-i-1];
  	} 	else	{
  		for(i=0;i<pos-text->part1len;i++)
  		text->text.ch[text->part1len+i]=text->text.ch[text->part1len+text->gap_len+i];
  	}
  	text->part1len=pos;
  	text->part2text.ch=text->text.ch+text->gap_len;
  } else {
  	if(pos<text->part1len)	{
  		for(i=0;i<text->part1len-pos;i++)
  			text->text.wc[text->part1len+text->gap_len - i -1]=text->text.wc[text->part1len-i-1];
  	}
  	else	{
  		for(i=0;i<pos-text->part1len;i++)
  			text->text.wc[text->part1len+i]=text->text.wc[text->part1len+text->gap_len+i];
  	}
  	text->part1len=pos;
  	text->part2text.wc=text->text.wc+text->gap_len;
  }
}

void 
make_room_for(GtkExText *text,gint len)
{
  gint newsize;
  gint count;

  if(text->gap_len<=len)
  {
    gap_to(text,text->length);

    count=len/text->size;
    if(count*text->size<len)
      count++;
		
    if(!text->use_wchar)	{
      newsize=text->gap_len + text->length + (text->size * count);
      text->text.ch=g_realloc(text->text.ch,newsize);

      text->gap_len=newsize - text->length;
      text->part2text.ch=text->text.ch + text->gap_len;
    } 
    else {
      newsize=text->gap_len + text->length + (text->size * count);
      text->text.wc=g_realloc(text->text.wc,newsize*sizeof(GdkWChar));

      text->gap_len=newsize - text->length;
      text->part2text.wc=text->text.wc + text->gap_len;
    }
  }
}

gint
text_set_pos(GtkExText *text, gint pos)
{
  GtkExTextLineData *linedataptr;
  gboolean forward;

  forward=FALSE;
  if(pos>=GTK_EDITABLE(text)->current_pos)
    forward=TRUE;

  if(pos>text->length)
    GTK_EDITABLE(text)->current_pos=text->length;
  else if(pos==-1)
    GTK_EDITABLE(text)->current_pos=0;
  else
    GTK_EDITABLE(text)->current_pos=pos;

  linedataptr=line_get_by_char_pos(text,pos);
  if(linedataptr->lineptr!=text->line_pos)	{
    linedataptr=line_set_by_ptr(text,linedataptr,forward);
  }	

  line_set_cursor_pos(text,GTK_EDITABLE(text)->current_pos 
                                    - linedataptr->startpos);

  g_free(linedataptr);

  return GTK_EDITABLE(text)->current_pos;
}

gint
text_insert_line_wrap(GtkExText *text, gint pos,gchar *txt,gint txtlen)
{
  gint lnum,start,i,len;
  GtkExTextLineData *lp;
  gint newpos;
 
 /*    g_print("text_insert_line_wrap is heavy broken :-((((\n"); */
  start = 0;
  newpos = pos;
  lp = gtk_extext_get_line_by_char_pos(text,pos);
  lnum = lp->line_number; 
  len = LINE_DATA_LENGTH(lp);

 /* wgb */
  for(i = 0; i < txtlen; i++)  {
    if (txt[i] == '\n') {
      newpos = text_insert(text, newpos, "\n", 1, TRUE);
      g_free(lp);
      lp = gtk_extext_get_line_by_char_pos(text, newpos);
    }
    else if (LINE_DATA_LENGTH(lp) >= text->line_max_chars) {
      GtkEditable *editable;
      int wpos;
      int epos;
        
      editable = GTK_EDITABLE(text);
      /*
      * This loop causes the text to flow forward.  We
      * wrap this line, and put the extra (what is being
      * wrapped) into a new line.  Then go to that line,
      * replace it's newline with a space, to join it and
      * the following line.  Continue doing this until we
      * reach a line that is less than line_max_chars long.
     */
      do {
        GtkExTextLineData *wlp;
        int next_line_len;
           
        wlp = gtk_extext_get_line_by_char_pos(text, lp->endpos);
        next_line_len = LINE_DATA_LENGTH(wlp);
        g_free(wlp);
        
        wpos = lp->startpos + text->line_max_chars;
        if (wpos >= lp->endpos)
            wpos = lp->endpos - 1;
            while (wpos > lp->startpos) {
               if (gtk_extext_get_char_at_pos(text, wpos) == ' ')
                  break;
               
                --wpos;
            }
                 /*
                   // Delete the space we identified, and insert a newline
                   // in it's place.
                 */
            gtk_editable_delete_text(editable, wpos, wpos + 1);
            wpos = text_insert(text, wpos, "\n", 1, TRUE);
                 /*
                  * Now, replace the newline in the new line we just
                  * created, with a space.  However, only do it if the
                 * next line has a length greater than 1 (indicating
                  * it has actual characters, not just a newline).
                  */
            g_free(lp);
            lp = gtk_extext_get_line_by_char_pos(text, wpos);
            if (next_line_len > 1) {
              if (lp->line_number < text->line_count) {
                epos = lp->endpos;
                text_insert(text, epos - 1, " ", 1, TRUE);
                gtk_editable_delete_text(editable, epos, epos + 1);
                g_free(lp);
                lp = gtk_extext_get_line_by_char_pos(text, wpos);
              }
            }
      } while (LINE_DATA_LENGTH(lp) >= text->line_max_chars);

      g_free(lp);
      lp = gtk_extext_get_line_by_char_pos(text, newpos);
      newpos = text_insert(text, newpos, txt + i, 1, TRUE);
    }
    else
      newpos = text_insert(text, newpos, txt + i, 1, TRUE);
  }
 
  g_free(lp);
 
  return newpos;
}


/* THIS CODE NEEDS A CLEANUP */

gint 
text_insert( GtkExText *text,gint pos,gchar *txt,gint len,gboolean removethis)
{
  GtkExTextLineData *linedataptr;
  GtkExTextProperty *property_first;
  GtkExTextProperty *prop;
  gint oldlinepos;
  gint linecount;
  gint lastline_endpos;
  gint start;
  gint end;
  gint linecurpos;
  gint lindex;
  gint diff;
  gint retcount;
  gint signal_return_value;

  g_return_val_if_fail (text != NULL,FALSE);
  g_return_val_if_fail (GTK_IS_EXTEXT (text),FALSE);

  if(!len)
    return text->length;
  if(len==-1)
    len=strlen(txt);

  prop=NULL;
  signal_return_value=FALSE;
  text->text_insert_delete_flag=TRUE;
  oldlinepos=text->line_pos_index;

  if(pos != GTK_EDITABLE(text)->current_pos)
    gtk_extext_set_position(GTK_EDITABLE(text),pos);  /* must call this to update correct */

  make_room_for(text,len);
  gap_to(text,pos);


  lindex=text->line_pos_index;
  linedataptr=gtk_extext_get_line_data(text,lindex,text->scroll_line_start);
  lastline_endpos=linedataptr->endpos;

  property_first=linedataptr->property_first;
  retcount=linecount=get_return_count(txt,len);
  if(retcount)	{
  	gboolean retvalue;
  	retvalue=FALSE;
  	/* linecurpos has chars before before this line */

  	end=get_next_return(txt,len);
  	start=end+1;

  	linecurpos=text->line_cursor_index;
  	/* diff is all chars after return in curline */ 
  	/*	this chars is moved to newline */

  	diff=LINE_DATA_LENGTH(linedataptr) -  text->line_cursor_index;

  	line_set_len(text,linedataptr,linecurpos + end + 1);
  	line_insert(text,linedataptr,retcount);
	
  	retcount--;
  	if(!retcount) /*last return */	
  	{	
  		lindex++;
  		linedataptr=gtk_extext_get_line_data(text,lindex,linedataptr);
  		line_set_len(text,linedataptr,diff);		
  	}
  	else
  	{
  		end=get_next_return(&txt[start],len-start);
  		while(end!=-1)
  		{
  			retcount--;
  			lindex++;

  			linedataptr=gtk_extext_get_line_data(text,lindex,linedataptr);
  			line_set_len(text,linedataptr,end+1);

  			start+=end+1;
  			end=get_next_return(&txt[start],len-start);

  			linecurpos=0; /* we have inserted a new line and has no chars yet*/
  		};
  		lindex++;
  		start+=end+1;
  		linedataptr=gtk_extext_get_line_data(text,lindex,linedataptr);
  		line_set_len(text,linedataptr,diff+(len - start));
  	}
  }	
  else	{
  	linecurpos=text->line_pos->length;
  	line_set_len(text,linedataptr,linecurpos + len);
  }

  if(!text->use_wchar)	{
  	memcpy(text->text.ch+text->part1len,txt,len);
  	text->part1len+=len;
  	text->length+=len;
  	text->gap_len-=len;
  	text->part2text.ch=text->text.ch+text->gap_len;
  }	
  else	  {
     char *chars_nt = (char *)txt;
     gint numwcs=0;
     if (len > 0)
	{
	  chars_nt = g_new (char, len+1);
	  memcpy (chars_nt, txt, len);
	  chars_nt[len] = 0;
	}
      numwcs = gdk_mbstowcs (text->text.wc + text->part1len, chars_nt,len);
      if (chars_nt != txt)
  		g_free(chars_nt);

  	if (numwcs < 0)
  		numwcs = 0;
 
   	g_print("[text_insert] (wchar) untested code\n");
  	text->part1len+=numwcs;
  	text->length+=numwcs;
  	text->gap_len-=numwcs;
  	text->part2text.wc=text->text.wc+text->gap_len;
  }
  
  lastline_endpos=linedataptr->endpos;

  g_free(linedataptr);
  linedataptr=NULL;

  gtk_signal_emit(GTK_OBJECT(text),extext_signals[PROPERTY_TEXT_INSERT],
				property_first,pos,pos+len,&signal_return_value);
 
  if(!signal_return_value)
    gtk_extext_real_property_text_insert(text,property_first,pos,pos+len);

  /* we need to update linedataptr with the new values */
  /* THIS IS THERE TO` MAKE SURE TO UPDATE ALL lines */
  /* between oldlinepos < text->scroll_line_index */

  linedataptr=text->scroll_line_start;
  if(oldlinepos < text->scroll_line_start->line_number)
    linedataptr=gtk_extext_get_line_data(text,oldlinepos,text->scroll_line_start);

  linedataptr=gtk_extext_get_line_data(text,linecurpos,linedataptr);
  recompute_scroll_line_width(text,linedataptr,linecount+1);

  g_free(linedataptr);

  gtk_extext_set_position(GTK_EDITABLE(text),pos+len);  /* must call this to update correct */
  extext_undo_insert(text,EXTEXT_UNDO_INSERT,pos, pos+len);

  text->text_insert_delete_flag=FALSE;
	
  return pos+len;
}

gint 
text_remove(GtkExText *text,gint pos,gint len)
{
  GtkExTextProperty *property_first;
  GtkExTextLineData* linedataptr;
  gint i;
  gint oldlinelen;
  gint lindex;
  gint linelen;
  gint signal_return_value;
  gint lastline_endpos;

  g_return_val_if_fail (text != NULL,0);
  g_return_val_if_fail (GTK_IS_EXTEXT (text),0);

  signal_return_value=FALSE;
  text->text_insert_delete_flag=TRUE;
	
  if(len <= 0 || !text->length || pos < 0)
  	return GTK_EDITABLE(text)->current_pos;

  if(pos>text->length)
  	pos= text->length;
	
  if(pos>=GTK_EDITABLE(text)->selection_start_pos)
  	GTK_EDITABLE(text)->selection_start_pos=-1;
  if(len+pos<=GTK_EDITABLE(text)->selection_end_pos)
  	GTK_EDITABLE(text)->selection_end_pos=-1;

  extext_undo_insert(text,EXTEXT_UNDO_REMOVE,pos, pos+len);

  linedataptr=line_get_by_char_pos(text,pos);
  property_first=linedataptr->property_first;
  linelen=LINE_DATA_LENGTH(linedataptr);
  lindex=linedataptr->line_number;
  for(i=pos;i<len+pos;i++)	{
  	if(GTK_EXTEXT_INDEX(text,i)=='\n')	{	
		/* always remove line AFTER Current line*/
		/* this because we should never remove FIRST line */
		/* ask for all characters before RETURN char */	

		/* get next line length */
  		lindex++;
  		linedataptr=gtk_extext_get_line_data(text,lindex,linedataptr);
  		oldlinelen=LINE_DATA_LENGTH(linedataptr)-1;

  		/* remove next line wish line */
  		linedataptr=line_remove(text,linedataptr);
  		/* line_remove returns previous line */
  		line_set_len(text,linedataptr,oldlinelen + LINE_DATA_LENGTH(linedataptr));

  		linelen=LINE_DATA_LENGTH(linedataptr);
  		lindex=linedataptr->line_number;
  	}
  	else	{
  		linelen--;
  		line_set_len(text,linedataptr,linelen);
  	}
  }
  lastline_endpos=linedataptr->endpos;
  g_free(linedataptr);
  linedataptr=gtk_extext_get_line_data(text,text->line_pos_index,text->scroll_line_start);

  gap_to(text,pos);
  text->length -=len;
  text->gap_len +=len;
  text->part2text.ch=text->text.ch + text->gap_len;

  text_set_pos(text,pos);

  gtk_signal_emit(GTK_OBJECT(text),extext_signals[PROPERTY_TEXT_REMOVE],
					property_first,pos,pos+len,&signal_return_value);

  if(!signal_return_value)
    gtk_extext_real_property_text_remove(text,property_first,pos,pos+len);

  gtk_extext_get_text_data_foreach_line(text,linedataptr,        
                                      &text_data_set_line_data,
                                  GINT_TO_POINTER(lastline_endpos));

  text->text_insert_delete_flag=FALSE;

  return GTK_EDITABLE(text)->current_pos;
}

static gint
gtk_extext_real_property_text_insert(GtkExText *text,GtkExTextProperty *property_first ,gint pos,gint endpos)
{
  GtkExTextProperty *prop; 
  gint len=endpos - pos;
  g_return_val_if_fail (text != NULL,TRUE);
  g_return_val_if_fail (GTK_IS_EXTEXT (text),TRUE);

  prop=gtk_extext_property_get_at_pos(text,pos ? pos-1 : 0,property_first);
  if(prop && prop->startpos != pos ) {
      prop->endpos+=len;
     gtk_extext_property_move_all(text,endpos,len,property_first); 
  }
  else
     gtk_extext_property_move_all(text,pos,len,property_first); 

  return TRUE;
}

static gint
gtk_extext_real_property_text_remove(GtkExText *text,GtkExTextProperty *property_first ,gint pos,gint endpos)
{
  GtkExTextProperty *prop,*prop_end; 
  gint len=endpos - pos;
  g_return_val_if_fail (text != NULL,TRUE);
  g_return_val_if_fail (GTK_IS_EXTEXT (text),TRUE);

  prop=gtk_extext_property_get_at_pos(text,pos,property_first);
  prop_end=gtk_extext_property_get_at_pos(text,endpos-1,property_first);

  if(prop==prop_end) {
    if(prop)
      prop->endpos-=len;

    gtk_extext_property_move_all(text,endpos,-len,property_first);

    return TRUE;
  }
  /* the same */
  else if(prop_end && prop_end->endpos >= endpos)  {
    prop_end->endpos=pos;
    gtk_extext_property_insert(text,prop->style->key,
                                                endpos,prop_end->endpos,
                                                  prop_end->user_data,PROPERTY_INSERT,property_first);

    gtk_extext_property_move_all(text,prop_end->endpos,-len,property_first);
    gtk_extext_property_remove_all(text,pos,endpos,property_first);
  }
  else
  {
    if(prop)   
      prop->endpos=pos;

    gtk_extext_property_move_all(text,endpos,-len,property_first);
    gtk_extext_property_remove_all(text,pos,endpos,property_first);

  }
  return TRUE;
}

/* TODO REMOVE THIS */
/* use GTK_EXTEXT_INDEX instead */
guchar 
gtk_extext_get_char_at_pos(GtkExText *text,gint pos)
{
  return  pos < text->part1len ? text->text.ch[pos] : text->part2text.ch[pos];
}

guchar *
text_get_text_with_expanded_tabs(GtkExText *text,gint start, gint end  ,gint *tabindexptr )
{
  guchar *newtxt;
  gint len;
  gint tabcount,tabs;
  gint i,n,tabindex;

  g_return_val_if_fail (text != NULL,NULL);
  g_return_val_if_fail (GTK_IS_EXTEXT (text),NULL);

  tabindex=0;
  tabindex=tabindex ? *tabindexptr : 0;
  tabs=i=n=0;
  len=end-start;

  tabcount=0;
  i=start;
  while(i<end)  {
    if(GTK_EXTEXT_INDEX(text,i)=='\t')  {
      tabcount++;
      tabs+=gtk_extext_get_tab_width(text,tabindex);
      tabindex++;
    }
    i++;
  }

  if(tabcount) {
    n=0; 
    tabindex= tabindexptr ? *tabindexptr : 0;
    newtxt=g_malloc(len + tabs + 1);
    for(i=start;i<end;i++)	{
  	if(GTK_EXTEXT_INDEX(text,i)=='\t')	{
  	  memset(&newtxt[n],' ',gtk_extext_get_tab_width(text,tabindex));
  	  n+=gtk_extext_get_tab_width(text,tabindex);
         tabindex++;
  	}
  	else
          newtxt[n++]=GTK_EXTEXT_INDEX(text,i);
    }
    newtxt[n]='\0';
  }
  else
    newtxt=text_get_text(text,start,end);

  *tabindexptr=tabindex;

  return newtxt;
}

guchar* 
text_get_text(GtkExText *text,gint startpos,gint endpos)
{
  guchar *txt;
  gint i;
  gint len;

  if(endpos<startpos)	{
  	g_warning("Startpos > endpos[text_get_text(%X,%d,%d)]\n",text,startpos,
		endpos);
  	return NULL;
  }

  if(startpos>text->length)	{
  	g_warning("Startpos out of range [text_get_text(%X,%d,%d)] text->length: %d\n",
		text,startpos,endpos,text->length);

  	return NULL;
  }

  if(endpos>text->length)
  	endpos=text->length;
	
  if(startpos==endpos)
  	return NULL;

  len=endpos-startpos;
  txt=g_malloc(len+1);
  for(i=0;i<len;i++)
  	txt[i]=GTK_EXTEXT_INDEX(text,startpos+i);

  txt[i]='\0';
	
  return txt;
}

/* the main handler to set cursor at pos */
/* called by text_set_pos */
/* and goto_line  */

gint 
gtk_extext_get_tab_width(GtkExText *text,gint index)
{
  GList *list;
  if(!text->tab_stops)
  	return text->default_tab_width;

  list=g_list_nth(text->tab_stops,index);
	
  return list ? (gint)list->data : text->default_tab_width;
} 

static gint 
line_set_cursor_pos(GtkExText *text,gint cursorpos)
{
  Lines* oldline; 
  gint len,oldcol;	
  gint width;
  gfloat val;
  GtkExTextProperty *prop;
  GtkExTextLineData *linedataptr;

  linedataptr=gtk_extext_get_line_data(text,text->line_pos_index,text->scroll_line_start);

  prop=text->property_current;
  oldcol=text->line_cursor_index;
  oldline=linedataptr->lineptr;
  len=LINE_DATA_LENGTH(linedataptr);
  if(cursorpos < 0 ) {
  	linedataptr=gtk_extext_get_line_data(text,text->line_pos_index-1,linedataptr);
		line_set_by_ptr(text,	linedataptr,FALSE);

  	if(oldline!=linedataptr->lineptr)
		text->line_cursor_index=text->line_pos->length -1;
  	else
		text->line_cursor_index=0;
  }
  else if(cursorpos >= len)	{
  	linedataptr=gtk_extext_get_line_data(text,text->line_pos_index+1,linedataptr);
  	line_set_by_ptr(text,linedataptr,TRUE);
		
  	if(oldline==linedataptr->lineptr)
  		text->line_cursor_index=len;
  	else
  		text->line_cursor_index=0;
  }
  else
  	text->line_cursor_index=cursorpos;

  GTK_EDITABLE(text)->current_pos=linedataptr->startpos + text->line_cursor_index;

  width=extext_get_line_width(text,linedataptr,text->line_cursor_index);
  val=text->hadj->value;
  if((gfloat)width>=val + text->hadj->page_size)	{
    val+=text->hadj->page_size;
    gtk_adjustment_set_value(text->hadj,val);
  }
  else if((gfloat)width<=val)	{
    val-=text->hadj->page_size;
    if(val<0)
       val=0;

    gtk_adjustment_set_value(text->hadj,val);
  }

  text->property_current=gtk_extext_property_get_at_pos(text,
										GTK_EDITABLE(text)->current_pos,linedataptr->property_first);

  if(prop!=text->property_current)
  	gtk_signal_emit_by_name(GTK_OBJECT(text),"property_mark");

  if(oldcol!=text->line_cursor_index)
  	gtk_signal_emit_by_name(GTK_OBJECT(text),"move-to-column",
						text->line_cursor_index);

	
  g_free(linedataptr);

  return text->line_cursor_index;
}

static GtkExTextLineData* 
line_insert(GtkExText *text,GtkExTextLineData *linedataptr,gint count)
{
	/* always insert after linedataptr->lineptr */

  gint lindex;
  gint i;

  Lines *cur;
  Lines *newc;
  Lines *next;
	
  g_return_val_if_fail(linedataptr!=NULL ,NULL);
  g_return_val_if_fail(count > 0,linedataptr);
 
  lindex=linedataptr->line_number;
  next = linedataptr->lineptr->next;
  cur=linedataptr->lineptr;
  for(i=0;i<count;i++)	{
  	newc=g_malloc0(sizeof(Lines));
  	if(!text->line_start) /* no lines at all */
  		text->line_start=newc;

  	newc->prev=cur;	
  	if(newc->prev)
  		newc->prev->next=newc;		

  	cur=newc;

   	gtk_signal_emit(GTK_OBJECT(text),extext_signals[LINE_INSERT],lindex+i+1);
  }

  cur->next=next;
  if(next)
  	next->prev=cur;

  text->line_count+=count;

  if(!next)
	text->line_end=cur;

  return linedataptr;
}

static gint 
line_get_height(GtkExText *text,gint start,gint end, GtkExTextProperty *prop)
{
  gint height=0;
  GtkExTextProperty *new;

  if(!gtk_extext_style_get(text,"Default"))
  	return MIN_TEXT_HEIGHT;

  height=TEXT_STYLE_HEIGHT(gtk_extext_style_get(text,"Default"));

  if(!prop)
  	prop=gtk_extext_property_nearest_forward(text,start,NULL);

  while(start<end && prop)	{
  	new=gtk_extext_property_get_at_pos(text,start,prop);
  	if(new && TEXT_STYLE_HEIGHT(new->style) > height)	{
  		height=TEXT_STYLE_HEIGHT(new->style);
  		prop=new;
  	}
  	else if(new)
  		prop=new;

  	start++;
  };
  return height;
}

void 
line_set_len(GtkExText *text,GtkExTextLineData *lptr,gint len)
{
  gint oldlen = LINE_DATA_LENGTH(lptr);

  lptr->endpos+=len - oldlen;
  LINE_DATA_LENGTH(lptr)=len;

  /* we need to set new width and height values */
/*  create_paint_cache(text,lptr,0,LINE_DATA_LENGTH(lptr),0xFFFF,&pcache);
  LINE_DATA_WIDTH(lptr)=pcache.width;  
  LINE_DATA_HEIGHT(lptr)=pcache.height;  
*/
  if(lptr->line_number < text->scroll_line_start->line_number)	{
  	text->scroll_line_start->startpos+=len - oldlen; 
  	text->scroll_line_start->endpos+=len  - oldlen;
  }
  if(lptr->line_number == text->scroll_line_start->line_number)
  	text->scroll_line_start->endpos+=len - oldlen;	
}

static GtkExTextLineData* 
line_remove(GtkExText *text,GtkExTextLineData *cur)
{
  gint lindex;
  GtkExTextLineData *linedataptr;
  Lines *remove;

  g_return_val_if_fail(GTK_EXTEXT(text),cur);
  g_return_val_if_fail(cur!=NULL,NULL);

  remove=cur->lineptr;
  linedataptr=cur;
  lindex=cur->line_number ;
  if(cur->lineptr==text->line_start)	{
  	g_warning("[line_remove] BUG!!!! tried to Remove first line this is impossible\n");	
  	return cur;	
  }	

  if(cur->lineptr==text->line_end) 
  	text->line_end=text->line_end->prev;

  gtk_signal_emit(GTK_OBJECT(text),extext_signals[LINE_REMOVE],lindex);
	
	/* update linedataptr so it points to previous line */

  lindex--;
  linedataptr=gtk_extext_get_line_data(text,lindex,linedataptr);

  if(remove->prev)
	remove->prev->next=remove->next;

  if(remove->next)
	remove->next->prev=remove->prev;

  g_free(remove);

  text->line_count--;

  /* if we removed the first visible line whe need to update */
  /* scroll_line_start so it points to prevline */
  if(remove==text->scroll_line_start->lineptr)	{
  	g_free(text->scroll_line_start);
  	text->scroll_line_start=linedataptr;
  	/* and yes create a new linedata struct to pass to the user */
  	/* else user freeing the text->scroll_line_start struct */
  	/* and it will segfault :-) */
  	linedataptr=gtk_extext_get_line_data(text,text->scroll_line_start->line_number,text->scroll_line_start);
  }

  /* we return previous line */
  return linedataptr;
}

GtkExTextLineData* 
line_get_by_char_pos(GtkExText *text,gint textoffset)
{
  GtkExTextLineData *linedataptr=NULL;
  gint lindex;
  gint val;

  g_return_val_if_fail(textoffset >= 0, NULL);
  if(textoffset >= text->length)
  	return gtk_extext_get_line_data(text,text->line_count,text->scroll_line_start);	 
  if(!textoffset )
 	return gtk_extext_get_line_data(text,0,NULL);	 
 
  lindex=text->scroll_line_start->line_number;
  linedataptr=gtk_extext_get_line_data(text,lindex,text->scroll_line_start);		

  /* check if text offset is in current line */
  val=is_value_ok(textoffset,linedataptr->startpos,linedataptr->endpos);
  if(!val)
  	return gtk_extext_get_line_data(text,lindex,linedataptr);
		
  if( val > 0)	{
  	while(TRUE ) {
  		linedataptr=gtk_extext_get_line_data(text,++lindex,linedataptr);		
  		if(!is_value_ok(textoffset,linedataptr->startpos,linedataptr->endpos))
  			return linedataptr;
    	};
  }

  while(lindex >= 0) {
  	linedataptr=gtk_extext_get_line_data(text,--lindex,linedataptr);		
  	if(!is_value_ok(textoffset,linedataptr->startpos,linedataptr->endpos))
  		return linedataptr;
  };

  g_warning("[line_get_by_char pos(%d)] failed????????\n",textoffset);

  return gtk_extext_get_line_data(text,0,NULL);
}

GtkExTextLineData*
line_set_by_ptr(GtkExText *text,GtkExTextLineData *linedataptr,gboolean forward)
{
  gint oldline;
  if(!linedataptr)
  	return NULL;

  oldline=text->line_pos_index;	
  text->line_pos=linedataptr->lineptr;
  text->line_pos_index=linedataptr->line_number;
  if(oldline!=text->line_pos_index)
  	gtk_signal_emit_by_name(GTK_OBJECT(text),"move-to-row",	
					text->line_pos_index);

  return linedataptr;	
}

Lines*
line_set(GtkExText *text,gint pos)
{
  GtkExTextLineData *linedataptr;
	
  linedataptr=gtk_extext_get_line_data(text,pos,text->scroll_line_start);
  line_set_by_ptr(text,linedataptr,FALSE); 
	
  g_free(linedataptr);

  return text->line_pos;	
}
/*****************************************************************************/
/*                           StyleCache declarations                             */
/*****************************************************************************/

/* this is called after realize to make sure all styles is updated */
/* if user has inserted styles with data wish was not initiated  */

void
style_init_each(gpointer key,GtkExTextStyle *style,GtkExText *text)
{

  gint width;
  GtkExTextStyle *def;

  def=gtk_extext_style_get(text,"Default");	
  if(!style->font)	{
  	style->font=def->font;
  	gdk_string_extents(style->font,
                 " `~!@#$%^&*()-_=+\\|[]{};:\"\'<,>.?/1234567890abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ",
                   &style->lbearing,
                   &style->rbearing,
                   &width,
                   &style->ascent,
                   &style->descent);
  }
  
  /* check if color initiated? */

  if(!style->bg_color.pixel)
	style->bg_color=def->bg_color;

  if(!style->fg_color.pixel)	
	style->fg_color=def->fg_color;
}

/* this is called after a new style has been inserted/removed to check if font already exist */
/* if exist ref->exist=TRUE. If not exist the gdk_font_[ref/unref] will be called */
/* by the main foreach loop */

void 
style_font_ref_each(gpointer key,GtkExTextStyle *style,FontRef *ref)
{
  if(style && style->font==ref->font)
  	ref->exist=TRUE;
}

void
style_destroy_each(gpointer key,gpointer freeme,gpointer data)
{
  gtk_extext_style_remove_real(GTK_EXTEXT(data),(gchar *)key);
}

void
propertys_destroy(GtkExText *text)
{
  GtkExTextProperty *next,*cur;

  g_hash_table_foreach(text->hash_styles,(GHFunc)style_destroy_each,text);
  g_hash_table_destroy(text->hash_styles);

  cur=text->property_start;
  while(cur)
  {
  	next=cur->next;

      gtk_signal_emit(GTK_OBJECT(text),
                                extext_signals[PROPERTY_DESTROY],cur);

  	g_free(cur);
  	cur=next;
  };
  text->property_start=text->property_end=NULL;
}

GtkExTextStyle *
gtk_extext_style_insert(GtkExText *text,gchar *key,
									GdkFont *font,GdkColor *fg,GdkColor *bg)
{
  FontRef *fontref=NULL;	
  GtkExTextStyle *style,*def;
  gint width;
	
  def=gtk_extext_style_get(text,"Default");
  style=gtk_extext_style_get(text,key);	
  if(!style)	{
    fontref=g_malloc0(sizeof(FontRef));
    fontref->exist=FALSE;
    fontref->font=font;
    style=g_malloc0(sizeof(GtkExTextStyle));
    strncpy(style->key,key,32);
    g_hash_table_foreach(text->hash_styles,(GHFunc)style_font_ref_each,fontref);
    if(!fontref->exist && font)	/* font dont exist in list */
      gdk_font_ref(font);	/* so we need to refcount++ it */

    g_free(fontref);
    g_hash_table_insert(text->hash_styles,(gpointer *)style->key,style);
  }

  if(strcmp(style->key,"Default")) 
    style->flags=bg ? STYLE_BG : 0; 

  if(def)	{	/* default is up means widget has been realized*/

    style->font=font ? font : def->font;
    style->bg_color=bg ? *bg : def->bg_color;
    style->fg_color=fg ? *fg : def->fg_color;

    gdk_color_alloc(gdk_colormap_get_system(),&style->bg_color);
    gdk_color_alloc(gdk_colormap_get_system(),&style->fg_color);
  }
  else  { /* nope default style yet not set */
				/* this means if some of the members is NULL */
				/* the values will be initiated when widget gets realized */
    style->font=font ? font : NULL;

    if(bg)	{	
      style->bg_color=*bg;	
      gdk_color_alloc(gdk_colormap_get_system(),&style->bg_color);	
    }
    if(fg)	{	
      style->fg_color=*fg; 	
      gdk_color_alloc(gdk_colormap_get_system(),&style->fg_color);	
    }
  }

  if(style->font)	
		gdk_string_extents(style->font,
                   	" `~!@#$%^&*()-_=+\\|[]{};:\"\'<,>.?/1234567890abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ",
                   	&style->lbearing,
                   	&style->rbearing,
                   	&width,
                   	&style->ascent,
                   	&style->descent);

  return style;
}

void 
gtk_extext_style_remove(GtkExText *text, gchar *key)
{
	if(!strcasecmp(key,"Default") && !strcasecmp(key,"Selected") 
				&& !strcasecmp(key,"Prelight")){		
			g_warning("you cannot remove default/selected/Prelight style\n"
					"use gtk_extext_style_insert to change the values\n");	

			return  ;
	}

	gtk_extext_style_remove_real(text,key);
	g_hash_table_remove(text->hash_styles,key);
}

void 
gtk_extext_style_remove_real(GtkExText *text ,gchar *key)
{
  FontRef *fontref=NULL;	
  GtkExTextStyle *style=NULL;	

  style=gtk_extext_style_get(text,key);
  g_hash_table_insert(text->hash_styles,(gpointer *)key,NULL);

  fontref=g_malloc0(sizeof(FontRef));
  fontref->font=style->font;
  fontref->exist=FALSE;
  g_free(style);
  
  g_hash_table_foreach(text->hash_styles,(GHFunc)style_font_ref_each,fontref);
  if(!fontref->exist && fontref->font)/* no style with removed font */					
    gdk_font_unref(fontref->font); /* so we unref it */

  g_free(fontref);
}

GtkExTextStyle *
gtk_extext_style_get(GtkExText *text, gchar *key)
{
  GtkExTextStyle *style;

  style=(gpointer)g_hash_table_lookup(text->hash_styles,key);
  return style;
}

/* we dont use  g_list because we need direct acces to ->next */
GtkExTextProperty *
property_remove_linked_list(GtkExText *text,GtkExTextProperty *cur)
{
	GtkExTextProperty *next;
	if(!cur )
		return NULL;

    text->property_count--;
	if(!cur->prev && !cur->next)	{ /*no more propertys left */
		text->property_start=NULL;
		text->property_end=NULL; 
		text->property_current=NULL; 
		text->scroll_line_start->property_first=NULL;	 

/*		recompute_scroll_line_data(text,text->scroll_line_start->line_number,TRUE);*/

 	    return NULL; 
	}

	next=cur->next;
	if(!next) {	/* LAST property */
		text->property_end=cur->prev;
		text->property_end->next=NULL;

	}
	else     /* somewhere in list */
	{
		if(cur->prev)
			cur->prev->next=next;

		next->prev=cur->prev;
	}
	
	if(!text->property_end || !text->property_end->prev)
		text->property_start=text->property_end;

	if(text->property_current==cur)
		text->property_current=NULL;

	if(text->property_start==cur)
		text->property_start=next;

	if(cur==text->scroll_line_start->property_first )	{
		text->scroll_line_start->property_first=cur->prev;
/*		recompute_scroll_line_data(text,text->scroll_line_start->line_number,TRUE);*/
	}

	return next;
}

void 
property_insert_linked_list(GtkExText *text,
										GtkExTextProperty *cur,
										  GtkExTextProperty *next)
{
	if(!cur)
		return;

    text->property_count++;
	cur->prev=NULL;
	cur->next=NULL;
	/* INSERT BEFORE */
	if(next)
	{
		cur->prev=next->prev;
		next->prev=cur;
		cur->next=next;
		if(cur->prev)
			cur->prev->next=cur;
	}

	if(!cur->next) { /* there is no after cur property_end */
		if(text->property_end)
			text->property_end->next=cur;
			
		cur->prev=text->property_end;
		text->property_end=cur;
	}

	if(!cur->prev) /* cur is the first in list */
		text->property_start=cur;
}

GtkExTextProperty* 
gtk_extext_property_remove_all(GtkExText *text,
											gint startpos,
											gint endpos,
											GtkExTextProperty *startprop)
{
	gint i;
	GtkExTextProperty *next;
	GtkExTextProperty *_this;
    _this=NULL;

	next= startprop ? startprop : text->property_start;

	i=startpos;
	do {
		_this=gtk_extext_property_get_at_pos(text,i,next);
		if(_this)	{
			next=property_remove_linked_list(text,_this);
              gtk_signal_emit(GTK_OBJECT(text),
                                extext_signals[PROPERTY_DESTROY],_this);

               if(_this==startprop)
                  startprop=_this->next;
                     
			g_free(_this);
		}	
		
		i++;
	}while(i<endpos);

	next=gtk_extext_property_move_all(text,endpos,0,startprop) ;
	
	return next;
}

GtkExTextProperty* 
gtk_extext_property_move_all(GtkExText *text,gint pos,
									gint diff,GtkExTextProperty *prop)
{
  GtkExTextProperty *next;
  gboolean oldread;

  oldread=text->read_only;

  next=NULL;  
  if(!prop)
    prop=text->property_start;

  text->read_only=TRUE;
  while(prop)
  {
    if(prop->startpos>=pos)
    {
        if(!next)	
        {
          next=prop;
          if(!diff)    
              goto end;  
        }

        prop->startpos+=diff;
        prop->endpos+=diff;
     }
	prop=prop->next;

/*     while(gtk_events_pending())
       gtk_main_iteration();*/
  };

end:  
  text->read_only=oldread;

  return next;
}

GtkExTextProperty *
gtk_extext_property_get_current(GtkExText *text)
{
  g_return_val_if_fail (GTK_IS_EXTEXT (text), NULL);
	
  return text->property_current;
}

GtkExTextProperty *
gtk_extext_property_insert(GtkExText *text,
										gchar *key,
										gint startpos,
										gint endpos, 
										gpointer data,
										gint type,
                                                    GtkExTextProperty *prev)
{
  gint diff;
  GtkExTextProperty *prop,*next,*extra;
  GtkExTextProperty *propstart,*propend;
	
  propend=NULL;
  propstart=NULL;
  extra=NULL;
  next=NULL;

  if(!prev)
    prev=gtk_extext_property_nearest_backward(text,startpos,NULL);

  diff=endpos-startpos;
  
  prop=g_malloc(sizeof(GtkExTextProperty));
  prop->startpos=0;
  prop->endpos=0;
  prop->style=NULL;
  prop->user_data=data;	
  prop->next=NULL;
  prop->prev=NULL;

  if(key && *key)	{
	  prop->style=gtk_extext_style_get(text,key);
  }	/* if keystyle not found or no key */
  if(!prop->style)	{
	  prop->style=gtk_extext_style_get(text,"Default");
  }

  if(type==PROPERTY_MERGE)  { 
     propstart=gtk_extext_property_get_at_pos(text,startpos,prev);
      propend=gtk_extext_property_get_at_pos(text,endpos-1,prev);
      if(propend && propstart!=propend && propend->endpos >= endpos)  {
         propend->startpos=endpos;
      }
      else if(propend && propstart==propend && propend->endpos >= endpos)  {
        extra=g_malloc(sizeof(GtkExTextProperty));
        extra->startpos=endpos;
	   extra->endpos=propend->endpos;
	   extra->style=propend->style;
	   extra->user_data=propend->user_data;	
	   extra->next=NULL;
	   extra->prev=NULL;     
        /*text->property_count++;*/
        property_insert_linked_list(text,extra,propend->next);
      }
      if(propstart)  {
         propstart->endpos=startpos-1;
      }
	 next=gtk_extext_property_move_all(text,extra ? extra->endpos : endpos,0,prev); 
	 gtk_extext_property_remove_all(text,startpos,endpos,prev);
      if(extra)
        next=extra;
  }
  else if(type==PROPERTY_INSERT) /* NO we dont update em. We only get the next prop to pass to linked_list */ 
	next=gtk_extext_property_move_all(text,startpos,0,prev); 
  else	
	next=gtk_extext_property_remove_all(text,startpos,endpos,prev);

  property_insert_linked_list(text,prop,next);

  prop->startpos=startpos;
  prop->endpos=endpos;

  if(!text->text_insert_delete_flag)
    recompute_text_segment(text,startpos,endpos);

/*
	recompute_scroll_line_data(text,text->scroll_line_start->line_number,TRUE);
*/
	return prop;
}

GtkExTextProperty*
gtk_extext_property_remove(GtkExText *text,
										GtkExTextProperty *prop)
{
  gint start,end;
  GtkExTextProperty *next;

  if(!prop)
  	return text->property_end;

  next=property_remove_linked_list(text,prop);

  start=prop->startpos;
  end=prop->endpos;
  g_free(prop);

  if(!text->text_insert_delete_flag)
    recompute_text_segment(text,start,end);

  return next;     /* we always return next property */
}

/* ALL propertys should be in sorted order */
/*****************************************************/

GtkExTextProperty *
gtk_extext_property_get_at_pos(GtkExText *text,gint pos,
											GtkExTextProperty *oldprop)
{
  GtkExTextProperty *cur,*curstart;

  if(!text->property_start)
  	return NULL;
		
  curstart=cur = oldprop  ? oldprop : text->property_start;
  while(cur)
  {
  	gint val;
  	val=is_value_ok(pos,cur->startpos,cur->endpos);
	/* if return 0, pos is in property interval :-) */
  	if(!val)
  		return cur;

  	/* if value returns more it probadly because property -> startpos */
  	/*  is after pos */
  	if(val<0)
  		return NULL;

  	cur=cur->next;
  };

  /* we started from curstart so we should start before curstart */
  cur=curstart->prev;

  /*	search backward if not found forward */	
  /* until val is less than cur->endpos */
  while(cur)  {
  	gint val;
  	val=is_value_ok(pos,cur->startpos,cur->endpos);
  	/* if val return 0 it is found */
  	if(!val)
  		return cur;

  	if(val>0)
  		return NULL;

  	cur=cur->prev;
  };

  return NULL;
}

/* call property_nearest_forward to return property after current pos  */

GtkExTextProperty *
gtk_extext_property_nearest_forward(GtkExText *text,
		gint pos,
		GtkExTextProperty *cur)
{
	if(!cur)
		cur=text->property_start;	
	
	if(!cur)
		return NULL;

	while(cur)	{
		if(cur->startpos > pos)
			return cur;

		cur=cur->next;
	};

	return cur;	
}

/* call property_nearest_forward to return property after the pos  */

GtkExTextProperty *
gtk_extext_property_nearest_backward(GtkExText *text,
		gint pos,
		GtkExTextProperty *prop)
{
  GtkExTextProperty *new_prop;
  gint val=0;

  if(!prop)
  	prop=text->property_start;	

  new_prop=prop;
  if(prop)	{
  	val=is_value_ok(pos,prop->startpos,prop->endpos);
  	if(val == 0 )
  		return prop->prev ;		
  	else if(val < 0 )	{
  		new_prop=text->property_start;	/* if something goes wrong */
  		while(prop)	{
  			if(is_value_ok(pos,prop->startpos,prop->endpos) >= 0)	
  				return prop->prev;	

  			prop=prop->prev;
  		};
            return NULL;
  	}		
  	else	{
  		new_prop=text->property_end; /* if something goes wrong */
  		while(prop)	{
  			if(is_value_ok(pos,prop->startpos,prop->endpos) <= 0)
  				return prop->prev ;	

  			prop=prop->next;
  		};
            return text->property_end;
  	}
  }	

  return NULL;	
}


/*****************************************************************************/
/*                       Static       Undo code                              */
/*****************************************************************************/

void
extext_undo_foreach(gpointer data,gpointer user_data)
{
  UndoBuffert *undo;
  undo=(UndoBuffert *)data;
  if(undo->buffert)
    g_free(undo->buffert);

  g_free(undo);
}

void
extext_undo_insert(GtkExText *text,gint type, gint start, gint end)
{

  UndoBuffert *undo;
  GSList *cur;

  g_return_if_fail (text != NULL);
  g_return_if_fail (GTK_IS_EXTEXT (text));


  if(text->undo_flag || !text->undo_max) /* we inserting text from undo buffert */
  	return ;			/* so we should NOT insert it in buffert again */

  if(text->undo_max == text->undo_count)  {
    cur=g_slist_last(text->undo_buffert); 
    undo=(UndoBuffert *)cur->data; 
    if(undo->type==EXTEXT_UNDO_REMOVE)
    {	g_free(undo->buffert);   undo->buffert=NULL; }
    
    text->undo_buffert=g_slist_remove_link(text->undo_buffert,cur);
    text->undo_count--;
    text->undo_last--;  
  }
  else
    undo=g_malloc0(sizeof(UndoBuffert));

  undo->type=type;
  undo->startpos=start;
  undo->endpos=end;
  if(undo->type==EXTEXT_UNDO_REMOVE)
  	undo->buffert=text_get_text(text,start,end);
	
  if(!text->undo_buffert)  /* first undo ? */
  	gtk_signal_emit(GTK_OBJECT(text),extext_signals[UNDO_CHANGED]);
  
  text->undo_buffert=g_slist_prepend(text->undo_buffert,undo);
  text->undo_count++;
  text->undo_last++;  
}

/*****************************************************************************/
/*                                 widget api                                   */
/*****************************************************************************/


static void
gtk_extext_set_arg (GtkObject        *object,
		  GtkArg           *arg,
		  guint             arg_id)
{
  GtkExText *text;
  
  text = GTK_EXTEXT (object);
  
  switch (arg_id)
    {
    case ARG_HADJUSTMENT:
      gtk_extext_set_adjustments (text,
				GTK_VALUE_POINTER (*arg),
				text->vadj);
      break;
    case ARG_VADJUSTMENT:
      gtk_extext_set_adjustments (text,
				text->hadj,
				GTK_VALUE_POINTER (*arg));
      break;
	default:
		break;
    }
}

static void
gtk_extext_get_arg (GtkObject        *object,
		  GtkArg           *arg,
		  guint             arg_id)
{
  GtkExText *text;
  
  text = GTK_EXTEXT (object);
  
  switch (arg_id)
    {
    case ARG_HADJUSTMENT:
      GTK_VALUE_POINTER (*arg) = text->hadj;
      break;
    case ARG_VADJUSTMENT:
      GTK_VALUE_POINTER (*arg) = text->vadj;
      break;
    default:
      arg->type = GTK_TYPE_INVALID;
      break;
    }
}

static void
gtk_extext_class_init(GtkExTextClass *_class)
{
  GtkObjectClass *object_class;
  GtkWidgetClass *widget_class;
  GtkEditableClass *editable_class;
  
  object_class = (GtkObjectClass*) _class;
  widget_class = (GtkWidgetClass*) _class;
  editable_class = (GtkEditableClass*) _class;
  parent_class = gtk_type_class (GTK_TYPE_EDITABLE);

  widget_class->set_scroll_adjustments_signal =
    gtk_signal_new ("set_scroll_adjustments",
		    GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GtkExTextClass, set_scroll_adjustments),
		    gtk_marshal_NONE__POINTER_POINTER,
		    GTK_TYPE_NONE, 2, GTK_TYPE_ADJUSTMENT, GTK_TYPE_ADJUSTMENT);

   extext_signals[PROPERTY_TEXT_INSERT]=gtk_signal_new ("property_text_insert",
		    GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GtkExTextClass, property_text_insert),
		    gtk_marshal_BOOL__POINTER_INT_INT,
		    GTK_TYPE_BOOL, 3,GTK_TYPE_POINTER,GTK_TYPE_INT,GTK_TYPE_INT);

   extext_signals[PROPERTY_TEXT_REMOVE]=gtk_signal_new ("property_text_remove",
		    GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GtkExTextClass, property_text_remove),
		    gtk_marshal_BOOL__POINTER_INT_INT,
		    GTK_TYPE_BOOL, 3,GTK_TYPE_POINTER,GTK_TYPE_INT,GTK_TYPE_INT);

   extext_signals[PROPERTY_DESTROY]=gtk_signal_new ("property_destroy",
		    GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GtkExTextClass, property_destroy),
		    gtk_marshal_NONE__POINTER,
		    GTK_TYPE_NONE, 1,GTK_TYPE_POINTER);

   extext_signals[UNDO_CHANGED]=gtk_signal_new ("undo_changed",
		    GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GtkExTextClass, undo_changed),
		    gtk_marshal_NONE__NONE,
		    GTK_TYPE_NONE, 0);

	/* called if cursor has moved to a new property */
   extext_signals[PROPERTY_MARK]=gtk_signal_new ("property_mark",
		    GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GtkExTextClass, property_mark),
		    gtk_marshal_NONE__NONE,
		    GTK_TYPE_NONE, 0);

   extext_signals[LINE_INSERT]=gtk_signal_new ("line_insert",
		    GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GtkExTextClass, line_insert),
		    gtk_marshal_NONE__INT,
		    GTK_TYPE_NONE, 1, GTK_TYPE_INT);

   extext_signals[LINE_REMOVE]=gtk_signal_new ("line_remove",
		    GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GtkExTextClass, line_remove),
		    gtk_marshal_NONE__INT,
		    GTK_TYPE_NONE, 1,GTK_TYPE_INT);
	
  gtk_object_class_add_signals (object_class, extext_signals, LAST_SIGNAL);

  gtk_object_add_arg_type ("GtkExText::hadjustment",
			   GTK_TYPE_ADJUSTMENT,
			   GTK_ARG_READWRITE | GTK_ARG_CONSTRUCT,
			   ARG_HADJUSTMENT);
  gtk_object_add_arg_type ("GtkExText::vadjustment",
			   GTK_TYPE_ADJUSTMENT,
			   GTK_ARG_READWRITE | GTK_ARG_CONSTRUCT,
			   ARG_VADJUSTMENT);

  object_class->set_arg = gtk_extext_set_arg; 
  object_class->get_arg = gtk_extext_get_arg; 

  object_class->destroy=gtk_extext_destroy;
  object_class->finalize=gtk_extext_finalize;



  widget_class->style_set=gtk_extext_style_set;
  widget_class->size_request=gtk_extext_size_request;
  widget_class->size_allocate=gtk_extext_size_allocate;
   widget_class->realize=gtk_extext_realize;
  widget_class->unrealize=gtk_extext_unrealize;
  widget_class->button_press_event = gtk_extext_button_press;
  widget_class->button_release_event = gtk_extext_button_release;
	widget_class->motion_notify_event=gtk_extext_motion_notify;
  widget_class->draw = gtk_extext_draw;
  widget_class->draw_focus=gtk_extext_draw_focus;
  widget_class->expose_event = gtk_extext_expose;
  widget_class->key_press_event=gtk_extext_key_press;
  widget_class->focus_in_event = gtk_extext_focus_in;
  widget_class->focus_out_event = gtk_extext_focus_out;
	

	editable_class->insert_text=gtk_extext_insert_text;
	editable_class->delete_text=gtk_extext_delete_text;
/*	editable_class->changed=gtk_extext_changed;*/
	editable_class->set_position=gtk_extext_set_position;	
	editable_class->update_text=gtk_extext_update_text;
/*	editable_class->get_position=gtk_extext_get_position;	*/
	editable_class->get_chars=gtk_extext_real_get_chars;
	editable_class->set_editable=gtk_extext_real_set_editable;
	editable_class->set_selection=gtk_extext_select_region;
	
	/*editable_class->visible=TRUE;
*/
  _class->property_destroy=NULL;
  _class->set_scroll_adjustments=gtk_extext_set_adjustments;
  _class->line_by_offset=line_get_by_offset;
  _class->column_by_offset=column_get_by_offset;
  _class->property_text_insert=NULL;
  _class->property_text_remove=NULL;
  _class->undo_changed=NULL;
  _class->property_mark=NULL;
  _class->line_insert=NULL;
  _class->line_remove=NULL;

}

void 
gtk_extext_draw_focus (GtkWidget *widget)
{
  GtkExText *text;
  gint width, height;
  gint x, y;
  
  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_EXTEXT (widget));
  
  text = GTK_EXTEXT (widget);
  
  if (GTK_WIDGET_DRAWABLE (widget))
    {
      gint ythick = widget->style->klass->ythickness;
      gint xthick = widget->style->klass->xthickness;
      gint xextra = TEXT_BORDER_ROOM;
      gint yextra = TEXT_BORDER_ROOM;
      
      
      x = 0;
      y = 0;
      width = widget->allocation.width;
      height = widget->allocation.height;
      
      if (GTK_WIDGET_HAS_FOCUS (widget))
	{
	  x += 1;
	  y += 1;
	  width -=  2;
	  height -= 2;
	  xextra -= 1;
	  yextra -= 1;

	  gtk_paint_focus (widget->style, widget->window,
			   NULL, widget, "text",
			   0, 0,
			   widget->allocation.width - 1,
			   widget->allocation.height - 1);
	}

      gtk_paint_shadow (widget->style, widget->window,
			GTK_STATE_NORMAL, GTK_SHADOW_IN,
			NULL, widget, "text",
			x, y, width, height);

      x += xthick; 
      y += ythick;
      width -= 2 * xthick;
      height -= 2 * ythick;
    }
  else
    {
    }
}

static void
gtk_extext_size_request (GtkWidget      *widget,
		       GtkRequisition *requisition)
{
  gint xthickness;
  gint ythickness;
  gint char_height;
  gint char_width; 
  
  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_EXTEXT (widget));
  g_return_if_fail (requisition != NULL);

  xthickness = widget->style->klass->xthickness + TEXT_BORDER_ROOM;
  ythickness = widget->style->klass->ythickness + TEXT_BORDER_ROOM;
  
  char_height = MIN_TEXT_HEIGHT_LINES * (widget->style->font->ascent +
					 widget->style->font->descent);
  
  char_width = MIN_TEXT_WIDTH_LINES * (gdk_text_width (widget->style->font,
						       "ABCDEFGHIJKLMNOPQRSTUVWXYZ",
						       26) 
				       / 26);
  
  requisition->width  = char_width  + xthickness * 2;
  requisition->height = char_height + ythickness * 2;

  if(GTK_EXTEXT(widget)->text_area==widget->window && GTK_EXTEXT(widget)->hadj)
	GTK_EXTEXT(widget)->hadj->page_increment=(gfloat)requisition->width;
}

static void
gtk_extext_size_allocate (GtkWidget     *widget,
			GtkAllocation *allocation)
{
  GtkExText *text;
  GtkEditable *editable;
  
  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_EXTEXT (widget));
  g_return_if_fail (allocation != NULL);
  
  text = GTK_EXTEXT (widget);
  editable = GTK_EDITABLE (widget);
  
  widget->allocation = *allocation;
  if (GTK_WIDGET_REALIZED (widget))
    {
      gdk_window_move_resize (widget->window,
			      allocation->x, allocation->y,
			      allocation->width, allocation->height);
      
      gdk_window_move_resize (text->text_area,
			      widget->style->klass->xthickness + TEXT_BORDER_ROOM,
			      widget->style->klass->ythickness + TEXT_BORDER_ROOM,
			      MAX (1, (gint)widget->allocation.width - (gint)(widget->style->klass->xthickness +
							  (gint)TEXT_BORDER_ROOM) * 2),
			      MAX (1, (gint)widget->allocation.height - (gint)(widget->style->klass->ythickness +
							   (gint)TEXT_BORDER_ROOM) * 2));
      
#ifdef USE_XfIM
      if (editable->ic && (gdk_ic_get_style (editable->ic) & GDK_IM_PREEDIT_POSITION))
	{
	  gint width, height;
	  
	  gdk_window_get_size (text->text_area, &width, &height);
	  editable->ic_attr->preedit_area.width = width;
	  editable->ic_attr->preedit_area.height = height;

	  gdk_ic_set_attr (editable->ic,
	      		   editable->ic_attr, GDK_IC_PREEDIT_AREA);
	}
#endif
      
      recompute_geometry (text);
    }
}

/* FIXME */
/* this code is fucked up */
static void
recompute_scroll_line_width(GtkExText *text, GtkExTextLineData *lp,gint count)
{
  gint i;
  InternalPaintCache *pcache;
  gboolean haschanged;

  g_return_if_fail (text != NULL);
  g_return_if_fail (GTK_IS_EXTEXT (text));
 
  if (text->line_wrap)  {
    text->hadj->upper=0;
   /* text->hadj->page_size=0;*/
   /* text->hadj->page_increment=0;*/   
     text->hadj->value=0;
     gtk_adjustment_value_changed(text->hadj); 
     gtk_adjustment_changed(text->hadj); 
    return ;
  }
  haschanged=FALSE;
  pcache=NULL;
  for(i=0; i < count ; i++)  {
    lp=gtk_extext_get_line_data(text,lp->line_number+i,lp);
    pcache=create_paint_cache(text,lp,0,LINE_DATA_LENGTH(lp),0xFFFF,pcache);
    if(pcache->width > (gint)text->hadj->upper)  {
      text->hadj->upper=(gfloat) pcache->width;  
      haschanged=TRUE;  
    }
  }
  if(haschanged)
    gtk_adjustment_changed(text->hadj);

  g_free(pcache);
}	 

static void 
recompute_scroll_line_height(GtkExText *text)
{
  gint height,width,count,h,startpos,i;
  InternalPaintCache *pcache;
  GtkExTextStyle *style;
  GtkExTextLineData *linedataptr;

  gdk_window_get_size (text->text_area, &width, &height);

  count=0;
  h=0;
  pcache=NULL;

  style=gtk_extext_style_get(text,"Default");
  linedataptr=text->scroll_line_start;
  while(TRUE)	{
    if(text->scroll_line_start->line_number + count <= text->line_count)  {
	  linedataptr=gtk_extext_get_line_data(text,text->scroll_line_start->line_number + count,linedataptr);
      startpos=linedataptr->startpos;
       for( i=0;i < LINE_DATA_LENGTH(linedataptr);i+=pcache->length)     {
         pcache=create_paint_cache(text,linedataptr,i,LINE_DATA_LENGTH(linedataptr)-i,text->line_wrap ? width- LINE_WRAP_WIDTH : 0xFFFF,pcache);
         h+=pcache->height;   
       }
    }
    else
      h+=TEXT_STYLE_HEIGHT(style);

	if(h>height )
          goto stop;

	count++;
  };

stop:
    g_free(pcache);
	g_free(linedataptr);
	text->scroll_line_count=count;

   text->vadj->page_increment=(gfloat)text->scroll_line_count;
   text->vadj->page_size=(gfloat)text->scroll_line_count;
   if(text->vadj->upper < text->vadj->page_size)
     text->vadj->upper=(gfloat)text->scroll_line_count;
}

static void 
recompute_geometry (GtkExText* text)
{	
  gint height,width;

  gdk_window_get_size (text->text_area, &width, &height);

  if(text->draw_area)
      gdk_pixmap_unref(text->draw_area);
          
  text->draw_area=gdk_pixmap_new(text->text_area,width,height,-1);

  recompute_scroll_line_height(text);


	text->hadj->page_size=width;
     if(text->line_wrap || text->word_wrap || text->hadj->upper < (gfloat) width)
  	text->hadj->upper=width;

     gtk_adjustment_changed(text->hadj); 


	text->vadj->step_increment=1.0;
	text->vadj->page_size=(gfloat)text->scroll_line_count;
	text->vadj->upper=(gfloat)text->line_count;
}

void 
recompute_scroll_line_data(GtkExText *text,gint pos,gboolean clearprop)
{
	GtkExTextLineData *lp;
	if(clearprop)	{
		text->scroll_line_start->property_first=NULL;
	}
	lp=gtk_extext_get_line_data(text,pos,text->scroll_line_start);
	g_free(text->scroll_line_start);
	text->scroll_line_start=lp;
}

void
gtk_extext_set_adjustments (GtkExText       *text,
			  GtkAdjustment *hadj,
			  GtkAdjustment *vadj)
{

  g_return_if_fail (text != NULL);
  g_return_if_fail (GTK_IS_EXTEXT (text));

  if (hadj)
    g_return_if_fail (GTK_IS_ADJUSTMENT (hadj));
  else
    hadj = GTK_ADJUSTMENT (gtk_adjustment_new (0.0, 0.0, 0.0, 0.0, 0.0, 0.0));
  if (vadj)
    g_return_if_fail (GTK_IS_ADJUSTMENT (vadj));
  else
    vadj = GTK_ADJUSTMENT (gtk_adjustment_new (0.0, 0.0, 0.0, 0.0, 0.0, 0.0));
  
  if (text->hadj && (text->hadj != hadj))
    {
      gtk_signal_disconnect_by_data (GTK_OBJECT (text->hadj), text);
      gtk_object_unref (GTK_OBJECT (text->hadj));
    }
  
  if (text->vadj && (text->vadj != vadj))
    {
      gtk_signal_disconnect_by_data (GTK_OBJECT (text->vadj), text);
      gtk_object_unref (GTK_OBJECT (text->vadj));
    }
  
  if (text->hadj != hadj)
    {
      text->hadj = hadj;
      gtk_object_ref (GTK_OBJECT (text->hadj));
      gtk_object_sink (GTK_OBJECT (text->hadj));
      
      gtk_signal_connect (GTK_OBJECT (text->hadj), "changed",
			  (GtkSignalFunc) gtk_extext_adjustment_changed,
			  text);
      gtk_signal_connect (GTK_OBJECT (text->hadj), "value_changed",
			  (GtkSignalFunc) gtk_extext_adjustment_value_changed,
			  text);
      gtk_signal_connect (GTK_OBJECT (text->hadj), "disconnect",
			  (GtkSignalFunc) gtk_extext_disconnect,
			  text);
      gtk_extext_adjustment_changed (hadj, text);
    }
  
  if (text->vadj != vadj)
    {
      text->vadj = vadj;
      gtk_object_ref (GTK_OBJECT (text->vadj));
      gtk_object_sink (GTK_OBJECT (text->vadj));
      
      gtk_signal_connect (GTK_OBJECT (text->vadj), "changed",
			  (GtkSignalFunc) gtk_extext_adjustment_changed,
			  text);
      gtk_signal_connect (GTK_OBJECT (text->vadj), "value_changed",
			  (GtkSignalFunc) gtk_extext_adjustment_value_changed,
			  text);
      gtk_signal_connect (GTK_OBJECT (text->vadj), "disconnect",
			  (GtkSignalFunc) gtk_extext_disconnect,
			  text);
      gtk_extext_adjustment_changed (vadj, text);
    }
	
	text->hadj->step_increment=0;
	text->hadj->page_increment=0;
	text->hadj->page_size=0;
	text->hadj->upper=0;
	text->hadj->value=0;
	
	
	text->vadj->step_increment=1.0;
	text->vadj->page_increment=text->scroll_line_count;
	text->vadj->page_size=text->scroll_line_count;
	text->vadj->upper=text->line_count;
}

static void
gtk_extext_init(GtkExText *text)
{
  GTK_WIDGET_SET_FLAGS(text,GTK_CAN_FOCUS);

  text->draw_area=NULL;
  text->gc=NULL;
  text->gc_xor=NULL;
  text->bg_gc=NULL;
  text->text_area=NULL;
  text->font=NULL;

  text->line_wrap=FALSE;
  text->word_wrap=FALSE;

  text->line_wrap_bitmap = NULL;
  text->line_arrow_bitmap = NULL;

	/* line max character count 0 = unlimted */
	/* if  this member is set the widget will insert  */
    /* '\n' character when line is to long   */
  text->line_max_chars=0;

  text->use_wchar=FALSE;
  text->text.ch=g_new0(guchar,TEXT_BUFFERT_DEFAULT_SIZE);
  text->size=TEXT_BUFFERT_DEFAULT_SIZE;	
  text->gap_len=TEXT_BUFFERT_DEFAULT_SIZE;
  text->part1len=0;
  text->part2text.ch=text->text.ch+text->gap_len;
  text->length=0;

  text->read_only=FALSE;

  text->hash_styles=g_hash_table_new(g_str_hash,g_str_equal);

  text->line_start=text->line_end=g_malloc0(sizeof(Lines));
  text->line_pos=text->line_start;
  text->line_pos_index=0;
  text->line_count=0;
  text->line_cursor_index=0;
  text->scroll_line_start=gtk_extext_get_line_data(text,0,NULL);

  text->default_tab_width = DEFAULT_TAB_STOP;
  text->tab_stops = NULL;

  text->property_count=0;
  text->property_start=NULL;
  text->property_end=NULL;
  text->property_current=NULL;

  text->undo_buffert=NULL;
  text->undo_max=5;
  text->undo_count=0;
  text->undo_last=0;
  text->text_insert_delete_flag=FALSE;
  text->undo_flag=FALSE;

  GTK_EDITABLE(text)->current_pos=0;
}

GtkWidget *
gtk_extext_new()
{
	GtkWidget *text;

	  text = gtk_widget_new (GTK_TYPE_EXTEXT,
			 "hadjustment", NULL,
			 "vadjustment", NULL,
			 NULL);


	return text;
}

static void
gtk_extext_destroy(GtkObject *object)
{
  GtkExText *text;
  
  g_return_if_fail (object != NULL);
  g_return_if_fail (GTK_IS_EXTEXT (object));
  
  text=GTK_EXTEXT(object);

  propertys_destroy(text);
  
  gtk_signal_disconnect_by_data (GTK_OBJECT (text->hadj), text);
  gtk_signal_disconnect_by_data (GTK_OBJECT (text->vadj), text);

  if (text->timer)
    {
      gtk_timeout_remove (text->timer);
      text->timer = 0;
    }
	if(text->cursor_timer)	{
		gtk_timeout_remove(text->cursor_timer);
		text->cursor_timer=0;
	}  

	GTK_OBJECT_CLASS(parent_class)->destroy(object);
}

static void 
gtk_extext_finalize(GtkObject *object)
{
  GtkExText *text;
  Lines *current,*next;

  g_return_if_fail (object != NULL);
  g_return_if_fail (GTK_IS_EXTEXT (object));

  
  text=GTK_EXTEXT(object);

  gtk_object_unref (GTK_OBJECT (text->hadj));
  gtk_object_unref (GTK_OBJECT (text->vadj));

  gtk_extext_undo_clear_all(GTK_EXTEXT(object));
  current=text->line_start;
  while(current)	{
  	next=current->next;
  	g_free(current);
  	current=next;
  };	

  if(text->tab_stops)
  	g_list_free(text->tab_stops);

  if(!text->use_wchar)
	g_free(text->text.ch);
  else 
	g_free(text->text.wc);

  GTK_OBJECT_CLASS(parent_class)->finalize (object);
}

static void 
gtk_extext_realize(GtkWidget *widget)
{
  GtkExText *text;
  GtkEditable *editable;
  GdkWindowAttr attributes;
  gint attributes_mask;
  GtkExTextLineData* lp;
  
  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_EXTEXT (widget));
  
  text = GTK_EXTEXT (widget);
  editable = GTK_EDITABLE (widget);
  GTK_WIDGET_SET_FLAGS (text, GTK_REALIZED);
  
  attributes.window_type = GDK_WINDOW_CHILD;
  attributes.x = widget->allocation.x;
  attributes.y = widget->allocation.y;
  attributes.width = widget->allocation.width;
  attributes.height = widget->allocation.height;
  attributes.wclass = GDK_INPUT_OUTPUT;
  attributes.visual = gtk_widget_get_visual (widget);
  attributes.colormap = gtk_widget_get_colormap (widget);
  attributes.event_mask = gtk_widget_get_events (widget);
  attributes.event_mask |= (GDK_EXPOSURE_MASK |
			    GDK_BUTTON_PRESS_MASK |
			    GDK_BUTTON_RELEASE_MASK |
			    GDK_BUTTON_MOTION_MASK |
			    GDK_ENTER_NOTIFY_MASK |
			    GDK_LEAVE_NOTIFY_MASK |
			    GDK_KEY_PRESS_MASK);
  attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;
  
  widget->window = gdk_window_new (gtk_widget_get_parent_window (widget), &attributes, attributes_mask);
  gdk_window_set_user_data (widget->window, text);
  
  attributes.x = (widget->style->klass->xthickness + TEXT_BORDER_ROOM);
  attributes.y = (widget->style->klass->ythickness + TEXT_BORDER_ROOM);
  attributes.width = MAX (1, (gint)widget->allocation.width - (gint)attributes.x * 2);
  attributes.height = MAX (1, (gint)widget->allocation.height - (gint)attributes.y * 2);
  
  text->text_area = gdk_window_new (widget->window, &attributes, attributes_mask);
  gdk_window_set_user_data (text->text_area, text);
  
  widget->style = gtk_style_attach (widget->style, widget->window);
  
  /* Can't call gtk_style_set_background here because it's handled specially */
  gdk_window_set_background (widget->window, &widget->style->bg[GTK_STATE_NORMAL]);
 gdk_window_set_background (text->text_area, &widget->style->bg[GTK_STATE_NORMAL]);

/*  if (widget->style->bg_pixmap[GTK_STATE_NORMAL])
    text->bg_gc = create_bg_gc (text);*/
  
  text->line_wrap_bitmap = gdk_bitmap_create_from_data (text->text_area,
							(gchar*) line_wrap_bits,
							line_wrap_width,
							line_wrap_height);
  
  text->line_arrow_bitmap = gdk_bitmap_create_from_data (text->text_area,
							 (gchar*)line_arrow_bits,
							 line_arrow_width,
							 line_arrow_height);
  
  if(GTK_WIDGET (text)->style->bg_pixmap[GTK_STATE_NORMAL])
  {
    GdkGCValues values;    
    values.tile = GTK_WIDGET (text)->style->bg_pixmap[GTK_STATE_NORMAL];
    values.fill = GDK_TILED;

    text->bg_gc=gdk_gc_new_with_values (text->text_area, &values,
				 GDK_GC_FILL | GDK_GC_TILE);

  }
  else
    text->bg_gc =  gdk_gc_new (text->text_area); 

  gdk_gc_set_foreground (text->bg_gc, &widget->style->bg[GTK_STATE_NORMAL]);
  gdk_gc_set_background (text->bg_gc, &widget->style->bg[GTK_STATE_NORMAL]);

  text->gc = gdk_gc_new (text->text_area);
  text->gc_xor = gdk_gc_new (text->text_area);
  gdk_gc_set_function(text->gc_xor, GDK_XOR);

  gdk_gc_set_exposures (text->gc, TRUE);
  gdk_gc_set_foreground (text->gc, &widget->style->text[GTK_STATE_NORMAL]);
  gdk_gc_set_background (text->gc, &widget->style->bg[GTK_STATE_NORMAL]);
  
#ifdef USE_XIM            /* WHAT THE FUCK IS XIM??? code stolen from gtktext */
  if (gdk_im_ready () && (editable->ic_attr = gdk_ic_attr_new ()) != NULL)
    {
      gint width, height;
      GdkColormap *colormap;
      GdkEventMask mask;
      GdkICAttr *attr = editable->ic_attr;
      GdkICAttributesType attrmask = GDK_IC_ALL_REQ;
      GdkIMStyle style;
      GdkIMStyle supported_style = GDK_IM_PREEDIT_NONE | 
	                           GDK_IM_PREEDIT_NOTHING |
	                           GDK_IM_PREEDIT_POSITION |
	                           GDK_IM_STATUS_NONE |
	                           GDK_IM_STATUS_NOTHING;
      
      if (widget->style && widget->style->font->type != GDK_FONT_FONTSET)
	supported_style &= ~GDK_IM_PREEDIT_POSITION;
      
      attr->style = style = gdk_im_decide_style (supported_style);
      attr->client_window = text->text_area;

      if ((colormap = gtk_widget_get_colormap (widget)) !=
	  gtk_widget_get_default_colormap ())
	{
	  attrmask |= GDK_IC_PREEDIT_COLORMAP;
	  attr->preedit_colormap = colormap;
	}

      switch (style & GDK_IM_PREEDIT_MASK)
	{
	case GDK_IM_PREEDIT_POSITION:
	  if (widget->style && widget->style->font->type != GDK_FONT_FONTSET)
	    {
	      g_warning ("over-the-spot style requires fontset");
	      break;
	    }

	  attrmask |= GDK_IC_PREEDIT_POSITION_REQ;
	  gdk_window_get_size (text->text_area, &width, &height);
	  attr->spot_location.x = 0;
	  attr->spot_location.y = height;
	  attr->preedit_area.x = 0;
	  attr->preedit_area.y = 0;
	  attr->preedit_area.width = width;
	  attr->preedit_area.height = height;
	  attr->preedit_fontset = widget->style->font;
	  
	  break;
	}
      editable->ic = gdk_ic_new (attr, attrmask);
      
      if (editable->ic == NULL)
	g_warning ("Can't create input context.");
      else
	{
	  mask = gdk_window_get_events (text->text_area);
	  mask |= gdk_ic_get_events (editable->ic);
	  gdk_window_set_events (text->text_area, mask);
	  
	  if (GTK_WIDGET_HAS_FOCUS (widget))
	    gdk_im_begin (editable->ic, text->text_area);
	}
    }
#endif

  text->font=widget->style->font;
  gdk_window_show (text->text_area);

  if (editable->selection_start_pos != editable->selection_end_pos)
    gtk_editable_claim_selection (editable, TRUE, GDK_CURRENT_TIME);


/* 
 if(!(style=gtk_extext_style_get(text,"Default")))
    style=gtk_extext_style_insert(text,"Default",
					GTK_WIDGET(text)->style->font,
					&GTK_WIDGET(text)->style->fg[GTK_STATE_NORMAL],
					&GTK_WIDGET(text)->style->bg[GTK_STATE_NORMAL]);

  if(!gtk_extext_style_get(text,"Selected"))
	gtk_extext_style_insert(text,"Selected",
					GTK_WIDGET(text)->style->font,
              		&GTK_WIDGET(text)->style->fg[GTK_STATE_SELECTED],
					&GTK_WIDGET(text)->style->bg[GTK_STATE_SELECTED]);

  if(!gtk_extext_style_get(text,"Prelight"))  
	gtk_extext_style_insert(text,"Prelight",
					GTK_WIDGET(text)->style->font,
              		&GTK_WIDGET(text)->style->fg[GTK_STATE_PRELIGHT],
					&GTK_WIDGET(text)->style->bg[GTK_STATE_PRELIGHT]);
*/

  /* oki now we need to check if user has setup any styles */
  /* unitialized Color/font values will be set to default values */

  g_hash_table_foreach(text->hash_styles,(GHFunc)style_init_each,text);

  text->timer=0;
  text->freeze_count=0;
  if(text->hadj)	{
    text->hadj->value=0;
    text->hadj->step_increment=gdk_char_width(text->font,' ');
    text->hadj->page_increment=gdk_char_width(text->font,' ') * 10;
     gtk_adjustment_value_changed(text->hadj); 
  }
	
  recompute_scroll_line_data(text,0,TRUE);
  recompute_geometry (text);

  lp=gtk_extext_get_line_data(text,0,NULL);
  gtk_extext_get_text_data_foreach_line(text,lp,        
                                      &text_data_set_line_data,
                                  GINT_TO_POINTER(text->length));
  g_free(lp);
}

static void
gtk_extext_unrealize(GtkWidget *widget)
{
 GtkExText *text;
  
  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_EXTEXT (widget));
  
  text = GTK_EXTEXT (widget);
 
#ifdef USE_XIM
  if (GTK_EDITABLE (widget)->ic)
    {
      gdk_ic_destroy (GTK_EDITABLE (widget)->ic);
      GTK_EDITABLE (widget)->ic = NULL;
    }
  if (GTK_EDITABLE (widget)->ic_attr)
    {
      gdk_ic_attr_destroy (GTK_EDITABLE (widget)->ic_attr);
      GTK_EDITABLE (widget)->ic_attr = NULL;
    }
#endif

  gdk_window_set_user_data (text->text_area, NULL);
  gdk_window_destroy (text->text_area);
  text->text_area = NULL;
  
  gdk_gc_destroy (text->gc);
  text->gc = NULL;


  if (text->bg_gc)
    {
      gdk_gc_destroy (text->bg_gc);
      text->bg_gc = NULL;
    }

	if(text->draw_area)	{
		gdk_pixmap_unref(text->draw_area);
		text->draw_area=NULL;
	}

  gdk_pixmap_unref (text->line_wrap_bitmap);
  gdk_pixmap_unref (text->line_arrow_bitmap);
  
  if (GTK_WIDGET_CLASS (parent_class)->unrealize)
    (* GTK_WIDGET_CLASS (parent_class)->unrealize) (widget);

 
}

static void
gtk_extext_style_set(GtkWidget *widget,GtkStyle *old)
{
  GtkExText *text;
  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_EXTEXT (widget));
  
  text=GTK_EXTEXT(widget);

  if(GTK_WIDGET_REALIZED(widget))
  {
 /*   gdk_window_set_background (widget->window, &widget->style->bg[GTK_STATE_NORMAL]);
    gdk_window_set_background (text->text_area, &widget->style->bg[GTK_STATE_NORMAL]);
*/
    gdk_gc_set_foreground (text->gc, &widget->style->text[GTK_STATE_NORMAL]);
    gdk_gc_set_background (text->gc, &widget->style->bg[GTK_STATE_NORMAL]);
    gdk_gc_set_foreground (text->bg_gc, &widget->style->bg[GTK_STATE_NORMAL]);
    gdk_gc_set_background (text->bg_gc, &widget->style->bg[GTK_STATE_NORMAL]);
  }

  if(!widget->style->font && old && old->font)
    widget->style->font=old->font;
  else if(!widget->style->font)  {
    g_warning("No default style->font set in gtkextext????");
    return ;
  }

  gtk_extext_style_insert(text,"Default",
					GTK_WIDGET(text)->style->font,
					&GTK_WIDGET(text)->style->text[GTK_STATE_NORMAL],
					&GTK_WIDGET(text)->style->bg[GTK_STATE_NORMAL]);


  gtk_extext_style_insert(text,"Selected",  
              GTK_WIDGET(text)->style->font,
            &GTK_WIDGET(text)->style->text[GTK_STATE_SELECTED],
            &GTK_WIDGET(text)->style->bg[GTK_STATE_SELECTED]	);

  gtk_extext_style_insert(text,"Prelight",
					GTK_WIDGET(text)->style->font,
		&GTK_WIDGET(text)->style->text[GTK_STATE_PRELIGHT],
					&GTK_WIDGET(text)->style->bg[GTK_STATE_PRELIGHT]	);
  
}

static void 
expose_text(GtkExText *text,GdkRectangle *area,gboolean hmm)
{
  gint lines;
  gint endline;	
  gint i;
  gint w,h,y;
  GtkExTextLineData *linedataptr;
  GdkDrawable *draw;

  g_return_if_fail (text != NULL);
  g_return_if_fail (GTK_IS_EXTEXT (text));

  if(text->freeze_count)
	return;

  if(text->cursor_timer)	{
  	gtk_timeout_remove(text->cursor_timer);
  	text->cursor_timer=0;
  }
	
  lines=text->scroll_line_start->line_number;
  if(text->scroll_line_count+text->scroll_line_start->line_number > text->line_count)
	endline=text->line_count;
  else
	endline=text->scroll_line_start->line_number  +text->scroll_line_count;

  i=0;
  if( text->draw_area)
    draw=text->draw_area;
  else
    draw=text->text_area;

  gdk_window_get_size(text->text_area,&w,&h);


  /* only draw if pixmap */
  gdk_gc_set_ts_origin (text->bg_gc,0,0);
  gdk_draw_rectangle(draw,text->bg_gc,TRUE,0,0,w,h);

  /* initiate linedataptr by passing first visible line */
 
  linedataptr=gtk_extext_get_line_data(text,lines,text->scroll_line_start);
  y=0;
  for(;lines<=endline;lines++)	{
      gtk_extext_get_line_data(text,lines,linedataptr);
     if(!text->line_wrap)
      y=expose_line_text(text,draw,linedataptr,0,y);
    else
      y=expose_line_text_wrap(text,draw,linedataptr,0,y);
  }

  gdk_draw_line(draw,GTK_WIDGET(text)->style->black_gc,
				text->cursor.x,text->cursor.y,
				text->cursor.w,text->cursor.h);

  if(draw!=text->text_area)
  {
    gdk_draw_pixmap(text->text_area,
			text->bg_gc,
			draw,area->x,area->y,area->x,area->y,
			area->width,area->height	);

  }
  gtk_widget_draw_focus(GTK_WIDGET(text)); 

  g_free(linedataptr);
  text->cursor_timer=gtk_timeout_add(250,gtk_extext_cursor_blink_start,text);
}

void recompute_text_segment(GtkExText *text, gint startpos,gint len)
{
  GtkExTextLineData *lp;

  g_return_if_fail (text != NULL);
  g_return_if_fail (GTK_IS_EXTEXT (text));


  /* how can we optimize this crap */
  return ;

  lp=gtk_extext_get_line_by_char_pos(text,startpos);

  len+=startpos-lp->startpos;
  startpos=lp->startpos;

  gtk_extext_get_text_data_foreach_line(text,lp,        
                                      &text_data_set_line_data,
                                  GINT_TO_POINTER(startpos+len));

  g_free(lp);
}

/* this routine will iterate from line start at startpos up to line including startpos+length+lastline length */

void
gtk_extext_get_text_data_foreach_line(GtkExText *text, GtkExTextLineData * lp, TextDataCallback cb,gpointer userdata)
{
  InternalPaintCache pcache;
  gint i;
  gint linecount;
  
  g_return_if_fail (text != NULL);
  g_return_if_fail (GTK_IS_EXTEXT (text));
  g_return_if_fail (cb != NULL);

	/* THIS CODE IS FUCKED UP */

	return ;

  if(!GTK_WIDGET_REALIZED(text))
    return ;

  linecount=lp->line_number;
  while(linecount <= text->line_count && LINE_DATA_LENGTH(lp)) {
    gtk_extext_get_line_data(text,linecount,lp);
    for(i=0; i < LINE_DATA_LENGTH(lp)  ; i+=pcache.length)  {
      create_paint_cache(text,lp,i,LINE_DATA_LENGTH(lp),0xFFFF,&pcache);    
      if(!cb(text,lp,&pcache,userdata))
        return ; 
    }
    linecount++;
  }    
}

static gint
text_data_set_line_data(GtkExText *text,GtkExTextLineData *lp,InternalPaintCache *pcache,gpointer data)
{
  g_return_val_if_fail (text != NULL,FALSE);
  g_return_val_if_fail (GTK_IS_EXTEXT (text),FALSE);
  
  if(lp->startpos > GPOINTER_TO_INT(data) )
    return FALSE;   /* stop iterate */

  LINE_DATA_HEIGHT(lp)=pcache->height;
  LINE_DATA_WIDTH(lp)=pcache->width;
  g_print("recompute line %d\n",lp->line_number);

  return TRUE;
}

/* NEED TO CLEAN UP THIS MESS */

static InternalPaintCache *
create_paint_cache(GtkExText *text,GtkExTextLineData *linedataptr,gint line_offset,gint length,gint max_width,InternalPaintCache *pcache)
{
  gchar chars[24];
  gint i,pindex,b;
  gint tabindex,tab;	
  GtkExTextStyle *style,*oldstyle; 
  GtkExTextStyle *style_selected; 
  gint curwidth;
  gint offset;
  GtkExTextProperty *prop; 
  GtkExTextProperty *oldprop; 
  gint startpos;
  InternalStyleCache *pstyle;

  g_return_val_if_fail (text != NULL,NULL);
  g_return_val_if_fail (GTK_IS_EXTEXT (text),NULL);


  if(!pcache)
  	pcache=g_malloc0(sizeof(InternalPaintCache));
  else
  	memset(pcache,0,sizeof(InternalPaintCache));

  if(!GTK_WIDGET_REALIZED(GTK_WIDGET(text)))
    return pcache;


  pstyle=pcache->styles;
  
  startpos=linedataptr->startpos+line_offset;

  oldprop=linedataptr->property_first;
  prop=NULL;
  offset=0;
  tabindex=0;
  tab=0;				/* is there a tab this is set to [tabindex].tab */
  pindex=0;				/* pstyle index */
  style=NULL;						/* Current style */
  oldstyle=NULL;
		
	/* Cache 0 is ALWAYS Default */
  
  style=oldstyle=gtk_extext_style_get(text,"Default");		
  memcpy(&pstyle[0].style,style,sizeof(GtkExTextStyle));
  pstyle[0].len=0;
  pstyle[0].height=TEXT_STYLE_HEIGHT(style);
  pcache->startpos=startpos;
  pcache->height=pstyle[0].height;
  pcache->width=0;
  pcache->length=0;
  pcache->cursor_x=-1;
  pcache->cursor_h=-1;
  pcache->style_count=0;

  if(GTK_WIDGET_HAS_FOCUS(text))
	style_selected=gtk_extext_style_get(text,"Selected");
  else
	style_selected=gtk_extext_style_get(text,"Prelight");

  /* we need to check for tabs at linestart to startpos */
  if(line_offset)	{
  	i=linedataptr->startpos;
  	tabindex=0;
  	while(i<startpos)	{
  		if(  GTK_EXTEXT_INDEX(text,i)=='\t' )
  			tabindex++;
  		else if(  GTK_EXTEXT_INDEX(text,i)=='\n' )
		      tabindex=0;

		i++;
  	}
  }

  tab=0;  
  for( i=0,b=0; i < length && 
  			   pcache->width < max_width && 
  			       pindex < MAX_STYLE_PER_LINE ;
                            i++ , b++)
  {
    if(tab)
      b+=tab;

    tab=0;	
    if( GTK_EXTEXT_INDEX(text,i+startpos)=='\t')	{	
      tab=gtk_extext_get_tab_width(text,tabindex);
      memset(chars,' ',tab < 24 ? tab : 24);	/* insert spaces */
      chars[tab]='\0';
      tab--;
      tabindex++;
    }
    else
    {	chars[0]=GTK_EXTEXT_INDEX(text,i+startpos);	chars[1]='\0';	}

    prop=gtk_extext_property_get_at_pos(text,i+startpos,oldprop);
    if(prop)	{
      style=prop->style;				
      oldprop=prop;
      if(TEXT_STYLE_HEIGHT(style) > pcache->height)
        pcache->height=TEXT_STYLE_HEIGHT(style);
    }
    else
      style=&pstyle[0].style; 
  	
    if(i+line_offset==text->line_cursor_index)	{
      if(prop && prop->startpos == i + line_offset)
        pcache->cursor_h=TEXT_STYLE_HEIGHT(style);
      else
        pcache->cursor_h=TEXT_STYLE_HEIGHT(oldstyle);
  	
        pcache->cursor_x=pcache->width;	
   }
			/* SELECTIONS ALWAYS CLEAR STYLE AT POS */
    if(!is_value_ok(i+startpos, GTK_EDITABLE(text)->selection_start_pos,GTK_EDITABLE(text)->selection_end_pos))	{
      if(style->font!=pstyle[pindex].style.font || oldstyle!=style_selected)	{
          pindex++;

          memcpy(&pstyle[pindex].style,style,sizeof(GtkExTextStyle));
          pstyle[pindex].style.bg_color=style_selected->bg_color;
          pstyle[pindex].style.fg_color=style_selected->fg_color;
          pstyle[pindex].style.flags=STYLE_BG;

          pstyle[pindex].len=tab+1;

          pstyle[pindex].height=TEXT_STYLE_HEIGHT(style);
          curwidth=gdk_text_width(style->font,chars,tab+1);

          pstyle[pindex].width=curwidth;
          pcache->width+=curwidth;

          oldstyle=style_selected;
      }
      else	{
        pstyle[pindex].len+=tab+1;
        curwidth=gdk_text_width(style->font,
							chars,tab+1);

         pstyle[pindex].width+=curwidth;
         pcache->width+=curwidth;
      }
    }	
	/* Init new style */
    else if(style!=oldstyle )
    {
      pindex++;
      memcpy(&pstyle[pindex].style,style,sizeof(GtkExTextStyle));
      pstyle[pindex].len=tab+1;
      pstyle[pindex].height=TEXT_STYLE_HEIGHT(style);

      curwidth=gdk_text_width(style->font,
							chars,tab+1);

      pstyle[pindex].width=curwidth;
      pcache->width+=curwidth;

      oldstyle=style;
    }
    else	{
      pstyle[pindex].len+=tab+1;				
      curwidth=gdk_text_width(style->font,chars,tab+1);

      pstyle[pindex].width+=curwidth;
      pcache->width+=curwidth;
    }
  }	
  if(i+line_offset==text->line_cursor_index)	{
      if(prop && prop->startpos==i + line_offset)
        pcache->cursor_h=TEXT_STYLE_HEIGHT(style);
      else
        pcache->cursor_h=TEXT_STYLE_HEIGHT(oldstyle);
  	
      pcache->cursor_x=pcache->width;	
  }
  if( pcache->length < LINE_DATA_LENGTH(linedataptr)  && text->word_wrap)	{
    g_print("line wordwrap is broken\n");
  } 

  pcache->length=i;
  pcache->style_count=pindex+1;

  return pcache;  
}

gint 
expose_line_text(GtkExText *text,GdkDrawable *draw_area,
			GtkExTextLineData *linedataptr,gint x,gint y)
{
  gchar *txt;
  gint i;
  gint tabindex;	
  gint offset;
  InternalPaintCache *pcache;
  InternalStyleCache *pstyle;
  GtkExTextStyle *style;

  pcache=create_paint_cache(text,linedataptr,0,
  								LINE_DATA_LENGTH(linedataptr),0xFFFF,NULL);
  pstyle=pcache->styles;
  tabindex=0;
  txt=text_get_text_with_expanded_tabs(text,linedataptr->startpos,linedataptr->endpos,&tabindex);
  if(txt)	{
  	x=(gint)-text->hadj->value;
  	offset=0;
  	for(i=0;i<pcache->style_count;i++)	{
  		if(pstyle[i].len)
  		{
  			style=&pstyle[i].style;
                if(style->flags & STYLE_BG)  {
  			gdk_gc_set_foreground(text->gc,&style->bg_color);
  			gdk_draw_rectangle(draw_area,text->gc,TRUE,
							x,y,pstyle[i].width,pcache->height);      
                }
  
  			gdk_gc_set_foreground(text->gc,&style->fg_color);
  			gdk_draw_text(draw_area,style->font,text->gc,
								x,y+ pcache->height- style->descent,&txt[offset],pstyle[i].len);
			
  			x+=pstyle[i].width;
  			offset+=pstyle[i].len;
  		}
  	}
  	if(linedataptr->line_number==text->line_pos_index)
  	{
  		text->cursor.x=pcache->cursor_x- text->hadj->value; 
  		text->cursor.y=y+pcache->height-pcache->cursor_h;
  		text->cursor.w=pcache->cursor_x - text->hadj->value; 
  		text->cursor.h=y+pcache->height;
  		text->cursor.hide=TRUE;
  	}
  	g_free(txt);
  }
  else if(linedataptr->line_number==text->line_pos_index)
  {
  	text->cursor.x=pcache->cursor_x- text->hadj->value; 
  	text->cursor.y=y+pcache->height-pcache->cursor_h;
  	text->cursor.w=pcache->cursor_x - text->hadj->value; 
  	text->cursor.h=y+pcache->height;
  	text->cursor.hide=TRUE;
  }

  y+=pcache->height;
  g_free(pcache);

  return y;
}

gint  
expose_line_text_wrap(GtkExText *text,GdkDrawable *draw_area,
					GtkExTextLineData *lp,
  					gint x, gint y)
{
  gint tabindex;
  gchar *txtreal,*txt;
  gint w,h,i,b;
  gint offset;
  GtkExTextStyle *style;
  InternalStyleCache *pstyle;
  InternalPaintCache *pcache;

  i=0,b=0;
  pcache=NULL;

  gdk_window_get_size(text->text_area,&w,&h);
  tabindex=0;
  txtreal=txt=text_get_text_with_expanded_tabs(text,lp->startpos,lp->endpos,&tabindex);
  if(!txtreal)	{
  	if(lp->line_number==text->line_pos_index)	{
  		text->cursor.x=0; 
  		text->cursor.y=y;
  		text->cursor.w=0; 
  		text->cursor.h=y+LINE_DATA_HEIGHT(lp);
  		text->cursor.hide=TRUE;
  	}
  	return y + LINE_DATA_HEIGHT(lp);
  }

  pcache=create_paint_cache(text,lp,0,
  							LINE_DATA_LENGTH(lp),w - LINE_WRAP_WIDTH,NULL);
  for(i=0;	i < LINE_DATA_LENGTH(lp) ; 
                        pcache=create_paint_cache(text,lp,i,LINE_DATA_LENGTH(lp)-i,w-LINE_WRAP_WIDTH,pcache) )	
  {
  	pstyle=pcache->styles;
  	if(pcache->cursor_x!=-1 && 
  					lp->line_number==text->line_pos_index)	
  	{
  		text->cursor.x=pcache->cursor_x; 
  		text->cursor.y=y+pcache->height-pcache->cursor_h;
  		text->cursor.w=pcache->cursor_x; 
  		text->cursor.h=y+pcache->height;
  		text->cursor.hide=TRUE;
  	}
  	x=0;
  	offset=0;
  	for(b=0;b<pcache->style_count;b++)	{
  		if(pstyle[b].len)
  		{
  			style=&pstyle[b].style;

                if(style->flags & STYLE_BG)  {
    			gdk_gc_set_foreground(text->gc,&style->bg_color);
    			gdk_draw_rectangle(draw_area,text->gc,TRUE,
						x,y,pstyle[b].width,pcache->height);      
                }

  			gdk_gc_set_foreground(text->gc,&style->fg_color);
  			gdk_draw_text(draw_area,style->font,text->gc,
								x,y+ pcache->height- style->descent,&txt[offset],pstyle[b].len);
			
  			x+=pstyle[b].width;
  			offset+=pstyle[b].len;
  		}
  	}
  	if(pcache->length+i < LINE_DATA_LENGTH(lp) )	{
  		gdk_gc_set_stipple (text->gc,
		      text->line_wrap_bitmap);
  
    	  gdk_gc_set_ts_origin (text->gc,
			w-line_wrap_width  -2, y );

  		gdk_gc_set_fill (text->gc, GDK_STIPPLED);
  
  		gdk_gc_set_foreground (text->gc, &GTK_WIDGET (text)->style->text[GTK_STATE_NORMAL]);

  		gdk_draw_rectangle (draw_area,
  					text->gc,TRUE,
  					  w - line_wrap_width - 2,
  						y,
  						line_wrap_width,
  					line_wrap_height);

  		gdk_gc_set_ts_origin (text->gc, 0, 0);
  
  		gdk_gc_set_fill (text->gc, GDK_SOLID);
  	}

  	i+=pcache->length;
  	txt=txtreal+i;
  	y+=pcache->height;

  }
  g_free(txtreal);
  g_free(pcache);
  
  return y;  
}

gint
gtk_extext_cursor_blink_start(gpointer data)
{
  g_return_val_if_fail (data != NULL,TRUE);
  g_return_val_if_fail (GTK_IS_EXTEXT (data),TRUE);

		GTK_EXTEXT(data)->cursor_timer=gtk_timeout_add(CURSOR_TIMEOUT,gtk_extext_cursor_blink,data);

		return FALSE;
}

gint
gtk_extext_cursor_blink(gpointer data)
{
  GdkColor *color;
  GtkExText *text;

  g_return_val_if_fail (data != NULL,TRUE);
  g_return_val_if_fail (GTK_IS_EXTEXT (data),TRUE);

  text=GTK_EXTEXT(data);

  if(!GTK_WIDGET_HAS_FOCUS(text) && text->cursor.hide)
    return TRUE;
		
  if(!text->cursor.hide)	{  
    color=&GTK_WIDGET(text)->style->text[GTK_STATE_NORMAL];
    text->cursor.hide=TRUE;
  }
  else {
    color=&GTK_WIDGET(text)->style->base[GTK_STATE_NORMAL];
    text->cursor.hide=FALSE;
  }

  gdk_gc_set_foreground(text->gc,color);
  gdk_draw_line(text->text_area,text->gc,
			text->cursor.x,text->cursor.y,
			text->cursor.w,text->cursor.h);

  return TRUE;
}

static void
gtk_extext_disconnect (GtkAdjustment *adjustment,
		     GtkExText       *text)
{
  g_return_if_fail (adjustment != NULL);
  g_return_if_fail (GTK_IS_ADJUSTMENT (adjustment));
  g_return_if_fail (text != NULL);
  g_return_if_fail (GTK_IS_EXTEXT (text));

  if (adjustment == text->hadj)
    gtk_extext_set_adjustments (text, NULL, text->vadj);
  if (adjustment == text->vadj)
    gtk_extext_set_adjustments (text, text->hadj, NULL);
}

static gint
gtk_extext_scroll_timeout (gpointer data)
{
  GtkExText *text;
  GdkEventMotion event;
  gint x, y;
  GdkModifierType mask;

  GDK_THREADS_ENTER ();

  text = GTK_EXTEXT (data);
  
  text->timer = 0;
  gdk_window_get_pointer (text->text_area, &x, &y, &mask);
  
  if (mask & (GDK_BUTTON1_MASK | GDK_BUTTON3_MASK))
    {
      event.is_hint = 0;
      event.x = x;
      event.y = y;
      event.state = mask;
      
      gtk_extext_motion_notify (GTK_WIDGET (text), &event);
    }

  GDK_THREADS_LEAVE ();
  
  return FALSE;
}

static void
gtk_extext_delete_to_line_end(GtkExText *text)
{
  GtkExTextLineData *lp;
  GtkEditable *editable;
  gint len;
  g_return_if_fail (GTK_IS_EXTEXT (text));
	
  editable=GTK_EDITABLE(text);

  lp=gtk_extext_get_line_data(text,text->line_pos_index,text->scroll_line_start);
  len=editable->current_pos - lp->startpos;
  gtk_editable_delete_text(editable,editable->current_pos,
								editable->current_pos + LINE_DATA_LENGTH(lp) - len);	

  g_free(lp);
}

static void 
gtk_extext_delete_backward_word(GtkExText *text)
{
  gint start,end;
  GtkEditable *editable;
  g_return_if_fail (GTK_IS_EXTEXT (text));
  editable=GTK_EDITABLE(text);

  start=end=editable->current_pos;
  if(!gtk_extext_get_current_word(text,&start,&end))
  	gtk_extext_get_previous_word(text,&start,&end);		

  gtk_editable_select_region(editable,start,editable->current_pos);
  gtk_editable_delete_selection(editable);
}

static void 
gtk_extext_delete_line(GtkExText *text)
{
  GtkExTextLineData *linedataptr;
  GtkEditable *editable;
  g_return_if_fail (GTK_IS_EXTEXT (text));

  editable=GTK_EDITABLE(text);
  linedataptr=gtk_extext_get_line_data(text,text->line_pos_index,text->scroll_line_start);

  gtk_editable_delete_text(editable,linedataptr->startpos,linedataptr->endpos);	

  g_free(linedataptr);
}

/*****************************************************************************/
/*                    Mousecursor TO line and linecursorpos                      */
/*                         Code 																*/
/*****************************************************************************/

gint 
gtk_extext_get_column_by_offset(GtkExText *text,GtkExTextLineData *linedataptr,gint x,gint *offsetx)
{
	GtkExTextClass *klass;
	g_return_val_if_fail(text,0);
	g_return_val_if_fail (GTK_IS_EXTEXT (text),0);

	klass=GTK_EXTEXT_CLASS(GTK_OBJECT(text)->klass);
	if(klass)
		return klass->column_by_offset(text,linedataptr,x,offsetx); 

    return 0;
}

GtkExTextLineData * 
gtk_extext_get_line_by_offset(GtkExText *text,gint y,gint *offsety)
{
  g_return_val_if_fail(text,0);
  g_return_val_if_fail (GTK_IS_EXTEXT (text),NULL);

  return GTK_EXTEXT_CLASS(GTK_OBJECT(text)->klass)->line_by_offset(text,y,offsety);
}

GtkExTextLineData * 
gtk_extext_get_line_by_char_pos(GtkExText *text,gint pos)
{
  g_return_val_if_fail (GTK_IS_EXTEXT (text),NULL);

  return line_get_by_char_pos(text,pos);
}

GtkExTextLineData *
line_get_by_offset(GtkExText *text,gint y,gint *offsety)
{
  InternalPaintCache *pcache;
  gint lnum;
  gint height=0;
  gint w,h,len,i;
  GtkExTextLineData *linedataptr=text->scroll_line_start;

  g_return_val_if_fail (text != NULL,NULL);
  g_return_val_if_fail (GTK_IS_EXTEXT (text),NULL);

  gdk_window_get_size (text->text_area, &w, &h);
  if(!text->line_wrap && !text->word_wrap)
  	w=0xFFFF;
  else
    w-=LINE_WRAP_WIDTH;

  pcache=NULL;
  *offsety=0;
  for(lnum=text->scroll_line_start->line_number;lnum <= text->line_count  ;lnum++)	{
  	i=0;
	  linedataptr=gtk_extext_get_line_data(text,lnum,linedataptr);
  	len=LINE_DATA_LENGTH(linedataptr);
  	pcache=create_paint_cache(text,linedataptr,0,LINE_DATA_LENGTH(linedataptr),w,pcache);
  	while(pcache)
  	{	
		height+=pcache->height;
		if( y <= height)	
			goto stop;

		*offsety=*offsety + height;
  		i+=pcache->length;
  		if(i < len)
  			pcache=create_paint_cache(text,linedataptr,i,LINE_DATA_LENGTH(linedataptr)-pcache->length,w,pcache);  
  		else
  			break;
  	}
  }
stop:
  if(pcache)
  	g_free(pcache);

  return linedataptr;
}

void 
find_cursor(GtkExText *text,gdouble x,gdouble y,gboolean select)
{
  GtkExTextStyle *defstyle;
  gint height;
  GtkExTextLineData *lp;

  gint cursor;
  gint tabs;
  gint tabindex;
  gint i;
  gint dummy;
				
  i=0;
  tabindex=0;
  tabs=0;		
  defstyle=gtk_extext_style_get(text,"Default");
  height=0;
  lp=NULL;

	/* Oki lets see were the fucking cursor is */

  cursor=-1;

  if(y<0){	
     lp=gtk_extext_get_line_data(text,text->scroll_line_start->line_number-1,text->scroll_line_start);
      
     goto_line(text,text->scroll_line_start->line_number - 1,select,TRUE);
	text->timer = gtk_timeout_add (SCROLL_TIME, 
					 gtk_extext_scroll_timeout,
					 text);
  }
  else 	{
      lp=gtk_extext_get_line_by_offset(text,y,&dummy);
      if(lp->line_number > text->scroll_line_start->line_number + text->scroll_line_count)	{
              text->timer = gtk_timeout_add (SCROLL_TIME, 
					 gtk_extext_scroll_timeout,text);
   }
   goto_line(text,lp->line_number,select,TRUE);

}
	
 lp=gtk_extext_get_line_data(text,text->line_pos_index,lp);
 cursor=gtk_extext_get_column_by_offset(text,lp,x,&dummy);
 if(cursor == -1 && lp->line_number==text->line_count)
    move_cursor(text, LINE_DATA_LENGTH(lp) ,select);
 else if(cursor == -1)
    move_cursor(text, LINE_DATA_LENGTH(lp) - 1 ,select);
 else
    move_cursor(text,cursor,select);

  g_free(lp);  
  gtk_widget_queue_draw(GTK_WIDGET(text));
}

gint 
column_get_by_offset(GtkExText *text,GtkExTextLineData *linedataptr,gint x,gint *newx)
{

#ifdef EXPRIMENTAL
  InternalPaintCache *pcache;
  gint w,h,cursor,len;

  g_return_val_if_fail(text,0);
  g_return_val_if_fail(GTK_IS_EXTEXT(text),0);

  gdk_window_get_size (text->text_area, &w, &h);
  cursor=0;
  if(text->line_wrap) 
  pcache=create_paint_cache();
  len=LINE_DATA_LENGTH(linedataptr);
#endif
  GtkExTextStyle *style;
  GtkExTextProperty *prop;
  GtkExTextStyle *defstyle;
  GtkExTextProperty *new;

  gchar chars[24];
  gint width,curwidth;
  gint cursor;
  guchar key;
  gint istab;
  gint tabindex;
  gint i;
  gint pos;

  g_return_val_if_fail(text,0);
  g_return_val_if_fail(GTK_IS_EXTEXT(text),0);

  cursor=-1;				
  i=0;
  defstyle=gtk_extext_style_get(text,"Default");

  prop=linedataptr->property_first;

	/* Oki we no the line but its little more tricky to now were the cursor is*/

  tabindex=0;
  width=0;
  istab=0;
  pos=linedataptr->startpos;
  x+=text->hadj->value;
  for(i=0; i < LINE_DATA_LENGTH(linedataptr);i++)	{
    new=gtk_extext_property_get_at_pos(text,pos,prop);
    if(new)	{
      prop=new;
      style=prop->style;
    }
    else
      style=defstyle;

    key=GTK_EXTEXT_INDEX(text,pos);
    if(key=='\t')	{		
      istab=gtk_extext_get_tab_width(text,tabindex);
      memset(chars,'\t',istab);
      chars[istab]='\0';  		
      istab--;
      tabindex++;
    }
    else
    {	chars[0]=key;	chars[1]='\0';	}

    curwidth=gdk_text_width(style->font,chars,1+istab);
    width+=curwidth;
    if(x+(gfloat)(curwidth*0.50) < (float)width)	{
      cursor=pos-linedataptr->startpos;
      break;
    }

    istab=0;
    pos++;
  }

  return cursor;
}

void 
gtk_extext_draw(GtkWidget *widget,GdkRectangle *area)
{
  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_EXTEXT (widget));
  g_return_if_fail (area != NULL);
 
  if (GTK_WIDGET_DRAWABLE (widget))
    {
      expose_text (GTK_EXTEXT (widget), area, TRUE);
 /*     gtk_widget_draw_focus (widget);*/
    }

}
/*****************************************************************************/
/*                              extext_key_press                                */
/*****************************************************************************/

static gint  
gtk_extext_key_press(GtkWidget *widget,GdkEventKey *event)
{
  GtkExText *text;
  GtkEditable *editable;
  gboolean shift_state;
  gboolean control_state;
  gboolean paint;
  gboolean return_value;
  gint key;
  gint curpos;  
  
  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_EXTEXT (widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);

  text=GTK_EXTEXT(widget);
  editable=GTK_EDITABLE(widget);
  paint=FALSE;


  curpos=editable->current_pos;

  return_value=FALSE;
  shift_state=event->state & GDK_SHIFT_MASK;
  control_state=event->state & GDK_CONTROL_MASK;      
  
  if(event->state & GDK_MOD1_MASK) 
    return FALSE;
	
  key=event->keyval;
  if (event->state & GDK_CONTROL_MASK)
  {
    if ((key >= 'A') && (key <= 'Z'))
      key -= 'A' - 'a';

    if(!text->read_only) 
    {	      
      if ((key >= 'a') && (key <= 'z') && control_keys[(int) (key - 'a')])
       {
          (* control_keys[(int) (key - 'a')]) (editable, event->time);
			return_value = TRUE;
			paint=TRUE;
	 }
    }
  }
  if(!return_value && !(event->state & GDK_MOD1_MASK)){
    return_value=TRUE;
    paint=TRUE;
    switch(event->keyval)
    {
      case GDK_Escape:
        return_value=FALSE;
        paint=TRUE;
      break;
      case GDK_Home:
        if(control_state)
          goto_line(text,0,shift_state,TRUE);
        else  {
          if(!text->line_cursor_index && shift_state)
            goto_line(text,text->line_pos_index-1,shift_state,TRUE);
          else {          
            line_set_cursor_pos(text,0);
            if(shift_state)
              update_select(text,editable->selection_start_pos,TRUE,FALSE);				
          }
        }
      break;
      case GDK_End:
        if(control_state)        
          goto_line(text,text->line_count,shift_state,TRUE);
        else  {
          line_set_cursor_pos(text, shift_state ? text->line_pos->length : text->line_pos->length-1);
          if(shift_state) {
            editable->selection_end_pos=editable->current_pos;
            if(editable->selection_start_pos==-1)
              editable->selection_start_pos=curpos;
          }
        }	
      break;
      case GDK_Page_Up:
        goto_line(text,text->line_pos_index 
									- text->scroll_line_count,shift_state,TRUE);
      break;
      case GDK_Page_Down:
        goto_line(text,text->line_pos_index
              + text->scroll_line_count,shift_state,TRUE);
      break;
      case GDK_Up:
        goto_line(text,text->line_pos_index-1,shift_state,TRUE);
      break;
      case GDK_Down:
        goto_line(text,text->line_pos_index+1,shift_state,TRUE);
      break;
      case GDK_Left:
         if(control_state) {
            gint oldselstart=editable->selection_start_pos;
            gint oldselend=editable->selection_end_pos;
            gint pos=0;
            gint start=curpos;
            gint end;
            if(gtk_extext_get_current_word(text,&start,&end) && start!=curpos)
               pos=start;
            else if(gtk_extext_get_previous_word(text,&start,&end))
               pos=start;
            else 
               pos=0;

            gtk_editable_set_position(GTK_EDITABLE(text),pos);

            if(shift_state)  {            
              editable->selection_start_pos=oldselstart;
              editable->selection_end_pos=oldselend !=-1 ? oldselend : curpos;
              update_select(text,pos,TRUE,FALSE);
            }
         }
         else  {
		  move_cursor(text, text->line_cursor_index  -1,shift_state );
            paint=TRUE;
        }
      break;
      case GDK_Right:
         if(control_state) {
            gint oldselstart=editable->selection_start_pos;
            gint pos=text->length;
            gint start=GTK_EDITABLE(text)->current_pos;
            gint end;
            if(gtk_extext_get_current_word(text,&start,&end))
               pos=end;
            else if(gtk_extext_get_next_word(text,&start,&end))
               pos=end;

            gtk_editable_set_position(GTK_EDITABLE(text),pos);
            if(shift_state)  {            
              editable->selection_start_pos=oldselstart != -1 ? oldselstart : curpos;
              editable->selection_end_pos=pos;
            }
         }
         else {
            move_cursor(text, text->line_cursor_index +1 ,shift_state);
            paint=TRUE;
          }
      break;
      default:
        if(text->read_only || control_state)
          return_value=FALSE;
        else if(!text->read_only)
        {
          switch(event->keyval)
		{
            case GDK_Return:
              gtk_editable_insert_text(editable,"\n",1,&curpos);
            break;
            case  GDK_KP_Enter:
              gtk_editable_insert_text(editable,"\n",1,&curpos);
            break;
            case GDK_BackSpace:
						gtk_editable_delete_text(GTK_EDITABLE(text),
							GTK_EDITABLE(text)->current_pos-1,
							GTK_EDITABLE(text)->current_pos);
            break;
            case GDK_Delete:
				 gtk_editable_delete_text(GTK_EDITABLE(text),
                            GTK_EDITABLE(text)->current_pos,
                        GTK_EDITABLE(text)->current_pos+1);
            break;
            case GDK_Tab:
              gtk_editable_insert_text(editable,"\t",1,&curpos);
            break;
            default:
		  if(event->length)										
               gtk_editable_insert_text(editable,(guchar*)event->string,event->length,&curpos);
            else 
              return_value=FALSE; /* let the parentwidget take care */
            break;
        };
      }
      break;
    }
  };
  if(paint)
    gtk_widget_queue_draw(widget);

  return return_value;
}

static void
gtk_extext_adjustment_changed (GtkAdjustment *adjustment,
		     GtkExText       *text)
{
  g_return_if_fail (adjustment != NULL);
  g_return_if_fail (GTK_IS_ADJUSTMENT (adjustment));
  g_return_if_fail (text != NULL);
  g_return_if_fail (GTK_IS_EXTEXT (text));
  
  /* Just ignore it if we haven't been size-allocated and realized yet */
  if (!GTK_WIDGET_REALIZED (text))
		return;

  if (adjustment == text->hadj)
    {
    }
  else if(adjustment==text->vadj)
    {
      recompute_scroll_line_height(text);
  	text->vadj->upper=(float)text->line_count;	
    }
}

static void 
gtk_extext_adjustment_value_changed (GtkAdjustment *adjustment,
		     GtkExText       *text)
{
	GtkExTextLineData *lp;
	gint firstline;
	gint curline;

  g_return_if_fail (text != NULL);
  g_return_if_fail (GTK_IS_EXTEXT (text));


	if(adjustment==text->vadj)	{
		firstline=(gint)text->vadj->value;
		curline=text->line_pos_index;
	
		lp=gtk_extext_get_line_data(text,firstline,text->scroll_line_start);
		text->scroll_line_start=lp;

		if(curline<text->scroll_line_start->line_number)	{
			goto_line(text,firstline,FALSE,FALSE);
		}
		/* if out o
f viewport */
		else if(curline >= text->scroll_line_start->line_number + text->scroll_line_count )	{
						goto_line(text,
									text->scroll_line_start->line_number + text->scroll_line_count-1,FALSE,FALSE);

		}
	}
	else if(adjustment==text->hadj)	{

	}

	
	gtk_widget_queue_draw(GTK_WIDGET(text));
}

static gint
gtk_extext_focus_in (GtkWidget     *widget,
		   GdkEventFocus *event)
{
  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_EXTEXT (widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);
  
  
  GTK_WIDGET_SET_FLAGS (widget, GTK_HAS_FOCUS);
  gtk_widget_draw_focus (widget);
      
  return FALSE;
}

static gint
gtk_extext_focus_out (GtkWidget     *widget,
		    GdkEventFocus *event)
{
  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_EXTEXT (widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);
  
  
  GTK_WIDGET_UNSET_FLAGS (widget, GTK_HAS_FOCUS);
  gtk_widget_draw_focus (widget);
  gtk_widget_queue_draw(widget);
    
  return FALSE;
}

static gint
gtk_extext_motion_notify (GtkWidget      *widget,
			GdkEventMotion *event)
{
  GtkExText *text;
  gint x, y;
  gint height;
  GdkModifierType mask;
  
  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_EXTEXT (widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);
  
  text = GTK_EXTEXT (widget);
  
  x = event->x;
  y = event->y;
  mask = event->state;
  if (event->is_hint || (text->text_area != event->window))
    {
      gdk_window_get_pointer (text->text_area, &x, &y, &mask);
    }
  
  if ((text->button == 0) ||
      !(mask & (GDK_BUTTON1_MASK | GDK_BUTTON3_MASK)))
    return FALSE;
  
  gdk_window_get_size (text->text_area, NULL, &height);
  
 	find_cursor (GTK_EXTEXT (widget), (float)x,(float) y,TRUE);

  
  return FALSE;
}

static gint
gtk_extext_button_press (GtkWidget      *widget,
		       GdkEventButton *event)
{
  gint start,end;
  GtkExText *text;
  GtkExTextLineData *linedataptr;
  GtkEditable *editable;
  static GdkAtom ctext_atom = GDK_NONE;


  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_EXTEXT (widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);
 

   if (ctext_atom == GDK_NONE)
  	ctext_atom = gdk_atom_intern ("COMPOUND_TEXT", FALSE);
  

  text = GTK_EXTEXT (widget);
  editable = GTK_EDITABLE (widget);
  
  if (text->button && (event->button != text->button))
    return FALSE;
  
  linedataptr=gtk_extext_get_line_data(text,text->line_pos_index,text->scroll_line_start);
  text->button = event->button;
  
  if (!GTK_WIDGET_HAS_FOCUS (widget))
    gtk_widget_grab_focus (widget);
  
  if (event->button == 1)
    {
      switch (event->type)
	{
	case GDK_BUTTON_PRESS:
	  gtk_grab_add (widget);
	  
	  
	  /* Set it now, so we display things right. We'll unset it
	   * later if things don't work out */
	  editable->has_selection = TRUE;
	find_cursor(text,event->x,event->y,FALSE);
 
	  break;
	  
	case GDK_2BUTTON_PRESS:
		start=end=editable->current_pos;
		if(!gtk_extext_get_current_word(text,&start,&end))
			gtk_extext_get_previous_word(text,&start,&end);		

		gtk_editable_select_region(editable,start,end);
	break;
	  
	case GDK_3BUTTON_PRESS:
		gtk_editable_select_region(editable,linedataptr->startpos,linedataptr->endpos);
	 break;
	  
	default:
	  break;
	}
    }
  else if (event->type == GDK_BUTTON_PRESS)
    {
      if ((event->button == 2) && editable->editable)
	{
	  if (editable->selection_start_pos == editable->selection_end_pos ||
	      editable->has_selection)
	    {
	      	g_print("do something\n");
	    }
	  
	  gtk_selection_convert (widget, GDK_SELECTION_PRIMARY,
				 ctext_atom, event->time);
	}
      else
	{
	  gtk_grab_add (widget);
	  
	  find_cursor (text, event->x, event->y,FALSE);
	  
	  
	  editable->has_selection = FALSE;
	  if (gdk_selection_owner_get (GDK_SELECTION_PRIMARY) == widget->window)
	    gtk_selection_owner_set (NULL, GDK_SELECTION_PRIMARY, event->time);
	}
    }
  
  g_free(linedataptr);

  return FALSE;	
}

static gint
gtk_extext_button_release (GtkWidget      *widget,
			 GdkEventButton *event)
{
  GtkExText *text;
  GtkEditable *editable;
  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_EXTEXT (widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);
  
  text = GTK_EXTEXT (widget);
  editable = GTK_EDITABLE (widget);

  gtk_grab_remove (widget);

  if (text->button != event->button)
    return FALSE;
 
  text->button = 0;
  
  if (text->timer)
    {
      gtk_timeout_remove (text->timer);
      text->timer = 0;
    }
  
   if (event->button == 1)
    {
      
      gtk_grab_remove (widget);
      
      editable->has_selection = FALSE;
      if (editable->selection_start_pos != editable->selection_end_pos)
	{
	  if (gtk_selection_owner_set (widget,
				       GDK_SELECTION_PRIMARY,
				       event->time))
	    editable->has_selection = TRUE;
	  else
		gtk_widget_queue_draw(widget);
	}
      else
	{
	  if (gdk_selection_owner_get (GDK_SELECTION_PRIMARY) == widget->window)
	    gtk_selection_owner_set (NULL, GDK_SELECTION_PRIMARY, event->time);
	}
    }
  else if (event->button == 3)
    {
      gtk_grab_remove (widget);
    }

	return FALSE;
}
/* Make sure scrolladjustment is up to date */
void 
scroll_to_view(GtkExText *text,gint moveto)
{
  GtkExTextLineData *lp;
  if(text->scroll_line_start->line_number + text->scroll_line_count <= moveto)	{
 /* 	text->vadj->value=(gfloat)text->scroll_line_start->line_number+1;*/
  	lp=gtk_extext_get_line_data(text,moveto,text->scroll_line_start);
  	g_free(text->scroll_line_start);
       text->scroll_line_start=lp;
  	text->vadj->value=(gfloat)moveto;
  	gtk_adjustment_changed(text->vadj);
  }
  else if(text->scroll_line_start->line_number > moveto)	{
  	lp=gtk_extext_get_line_data(text,moveto,text->scroll_line_start);
  	g_free(text->scroll_line_start);
  	text->scroll_line_start=lp;
  	text->vadj->value=(gfloat)moveto;
  	gtk_adjustment_changed(text->vadj);
  }
}

Lines *
goto_line(GtkExText *text,gint posindex,gboolean select,gboolean killselect)
{
  gint curpos,oldpos,len;
  Lines *pos;

  oldpos=GTK_EDITABLE(text)->current_pos;
  curpos=text->line_cursor_index;
  pos=line_set(text,posindex);
  len=pos->length;
  if(curpos>len)	
  	line_set_cursor_pos(text,len ? len - 1 : 0);
  else
  	line_set_cursor_pos(text,curpos);

  scroll_to_view(text,text->line_pos_index);
  update_select(text,oldpos,select,killselect);
  recompute_scroll_line_width(text,text->scroll_line_start,text->scroll_line_count);
	
  return text->line_pos;
}

gint 
move_cursor(GtkExText *text,gint moveto,gboolean select)
{
	/* if move cursor we should always kill selection if select is false */ 
	gint oldpos;

	oldpos=GTK_EDITABLE(text)->current_pos;
	line_set_cursor_pos(text, moveto);

	scroll_to_view(text,text->line_pos_index);

	update_select(text,oldpos,select,TRUE);	

	return GTK_EDITABLE(text)->current_pos;
}

static void 
update_select(GtkExText *text,gint oldpos,gboolean select,gboolean killselect)
{
  g_return_if_fail (text != NULL);
  g_return_if_fail (GTK_IS_EXTEXT (text));

  if(select)	
  { 
	/* FORWARD SELECT */
  	if(oldpos<GTK_EDITABLE(text)->current_pos)
  	{
   		if(GTK_EDITABLE(text)->selection_start_pos==-1  )	
  		{	
  			GTK_EDITABLE(text)->selection_start_pos=oldpos;
  			GTK_EDITABLE(text)->selection_end_pos=GTK_EDITABLE(text)->current_pos;	
  		} 	
  		else if(GTK_EDITABLE(text)->selection_start_pos==oldpos)
  			GTK_EDITABLE(text)->selection_start_pos=GTK_EDITABLE(text)->current_pos;
  		else 
  			GTK_EDITABLE(text)->selection_end_pos=GTK_EDITABLE(text)->current_pos;
  	}
  	else 
  	{
  		if(GTK_EDITABLE(text)->selection_end_pos==-1)
  		{
  			GTK_EDITABLE(text)->selection_end_pos=oldpos;
  			GTK_EDITABLE(text)->selection_start_pos=GTK_EDITABLE(text)->current_pos;
  		}
  		else if(GTK_EDITABLE(text)->selection_end_pos==oldpos)
  			GTK_EDITABLE(text)->selection_end_pos=GTK_EDITABLE(text)->current_pos;
  		else 
  			GTK_EDITABLE(text)->selection_start_pos=GTK_EDITABLE(text)->current_pos;	
  	}
  	/* this is to make sure start end is not swaped */
  	if(GTK_EDITABLE(text)->selection_end_pos<GTK_EDITABLE(text)->selection_start_pos)
  	{
  		gint swap=GTK_EDITABLE(text)->selection_start_pos;
  		GTK_EDITABLE(text)->selection_start_pos=GTK_EDITABLE(text)->selection_end_pos;
  		GTK_EDITABLE(text)->selection_end_pos=swap;
  	}
  }
  else if(killselect)
  {	
  	GTK_EDITABLE(text)->selection_start_pos=-1;
  	GTK_EDITABLE(text)->selection_end_pos=-1;
  }
}

static gint
gtk_extext_expose(GtkWidget *widget,GdkEventExpose *event)
{
  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_EXTEXT (widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);
 
  if(event->window==GTK_EXTEXT(widget)->text_area)	
	expose_text(GTK_EXTEXT(widget),&event->area ,TRUE);
  else if(event->count == 0)
      gtk_widget_draw_focus (widget);

  return FALSE;
}

GtkType 
gtk_extext_get_type(void)
{
	static GtkType extext_type=0;

	if(!extext_type)
	{
		static const GtkTypeInfo extext_info=	
		{	
			"GtkExText",
			sizeof(GtkExText),
			sizeof(GtkExTextClass),
			(GtkClassInitFunc) gtk_extext_class_init,
			(GtkObjectInitFunc) gtk_extext_init,
			NULL,
			NULL,
			(GtkClassInitFunc)NULL,
		};
		extext_type=gtk_type_unique(GTK_TYPE_EDITABLE,&extext_info);
	}
	return extext_type;
}

/*****************************************************************************/
/*                               Public USER API                                 */
/*****************************************************************************/

void
gtk_extext_set_line_max_chars(GtkExText *text, gint c)
{
  g_return_if_fail (text != NULL);
  g_return_if_fail (GTK_IS_EXTEXT (text));

  text->line_max_chars=c;
}

void
gtk_extext_set_word_wrap(GtkExText *text,gboolean set)
{
  g_return_if_fail (text != NULL);
  g_return_if_fail (GTK_IS_EXTEXT (text));

  text->word_wrap=set;  
}

void
gtk_extext_set_line_wrap(GtkExText *text,gboolean set)
{
  GtkExTextLineData *lp;
  g_return_if_fail (text != NULL);
  g_return_if_fail (GTK_IS_EXTEXT (text));


  text->line_wrap=set;  
  if(GTK_WIDGET_REALIZED(text))  {
    /* TODO cleanup this mess this is hell to slow :-((( */
    lp=gtk_extext_get_line_data(text,0,NULL);
    recompute_geometry(text);
    recompute_scroll_line_width(text,lp,text->line_count);
    g_free(lp);
  }
}

void 	
gtk_extext_insert(GtkExText *text,char *chars,gint length)
{
  g_return_if_fail (text != NULL);
  g_return_if_fail (GTK_IS_EXTEXT (text));

  gtk_extext_insert_with_style(text,chars,length,NULL,NULL);
}

void       
gtk_extext_insert_with_style(GtkExText       *text, char    *chars,
				     gint length,gchar *stylekey, gpointer userdata)
{
  gint pos;
  g_return_if_fail (text != NULL);
  g_return_if_fail (GTK_IS_EXTEXT (text));

	pos=GTK_EDITABLE(text)->current_pos;
	if(length==-1)
		length=strlen(chars);

	gtk_editable_insert_text(GTK_EDITABLE(text),chars,length,&pos);

	if(stylekey)
		gtk_extext_property_insert(text,stylekey,pos - length,pos,userdata,PROPERTY_MERGE,text->scroll_line_start->property_first);
}

/* this routine will be removed use create_cache_insteadf */

static gint 
extext_get_line_width(GtkExText *text,GtkExTextLineData *linedataptr,gint pos)
{
  gint width;
  InternalPaintCache *pcache;
  
  pcache=create_paint_cache(text,linedataptr,0,pos,0xFFFF,NULL);
  width=pcache->width;
  g_free(pcache);

  return width;  
#ifdef OLDCRAP
  gchar *txt;
  gint cursor_width;
  gint cursortabcount;
  gint i,pindex;
  gint tabindex,tab;	
  GtkExTextStyle *style,*oldstyle; 
  gint height;
  gint cursor_height;
  gint offset;
  GtkExTextProperty *prop; 
  GtkExTextProperty *oldprop; 
  gint startpos,endpos;
		
	/* hopefully there is not more than 64 propertys in one line  :-( */
	/* if there is the text will not appear */
	/* This is a very dirty hack */
	/* maybe we should do it better in the feature */

  InternalStyleCache pstyle[64];
  g_return_val_if_fail (text != NULL, 0);
  g_return_val_if_fail (GTK_IS_EXTEXT (text), 0);


  if (!GTK_WIDGET_REALIZED (text))
  	return 0;

  startpos=linedataptr->startpos;
  endpos=linedataptr->startpos+pos;

  oldprop=linedataptr->property_first;
  prop=NULL;
  offset=0;
  tabindex=0;
  tab=0;          				/* is there a tab this is set to [tabindex].tab */
  pindex=0;				/* pstyle index */
  style=NULL;						/* Current style */
  oldstyle=NULL;
  cursortabcount=0;				/* tabs before cursor pos */ 
		
	/* Cache 0 is ALWAYS Default */
  
	style=oldstyle=gtk_extext_style_get(text,"Default");		
	memcpy(&pstyle[0].style,style,sizeof(GtkExTextStyle));
	pstyle[0].len=0;

	txt=text_get_text(text,startpos,endpos);
	if(!txt)	
		return 0;
		
	height=LINE_DATA_HEIGHT(linedataptr);
	for( i=startpos;i<endpos;i++)
	{				
		if(pindex>64)
		{
			g_warning("WHAT THE FUCK property line overflo\n");
				break;
		}

		tab=0;	
		if( GTK_EXTEXT_INDEX(text,i)=='\t')	{	
			tab=gtk_extext_get_tab_width(text,tabindex);
			tab--;

  		if(linedataptr->line_number==text->line_pos_index 
					&& i - startpos<text->line_cursor_index 
								&& text->line_cursor_index)					
				cursortabcount+=tab;
				tabindex++;
		}
		prop=gtk_extext_property_get_at_pos(text,i,oldprop);
		if(prop)	{
			style=prop->style;				
			oldprop=prop;
		}
		else
			style=&pstyle[0].style; 


		/* Init new style */
		if(style!=oldstyle )
		{
			pindex++;
			memcpy(&pstyle[pindex].style,style,sizeof(GtkExTextStyle));
			pstyle[pindex].len=tab+1;

			oldstyle=style;
		}
		else	{
			pstyle[pindex].len+=tab+1;				
		}
	}	
		
	
	g_free(txt);
	
	tabindex=0;
	txt=text_get_text_with_expanded_tabs(text,startpos,endpos,&tabindex);

	cursor_width=0;
	if(txt)	{
		gint cursoroffset=pos+cursortabcount;
		offset=0;
		for(i=0;i<=pindex;i++)	{
			if(pstyle[i].len)
			{
				gint width;
				GtkExTextStyle *style;
				gint descent;

				style=&pstyle[i].style;

				descent=style->descent;
				width=gdk_text_width(style->font,
								&txt[offset],pstyle[i].len);

				if(cursoroffset && cursoroffset<pstyle[i].len)	{
					cursor_width+=gdk_text_width(style->font,
							&txt[offset],cursoroffset);
					cursoroffset=0;
				}
				else	{
					cursor_width+=gdk_text_width(style->font,
							&txt[offset],pstyle[i].len);

					cursoroffset-=pstyle[i].len;
				}
			}
			offset+=pstyle[i].len;
		}
		g_free(txt);
	}
	return cursor_width;
#endif
}

GtkExTextLineData *
gtk_extext_get_first_visible_line(GtkExText *text)
{
	GtkExTextLineData *cur;
     g_return_val_if_fail (text != NULL, NULL);
     g_return_val_if_fail (GTK_IS_EXTEXT (text), NULL);

	cur=g_malloc(sizeof(GtkExTextLineData));
	memcpy(cur,text->scroll_line_start,sizeof(GtkExTextLineData));

	return cur;	
}

GtkExTextLineData *
gtk_extext_get_line_init(GtkExText *text,gint line)
{
  GtkExTextLineData *cur;
  g_return_val_if_fail (text != NULL, NULL);
  g_return_val_if_fail (GTK_IS_EXTEXT (text), NULL);


  cur=g_malloc0(sizeof(GtkExTextLineData));
  if(line <= (gint) (text->line_count/2 ))	{
  	cur->lineptr=text->line_start;	
  	cur->endpos=LINE_DATA_LENGTH(cur);
  }
  else	{	
  	cur->lineptr=text->line_end;	
  	cur->startpos=text->length - LINE_DATA_LENGTH(cur);
  	cur->endpos=text->length;
  	cur->line_number=text->line_count;
  }

  return cur;
}

GtkExTextLineData *
gtk_extext_get_line_data(GtkExText *text, gint newline, GtkExTextLineData *cur)
{
     g_return_val_if_fail (text != NULL, NULL);
     g_return_val_if_fail (GTK_IS_EXTEXT (text), NULL);

	if(cur && cur==text->scroll_line_start)	{
		cur=g_malloc(sizeof(GtkExTextLineData));
		memcpy(cur,text->scroll_line_start,sizeof(GtkExTextLineData));
	}
	else if(cur && !cur->lineptr)	{
		memset(cur,0,sizeof(GtkExTextLineData));
		cur->lineptr=text->line_start;
		cur->endpos=LINE_DATA_LENGTH(cur);
	} else if(!cur) {
        cur=g_malloc0(sizeof(GtkExTextLineData));
	  cur=gtk_extext_get_line_init(text,newline);
	}

    /* optimize start from begin/pos/end */
     /*and make sure user dont request line out of range */
    /* if request minus 0 set to first line */
    /* if request more than last line set to endline */
	if(newline <= 0 || (newline < cur->line_number &&  cur->line_number - newline  > (text->line_count/2)) )	{
		cur->line_number=0;
		cur->lineptr=text->line_start;
		cur->startpos=0;
		cur->endpos=LINE_DATA_LENGTH(cur);
		cur->property_first=NULL;
	}
	else if(newline >= text->line_count || (newline > cur->line_number && newline - cur->line_number  > (text->line_count/2)))	{
		cur->line_number=text->line_count;
		cur->lineptr=text->line_end;		
		cur->startpos=text->length - LINE_DATA_LENGTH(cur);
		cur->endpos=text->length;
		cur->property_first=NULL;
	}

    /* make sure dont loop if user request out range */
     if(newline > cur->line_number && newline < text->line_count) {	/*forweard */	
		while(cur->lineptr)	{
			if(newline==cur->line_number)
				break;

			cur->startpos+=LINE_DATA_LENGTH(cur);
			cur->line_number++;

        	cur->property_first=gtk_extext_property_nearest_backward(text,cur->startpos, cur->property_first);
/*	        if(!cur->property_first)
		    LINE_DATA_HEIGHT(cur)=TEXT_STYLE_HEIGHT(gtk_extext_style_get(text,"Default"));
	        else
		     LINE_DATA_HEIGHT(cur)=line_get_height(text,cur->startpos,cur->endpos,cur->property_first);
*/
			cur->lineptr=LINE_DATA_NEXT(cur);			
			cur->endpos+=LINE_DATA_LENGTH(cur);

		};
	}	
	else if(newline < cur->line_number && newline >= 0)	{	/* backward */
		while(cur->lineptr)	{
			if(newline==cur->line_number)
				break;

			cur->endpos-=LINE_DATA_LENGTH(cur);
			cur->line_number--;
        	cur->property_first=gtk_extext_property_nearest_backward(text,cur->startpos, cur->property_first);
/*	        if(!cur->property_first)
		    LINE_DATA_HEIGHT(cur)=TEXT_STYLE_HEIGHT(gtk_extext_style_get(text,"Default"));
	        else
		     LINE_DATA_HEIGHT(cur)=line_get_height(text,cur->startpos,cur->endpos,cur->property_first);
*/
			cur->lineptr=LINE_DATA_PREV(cur);			
			cur->startpos-=LINE_DATA_LENGTH(cur);
		};
	}		

	cur->property_first=gtk_extext_property_nearest_backward(text,cur->startpos, cur->property_first);
	if(!cur->property_first)
		LINE_DATA_HEIGHT(cur)=TEXT_STYLE_HEIGHT(gtk_extext_style_get(text,"Default"));
	else
		LINE_DATA_HEIGHT(cur)=line_get_height(text,cur->startpos,cur->endpos,cur->property_first);

	if(!cur->lineptr)
		g_warning("WHAT THE FUCK THIS SHOULD NEVER HAPPEN\n");

	/* user MUST free structure as soon it's not used anymore */
	return cur;
}

void
gtk_extext_line_set_userdata(GtkExText *text,gint line,gpointer userdata)
{
  GtkExTextLineData linedata;
  g_return_if_fail (text != NULL);
  g_return_if_fail (GTK_IS_EXTEXT (text));
  g_return_if_fail(line > 0);
  g_return_if_fail(line <= text->line_count );

  /* if  we use the stack we MUST init lineptr to NULL else if will segfault */
  /* so the get_line_data init the values */

  linedata.lineptr=NULL;
  gtk_extext_get_line_data(text,line,&linedata);	
  linedata.lineptr->user_data=userdata;
}

gint
gtk_extext_set_line(GtkExText *text,gint line)
{
  g_return_val_if_fail (text != NULL,-1);
  g_return_val_if_fail (GTK_IS_EXTEXT (text),-1);

  goto_line(text,line,FALSE,FALSE);
  gtk_widget_queue_draw(GTK_WIDGET(text));

  return text->line_pos_index;
}

gint
gtk_extext_set_column(GtkExText *text,gint col)
{
  g_return_val_if_fail (text != NULL,-1);
  g_return_val_if_fail (GTK_IS_EXTEXT (text),-1);

  col= line_set_cursor_pos(text,col);
  gtk_widget_queue_draw(GTK_WIDGET(text));

  return col;
}

gint
gtk_extext_get_line(GtkExText *text)
{
  g_return_val_if_fail (text != NULL,-1);
  g_return_val_if_fail (GTK_IS_EXTEXT (text),-1);

  return text->line_pos_index;
}

gint
gtk_extext_get_column(GtkExText *text)
{
  g_return_val_if_fail (text != NULL,-1);
  g_return_val_if_fail (GTK_IS_EXTEXT (text),-1);

  return text->line_cursor_index;
}

void
gtk_extext_freeze(GtkExText *text)
{
  g_return_if_fail (text != NULL);
  g_return_if_fail (GTK_IS_EXTEXT (text));

  text->freeze_count++;	
}

void 
gtk_extext_thaw(GtkExText *text)
{
  g_return_if_fail (text != NULL);
  g_return_if_fail (GTK_IS_EXTEXT (text));

  text->freeze_count=0;
  goto_line(text,text->line_pos_index,FALSE,TRUE);
  gtk_widget_queue_draw(GTK_WIDGET(text));
}

void 
gtk_extext_undo_set_max(GtkExText *text,gint max)
{
  g_return_if_fail (text != NULL);
  g_return_if_fail (GTK_IS_EXTEXT (text));

  text->undo_max=max;
  gtk_extext_undo_clear_all(text);
}

gboolean 
gtk_extext_undo_is_empty(GtkExText *text)
{
  g_return_val_if_fail (text != NULL,FALSE);
  g_return_val_if_fail (GTK_IS_EXTEXT (text),FALSE );

  return text->undo_buffert ? FALSE : TRUE ;
}

gboolean 
gtk_extext_redo_is_empty(GtkExText *text)
{
  g_return_val_if_fail (text != NULL,FALSE );
  g_return_val_if_fail (GTK_IS_EXTEXT (text), FALSE);

  return text->undo_buffert->next ? FALSE : TRUE ;
}

void
gtk_extext_undo_clear_all(GtkExText *text)
{
  g_return_if_fail (text != NULL);
  g_return_if_fail (GTK_IS_EXTEXT (text));
 
  if(!text->undo_buffert)
	return ; 
 
  g_slist_foreach(text->undo_buffert,extext_undo_foreach,NULL);
  g_slist_free(text->undo_buffert);
  text->undo_buffert=NULL;
  text->undo_count=0;
  text->undo_last=0;

  if (GTK_WIDGET_REALIZED (text))
	  gtk_signal_emit(GTK_OBJECT(text),extext_signals[UNDO_CHANGED]);
}

void
gtk_extext_undo(GtkExText *text)
{
  GSList *cur;
  UndoBuffert *undo;
  g_return_if_fail (text != NULL);
  g_return_if_fail (GTK_IS_EXTEXT (text));
  if(!text->undo_buffert)	
	return ;

  text->undo_flag=TRUE;
  cur=g_slist_nth(text->undo_buffert,0);    
  undo=(UndoBuffert *)cur->data;	
  switch(undo->type) {
	case EXTEXT_UNDO_INSERT:
		gtk_editable_delete_text(GTK_EDITABLE(text),undo->startpos,undo->endpos);
	break;
	case EXTEXT_UNDO_REMOVE:
		gtk_editable_insert_text(GTK_EDITABLE(text), undo->buffert,
												undo->endpos - undo->startpos,&undo->startpos);
	break;
  };
 
  text->undo_flag=FALSE;

  if(undo->buffert)
     g_free(undo->buffert);

  g_free(undo);

  text->undo_buffert=g_slist_remove_link(text->undo_buffert,cur); 
  text->undo_last--;
  text->undo_count--;
/* the last element */
  if(!text->undo_buffert)  
	gtk_signal_emit(GTK_OBJECT(text),extext_signals[UNDO_CHANGED]);
}

void 
gtk_extext_redo(GtkExText *text)
{
  g_return_if_fail (text != NULL);
  g_return_if_fail (GTK_IS_EXTEXT (text));
	
}

gint 
gtk_extext_get_length(GtkExText *text)
{
  g_return_val_if_fail (GTK_IS_EXTEXT (text), FALSE);
 
  return text->length;	
}

gboolean 
gtk_extext_get_editable(GtkExText *text)
{
  g_return_val_if_fail (GTK_IS_EXTEXT (text), FALSE);

  return text->read_only ? FALSE : TRUE;
}

/*****************************************************************************/
/*          PROTECTED                  editable overrides                               */
/*****************************************************************************/

static gchar *
gtk_extext_real_get_chars(GtkEditable *editable,gint start,gint end)
{
  g_return_val_if_fail (editable != NULL,NULL);
  g_return_val_if_fail (GTK_IS_EXTEXT (editable),NULL);

  return text_get_text(GTK_EXTEXT(editable),start,end);
}

static void 
gtk_extext_delete_text       (GtkEditable       *editable,
					gint              startpos,
					gint               endpos)
{
  GtkExText  *text;
  gint oldlinecount;
  g_return_if_fail (editable != NULL);
  g_return_if_fail (GTK_IS_EDITABLE (editable));
  g_return_if_fail (GTK_IS_EXTEXT (editable));

  text=GTK_EXTEXT(editable);	

  if(text->read_only)
    return ;
		
  oldlinecount=GTK_EXTEXT(editable)->line_count;	
  if(editable->selection_start_pos!=editable->selection_end_pos && editable->selection_start_pos!=-1)
  	text_remove(GTK_EXTEXT(editable),editable->selection_start_pos,
						 editable->selection_end_pos - editable->selection_start_pos);
  else
  	text_remove(GTK_EXTEXT(editable),startpos,endpos - startpos);

  if(oldlinecount!=text->line_count)	{
  	text->vadj->upper=text->line_count;
  	if(text->scroll_line_count)
  		text->vadj->page_size=text->line_count/text->scroll_line_count;
  	if(text->scroll_line_start->line_number < text->vadj->value)
  		text->vadj->value=text->scroll_line_start->line_number;

  	gtk_adjustment_changed(text->vadj);
  }
  gtk_widget_queue_draw(GTK_WIDGET(editable));
}

static void 
gtk_extext_select_region(GtkEditable *editable,gint start,gint end)
{
  g_return_if_fail (editable != NULL);
  g_return_if_fail (GTK_IS_EXTEXT (editable));

  editable->selection_start_pos=start;
  editable->selection_end_pos=end;

  gtk_widget_queue_draw(GTK_WIDGET(editable));
}

static void 
gtk_extext_real_set_editable(GtkEditable *text,gboolean r)
{
   g_return_if_fail (GTK_IS_EDITABLE (text));

   GTK_EXTEXT(text)->read_only = r ? FALSE : TRUE;
}

void 
gtk_extext_insert_text       (GtkEditable       *editable,
					const gchar       *new_text,
					gint               new_len,
					gint               *position)
{
  GtkExText *text;
  g_return_if_fail (editable != NULL);
  g_return_if_fail (GTK_IS_EDITABLE (editable));
  g_return_if_fail (GTK_IS_EXTEXT (editable));

  text=GTK_EXTEXT(editable);

  if(GTK_EDITABLE(text)->selection_start_pos!=-1 && GTK_EDITABLE(text)->selection_end_pos!=-1)	{
  	if(*position > editable->selection_start_pos  && *position<=editable->selection_end_pos) {
  		gtk_editable_delete_text(editable,editable->selection_start_pos,editable->selection_end_pos);
  		*position=editable->current_pos;
  	}
  	else
  		gtk_editable_delete_text(editable,editable->selection_start_pos,editable->selection_end_pos);
  }

  if(text->line_max_chars)
    *position=text_insert_line_wrap(text,*position,(gchar *)new_text,new_len);
  else
    *position=text_insert(text,*position,(gchar*)new_text,new_len,TRUE);
	
  text->vadj->upper=text->line_count;
  if(text->scroll_line_count)
  	text->vadj->page_size=text->line_count/text->scroll_line_count;

  gtk_adjustment_changed(text->vadj);
  gtk_widget_queue_draw(GTK_WIDGET(text));
}

static void 
gtk_extext_update_text(GtkEditable *text,gint start,gint end)
{
  /* hmm not sure about this */
  gtk_widget_queue_draw(GTK_WIDGET(text));
}

/*****************************************************************************/
/*           PUBLIC                  editable overrides                               */
/*****************************************************************************/

void 
gtk_extext_set_position(GtkEditable *text,gint pos)
{
  g_return_if_fail (GTK_IS_EXTEXT (text));

  text_set_pos(GTK_EXTEXT(text),pos);
  goto_line(GTK_EXTEXT(text),GTK_EXTEXT(text)->line_pos_index,FALSE,TRUE);

  gtk_widget_queue_draw(GTK_WIDGET(text));
}

gint
gtk_extext_get_position(GtkEditable *edit)
{
  g_return_val_if_fail (edit != NULL,FALSE);
  g_return_val_if_fail (GTK_IS_EXTEXT (edit),FALSE);

  return GTK_EDITABLE(edit)->current_pos;
}

/*****************************************************************************/
/*                              Search functions                                */
/*****************************************************************************/

gboolean 
gtk_extext_search (GtkExText *text, gchar *search,gint pos,
	      gboolean iscase,gboolean forward, GtkExTextMatch *m)
{
  guchar *txt;
  gint len;
  gint found;
  g_return_val_if_fail (pos <= text->length, FALSE);
  g_return_val_if_fail (pos > -1, FALSE);
  g_return_val_if_fail (m != NULL, FALSE);

  m->startpos=m->endpos=0;
  txt=text->text.ch;
  len=strlen(search);
  if(!forward)	{
	 g_warning("SEARCH BACKWARD NOT YET IMPLEMENTED\n");
	return FALSE;
  }	

  while(pos < text->length-len)	{
  	if(pos+len > text->part1len && txt==text->text.ch)	{
  		gint i=0;
  		txt=gtk_extext_get_text(text,pos,pos+len);
  		while(pos < text->part1len)	{
  			found=iscase ? strncmp(&txt[i],search,len) : g_strncasecmp(&txt[i],search,len);
  			if(!found)
  			{	g_free(txt);	m->startpos=pos;	m->endpos=pos+len;	return TRUE;	}
  			pos++;
  			i++;
  		}
  		g_free(txt);
  		txt=text->part2text.ch;
  	}
  	else	{				
  		found=iscase ? strncmp(&txt[pos],search,len) : g_strncasecmp(&txt[pos],search,len);
  		if(!found)
  		{	m->startpos=pos;	m->endpos=pos+len;	return TRUE;	}
  		pos++;
  	};
  }

  return FALSE;	
}

#ifdef WITH_REGEX
/*****************************************************************************/
/*                                  Regex API                                    */
/* 	stolen from GtkEditor by Thomas Maulund */
/*****************************************************************************/

/* arch -- searches for regex in text from position
 * 'start'. If 'forward' is TRUE, it searches forward, otherwise
 * backwards.  If  found, returns index of beginning of the match,
 * otherwise returns < 0.  If found the match interval is
 * [m.from,m.to[.  m.from is always equal to the return value, but
 * m.to is undefined if no match.  I search in GtkSCText, since I don't
 * really need an editor for this function. This will also make it
 * easier to move this function to the text widget, should we want
 * that. */
gint
gtk_extext_regex_search (GtkExText *text, gint start, Regex *regex,
	     gboolean forward, 
		GtkExTextMatch *m)
{
  g_return_val_if_fail (start <= text->length, -1);
  g_return_val_if_fail (regex != NULL, -1);
  g_return_val_if_fail (m != NULL, -1);

  m->startpos = re_search_2 (&regex->buf,
			 /* text before gap */
			 text->text.ch,
			text->part1len,
			 /* text after gap */
			 &text->part2text.ch[text->part1len],
				text->length - text->part1len+1,
			 /* from 'start' and forward to the end */
			 start,
			 (forward ? text->length - start : 
              -start),
			 &regex->reg,
			 text->length - start);

  if (m->startpos > -1) m->endpos = regex->reg.end[0];

  return m->startpos;    
}

/* regex_match -- tries to match regex at the 'pos' position in the
 * text. It returns the number of chars matched, or -1 if no match.
 * Warning!  The number matched can be 0, if the regex matches the
 * empty string.  The reason for workin on GtkSCText is the same as in
 * regex_search. */
gint
gtk_extext_regex_match (GtkExText *text, gint pos, Regex *regex)
{
  g_return_val_if_fail (pos <= text->length, -1);
  g_return_val_if_fail (regex != NULL, -1);

  return re_match_2 (&regex->buf,
		     /* text before gap */
		     text->text.ch, text->part1len,
		     /* text after gap */
		     &text->part2text.ch[text->part1len],
		     text->length - text->part1len+1,
		     /* from pos and not after the end */
		     pos, &regex->reg, text->length);
}
#endif
