/* modrdn.c - ldbm backend modrdn routine */

#include "portable.h"

#include <stdio.h>

#include <ac/string.h>
#include <ac/socket.h>

#include "slap.h"
#include "back-ldbm.h"
#include "proto-back-ldbm.h"

int
ldbm_back_modrdn(
    Backend	*be,
    Connection	*conn,
    Operation	*op,
    char	*dn,
    char	*newrdn,
    int		deleteoldrdn
)
{
	struct ldbminfo	*li = (struct ldbminfo *) be->be_private;
	char		*matched = NULL;
	char		*p_dn = NULL, *p_ndn = NULL;
	char		*new_dn = NULL, *new_ndn = NULL;
	char		sep[2];
	Entry		*e, *p = NULL;
	int		rootlock = 0;
	int		rc = -1;

	char		*new_rdn_val = NULL;	/* Val of new rdn */
	char		*new_rdn_type = NULL;	/* Type of new rdn */
	char		*old_rdn;		/* Old rdn's attr type & val */
	char		*old_rdn_type = NULL;	/* Type of old rdn attr. */
	char		*old_rdn_val = NULL;	/* Old rdn attribute value */
	struct berval	add_bv;			/* Stores new rdn att */
	struct berval	*add_bvals[2];		/* Stores new rdn att */
	struct berval	del_bv;			/* Stores old rdn att */
	struct berval	*del_bvals[2];		/* Stores old rdn att */
	LDAPMod		mod[2];			/* Used to delete old rdn */

	Debug( LDAP_DEBUG_TRACE,
	       "ldbm_back_modrdn:()==>\n",
	       0, 0, 0 );

	/* get entry with writer lock */
	if ( (e = dn2entry_w( be, dn, &matched )) == NULL ) {
		send_ldap_result( conn, op, LDAP_NO_SUCH_OBJECT, matched, "" );
		if ( matched != NULL ) {
			free( matched );
		}
		return( -1 );
	}

#ifdef SLAPD_CHILD_MODIFICATION_WITH_ENTRY_ACL
		/* check parent for "children" acl */
	if ( ! access_allowed( be, conn, op, e,
		"entry", NULL, ACL_WRITE ) )
	{
		Debug( LDAP_DEBUG_TRACE, "no access to entry\n", 0,
			0, 0 );
		send_ldap_result( conn, op, LDAP_INSUFFICIENT_ACCESS,
			"", "" );
		goto return_results;
	}
#endif

	if ( (p_ndn = dn_parent( be, e->e_ndn )) != NULL ) {
		/* parent + rdn + separator(s) + null */
		if( (p = dn2entry_w( be, p_ndn, &matched )) == NULL) {
			Debug( LDAP_DEBUG_TRACE, "parent does not exist\n",
				0, 0, 0);
			send_ldap_result( conn, op, LDAP_OPERATIONS_ERROR,
				"", "");
			goto return_results;
		}

#ifndef SLAPD_CHILD_MODIFICATION_WITH_ENTRY_ACL
		/* check parent for "children" acl */
		if ( ! access_allowed( be, conn, op, p,
			"children", NULL, ACL_WRITE ) )
		{
			Debug( LDAP_DEBUG_TRACE, "no access to parent\n", 0,
				0, 0 );
			send_ldap_result( conn, op, LDAP_INSUFFICIENT_ACCESS,
				"", "" );
			goto return_results;
		}
#endif

		p_dn = dn_parent( be, e->e_dn );


	} else {
		/* no parent, modrdn entry directly under root */
		if( ! be_isroot( be, op->o_ndn ) ) {
			Debug( LDAP_DEBUG_TRACE, "no parent & not root\n",
				0, 0, 0);
			send_ldap_result( conn, op, LDAP_INSUFFICIENT_ACCESS,
				"", "");
			goto return_results;
		}

		ldap_pvt_thread_mutex_lock(&li->li_root_mutex);
		rootlock = 1;


	}

	build_new_dn( &new_dn, e->e_dn, p_dn, newrdn ); 
	new_ndn = dn_normalize_case( ch_strdup( new_dn ) );

	/* Get attribute type and attribute value of our new rdn, we will
	 * need to add that to our new entry
	 */

	if ( (new_rdn_type = rdn_attr_type( newrdn )) == NULL ) {
	    
		Debug( LDAP_DEBUG_TRACE,
		       "ldbm_back_modrdn: can't figure out type of newrdn\n",
		       0, 0, 0 );
		send_ldap_result( conn, op, LDAP_OPERATIONS_ERROR, "", "" );
		goto return_results;		

	}

	if ( (new_rdn_val = rdn_attr_value( newrdn )) == NULL ) {
	    
		Debug( LDAP_DEBUG_TRACE,
		       "ldbm_back_modrdn: can't figure out val of newrdn\n",
		       0, 0, 0 );
		send_ldap_result( conn, op, LDAP_OPERATIONS_ERROR, "", "" );
		goto return_results;		

	}

	Debug( LDAP_DEBUG_TRACE,
	       "ldbm_back_modrdn: new_rdn_val=\"%s\", new_rdn_type=\"%s\"\n",
	       new_rdn_val, new_rdn_type, 0 );

	/* Retrieve the old rdn from the entry's dn */

	if ( (old_rdn = dn_rdn( be, dn )) == NULL ) {

		Debug( LDAP_DEBUG_TRACE,
		       "ldbm_back_modrdn: can't figure out old_rdn from dn\n",
		       0, 0, 0 );
		send_ldap_result( conn, op, LDAP_OPERATIONS_ERROR, "", "" );
		goto return_results;		

	}

	if ( (old_rdn_type = rdn_attr_type( old_rdn )) == NULL ) {
	    
		Debug( LDAP_DEBUG_TRACE,
		       "ldbm_back_modrdn: can't figure out the old_rdn type\n",
		       0, 0, 0 );
		send_ldap_result( conn, op, LDAP_OPERATIONS_ERROR, "", "" );
		goto return_results;		
		
	}
	
	if ( strcasecmp( old_rdn_type, new_rdn_type ) != 0 ) {

	    /* Not a big deal but we may say something */
	    Debug( LDAP_DEBUG_TRACE,
		   "ldbm_back_modrdn: old_rdn_type=%s, new_rdn_type=%s!\n",
		   old_rdn_type, new_rdn_type, 0 );
	    
	}		

	if ( dn_type( old_rdn ) == DN_X500 ) {

		Debug( LDAP_DEBUG_TRACE, "ldbm_back_modrdn: DN_X500\n",
		       0, 0, 0 );
		
		/* Add new attribute value to the entry.
		 */

		add_bvals[0] = &add_bv;		/* Array of bervals */
		add_bvals[1] = NULL;

		add_bv.bv_val = new_rdn_val;
		add_bv.bv_len = strlen(new_rdn_val);
		
		mod[0].mod_type = new_rdn_type;	
		mod[0].mod_bvalues = add_bvals;
		mod[0].mod_op = LDAP_MOD_SOFTADD;
		mod[0].mod_next = NULL;
		
		Debug( LDAP_DEBUG_TRACE,
		       "ldbm_back_modrdn: adding new rdn attr val =\"%s\"\n",
		       new_rdn_val, 0, 0 );

		/* Remove old rdn value if required */

		if (deleteoldrdn) {

			del_bvals[0] = &del_bv;		/* Array of bervals */
			del_bvals[1] = NULL;
			/* Get value of old rdn */
	
			if ((old_rdn_val = rdn_attr_value( old_rdn ))
			    == NULL) {
			    
				Debug( LDAP_DEBUG_TRACE,
				       "ldbm_back_modrdn: can't figure out old_rdn_val from old_rdn\n",
				       0, 0, 0 );
				send_ldap_result( conn, op,
						  LDAP_OPERATIONS_ERROR,
						  "", "" );
				goto return_results;		


			}

			/* Remove old value of rdn as an attribute. */
		    
			del_bv.bv_val = old_rdn_val;
			del_bv.bv_len = strlen(old_rdn_val);

			/* No need to normalize old_rdn_type, delete_values()
			 * does that for us
			 */
			mod[0].mod_next = &mod[1];
			mod[1].mod_type = old_rdn_type;	
			mod[1].mod_bvalues = del_bvals;
			mod[1].mod_op = LDAP_MOD_DELETE;
			mod[1].mod_next = NULL;

		}/* if (deleteoldrdn) */

		/* modify memory copy of entry */
		if ( ldbm_internal_modify( be, conn, op, dn, &mod[0], e )
		     != 0 ) {
		    
			goto return_results;
			
		}
	
	} else {
	    

		Debug( LDAP_DEBUG_TRACE, "ldbm_back_modrdn: DNS DN\n",
		       0, 0, 0 );
		/* XXXV3: not sure of what to do here */
		Debug( LDAP_DEBUG_TRACE,
		       "ldbm_back_modrdn: not fully implemented...\n",
		       0, 0, 0 );  
		send_ldap_result( conn, op, LDAP_OPERATIONS_ERROR, NULL, NULL );
		goto return_results;

	}

	if ( (dn2id ( be, new_ndn ) ) != NOID ) {
		send_ldap_result( conn, op, LDAP_ALREADY_EXISTS, NULL, NULL );
		goto return_results;
	}

	/* check for abandon */
	ldap_pvt_thread_mutex_lock( &op->o_abandonmutex );
	if ( op->o_abandon ) {
		ldap_pvt_thread_mutex_unlock( &op->o_abandonmutex );
		goto return_results;
	}
	ldap_pvt_thread_mutex_unlock( &op->o_abandonmutex );

	/* add new one */
	if ( dn2id_add( be, new_ndn, e->e_id ) != 0 ) {
		send_ldap_result( conn, op, LDAP_OPERATIONS_ERROR, NULL, NULL );
		goto return_results;
	}

	/* delete old one */
	if ( dn2id_delete( be, e->e_ndn ) != 0 ) {
		send_ldap_result( conn, op, LDAP_OPERATIONS_ERROR, NULL, NULL );
		goto return_results;
	}


	(void) cache_delete_entry( &li->li_cache, e );
	free( e->e_dn );
	free( e->e_ndn );
	e->e_dn = new_dn;
	e->e_ndn = new_ndn;
	(void) cache_update_entry( &li->li_cache, e );

	/* id2entry index: commit */
	if ( id2entry_add( be, e ) != 0 ) {
		entry_free( e );
		send_ldap_result( conn, op, LDAP_OPERATIONS_ERROR, "", "" );
		goto return_results_new;
	}

	send_ldap_result( conn, op, LDAP_SUCCESS, NULL, NULL );
	rc = 0;

	Debug( LDAP_DEBUG_TRACE,
	       "ldbm_back_modrdn:()<==\n",
	       0, 0, 0 );
	goto return_results_new;

return_results:	
	if( new_dn != NULL ) free( new_dn );
	if( new_ndn != NULL ) free( new_ndn );
return_results_new:
	/* NOTE:
	 * new_dn and new_ndn are not deallocated because they are used by
	 * the cache entry at this point.
	 */
	if( p_dn != NULL ) free( p_dn );
	if( p_ndn != NULL ) free( p_ndn );

	if( matched != NULL ) free( matched );

	/* LDAP v2 supporting correct attribute handling. */
	if( new_rdn_type != NULL ) free(new_rdn_type);
	if( new_rdn_val != NULL ) free(new_rdn_val);
	if( old_rdn != NULL ) free(old_rdn);
	if( old_rdn_type != NULL ) free(old_rdn_type);
	if( old_rdn_val != NULL ) free(old_rdn_val);

	if( p != NULL ) {
		/* free parent and writer lock */
		cache_return_entry_w( &li->li_cache, p );
	}

	if ( rootlock ) {
		/* release root writer lock */
		ldap_pvt_thread_mutex_unlock(&li->li_root_mutex);
	}

	/* free entry and writer lock */
	cache_return_entry_w( &li->li_cache, e );
	return( rc );
}
