/*
**  LocalMessage.m
**
**  Copyright (c) 2001, 2002, 2003
**
**  Author: Ludovic Marcotte <ludovic@Sophos.ca>
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**  
**  This library is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
**  Lesser General Public License for more details.
**  
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include <Pantomime/LocalMessage.h>

#include <Pantomime/Constants.h>
#include <Pantomime/LocalFolder.h>
#include <Pantomime/LocalStore.h>
#include <Pantomime/NSData+Extensions.h>

#include <Foundation/NSDebug.h>
#include <Foundation/NSException.h>
#include <Foundation/NSValue.h>

#include <string.h>

static int currentLocalMessageVersion = 1;

//
//
//
@implementation LocalMessage 

- (id) init
{
  self = [super init];

  [LocalMessage setVersion: currentLocalMessageVersion];

  return self;
}


//
// NSCoding protocol
//
- (void) encodeWithCoder: (NSCoder *) theCoder
{
  [super encodeWithCoder: theCoder];

  [LocalMessage setVersion: currentLocalMessageVersion];
	
  [theCoder encodeObject: [NSNumber numberWithLong: [self filePosition]]];
  [theCoder encodeObject: [NSNumber numberWithLong: [self bodyFilePosition]]];

  // Store the name of the file; we need it for local.
  [theCoder encodeObject: mailFilename];

  // Store the message type; useful to have.
  [theCoder encodeObject: [NSNumber numberWithInt: [self messageType]]];
}


//
//
//
- (id) initWithCoder: (NSCoder *) theCoder
{ 
  int version;

  self = [super initWithCoder: theCoder];
  
  [self setFilePosition: [[theCoder decodeObject] longValue]];
  [self setBodyFilePosition: [[theCoder decodeObject] longValue]];

  version = [theCoder versionForClassName: @"LocalMessage"];

  if ( version == 1 )
    {
      // Retrieve the mail file name which we need for local storage.
      [self setMailFilename: [theCoder decodeObject]];
      
      // Retrieve the message type
      [self setMessageType: [[theCoder decodeObject] intValue]];
    }
  else
    {
      [NSException raise: NSInternalInconsistencyException
		   format: @"LocalMessage cache error."];
    }

  return self;
}


//
// access / mutation methods
//
- (long) filePosition
{
  return filePosition;
}

- (void) setFilePosition: (long) theFilePosition
{
  filePosition = theFilePosition;
}


//
//
//
- (long) bodyFilePosition
{
  return bodyFilePosition;
}


- (void) setBodyFilePosition: (long) theBodyFilePosition
{
  bodyFilePosition = theBodyFilePosition;
}


//
//
//
- (int) messageType
{
  return messageType;
}


- (void) setMessageType: (int) theType
{
  messageType = theType;
}


//
//
//
- (NSString *) mailFilename
{
  return (mailFilename);
}


- (void) setMailFilename: (NSString *) theFilename
{	
  if ( theFilename )
    {
      RETAIN(theFilename);
      RELEASE(mailFilename);
      mailFilename = theFilename;
    }
  else
    {
      DESTROY(mailFilename);
    }
}


//
//
//
- (void) dealloc
{
  TEST_RELEASE(mailFilename);
  
  [super dealloc];
}


//
//
//
- (NSData *) rawSource
{
  NSMutableData *aMutableData;
  LocalFolder *aLocalFolder;
  char aLine[1024];

  FILE *aStream;
  long mark;

  aLocalFolder = (LocalFolder *)[self folder];
  aStream = NULL;

  // If we are reading from a mbox file, the file is already open
  if ([self messageType] == MAILBOX_FORMAT_MBOX)
    {
      aStream = [aLocalFolder stream];
    }
  // we need to open the specific file
  else if ([self messageType] == MAILBOX_FORMAT_MAILDIR)
    { 
      aStream = fopen([[self mailFilename] cString], "r");
    }

  if ( !aStream )
    {
      NSDebugLog(@"Stream is null!");
      return nil;
    }
  
  mark = ftell(aStream);
  
  if (fseek(aStream, [self filePosition], SEEK_SET) < 0)
    {
      NSDebugLog( @"Seek operation to position %d failed!", [self filePosition]);
      return nil;
    }
  
  // We initialize our mutable data and our buffer
  aMutableData = [[NSMutableData alloc] init];
  memset(aLine, 0, 1024);

  while ( fgets(aLine, 1024, aStream) != NULL && 
	  ftell(aStream) < ([self filePosition] + [self size]) )
    {
      [aMutableData appendBytes: aLine  length: strlen(aLine) ];
      memset(aLine, 0, 1024);
    }
  
  fseek(aStream, mark, SEEK_SET);
  
  // If we are operating on a local file, close it.
  if ([self messageType] == MAILBOX_FORMAT_MAILDIR)
    {
      fclose(aStream);
    }
  
  return AUTORELEASE(aMutableData);
}


//
// This method is called to initialize the message if it wasn't.
// If we set it to NO and we HAD a content, we release the content;
//
- (void) setInitialized: (BOOL) aBOOL
{
  [super setInitialized: aBOOL];

  if ( aBOOL )
    {
      NSData *aData;

      aData = [self rawSource];

      if ( aData )
	{
	  NSRange aRange;

	  aRange = [aData rangeOfCString: "\n\n"];
	  
	  if (aRange.length == 0)
	    {
	      NSDebugLog(@"LocalMessage: Failed to initialize the message from data.");
	      [super setInitialized: NO];
	      return;
	    }
	  
	  [self setHeadersFromData: [aData subdataWithRange: NSMakeRange(0,aRange.location)]];
	  [self setContentFromRawSource:
		  [aData subdataWithRange:
			   NSMakeRange(aRange.location + 2, [aData length]-(aRange.location+2))]];
	}
      else
	{
	  NSDebugLog(@"LocalMessage: Failed to get the raw source of the local message.");
	  [super setInitialized: NO];
	  return;
	}
    }
  else
    {
      DESTROY(content);
    }  
  
}

@end
