/*
 * jMax
 * Copyright (C) 1994, 1995, 1998, 1999 by IRCAM-Centre Georges Pompidou, Paris, France.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * See file LICENSE for further informations on licensing terms.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 * Based on Max/ISPW by Miller Puckette.
 *
 * Authors: Maurizio De Cecco, Francois Dechelle, Enzo Maggi, Norbert Schnell.
 *
 */

/* "expr" was written by Shahrokh Yadegari c. 1989. -msp */

#include <stdio.h>

#include "m_pd.h"
#include "fts_to_pd.h"
#include "vexp.h"
#include "malloc.h"

extern struct ex_ex *ex_eval(struct expr *exp, struct ex_ex *eptr,
    struct ex_ex *optr);

static t_class *expr_class;

/*------------------------- expr class -------------------------------------*/

extern int expr_donew(struct expr *expr, int ac, t_atom *av);

/*#define EXPR_DEBUG*/

static void expr_bang(t_expr *x);

static void expr_list(t_expr *x, t_symbol *s,
    int argc, const fts_atom_t *argv)
{
    int i;

    if (argc > MAX_VARS) argc = MAX_VARS;

    for (i = 0; i < argc; i++)
    {
	if (argv[i].a_type == A_FLOAT)
	{
    	    if (x->exp_var[i].ex_type == ET_FI)
    	    	x->exp_var[i].ex_flt = argv[i].a_w.w_float;
    	    else if (x->exp_var[i].ex_type == ET_II)
    	    	x->exp_var[i].ex_int = argv[i].a_w.w_float;
    	    else pd_error(x, "expr: type mismatch");
	}
	else if (argv[i].a_type == A_SYMBOL)
	{
    	    if (x->exp_var[i].ex_type == ET_SI)
    	    	x->exp_var[i].ex_ptr = (char *)argv[i].a_w.w_symbol;
    	    else pd_error(x, "expr: type mismatch");
	}
    }
    expr_bang(x);
}

static void expr_flt(t_expr *x, t_float f, int in)
{
    if (in > MAX_VARS)
    	return;

    if (x->exp_var[in].ex_type == ET_FI)
    	x->exp_var[in].ex_flt = f;
    else if (x->exp_var[in].ex_type == ET_II)
    	x->exp_var[in].ex_int = f;
}

static t_class *exprproxy_class;

typedef struct _exprproxy
{
    t_pd p_pd;
    int p_index;
    t_expr *p_owner;
    struct _exprproxy *p_next;
} t_exprproxy;

t_exprproxy *exprproxy_new(t_expr *owner, int index)
{
    t_exprproxy *x = (t_exprproxy *)pd_new(exprproxy_class);
    x->p_owner = owner;
    x->p_index = index;
    x->p_next = owner->exp_proxy;
    owner->exp_proxy = x;
    return (x);
}

void exprproxy_float(t_exprproxy *p, t_floatarg f)
{
    t_expr *x = p->p_owner;
    int in = p->p_index;

    if (in > MAX_VARS)
    	return;

    if (x->exp_var[in].ex_type == ET_FI)
    	x->exp_var[in].ex_flt = f;
    else if (x->exp_var[in].ex_type == ET_II)
    	x->exp_var[in].ex_int = f;
}

/* method definitions */
static void
expr_ff(t_expr *x)
{
    t_exprproxy *y;
    while (y = x->exp_proxy)
    {
    	x->exp_proxy = y->p_next;
	pd_free(&y->p_pd);
    }
    if (x->exp_stack)
    	fts_free(x->exp_stack);
}

static void
expr_bang(t_expr *x)
{

#ifdef EXPR_DEBUG
  {
    int i;
    struct ex_ex *eptr;

    for (i = 0, eptr = x->exp_var;  ; eptr++, i++)
      {
	if (!eptr->ex_type)
	  break;
	switch (eptr->ex_type)
	  {
	  case ET_II:
	    fprintf(stderr,"ET_II: %d \n", eptr->ex_int);
	    break;

	  case ET_FI:
	    fprintf(stderr,"ET_FT: %f \n", eptr->ex_flt);
	    break;

	  default:
	    fprintf(stderr,"oups\n");
	  }
      }
  }
#endif

  if (!ex_eval(x, x->exp_stack, &x->exp_res))
    {
      /*       fprintf(stderr,"expr_bang(error evaluation)\n"); */
      return;
    }
  

  switch(x->exp_res.ex_type)
    {
    case ET_INT:
      outlet_float(x->exp_outlet,  x->exp_res.ex_int);
      break;

    case ET_FLT:
      outlet_float(x->exp_outlet,  x->exp_res.ex_flt);
      break;

    case ET_SYM:
      /* CHANGE this will have to be taken care of */

    default:
      post("expr: bang: unrecognized result %ld\n", x->exp_res.ex_type);
    }
}

static t_expr *expr_new(t_symbol *s, int ac, t_atom *av)
{
  struct expr *x;
  int i, ninlet;
  struct ex_ex *eptr;
  t_atom fakearg;

  if (!ac)
  {
    ac = 1;
    av = &fakearg;
    SETFLOAT(&fakearg, 0);
  }

  x = (t_expr *)pd_new(expr_class);
  x->exp_stack = (struct ex_ex *)0;
  x->exp_proxy = 0;

  if (expr_donew(x, ac, av))
    {
      pd_error(x, "expr: syntax error");
      pd_free(&x->exp_ob.ob_pd);
      return (0);
    }

  for (i = 0, eptr = x->exp_var, ninlet = 0; i < MAX_VARS ; i++, eptr++)
    if (!eptr->ex_type) 
  {
    ninlet = i;
    break;
  }

  for (i = 1, eptr = x->exp_var + 1; i < ninlet ; i++, eptr++)
  {
    t_exprproxy *p;
    switch (eptr->ex_type)
    {
   case 0:
      /* nothing is using this inlet */
      if (i < ninlet)
	floatinlet_new(&x->exp_ob, &eptr->ex_flt);
      break;
   case ET_II:
   case ET_FI:
      p = exprproxy_new(x, i);
      inlet_new(&x->exp_ob, &p->p_pd, &s_float, &s_float);
      break;
   case ET_SI:
      symbolinlet_new(&x->exp_ob, (t_symbol **)&eptr->ex_ptr);
      break;
   default:
      pd_error(x, "expr: bad type (%lx) inlet = %d\n", eptr->ex_type, i, 0, 0, 0);
      break;
    }
  }
  x->exp_outlet = outlet_new(&x->exp_ob, 0);

  return (x);
}

void expr_setup(void)
{
    expr_class = class_new(gensym("expr"), (t_newmethod)expr_new,
    	(t_method)expr_ff, sizeof(t_expr), 0, A_GIMME, 0);
    class_addlist(expr_class, expr_list);
    exprproxy_class = class_new(gensym("exprproxy"), 0,
    	0, sizeof(t_exprproxy), CLASS_PD, 0);
    class_addfloat(exprproxy_class, exprproxy_float);
}

/* -- the following functions use Pd internals and so are in the "if" file. */


int ex_getsym(char *p, fts_symbol_t *s)
{
  *s = gensym(p);
  return (0);
}

const char *
ex_symname(fts_symbol_t s)
{
  return (fts_symbol_name(s));
}

/*
   * max_ex_tab -- evaluate this table access
   *		 eptr is the name of the table and arg is the index we
   *		 have to put the result in optr
   *		 return 1 on error and 0 otherwise
   *
   * Arguments:
   *  the expr object
   *  table 
   *  the argument 
   *  the result pointer 
   */
int
max_ex_tab(struct expr *exp, fts_symbol_t s, struct ex_ex *arg, struct ex_ex *optr)
{
  t_garray *garray;
  int size, indx;
  t_float *vec;

  if (!s || !(garray = (t_garray *)pd_findbyclass(s, garray_class)) ||
    !garray_getfloatarray(garray, &size, &vec))
    {
      optr->ex_type = ET_FLT;					
      optr->ex_int = 0;					
      pd_error(exp, "no such table '%s'", s->s_name);		
      return (1);						
    }
  optr->ex_type = ET_FLT;

  switch (arg->ex_type) {
  case ET_INT:
    indx = arg->ex_int;
    break;
  case ET_FLT:
    	/* strange interpolation code deleted here -msp */
    indx = arg->ex_flt;
    break;

  default:	/* do something with strings */
    pd_error(exp, "expr: bad argument for table '%s'\n", fts_symbol_name(s));
    indx = 0;
  }
  if (indx < 0) indx = 0;
  else if (indx >= size) indx = size - 1;
  optr->ex_flt = vec[indx];
  return (0);
}

#define	ISTABLE(sym, garray, size, vec)			    	      \
if (!sym || !(garray = (t_garray *)pd_findbyclass(sym, garray_class)) || \
!garray_getfloatarray(garray, &size, &vec))  {		\
optr->ex_type = ET_FLT;					\
optr->ex_int = 0;					\
error("no such table '%s'", sym->s_name);		\
return;						\
}

/*
 * ex_size -- find the size of a table
 */

void
ex_size(long int argc, struct ex_ex *argv, struct ex_ex *optr)
{
  t_symbol *s;
  t_garray *garray;
  int size;
  t_float *vec;
  
  if (argv->ex_type != ET_SYM)
    {
      post("expr: size: need a table name\n");
      optr->ex_type = ET_INT;
      optr->ex_int = 0;
      return;
    }
  
  s = (fts_symbol_t ) argv->ex_ptr;

  ISTABLE(s, garray, size, vec);

  optr->ex_type = ET_INT;
  optr->ex_int = size;
}

/*
 * ex_sum -- calculate the sum of all elements of a table
 */

void
ex_sum(long int argc, struct ex_ex *argv, struct ex_ex *optr)
{
  t_symbol *s;
  t_garray *garray;
  int size;
  t_float *vec, sum;
  int indx;
  
  if (argv->ex_type != ET_SYM)
    {
      post("expr: sum: need a table name\n");
      optr->ex_type = ET_INT;
      optr->ex_int = 0;
      return;
    }
  
  s = (fts_symbol_t ) argv->ex_ptr;

  ISTABLE(s, garray, size, vec);

  for (indx = 0, sum = 0; indx < size; indx++)
    sum += vec[indx];

  optr->ex_type = ET_FLT;
  optr->ex_flt = sum;
}


/*
 * ex_Sum -- calculate the sum of table with the given boundries
 */

void
ex_Sum(long int argc, struct ex_ex *argv, struct ex_ex *optr)
{
  t_symbol *s;
  t_garray *garray;
  int size;
  t_float *vec, sum;
  int indx, n1, n2;
  
  if (argv->ex_type != ET_SYM)
    {
      post("expr: sum: need a table name\n");
      optr->ex_type = ET_INT;
      optr->ex_int = 0;
      return;
    }
  
  s = (fts_symbol_t ) argv->ex_ptr;

  ISTABLE(s, garray, size, vec);

  if (argv->ex_type != ET_INT || argv[1].ex_type != ET_INT)
    {
      post("expr: Sum: boundries have to be fix values\n");
      optr->ex_type = ET_INT;
      optr->ex_int = 0;
      return;
    }
  n1 = argv->ex_int;
  n2 = argv[1].ex_int;

  for (indx = n1, sum = 0; indx < n2; indx++)
    if (indx >= 0 && indx < size)
      sum += vec[indx];

  optr->ex_type = ET_FLT;
  optr->ex_flt = sum;
}

/*
 * ex_avg -- calculate the avarage of a table
 */

void
ex_avg(long int argc, struct ex_ex *argv, struct ex_ex *optr)
{
#if 0
  fts_symbol_t s;
  fts_integer_vector_t *tw = 0;
  
  if (argv->ex_type != ET_SYM)
    {
      post("expr: avg: need a table name\n");
      optr->ex_type = ET_INT;
      optr->ex_int = 0;
    }

  s = (fts_symbol_t ) argv->ex_ptr;

  tw = table_integer_vector_get_by_name(s);

  if (tw)
    {
      optr->ex_type = ET_INT;

      if (! fts_integer_vector_get_size(tw))
	optr->ex_int = 0;
      else
	optr->ex_int = fts_integer_vector_get_sum(tw) / fts_integer_vector_get_size(tw);
    }
  else
    {
      optr->ex_type = ET_INT;
      optr->ex_int = 0;
      post("expr: avg: no such table %s\n", fts_symbol_name(s));
    }
#endif
}


/*
 * ex_Avg -- calculate the avarage of table with the given boundries
 */

void
ex_Avg(long int argc, struct ex_ex *argv, struct ex_ex *optr)
{
#if 0
  fts_symbol_t s;
  fts_integer_vector_t *tw = 0;
  
  if (argv->ex_type != ET_SYM)
    {
      post("expr: Avg: need a table name\n");
      optr->ex_type = ET_INT;
      optr->ex_int = 0;
    }

  s = (fts_symbol_t ) (argv++)->ex_ptr;

  tw = table_integer_vector_get_by_name(s);

  if (! tw)
    {
      optr->ex_type = ET_INT;
      optr->ex_int = 0;
      post("expr: Avg: no such table %s\n", fts_symbol_name(s));
      return;
    }

  if (argv->ex_type != ET_INT || argv[1].ex_type != ET_INT)
    {
      post("expr: Avg: boundries have to be fix values\n");
      optr->ex_type = ET_INT;
      optr->ex_int = 0;
      return;
    }

  optr->ex_type = ET_INT;

  if (argv[1].ex_int - argv->ex_int <= 0)
    optr->ex_int = 0;
  else
    optr->ex_int = (fts_integer_vector_get_sub_sum(tw, argv->ex_int, argv[1].ex_int) /
		    (argv[1].ex_int - argv->ex_int));
#endif
}

/*
 * ex_store -- store a value in a table
 *			   if the index is greater the size of the table, we will make a modulo the
 *			   size of the table
 */

void
ex_store(long int argc, struct ex_ex *argv, struct ex_ex *optr)
{
#if 0
  fts_symbol_t s;
  fts_integer_vector_t *tw = 0;
  
  if (argv->ex_type != ET_SYM)
    {
      post("expr: store: need a table name\n");
    }

  s = (fts_symbol_t ) (argv++)->ex_ptr;

  tw = table_integer_vector_get_by_name(s);

  if (! tw)
    {
      optr->ex_type = ET_INT;
      optr->ex_int = 0;
      post("expr: store: no such table %s\n", fts_symbol_name(s));
      return;
    }

  if (argv->ex_type != ET_INT || argv[1].ex_type != ET_INT)
    {
      post("expr: store: arguments have to be integer\n");
      optr->ex_type = ET_INT;
      optr->ex_int = 0;
    }

  fts_integer_vector_set_element(tw, argv->ex_int < 0 ? 0 : argv->ex_int % fts_integer_vector_get_size(tw), argv[1].ex_int);
  *optr = argv[1]; 
#endif
}

