/***************************************************************************
 *                                                                         *
 *                         Powersave Daemon                                *
 *                                                                         *
 *          Copyright (C) 2004,2005 SUSE Linux Products GmbH               *
 *                                                                         *
 *               Author(s): Holger Macht <hmacht@suse.de>                  *
 *                                                                         *
 * This program is free software; you can redistribute it and/or modify it *
 * under the terms of the GNU General Public License as published by the   *
 * Free Software Foundation; either version 2 of the License, or (at you   *
 * option) any later version.                                              *
 *                                                                         *
 * This program is distributed in the hope that it will be useful, but     *
 * WITHOUT ANY WARRANTY; without even the implied warranty of              *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU       *
 * General Public License for more details.                                *
 *                                                                         *
 * You should have received a copy of the GNU General Public License along *
 * with this program; if not, write to the Free Software Foundation, Inc., *
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA                  *
 *                                                                         *
 ***************************************************************************/

#include <sys/wait.h>
#include <sys/time.h>

#include "globals.h"
#include "errno.h"
#include "event.h"
#include "config.h"

using namespace std;
using namespace __gnu_cxx;

Event::Event()
{
	_id = -1;
	_timeout = -1;
	_starttime = -1;
	_sleep_time = 0;
	_name = "default";
	_current_program_2_execute = _programs_2_execute.begin();
	setupInternalActions();
}

Event::Event(const string &n):_name(n)
{
	_id = -1;
	_timeout = -1;
	_starttime = -1;
	_sleep_time = 0;
	_current_program_2_execute = _programs_2_execute.begin();
	setupInternalActions();
	setActions2Execute("ignore");
}

Event::Event(const Event & e)
{
	_id = e._id;
	_starttime = e._starttime;
	_sleep_time = e._sleep_time;
	_timeout = e._timeout;
	_name = e._name;
	_programs_2_execute = e._programs_2_execute;
	_current_program_2_execute = _programs_2_execute.begin();
	setupInternalActions();
}

Event & Event::operator=(const Event & e)
{
	_id = e._id;
	_starttime = e._starttime;
	_sleep_time = e._sleep_time;
	_timeout = e._timeout;
	_name = e._name;
	_programs_2_execute = e._programs_2_execute;
	_current_program_2_execute = _programs_2_execute.begin();
	setupInternalActions();
	return *this;
}

Event::~Event()
{
	if (!_programs_2_execute.empty())
		_programs_2_execute.clear();

	if (!_internalActions.empty())
		_internalActions.clear();
}

void Event::setupInternalActions()
{
	_internalActions.push_back("ignore");
	_internalActions.push_back("throttle");
	_internalActions.push_back("dethrottle");
	_internalActions.push_back("do_suspend_to_disk");
	_internalActions.push_back("do_suspend_to_ram");
	_internalActions.push_back("do_standby");
	_internalActions.push_back("suspend_to_disk");
	_internalActions.push_back("suspend_to_ram");
	_internalActions.push_back("standby");
	_internalActions.push_back("screen_saver");
	_internalActions.push_back("reread_cpu_capabilities");
	_internalActions.push_back("suspend_networkmanager");
	_internalActions.push_back("resume_networkmanager");
}

string Event::name()
{
	return _name;
}

bool Event::execute_program(string acpi_event_line)
{
	pid_t pid;
	int x = 0;
	int eventID = ID();

	// set the program to execute to the current one
	string exec_script_path;
	string pub_script_dir;

	/* This event has no actions, needed to not segfault if empty */

	pub_script_dir = Powersave::Globals::config_obj->pub_script_dir;
	exec_script_path = pub_script_dir + "/" + *_current_program_2_execute;
	if (access(exec_script_path.c_str(), X_OK)) {
		return false;
	}

	pid = fork();
	//child
	if (pid == 0) {
		int argCount = 6;
		char *argv[argCount];
		string env = "";

		signal(SIGHUP, SIG_DFL);
		signal(SIGTERM, SIG_DFL);
		signal(SIGINT, SIG_DFL);
		signal(SIGQUIT, SIG_DFL);
		signal(SIGPIPE, SIG_DFL);
		signal(SIGCHLD, SIG_DFL);
		// close all fds
		int fdlimit = sysconf(_SC_OPEN_MAX);

		if (nice(_proxy_nice_factor) == -1) {
			pDebug(DBG_WARN,
			       "Could not set nice factor for proxy execution: %s",
			       strerror(errno));
		}

		/* don't close stdin, stdout and stderr for scripts. In
		 * daemon mode, they are redirected to /dev/null
		 * anyway. In non-daemon mode, the script's output is
		 * echoed to console. */
		for (int fd = 3; fd < fdlimit; fd++)
			close(fd);

		for (x = 0; x < argCount; x++)
			argv[x] = NULL;
		x = 0;

		pDebug(DBG_INFO,
		       "Going to execute external action %s",
		       (*_current_program_2_execute).c_str());

		/* first give name of the program as parameter */
		argv[x] = strdup((*_current_program_2_execute).c_str());
		x++;

		/* second: give name of happening event */
		argv[x] = strdup(_name.c_str());
		x++;

		/* third is the current scheme file name */
		argv[x] = strdup(Powersave::Globals::config_obj->current_scheme->file_name.c_str());
		x++;

		/* forth is acpi line */
		if (acpi_event_line == "") {
			acpi_event_line = "empty";
		}
		argv[x] = strdup(acpi_event_line.c_str());
		x++;

		/* fifth is the event id */
		argv[x] = new char[5];
		sprintf(argv[x], "%d", eventID);
		x++;

		argv[x] = NULL;

		pDebug(DBG_INFO,
		       "Program %s will be executed for event %s. Param 1: %s,"
		       "Param 2: %s",
		       exec_script_path.c_str(), _name.c_str(), argv[1], argv[2]);

		// set environment variables for scripts
		setenv("KDE_BINDIR", KDE_BINDIR, 1);
		setenv("GNOME_BINDIR", GNOME_BINDIR, 1);
		env = Powersave::Globals::config_obj->script_dir;
		setenv("SCRIPT_DIR", env.c_str(), 1);
		setenv("PUB_SCRIPT_DIR", pub_script_dir.c_str(), 1);
		env = Powersave::Globals::config_obj->config_dir;
		setenv("SYSCONF_DIR", env.c_str(), 1);
		setenv("BIN_DIR", BIN_DIR, 1);

		execv(exec_script_path.c_str(), argv);

		pDebug(DBG_ERR,
		       "Came back from subprocess when trying to execute %s for "
		       "event %s; Error: %s ",
		       exec_script_path.c_str(), _name.c_str(), strerror(errno));
		_exit(127);
	} else if (pid == -1) {	// fork failed
		pDebug(DBG_ERR, "Could not fork process: %s", strerror(errno));
		return false;
	} else {
		// parent
		// no need to wait for return of script because it returns
		// through the script_return_helper. So return to main loop and wait
		// for other events from acpi or sockets
	}
	return true;
}

void Event::setActions2Execute(list< string > actions)
{
	_programs_2_execute.clear();

	insertAction(actions, true, "prepare_suspend_to_disk", "suspend_networkmanager");
	insertAction(actions, false, "restore_after_suspend_to_disk", "resume_networkmanager");

	insertAction(actions, true, "prepare_suspend_to_ram", "suspend_networkmanager");
	insertAction(actions, false, "restore_after_suspend_to_ram", "resume_networkmanager");

	insertAction(actions, true, "prepare_standby", "suspend_networkmanager");
	insertAction(actions, false, "restore_after_standby", "resume_networkmanager");


	_programs_2_execute = actions;

}

void Event::setActions2Execute(const string &action)
{
	_programs_2_execute.clear();
	_programs_2_execute.push_front(action);
}

void Event::insertAction(list< string > &actions, bool before, const string &match, const string &action)
{
	list< string >::iterator it;

	if ((it = find(actions.begin(), actions.end(), match))
		 == actions.end()) {
		return;
	}

	if (before) {
		actions.insert(it, action);
	}

	if (it == --actions.end()) {
		actions.push_back(action);
	}
}


bool Event::isInternalAction()
{
	for (std::list< string >::iterator it = _internalActions.begin();
	     it != _internalActions.end(); ++it) {
		
		if (!_current_program_2_execute->compare(*it)) {
			return true;
		}
	}
	return false;
}

/*
---------------------------------------------------------------------
   the following is needed for managing events
---------------------------------------------------------------------
*/

unsigned long Event::timeElapsed()
{
	struct timeval cur_time;
	struct timezone dummy;
	gettimeofday(&cur_time, &dummy);

	// time elapsed in milliseconds
	unsigned long time_elapsed = cur_time.tv_sec * 1000 + cur_time.tv_usec /
		1000 - _starttime - _sleep_time;

	return time_elapsed;
}

void Event::setTimeout(unsigned long m_seconds)
{
	_timeout = m_seconds;
}

unsigned long Event::getTimeout()
{
	return _timeout;
}

void Event::setID(int i)
{
	_id = i;
}

int Event::ID()
{
	return _id;
}

list < string >::iterator Event::end()
{
	return _programs_2_execute.end();
}

void Event::start()
{
	struct timeval cur_time;
	struct timezone dummy;
	gettimeofday(&cur_time, &dummy);

	this->_starttime = cur_time.tv_sec * 1000 + cur_time.tv_usec / 1000;
}

void Event::increaseSleeptime(unsigned long time)
{
	_sleep_time += time;
}

string Event::currentAction()
{
	return *_current_program_2_execute;
}

/*
---------------------------------------------------------------------
   end of managing part
---------------------------------------------------------------------
*/

#ifdef CREATE_DEBUG_OUTPUT

ostream & operator<<(ostream & os, const Event & ev)
{
	copy(ev.programs_2_execute.begin(), ev.programs_2_execute.end(), ostream_iterator < string > (os));
	return os;
}

#endif
