/***************************************************************************
 *                                                                         *
 *                         Powersave Daemon                                *
 *                                                                         *
 *          Copyright (C) 2004,2005 SUSE Linux Products GmbH               *
 *                                                                         *
 *               Author(s): Holger Macht <hmacht@suse.de>                  *
 *                                                                         *
 * This program is free software; you can redistribute it and/or modify it *
 * under the terms of the GNU General Public License as published by the   *
 * Free Software Foundation; either version 2 of the License, or (at you   *
 * option) any later version.                                              *
 *                                                                         *
 * This program is distributed in the hope that it will be useful, but     *
 * WITHOUT ANY WARRANTY; without even the implied warranty of              *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU       *
 * General Public License for more details.                                *
 *                                                                         *
 * You should have received a copy of the GNU General Public License along *
 * with this program; if not, write to the Free Software Foundation, Inc., *
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA                  *
 *                                                                         *
 ***************************************************************************/

#include "throttle_interface.h"
#include "config_pm.h"
#include "event_management.h"
#include "globals.h"

using namespace Powersave::Globals;

ThrottleInterface::ThrottleInterface()
{
	_msecs_cpu_idle = 0;
	_cpu_is_idle = false;
	_event_state = NO_PROCESSOR_EVENT_THROWN;
	_throttling_supported = true;
}

bool ThrottleInterface::updateCPUState(EventManagement *eM)
{
	int cpu_load, ret;

	/* build the array of cpu load values
	   ideally, this should be done elsewhere... */
	ret = calcCPULoad(config_obj->current_scheme->CONSIDER_NICE);
	if (ret) {	// something went wrong
		pDebug(DBG_WARN, "calcCPULoad failed: %d", ret);
		return false;
	}
	
	// use all cpus. For determining if we want to throttle, this is sufficient
	cpu_load = getCPULoad(-1);
	/*******************************************/
	/* this checks whether CPU should get throttled ... ****** */
	/* CPUs get throttled if load of CPU stays below CPU_IDLE_LIMIT
	   longer than CPU_IDLE_TIMEOUT */
	if (config_obj->current_scheme && config_obj->current_scheme->ALLOW_THROTTLING
	    && config_obj->current_scheme->CPU_IDLE_TIMEOUT > 0) {
		if (cpu_load < config_obj->current_scheme->CPU_IDLE_LIMIT) {
			_cpu_is_idle = true;
			_msecs_cpu_idle += config_obj->current_scheme->POLL_INTERVAL;
		}
		else {
			_cpu_is_idle = false;
			if (_event_state == PROCESSOR_BUSY_THROWN)
				_msecs_cpu_idle = 0;
		}
		if (!_cpu_is_idle && 
		    (_event_state == NO_PROCESSOR_EVENT_THROWN || _event_state == PROCESSOR_IDLE_THROWN)) {
			// dethrottle CPU immediately if CPU is above CPU_BELOW_DYNAMIC usage limit
			// and a low or no processor event has been thrown

			// be careful! Do not throw a processor.busy event immediately after a processor.idle
			// event. The idle event may cause such high load (dbus signals, hal, clients)
			// that a processor.busy event maybe thrown --> this would result in a idle time
			// of only config_obj->current_scheme->POLL_INTERVAL
			if (_msecs_cpu_idle != 0) {
				eM->executeEvent("processor.busy");
				_event_state = PROCESSOR_BUSY_THROWN;
			}
		} else if (_cpu_is_idle &&
			   (_event_state == NO_PROCESSOR_EVENT_THROWN || _event_state == PROCESSOR_BUSY_THROWN)) {
			// count up idle time and send a throttle CPU when time limit
			// (config_obj->current_scheme->CPU_IDLE_TIMEOUT variable) is reached
			pDebug(DBG_DEBUG, "cpu_idle_time: %lu; timeout: %lu; low event already thrown : %d"
			       ", poll_Intervall: %lu; _cpu_is_idle: %d",
			       _msecs_cpu_idle, config_obj->current_scheme->CPU_IDLE_TIMEOUT * 1000, _event_state,
			       config_obj->current_scheme->POLL_INTERVAL, _cpu_is_idle);
			if (_msecs_cpu_idle > config_obj->current_scheme->CPU_IDLE_TIMEOUT * 1000) {
				eM->executeEvent("processor.idle");
				_msecs_cpu_idle = 0;
				_event_state = PROCESSOR_IDLE_THROWN;
			}
		}
	}
	/*******************************************************/
	return true;
}

void ThrottleInterface::restart()
{
	_msecs_cpu_idle = 0;
	_cpu_is_idle = false;
	_event_state = NO_PROCESSOR_EVENT_THROWN;
}

bool ThrottleInterface::throttle(int percent)
{
	if (!_throttling_supported || !config_obj->current_scheme->ALLOW_THROTTLING) {
		return false;
	}

	if (percent == -1)
		percent = config_obj->current_scheme->MAX_CPU_THROTTLING;
	if (setThrottlingPercent(percent) <= 0)
		_throttling_supported = false;
	else {
		pDebug(DBG_INFO, "CPU(s) throttled to %d percent", percent);
	}

	return _throttling_supported;
}

bool ThrottleInterface::dethrottle()
{
	if (_throttling_supported) {
		if (setThrottlingPercent(0) <= 0)
			_throttling_supported = false;
		else {
			pDebug(DBG_INFO, "CPU(s) dethrottled");
		}
	}
	
	return _throttling_supported;
}


void ThrottleInterface::disableThrottling() {
	_throttling_supported = false;
}
