# arch-tag: david@allouche.net - 2003-11-17 15:23:30 332029000
# Copyright (C) 2003 David Allouche <david@allouche.net>
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""
Internal module providing top-level pybaz package names

This module implements the top-level public interface for the
pybaz_ package. But for convenience reasons the author prefers
to store this code in a file separate from ``__init__.py``.

.. _pybaz: pybaz-module.html

This module is strictly internal and should never be used.

:var backend: See `pybaz.backend`

:var _arch: Internal deprecated interface to the backend
"""

### Package docstring ###

import os
import shutil
import itertools
import re
from sets import ImmutableSet
import errors
import util
from pathname import PathName, FileName, DirName
from deprecation import deprecated_usage, deprecated_callable
from _escaping import *
from _output import *
from _location import *

__all__ = []

def public(*names):
    __all__.extend(names)


### Backend abstraction and compatibility

class _BackendProxy(object):
    """Internal hack for compatibility with _arch"""
    def __getattribute__(self, name):
        return getattr(backend._module, name)
    def __setattr__(self, name, value):
        setattr(backend._module, name, value)

_arch = _BackendProxy()

from backends import commandline

backend = commandline.default_backend()

public('backend')


### Factory

public('Factory', 'factory')

def _check_param_type(name, value, type):
    if not isinstance(value, type):
        if type.__module__ == '__builtin__':
            typename = type.__name__
        else:
            typename = type.__module__ + '.' + type.__name__
        raise TypeError("%s must be a %s, but was: %r"
                        % (name, typename, value))


class Factory(object):

    """Abstract factory for objects created by the public interface.

    Eventually, it will possible to alter the type all objects created by
    PyBaz, system-wide (by assigning to `pybaz.factory`) or locally (by using a
    factory attribute in instances).

    Currently, this is only used internally to localise the cyclic dependencies
    between various internal modules.
    """

    def Archive(self, name):
        """Create an Archive.

        :param name: archive name
        :type name: str
        :rtype: `Archive`
        """
        _check_param_type('name', name, str)
        return Archive(name)

    def isArchive(self, obj):
        """Does the object implement the Archive interface?

        :rtype: bool
        """
        return isinstance(obj, Archive)

    def Version(self, name):
        """Create a Version.

        :param name: fully qualified name of the version.
        :type name: str
        :rtype: `Version`
        """
        _check_param_type('name', name, str)
        return Version(name)

    def isVersion(self, obj):
        """Does the object implement the Version interface?

        :rtype: bool
        """
        return isinstance(obj, Version)

    def Revision(self, name):
        """Create a Revision.

        :param name: fully qualified name of the revision
        :type name: str
        :rtype: `Revision`
        """
        _check_param_type('name', name, str)
        return Revision(name)

    def isRevision(self, obj):
        """Does the object implement the Revision interface?

        :rtype: bool
        """
        return isinstance(obj, Revision)

    def isSourceTree(self, obj):
        """Does the object implement the SourceTree interface?

        :rtype: bool
        """
        return isinstance(obj, ArchSourceTree)

    def ArchiveLocation(self, url):
        """Create an ArchiveLocation.

        :type url: str
        """
        _check_param_type('url', url, str)
        return ArchiveLocation(url)

    def isArchiveLocation(self, obj):
        """Does the object implement the ArchiveLocation interface?

        :rtype: bool
        """
        return isinstance(obj, ArchiveLocation)


factory = Factory()


### Internal helpers for namespace sanity checks

class _unsafe(tuple):
    """Used internally to disable namespace sanity checks"""


def _archive_param(archive):
    """Internal argument checking utility"""
    if isinstance(archive, Archive):
        return archive.name
    else:
        if not NameParser.is_archive_name(archive):
            raise errors.NamespaceError(archive, 'archive name')
        return archive

def _version_param(vsn):
    """Internal argument checking utility"""
    if isinstance(vsn, Version):
        return vsn.fullname
    else:
        p = NameParser(vsn)
        if not p.has_archive() or not p.is_version():
            raise errors.NamespaceError(vsn, 'fully-qualified version')
        return vsn

def _revision_param(rev):
    """Internal argument checking utility."""
    if isinstance(rev, Revision):
        return rev.fullname
    else:
        p = NameParser(rev)
        if not p.has_archive() or not p.has_patchlevel():
            raise errors.NamespaceError(rev, 'fully-qualified revision')
        return rev

def _version_revision_param(vsn_rev):
    """Internal argument checking utility"""
    if isinstance(vsn_rev, BranchItem):
        return vsn_rev.fullname
    else:
        p = NameParser(vsn_rev)
        if not p.has_archive() or not p.has_version():
            raise errors.NamespaceError(vsn_rev,
                                        'fully-qualified version or revision')
        return vsn_rev

def _package_revision_param(pkg_rev):
    """Internal argument checking utility"""
    if isinstance(pkg_rev, CategoryItem):
        return pkg_rev.fullname
    else:
        p = NameParser(pkg_rev)
        if not p.has_archive() or not p.has_package():
            raise errors.NamespaceError(pkg_rev,
                                        'fully-qualified package or revision')
        return pkg_rev

def _check_version_param(param, name):
    """Internal argument checking utility"""
    if not isinstance(param, Version):
        raise TypeError("Parameter \"%r\" must be a Version, but was: %r" \
                        % (name, param))

def _check_revision_param(param, name):
    """Internal argument checking utility"""
    if not isinstance(param, Revision):
        raise TypeError("Parameter \"%s\" must be a Revision"
                        " but was: %r" % (name, param))


### Base class for all namespace classes ###

public('NamespaceObject')

class NamespaceObject(object):

    """Base class for all archive objects."""

    def get_fullname(self):
        """Deprecated

        Fully qualified name of this namespace object.

        :rtype: str
        :see: `NamespaceObject.fullname`
        """
        raise NotImplementedError

    fullname = property(get_fullname, doc = """
    Fully qualfied name of this namespace object.

    :type: str
    """)

    def exists(self):
        """Does this namespace exists?

        Within the Arch model, history cannot be changed: created archive
        entries cannot be deleted. However, it is possible to ``unregister`` an
        archive, or to find references to archives whose location is not known.
        Thus, existence cannot always be decided. Testing for the existence of
        a name in a non-registered archive raises
        `errors.ArchiveNotRegistered`.

        :return: whether this namespace object exists.
        :rtype: bool
        :raise errors.ArchiveNotRegistered: the archive name is not registered,
            so existence cannot be decided.
        :raise errors.ExecProblem: there was a problem accessing the archive.
        """
        raise NotImplementedError

    def __repr__(self):
        """Fully-qualified name in angle brackets.

        :rtype: str
        """
        return '%s.%s(%r)' % (
            self.__class__.__module__, self.__class__.__name__, self.fullname)

    def __str__(self):
        """Fully-qualified name.

        Returns the value of the fullname attribute.

        :rtype: str
        """
        return self.fullname

    def __eq__(self, x):
        """Compare types and fully-qualified names.

        :return: wether objects have the same types and names.
        :rtype: bool
        """
        return type(self) is type(x) \
               and self.__class__ is x.__class__ \
               and self.fullname == x.fullname

    def __ne__(self, x):
        """Compare types and fully-qualified names.

        :return: whether objects have different types or names.
        :rtype: bool
        """
        return type(self) is not type(x) \
               or self.__class__ is not x.__class__ \
               or self.fullname != x.fullname


### Mixins for archive iteration aspect ###

public(
    'RevisionIterable',
    'VersionIterable',
    'BranchIterable',
    'CategoryIterable',
    )

class RevisionIterable(NamespaceObject):

    """Abstract class for namespace classes above Revision.

    RevisionIterable provides features which are common to all objects
    containing revisions.
    """

    def iter_revisions(self, reverse=False):
        """Iterate over archive revisions.

        :param reverse: reverse order, recent revisions first.
        :type reverse: bool
        :return: all existing revisions in this namespace.
        :rtype: iterable of `Revision`

        :precondition: `self.exists()` returns ``True``.
        """
        raise NotImplementedError

    def iter_library_revisions(self, reverse=False):
        """Iterate over library revisions.

        :param reverse: reverse order, recent revisions first.
        :type reverse: bool
        :return: revisions in this namespace which are present in the
            revision library.
        :rtype: iterable of `Revision`
        """
        raise NotImplementedError


class VersionIterable(RevisionIterable):

    """Abstract class for archive classes above Version.

    VersionIterable provides features which are common to all objects
    containing versions.
    """

    def iter_versions(self, reverse=False):
        """Iterate over archive versions.

        :param reverse: reverse order, higher versions first.
        :type reverse: bool
        :return: all existing versions in this namespace.
        :rtype: iterable of `Version`

        :precondition: `self.exists()` returns ``True``.
        """
        raise NotImplementedError

    def iter_library_versions(self, reverse=False):
        """Iterate over library revisions.

        :param reverse: reverse order, higher versions first.
        :type reverse: bool
        :return: versions in this namespace which are present in the
            revision library.
        :rtype: iterable of `Version`
        """
        raise NotImplementedError

    def iter_revisions(self, reverse=False):
        for v in self.iter_versions(reverse):
            for r in v.iter_revisions(reverse): yield r

    def iter_library_revisions(self, reverse=False):
        for v in self.iter_library_versions(reverse):
            for r in v.iter_library_revisions(reverse): yield r


class BranchIterable(VersionIterable):

    """Base class for archive classes above Branch.

    BranchIterable provides features which are common to all objects
    containing branches.
    """

    def iter_branches(self):
        """Iterate over archive branches.

        :return: all existing branches in this namespace.
        :rtype: iterable of `Branch`
        :precondition: `self.exists()` returns ``True``.
        """
        raise NotImplementedError

    def iter_library_branches(self):
        """Iterate over library branches.

        :return: branches in this namespace which are present in the
            revision library.
        :rtype: iterable of `Branch`
        """
        raise NotImplementedError

    def iter_versions(self, reverse=False):
        for b in self.iter_branches():
            for v in b.iter_versions(reverse): yield v

    def iter_library_versions(self, reverse=False):
        for b in self.iter_library_branches():
            for v in b.iter_library_versions(reverse): yield v


class CategoryIterable(BranchIterable):

    """Base class for Archive.

    CategoryIterable provides features for the aspect of Archive wich
    relates to its containing other archive objects.
    """

    def iter_categories(self):
        """Iterate over archive categories.

        :return: all existing categories in this namespace.
        :rtype: iterable of `Category`
        :precondition: `self.exists()` returns ``True``.
        """
        raise NotImplementedError

    def iter_library_categories(self):
        """Iterate over library categories.

        :return: categories in this namespace which are present in the
            revision library.
        :rtype: iterable of `Category`
        """
        raise NotImplementedError

    def iter_branches(self):
        for c in self.iter_categories():
            for b in c.iter_branches(): yield b

    def iter_library_branches(self):
        for c in self.iter_library_categories():
            for b in c.iter_library_branches(): yield b


### Base classes for archive containment aspect ###

public(
    'ArchiveItem',
    'CategoryItem',
    'BranchItem',
    'VersionItem',
    )

class ArchiveItem(NamespaceObject):

    """Base class for all archive components classes.

    ArchiveItem provides features common to all objects which are
    structural components of Archive.
    """

    def __init__(self, archive, nonarch):
        self._archive = archive
        self._nonarch = nonarch

    def get_archive(self):
        """Deprecated.

        Archive which contains this namespace object.

        :rtype: `Archive`
        :see: `ArchiveItem.archive`
        """
        deprecated_callable(self.get_archive, (type(self), 'archive'))
        return self.archive

    def _get_archive(self):
        return Archive(_unsafe((self._archive,)))

    archive = property(_get_archive, doc="""
    Archive which contains this namespace object.

    :type: `Archive`
    """)

    def get_nonarch(self):
        """Deprecated.

        Non-arch part of this namespace name.

        :rtype: str
        :see: `ArchiveItem.nonarch`
        """
        deprecated_callable(self.get_nonarch, (type(self), 'nonarch'))
        return self.nonarch

    def _get_nonarch(self):
        return self._nonarch

    nonarch = property(_get_nonarch, doc="""
    Non-arch part of this namespace name.

    :type: str
    """)

    def get_fullname(self):
        deprecated_callable(self.get_fullname, (type(self), 'fullname'))
        return self.fullname

    def _get_fullname(self):
        return '%s/%s' % (self._archive, self._nonarch)

    fullname = property(_get_fullname, doc=NamespaceObject.fullname.__doc__)


class CategoryItem(ArchiveItem):

    """Base class for archive classes below Category.

    CategoryItem provides features common to all objects which are
    structural components of Category.
    """

    def get_category(self):
        """Deprecated.

        Category which contains this namespace object.

        :rtype: `Category`
        :see: `CategoryItem.category`
        """
        deprecated_callable(self.get_category, (type(self), 'category'))
        return self.category

    def _get_category(self):
        return Category(_unsafe((self._archive, self._nonarch.split('--')[0])))

    category = property(_get_category, doc="""
    Category which contains this object.

    :type: `Category`
    """)


class BranchItem(CategoryItem):

    """Base class for archive classes Version and Revision.

    BranchItem provides features common to all objects which are
    structural components of Branch.
    """

    def get_branch(self):
        """Deprecated.

        Branch which contains this object.

        :rtype: `Branch`
        :see: `BranchItem.branch`
        """
        deprecated_callable(self.get_branch, (type(self), 'branch'))
        return self.branch

    def _get_branch(self):
        assert isinstance(self, Version)
        package = '--'.join(self._nonarch.split('--')[0:-1])
        return Branch(_unsafe((self._archive, package)))

    branch = property(_get_branch, doc="""
    Branch which contains this namespace object.

    :type: `Branch`
    """)


class VersionItem(BranchItem):

    """Base class for Revision.

    VersionItem provides features for the aspect of Revision which
    relate to its containment within other archive objects.
    """

    def __init__(self, archive, version, patchlevel):
        BranchItem.__init__(self, archive, "%s--%s" % (version, patchlevel))
        self._version, self._patchlevel = version, patchlevel

    def get_branch(self):
        deprecated_callable(self.get_branch, (type(self), 'branch'))
        return self.branch

    def _get_branch(self):
        assert isinstance(self, Revision)
        package = '--'.join(self._version.split('--')[0:-1])
        return Branch(_unsafe((self._archive, package)))

    branch = property(_get_branch, doc=BranchItem.branch.__doc__)

    def get_version(self):
        """Deprecated.

        Version which contains this revision.

        :rtype: `Version`
        :see: `VersionItem.version`
        """
        deprecated_callable(self.get_version, (type(self), 'version'))
        return self.version

    def _get_version(self):
        return Version(_unsafe((self._archive, self._version)))

    version = property(_get_version, doc="""
    Version which contains this revision.

    :type: `Version`
    """)

    def get_patchlevel(self):
        """Deprecated.

        Patch-level part of this object's name.

        :rtype: str
        :see: `VersionItem.patchlevel`
        """
        deprecated_callable(self.get_patchlevel, (type(self), 'patchlevel'))
        return self.patchlevel

    def _get_patchlevel(self):
        return self._patchlevel

    patchlevel = property(_get_patchlevel, doc="""
    Patch-level part of this object's name.

    :type: str
    """)


### Mixins for misc. features of archive objects ###

public('Setupable', 'Package')

class Setupable(ArchiveItem):

    """Base class for container archive objects."""

    def setup(self):
        """Deprecated.

        PyBaz 1.1 and later do archive-setup implicitly.
        """
        deprecated_callable(self.setup, because=
                            'pybaz 1.1 and later do archive-setup implicitly.')


class Package(Setupable, RevisionIterable):

    """Base class for ordered container archive objects."""

    def as_revision(self):
        """Deprecated.

        Latest revision in this package.

        :rtype: `Revision`
        :precondition: `self.exists()` returns ``True``
        :precondition: `self.iter_revisions()` yields at least one object.
        :raises StopIteration: this package contains no revision
        :see: `Package.latest_revision`
        """
        deprecated_callable(self.as_revision, self.latest_revision)
        return self.iter_revisions(reverse=True).next()

    def latest_revision(self):
        """Latest revision in this package.

        :rtype: `Revision`
        :precondition: `self.exists()` returns ``True``
        :precondition: `self.iter_revisions()` yields at least one object.
        :raises ValueError: the archive is not registered, or this
            package does not exist, or it contains no revision.
        """
        try:
            return self.iter_revisions(reverse=True).next()
        except errors.ExecProblem:
            try:
                self_exists = self.exists()
            except errors.ArchiveNotRegistered:
                raise ValueError('Archive is not registered: %s'
                                 % self.archive)
            if not self_exists:
                raise ValueError('Package does not exist: %s' % self)
            raise
        except StopIteration:
            raise ValueError('Package contains no revision: %s' % self)


### Archive classes ###

public('Archive', 'Category', 'Branch', 'Version', 'Revision')


class Archive(CategoryIterable):

    """Arch archive namespace object.

    In the Arch revision control system, archives are the units of
    storage. They store revisions organized in categories, branches
    and versions, and are associated to a `name` and a `location`.

    :see: `Category`, `Branch`, `Version`, `Revision`
    """

    def __init__(self, name):
        """Create an archive object from its registered name.

        :param name: archive name, like "jdoe@example.com--2003"
        :type name: str
        :raise errors.NamespaceError: invalid archive name.
        """
        if isinstance(name, Archive):
            name = name.__name
        if isinstance(name, _unsafe):
            assert len(name) == 1
            name = name[0]
        elif not NameParser.is_archive_name(name):
            raise errors.NamespaceError(name, 'archive name')
        self.__name = name
        self._impl = _archive_impl()

    def get_name(self):
        """Deprecated.

        Logical name of the archive.

        :rtype: str
        :see: `Archive.name`
        """
        deprecated_callable(Archive.get_name, (Archive, 'name'))
        return self.name

    def _get_name(self):
        return self.__name

    name = property(_get_name, doc="""
    Logical name of the archive.

    :type: str
    """)

    def get_fullname(self):
        deprecated_callable(Archive.get_fullname, (Archive, 'fullname'))
        return self.name

    fullname = property(_get_name, doc=NamespaceObject.fullname.__doc__)

    def get_location(self):
        """Deprecated.

        URI of the archive, specifies location and access method.

        :rtype: str
        :see: `Archive.all_locations`
        """
        deprecated_callable(Archive.get_location, Archive.all_locations)
        return self.location

    def _get_location(self):
        deprecated_callable((Archive, 'location'), Archive.all_locations)
        status, location = backend.status_one_cmd(
            ('whereis-archive', self.name), (0,1))
        if status == 0:
            return location
        else:
            return None

    location = property(_get_location, doc="""
    Deprecated.

    For example 'http://ddaa.net/arch/2004', or
    'sftp://user@sourcecontrol.net/public_html/2004'.

    :see: `Archive.all_locations`
    :type: str
    """)

    def all_locations(self):
        """All registered locations for this archive.

        :rtype: list of `ArchiveLocation`
        """
        args = ['whereis-archive', '--all-locations', self.name]
        try:
            raw_locations = [factory.ArchiveLocation(url)
                             for url in backend.sequence_cmd(args)]
        except errors.ExecProblem:
            if not self.is_registered():
                return []
            else:
                raise
        # may report duplicate locations with old-style registrations
        unique_locations = []
        for location in raw_locations:
            if location not in unique_locations:
                unique_locations.append(location)
        return unique_locations

    def _meta_info(self, key):
        archive = self.name
        args = self._impl.archive_meta_info_args(self.name, key)
        status, output = backend.status_one_cmd(args, expected=(0, 1))
        if status != 0:
            raise KeyError(
                "No such meta-info in archive %s: %s" % (archive, key))
        return output

    def _has_meta_info(self, key):
        try:
            self._meta_info(key)
            return True
        except KeyError:
            return False

    def get_is_mirror(self):
        """Deprecated.

        Is this archive registration a mirror?

        :see: `ArchiveLocation._meta_info_present`
        :rtype: bool
        """
        deprecated_callable(Archive.get_is_mirror,
                            ArchiveLocation._meta_info_present)
        return self.is_mirror

    def _get_is_mirror(self):
        deprecated_callable((Archive, 'is_mirror'),
                             ArchiveLocation._meta_info_present)
        return self._has_meta_info('mirror')

    is_mirror = property(_get_is_mirror, doc="""
    Deprecated.

    :see: `ArchiveLocation._meta_info_present`
    :type: bool
    """)

    def get_official_name(self):
        """Deprecated.

        Official archive name of this archive registration.

        :see: `ArchiveLocation._meta_info_present`
        :rtype: str
        """
        deprecated_callable(Archive.get_official_name,
                            ArchiveLocation._meta_info_present)
        return self.official_name

    def _get_official_name(self):
        deprecated_callable((Archive, 'official_name'),
                            ArchiveLocation._meta_info_present)
        return self._meta_info('name')

    official_name = property(_get_official_name, doc="""
    Deprecated.

    :see: `ArchiveLocation._meta_info_present`
    :type: str
    """)

    def get_is_signed(self):
        """Deprecated.

        Is the archive GPG-signed?

        :see: `ArchiveLocation._meta_info_present`
        :rtype: bool
        """
        deprecated_callable(Archive.get_is_signed,
                            ArchiveLocation._meta_info_present)
        return self.is_signed

    def _get_is_signed(self):
        deprecated_callable((Archive, 'is_signed'),
                            ArchiveLocation._meta_info_present)
        return self._has_meta_info('signed-archive')

    is_signed = property(_get_is_signed, doc="""
    Deprecated.

    :see: `ArchiveLocation._meta_info_present`
    :type: bool
    """)

    def get_has_listings(self):
        """Deprecated.

        Does the archive provide .listing file for http access?

        :see: `ArchiveLocation._meta_info_present`
        :rtype: bool
        """
        deprecated_callable(Archive.get_has_listings,
                            ArchiveLocation._meta_info_present)
        return self.has_listings

    def _get_has_listings(self):
        deprecated_callable((Archive, 'has_listings'),
                            ArchiveLocation._meta_info_present)
        return self._has_meta_info('http-blows')

    has_listings = property(_get_has_listings, doc="""
    Deprecated.

    :see: `ArchiveLocation._meta_info_present`
    :type: bool
    """)

    def _get_version_string(self):
        deprecated_callable((Archive, 'version_string'),
                            ArchiveLocation._version_string)
        location = '/'.join((self.location, '.archive-version'))
        import urllib
        version_file = urllib.urlopen(location)
        try:
            ret = version_file.read()
        finally:
            version_file.close()
        return ret.rstrip()

    version_string = property(_get_version_string, doc="""
    Deprecated.

    Contents of the ``.archive-version`` file at the root of the archive.

    :see: `ArchiveLocation._version_string`
    :type: str
    """)

    def _is_tla_format(self):
        expected = ('Hackerlab arch archive directory, format version 2.',
                    'Hackerlab arch archive directory, format version 1.')
        return self.version_string in expected

    def _is_baz_format(self):
        expected = 'Bazaar archive format 1 0'
        return self.version_string == expected

    def __getitem__(self, category):
        """Instanciate a Category belonging to this archive.

        :param category: unqualified category name
        :type category: str
        :rtype: `Category`
        """
        if not NameParser.is_category_name(category):
            raise errors.NamespaceError(category, 'unqualified category name')
        return Category(_unsafe((self.name, category)))

    def exists(self):
        if self.is_registered():
            return True
        else:
            raise errors.ArchiveNotRegistered(self.name)

    def is_registered(self):
        """Is this archive registered?

        :return: Whether the location associated to this registration name is
            known.
        :rtype: bool
        :see: `register_archive`, `Archive.unregister`
        """
        for archive in iter_archives():
            if archive.name == self.name:
                return True
        else:
            return False

    def iter_categories(self):
        args = self._impl.iter_categories_args(self.name)
        for c in  backend.sequence_cmd(args):
            yield Category(_unsafe((self.name, c)))

    def iter_location_versions(self, location):
        """Versions present in the specified archive location.

        :warning: This is a temporary facility that does no sanity checking. It
        will be removed shortly after bound namespace objects are properly
        implemented.
        """
        for category in backend.sequence_cmd(['categories', location.url]):
            category_url = location.url + '/' + category
            for branch in backend.sequence_cmd(['branches', category_url]):
                branch_url = location.url + '/' + branch
                for version in backend.sequence_cmd(['versions', branch_url]):
                    yield Version(_unsafe((self.name, version)))

    def iter_location_revisions(self, location):
        """Revisions present in the specified archive location.

        :warning: This is a temporary facility that does no sanity checking. It
        will be removed shortly after bound namespace objects are properly
        implemented.
        """
        for version in self.iter_location_versions(location):
            for revision in version.iter_location_revisions(location):
                yield revision

    def iter_library_categories(self):
        for c in _arch.library_categories(self.name):
            yield Category(_unsafe((self.name, c)))

    def get_categories(self):
        """Deprecated.

        Categories in this archive.

        :rtype: tuple of `Category`
        :see: `iter_categories`
        """
        deprecated_callable(Archive.get_categories, Archive.iter_categories)
        return tuple(self.iter_categories())

    def _get_categories(self):
        deprecated_callable((Archive, 'categories'), Archive.iter_categories)
        return tuple(self.iter_categories())

    categories = property(_get_categories, doc="""
        Deprecated.

        Categories in this archive.

        :type: tuple of `Category`
        :see: `iter_categories`
        """)

    def get_library_categories(self):
        """Deprecated.

        Categories in this archive  present in the library.

        :rtype: tuple of `Category`
        :see: `iter_library_categories`
        """
        deprecated_callable(Archive.get_library_categories,
                            Archive.iter_library_categories)
        return tuple(self.iter_library_categories())

    def _get_library_categories(self):
        deprecated_callable((Archive, 'library_categories'),
                            Archive.iter_library_categories)
        return tuple(self.iter_library_categories())

    library_categories = property(_get_library_categories, doc="""
    Deprecated.

    Categories in this archive  present in the library.

    :type; tuple of `Category`
    :see: `iter_library_categories`
    """)

    def unregister(self):
        """Unregister this archive.

        :precondition: `self.is_registered()`
        :postcondition: not `self.is_registered()`
        :see: `register_archive`
        """
        backend.null_cmd(('register-archive', '--delete', self.name))

    def make_mirror(self, name, location, signed=False, listing=False,
                    tla=False):
        """Deprecated.

        :see: `ArchiveLocation.create_mirror`

        :param name: name of the new mirror (for example
            'david@allouche.net--2003b-MIRROR').
        :type name: str
        :param location: writeable URI were to create the archive mirror.
        :type location: str
        :param signed: create GPG signatures for the mirror contents
        :type signed: bool
        :param listing: maintains ''.listing'' files to enable HTTP access.
        :type listing: bool
        :param tla: create a tla archive instead of a baz archive.
        :type tla: bool

        :return: object for the newly created archive mirror.
        :rtype: `Archive`

        :precondition: `self.is_registered()`
        :precondition: ``name`` is not a registered archive name
        :precondition: ``location`` does not exist and can be created
        :postcondition: Archive(name).is_registered()

        :raise errors.NamespaceError: ``name`` is not a valid archive name.
        """
        deprecated_callable(Archive.make_mirror, ArchiveLocation.create_mirror)
        a = Archive(name) # check archive name validity first
        official_name = self.official_name
        self._impl.make_mirror(official_name, name, location,
                               signed, listing, tla)
        return a

    def mirror(self, limit=None, fromto=None,
               no_cached=False, cached_tags=False):
        """Deprecated.

        :see: `ArchiveLocation.make_mirrorer`

        :param limit: restrict mirrorring to those archive items. All items
            must belong to this archive.
        :type limit: iterable of at least one ArchiveItem or str

        :param fromto: update the mirror specified by the second item with the
            contents of the archive specified by the first item.
        :type fromto: sequence of exactly two Archive or str.

        :precondition: If ``fromto`` is provided, both items must be registered
            archives names whose official name is this archive.

        :param no_cached: do not copy cached revisions.
        :type no_cached: bool

        :param cached_tags: copy only cachedrevs for tags to other archives.
        :type cached_tags: bool
        """
        deprecated_callable(Archive.mirror, ArchiveLocation.make_mirrorer)
        assert no_cached + cached_tags < 2
        assert limit is None or len(limit) > 0
        if fromto is None:
            from_, to = self.name, None
        else:
            from_, to = self._impl.mirror_fromto(fromto)
        args = ['archive-mirror']
        if no_cached:
            args.append('--no-cached')
        if cached_tags:
            args.append('--cached-tags')
        args.append(from_)
        if to is not None:
            args.append(to)
        if limit is not None:
            args.extend([self._archive_limit_param(L) for L in limit])
        backend.null_cmd(args)

    def _archive_limit_param(self, item):
        """Internal argument checking utility"""
        if isinstance(item, ArchiveItem):
            if self != item.archive:
                message = "%s is not an item of %s" % (item.fullname, self.name)
                raise ValueError(message)
            return item.nonarch
        else:
            p = NameParser(item)
            if not p.has_category():
                raise errors.NamespaceError(item, 'archive limit')
            if not p.has_archive():
                return item
            elif self.name != p.get_archive():
                message = "%s is not an item of %s" % (item.fullname, self.name)
                raise ValueError(message)
            else:
                return p.get_nonarch()


def _archive_impl():
    if backend.version.release < (1, 3, 0):
        return _Archive_Baz_1_0
    else:
        return _Archive_Baz_1_3


class _Archive_Baz_1_0(object):

    def archive_meta_info_args(archive, key):
        return ('archive-meta-info', '--archive', archive, key)

    archive_meta_info_args = staticmethod(archive_meta_info_args)

    def register_archive(name, location):
        args = ['register-archive']
        if name is not None:
            args.append(name)
        args.append(location)
        backend.null_cmd(args)
        if name is not None:
            return Archive(_unsafe((name,)))
        else:
            for archive in iter_archives():
                if archive.location == location:
                    return archive
            raise AssertionError, 'not reached'

    register_archive = staticmethod(register_archive)

    def make_mirror(master, name, location, signed, listing, tla):
        args = ['make-archive', '--mirror', master]
        if signed:
            args.append('--signed')
        if listing:
            args.append('--listing')
        if tla:
            args.append('--tla')
        args.extend((name, location))
        backend.null_cmd(args)

    make_mirror = staticmethod(make_mirror)

    def iter_categories_args(name):
        return ('categories', '--archive', name)

    iter_categories_args = staticmethod(iter_categories_args)

    def _registered_archive_param(archive):
        """Internal argument checking utility"""
        archive = _archive_param(archive)
        if not Archive(_unsafe((archive,))).is_registered():
            raise errors.ArchiveNotRegistered(archive)
        return archive

    _registered_archive_param = staticmethod(_registered_archive_param)

    def mirror_fromto(fromto):
        return [_Archive_Baz_1_0._registered_archive_param(A) for A in fromto]

    mirror_fromto = staticmethod(mirror_fromto)


class _Archive_Baz_1_3(object):

    def archive_meta_info_args(archive, key):
        return ('archive-meta-info', archive + '/' + key)

    archive_meta_info_args = staticmethod(archive_meta_info_args)

    def _is_url(location):
        match = re.match('[^/]+://', location)
        return match is not None

    _is_url = staticmethod(_is_url)

    def _absolutize(cls, location):
        if cls._is_url(location):
            return location
        return os.path.abspath(location)

    _absolutize = classmethod(_absolutize)

    def _canonical_location_unregister(official_name, absolute_location):
        # Figure out the full location by examining the registration and
        # unregister the mirror location in the process. The location must be
        # unregistered because we will create an old-style registration and it
        # should not be redundant.
        args = ['whereis-archive', '--all-locations', official_name]
        import sets
        locations_with = sets.Set(backend.sequence_cmd(args))
        args = ['register-archive', '--delete', absolute_location]
        backend.null_cmd(args)
        if len(locations_with) == 1:
            difference = locations_with
        else:
            args = ['whereis-archive', '--all-locations', official_name]
            locations_without = sets.Set(backend.sequence_cmd(args))
            difference = locations_with - locations_without
        assert len(difference) == 1
        canonical_location = difference.pop().rstrip('\n')
        return canonical_location

    _canonical_location_unregister = staticmethod(
        _canonical_location_unregister)

    def _register_name(name, canonical_location):
        home = DirName(os.environ['HOME'])
        locations_dir = home/'.arch-params'/'=locations'
        if not os.path.isdir(locations_dir):
            os.mkdir(locations_dir)
        print >> open(locations_dir/name, 'w'), canonical_location

    _register_name = staticmethod(_register_name)

    def _location_meta_info(location, label):
        args = ['archive-meta-info', location + '/' + label]
        return backend.text_cmd(args).strip()

    _location_meta_info = staticmethod(_location_meta_info)

    def register_archive(cls, name, location):
        absolute_location = cls._absolutize(location)
        backend.null_cmd(['register-archive', location])
        official_name = cls._location_meta_info(absolute_location, 'name')
        if name is None:
            return Archive(_unsafe((official_name,)))
        if name != official_name:
            # given name is not official name, create registered name
            canonical_loc = cls._canonical_location_unregister(
                official_name, absolute_location)
            cls._register_name(name, canonical_loc)
        return Archive(_unsafe((name,)))

    register_archive = classmethod(register_archive)

    def make_mirror(cls, master, name, location, signed, listing, tla):
        absolute_location = cls._absolutize(location)
        _Archive_Baz_1_0.make_mirror(master, name, absolute_location,
                                     signed, listing, tla)
        canonical_location = cls._canonical_location_unregister(
            master, absolute_location)
        cls._register_name(name, canonical_location)

    make_mirror = classmethod(make_mirror)

    def iter_categories_args(name):
        return ('categories', name)

    iter_categories_args = staticmethod(iter_categories_args)

    def mirror_fromto(fromto):
        return [Archive(_unsafe((A,))).location
                for A in _Archive_Baz_1_0.mirror_fromto(fromto)]

    mirror_fromto = staticmethod(mirror_fromto)


class Category(Setupable, BranchIterable):

    """Arch category namespace object.

    :see: `Archive`, `Branch`, `Version`, `Revision`
    """

    def __init__(self, name):
        """Create a category object from its name.

        :param name: fully-qualified category name, like
          "jdoe@example.com/frob"
        :type name: str
        :raise errors.NamespaceError: ``name`` is not a valid category name.
        """
        if isinstance(name, Category):
            name = (name._archive, name._nonarch)
        elif not isinstance(name, _unsafe):
            p = NameParser(name)
            if not p.has_archive() or not p.is_category():
                raise errors.NamespaceError(name, 'fully-qualified category')
            name = (p.get_archive(), p.get_nonarch())
        ArchiveItem.__init__(self, *name)

    def __getitem__(self, b):
        """Instanciate a branch belonging to this category.

        For example ``Category('jdoe@example.com/frob')['devel']`` is
        equivalent to ``Branch('jdoe@example.com/frob--devel')``.

        :param b: branch id.
        :type b: str
        :rtype: `Category`

        :raise NamespaceError: argument is not a valid branch id.
        """
        if b is not None and not NameParser.is_branch_name(b):
            raise errors.NamespaceError(b, 'unqualified branch name')
        if b is None:                   # nameless branch
            return Branch(_unsafe((self._archive, self._nonarch)))
        else:                           # named branch
            return Branch(_unsafe((self._archive,
                                   "%s--%s" % (self._nonarch, b))))

    def exists(self):
        if not self.archive.is_registered():
            raise errors.ArchiveNotRegistered(self.archive)
        for category in self.archive.iter_categories():
            if category.fullname == self.fullname:
                return True
        else:
            return False

    def iter_branches(self):
        for b in backend.sequence_cmd(('branches', self.fullname)):
            yield Branch(_unsafe((self._archive, b)))

    def iter_library_branches(self):
        for b in _arch.library_branches(self.fullname):
            yield Branch(_unsafe((self._archive, b)))

    def get_branches(self):
        """Deprecated.

        Branches in this category.

        :rtype: tuple of `Branch`
        :see: `iter_branches`
        """
        deprecated_callable(self.get_branches, self.iter_branches)
        return tuple(self.iter_branches())

    def _get_branches(self):
        deprecated_callable((type(self), 'branches'), self.iter_branches)
        return tuple(self.iter_branches())

    branches = property(_get_branches, doc="""
    Deprecated.

    Branches in this category.

    :type: tuple of `Branch`
    :see: `Category.iter_branches`
    """)

    def get_library_branches(self):
        """Deprecated.

        Branches in this category present in the library.

        :rtype: tuple of `Branch`
        :see: `iter_library_branches`
        """
        deprecated_callable(self.get_library_branches,
                            self.iter_library_branches)
        return tuple(self.iter_library_branches())

    def _get_library_branches(self):
        deprecated_callable((Category, 'library_branches'),
                            self.iter_library_branches)
        return tuple(self.iter_library_branches())

    library_branches = property(_get_library_branches, doc="""
    Deprecated.

    Branches in this category present in the library.

    :type: tuple of `Branch`
    :see: `Category.iter_branches`
    """)


class Branch(CategoryItem, Package, VersionIterable):

    """Arch branch namespace object.

    :see: `Archive`, `Category`, `Version`, `Revision`
    """

    def __init__(self, name):
        """Create a Branch object from its name.

        :param name: fully-qualified branch name, like
            "jdoe@example.com--2004/frob--devo" or
            "jdoe@example.com--2004/frob".
        :type name: str
        :raise errors.NamespaceError: ``name`` is not a valid branch name.
        """
        if isinstance(name, Branch):
            name = (name._archive, name._nonarch)
        elif not isinstance(name, _unsafe):
            p = NameParser(name)
            if not p.has_archive() or not p.is_package():
                raise errors.NamespaceError(name, 'fully-qualified branch')
            assert p.is_package()
            name = (p.get_archive(), p.get_nonarch())
        CategoryItem.__init__(self, *name)

    def __getitem__(self, v):
        """Instanciate a version belonging to this branch.

        For example ``Branch('jdoe@example.com/frob--devel')['0']`` is
        equivalent to ``Branch('jdoe@example.com/frob--devel--0')``.

        :param v: branch id.
        :type v: str
        :rtype: `Version`

        :raise NamespaceError: argument is not a valid version id.
        """
        if not NameParser.is_version_id(v):
            raise errors.NamespaceError(v, 'version id')
        return Version(_unsafe((self._archive, "%s--%s" % (self._nonarch, v))))

    def exists(self):
        if not self.category.exists():
            return False
        for branch in self.category.iter_branches():
            if branch.fullname == self.fullname:
                return True
        else:
            return False

    def iter_versions(self, reverse=False):
        args = ['versions']
        if reverse:
            args.append('--reverse')
        args.append(self.fullname)
        for v in backend.sequence_cmd(args):
            yield Version(_unsafe((self._archive, v)))

    def iter_library_versions(self, reverse=False):
        for v in _arch.library_versions(self.fullname, reverse):
            yield Version(_unsafe((self._archive, v)))

    def get_versions(self, reverse=False):
        """Deprecated.

        Versions in this branch.

        :rtype: tuple of `Version`
        :see: `iter_versions`
        """
        deprecated_callable(self.get_versions, self.iter_versions)
        return tuple(self.iter_versions(reverse))

    def _get_versions(self):
        deprecated_callable((Branch, 'versions'), self.iter_versions)
        return tuple(self.iter_versions(reverse=False))

    versions = property(_get_versions, doc="""
    Deprecated.

    Versions in this branch.

    :type: tuple of `Version`
    :see: `iter_versions`
    """)

    def get_library_versions(self, reverse=False):
        """Deprecated.

        Versions in this branch present in the library.

        :rtype: tuple of `Version`
        :see: `iter_library_versions`
        """
        deprecated_callable(self.get_library_versions,
                            self.iter_library_versions)
        return tuple(self.iter_library_versions(reverse))

    def _get_library_versions(self):
        deprecated_callable((Branch, 'library_versions'),
                            self.iter_library_versions)
        return tuple(self.iter_library_versions(reverse=False))

    library_versions = property(_get_library_versions, doc="""
    Deprecated.

    Versions in this branch present in the library.

    :type: tuple of `Version`
    :see: `iter_library_versions`
    """)

    def as_version(self):
        """Deprecated.

        Latest version in this branch.

        :rtype: `Version`
        :precondition: `self.exists()` returns ``True``
        :precondition: `self.iter_versions` yields at least one object.
        :raise IndexError: this branch is empty.
        :see: `latest_version`
        """
        deprecated_callable(self.as_version, self.latest_version)
        return list(self.iter_versions(reverse=True))[0]

    def latest_version(self):
        """Latest version in this branch.

        :rtype: `Version`
        :precondition: `self.exists()` returns ``True``
        :precondition: `self.iter_versions` yields at least one object.
        :raise ValueError: the archive is not registered, or this branch does
            not exist, or it contains no version.
        """
        try:
            return self.iter_versions(reverse=True).next()
        except errors.ExecProblem:
            try:
                self_exists = self.exists()
            except errors.ArchiveNotRegistered:
                raise ValueError('Archive is not registered: %s'
                                 % self.archive)
            if not self_exists:
                raise ValueError('Branch does not exist: %s' % self)
            raise
        except StopIteration:
            raise ValueError('Branch contains no version: %s' % self)



class Version(BranchItem, Package, RevisionIterable):

    """Arch version namespace object.

    :see: `Archive`, `Category`, `Branch`, `Revision`
    """

    def __init__(self, name):
        """Create a Version object from its name.

        :param name: fully-qualified version name, like
           "jdoe@example.com--2004/frob--devo--1.2".
        :type name: str

        :note: Nameless branches have no "branch" part in their name.
        """
        if isinstance(name, Version):
            name = (name._archive, name._nonarch)
        elif not isinstance(name, _unsafe):
            p = NameParser(name)
            if not p.has_archive() or not p.is_version():
                raise errors.NamespaceError(name, 'fully-qualified version')
            name = (p.get_archive(), p.get_nonarch())
        BranchItem.__init__(self, *name)
        self._impl = _Namespace_Baz_1_0

    def __getitem__(self, idx):
        """Instanciate a revision belonging to this version.

        Given a string, instanciate the revision of this version with the given
        patchlevel. For example
        ``Version('jdoe@example.com/frob--devel--0')['patch-1']`` is equivalent
        to ``Revision('jdoe@example.com/frob--devel--0--patch-1')``.

        Given an integer, instanciate the *existing* revision of this version
        with the given index. For example, if ``version`` contains at least one
        revision, ``version[0]`` is equivalent to ``version['base-0']``, and
        ``version[-1]`` is equivalent to ``version.latest_revision()``.

        :param idx: patch level, or revision number
        :type idx: str, int
        :rtype: `Revision`

        :raise NamespaceError: argument is a string not a valid version
            patchlevel.
        :raise ValueError: argument is an integer and the version contains no
            revision with that index.
        """
        if isinstance(idx, str):
            return self._getitem_str(idx)
        if isinstance(idx, int):
            return self._getitem_int(idx)
        else:
            raise TypeError(
                'Version indices must be str or int, but was %r' % (idx,))

    def _getitem_str(self, lvl):
        if not NameParser.is_patchlevel(lvl):
            raise errors.NamespaceError(lvl, 'unqualified patchlevel')
        return Revision(_unsafe((self._archive, self._nonarch, lvl)))

    def _getitem_int(self, idx):
        if idx < 0:
            reverse = True
            decounter = - idx - 1
        else:
            reverse = False
            decounter = idx
        assert decounter >= 0
        rev_iter = self.iter_revisions(reverse)
        try:
            while decounter > 0:
                decounter -= 1
                rev_iter.next()
            return rev_iter.next()
        except StopIteration:
            raise IndexError(
                'Version index out of bounds: %r[%d]' % (self, idx))

    def as_version(self):
        """Deprecated.

        This version.

        :rtype: `Version`
        """
        deprecated_callable(self.as_version, because='Foolish consistency.')
        return self

    def exists(self):
        if not self.branch.exists():
            return False
        for version in self.branch.iter_versions():
            if version.fullname == self.fullname:
                return True
        else:
            return False

    def get(self, dir, link=False):
        """Construct a project tree for this version.

        Extract the latest revision for this version from the archive. That is
        a shortcut for ``version.latest_revision.get(dir)``. It can be
        susceptible to race conditions when a concurrent transaction occurs on
        the same version, yielding a latter revision that what you may have
        meant.

        :param dir: path of the project tree to create. Must not
            already exist.
        :type dir: str
        :param link: hardlink files to revision library instead of copying
        :type link: bool
        :return: newly created project tree.
        :rtype: `WorkingTree`
        """
        args = self._impl.get_args(self, dir, link)
        backend.null_cmd(args)
        return WorkingTree(dir)

    def iter_revisions(self, reverse=False):
        args = ['revisions']
        if reverse:
            args.append('--reverse')
        args.append(self.fullname)
        for lvl in backend.sequence_cmd(args):
            yield Revision(_unsafe((self._archive, self._nonarch, lvl)))

    def iter_location_revisions(self, location):
        """Revisions present in this version at specified archive location.

        :warning: This is a temporary facility that does no sanity checking. It
        will be removed shortly after bound namespace objects are properly
        implemented.
        """
        version_url = location.url + '/' + self.nonarch
        for level in backend.sequence_cmd(['revisions', version_url]):
            yield Revision(_unsafe((self._archive, self._nonarch, level)))

    def iter_library_revisions(self, reverse=False):
        for lvl in _arch.library_revisions(self.fullname, reverse):
            yield Revision(_unsafe((self._archive, self._nonarch, lvl)))

    def get_revisions(self, reverse=False):
        """Deprecated.

        Revisions in this version.

        :rtype: tuple of `Revision`
        :see: `iter_revisions`
        """
        deprecated_callable(self.get_revisions, self.iter_revisions)
        return tuple(self.iter_revisions(reverse))

    def _get_revisions(self):
        deprecated_callable((Version, 'revisions'), self.iter_revisions)
        return tuple(self.iter_revisions(reverse=False))

    revisions = property(_get_revisions, doc="""
    Deprecated.

    Revisions in this version.

    :type: tuple of `Revision`
    :see: `iter_revisions`
    """)

    def get_library_revisions(self, reverse=False):
        """Deprecated.

        Revisions in this version present in the library.

        :rtype: tuple of `Revision`
        :see: `iter_library_revisions`
        """
        deprecated_callable(self.get_library_revisions,
                            self.iter_library_revisions)
        return tuple(self.iter_library_revisions(reverse))

    def _get_library_revisions(self):
        deprecated_callable((Version, 'library_revisions'),
                            self.iter_library_revisions)
        return tuple(self.iter_library_revisions(reverse=False))

    library_revisions = property(_get_library_revisions, doc="""
    Deprecated.

    Revisions in this version present in the library.

    :type: tuple of `Revision`
    :see: `iter_library_revisions`
    """)

    def iter_merges(self, other=None, reverse=False, metoo=True):
        """Iterate over merge points in this version.

        This method is mostly useful to save multiple invocations of
        the command-line tool and multiple connection to a remote
        archives when building an ancestry graph. Ideally, it would
        not be present and the desired merge graph traversal would be
        done using the new_patches and merges properties of Patchlog
        objects.

        :param other: list merges with that version.
        :type other: `Version`
        :param reverse: reverse order, recent revisions first.
        :type reverse: bool
        :param metoo: do not report the presence of a patch within itself
        :type metoo: bool

        :return: Iterator of tuples (R, T) where R are revisions in this
            version and T are iterable of revisions in the ``other`` version.
        :rtype: iterable of `Revision`
        """
        if other is None: othername = None
        else: othername = other.fullname
        flow = _arch.iter_merges(self.fullname, othername, reverse, metoo)
        last_target = None
        sources = []
        for target, source in flow:
            if last_target is None:
                last_target = target
            if target == last_target:
                sources.append(source)
            else:
                yield self[last_target], itertools.imap(Revision, sources)
                last_target = target
                sources = [source]
        yield self[last_target], itertools.imap(Revision, sources)

    def iter_cachedrevs(self):
        """Iterate over the cached revisions in this version.

        :rtype: iterator of `Revision`
        """
        for rev in _arch.cachedrevs(self.fullname):
            lvl = rev.split('--')[-1]
            yield self[lvl]


class Revision(VersionItem):

    """Arch revision namespace object.

    :see: `Archive`, `Category`, `Branch`, `Version`
    :group Libray Methods: library_add, library_remove, library_find
    :group History Methods: get_ancestor, get_previous, iter_ancestors
    """

    def __init__(self, name):
        """Create a Revision object from its name.

        :param name: fully-qualified revision, like
            "jdoe@example.com--2004/frob--devo--1.2--patch-2".
        :type name: str

        :note: Nameless branches have no "branch" part in their name.
        """
        if isinstance(name, Revision):
            name = (name._archive, name._version, name._patchlevel)
        elif not isinstance(name, _unsafe):
            p = NameParser(name)
            if not p.has_archive() or not p.has_patchlevel():
                raise errors.NamespaceError(name, 'fully-qualified revision')
            name = (p.get_archive(), p.get_package_version(),
                    p.get_patchlevel())
        VersionItem.__init__(self, *name)
        self.__patchlog = Patchlog(self)
        self._impl = _revision_impl()

    def as_revision(self):
        """Deprecated

        Returns this revision. For consistency with `Package.as_revision()`.

        :rtype: `Revision`
        """
        deprecated_callable(self.as_revision, because='Foolish consistency.')
        return self

    def exists(self):
        try:
            for revision in self.version.iter_revisions():
                if revision.fullname == self.fullname:
                    return True
            else:
                return False
        except Exception:
            if self.version.exists():
                raise
            else:
                return False

    def get(self, dir, link=False):
        """Construct a project tree for this revision.

        Extract this revision from the archive.

        :param dir: path of the project tree to create. Must not
            already exist.
        :type dir: str
        :param link: hardlink files to revision library instead of copying
        :type link: bool
        :return: newly created project tree.
        :rtype: `WorkingTree`
        """
        args = self._impl.get_args(self, dir, link)
        backend.null_cmd(args)
        return WorkingTree(dir)

    def get_patch(self, dir):
        """Fetch the changeset associated to this revision.

        :param dir: name of the changeset directory to create. Must
            not already exist.
        :type dir: str
        :return: changeset associated to this revision.
        :rtype: `Changeset`
        """
        _arch.get_patch(self.fullname, dir)
        return Changeset(dir)

    def get_patchlog(self):
        """Deprecated.

        Patchlog associated to this revision.

        :rtype: `Patchlog`
        :see: `Revision.patchlog`
        """
        deprecated_callable(self.get_patchlog, (Revision, 'patchlog'))
        return self.patchlog

    def _get_patchlog(self):
        return self.__patchlog

    patchlog = property(_get_patchlog, doc="""
    Patchlog associated to this revision.

    The `Patchlog` object is created in `__init__`, since log parsing is
    deferred that has little overhead and avoid parsing the log for a given
    revision several times. The patchlog data is read from the archive.

    :type: `Patchlog`
    :see: `ArchSourceTree.iter_logs`
    """)

    def make_continuation(self, target):
        """Create a continuation of this revision in the target version.

        :param target: version to create a continuation into. If it does not
            exist yet, it is created.
        :type target: Version
        """
        _check_version_param(target, 'target')
        target_name = target.fullname
        _arch.tag(self.fullname, target_name)

    def library_add(self):
        """Add this revision to the library.

        :postcondition: self in self.version.iter_library_revisions()
        """
        _arch.library_add(self.fullname)

    def library_remove(self):
        """Remove this revision from the library.

        :precondition: self in self.version.iter_library_revisions()
        :postcondition: self not in self.version.iter_library_revisions()
        """
        _arch.library_remove(self.fullname)

    def library_find(self):
        """The copy of this revision in the library.

        :rtype: `LibraryTree`
        :precondition: self in self.version.iter_library_revisions()
        """
        return LibraryTree(_arch.library_find(self.fullname))

    #def library_file(self, file):
    #    raise NotImplementedError, "library_file is not yet implemented"

    def get_ancestor(self):
        """Deprecated.

        Parent revision.

        :return:
            - The previous namespace revision, if this revision is regular
              commit.
            - The tag origin, if this revision is a continuation
            - ``None`` if this revision is an import.

        :rtype: `Revision` or None
        :see: `Revision.ancestor`
        """
        deprecated_callable(self.get_ancestor, (Revision, 'ancestor'))
        return self.ancestor

    def _get_ancestor(self):
        args = ['ancestry-graph', '--immediate', self.fullname]
        # ancestry-graph --immediate always gives a fully qualified revision id
        revision_id = backend.one_cmd(args)
        if revision_id == '(null)':
            return None
        return Revision(revision_id)

    ancestor = property(_get_ancestor, doc="""
    Parent revision.

    - The previous namespace revision, if this revision is regular commit.
    - The tag origin, if this revision is a continuation
    - ``None`` if this revision is an import.

    :type: `Revision` or None
    """)

    def get_previous(self):
        """Deprecated.

        Previous namespace revision.

        :return: the previous revision in the same version, or None if this
            revision is a ``base-0``.
        :rtype: `Revision` or None
        :see: `Revision.previous`
        """
        deprecated_callable(self.get_previous, (Revision, 'previous'))
        return self.previous

    def _get_previous(self):
        args = ['ancestry-graph', '--previous', self.fullname]
        previous = backend.one_cmd(args)
        if previous in ('(null)', '<<<no previous revision>>>'):
            return None
        revision_id = self._impl.revision_from_previous(self.archive.name,
                                                        previous)
        return Revision(revision_id)

    previous = property(_get_previous, doc="""
    Previous namespace revision.

    The previous revision in the same version, or None if this revision is a
    ``base-0``.

    :type: `Revision` or None
    """)

    def iter_ancestors(self, metoo=False):
        """Ancestor revisions.

        :param metoo: yield ``self`` as the first revision.
        :type metoo: bool
        :return: all the revisions in that line of development.
        :rtype: iterable of `Revision`
        """
        args = ['ancestry-graph']
        args.append(self.fullname)
        lines = iter(backend.sequence_cmd(args))
        if not metoo:
            lines.next() # the first revision is self
        for line in lines:
            ancestor, merge = line.split('\t')
            yield Revision(ancestor)

    def cache(self, cache=None):
        """Cache a full source tree for this revision in its archive.

        :param cache: cache root for trees with pristines.
        :type cache: bool
        """
        _arch.cacherev(self.fullname, cache)

    def uncache(self):
        """Remove the cached tree of this revision from its archive."""
        _arch.uncacherev(self.fullname)

    def _tla_file_url(self, name):
        return '/'.join([self.archive.location, self.category.nonarch,
                         self.branch.nonarch, self.version.nonarch,
                         self.patchlevel, name])

    def _baz_file_url(self, name):
        return '/'.join([self.archive.location, self.version.nonarch,
                         self.patchlevel, name])

    def _file_url(self, name):
        if self.archive._is_tla_format():
            return self._tla_file_url(name)
        if self.archive._is_baz_format():
            return self._baz_file_url(name)
        raise AssertionError, \
              'did not recognize archive version of %s' % self.archive

    _checksum_regex = re.compile(
        "^Signature-for: (.*)/(.*)\n"
        "([a-zA-Z0-9]+ [^/\\s]+ [a-fA-F0-9]+\n)*",
        re.MULTILINE)

    def _parse_checksum(self, text):
        match = self._checksum_regex.search(text)
        checksum_body = match.group()
        checksum_lines = checksum_body.strip().split('\n')
        checksum_words = map(lambda x: x.split(), checksum_lines)
        assert checksum_words[0] == ['Signature-for:', self.fullname]
        del checksum_words[0]
        checksums = {}
        for algo, name, value in checksum_words:
            name_sums = checksums.setdefault(name, dict())
            name_sums[algo] = value
        return checksums

    def iter_files(self):
        """Files stored in the archive for that revision.

        :rtype: iterable of `RevisionFile`
        """
        import urllib
        for checksum_name, can_fail in \
                (('checksum', False), ('checksum.cacherev', True)):
            try:
                checksum_file = urllib.urlopen(self._file_url(checksum_name))
            except IOError:
                if can_fail: continue
                else: raise
            try:
                checksum_data = checksum_file.read()
            finally:
                checksum_file.close()
            yield RevisionFile(self, checksum_name, {})
            checksums = self._parse_checksum(checksum_data)
            for name, name_sums in checksums.items():
                yield RevisionFile(self, name, name_sums)

    def apply(self, tree, reverse=False):
        """Replay this revision on this tree. Raise on conflict.

        :param tree: the tree to apply changes to.
        :type tree: `WorkingTree`
        :param reverse: invert the meaning of the changeset; adds
            become deletes, etc.
        :type reverse: bool
        :raise errors.ChangesetConflict: a conflict occured while replaying the
            revision.
        """
        _check_working_tree_param(tree, 'tree')
        args = self._impl.replay_args(self, tree, reverse)
        status = backend.status_cmd(args, expected=(0,1))
        if status == 1:
            raise errors.ChangesetConflict(tree, self)


class _Namespace_Baz_1_0(object):

    def replay_args(rev, tree, reverse):
        args = ['replay', '--dir', str(tree)]
        if reverse:
            args.append('--reverse')
        args.append(rev.fullname)
        return args

    replay_args = staticmethod(replay_args)

    def get_args(revision, dir, link=False):
        args = ['get']
        if link:
            args.append('--link')
        args.extend((revision.fullname, str(dir)))
        return args

    get_args = staticmethod(get_args)


def _revision_impl():
    if backend.version.release < (1, 4, 0):
        return _Revision_Baz_1_0
    else:
        return _Revision_Baz_1_4


class _Revision_Baz_1_0(_Namespace_Baz_1_0):

    def revision_from_previous(archive, previous):
        return archive + '/' + previous

    revision_from_previous = staticmethod(revision_from_previous)


class _Revision_Baz_1_4(_Namespace_Baz_1_0):

    def revision_from_previous(archive, previous):
        return previous

    revision_from_previous = staticmethod(revision_from_previous)


public('RevisionFile')

class RevisionFile(object):

    """File component of an archived revision.

    :ivar revision: revision this file belongs to.
    :type revision: `Revision`
    :ivar name: name of that file.
    :type name: str
    :ivar checksums: dictionnary whose keys are checksum algorithms
        (e.g. ``"md5"``, ``"sha1"``) and whose values are checksum
        values (strings of hexadecimal digits).
    :type checksums: dict of str to str
    """

    def __init__(self, revision, name, checksums):
        self.revision = revision
        self.name = name
        self.checksums = checksums

    def _get_data(self):
        import urllib
        my_file = urllib.urlopen(self.revision._file_url(self.name))
        try:
            ret = my_file.read()
        finally:
            my_file.close()
        return ret

    data = property(_get_data,
                    """Content of of that file.

                    :type: str
                    """)


### Patch logs ###

public('Patchlog')

from _patchlog import Patchlog

public('LogMessage')

from _logmessage import LogMessage


### Source trees ###

public(
    'init_tree',
    'in_source_tree',
    'tree_root',
    )

def init_tree(directory, version=None, nested=False):
    """Initialize a new project tree.

    :param directory: directory to initialize as a source tree.
    :type directory: str
    :param version: if given, set the the ``tree-version`` and create an empty
       log version.
    :type version: `Version`
    :param nested: if true, the command will succeed even if 'directory'
        is already within a source tree.
    :type nested: bool
    :return: source tree object for the given directory.
    :rtype: `WorkingTree`
    """
    if version is not None:
        version = _version_param(version)
    _arch.init_tree(directory, version, nested)
    return WorkingTree(directory)


def in_source_tree(directory=None):
    """Is directory inside a Arch source tree?

    :param directory: test if that directory is in an Arch source tree.
    :type directory: str
    :return: whether this directory is inside an Arch source tree.
    :rtype: bool

    :warning: omitting the ``directory`` argument is deprecated.
    """
    if directory is None:
        deprecated_usage(
            in_source_tree, "Argument defaults to current direcotry.")
        directory = '.'
    return _arch.in_tree(directory)


def tree_root(directory=None):
    """SourceTree containing the given directory.

    :param directory: give the ``tree-root`` of this directory. Specify "." to
        get the ``tree-root`` of the current directory.
    :type directory: str
    :return: source tree containing ``directory``.
    :rtype: `ArchSourceTree`

    :warning: omitting the ``directory`` argument is deprecated.
    """
    if directory is None:
        deprecated_usage(tree_root, "Argument defaults to current directory.")
        directory = '.'
    root = _arch.tree_root(directory)
    # XXX use SourceTree to instanciate either WorkingTree or LibraryTree
    return SourceTree(root)


public('SourceTree',
       'ForeignTree',
       'ArchSourceTree',
       'LibraryTree',
       'WorkingTree')

class SourceTree(DirName):

    """Abstract base class for `ForeignTree` and `ArchSourceTree`."""

    def __new__(cls, root=None):
        """Create a source tree object for the given root path.

        `ForeignTree` if root does not point to a Arch source tree.
        `LibraryTree` if root is a tree in the revision library. `WorkingTree`
        if root is a Arch source tree outside of the revision library.

        If root is omitted, use the tree-root of the current working directory.
        """
        __pychecker__ = 'no-returnvalues'
        # checking root actually is a root done by LibraryTree and WorkingTree
        if cls is not SourceTree:
            assert root is not None
            return DirName.__new__(cls, root)
        else:
            if not root:
                root = _arch.tree_root(os.getcwd())
            root = DirName(root).realpath()
            if _is_tree_root(root):
                for revlib in _arch.iter_revision_libraries():
                    if root.startswith(revlib):
                      return LibraryTree(root)
                return WorkingTree(root)
            else:
                assert not _arch.in_tree(root)
                return ForeignTree(root)


class ForeignTree(SourceTree):
    """Generic source tree without Arch support.

    Unlike Arch source trees, the root may not even exist.
    """
    def __init__(self, root): pass


def _is_tree_root(dir):
    """Return True if dir is a Arch source tree."""
    return os.path.isdir(os.path.join(dir, '{arch}'))


class ArchSourceTree(SourceTree):

    """Abstract base class for Arch source trees."""

    def get_tree_version(self):
        """Default version of the tree, also called tree-version.

        :raise errors.TreeVersionError: no valid default version set.
        :raise IOError: unable to read the ++default-version file.
        """
        data = self._read_tree_version()
        if data is None:
            raise errors.TreeVersionError(self)
        try:
            return Version(data)
        except errors.NamespaceError:
            raise errors.TreeVersionError(self, data)
    tree_version = property(get_tree_version)

    def _read_tree_version(self):
        path = os.path.join(self, '{arch}', '++default-version')
        try:
            return open(path, 'r').read().rstrip('\n')
        except IOError:
            if not os.path.exists(vsn_name):
                return None
            raise

    def _get_tree_revision(self):
        """Revision of the last patchlog for the tree-version.

        :raise errors.TreeVersionError: no valid default version set.
        :raise IOError: unable to read the ++default-version file.
        """
        reverse_logs = self.iter_logs(reverse=True)
        try: last_log = reverse_logs.next()
        except StopIteration:
            raise RuntimeError('no logs for the tree-version')
        return last_log.revision
    tree_revision = property(_get_tree_revision)

    def check_is_tree_root(self):
        if not _is_tree_root(self):
            raise errors.SourceTreeError(str(self))
        assert os.path.isdir(self)

    def get_tagging_method(self):
        return _arch.tagging_method(self)
    tagging_method = property(get_tagging_method)


    def iter_inventory(self, source=False, precious=False, backups=False,
                       junk=False, unrecognized=False, trees=False,
                       directories=False, files=False, both=False,
                       names=False, limit=None):
        """Tree inventory.

        The kind of files looked for is specified by setting to
        ``True`` exactly one of the following keyword arguments:

            ``source``, ``precious``, ``backups``, ``junk``,
            ``unrecognized``, ``trees``.

        If the ``trees`` argument is not set, whether files, directory
        or both should be listed is specified by setting to ``True``
        exactly one of the following keyword arguments:

            ``directories``, ``files``, ``both``.

        :keyword source: list source files only.
        :keyword precious: list precious files only.
        :keyword backups: list backup files only.
        :keyword junk: list junk files only.
        :keyword unrecognized: list unrecognized files only.
        :keyword trees: list nested trees only. If this is true, the
            iterator wil only yield `ArchSourceTree` objects.

        :keyword directories: list directories only,
            yield only `FileName` objects.
        :keyword files: list files only, yield only `DirName` objects.
        :keyword both: list both files and directories,
            yield both FileName and `DirName` objects.

        :keyword name: do inventory as if the id-tagging-method was
            set to ``names``. That is useful to ignore
            ``untagged-source`` classification.

        :keyword limit: restrict the inventory to this directory. Must
            be the name of a directory relative to the tree root.

        :rtype: iterable of `FileName`, `DirName`, `ArchSourceTree`
            according to the arguments.
        """
        __pychecker__ = 'maxargs=12'
        name_filter = self.__inventory_filter(trees, directories, files, both)
        name_iterator = self.__inventory_helper(
            ['inventory'], source, precious, backups, junk, unrecognized,
            trees, directories, files, both, names, limit)
        # using a generator here would delay argument checking
        return itertools.imap(name_filter, name_iterator)

    def iter_inventory_ids(self,source=False, precious=False, backups=False,
                           junk=False, unrecognized=False, trees=False,
                           directories=False, files=False, both=False,
                           limit=None):
        """Tree inventory with file ids.

        :see: `ArchSourceTree.iter_inventory`

        :rtype: iterable of tuples ``(id, item)``, where ``item`` is
            `FileName`, `DirName`, `ArchSourceTree` according to the
            arguments and ``id`` is the associated inventory id.
        """
        __pychecker__ = 'maxargs=11'
        name_filter = self.__inventory_filter(trees, directories, files, both)
        def id_name_filter(line):
            name, id_ = line.split('\t')
            return id_, name_filter(name)
        id_name_iterator = self.__inventory_helper(
            ['inventory', '--ids'], source, precious, backups, junk,
            unrecognized, trees, directories, files, both,
            names=False, limit=limit)
        # using a generator here would delay argument checking
        return itertools.imap(id_name_filter, id_name_iterator)

    def __inventory_filter(self, trees, directories, files, both):
        types = trees, directories, files, both
        assert 1 == sum([bool(t) for t in types])
        trees, directories, files, both = types
        if trees:
            return self.__inventory_filter_trees
        if directories:
            return self.__inventory_filter_directories
        if files:
            return self.__inventory_filter_files
        if both:
            return self.__inventory_filter_both
        raise AssertionError('unreachable')

    def __inventory_filter_trees(self, t):
        return SourceTree(self/name_unescape(t))

    def __inventory_filter_directories(self, d):
        return DirName(name_unescape(d))

    def __inventory_filter_files(self, f):
        return FileName(name_unescape(f))

    def __inventory_filter_both(self, f):
        __pychecker__ = 'no-returnvalues'
        f = name_unescape(f)
        if os.path.isfile(self/f):
            return FileName(f)
        elif os.path.islink(self/f):
            return FileName(f)
        elif os.path.isdir(self/f):
            return DirName(f)
        else:
            raise AssertionError("neither file, link, nor dir: %r" % (self/f,))

    def __inventory_helper(self, opts,
                           source, precious, backups, junk, unrecognized,trees,
                           directories, files, both, names, limit):
        __pychecker__ = 'maxargs=13'
        # Asserts there is only one class and one type
        classes = source, precious, backups, junk, unrecognized, trees
        classes = [bool(X) for X in classes]
        assert 1 == sum(classes)
        class_opts = ['--source', '--precious', '--backups', '--junk',
                      '--unrecognized', '--trees']
        types = directories, files, both
        types = [bool(X) for X in types]
        assert 1 == sum(types) + bool(trees)
        type_opts = ['--directories', '--files', '--both']
        for flag, opt in zip(classes + types, class_opts + type_opts):
            if flag:
                opts.append(opt)
        if names:
            opts.append('--names')
        if limit is not None:
            self._check_relname_param(limit, 'limit')
            opts.append(str(limit))
        return backend.sequence_cmd(opts, chdir=str(self))


    def _check_relname_param(self, param, name):
        """Internal argument checking utility"""
        if not isinstance(param, basestring):
            exc_type = TypeError
        elif os.path.isabs(param):
            exc_type = ValueError
        else:
            return
        raise exc_type("Parameter \"%s\" must be a relative path (string)"
                       " but was: %r" % (name, param))

    def inventory(self, *args, **kw):
        """Deprecated.

        Inventory of the source tree as a list.

        :see: `iter_inventory`
        """
        # deprecated_callable(self.inventory, self.iter_inventory)
        return list (self.iter_inventory(*args, **kw))

    def get_tree(self):
        return util.sorttree(self.inventory(source=True, both=True))

    def iter_log_versions(self, limit=None, reverse=False):
        """Iterate over versions this tree has a log-version for.

        :param limit: only iterate log-versions in this namespace.
        :type limit: `Archive`, `Category`, `Branch`, `Version`
        :param reverse: yield log-versions in reverse order.
        :type reverse: bool
        """
        kwargs = {}
        if isinstance(limit, Archive):
            kwargs['archive'] = limit.name
        elif isinstance(limit, (Category, Branch, Version)):
            kwargs['archive'] = limit.archive.name
        if isinstance(limit, Category): kwargs['category'] = limit.nonarch
        elif isinstance(limit, Branch): kwargs['branch'] = limit.nonarch
        elif isinstance(limit, Version): kwargs['version'] = limit.nonarch
        elif limit is not None and not isinstance(limit, Archive):
            raise TypeError("Expected Archive, Category, Branch or Version"
                            " but got: %r" % limit)
        for vsn in _arch.iter_log_versions(self, reverse=reverse, **kwargs):
            yield Version(vsn)

    def iter_logs(self, version=None, reverse=False):
        """Iterate over patchlogs present in this tree.

        :param version: list patchlogs from this version. Defaults to
            the tree-version.
        :type version: `Version`
        :param reverse: iterate more recent logs first.
        :type reverse: bool
        :return: patchlogs from ``version``.
        :rtype: iterator of `Patchlog`.
        :raise errors.TreeVersionError: no valid default version set.
        :raise IOError: unable to read the ++default-version file.
        """
        if version is None:
            version = self.tree_version.fullname
        else:
            version = _version_param(version)
        for rvsn in _arch.iter_log_ls(self, version, reverse=reverse):
            yield Patchlog(_unsafe((rvsn,)), tree=self)

    def get_tag(self, name):
        """Read a file id set by an explicit id tag or tagline.

        FIXME: update docstring when support for pybaz<1.3 is dropped, as this
        release fixed "baz id" to report name-based ids correctly.

        :param name: name of a source file relative to the tree-root.
        :type name: str
        :return: file id if the file has an explicit id or a tagline.
        :rtype: str
        """
        absname = str(self/name)
        if not os.path.exists(absname) and not os.path.islink(absname):
            raise IOError(2, 'No such file or directory', absname)
        args = ('id', absname)
        status, output = backend.status_one_cmd(args, expected=(0,1))
        if status == 1:
            return None
        return output.split('\t')[1]

    def file_find(self, name, revision):
         """Find a pristine copy of a file for a given revision.

         Will create the requested revision in the library or in the
         current tree pristines if needed.

         :param name: name of the file.
         :type name: str
         :param revision: revision to look for the file into.
         :type revision: `Revision`
         :return: absolute path name of a pristine copy of the file.
         :rtype: `pathname.PathName`
         :raise errors.MissingFileError: file is not source or is not
             present in the specified revision.
         """
         revision = _revision_param(revision)
         result = _arch.file_find(self, name, revision)
         if result is None:
             raise errors.MissingFileError(self, name, revision)
         # XXX Work around relative path when using pristines
         result = os.path.join(self, result)
         return PathName(result)


class LibraryTree(ArchSourceTree):

    """Read-only Arch source tree."""

    def __init__(self, root):
        """Create a LibraryTree object with the given root path.

        Root must be a directory containing a Arch source tree in the
        revision library.
        """
        ArchSourceTree.__init__(self, root)
        self.check_is_tree_root()


class WorkingTree(ArchSourceTree):

    """Working source tree, Arch source tree which can be modified."""

    def __init__(self, root):
        """Create WorkingTree object with the given root path.

        Root must be a directory containing a valid Arch source tree
        outside of the revision library.
        """
        ArchSourceTree.__init__(self, root)
        self.check_is_tree_root()
        self._impl = _workingtree_impl()

    def sync_tree(self, revision):
        """Adds the patchlogs in the given revision to the current tree.

        Create a temporary source tree for ``revision``, then add all the
        patchlogs present in that tree to the current tree. No content
        is touched besides the patchlogs.

        :param revision: revision to synchronize with.
        :type revision: `Version`, `Revision` or str
        :raise errors.NamespaceError: ``revision`` is not a valid
            version or revision name.
        """
        revision = _version_revision_param(revision)
        _arch.sync_tree(self, revision)

    def set_tree_version(self, version):
        version = _version_param(version)
        version_path = os.path.join(str(self), '{arch}', '++default-version')
        print >> open(version_path, 'w'), version

    def resolved(self, all=False):
        _arch.resolved(self, all)

    tree_version = property(ArchSourceTree.get_tree_version, set_tree_version)

    def has_changes(self):
        """Are there uncommited changes is this source tree?

        :rtype: bool
        """
        return self._impl.has_changes(str(self))

    def changes(self, revision=None, output=None):
        """Uncommited changes in this tree.

        :param revision: produce the changeset between this revision
            and the tree. If ``revision`` is ``None``, use the
            tree-revision.
        :type revision: `Revision`, None
        :param output: absolute path of the changeset to produce.
        :type output: string
        :return: changeset between ``revision`` and the tree.
        :rtype: Changeset
        :raise errors.TreeVersionError: no valid default version set.
        :raise IOError: unable to read the ++default-version file.
        """
        if revision is None: revision = self.tree_revision
        _check_str_param(output, 'output')
        _check_revision_param(revision, 'revision')
        return delta(revision, self, output)

    def star_merge(self, from_=None, reference=None,
                   forward=False, diff3=False):
        """Merge mutually merged branches.

        :bug: if the merge causes a conflict, a RuntimeError is
            raised. You should not rely on this behaviour as it is
            likely to change in the future. If you want to support
            conflicting merges, use `iter_star_merge` instead.

        :param `from_`: branch to merge changes from, ``None`` means the
            ``tree-version``.
        :type `from_`: None, `Version`, `Revision`, or str
        :param reference: reference version for the merge, ``None``
            means the ``tree-version``.
        :type reference: None, `Version` or str
        :param forward: ignore already applied patch hunks.
        :type forward: bool
        :param diff3: produce inline conflict markers instead of
            ``.rej`` files.
        :type diff3: bool
        :raise errors.NamespaceError: ``from_`` or ``reference`` is
            not a valid version or revision name.
        """
        if from_ is not None:
            from_ = _version_revision_param(from_)
        if reference is not None:
            reference = _version_revision_param(reference)
        if forward:
            deprecated_usage(WorkingTree.star_merge, (
                "forward=True has always been a no-op."))
        args = self._impl.star_merge_args(diff3)
        args.extend(('--dir', str(self)))
        if reference is not None:
            args.extend(('--reference', reference))
        if from_ is not None:
            args.append(from_)
        # yes, we want to raise if there was a conflict.
        backend.null_cmd(args)

    def iter_star_merge(self, from_=None, reference=None,
                        forward=False, diff3=False):
        """Merge mutually merged branches.

        :param `from_`: branch to merge changes from, ``None`` means the
            ``tree-version``.
        :type `from_`: None, `Version`, `Revision`, or str
        :param reference: reference version for the merge, ``None``
            means the ``tree-version``.
        :type reference: None, `Version` or str
        :param forward: ignore already applied patch hunks.
        :type forward: bool
        :param diff3: produce inline conflict markers instead of
            ``.rej`` files.
        :type diff3: bool
        :raise errors.NamespaceError: ``from_`` or ``reference`` is
            not a valid version or revision name.
        :rtype: `ChangesetApplication`
        """
        # the --changes option must be supported by another method
        # which returns a Changeset.
        if from_ is not None:
            from_ = _version_revision_param(from_)
        if reference is not None:
            reference = _version_revision_param(reference)
        if forward:
            deprecated_usage(WorkingTree.iter_star_merge, (
                "forward=True has always been a no-op."))
        args = self._impl.star_merge_args(diff3)
        args.extend(('--dir', str(self)))
        if reference is not None:
            args.extend(('--reference', reference))
        if from_ is not None:
            args.append(from_)
        merge = backend.sequence_cmd(args, expected=(0,1))
        return ChangesetApplication(merge)

    def undo(self, revision=None, output=None, quiet=False, throw_away=False):
        """Undo and save changes in a project tree.

        Remove local changes since revision and optionally save them
        as a changeset.

        :keyword revision: revision to revert to. Default to the last
            revision of the tree-version for which a patchlog is present.
        :type revision: `Revision`, str
        :keyword output: name of the output changeset directory. Must
            not already exist. Default to an automatic ,,undo-N name
            in the working tree.
        :type output: str
        :keyword quiet: OBSOLETE. Incremental output is always discarded.
        :type quiet: bool
        :keyword throw_away: discard the output changeset and return
            ``None``. Must not be used at the same time as ``output``.
        :type throw_away: bool
        :return: changeset restoring the undone changes,
            or None if ``throw_away``.
        :rtype: `Changeset`, None
        """
        assert sum(map(bool, (output, throw_away))) < 2
        if output is None:
            output = util.new_numbered_name(self, ',,undo-')
        if revision is not None:
            revision = _revision_param(revision)
        _arch.undo(self, revision, output, quiet, throw_away)
        if throw_away: return None
        return Changeset(output)

    def redo(self, patch=None, keep=False, quiet=False):
        """Redo changes in a project tree.

        Apply patch to the project tree and delete patch.

        If patch is provided, it must be a Changeset object. Else, the highest
        numbered ,,undo-N directory in the project tree root is used.

        If keep is true, the patch directory is not deleted.
        """
        _arch.redo(self, patch, keep, quiet)

    def set_tagging_method(self, method):
        _arch.set_tagging_method(self, method)
    tagging_method = property(ArchSourceTree.get_tagging_method,
                              set_tagging_method)

    def add_tag(self, file):
        _arch.add(self/file)

    def move_tag(self, src, dest):
        _arch.move(self/src, self/dest)

    def move_file(self, src, dest):
        # FIXME: move explicit tag if present
        dest = PathName(dest)
        assert os.path.exists((self/dest).dirname())
        os.rename(self/src, self/dest)

    def delete(self, file):
        fullfile = self/file
        if os.path.isfile(fullfile):
            if _arch.has_explicit_id(fullfile):
                _arch.delete(fullfile)
            os.unlink(fullfile)
        elif os.path.isdir(fullfile):
            shutil.rmtree(fullfile)

    def del_file(self, file):
        fullfile = self/file
        if os.path.isfile(fullfile):
            os.unlink(fullfile)
        else:
            shutil.rmtree(fullfile)

    def del_tag(self, file):
        if not os.path.islink(self/file):
            assert os.path.exists(self/file)
        _arch.delete(self/file)

    def import_(self, log=None):
        """Archive a full-source base-0 revision.

        If log is specified, it must be a LogMessage object or a file
        name as a string. If omitted, the default log message file of
        the tree is used.

        The --summary, --log-message and --setup options to tla are
        mere CLI convenience features and are not directly supported.
        """
        if isinstance(log, LogMessage):
            log.save()
            log = log.name
        assert log is None or isinstance(log, str)
        if log is not None:
            log = os.path.abspath(log)
        self._impl.import_(str(self), log)

    def commit(self, log=None, strict=False, seal=False, fix=False,
               out_of_date_ok=False, file_list=None, version=None,
               just_commit=False):
        """Archive a changeset-based revision.

        :keyword version: version in which to commit the revision.
            Defaults to the `tree_version`.
        :type version: `Version`, str
        :keyword log: Log message for this revision. Defaults to the log
            message file of the tree-version (the file created by
            ``tla make-log``.
        :type log: `LogMessage`
        :param strict: perform a strict tree-lint before commiting.
        :type strict: bool
        :param seal: create a ``version-0`` revision.
        :type seal: bool
        :param fix: create a ``versionfix`` revision.
        :type fix: bool
        :param out_of_date_ok: commit even if the tree is out of
            date.
        :type out_of_date_ok: bool
        :param file_list: Only commit changes to those files,
            specified relative to the tree-root.
        :type file_list: iterable of str, with at least one item.
        :param just_commit: only create new revision, do not add ancillary data
            like cachedrevs or ancestry files.
        :type just_commit: bool

        The --summary and --log-message options to tla are mere CLI
        convenience features and are not directly supported.

        :see: `WorkingTree.iter_commit`
        """
        for unused in self.iter_commit(
            log, strict, seal, fix, out_of_date_ok, file_list, version,
            just_commit):
            pass

    def log_for_merge(self):
        """Standard arch log of newly merged patches.

        :rtype: str
        """
        return _arch.log_for_merge(self)


    def iter_commit(self, log=None, strict=False, seal=False, fix=False,
                    out_of_date_ok=False, file_list=None, version=None,
                    just_commit=False, stderr_too=False):
        """Archive a changeset-based revision, returning an iterator.


        :keyword version: version in which to commit the revision.
            Defaults to the `tree_version`.
        :type version: `Version`, str
        :keyword log: Log message for this revision. Defaults to the log
            message file of the tree-version (the file created by
            ``tla make-log``.
        :type log: `LogMessage`
        :keyword strict: perform a strict tree-lint before commiting.
        :type strict: bool
        :keyword seal: create a ``version-0`` revision.
        :type seal: bool
        :keyword fix: create a ``versionfix`` revision.
        :type fix: bool
        :keyword out_of_date_ok: commit even if the tree is out of
            date.
        :type out_of_date_ok: bool
        :keyword file_list: Only commit changes to those files,
            specified relative to the tree-root.
        :type file_list: iterable of str, with at least one item.
        :param just_commit: only create new revision, do not add ancillary data
            like cachedrevs or ancestry files.
        :type just_commit: bool
        :param stderr_too: iterate over stderr output as well as stdout.
        :type stderr_too: bool
        :rtype: iterator of `TreeChange`, `Chatter` or str

        :warning: ``stderr_too=True`` is only supported with the
        PyArchSpawningStrategy. Using it will cause a ArgumentError with the
        TwistedSpawningStrategy. That will be fixed when the process handling
        subsystem is replaced by Gnarly.

        The --summary and --log-message options to tla are mere CLI
        convenience features and are not directly supported.

        :see: `WorkingTree.commit`
        """
        # XXX stderr_too=True will fail with the TwistedSpawningStrategy. That
        # XXX will be fixed when the process handling subsystem is replaced by
        # XXX Gnarly. -- David Allouche 2005-05-27
        args = ['commit']
        if just_commit:
            args.extend(self._impl.commit_just_commit)
        if log is not None:
            if isinstance(log, LogMessage):
                log.save()
                log = log.name
            assert isinstance(log, str)
            log = os.path.abspath(log)
            args.extend(('--log', log))
        if strict:
            args.append('--strict')
        assert not (seal and fix)
        if seal:
            args.append('--seal')
        if fix:
            args.append('--fix')
        if out_of_date_ok:
            args.append('--out-of-date-ok')
        self._maybe_commit_version(version, args)
        if file_list is not None:
            file_list = tuple(file_list)
            assert 0 < len(file_list)
            args.append('--')
            args.extend(map(str, file_list))
        iterator = backend.sequence_cmd(args, chdir=str(self),
                                        stderr_too=stderr_too)
        return classify_changeset_creation(iterator)

    def _maybe_commit_version(self, version, args):
        """Add a version to the commit command line if needed"""
        if version is None:
            # raises on bad versions
            self.tree_version.fullname
            return
        version = _version_param(version)
        args.extend(self._impl.commit_version_args(version))

    def log_message(self, create=True):
        """Default log-message object used by import and commit.

        If `create` is False, and the standard log file does not already
        exists, return None. If `create` is True, use ``tla make-log`` if
        needed.
        """
        path = self._message_path()
        if not os.path.exists(path):
            if not create: return None
            _arch.make_log(self)
        return LogMessage(path)

    def _message_path(self):
        version = self.tree_version
        return self / FileName('++log.%s--%s'
                               % (version.nonarch, version.archive.name))

    def add_log_version(self, version):
        """Add a patch log version to the project tree."""
        version = _version_param(version)
        _arch.add_log_version(self, version)

    def remove_log_version(self, version):
        """Remove a patch log version from the project tree."""
        version = _version_param(version)
        _arch.remove_log_version(self, version)

    def replay(self):
        """Replay changesets into this working tree."""
        _arch.replay(self)

    def update(self):
        """Apply delta of new revisions in the archive.

        Apply delta(A,B) on this working tree, where A and B are both
        revisions of the tree version, A is the latest whose patchlog
        is present in the tree and B is the latest present in the
        archive.
        """
        _arch.update(self)

    def iter_pristines(self):
        """Pristines present in that source tree.

        :return: Revisions which have pristine trees in that source tree.
        :rtype: iterable of `Revision`
        """
        for rvsn in _arch.iter_pristines(self):
            yield Revision(rvsn)

    def add_pristine(self, revision):
        """Ensure that the project tree has a particular pristine revision.

        :param revision: revision to add a pristine for.
        :type revision: `Revision`
        """
        revision = _revision_param(revision)
        _arch.add_pristine(self, revision)

    def find_pristine(self, revision):
        """Path to a pristine tree for the given revision.

        :param revision: find a pristine for that revision.
        :type revision: `Revision`
        :rtype: `ArchSourceTree`
        :raise errors.NoPristineFoundError: no pristine tree was found
            for the given revision.
        """
        if revision not in self.iter_pristines():
            raise errors.NoPristineFoundError(self, revision)
        revision = _revision_param(revision)
        return ArchSourceTree(_arch.find_pristine(self, revision))


# This is crufty. Implementations are not flyweights.
def _workingtree_impl():
    if backend.version.release < (1, 1, 0):
        return _WorkingTree_Baz_1_0
    elif backend.version.release == (1, 2, 0):
        return _WorkingTree_Baz_2_0_BUG
    elif backend.version.release < (1, 4, 0):
        return _WorkingTree_Baz_1_1
    else:
        return _WorkingTree_Baz_1_4


class _WorkingTree_Baz_1_0(object):

    def import_(tree, log):
        args = ['import', '--dir', tree, '--setup']
        if log is not None:
            args.extend(('--log', log))
        backend.null_cmd(args)

    import_ = staticmethod(import_)

    def has_changes(tree):
        args = ('changes', '--dir', tree, '--quiet')
        return 1 == backend.status_cmd(args, expected=(0,1))

    has_changes = staticmethod(has_changes)

    commit_just_commit = []

    def star_merge_args(diff3):
        args = ['star-merge']
        if diff3:
            args.append('--three-way')
        return args

    star_merge_args = staticmethod(star_merge_args)

    def commit_version_args(version):
        return [version]

    commit_version_args = staticmethod(commit_version_args)


class _WorkingTree_Baz_1_1(_WorkingTree_Baz_1_0):

    def import_(tree, log):
        args = ['import', '--dir', tree]
        if log is not None:
            args.extend(('--log', log))
        backend.null_cmd(args)

    import_ = staticmethod(import_)

    def has_changes(tree):
        args = ('diff', '--dir', tree, '--quiet', '--summary')
        return 1 == backend.status_cmd(args, expected=(0,1))

    has_changes = staticmethod(has_changes)

    commit_just_commit = ['--just-commit']

    def star_merge_args(diff3):
        args = ['merge', '--star-merge']
        if not diff3:
            args.append('--two-way')
        return args

    star_merge_args = staticmethod(star_merge_args)


class _WorkingTree_Baz_2_0_BUG(_WorkingTree_Baz_1_1):

    def import_(tree, log):
        # baz-1.2 has a bug preventing use of the --dir option
        args = ['import']
        if log is not None:
            args.extend(('--log', log))
        backend.null_cmd(args, chdir=tree)

    import_ = staticmethod(import_)


class _WorkingTree_Baz_1_4(_WorkingTree_Baz_1_1):

    def commit_version_args(version):
        return ['--branch', version]

    commit_version_args = staticmethod(commit_version_args)


### Changesets ###

from _changeset import *

public('Changeset')
public('ChangesetApplication', 'ChangesetCreation')
public('delta', 'iter_delta')

public('changeset')

def changeset(orig, mod, dest):
    """Deprecated.

    :see: `delta`
    :rtype: `Changeset`
    """
    deprecated_callable(changeset, delta)
    return delta(ArchSourceTree(orig), ArchSourceTree(mod), dest)


def _check_str_param(param, name):
    """Internal argument checking utility"""
    if not isinstance(param, basestring):
        raise TypeError("Parameter \"%s\" must be a string"
                        " but was: %r" % (name, param))

def _check_working_tree_param(param, name):
    """Internal argument checking utility"""
    if not isinstance(param, WorkingTree):
        raise TypeError("Parameter \"%s\" must be a WorkingTree"
                        " but was: %r" % (name, param))


### Top level archive functions ###

public('my_id', 'set_my_id', 'make_archive', 'register_archive')

from _my_id import my_id
from _my_id import set_my_id

def make_archive(name, location, signed=False, listing=False, tla=False):
    """Deprecated.

    :see: `ArchiveLocation.create_master`

    :param name: archive name (e.g. "david@allouche.net--2003b").
    :type name: `Archive` or str
    :param location: URL of the archive
    :type location: str
    :param signed: create GPG signatures for the archive contents.
    :type signed: bool
    :param listing: maintains ''.listing'' files to enable HTTP access.
    :type listing: bool
    :param tla: create a tla archive instead of a baz archive.
    :type tla: bool

    :return: an `Archive` instance for the given name.
    :rtype: `Archive`

    :raise errors.NamespaceError: ``name`` is not a valid archive name.
    """
    deprecated_callable(make_archive, ArchiveLocation.create_master)
    name = _archive_param(name)
    archive = Archive(_unsafe((name,)))
    params = ArchiveLocationParams()
    params.signed = signed
    params.listing = listing
    if tla:
        params.tla_format()
    location = ArchiveLocation(location)
    location.create_master(archive, params)
    return archive

def register_archive(name, location):
    """Deprecated.

    :see: `ArchiveLocation.register`

    :param name: archive name, or None to use the official name stored in the
        archive.
    :type name: str, None
    :param location: URL of the archive.
    :type location: str
    :return: newly registered archive.
    :rtype: `Archive`.
    """
    deprecated_callable(register_archive, ArchiveLocation.register)
    if name is not None:
        name = _archive_param(name)
        if Archive(name).is_registered():
            raise ValueError('%r is already a registered name' % (name,))

    _check_str_param(location, 'location')
    return _archive_impl().register_archive(name, location)


public('iter_archives', 'archives')

def iter_archives():
    """Iterate over registered archives.

    :return: all registered archives.
    :rtype: iterable of `Archive`
    """
    for n in backend.sequence_cmd(('archives', '--names')):
        yield Archive(_unsafe((n,)))


def archives():
    """Deprecated.

    List of registered archives.

    :rtype: sequence of `Archive`
    :see: `iter_archives`
    """
    deprecated_callable(archives, iter_archives)
    return list(iter_archives())


public('iter_library_archives', 'library_archives')

def iter_library_archives():
    """Iterate over archives present in the revision library.

    :returns: all archives which are present in the revision library.
    :rtype: iterable of `Archive`
    """
    for n in _arch.library_archives():
        yield Archive(_unsafe((n,)))


def library_archives():
    """Deprecated.

    List of archives present in the revision library.

    :rtype: sequence of `Archive`
    :see: `iter_library_archives`
    """
    deprecated_callable(library_archives, iter_library_archives)
    return list(iter_library_archives())


public('default_archive')

def default_archive():
    """Default Archive object or None.

    :return: the default archive, or None.
    :rtype: `Archive`, None
    """
    name = _arch.default_archive()
    if name is None:
        return None
    else:
        return Archive(_unsafe((name,)))


public('make_continuation')

def make_continuation(source_revision, tag_version):
    """Deprecated.

    :see: `Revision.make_continuation`
    """
    deprecated_callable(make_continuation, Revision.make_continuation)
    source_revision = _version_revision_param(source_revision)
    tag_version = _version_param(tag_version)
    _arch.tag(source_revision, tag_version)

public('get', 'get_patch')

def get(revision, dir, link=False):
    """ Construct a project tree for a revision.

    :rtype: `WorkingTree`
    :see: `Revision.get`
    """
    # FIXME: fragile, code duplication
    revision = _package_revision_param(revision)
    args = ['get']
    if link:
        args.append('--link')
    args.extend((revision, str(dir)))
    backend.null_cmd(args)
    return WorkingTree(dir)

def get_patch(revision, dir):
    """Deprecated.

    :rtype: `Changeset`
    :see: `Revision.get_patch`
    """
    deprecated_callable(get_patch, Revision.get_patch)
    revision = _revision_param(revision)
    _arch.get_patch(revision, dir)
    return Changeset(dir)


public(
    'iter_revision_libraries',
    'register_revision_library',
    'unregister_revision_library',
    )

def iter_revision_libraries():
    """Iterate over registered revision library directories.

    :return: directory names of all registered revision libraries.
    :rtype: iterable of str
    """
    return _arch.iter_revision_libraries()

def register_revision_library(dirname):
    """Register an existing revision library directory.

    :param dirname: absolute path name of existing user-writable directory.
    :type dirname: str
    :todo: create_revision_library which abstracts out revlib construction.
    :postcondition: ``dirname`` is present in `iter_revision_libraries` output.
    """
    if not os.path.isabs(dirname):
        raise ValueError, "not an absolute path: %r" % dirname
    if not os.path.isdir(dirname):
        raise ValueError, "directory does not exist: %r" % dirname
    _arch.register_revision_library(dirname)

def unregister_revision_library(dirname):
    """Unregister a revision library directory.

    :param dirname: registered revision library directory.
    :type dirname: str
    :todo: delete_revision_library which abstracts out revlib destruction.
    :precondition: ``dirname`` is present in `iter_revision_libraries` output.
    :postcondition: ``dirname`` is not listed by `iter_revision_libraries`.
    """
    if not os.path.isabs(dirname):
        raise ValueError, "not an absolute path: %r" % dirname
    _arch.unregister_revision_library(dirname)


### Parsing Arch names ###

public('NameParser')

from _nameparser import NameParser


### Searching in Archives ###

public(
    'filter_archive_logs',
    'filter_revisions',
    'grep_summary',
    'grep_summary_interactive',
    'suspected_move',
    'revisions_merging',
    'temphack',
    'revision_which_created',
    'last_revision',
    'map_name_id',
    )

from _deprecated_helpers import *
