# arch-tag: 75709428-970d-4bba-ac1c-7e1e7428345f
# Copyright (C) 2004  David Allouche <david@allouche.net>
#                     Canonical Ltd.
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""File name manipulation."""

__all__ = ['PathName', 'DirName', 'FileName']

import os


class PathName(str):

    """String with extra methods for filename operations."""

    def __new__(cls, path='.'):
        return str.__new__(cls, os.path.normpath(path))

    def __div__(self, path):
        if isinstance(path, (DirName, FileName)):
            ctor = path.__class__
        else:
            ctor = PathName
        return ctor(os.path.join(self, path))

    def __repr__(self):
        return '%s(%s)' % (type(self).__name__, str.__repr__(self))

    def abspath(self):
        return self.__class__(os.path.abspath(self))
    def dirname(self):
        return DirName(os.path.dirname(self))
    def basename(self):
        return self.__class__(os.path.basename(self))

    def realpath(self):
        if 'realpath' in dir(os.path):
            return self.__class__(os.path.realpath(self))
        else:
            return self.abspath()

    def splitname(self):
        dir_, base = os.path.split(self)
        return DirName(dir_), self.__class__(base)


class DirName(PathName):

    """PathName, further characterized as a directory name."""


class FileName(PathName):

    """PathName further characterized as a file name."""
