#!/usr/bin/env python
# arch-tag: eb042c78-5862-40b9-8570-bf7d62419dca
# Copyright (C) 2004 David Allouche <david@allouche.net>
#               2004 Canonical Ltd.
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""Test cases for revision library handling
"""

import os
import pybaz as arch
from pybaz.tests import framework
from pybaz.tests.framework import TestCase


class Empty(TestCase):

    tests = []

    def list_revlibs_empty(self):
        """iter_revision_libraries works when no revlib is registered"""
        self.assertEqual([], list(arch.iter_revision_libraries()))
    tests.append('list_revlibs_empty')


# XXX: baz bug, won't properly register a revision library in a path with a
# space in it. -- David Allouche 2005-03-31

class Register(TestCase):

    tests = []

    def register_revision_library(self):
        """register_revision_library works"""
        root = self.params.sandbox_dir # XXX: path whitespace baz bug
        revlib1_dir = root/'revlib1'
        os.mkdir(revlib1_dir)
        arch.register_revision_library(revlib1_dir)
        expected = [revlib1_dir]
        result = list(arch.iter_revision_libraries())
        self.assertEqual(expected, result)
        revlib2_dir = root/'revlib2'
        os.mkdir(revlib2_dir)
        arch.register_revision_library(revlib2_dir)
        expected = [revlib1_dir, revlib2_dir]
        expected.sort()
        result = list(arch.iter_revision_libraries())
        result.sort()
        self.assertEqual(expected, result)
    tests.append('register_revision_library')

    def register_revision_library_wants_abspath(self):
        """register_revision_library requires absolute path"""
        os.chdir(self.params.arch_dir)
        revlib_dir = 'revlib'
        os.mkdir(revlib_dir)
        self.assertRaises(ValueError, arch.register_revision_library,
                          revlib_dir)
    tests.append('register_revision_library_wants_abspath')


class Unregister(TestCase):

    tests = []

    def unregister_revision_library(self):
        """unregister_revision_library works"""
        root = self.params.sandbox_dir # XXX: path whitespace baz bug
        revlib1_dir = root/'revlib1'
        os.mkdir(revlib1_dir)
        arch.register_revision_library(revlib1_dir)
        revlib2_dir = root/'revlib2'
        os.mkdir(revlib2_dir)
        arch.register_revision_library(revlib2_dir)
        arch.unregister_revision_library(revlib1_dir)
        result = list(arch.iter_revision_libraries())
        self.assertEqual([revlib2_dir], result)
        arch.unregister_revision_library(revlib2_dir)
        result = list(arch.iter_revision_libraries())
        self.assertEqual([], result)
    tests.append('unregister_revision_library')

    def unregister_relative_revlib(self):
        """unregister_revision_library requires absolute path"""
        os.chdir(self.params.arch_dir)
        revlib_dir = self.params.arch_dir/'revlib'
        os.mkdir(revlib_dir)
        arch.register_revision_library(revlib_dir)
        self.assertRaises(ValueError, arch.unregister_revision_library,
                          os.path.basename(revlib_dir))
    tests.append('unregister_relative_revlib')


class List(TestCase):

    tests = []

    def extraSetup(self):
        self.params.set_my_id()
        self.params.create_archive()
        root = self.params.sandbox_dir # XXX: path whitespace baz bug
        revlib1_dir = root/'revlib1'
        os.mkdir(revlib1_dir)
        arch.register_revision_library(revlib1_dir)

    def empty(self):
        self.assertEqual([], list(arch.iter_library_archives()))
    tests.append('empty')

    def one(self):
        self.params.create_working_tree(self.params.version)
        tree = self.params.working_tree
        tree.import_()
        tree.tree_revision.library_add()
        archive = tree.tree_version.archive
        self.assertEqual([archive], list(arch.iter_library_archives()))
        category = tree.tree_version.category
        self.assertEqual([category], list(archive.iter_library_categories()))
        branch = tree.tree_version.branch
        self.assertEqual([branch], list(category.iter_library_branches()))
        version = tree.tree_version
        self.assertEqual([version], list(branch.iter_library_versions()))
    tests.append('one')


framework.register(__name__)
