#ifdef DEBUG
#endif
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <assert.h>
#include "xmlparse.h"
#include "domlette.h"
#include "expat2domlette.h"
#include "ds_primitives.h"
#include "ds_tools.h"

int readdata(char *buf,FILE *stream,PyObject *readmethod);
void buildUniversalName(const State *state, const XML_Char *expanded_name,
                        UniversalName **new_un);
PyObject *g_readerErrorObject = NULL;


void completeText(State *state)
{
  XML_Char *cdata = NULL;
  PyNodeObject *new_text;
  PyNodeObject *top_node;
  int preserve_state;
  int wsonly = 1;

  if (!state->curr_text || !state->curr_text[0]) return;

  /* If text contains any non-WS chars, we can't strip it in any case */
  for (cdata=state->curr_text; *cdata && wsonly; cdata++){
    if (!isspace(*cdata)) wsonly = 0;
  }
  cdata = state->curr_text;
  preserve_state = *((int *)_stack_peek(state->preserve_state));
  if (preserve_state || !wsonly){
    new_text = (PyNodeObject *)Document_CreateTextNode(state->ownerDoc,
                                                         cdata, state->docIx);
    top_node = _stack_peek(state->node_stack);
    Node_AppendChild(top_node, new_text);
  }
  free(cdata);
  if ((cdata = (XML_Char *)malloc(sizeof(XML_Char))) == NULL) {
    /* Exception: Out of memory */
  }
  cdata[0] = '\0';
  state->curr_text = cdata;
}


void buildUniversalName(const State *state, const XML_Char *expanded_name,
                        UniversalName **new_un)
{
  UniversalName *un=NULL;
  XML_Char *ns_split=NULL, *namecpy;
  ListElmt *node = NULL;

  if ((un = (UniversalName *)calloc(1, sizeof(UniversalName))) == NULL) {
    return;
    /* Exception: Out of memory */
  }
  namecpy = strdup(expanded_name);
  ns_split = strchr(namecpy, NSSEP);
  if (ns_split) {
    *ns_split = '\0';
    un->uri = strdup(namecpy);
    un->local = strdup(ns_split + 1);
    node = list_find(state->namespaces, (void *)un->uri, matchUri);
    if (node){
      if ((un->prefix = strdup(((NsMapping *)list_data(node))->prefix)) == NULL) {
        /* Exception: Out of memory */
        assert(0);
      }
    } else {
      /*I think then it must be the XML namespace per expat contract*/
      un->prefix = strdup("xml");
    }
  } else if (namecpy[0] == 'x' && namecpy[1] == 'm' &&
             namecpy[2] == 'l' && namecpy[3] == ':') {
    namecpy[3] = '\0';
    un->prefix = strdup(namecpy);
    un->local = strdup(namecpy+4);
    un->uri = strdup(XML_NAMESPACE);
  } else {
    un->prefix = strdup("");
    un->local = strdup(namecpy);
    un->uri = strdup("");
  }
  *new_un = un;
  free(namecpy);
}


void destroyUniversalName(UniversalName *un)
{
  if (un->prefix) free(un->prefix);
  if (un->local) free(un->local);
  if (un->uri) free(un->uri);
  free(un);
}


void startElement(void *userData, const XML_Char *name, const XML_Char **atts)
{
  UniversalName *un = NULL;
  State *state = (State *)userData;
  PyElementObject *elem = NULL;
  int i, attrs_len, found = 0;
  int *new_preserve_state;
  const XML_Char **ppattr = NULL, *curr_att_name = NULL, *curr_att_value = NULL;
  ListElmt *curr = NULL;
  NsMapping *nsm = NULL;
  void *ptr;

#ifdef DEBUG
  fprintf(stderr, "Start element: %s\n", name);
  dumpNsMappingList(state->namespaces, "namespaces");
  dumpNsMappingList(state->new_nss, "new_nss");
  dumpIntList(state->preserve_state, "preserve_state");
#endif
  completeText(state);

  buildUniversalName(state, name, &un);

  elem = Document_CreateElementNS(state->ownerDoc, (const char *)un->uri, (const char *)un->prefix,(const char *)un->local, state->docIx);

  /* Check for preserve state */
  if ((new_preserve_state = (int *)malloc(sizeof(int))) == NULL) {
    /* Exception: Out of memory */
  }

  *new_preserve_state = *((int *)_stack_peek(state->preserve_state));
  for (i=0, found=0; i < state->num_strip_elements && !found; i++){
    if (!strcmp(state->strip_elements[i].local, "*") && !strcmp(state->strip_elements[i].uri, un->uri)){
      found = 1;
    }
    else if (!strcmp(state->strip_elements[i].local, "*") && !strcmp(state->strip_elements[i].uri, un->uri)){
      found = 1;
    }
    else if (!strcmp(state->strip_elements[i].local, un->local) && !strcmp(state->strip_elements[i].uri, un->uri)){
      found = 1;
    }
    if (found)
      *new_preserve_state = !(state->strip_elements[i].strip_flag);
  }

  destroyUniversalName(un);

  _stack_push(state->preserve_state, new_preserve_state);

  /* Apply regular attributes */
  for(attrs_len = 0, ppattr = atts; *ppattr; ppattr++, attrs_len++) {
    if (attrs_len%2) {
      curr_att_value = *ppattr;
      buildUniversalName(state, curr_att_name, &un);
      Element_SetAttributeNS(elem, un->uri, un->local, un->prefix, curr_att_value, state->docIx);
      destroyUniversalName(un);
    } else {
      curr_att_name = *ppattr;
    }
  }

  /* Now apply namespace decl attributes */
  for (curr = list_head(state->new_nss); curr; curr = list_next(curr)){
    nsm = list_data(curr);
    if (nsm->prefix[0] == '\0'){
      Element_SetAttributeNS(elem, XMLNS_NAMESPACE, "xmlns", "", nsm->uri, state->docIx);
    } else {
      Element_SetAttributeNS(elem, XMLNS_NAMESPACE, nsm->prefix, "xmlns", nsm->uri, state->docIx);
    }
  }
  /* Now clean up the namespace declarations.  */
  /* Note: With destroy fuct NULL, could destroy and init again */
  for (; _stack_peek(state->new_nss); _stack_pop(state->new_nss, &ptr));

  _stack_push(state->node_stack, elem);
  return;
}


void endElement(void *userData, const XML_Char *name)
{
  State *state = (State *)userData;
  int *top_ps;
  void *ptr;
  PyNodeObject *new_elem;
  PyNodeObject *top_node;

  completeText(state);

#ifdef DEBUG
  fprintf(stderr, "End element: %s\n", name);
  dumpState(state);
#endif

  _stack_pop(state->node_stack, &ptr);
  new_elem = (PyNodeObject *)ptr;
  if (!_stack_pop(state->preserve_state, &ptr)){
    top_ps = (int *)ptr;
    free(top_ps);
  }
  top_node = (PyNodeObject *)_stack_peek(state->node_stack);
  Node_AppendChild(top_node, new_elem);
}


void startNsScope(void *userData, const XML_Char *prefix, const XML_Char *uri)
{
  State *state = (State *)userData;
  NsMapping *nsm = NULL;

#ifdef DEBUG
  fprintf(stderr, "Start NS %s{}%s\n", prefix, uri);
  fprintf(stderr, "Begin namespace scope: %s\n", prefix);
  dumpNsMappingList(state->namespaces, "namespaces");
  dumpNsMappingList(state->new_nss, "new_nss");
#endif

  if ((nsm = (NsMapping *)malloc(sizeof(NsMapping))) == NULL) {
    /* Exception: Out of memory */
  }

  nsm->uri = strdup(uri ? uri : "");
  nsm->prefix = strdup(prefix ? prefix : "");

  list_ins_next(state->namespaces, NULL, nsm);
  /* Be careful to avoid double frees */
  set_insert(state->new_nss, nsm);

  return;
}


void endNsScope(void *userData, const XML_Char *prefix)
{
  State *state = (State *)userData;
  void *data;

#ifdef DEBUG
  fprintf(stderr, "End NS %s\n", prefix);
  fprintf(stderr, "End namespace scope: %s\n", prefix);
  dumpState(state);
#endif

  list_find_remove(state->namespaces, (void *)(prefix ? prefix : ""),
                   &(data), matchPrefix);
  free_nsmapping(data);
  return;
}


void characterData(void *userData, const XML_Char *s, int len)
{
  State *state = (State *)userData;
  XML_Char *cdata = NULL;
  int total_len;

  total_len = strlen(state->curr_text)+1;
  if ((cdata = (XML_Char *)realloc(state->curr_text, (total_len + len) * sizeof(XML_Char))) == NULL) {
    /* Exception: Out of memory */
  }
  total_len += len;
  strncat(cdata, s, len);
  state->curr_text = cdata;

#ifdef DEBUG
  fprintf(stderr, "New character data\n");
  dumpState(state);
#endif
}


void processingInstruction(void *userData, const XML_Char *target, const XML_Char *data)
{
  State *state = (State *)userData;
  PyNodeObject *new_pi;
  PyNodeObject *top_node;


  new_pi = (PyNodeObject *)Document_CreateProcessingInstruction(state->ownerDoc, target, data, state->docIx);
  top_node = (PyNodeObject *)_stack_peek(state->node_stack);
  Node_AppendChild(top_node, new_pi);

#ifdef DEBUG
  fprintf(stderr, "New processing instruction\n");
  dumpState(state);
#endif
}


void comment(void *userData, const XML_Char *data)
{
  State *state = (State *)userData;
  PyNodeObject *new_comment;
  PyNodeObject *top_node;


  completeText(state);

  new_comment = (PyNodeObject *)Document_CreateComment(state->ownerDoc, data, state->docIx);

  top_node = (PyNodeObject *)_stack_peek(state->node_stack);
  Node_AppendChild(top_node, new_comment);

#ifdef DEBUG
  fprintf(stderr, "New processing instruction\n");
  dumpState(state);
#endif
}


PyDocumentObject *beginParse(FILE *stream, PyObject *readmethod,
                             StripSpec *stripElements, int numStripElements,
                             const char *refUri)
{
  char buf[BUFSIZ];
  XML_Parser parser = XML_ParserCreateNS(NULL, NSSEP);
  int done;
  size_t len;
  State *state;
  Stack *ps;
  Stack *ns;
  List *nss;
  Set *nnss;
  unsigned long *docIx;
  PyDocumentObject *doc = NULL;
  int *new_preserve_state;
  void *ptr;

  if ((docIx = (unsigned long *)malloc(sizeof(unsigned long))) == NULL) {
    /* Exception: Out of memory */
  }
  *docIx = 0;
  if ((doc = (PyDocumentObject *)Document_NEW(docIx, refUri)) == NULL) {
    /* Exception: Out of memory */
    return NULL;
  }
  if ((state = (State *)malloc(sizeof(State))) == NULL) {
    /* Exception: Out of memory */
  }
  if ((ps = (Stack *)malloc(sizeof(Stack))) == NULL) {
    /* Exception: Out of memory */
  }
  _stack_init(ps, free);
  if ((new_preserve_state = (int *)malloc(sizeof(int))) == NULL) {
    /* Exception: Out of memory */
  }
  *new_preserve_state = 1;
  _stack_push(ps, ((void *)new_preserve_state));
  state->preserve_state = ps;

  if ((ns = (Stack *)malloc(sizeof(Stack))) == NULL) {
    /* Exception: Out of memory */
  }
  _stack_init(ns, free);
  state->node_stack = ns;

  if ((nnss = (Set *)malloc(sizeof(Set))) == NULL) {
    /* Exception: Out of memory */
  }
  set_init(nnss, matchNsMapping, free_nsmapping);
  state->new_nss = nnss;
  if ((nss = (List *)malloc(sizeof(List))) == NULL) {
    /* Exception: Out of memory */
  }
  list_init(nss, free_nsmapping);
  state->namespaces = nss;

  state->ownerDoc = doc;
  _stack_push(state->node_stack, doc);
  if ((state->curr_text = (XML_Char *)malloc(sizeof(XML_Char))) == NULL) {
    /* Exception: Out of memory */
  }
  state->curr_text[0] = '\0';
  state->docIx = docIx;
  state->strip_elements = stripElements;
  state->num_strip_elements = numStripElements;

  XML_SetUserData(parser, state);
  XML_SetElementHandler(parser, startElement, endElement);
  XML_SetCharacterDataHandler(parser, characterData);
  XML_SetNamespaceDeclHandler(parser, startNsScope, endNsScope);
  XML_SetProcessingInstructionHandler(parser, processingInstruction);
  XML_SetCommentHandler(parser, comment);

  do {
    len = readdata(buf,stream,readmethod);
    if (PyErr_Occurred()) {
      return NULL;
    }
    done = len < sizeof(buf);
    if (!XML_Parse(parser, buf, len, done)) {
      /* Exception: Well-formedness error */
      /*PyErr_Format(g_readerErrorObject, ...);*/
      PyErr_Format(PyExc_SyntaxError, "%d:%d:%s",
		   XML_GetCurrentLineNumber(parser),
		   XML_GetCurrentColumnNumber(parser),
		   XML_ErrorString(XML_GetErrorCode(parser)));
      return NULL;
    }
  } while (!done);

  _stack_pop(state->node_stack, &ptr);
  _stack_pop(state->preserve_state, &ptr);

  /*Clean-up*/
  _stack_destroy(state->preserve_state);
  _stack_destroy(state->node_stack);
  list_destroy(state->namespaces);
  set_destroy(state->new_nss);

  doc = state->ownerDoc;
  free(state->preserve_state);
  free(state->node_stack);
  free(state->namespaces);
  free(state->new_nss);
  free(state->curr_text);
  free(new_preserve_state);
  free(state->docIx);
  free(state);

  XML_ParserFree(parser);
  return doc;
}

int readdata(char *buf, FILE *stream, PyObject *readmethod) {
  size_t len;
  PyObject *bytes=NULL;
  PyObject *str=NULL;

  if (stream) {
    len = fread(buf, 1, BUFSIZ, stream);
  } else {
    if (!(str = PyObject_CallFunction(readmethod, "i", BUFSIZ))) {
      return 0;
    }

    if (!(PyString_Check( str ))) {
      return 0;
    }

    len=PyString_GET_SIZE( str );
    strncpy( buf, PyString_AsString(str), len );
    Py_XDECREF( str );
  }
  return len;
}
