

/*
 Copyright (C) 2000, 2001, 2002 RiskMap srl

 This file is part of QuantLib, a free-software/open-source library
 for financial quantitative analysts and developers - http://quantlib.org/

 QuantLib is free software: you can redistribute it and/or modify it under the
 terms of the QuantLib license.  You should have received a copy of the
 license along with this program; if not, please email ferdinando@ametrano.net
 The license is also available online at http://quantlib.org/html/license.html

 This program is distributed in the hope that it will be useful, but WITHOUT
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 FOR A PARTICULAR PURPOSE.  See the license for more details.
*/
/*! \file interpolation.hpp
    \brief abstract base classes for interpolations

    \fullpath
    ql/Math/%interpolation.hpp
*/

// $Id: interpolation.hpp,v 1.8 2002/03/12 10:55:29 nando Exp $

#ifndef quantlib_interpolation_h
#define quantlib_interpolation_h

#include <ql/errors.hpp>

namespace QuantLib {

    namespace Math {

        //! abstract base class for 1-D interpolations
        /*! Classes derived from this class will override operator() to
            provide interpolated values from two sequences of equal length,
            representing discretized values of a variable and a function of
            the former, respectively.
        */
        template <class RandomAccessIterator1, class RandomAccessIterator2>
        class Interpolation {
          public:
            // We do not inherit from std::unary_function because we wouldn't 
            // inherit these typedefs anyway (see 14.6.2.3 of the standard)
            typedef
              typename QL_ITERATOR_TRAITS<RandomAccessIterator1>::value_type
                argument_type;
            typedef
              typename QL_ITERATOR_TRAITS<RandomAccessIterator2>::value_type
                result_type;
            Interpolation(const RandomAccessIterator1& xBegin,
                const RandomAccessIterator1& xEnd,
                const RandomAccessIterator2& yBegin,
                bool allowExtrapolation);
            /*! This operator must be overridden to provide an implementation
                of the actual interpolation.

                \pre The sequence of values for x must have been sorted for
                the result to make sense.
            */
            virtual result_type operator()(const argument_type& x) const = 0;
          protected:
            RandomAccessIterator1 xBegin_, xEnd_;
            RandomAccessIterator2 yBegin_;
            bool allowExtrapolation_;
        };

        // inline definitions

        template <class I1, class I2>
        inline Interpolation<I1,I2>::Interpolation(const I1& xBegin,
            const I1& xEnd, const I2& yBegin, bool allowExtrapolation)
        : xBegin_(xBegin), xEnd_(xEnd), yBegin_(yBegin),
          allowExtrapolation_(allowExtrapolation) {
            #ifdef QL_DEBUG
                QL_REQUIRE(xEnd_-xBegin_ >= 2,
                    "not enough points to interpolate");
            #endif
        }

    }

}


#endif
